#!/bin/bash

###################################
#
# Copyright (c) 2020 Ribbon Communication, Inc.
# All Rights Reserved.
#
# Script to setup running the HFE_AZ.sh as 
# a systemd service. TO be used in Azure
# with distros missing cloudinit functionality.
#
# This scipt needs to be run with heightened
# privileges.
##################################

PROG=${0##*/}
DIR=$(cd -P "$(dirname $0)" && pwd)

HFE_DIR="/opt/HFE"
HFE_LOG_DIR="$HFE_DIR/log"
HFE_FILE="$HFE_DIR/HFE_AZ.sh"
LOG_FILE="$HFE_LOG_DIR/cloud-init-nat.log"
NAT_VAR="$HFE_DIR/natVars.input"

###############################
# UPDATE VARIABLES IN THIS SECTION
AZ_BLOB_URL="<HFE_SCRIPT_LOCATION>" # URL of uploaded HFE script
ACTIVE_SBC_NAME="<ACTIVE_SBC_NAME>"
STANDBY_SBC_NAME="<STANDBY_SBC_NAME>"
REMOTE_SSH_MACHINE_IP="<REMOTE_SSH_MACHINE_IP>"
SBC_PKT_PORT_NAME="<SBC_PKT_PORT_NAME>"    # Only use for HFE 2.1
#
##############################


timestamp()
{
    date +"%Y-%m-%d %T"
}

log()
{
    echo $(timestamp) "$1" >> $LOG_FILE
    logger -t "ribbon-hfe" "$1"
}

usage()
{
   echo ""
   echo "$PROG usage:"
   echo " Script used to setup to initialise the HFE script and also create the systemd file"
   echo " To be used with distros without cloudinit support"
   echo "  -s : Setup systemd to run this script to initialize the HFE at startup"
   echo "  -r : Run the initial setup of HFE"
   echo "  -h : Prints this message"
   echo ""
}

runHfe()
{
    log " ========================= Initial configuration for HFE =========================================="
    curl "$AZ_BLOB_URL" -H 'x-ms-version : 2019-02-02' -o $HFE_FILE
    if [ $? -ne 0 ]; then
        log "Error: Could not copy HFE script from Azure Blob Container."
    else
        log "Copied HFE script from Azure Blob Container." 
    fi

    echo > $NAT_VAR
    echo "ACTIVE_SBC_VM_NAME=\"$ACTIVE_SBC_NAME\"" >> $NAT_VAR
    echo "STANDBY_SBC_VM_NAME=\"$STANDBY_SBC_NAME\"" >> $NAT_VAR
    echo "REMOTE_SSH_MACHINE_IP=\"$REMOTE_SSH_MACHINE_IP\"" >> $NAT_VAR
    if [[ $SBC_PKT_PORT_NAME != "" ]]; then
        echo "SBC_PKT_PORT_NAME=\"$SBC_PKT_PORT_NAME\"" >> $NAT_VAR
    fi
    log "Copied natVars.input"
    sudo chmod 744 $HFE_FILE
    log "Configured using HFE script - $HFE_FILE"
    log " ========================= Done =========================================="
    $HFE_FILE setup
}

setupHfe()
{
    if [[ $(command -v setenforce) != "" ]]; then
        log "Setting SELinux to Permissive"
        setenforce Permissive
        sed -i 's/SELINUX=.*/SELINUX=permissive/' /etc/selinux/config
    fi

    log "Creating a systemd file for ribbon-hfe"
    systemdHfe=/usr/lib/systemd/system/ribbon-hfe.service
    echo "[Unit]
Description=Ribbon Communications HFE script for use with HA SBX in Azure
Wants=network-online.target
After=network-online.target
StartLimitIntervalSec=15
StartLimitBurst=5

ConditionFileIsExecutable=$DIR/$PROG

[Service]
Type=simple
ExecStart=$DIR/$PROG -r
Restart=always
RestartSec=1

[Install]
WantedBy=multi-user.target" > $systemdHfe

   systemctl enable ribbon-hfe 
}

if [ ! -d $HFE_LOG_DIR ]; then
    mkdir -p $HFE_LOG_DIR;
fi

if [[ $AZ_BLOB_URL == \<*\> ]] || [[ $ACTIVE_SBC_NAME == \<*\> ]] || [[ $STANDBY_SBC_NAME == \<*\> ]] || [[ $REMOTE_SSH_MACHINE_IP == \<*\> ]] || [[ $SBC_PKT_PORT_NAME == \<*\> ]]; then
    msg="Not all of the HFE variables are updated. Exiting!"
    log "$msg"
    echo $msg
    exit 1
fi

while getopts srh o
do
    case $o in 
    s) setupHfe ;;
    r) runHfe ;;
    h) usage ;;
    *) usage && exit 1 ;;
    esac
done
