#!/bin/bash

################################################################################
#
# Bash wrapper for handling of repave command line options.  Check and validate
# the options before passing them to the expect script.
#
################################################################################
PROG=${0##*/}
PROG_DIR=$(cd "$(/usr/bin/dirname $(readlink -f "${BASH_SOURCE[0]}" ))" && pwd)
image=""
username=""
unameDefault="root"
userpwd=""
configopt=""
verbose=0
mgmtIP=""
host=`hostname`
cmdArgs="$*"
reload=0

# guarantee the user's terminal type doesn't interfere with expect matching
export TERM=xterm

usage()
{
cat << EOF
usage: $PROG [-h] -m <mgmt_ip> -i <image> [-u <user>] -r <pwd> [-c <configfile>] [-v] [-R]

OPTIONS:
   -h               Print this message.
   -m <mgmt_ip>     Specify the machine IP address to be installed
   -i <image>       Specify the image to be installed
   -u <user>        Specify the username used to log into the management port (default: $unameDefault)
   -r <pwd>         Specify the password used to log into the management port
   -c <configfile>  sbx.conf file for auto-install of the application
   -v               Verbose mode: bitwise combination of the following:
                    0x000 - off
                    0x001 - extra user logging
                    0x002 - full expect logging
                    0x004 - mgmt login logging
                    0x008 - mgmt ssh logging (adds -vvv to ssh)
   -R               reload an existing SBC


   NOTE: if user is specified, that user must be configured within sudo to allow for
         passowrd-less promotion to root
EOF
}

while getopts "hm:i:u:r:p:c:v:R" OPTION
do
   case $OPTION in
   h)
      usage
      exit 0
      ;;
   m)
      mgmtIP="$OPTARG"
      ;;
   i)
      image="$OPTARG"
      ;;
   u)
      username="$OPTARG"
      ;;
   r)
      userpwd="$OPTARG"
      ;;
   c)
      configopt="$OPTARG"
      ;;
   v)
      verbose=$(( OPTARG ))
      ;;
   R)
      reload=1
      ;;
   ?)
      echo -e "$PROG: invalid argument specified\n"
      usage
      exit 1
      ;;
   esac
done

shift $(($OPTIND - 1))

if [ $# -ne 0 ]; then
  echo -e "$PROG: additional/invalid options specified\n"
  usage
  exit 1
elif [ -z "$mgmtIP" ]; then
  echo -e "$PROG: mgmt IP must be specified\n"
  usage
  exit 1
elif [ -z "$image" ]; then
  echo -e "$PROG: image must be specified\n"
  usage
  exit 1
elif [ -z "$userpwd" ]; then
  echo -e "$PROG: ${username:-$unameDefault} password must be specified\n"
  usage
  exit 1
elif [ $verbose -lt 0 ]; then
  echo -e "$PROG: verbosity setting must be greater or equal to 0\n"
  usage
  exit 1
fi

# attempt to timestamp the logs by using the ts utility from package moreutils
useTimestamp="n"
if [ -e /etc/debian_version ]; then
  dpkg -s moreutils >& /dev/null
  if [ $? -eq 0 ]; then
      useTimestamp="y";
  fi
elif [ -e /etc/redhat-release ]; then
  rpm -q moreutils >& /dev/null
  if [ $? -eq 0 ]; then
      useTimestamp="y";
  fi
fi

# use a variable for options to reduce code duplication
expectOptions="$mgmtIP $image ${username:-$unameDefault} $userpwd $verbose $reload $configopt"

if [ $verbose -ne 0 ]; then
  echo "calling $PROG_DIR/repave.expect $expectOptions"
fi

if [ "$useTimestamp" == "y" ]; then
  $PROG_DIR/repave.expect $expectOptions | ts
else
  $PROG_DIR/repave.expect $expectOptions
fi
