#!/bin/bash

#######################################$#########################################
#
# Bash wrapper for handling of repaveHW command line options.  Check and validate
# the options before passing them to the expect script.
#
#################################################################################
PROG=${0##*/}
PROG_DIR=$(cd "$(/usr/bin/dirname $(readlink -f "${BASH_SOURCE[0]}" ))" && pwd)
machine=""
image=""
rootpwd=""
bmcpwd=""
configopt=""
verbose=0
mgmtIP=""
bmcIP=""
host=`hostname`
cmdArgs="$*"

# guarantee the user's terminal type doesn't interfere with expect matching
export TERM=xterm

usage()
{
cat << EOF
usage: $PROG [-h] [-m <machine>] -i <image> -r <pwd> -b <pwd> [-c <configfile>] [-M mgmt_ip -B bmc_ip] [-v]

OPTIONS:
   -h               Print this message.
   -m <machine>     Specify the machine name to be installed
   -i <image>       Specify the image to be installed
   -r <pwd>         Specify the root user password
   -b <pwd>         Specify the BMC user password
   -c <configfile>  sbx.conf file for auto-install of the application
   -M <mgmt_ip>     Specify the machine to be installed via IP address
   -B <bmc_ip>      Specify the IP address for the BMC
   -v               Verbose mode: bitwise combination of the following:
                    0x000 - off
                    0x001 - extra user logging
                    0x002 - full expect logging
                    0x004 - mgmt login logging
                    0x008 - watchdog logging
                    0x010 - mgmt ssh logging (adds -vvv to ssh)
                    0x020 - bmc ssh logging (adds -vvv to ssh)

  NOTE: Either -m or -M must be specified
  NOTE: -B is mandatory if -M is used, and is not utilized when -m is used
EOF
}

while getopts "hm:i:r:b:c:B:M:v:" OPTION
do
   case $OPTION in
   h)
      usage
      exit 0
      ;;
   m)
      machine="$OPTARG"
      ;;
   i)
      image="$OPTARG"
      ;;
   r)
      rootpwd="$OPTARG"
      ;;
   b)
      bmcpwd="$OPTARG"
      ;;
   c)
      configopt="$OPTARG"
      ;;
   B)
      bmcIP="$OPTARG"
      ;;
   M)
      mgmtIP="$OPTARG"
      ;;
   v)
      verbose=$(( OPTARG ))
      ;;
   ?)
      echo -e "$PROG: invalid argument specified\n"
      usage
      exit 1
      ;;
   esac
done

shift $(($OPTIND - 1))

if [ $# -ne 0 ]; then
  # note: catch extra args and when 1st arg is an option but old usage is used
  echo -e "$PROG: additional/invalid options specified\n"
  usage
  exit 1
elif [ -z "$machine" -a -z "$mgmtIP" ]; then
  echo -e "$PROG: machine name or mgmt IP must be specified\n"
  usage
  exit 1
elif [ -n "$machine" -a -n "$mgmtIP" ]; then
  echo -e "$PROG: only one of machine name or mgmt IP may be specified\n"
  usage
  exit 1
elif [ -n "$machine" -a -n "$bmcIP" ]; then
   echo -e "$PROG: must NOT specify BMC IP when specifying machine name\n"
   usage
   exit 1
elif [ -n "$mgmtIP" -a -z "$bmcIP" ]; then
  echo -e "$PROG: must specify BMC IP when specifying mgmt IP\n"
  usage
  exit 1
elif [ -z "$image" ]; then
  echo -e "$PROG: image must be specified\n"
  usage
  exit 1
elif [ -z "$rootpwd" -o -z "$bmcpwd" ]; then
  echo -e "$PROG: both root user password and BMC user password must be specified\n"
  usage
  exit 1
elif [ $verbose -lt 0 ]; then
  echo -e "$PROG: verbosity setting must be greater or equal to 0\n"
  usage
  exit 1
fi

# for debug purposes....
echo "$PROG is running on $host: called with \"$cmdArgs\""


# attempt to timestamp the logs by using the ts utility from package moreutils
timestampOption="n"
if [ -e /etc/debian_version ]; then
  dpkg -s moreutils >& /dev/null
  if [ $? -eq 0 ]; then
      timestampOption="y"
  fi
elif [ -e /etc/redhat-release ]; then
  rpm -q moreutils >& /dev/null
  if [ $? -eq 0 ]; then
      timestampOption="y"
  fi
fi

# setup some positional variables to reduce code duplication
# note: add params for both usage types before bmcIParg
machineArg=${machine:=$mgmtIP}
bmcIPArg=${bmcIP:=0.0.0.0}
expectOptions="$machineArg $image $rootpwd $bmcpwd $verbose 0 0 0 0 0 0 $bmcIPArg $configopt"

# NOTE: timestampOption is NOT passed to repave.expect. it is either null or a pipe to the
# ts utility!
echo "$PROG: calling $PROG_DIR/isoSbx.expect $expectOptions"

if [ "$timestampOption" == "y" ]; then
  $PROG_DIR/repave.expect $expectOptions | ts
else
  $PROG_DIR/repave.expect $expectOptions
fi
