#!/bin/bash

#############################################################
#
# Copyright (c) 2018 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# HFE_GCE.sh
#
# Chris John
# 17/7/2019
#
# Module Description:
# Script to enable HFE(High-Availability Front End) instance as frontend for
# PKT ports of SBC.
# Call "HFE_GCE.sh setup [1]" - to configure HFE with Alias or Primary IPs
# The script will perform the following steps when called from cloud-init (setup function):
# 1) Save and clear old iptables rules : preConfigure
# 2) Enable IP Forwarding :configureNATRules
# 3) Read variables from natVars.input configured from userdata: readConfig
# 4) Determine interface names and GW IPs on HFE: setInterfaceMap
# 5) Extract the interface IPs required to configure the HFE for pkt0 and pkt1
#    ports: extractHfeConfigIps
# 6) Setup DNATs, SNATs and routes to route traffic through HFE to/from SBC for
#        - public IP for SBC's pkt0 port and forward it to Alias IP over ens7
#        - private IP for SBC's pkt1 port and forward to Alias IP over ens8
#    : configureNATRules
# 7) Configure Management interface to HFE: configureMgmtNAT
# 8) Configure customer specified routes: addCustomerRoutes
# 9) Log applied iptables configuration and routes: showCurrentConfig 
# 10) Check connectivity to currently Active SBC using ping and switchover to Standby SBC
#    if connection is lost to current Active: checkSbcConnectivity
#
#
# Call "HFE_GCE.sh cleanup" to clear all IP tables and Routes on the HFE.
# 1) Save and clear old iptables rules : preConfigure
# 2) Read variables from natVars.input configured from userdata: readConfig
# 3) Determine interface names and GW IPs on HFE: setInterfaceMap
# 4) Extract the interface IPs required to configure the HFE for pkt0 and pkt1
#    ports: extractHfeConfigIps
# 5) Route clean up for Pkt0 and Pkt1 IP and remote machine's public IP: routeCleanUp
#
# This option is useful to debug connectivity of end-point with HFE, after
# calling this no packet is forwarded to SBC, user can ping IP on ens4 to
# make sure connectivity between end-point and HFE is working fine.
# Once this is done user MUST reboot HFE node to restore all forwarding rules
# and routes.
# 
# NOTE: This script is run by google_metadata_script_runner in HFE instance.
#
# This script should be uploaded to Google Cloud Storage and
# GCE HFE Terraform tempates configures HFE instance to
# get this script from storage and run with google_metadata_script_runner.
#
# NOTE: This can used for HFE 2.1 or 2.0. For HFE 2.1 include
# the variable SBC_PKT_PORT_NAME in the HFE instance user data.
#
# NOTE: This script requires either apt or yum package managers
#############################################################

declare -A interfaceMap
declare -A commandMap
declare -A zoneMap


HFERoot="/opt/HFE"
GCLOUD="$(command -v gcloud)"
varFile="$HFERoot/natVars.input"
confLogFile="$HFERoot/log/HFE_conf.log"
statusLogFile="$HFERoot/log/HFE.log"
oldRules="$HFERoot/iptables.rules.prev"
IPTABLES="$(command -v iptables)"
ACTIVE_SBC_IP=""
STANDBY_SBC_IP=""
PRIMARY_IP_HFE_ETH0=""
PRIMARY_GCE_IP=0
APP_INSTALL_MARKER="$HFERoot/app_install_marker"
hfeLogRotateConf="/etc/logrotate.d/hfe"
TMP_LOG_DIR="/tmp/sysdump"

DHCLIENT="$(command -v dhclient)"

# PRIMARY IP DNS VARS
ENABLE_PKT_DNS_QUERY=0
declare -A primaryIpMap
PRIMARY_FILE="$(mktemp)"

# custom routes
CUSTOM_ROUTES_ARR=( )

SBC_PKT_PORT_NAME=""
MODE=""
INIT_LOG=1

DEBUG_MODE=0

# Gcloud timeout vars
CMD_TIMEOUT="1m"
CMD_TIMEOUT_CODE=124

# HA related 
initPubIpCfg=""
initAliasIps=""
initRoute=""
HA_MODE=0
HA_SWITCHOVER_RETRY=2
declare -A routeInfoMap
HFE_NAME=""

PROG=${0##*/}

usage()
{
    echo $1
    echo "usage: $PROG <setup [1]|cleanup | sysdump>"
    echo "Example:"
    echo "$PROG setup - Setup HFE with Alias IPs" 
    echo "$PROG setup 1 - Setup HFE with Primary IPs"
    echo "$PROG cleanup - Cleanup IP tables and route on HFE" 
    echo "$PROG sysdump - Collect logs used for debugging HFE" 
    exit
}

timestamp()
{
 date +"%Y-%m-%d %T"
}

timestampDebug()
{
 date +"%F %T,%3N"
}


loggerHfe()
{
    echo $(timestamp) "$1" >> $confLogFile 2>&1
}

statusLoggerHfeDebug()
{ 
    if [ $DEBUG_MODE -eq 1 ]; then
        echo $(timestamp) "$1" >> $statusLogFile 2>&1
    fi
}

statusLoggerHfe()
{
    echo $(timestamp) "$1" >> $statusLogFile 2>&1
}

doneMessage()
{
    loggerHfe " =========================    DONE HFE_GCE.sh     =========================================="
    exit 
}

errorAndExit()
{
    loggerHfe " Error: $1"
    doneMessage
}

startReboot()
{
    if [ ! -z "$1" ]; then
        loggerHfe " Error: $1"
    fi
    reboot --force
    doneMessage
}

clearOldIptableRules()
{
    # reset the default policies in the filter table.
    $IPTABLES -P INPUT ACCEPT
    $IPTABLES -P FORWARD ACCEPT
    $IPTABLES -P OUTPUT ACCEPT

    # reset the default policies in the nat table.
    $IPTABLES -t nat -P PREROUTING ACCEPT
    $IPTABLES -t nat -P POSTROUTING ACCEPT
    $IPTABLES -t nat -P OUTPUT ACCEPT

    # reset the default policies in the mangle table.
    $IPTABLES -t mangle -P PREROUTING ACCEPT
    $IPTABLES -t mangle -P POSTROUTING ACCEPT
    $IPTABLES -t mangle -P INPUT ACCEPT
    $IPTABLES -t mangle -P OUTPUT ACCEPT
    $IPTABLES -t mangle -P FORWARD ACCEPT

    # flush all the rules in the filter and nat tables.
    $IPTABLES -F
    $IPTABLES -t nat -F
    $IPTABLES -t mangle -F

    # erase all chains that's not default in filter and nat table.
    $IPTABLES -X
    $IPTABLES -t nat -X
    $IPTABLES -t mangle -X
}

removeRoute()
{
    $IP route | grep -E "$1.*$2.*$3"
    if [ "$?" = "0" ]; then
        $IP route del $1 via $2 dev $3
        if [ "$?" = "0" ]; then
            loggerHfe " Route deleted to for $1 "
        else
            errorAndExit " Failed to delete route for $1 "
        fi
    else
        loggerHfe " Route not available for $1 "
    fi
}

routeCleanUp()
{
    loggerHfe "Route clean up for Active and Standby IP and remote machine's public IP."

    ## Cleanup IP routes to SBC
    if [[ $MODE == "single" ]]; then
        #PKT0
        removeRoute $ACTIVE_SBC_IP $ETH3_GW $ETH3
        removeRoute $STANDBY_SBC_IP $ETH3_GW $ETH3
        #PKT1
        removeRoute $ACTIVE_SBC_IP_PKT1 $ETH4_GW $ETH4
        removeRoute $STANDBY_SBC_IP_PKT1 $ETH4_GW $ETH4
    else
        removeRoute $ACTIVE_SBC_IP $ETH2_GW $ETH2
        removeRoute $STANDBY_SBC_IP $ETH2_GW $ETH2
    fi

    ## Cleanup Ip route from Remote Machine to HFE
    IFS=',' read -ra SSH_IP_LIST <<< "$REMOTE_SSH_MACHINE_IP"
    for REMOTE_MACHINE_IP in "${SSH_IP_LIST[@]}"; do
       REMOTE_MACHINE_IP="$(echo $REMOTE_MACHINE_IP | sed 's~/32$~~')" # IP route will ignore /32/
       removeRoute $REMOTE_MACHINE_IP $MGMT_GW $MGMT
    done

    for CUSTOM_ROUTE in "${CUSTOM_ROUTES_ARR[@]}"; do
        removeRoute "$(echo $CUSTOM_ROUTE | awk -F'_' '{print $1}')" "$(echo $CUSTOM_ROUTE | awk -F'_' '{print $2}')" "$(echo $CUSTOM_ROUTE | awk -F'_' '{print $3}')"
    done

    # Remove mgmt route to metadata server
    removeRoute "169.254.169.254" $MGMT_GW $MGMT
}

# $1 - IP / Subnet Prefix
# $2 - Gateway
# $3 - Interface
verifyRoute()
{
    statusLoggerHfeDebug "verifyRoute $1 $2 $3"
    $IP route | grep -E "$1.*$3"
    if [ $? -ne 0 ]; then
        statusLoggerHfeDebug "Adding missing route for $1."
        $IP route add $1 via $2 dev $3
        return 1
    fi
    return 0
}

verifyRoutes()
{
    ret=0
    if [[ $MODE == "single" ]]; then
        verifyRoute $ACTIVE_SBC_IP $ETH3_GW $ETH3
        [ $? -eq 1 ] && ret=1
        verifyRoute $ACTIVE_SBC_IP_PKT1 $ETH4_GW $ETH4
        [ $? -eq 1 ] && ret=1
    else
        verifyRoute $ACTIVE_SBC_IP $ETH2_GW $ETH2
        [ $? -eq 1 ] && ret=1
    fi

    IFS=',' read -ra SSH_IP_LIST <<< "$REMOTE_SSH_MACHINE_IP"
    for REMOTE_MACHINE_IP in "${SSH_IP_LIST[@]}"; do
       verifyRoute $REMOTE_MACHINE_IP $MGMT_GW $MGMT
       [ $? -eq 1 ] && ret=1
    done

    for CUSTOM_ROUTE in "${CUSTOM_ROUTES_ARR[@]}"; do
        verifyRoute "$(echo $CUSTOM_ROUTE | awk -F'_' '{print $1}')" "$(echo $CUSTOM_ROUTE | awk -F'_' '{print $2}')" "$(echo $CUSTOM_ROUTE | awk -F'_' '{print $3}')"
        [ $? -eq 1 ] && ret=1
    done

    return $ret
}

verifyPackages()
{
    local missingCmd=0
    
    commandMap["gcloud"]=$GCLOUD
    commandMap["iptables"]=$IPTABLES
    commandMap["jq"]=$JQ
    commandMap["conntrack"]=$CONNTRACK
    commandMap["fping"]=$FPING
    commandMap["ip"]=$IP
    commandMap["dhclient"]=$DHCLIENT

    for i in "${!commandMap[@]}"
    do
        if [[ ${commandMap[$i]} == "" ]]; then
            loggerHfe "Required packages $i is missing."
            missingCmd=1
        fi
    done

    if [ $missingCmd -eq 1 ]; then
       errorAndExit "Missing required packages. Exiting!!!"
    fi
}


prepareHFEInstance()
{
    ### Configure ip_forward - Enable ip forwarding
    echo 1 >  /proc/sys/net/ipv4/ip_forward

    if [ ! -f $APP_INSTALL_MARKER ]; then
        ### Install required packages
 
        if [[ $(command -v yum) == "" ]]; then
            sudo apt-get update
            sudo apt-get -y install jq
            sudo apt-get -y install conntrack
            sudo apt-get -y install fping

            # Kill auto package updates
            sudo systemctl stop apt-daily.service
            sudo systemctl stop apt-daily.timer
            sudo systemctl stop apt-daily-upgrade.timer
            sudo systemctl stop apt-daily-upgrade.service
            sudo systemctl disable apt-daily.service
            sudo systemctl disable apt-daily.timer
            sudo systemctl disable apt-daily-upgrade.timer
            sudo systemctl disable apt-daily-upgrade.service
        else
            sudo yum -y update
            sudo yum -y install epel-release
            if [ $? -ne 0 ]; then
                majorOsVer=$(cat /etc/os-release | grep VERSION_ID | awk -F= '{print $2}' | sed 's/"//g' | sed 's/\..*//')
                sudo yum -y install wget
                cd /tmp/
                wget https://dl.fedoraproject.org/pub/epel/epel-release-latest-$majorOsVer.noarch.rpm
                epelRpm=$(ls epel*.rpm)
                sudo yum -y install $epelRpm
            fi
            sudo yum -y install jq
            sudo yum -y install conntrack
            sudo yum -y install fping
        fi
        setLogRotate
    fi
    
    JQ="$(command -v jq)"
    CONNTRACK="$(command -v conntrack)"
    FPING="$(command -v fping)"
    IP="$(command -v ip)"
    verifyPackages

    # If we get this far, packages are good
    echo 1 > $APP_INSTALL_MARKER
}

preConfigure()
{
    if [ -f $confLogFile ]; then
        mv -f $confLogFile $confLogFile.prev
    fi
    ### Redirect all echo $(timestamp) to file after writing ip_forward
    loggerHfe " ==========================   Starting HFE_GCE.sh  ============================"
    loggerHfe " Enabled IP forwarding"
    loggerHfe " This script will setup DNAT, SNAT and IP forwarding."
    loggerHfe " Save old rules in $HFERoot/firewall.rules"

    ### Save old iptable rules
    sudo iptables-save > $oldRules
    if [ "$?" = "0" ]; then
        ### Clear the iptables
        clearOldIptableRules
    else
        errorAndExit "Cound not save old iptables rules. Exiting"
    fi


}

# Run dhclient on interfaces which are down
fixInterfaces()
{
    local dhclinetRun=0
    local lastIntf=""
    while read intf; do
        $DHCLIENT $intf
        if [ $? -ne 0 ]; then
            errorAndExit "dhclient failed to bring up $intf"
        else
            loggerHfe " Brought up $intf using dhclient"
        fi
        dhclientRun=1
        lastIntf=$intf
    done < <($IP a | grep -E eth.: | grep DOWN | awk -F' ' '{print $2}' | sed 's/://')

    # Wait for everything to configured before continuing
    if [ $dhclinetRun -eq 1 ]; then
        for i in $(seq 0 4); do
            if [[ "$($IP route | tr -s ' ' | grep "$lastIntf scope link" |  awk '{ print $1}')" == "" ]]; then
                sleep 0.5s
            else
                return    # We are ready to continue
            fi
        done
        errorAndExit "Failed to find routes for $intf after 2 seconds. IP routes: \n\n $($IP route)"
    fi
}

setInterfaceMap()
{
    instanceInfo=`$GCLOUD compute instances describe $HFE_NAME --format="json" --zone=${zoneMap[$HFE_NAME]}`
    instanceNum=$(echo $instanceInfo | $JQ '.networkInterfaces' | $JQ length)

    # wait until all interfaces are configured
    while [ $($IP address show | egrep ^[0-9]: | wc -l) -gt $($IP address | grep -w inet | wc -l) ]; do
        sleep 0.1s
    done
    fixInterfaces

    for i in $(seq 0 $((instanceNum-1))); do
        nicIp=$(echo $instanceInfo | $JQ ".networkInterfaces[] | select(.name == \"nic$i\") | .networkIP " | awk -F'"' '{print $2}')

        while read IF; do
            ifIp=$($IP address show $IF | grep -w inet | awk -F ' ' '{print $2}' | awk -F '/' '{print $1}')

            if [ $(echo $ifIp | grep -cw $nicIp) -gt 0 ]; then
                interfaceMap["eth$i"]=$IF
            fi

        done < <($IP address show | egrep ^[0-9]: | awk -F ' ' '{print $2}' | sed 's/://')
    done

    ETH0=${interfaceMap['eth0']}
    ETH1=${interfaceMap['eth1']}
    ETH2=${interfaceMap['eth2']}
    if [[ $MODE == "single" ]]; then
        ETH3=${interfaceMap['eth3']}
        ETH4=${interfaceMap['eth4']}
    fi

    if [[ $MODE == "single" ]]; then
        ETH2_GW=`$IP route | tr -s ' ' | grep "$ETH2 scope link" |  awk '{ print $1}'`
        ETH3_GW=`$IP route | tr -s ' ' | grep "$ETH3 scope link" |  awk '{ print $1}'`
        ETH4_GW=`$IP route | tr -s ' ' | grep "$ETH4 scope link" |  awk '{ print $1}'`
        loggerHfe ""
        loggerHfe " Default GWs:"
        loggerHfe " ETH2_GW          $ETH2_GW"
        loggerHfe " ETH3_GW          $ETH3_GW"
        loggerHfe " ETH4_GW          $ETH4_GW"
    else
        ETH1_GW=`$IP route | tr -s ' ' | grep "$ETH1 scope link" |  awk '{ print $1}'`
        ETH2_GW=`$IP route | tr -s ' ' | grep "$ETH2 scope link" |  awk '{ print $1}'`
        loggerHfe ""
        loggerHfe " Default GWs:"
        loggerHfe " ETH1_GW          $ETH1_GW"
        loggerHfe " ETH2_GW          $ETH2_GW"
    fi

    # Set the MGMT interface information
    if [[ $MODE == "single" ]]; then
        MGMT=$ETH2
        MGMT_GW=$ETH2_GW
    else
        MGMT=$ETH1
        MGMT_GW=$ETH1_GW
    fi
}

isInDifferentZone()
{
    [ $(echo $1 | grep -c '/zones/') -gt 0 ] && return 0
    return 1
}

## Support for multi zone deployements
## Remove all of the zonal information
## from the natvars. Also create map of
## zones so we only work it out once.
stripZonesAndMap()
{
    local zone=$ZONE
    #zoneMap
    for natVar in "ACTIVE_SBC_VM_NAME" "STANDBY_SBC_VM_NAME" "HFE_NAME" "PEER_HFE_NAME"; do
        [[ -z "${!natVar}" ]] && continue     # No value set for variable
        zone=$ZONE
        isInDifferentZone "${!natVar}"
        if [ $? -eq 0 ]; then
            zone="$(echo ${!natVar} | awk -F'/' '{print $3}')"
            loggerHfe " NatVar $natVar has zone information. Removing."
            declare -g $natVar="$(echo ${!natVar} | awk -F'/' '{print $5}')"    #Overwrite existing variable
            loggerHfe " NatVar $natVar set as ${!natVar}"
        fi
        zoneMap["${!natVar}"]="$zone"
    done
}

readConfig()
{
    loggerHfe " Read variables from file $varFile"

    source $varFile
    SBC_PKT_PORT_NAME=$(echo $SBC_PKT_PORT_NAME | awk '{print toupper($0)}') #Force UpperCase
    loggerHfe " Data from $varFile:"
    if [[ $SBC_PKT_PORT_NAME == "" ]]; then
        MODE="single"
    else
        MODE="split"
        loggerHfe " SBC_PKT_PORT_NAME $SBC_PKT_PORT_NAME"
    fi
    loggerHfe " ACTIVE_SBC_VM_NAME $ACTIVE_SBC_VM_NAME"
    loggerHfe " STANDBY_SBC_VM_NAME $STANDBY_SBC_VM_NAME"
    loggerHfe " REMOTE_SSH_MACHINE_IP $REMOTE_SSH_MACHINE_IP"
    loggerHfe " ZONE $ZONE"
    loggerHfe " ENABLE_PKT_DNS_QUERY $ENABLE_PKT_DNS_QUERY"
    loggerHfe " CUSTOM_ROUTES $CUSTOM_ROUTES"
    loggerHfe " PEER_HFE_NAME $PEER_HFE_NAME"
    loggerHfe " SWITCHOVER_MODE $SWITCHOVER_MODE"
    loggerHfe " ROUTE_NAME $ROUTE_NAME"
    loggerHfe ""

    if [[ $MODE == "single" ]]; then
        loggerHfe " SBC_PKT_PORT_NAME is not configured. Assuming single HFE instance with 5 interfaces"
        loggerHfe ""
    fi

    HFE_NAME=$(curl 'http://169.254.169.254/computeMetadata/v1/instance/name' -H 'Metadata-Flavor:Google')

    stripZonesAndMap
}

## Parameters passed in to configureNATRules
## $1 - Interface Name for Public/Private connection in to HFE
## $2 - Interface Name for Pkt0/Pkt1 to Active SBC
## $3 - Gateway for Pkt0/Pkt1 to Active SBC
## $4 - IP of Active SBC to Pkt0/Pkt1
## $5 - Input IP of HFE
## $6 - Print configuration messages to HFE log
configureNATRules()
{
    ## Set forward ACCEPT rule for packets coming into HFE
    $IPTABLES -I FORWARD -i $1 -o $2 -j ACCEPT
    if [ "$?" = "0" ]; then
        [[ ! -z $6 ]] && loggerHfe " Set Forward ACCEPT rule all packets coming from outside $1 to $2 towards SBC"
    else
        errorAndExit "Failed to set forward ACCEPT rule for all packets coming on IP($1)"
    fi
    
    ## Set route to reach SBC Pkt Port
    $IP route | grep -E "$4.*$3.*$2"
    if [ "$?" = "0" ]; then
        [[ ! -z $6 ]] && loggerHfe " Route is already available to reach SBC $SBC_PKT_PORT_NAME $4 from $2"
    else
        $IP route add $4 via $3 dev $2
        if [ "$?" = "0" ]; then
            [[ ! -z $6 ]] && loggerHfe " Set route to reach SBC $SBC_PKT_PORT_NAME $4 from $2"
        else
            errorAndExit "Failed to set route to reach SBC $4 from $2"
        fi
    fi
 
    ## Set DNAT for detination IP
    $IPTABLES -t nat -I PREROUTING -d $5 -j DNAT --to $4
    if [ "$?" = "0" ]; then
        [[ ! -z $6 ]] && loggerHfe " Set up proper DNAT for destination IP $5 to offset $4."
    else
        errorAndExit "Failed to set DNAT rule for destination IP $5 to offset $4."
    fi

    ## Set forward ACCEPT rule for packets coming from SBC
    $IPTABLES -I FORWARD -i $2 -o $1 -j ACCEPT
    if [ "$?" = "0" ]; then
        [[ ! -z $6 ]] && loggerHfe " Set Forward ACCEPT rule all packets coming from SBC ($2) to $1"
    else
        errorAndExit "Failed to set ACCEPT rule all packets coming from SBC ($2) to $1"
    fi

    ## Set SNAT for external interface to HFE
    $IPTABLES -t nat -I POSTROUTING -o $1 -s $4 -j SNAT --to $5
    if [ "$?" = "0" ]; then
        [[ ! -z $6 ]] && loggerHfe " Set up POSTROUTING rule (source IP $4, to offset $5) for packet sent on $1"
    else
        errorAndExit "Failed to set POSTROUTING rule (source IP $4, to offset $5) for packet sent on $1"
    fi
 
    # Handle DNS requests on PKT ports.
    # This is done using Primary IPs, so create SNAT rules for primary IPs
    if [ $ENABLE_PKT_DNS_QUERY -eq 1 ]; then 
        if [[ $1 == $ETH0 ]]; then
            primarySbcIp=${primaryIpMap["$ACTIVE_SBC_VM_NAME"]}
        else
            primarySbcIp=${primaryIpMap["$ACTIVE_SBC_VM_NAME""-PKT1"]}
        fi
        primaryHfeIp=${primaryIpMap["HFE_$1"]}

        $IPTABLES -t nat -I POSTROUTING -o $1 -s $primarySbcIp -j SNAT --to $primaryHfeIp
        if [ $? -eq 0 ]; then
            [[ ! -z $6 ]] && loggerHfe " Set up primary IP POSTROUTING rule (source IP $primarySbcIp, to offset $primaryHfeIp) for packet sent on $1"
        else
            errorAndExit "Failed to set primary IP POSTROUTING rule (source IP $primarySbcIp, to offset $primaryHfeIp) for packet sent on $1"
        fi
    else
        [[ ! -z $6 ]] && loggerHfe " ENABLE_PKT_DNS_QUERY not enabled. Not forwarding primary PKT IP"
    fi
}

###################################################################################################
####                                         WARNING 
####
####                Each call of this fucntion  will result in pkt drop.
####
#### Each conntrack flush operation has a associated time penalty (pkt drop), call this function
#### only when you are all set with setting up new iptables rules (and old rules are cleaned up).
####
####
####
###################################################################################################

clearOldRules()
{
    ## Reset connection tracking
    ## Any packet received on input interfaces before NAT rules are set are not forwarded
    ## to sbc, connection tracking will not forward those packets even
    ## if NAT rules are set after receiving first packet from that source
    ## IP/Port as it has cache entry for source IP and Port.
    ## Reset connection tracking will treat them as new stream and those packets
    ## will be forwarded to SBC once SNAT and DNAT rules are setup
    ## properly.

    $CONNTRACK -F conntrack
    if [ "$?" = "0" ]; then
        [[ ! -z $1 ]] && loggerHfe " Flushing connection tracking rules."
    else
        [[ ! -z $1 ]] && loggerHfe " (WARNING):Flushing connection tracking rules failed."
    fi
}

# Add customer specific routes to specific interfaces
# Example values would be:
#    1.1.1.1_eth1,2.2.2.2_eth2,3.3.3.3_eth3
addCustomerRoutes()
{
    if [ -z $CUSTOM_ROUTES ]; then
        loggerHfe " No custom routes supplied. Continuing"
    else
        IFS=',' read -ra CUSTOMER_ROUTES <<< "$CUSTOM_ROUTES"
        for ROUTE_INFO in "${CUSTOMER_ROUTES[@]}"; do
            DEST_IP="$(echo $ROUTE_INFO | awk -F'_' '{print $1}' | sed 's~/32$~~')"  #IP route will ignore /32. Therefore will fail to find it.
            INTF=${interfaceMap[$(echo $ROUTE_INFO | awk -F'_' '{print $2}')]}
            GW=$($IP route | tr -s ' ' | grep "$INTF scope link" |  awk '{ print $1}')
            CUSTOM_ROUTES_ARR+=( "$DEST_IP""_$GW""_$INTF" ) # Create an array so we don't need to find intf gw everytime
            $IP route | grep -E "$DEST_IP.*$GW.*$INTF"
            if [ $? -ne 0 ]; then
                $IP route add $DEST_IP via $GW dev $INTF
                if [ $? -eq 0 ]; then
                    loggerHfe " Added customer route to $DEST_IP through interface $INTF"
                else
                    errorAndExit "Failed to add custom route $DEST_IP via $GW through interface $INTF"
                fi
            else
                loggerHfe " Route already exists to $DEST_IP through interface $INTF"
            fi
        done
    fi
}

configureMgmtNAT()
{
    loggerHfe " Optional configuration to reach $MGMT using Remote IP. "

    if [ -z "${REMOTE_SSH_MACHINE_IP// }" ]; then
        loggerHfe " No IP is given for REMOTE_SSH_MACHINE_IP field, no route is set for managing this instance over $MGMT"
    else
        loggerHfe " $MGMT is used to manage this HFE instance, we can login using private IP to manage HFE machine without setting default route"
        loggerHfe " default route points to eth0 which will be used to interface all traffic for SBC"

        IFS=',' read -ra SSH_IP_LIST <<< "$REMOTE_SSH_MACHINE_IP"
        for REMOTE_MACHINE_IP in "${SSH_IP_LIST[@]}"; do
            REMOTE_MACHINE_IP="$(echo $REMOTE_MACHINE_IP | sed 's~/32$~~')" # IP route will ignore /32. Therefore will fail to find it.
            $IP route | grep -E "$REMOTE_MACHINE_IP.*$MGMT_GW.*$MGMT"
            if [ "$?" = "0" ]; then
                loggerHfe " Route is already available for remote machine's public IP($REMOTE_MACHINE_IP), from this IP you can SSH to HFE over Remote IP($MGMT)"
            else
                $IP route add $REMOTE_MACHINE_IP via $MGMT_GW dev $MGMT
                if [ "$?" = "0" ]; then
                    loggerHfe " Route added for remote machine's public IP($REMOTE_MACHINE_IP), from this IP you can SSH to HFE over Remote IP($MGMT)"
                else
                    errorAndExit "Failed to add route for ($REMOTE_MACHINE_IP)"
                fi
            fi
        done
    fi

    # Add route to metadata server for mgmt interface to be default
    # Keep default route for redundancy in case mgmt drops
    loggerHfe " Adding metadata server route via $MGMT interface"
    MD_SERVER="169.254.169.254"
    $IP route | grep -E "$MD_SERVER.*$MGMT_GW.*$MGMT"
    if [ $? -ne 0 ]; then
        $IP route add $MD_SERVER via $MGMT_GW dev $MGMT metric 50
        if [ $? -ne 0 ]; then
            errorAndExit "Failed to add metadata server route via $MGMT"
        fi
    fi
}

configurePrimaryIpMap()
{
    if [ $ENABLE_PKT_DNS_QUERY -eq 1 ]; then
        while read line; do
            key=$(echo $line | awk -F: '{print $1}')
            ip=$(echo $line | awk -F: '{print $2}')
            primaryIpMap[$key]=$ip
        done < $PRIMARY_FILE
    fi

    rm -f $PRIMARY_FILE
}

showCurrentConfig()
{
    loggerHfe " Applied iptable rules and kernel routing table. "
    natTable=`$IPTABLES -t nat -vnL`

    loggerHfe " NAT tables:"
    loggerHfe " $natTable "

    filterTable=`$IPTABLES -t filter -vnL`

    loggerHfe " Filter tables:"
    loggerHfe " $filterTable "

    routeOutput=`$IP route`
    loggerHfe " Route:"
    loggerHfe "$routeOutput"
}

getSbcPktIp()
{
    local SBC_NAME=$1
    primaryMapKey="$SBC_NAME"
    if [[ $2 == "PKT0" ]]; then
        NIC="nic2"
    else
        NIC="nic3"
        [[ $MODE == "single" ]] && primaryMapKey="$SBC_NAME""-PKT1"
    fi

    # Get Primary IPs or Alias IPs
    nicMeta=$($GCLOUD compute instances describe $SBC_NAME --format="json" --zone=${zoneMap[$SBC_NAME]} | $JQ ".networkInterfaces[] | select(.name ==\"$NIC\")")
    ipOnSbcPkt="$(echo $nicMeta | $JQ -r .networkIP)"
    echo "$primaryMapKey:$ipOnSbcPkt" >> $PRIMARY_FILE
    if [ $PRIMARY_GCE_IP -ne 1 ]; then
        # Get the Alias ip assigned to Active PKT0 (nic2) interface.
        ipOnSbcPkt="$(echo $nicMeta | $JQ .aliasIpRanges[0] | awk '{print $2}' | awk -F"/" '{print $1}' | awk -F"\"" '{print $2}')"
    fi
    ipOnSbcPkt=${ipOnSbcPkt//$'\n'/}

    if [[ -z "$ipOnSbcPkt" ]];then
        loggerHfe " ERROR: Failed to get IP assigned to $2 on $1."
        return
    fi
    loggerHfe " $SBC_NAME - IP for $2 is $ipOnSbcPkt"
    echo $ipOnSbcPkt
}


getPrimaryHfeEth()
{
    if [[ "$1" == "$ETH1" ]]; then
        intf=1
    else
        intf=0
    fi
    attemptCount=0
    if [ $HA_MODE -eq 1 ] && [[ $SWITCHOVER_TYPE == "private" ]]; then
        url="http://metadata.google.internal/computeMetadata/v1/instance/network-interfaces/$intf/ip-aliases/0"
        attemptMax=90
    else
        url="http://metadata.google.internal/computeMetadata/v1/instance/network-interfaces/$intf/ip"
        attemptMax=1
    fi

    # In HA mode, we might end up being active
    # without alias IP. If that is the case
    # wait to be given IP by standby.
    while [ $attemptCount -lt $attemptMax ]; do
        # Get primary ip assigned on HFE public interface(ens4/eth0) and private interface on (ens5/eth1).
        privIpOnHfeEth=$(curl -f -H "Metadata-Flavor: Google" $url)
        if [ $? -ne 0 ]; then
            if [ $HA_MODE -eq 1 ] && [[ $SWITCHOVER_TYPE == "private" ]]; then
                [ $((attemptCount % 10)) -eq 0 ] && loggerHfe " Waiting to be given alias IP..."
                sleep 2
                attemptCount=$((attemptCount+1))
            fi
        else
            break
        fi
    done
 
    if [[ -z "$privIpOnHfeEth" ]];then
        if [ $HA_MODE -eq 1 ] && [[ $SWITCHOVER_TYPE == "private" ]]; then
            errorAndExit "Switchover: Did not get given alias IPs after 3 minutes from standby. Check alias IPs and reboot HFE to recover"
        else
            errorAndExit " ERROR: Failed to get primary IP assigned on HFE $1 interface."
        fi
    fi

    if [ $HA_MODE -eq 1 ] && [[ $SWITCHOVER_TYPE == "private" ]]; then
        # GCP Ubuntu images do not have all of the alias IPs
        # assigned to the interface. Assign them for consistenancy
        # with Azure / information in logs.
        # TODO: Update logic if we ever support multiple IPs
        if [ $( $IP addr show $ETH0 | grep -c "$privIpOnHfeEth/" ) -eq 0 ]; then
            $IP addr add "$privIpOnHfeEth" dev $ETH0 scope global
        fi
        if [ $($IP addr show dev $ETH0 | grep -ic 'state down') -gt 0 ]; then
            if [ -e $NETPLAN_ROOT ]; then
                netplan apply
                if [ $? -ne 0 ]; then
                    startReboot "Switchover: Failed to bring up $ETH0. Rebooting"
                fi
            fi
        fi
    fi

    # Remove CIDR prefix as not needed for further processing
    privIpOnHfeEth=$(echo $privIpOnHfeEth | awk -F'/' '{print $1}')
    echo "HFE_$1:$privIpOnHfeEth" >> $PRIMARY_FILE
    echo $privIpOnHfeEth
}


setLogRotate()
{
    # Setup HFE logRotate to backup and rotate logs which report status of connection between HFE and Active SBC

    if [ ! -f $hfeLogRotateConf ]; then
        echo -e "$statusLogFile" >> $hfeLogRotateConf
        echo -e "{" >> $hfeLogRotateConf
        echo -e "    rotate 4" >> $hfeLogRotateConf
        echo -e "    size=250M" >> $hfeLogRotateConf
        echo -e "    missingok" >> $hfeLogRotateConf
        echo -e "    notifempty" >> $hfeLogRotateConf
        echo -e "    dateformat -%d%m%Y" >> $hfeLogRotateConf
        echo -e "    compress" >> $hfeLogRotateConf
        echo -e "}" >> $hfeLogRotateConf
    fi
}



extractHfeConfigIps()
{
    # Get PKT interface
    if [[ $MODE == "single" ]]; then
        ACTIVE_SBC_IP=$(getSbcPktIp $ACTIVE_SBC_VM_NAME "PKT0")
        STANDBY_SBC_IP=$(getSbcPktIp $STANDBY_SBC_VM_NAME "PKT0")
        PRIMARY_IP_HFE_ETH0=$(getPrimaryHfeEth $ETH0)

        ACTIVE_SBC_IP_PKT1=$(getSbcPktIp $ACTIVE_SBC_VM_NAME "PKT1")
        STANDBY_SBC_IP_PKT1=$(getSbcPktIp $STANDBY_SBC_VM_NAME "PKT1")
        PRIMARY_IP_HFE_ETH1=$(getPrimaryHfeEth $ETH1)
    else
        ACTIVE_SBC_IP=$(getSbcPktIp $ACTIVE_SBC_VM_NAME $SBC_PKT_PORT_NAME)
        STANDBY_SBC_IP=$(getSbcPktIp $STANDBY_SBC_VM_NAME $SBC_PKT_PORT_NAME)
        PRIMARY_IP_HFE_ETH0=$(getPrimaryHfeEth $ETH0)
    fi

    if [[ $ACTIVE_SBC_IP == "" ]] || [[ $STANDBY_SBC_IP == "" ]]; then
        errorAndExit "Missing SBC IPs. Exiting! Verify SBCs have been created and reboot HFE instance" 
    fi
    if [[ $PRIMARY_IP_HFE_ETH0 == "" ]]; then
        errorAndExit "Failed to get primary IP on $ETH0. Exiting!"
    fi
 
    if [[ $MODE == "single" ]]; then
        if [[ $ACTIVE_SBC_IP_PKT1 == "" ]] || [[ $STANDBY_SBC_IP_PKT1 == "" ]]; then
            errorAndExit "Missing SBC PKT1 IPs. Exiting! Verify SBCs have been created and reboot HFE instance"
        fi
        if [[ $PRIMARY_IP_HFE_ETH1 == "" ]]; then
            errorAndExit "Failed to get primary IP on $ETH1. Exiting!"
        fi
    fi
}

# Create a sysdump of common files needed by Ribbon
# for debugging HFE failures in GCP
sysdump()
{
    if [ ! -e $HFERoot/sysdump ]; then
        mkdir $HFERoot/sysdump
    fi

    if [ -e $TMP_LOG_DIR ]; then
        rm -rf $TMP_LOG_DIR
    fi
    mkdir $TMP_LOG_DIR

    $IP a > $TMP_LOG_DIR/interfaces.out
    $IP r > $TMP_LOG_DIR/routes.out
    $IPTABLES -vnL > $TMP_LOG_DIR/iptables.out
    echo -E "\n\n\n\n" >> $TMP_LOG_DIR/iptables.out
    $IPTABLES -t nat -vnL >> $TMP_LOG_DIR/iptables.out
    dmesg > $TMP_LOG_DIR/dmesg.out
    $CONNTRACK -L -o extended > $TMP_LOG_DIR/conntrack.out
    $CONNTRACK -C > $TMP_LOG_DIR/conntrack.count.out
    curl "http://metadata.google.internal/computeMetadata/v1/?recursive=true&alt=json" -H "Metadata-Flavor: Google" | $JQ . > $TMP_LOG_DIR/metadata.json
    curl "http://metadata.google.internal/computeMetadata/v1/instance/attributes/startup-script" -H "Metadata-Flavor: Google" > $TMP_LOG_DIR/startup_script
 
    journalctl -xe > $TMP_LOG_DIR/journalctl.out
    journalctl -t dhclient > $TMP_LOG_DIR/dhclient.log
    journalctl -t systemd-networkd > $TMP_LOG_DIR/system-networkd.log

    cp /var/log/syslog $TMP_LOG_DIR/
    cp /var/log/messages $TMP_LOG_DIR/
    cp /var/log/dpkg.log $TMP_LOG_DIR/
    cp /var/log/cloud-init* $TMP_LOG_DIR/
    cp /var/lib/cloud/instance/user-data.txt* $TMP_LOG_DIR/

    mkdir $TMP_LOG_DIR/bash_history/
    ls /home/*/.bash_history /root/.bash_history | while read hist; do
        cp $hist "$TMP_LOG_DIR/bash_history/$(echo $hist | awk -F/ '{print $(NF-1)}')"
    done

    cp -r $HFERoot $TMP_LOG_DIR/
    rm -fr $TMP_LOG_DIR/HFE/sysdump/     #Remove previous sysdumps

    fileName="HFE_sysdump_$(hostname)_$(date +%y-%m-%dT%H-%M-%S).tar.gz"
    tar -zcvf $fileName -C $TMP_LOG_DIR .
    mv $fileName $HFERoot/sysdump/
    rm -rf $TMP_LOG_DIR
    echo "SYSDUMP available at $HFERoot/sysdump/$fileName"
    echo "Remove any old sysdumps to preserve disk space!"
}


# Run the gcloud commands
# to move the alias IP between
# the HFE nodes.
haPerformPrivateSwitchover()
{
    local sourceVmName="$1"
    local destVmName="$2"
    local aliasIp="$3"
    local errorExit=$4
    local noDetach=$5

    local attempt=0

    if [ $noDetach -eq 0 ]; then
        while [ $attempt -le $HA_SWITCHOVER_RETRY ]; do
            resp=$(timeout $CMD_TIMEOUT $GCLOUD compute instances network-interfaces update $sourceVmName --network-interface=nic0 --aliases="" --zone=${zoneMap[$sourceVmName]} 2>&1)
            retCode=$?
            if [ $retCode -eq $CMD_TIMEOUT_CODE ]; then
                loggerHfe "   Had to terminate gcloud command"
                if [[ $(haGetHfeInfoJson $sourceVmName | $JQ -r '.eth0.aliasIps') == "null" ]]; then
                    # Command was actually a success despite timeout
                    retCode=0
                fi
            fi
            if [ $retCode -ne 0 ]; then
                loggerHfe "  Got error response from GCP: $resp"
                if [ $attempt -eq $HA_SWITCHOVER_RETRY ]; then
                    if [ $errorExit -ne 0 ]; then
                        errorAndExit "Switchover: Failed to remove alias IP from $sourceVmName"
                    else
                        return 1
                    fi
                fi
                loggerHfe "Switchover: Failed to remove alias IP from $sourceVmName. Retrying"
                attempt=$((attempt+1))
            else
                break
            fi
        done
        loggerHfe "Switchover: Successfully removed alias IP ($aliasIp) from $sourceVmName"
    fi

    attempt=0
    loggerHfe "Switchover: Adding alias IP ($aliasIp) to $destVmName"
    while [ $attempt -le $HA_SWITCHOVER_RETRY ]; do
        resp=$(timeout $CMD_TIMEOUT $GCLOUD compute instances network-interfaces update $destVmName --network-interface=nic0 --aliases="$aliasIp" --zone=${zoneMap[$destVmName]} 2>&1)
        retCode=$?
        if [ $retCode -eq $CMD_TIMEOUT_CODE ]; then
            loggerHfe "   Had to terminate gcloud command"
            if [[ $(haGetHfeInfoJson $destVmName | $JQ -r '.eth0.aliasIps') != "null" ]]; then
               # Command was actually a success despite timeout
                retCode=0
            fi
        fi
        if [ $retCode -ne 0 ]; then
            loggerHfe "   Got error response from GCP: $resp"
            if [ $attempt -eq $HA_SWITCHOVER_RETRY ]; then
                if [ $errorExit -ne 0 ]; then
                    errorAndExit "Switchover: Failed to attach IP ($aliasIp) to $destVmName"
                else
                    return 1
                fi
            fi
            loggerHfe "Switchover: Failed to attach IP ($aliasIp) to $destVmName. Retrying"
            attempt=$((attempt+1))
        else
            break
        fi
    done

    loggerHfe "Switchover: Successfully attached alias IP ($aliasIp) to $destVmName"
    return 0
}

# Runs the gcloud requests
# to move the public IP between
# the HFE nodes.
haPerformPublicSwitchover()
{
    local sourceVmName="$1"
    local destVmName="$2"
    local publicIpCfg="$3"   # ACCESS_CONFIG_NAME;$IP;$NETWORK_TIER
    local errorExit=$4
    local noDetach=$5

    local accessName=$(echo $publicIpCfg | awk -F';' '{print $1}') 
    local publicIp=$(echo $publicIpCfg | awk -F';' '{print $2}')
    local networkTier=$(echo $publicIpCfg | awk -F';' '{print $3}')
    local attempt=0

    if [ $noDetach -eq 0 ]; then
        loggerHfe "Switchover: Detaching public IP ($publicIp) from $sourceVmName"
        while [ $attempt -le $HA_SWITCHOVER_RETRY ]; do
            #Full Public Switchover
            resp=$(timeout $CMD_TIMEOUT $GCLOUD compute instances delete-access-config $sourceVmName --zone=${zoneMap[$sourceVmName]} --network-interface=nic0 --access-config-name=$accessName 2>&1)
            retCode=$?
            if [ $retCode -eq $CMD_TIMEOUT_CODE ]; then
                loggerHfe "   Had to terminate gcloud command"
                if [[ $(haGetHfeInfoJson $sourceVmName | $JQ -r '.eth0.publicIp') == "null" ]]; then
                    # Command was actually a success despite timeout
                    retCode=0
                fi
            fi
            if [ $retCode -ne 0 ]; then
                loggerHfe "   Got bad response from GCP: $resp"
                if [ $attempt -eq $HA_SWITCHOVER_RETRY ]; then
                    if [ $errorExit -ne 0 ]; then
                        errorAndExit "Switchover: Failed to remove public IP ($publicIp) from $sourceVmName"
                    else
                        return 1
                    fi
                fi
                loggerHfe "Switchover: Failed to remove public IP ($publicIp) from $sourceVmName. Retrying"
                attempt=$((attempt+1))
            else
                break
            fi
        done
        loggerHfe "Switchover: Successfully removed public IP ($publicIp) from $sourceVmName."
    fi

    attempt=0
    loggerHfe "Switchover: Attaching public IP ($publicIp) to $destVmName"
    while [ $attempt -le $HA_SWITCHOVER_RETRY ]; do
        resp=$(timeout $CMD_TIMEOUT $GCLOUD compute instances add-access-config $destVmName --zone=${zoneMap[$destVmName]} --access-config-name=$accessName --address=$publicIp --network-interface=nic0 --network-tier=$networkTier 2>&1)
        retCode=$?
        if [ $retCode -eq $CMD_TIMEOUT_CODE ]; then
            loggerHfe "   Had to terminate gcloud command"
            if [[ $(haGetHfeInfoJson $destVmName | $JQ -r '.eth0.publicIp') != "null" ]]; then
               # Command was actually a success despite timeout
                retCode=0
            fi
        fi
        if [ $retCode -ne 0 ]; then
            loggerHfe "   Got bad response from GCP:  $resp"
            if [ $attempt -eq $HA_SWITCHOVER_RETRY ]; then
                if [ $errorExit -ne 0 ]; then
                    errorAndExit "Switchover: Failed to attach public IP ($publicIp) to $destVmName"
                else
                    return 1
                fi
            fi
            loggerHfe "Switchover: Failed to attach public IP ($publicIp) to $destVmName. Retrying"
            attempt=$((attempt+1))
        else
            break
        fi
    done
    loggerHfe "Switchover: Successfully attached public IP ($publicIp) to $destVmName"
    return 0
}

# Util function to return
# an IP address instead of
# performing full switchover.
# Used when standby has the IPs or
# if neither ce has IPs
haReturnIpAddress()
{
    local sourceVmName="$1"
    local destVmName="$2"
    local targetIp="$3"
    local noDetach=$4

    if [ $noDetach -eq 0 ]; then
        actionType="return IP address to active"
    else
        actionType="recover IP address from GCP"
    fi

    loggerHfe "Switchover: Attempting to $actionType"

    # Note: No way to check if interface is being updated
    # We will instead fail and retry

    if [[ $SWITCHOVER_TYPE == "public" ]]; then
        haPerformPublicSwitchover "$sourceVmName" "$destVmName" "$targetIp" 0 $noDetach
        [ $? -ne 0 ] && loggerHfe "Switchover: Failed to perform public switchover to give IP to $destVmName. Will try again next time" && return 1
    else
        haPerformPrivateSwitchover "$sourceVmName" "$destVmName" "$targetIp" 0 $noDetach
        [ $? -ne 0 ] && loggerHfe "Switchover: Failed to perform private switchover to give IP to $destVmName. Will try again next time" && return 1
    fi
    return 0
}


# Util function to get the relevant HFE info.
# Return in a JSON string format for processing.
# For switchover logic to check who is active
haGetHfeInfoJson()
{
    local targetHfe="$1"
    local simpleHfeJson=""

    local ipArr=( )
    local aliasArr=( )
    local pubArr=( )
 
    networkIntfJson=$($GCLOUD compute instances describe $targetHfe --zone=${zoneMap[$targetHfe]} --format="json(networkInterfaces)" | $JQ .networkInterfaces)
    interfaceNum=$(echo $networkIntfJson | $JQ length)
    for i in $(seq 0 $((interfaceNum-1))); do
        ipArr[$i]=$(echo $networkIntfJson | $JQ -r --arg NIC "nic$i" '.[] | select(.name==$NIC) | .networkIP')
        if [[ $(echo $networkIntfJson | $JQ -r --arg NIC "nic$i" '.[] | select(.name==$NIC) | .aliasIpRanges') == "null" ]]; then
            aliasArr[$i]="null"
        else
            aliasArr[$i]=$(echo $networkIntfJson | $JQ -r --arg NIC "nic$i" '.[] | select(.name==$NIC) | .aliasIpRanges[].ipCidrRange' | tr '\n' ';' | sed 's/;$//')
        fi
        if [[ $(echo $networkIntfJson | $JQ -r --arg NIC "nic$i" '.[] | select(.name==$NIC) | .accessConfigs') == "null" ]]; then
            pubArr[$i]="null"
        else
            pubArr[$i]=$(echo $networkIntfJson | $JQ -r --arg NIC "nic$i" '.[] | select(.name==$NIC) | .accessConfigs[] | .name, .natIP, .networkTier' | tr '\n' ';' | sed 's/;$//')
        fi
    done

    simpleHfeJson=$( $JQ -n \
      --arg HFENAME "$targetHfe" \
      --arg ETH0IP "${ipArr[0]}" \
      --arg ETH0ALIAS "${aliasArr[0]}" \
      --arg ETH0PUBIP "${pubArr[0]}" \
      --arg ETH1IP "${ipArr[1]}" \
      --arg ETH1ALIAS "${aliasArr[1]}" \
      --arg ETH1PUBIP "${pubArr[1]}" \
      --arg ETH2IP "${ipArr[2]}" \
      --arg ETH2ALIAS "${aliasArr[2]}" \
      --arg ETH2PUBIP "${pubArr[2]}" \
    '{
    "eth0": {
        "privateIp": $ETH0IP,
        "aliasIps": $ETH0ALIAS,
        "publicIp": $ETH0PUBIP
    },
    "eth1": {
        "privateIp": $ETH1IP,
        "aliasIps": $ETH1ALIAS,
        "publicIp": $ETH1PUBIP
    },
    "eth2": {
        "privateIp": $ETH2IP,
        "aliasIps": $ETH2ALIAS,
        "publicIp": $ETH2PUBIP
    },
    "name": $HFENAME
}')
 
    echo $simpleHfeJson
}

# Main standby function which performs switchover logic. 
# Will do the following:
#   1. Check if we are in HA mode
#   2. Gather information about the GCP routes, local IPs and peer IPs
#   3. If the GCP routes are pointing to us, we are active. Exit function and continue
#   4. Conitually monitor the active HFE:
#     i. Check we do not have the IPs. If so return it
#    ii. Check maintenance status of active. Perform switchover SWITCHOVER_MODE is FULL
#   iii. Use fping to healthcheck the active.
#  5. Perform switchover logic:
#     i. Update the GCP routes to use our eth2 IP address
#    ii. Move IP address to our eth0
#   iii. Update the private IP at linux level if private switchover
haCheckIfStandby()
{
    if [[ $PEER_HFE_NAME == "" ]]; then
        loggerHfe "Switchover: Natvar PEER_HFE_NAME not configured. Not in HA mode"
        return
    elif [[ $ROUTE_NAME == "" ]]; then
        loggerHfe "Switchover: Natvar ROUTE_NAME not configured. Not in HA mode"
        return
    fi

    [ -z $SWITCHOVER_MODE ] && loggerHfe "Switchover: NatVar SWITCHOVER_MODE not set. Assuming FULL"

    HA_MODE=1
    SWITCHOVER_TYPE="private"
    loggerHfe "Switchover: We are paired with VM $PEER_HFE_NAME. Checking who is active"
 
    localCeJson="$(haGetHfeInfoJson $HFE_NAME)"
    [ -z "$localCeJson" ] && errorAndExit "Switchover: Failed to get info on HFE $HFE_NAME"
    peerCeJson="$(haGetHfeInfoJson $PEER_HFE_NAME)"
    [ -z "$peerCeJson" ] && errorAndExit "Switchover: Failed to get info on HFE $PEER_HFE_NAME"

    # Get some values immediately so that we can apply them if they get removed
    # If active: apply to self or apply to standby if we are about to reboot
    # If standby: apply when switching over
    retry=0
    initRoute=""
    ## Retry a couple of times. Force reboot can lead to issues with snap and gcloud"
    while [ -z "$initRoute" ] && [ $retry -le 6 ]; do
        initRoute="$($GCLOUD compute routes describe $ROUTE_NAME --format='json')"
        if [ -z "$initRoute" ]; then 
            ((retry++))
            loggerHfe "Switchover: Attempt $retry. Route returned empty. Assume issue with gcloud and snap. Retrying in 5..." 
            sleep 5
        fi
    done

    peerPubIpCfg="$(echo $peerCeJson | $JQ -r '.eth0.publicIp')"
    localPubIpCfg="$(echo $localCeJson | $JQ -r '.eth0.publicIp')"
    if [[ $peerPubIpCfg != "null" ]]; then
        SWITCHOVER_TYPE="public"
        initPubIpCfg=$peerPubIpCfg
    elif [[ $localPubIpCfg != "null" ]]; then
        SWITCHOVER_TYPE="public"
        initPubIpCfg=$localPubIpCfg
    fi

    if [[ $(echo $peerCeJson | $JQ -r '.eth0.aliasIps') != "null" ]]; then
        initAliasIps=$(echo $peerCeJson | $JQ -r '.eth0.aliasIps')
    elif [[ $(echo $localCeJson | $JQ -r '.eth0.aliasIps') != "null" ]]; then
        initAliasIps=$(echo $localCeJson | $JQ -r '.eth0.aliasIps')
    fi
    loggerHfe "Switchover: Got the following initial values: SwitchoverType=$SWITCHOVER_TYPE, PublicIP=$initPubIpCfg, AliasIP=$initAliasIps, Route=$initRoute"


    # eth2 IPs for checking the Azure routes
    localEth2=$(echo $localCeJson | $JQ -r '.eth2.privateIp')
    peerEth2=$(echo $peerCeJson | $JQ -r '.eth2.privateIp')


    while true; do
        route="$($GCLOUD compute routes describe $ROUTE_NAME --format='json')"

        localCount=$(echo "$route" | grep -cw $localEth2) 
        peerCount=$(echo "$route" | grep -cw $peerEth2)
        if [ $localCount -gt 0 ] && [ $peerCount -gt 0 ]; then
            errorAndExit "Switchover: GCP route contains nextHopIp for both peer HFE nodes! Unable to determine active. Exiting!"
        elif [ $localCount -eq 0 ] && [ $peerCount -eq 0 ]; then
            loggerHfe "Switchover:  Route does not contain eth2 IPs for either peer HFE nodes. Waiting 1 second and retrying"
            sleep 1
            continue
        elif [ $localCount -gt 0 ]; then
            loggerHfe "Switchover: We are the active HFE node. Continuing to startup"
            break
        elif [ $peerCount -gt 0 ]; then
            loggerHfe "Switchover: We are the standby HFE node. Starting monitoring of active HFE."
            hfeMgt=$(echo $peerCeJson | $JQ -r '.eth1.privateIp')
            triggerSwitchover=0
            checkRoute=1
            checkIPs=1
            waitCount=0

            # Add mgmt routes now so we can get on the standby HFE
            setInterfaceMap
            configureMgmtNAT

            while [ $triggerSwitchover -eq 0 ]; do
                if [ $checkIPs -eq 1 ]; then
                    checkIPs=0
                    checkState=0    # Do we need to check the state of the interfaces
                    localCeJson="$(haGetHfeInfoJson $HFE_NAME)"
                    if [ -z "$localCeJson" ]; then
                        loggerHfe "Switchover: Failed to get local HFE information. Will try again next time"
                        checkIPs=1
                    else
                        if [[ $SWITCHOVER_TYPE == "public" ]]; then
                            targetIp="$(echo $localCeJson | $JQ -r '.eth0.publicIp')"
                            if [[ $targetIp != "null" ]]; then
                                checkState=1
                                loggerHfe "Switchover: Detected we have the public IP. Attempting to return it to active"
                            fi
                        else ##Check private
                            targetIp=$(echo $localCeJson | $JQ -r '.eth0.aliasIps')
                            if [[ $targetIp != "null" ]]; then
                                checkState=1
                                loggerHfe "Switchover: Detected we have the alias IPs ($targetIp). Attempting to return to them to active"
                            fi 
                        fi
                        if [ $checkState -eq 1 ]; then    # We have the IPs but are not active. Give them back
                            removeIp=0
                            haReturnIpAddress "$HFE_NAME" "$PEER_HFE_NAME" "$targetIp" 0
                            case $? in
                                1) checkIPs=1 ;;
                                2) waitCount=50 ;;
                                *) removeIp=1 ;;
                            esac
                            if [ $removeIp -eq 1 ]; then
                                for cidr in $(echo $targetIp | tr ',' ' '); do
                                    if [ $($IP addr show dev $ETH0 | grep -c "$cidr/") -gt 0 ]; then
                                        #Delete the IP locally
                                        $IP addr del $cidr dev $ETH0
                                        if [ $($IP addr show dev $ETH0 | grep -ic 'state down') -gt 0 ]; then
                                            if [ -e $NETPLAN_ROOT ]; then
                                                netplan apply
                                                if [ $? -ne 0 ]; then
                                                    startReboot "Switchover: Failed to bring up eth0. Rebooting"
                                                fi
                                            fi
                                        fi
                                    fi
                                done
                            fi
                        fi
                    fi
                fi

                if [ $checkRoute -eq 1 ]; then
                    checkRoute=0
                    # In GCP the route update takes 2 actions to change.
                    # If the route goes missing, return it to the init
                    # value
                    $GCLOUD compute routes describe $ROUTE_NAME
                    if [ $? -ne 0 ]; then
                        loggerHfe "Switchover: Deteced no GCP route exists. Re-adding the initial route"
                        # Route does not exit. Add it 
                        routeInfoMap["desc"]=$(echo $initRoute | $JQ -r '.description')
                        routeInfoMap["dest"]=$(echo $initRoute | $JQ -r '.destRange')
                        routeInfoMap["network"]=$(basename $(echo $initRoute | $JQ -r '.network'))
                        routeInfoMap["priority"]=$(echo $initRoute | $JQ -r '.priority')
                        routeInfoMap["tags"]=$(echo $initRoute | $JQ -r '.tags[]' | tr '\n' ',' | sed 's/,$//')
                        routeInfoMap["hop"]=$(echo $initRoute | $JQ -r '.nextHopIp')
                        timeout $CMD_TIMEOUT $GCLOUD compute routes create $ROUTE_NAME --destination-range=${routeInfoMap['dest']} --next-hop-address=${routeInfoMap['hop']} --description=${routeInfoMap['desc']} --network=${routeInfoMap['network']} --priority=${routeInfoMap['priority']} --tags="${routeInfoMap['tags']}"
                        if [ $? -ne 0 ]; then
                            loggerHfe "Switchover: Failed to re-add initial route. Will retry next time"
                            checkRoute=1
                        fi
                    fi
                fi

                # Check the active is up
                $FPING -c 3 -t 100 -p 200 $hfeMgt
                if [ $? -ne 0 ]; then
                    msg="Switchover: $PEER_HFE_NAME failed to respond in time."
                    if [[ $SWITCHOVER_MODE != "NONE" ]]; then
                        msg="$msg Performing Switchover"
                        triggerSwitchover=1
                    fi
                    loggerHfe "$msg"
                else
                    waitCount=$((waitCount+1))
                    sleep 0.5s
                    if [ $waitCount -eq 60 ]; then
                         checkIPs=1 
                         checkRoute=1
                         waitCount=0
                    fi
                fi
            done

            # Re-get route so we have the latest
            route="$($GCLOUD compute routes describe $ROUTE_NAME --format='json')"
            delRoute=1
            if [[ -z "$route" ]]; then
                route="$initRoute" 
                delRoute=0  # Route does not exist, use initial
                loggerHfe "Switchover: GCP route $ROUTE_NAME not found. Using initial value"
            fi
            routeInfoMap["desc"]=$(echo $route | $JQ -r '.description')
            routeInfoMap["dest"]=$(echo $route | $JQ -r '.destRange')
            routeInfoMap["network"]=$(basename $(echo $route | $JQ -r '.network'))
            routeInfoMap["priority"]=$(echo $route | $JQ -r '.priority')
            routeInfoMap["tags"]=$(echo $route | $JQ -r '.tags[]' | tr '\n' ',' | sed 's/,$//')

            ##Update the routes
            loggerHfe "Switchover: Updating GCP route to new HFE"

            attempt=0
            while [ $attempt -le $HA_SWITCHOVER_RETRY ]; do
                if [ $delRoute -eq 1 ]; then
                    resp=$(timeout $CMD_TIMEOUT $GCLOUD --quiet compute routes delete $ROUTE_NAME 2>&1)
                    retCode=$?
                    if [ $retCode -eq $CMD_TIMEOUT_CODE ]; then
                        loggerHfe "   Had to terminate gcloud command"
                        if [[ $($GCLOUD compute routes describe $ROUTE_NAME --format='json') == "" ]]; then
                            # Command was actually a success despite timeout
                            retCode=0
                        fi
                    fi
                    if [ $retCode -ne 0 ]; then
                        if [ $routeAttempt -eq $routeAttemptMax ]; then
                            loggerHfe "   Got bad response from GCP: $resp"
                            errorAndExit "Switchover: Failed to delete route $ROUTE_NAME"
                        fi
                        loggerHfe "Switchover: Failed to delete route. Retrying"
                    else
                        loggerHfe "Switchover: Deleted route $ROUTE_NAME"
                        delRoute=0 
                    fi
                fi
                if [ $delRoute -eq 0 ]; then
                    resp=$(timeout $CMD_TIMEOUT $GCLOUD compute routes create $ROUTE_NAME --destination-range=${routeInfoMap['dest']} --next-hop-address=$localEth2 --description=${routeInfoMap['desc']} --network=${routeInfoMap['network']} --priority=${routeInfoMap['priority']} --tags="${routeInfoMap['tags']}" 2>&1)
                    retCode=$?
                    if [ $retCode -eq $CMD_TIMEOUT_CODE ]; then
                        loggerHfe "   Had to terminate gcloud command"
                        if [[ $($GCLOUD compute routes describe $ROUTE_NAME --format='json') != "" ]]; then
                            # Command was actually a success despite timeout
                            retCode=0
                        fi
                    fi
                    if [ $retCode -ne 0 ]; then
                        if [ $routeAttempt -eq $routeAttemptMax ]; then
                            loggerHfe "   Got bad response from GCP: $resp"
                            errorAndExit "Switchover: Failed to add route $ROUTE_NAME"
                        fi
                        loggerHfe "Switchover: Failed to add route. Retrying"
                    else
                        loggerHfe "Switchover: Added route $ROUTE_NAME with new next hop IP"
                        break
                    fi
                fi
                attempt=$((attempt+1))
            done


            # Make sure we have the latest information 
            localCeJson="$(haGetHfeInfoJson $HFE_NAME)"
            [ -z "$localCeJson" ] && errorAndExit "Switchover: Failed to get info on HFE $HFE_NAME"
            peerCeJson="$(haGetHfeInfoJson $PEER_HFE_NAME)"
            [ -z "$peerCeJson" ] && errorAndExit "Switchover: Failed to get info on HFE $PEER_HFE_NAME"

            peerPubIpCfg="$(echo $peerCeJson | $JQ -r '.eth0.publicIp')"
            localPubIpCfg="$(echo $localCeJson | $JQ -r '.eth0.publicIp')"
 
            if [[ $SWITCHOVER_TYPE == "public" ]]; then 
                if [[ $peerPubIpCfg != "null" ]]; then
                    haPerformPublicSwitchover "$PEER_HFE_NAME" "$HFE_NAME" "$peerPubIpCfg" 1 0
                elif [[ $localPubIpCfg != "null" ]]; then
                    # Just update the routetables
                    loggerHfe "Switchover: We are standby but already have public IP. Just updating the GCP routes"
                else
                    loggerHfe "Switchover: Neither HFE nodes have a public IP. Attempting to attach last know public IP config $initPubIpCfg"
                    haPerformPublicSwitchover "$PEER_HFE_NAME" "$HFE_NAME" "$initPubIpCfg" 1 1
                fi
            else
                loggerHfe "Switchover: Private IP switchover" 
                peerAliasIps=$(echo $peerCeJson | $JQ -r '.eth0.aliasIps')
                localAliasIps=$(echo $localCeJson | $JQ -r '.eth0.aliasIps')
                localPrimaryIp=$(echo $localCeJson | $JQ -r '.eth0.privateIp')

                if [[ $peerAliasIps != "null" ]]; then
                    haPerformPrivateSwitchover "$PEER_HFE_NAME" "$HFE_NAME" "$peerAliasIps" 1 0
                elif [[ $localAliasIps != "null" ]]; then
                    loggerHfe "Switchover: We are standby but already have alias IP(s). Just updating the GCP routes"
                else
                    loggerHfe "Switchover: Neither HFE nodes have an alias IP address. Attempting to use last known alias IP $initAliasIps"
                    haPerformPrivateSwitchover "$PEER_HFE_NAME" "$HFE_NAME" "$initAliasIps" 1 1
                    peerAliasIps="$initAliasIps"
                fi
            fi

            ipMoved=0
            routeUpdated=0
            failTimer=0 #Give 1.5 minute for switchover..
            while [ $ipMoved -ne 1 ] && [ $routeUpdated -ne 1 ] && [ $failTimer -lt 90 ]; do
                failTimer=$((failTimer+2))
                if [ $routeUpdated -ne 1 ]; then
                    if [[ "$($GCLOUD compute routes describe $ROUTE_NAME --format='value(nextHopIp)')" == "" ]]; then
                        loggerHfe "Switchover: Some routes still going via peer's eth2. Waiting 2 seconds and retrying"
                        sleep 2
                        continue
                    else
                        loggerHfe "Switchover: Routes are successively updated to our eth2"
                        routeUpdated=1
                    fi
                fi

                if [ $ipMoved -ne 1 ]; then
                    updatedlocalCeJson="$(haGetHfeInfoJson $HFE_NAME)"
                    if [[ $SWITCHOVER_TYPE == "public" ]]; then
                        if [[ $(echo $updatedlocalCeJson | $JQ -r '.eth0.publicIp') == "null" ]]; then
                            loggerHfe "Switchover: No public IP found attached to eth0 primary config. Waiting 2 seconds and retrying"
                            sleep 2
                            continue
                        else
                            loggerHfe "Switchover: Public IP successively attached to eth0"
                            ipMoved=1
                        fi
                    else
                        ##Private
                        if [[ $(echo $updatedlocalCeJson | $JQ -r '.eth0.aliasIps') != "$peerAliasIps" ]]; then
                            loggerHfe "Switchover: The alias IP(s) have not been assigned to eth0 yet. Waiting 2 seconds and retrying"
                            sleep 2
                            continue
                        else
                            loggerHfe "Switchover: Alias IP(s) successively attached to eth0"
                            ipMoved=1

                            loggerHfe "Switchover: Adding new IP to $ETH0 as secondary"
                            for cidr in $(echo $peerAliasIps | tr ';' ' '); do
                                $IP addr add $cidr dev $ETH0 scope global
                                if [ $($IP addr show dev $ETH0 | grep -ic 'state down') -gt 0 ]; then
                                    if [ -e $NETPLAN_ROOT ]; then
                                        netplan apply
                                        if [ $? -ne 0 ]; then
                                            startReboot "Switchover: Failed to bring up eth0. Rebooting"
                                        fi
                                    fi
                                fi
 
                                if [ $($IP addr show $ETH0 | grep -c $cidr) -eq 0 ]; then
                                    errorAndExit "Switchover: New IP address $cidr not added $ETH0. Exiting!"
                                fi
                            done
                        fi
                    fi
                fi
            done

            if [ $failTimer -eq 90 ]; then
                loggerHfe "Switchover: Switchover mechanism failed! Allowing HFE to re-check if old active is up"
                continue
            fi

            loggerHfe "Switchover: Successful switchover to active. Starting HFE setup."
            break
        else
            errorAndExit "Switchover: Value returned for number of IPs in route is less than zero! localCount=$localCount peerCount=$peerCount. Exiting!"
        fi
    done
}

# Fuction called from checkSbcConnectivity
# to check if we are still active or if 
# we need to return the IPs back to us.
# Note: This is run as a background process
#       as gcloud requests are taking too long 
#       and interuppting the SBC healthcheck
haCheckIfStillActive()
{
    statusLoggerHfeDebug "Checking if we are still active CE and if IPs are attached"
    recoverIp=""

    for attempt in 1 2; do

        nextHopIp="$(timeout 15s $GCLOUD compute routes describe $ROUTE_NAME --format='value(nextHopIp)')"
        ret=$?
        if [ $ret -eq $CMD_TIMEOUT_CODE ]; then
            # Timeout means GCP has not responsed to simple get request
            if [ $attempt -eq 2 ]; then
                startReboot "Switchover: Timeout when trying to get route ($ROUTE_NAME) information. This is attempt $attempt. Assuming no traffic can get to $ETH0. Rebooting to switchover!"
            else
                loggerHfe "Switchover: Timeout when trying to get route ($ROUTE_NAME) information. This is attempt $attempt. Will check again next time."
                continue
            fi
        elif [ $ret -ne 0 ]; then
            # Route creation in GCP is slow. Wait for us to get the actual IP
            loggerHfe "Switchover: Failed to get route ($ROUTE_NAME) information. Assuming some change is occurring. Will check again next time"
            continue
        fi
        localCeJson="$(haGetHfeInfoJson $HFE_NAME)"
        if [ -z "$localCeJson" ]; then
            statusLoggerHfe "Request timed out getting local CE information. Retrying"
            continue
        fi
        peerCeJson="$(haGetHfeInfoJson $PEER_HFE_NAME)"
        if [ -z "$peerCeJson" ]; then
            statusLoggerHfe "Request timed out getting peer CE information. Retrying"
            continue
        fi
        if [[ "$SWITCHOVER_TYPE" == "public" ]]; then
            if [[ $(echo "$localCeJson" | $JQ -r '.eth0.publicIp') == "null" ]] && [[ $(echo "$peerCeJson" |  $JQ -r '.eth0.publicIp') == "null" ]]; then
                # Neither of us have public IPs. Try to get it back from initPubIpCfg
                recoverIp="$initPubIpCfg"
            fi
        else
            if [[ $(echo "$localCeJson" | $JQ -r '.eth0.aliasIps') == "null" ]] && [[ $(echo "$peerCeJson" | $JQ -r '.eth0.aliasIps') == "null" ]]; then
                # Neither of us have alias IPs. Try to get it back from initAliasIps
                recoverIp="$initAliasIps"
            fi
        fi

        eth2ip=$($IP addr show $ETH2 | grep -w inet | awk '{print $2}')
        if [ $(echo $nextHopIp | grep -c '/') -eq 0 ]; then
            # nextHop is not in CIDR format, remove our prefix
            eth2ip=$(echo $eth2ip | awk -F'/' '{print $1}')
        fi
        if [[ $nextHopIp != $eth2ip ]]; then
            statusLoggerHfe "Detected the GCP route is not using our IP address. Rebooting to become standby"
            loggerHfe "Switchover: Detected the GCP route is not using our IP address. Rebooting to become standby"
            if [ ! -z "$recoverIp" ]; then
                loggerHfe "Switchover: No IPs found for either CE. Attempting to recover IPs and give to peer before reboot"
                haReturnIpAddress "$HFE_NAME" "$PEER_HFE_NAME" "$recoverIp" 1
                if [ $? -ne 0 ] && [ $attempt -eq 1 ]; then
                    loggerHfe "Switchover: Failed to recover IPs and give to peer. Will try again before rebooting"
                else
                    startReboot
                fi
            else 
                startReboot
            fi
        else
            if [ ! -z "$recoverIp" ] ; then
                # We are active but no-one has the IPs. Recover them to us
                haReturnIpAddress "$PEER_HFE_NAME" "$HFE_NAME" "$recoverIp" 1
                if [ $? -eq 0 ] && [[ "$SWITCHOVER_TYPE" == "private" ]]; then
                    for cidr in $(echo $peerAliasIps | tr ';' ' '); do
                        # Check if we need to add the address
                        if [ $($IP addr show dev $ETH0 | grep -c "$cidr") -eq 0 ]; then
                            $IP addr add $cidr dev $ETH0 scope global
                            if [ $($IP addr show dev $ETH0 | grep -ic 'state down') -gt 0 ]; then
                                if [ -e $NETPLAN_ROOT ]; then
                                    netplan apply
                                    if [ $? -ne 0 ]; then
                                        startReboot "Switchover: Failed to bring up eth0, Rebooting"
                                    fi
                                fi
                            fi
                        fi
                    done 
                fi
            fi
        fi
    done
}


# Main logic of HFE
checkSbcConnectivity()
{
    local switched=0
    local counter=0
    local verifyCount=0
    local haCheckCount=0

    while :; do
############################################### CAUTION #################################
#
#        -t (value) < -P (value)
#
#########################################################################################
#      -c, --count=N
#
#        Number of request packets to send to each target. In this mode, a line is
#        displayed for each received response (this can suppressed with -q or -Q). Also,
#        statistics about responses for each target are displayed when all requests have
#        been sent (or when interrupted).
#
#
#      -p, --period= MSEC
#
#        In looping or counting modes (-l, -c, or -C), this parameter sets the time in
#        milliseconds that fping waits between successive packets to an individual
#        target. Default is 1000 and minimum is 10.
#
#
#
#      -t, --timeout= MSEC
#
#        Initial target timeout in milliseconds. In the default, non-loop mode,
#        the default timeout is 500ms, and it represents the amount of time that
#        fping waits for a response to its first request. Successive timeouts are
#        multiplied by the backoff factor specified with -B.
#
#        In loop/count mode, the default timeout is automatically adjusted to
#        match the "period" value (but not more than 2000ms). You can still
#        adjust the timeout value with this option, if you wish to, but note that
#        setting a value larger than "period" produces inconsistent results,
#        because the timeout value can be respected only for the last ping.
#
#        Also note that any received replies that are larger than the timeout
#        value, will be discarded.
############################################################################################


        $FPING -c 3 -t 20 -p 30 $ACTIVE_SBC_IP &> /dev/null
        if [ $? -ne 0 ]
        then
            if [ $switched -eq 0 ]; then
                statusLoggerHfe "Connection error detected to Active SBC $ACTIVE_SBC_VM_NAME. $ACTIVE_SBC_IP did not respond. Attempting switchover."
                switched=1
            elif [ $switched -eq 1 ] && [ $(($counter % 10)) -eq 0 ]; then
                statusLoggerHfe "Connection error ongoing - No connection to SBC $SBC_PKT_PORT_NAME from HFE"
            fi
            
            # If single HFE (2.0), this is PKT0 config
            TEMP_SBC_IP=$ACTIVE_SBC_IP
            ACTIVE_SBC_IP=$STANDBY_SBC_IP
            STANDBY_SBC_IP=$TEMP_SBC_IP

            TEMP_SBC_NAME=$ACTIVE_SBC_VM_NAME
            ACTIVE_SBC_VM_NAME=$STANDBY_SBC_VM_NAME
            STANDBY_SBC_VM_NAME=$TEMP_SBC_NAME

            if [[ $MODE == "single" ]]; then
                TEMP_SBC_IP_PKT1=$ACTIVE_SBC_IP_PKT1
                ACTIVE_SBC_IP_PKT1=$STANDBY_SBC_IP_PKT1
                STANDBY_SBC_IP_PKT1=$TEMP_SBC_IP_PKT1
            fi

            statusLoggerHfeDebug "Before clearOldIptableRules"
            clearOldIptableRules
            statusLoggerHfeDebug "After clearOldIptableRules, and just before configureNATRules"

            statusLoggerHfeDebug "Verifying Routes"
            verifyRoutes
            verifyCount=0

            if [[ $MODE == "single" ]]; then
                configureNATRules $ETH0 $ETH3 $ETH3_GW $ACTIVE_SBC_IP $PRIMARY_IP_HFE_ETH0
                configureNATRules $ETH1 $ETH4 $ETH4_GW $ACTIVE_SBC_IP_PKT1 $PRIMARY_IP_HFE_ETH1
            else
                configureNATRules $ETH0 $ETH2 $ETH2_GW $ACTIVE_SBC_IP $PRIMARY_IP_HFE_ETH0
            fi

            statusLoggerHfeDebug "After configureNATRules and just before clearOldRules"
            clearOldRules
            statusLoggerHfeDebug "After clearOldRules"
            
            let counter=$counter+1 #increment log count
            statusLoggerHfeDebug "Wait for SBC to complete switchover."
            statusLoggerHfeDebug ""
            statusLoggerHfeDebug ""
            sleep 2s
        else
            if [ $INIT_LOG -eq 1 ]; then
                statusLoggerHfe "Initial HFE startup configuration complete. Successfully connected to $ACTIVE_SBC_VM_NAME"
                INIT_LOG=0
                switched=0
                counter=0
            elif [ $switched -eq 1 ]; then
                statusLoggerHfe "Switchover from old Active $TEMP_SBC_NAME to new Active $ACTIVE_SBC_VM_NAME complete. Connection established."
                switched=0
                counter=0
            fi
            statusLoggerHfeDebug "Nothing needs to be done, active is up $ACTIVE_SBC_VM_NAME"
            sleep 0.5s
        fi
        
        # Verify Routes every ~10 secs
        if [ $verifyCount -eq 20 ]; then
            verifyRoutes
            if [ $? -eq 1 ]; then
                statusLoggerHfe "Routes have changed. Flushing conntrack"
                clearOldRules
            fi
            verifyCount=0
        fi
        let verifyCount=$verifyCount+1

        # Verify every ~30seconds we are still supposed to be active
        if [ $haCheckCount -eq 60 ]; then
            if [ $HA_MODE -eq 1 ]; then
                haCheckIfStillActive &
            fi
            haCheckCount=0
        fi
        let haCheckCount=$haCheckCount+1
    done
}

main()
{
    case $1 in
        "setup") 
            prepareHFEInstance
            preConfigure
            readConfig
            haCheckIfStandby
            setInterfaceMap
            extractHfeConfigIps
            configurePrimaryIpMap
            # Configure Routes for Active SBC
            if [[ $MODE == "single" ]]; then
                configureNATRules $ETH0 $ETH3 $ETH3_GW $ACTIVE_SBC_IP $PRIMARY_IP_HFE_ETH0 1
                configureNATRules $ETH1 $ETH4 $ETH4_GW $ACTIVE_SBC_IP_PKT1 $PRIMARY_IP_HFE_ETH1 1
            else
                configureNATRules $ETH0 $ETH2 $ETH2_GW $ACTIVE_SBC_IP $PRIMARY_IP_HFE_ETH0 1
            fi
            configureMgmtNAT
            addCustomerRoutes
            clearOldRules 1
            showCurrentConfig
            checkSbcConnectivity
            ;;
        "cleanup")
            prepareHFEInstance
            preConfigure
            readConfig
            setInterfaceMap
            extractHfeConfigIps
            routeCleanUp
            clearOldRules 1
            doneMessage
            ;;
        "sysdump")
            prepareHFEInstance
            sysdump
            ;;
        *) 
            usage "Unrecognized switch"
            ;;
    esac
}

[[ $# -lt 1 ]] || [[ $# -gt 2 ]] && usage

if [ ! -z $2 ]; then
    PRIMARY_GCE_IP=1
    loggerHfe "Option passed in. Using primary IPs"
fi

main $1
