#!/bin/bash
#############################################################################################################################################################
# This script should be run after creating the stacks/instances that form a Redundancy Group (RG).
# It can be run multiple times without causing any harm. If a new instance is added to an aleardy running RG,
# the script shall be run again and provide ALL the stacks as parameters.
#
# Usage Examples:
# ./rgAllowedAddressPairUpdate.sh -h
# ./rgAllowedAddressPairUpdate.sh -p <path_to_openrc_file> -a <active_stack_name1> -a <active_stack_name2> -a <active_stack_name3> -a <active_stack_name4> -s <standby_stack_name>
#
# Prerequisites:
#   1. This script should be run on an instance/server where the latest openstack neutron and heat python-clients are installed.
#   2. The instance/server, should contain the openrc file to be able to communicate with the openstack controller node.
#   3. Controller node should be routable from the instance/server where the user runs this script.
# Max numbers of stacks that can be supplied :
#   Actives - 4
#   Standby - 1
#
# Gets the IP address of all the virtual ports created in the active instances.
# Create separate array for pkt0 and pkt1 for each active instance.
# Construct a global array of IP address for pkt0 and pkt1 and mgt0, and remove the IP address from the individual array of pkt array's of stack.
# Updates all the stacks with Allowed Address Pairs 
#############################################################################################################################################################

function executeCommand()
{
    commandToExecute=$1
    value=""
    timeOfExecution=`date +'%a %b %e %H:%M:%S %Z %Y'`
    echo -e "$timeOfExecution Executing: \n$commandToExecute" >> rgUpdate.logs

    value=`$commandToExecute`
    res=$?
    if [[ $res -ne 0 ]];then
        logMsg "Error executing $commandToExecute . Error code: $res."
        logMsg "Return Value: \n$value \n"
        return $res
    else
        echo -e "$value"
        return 0
    fi

}

function logMsg()
{
    logString=$1
    timeOfExecution=`date +'%a %b %e %H:%M:%S %Z %Y'`

    echo -e "\n$logString" >&2
    echo -e "\n$timeOfExecution \n $logString" >> rgUpdate.logs

    return 0
}

function validateStackExistance()
{
    stackName=$1

    stackCheckCommand="openstack stack show $stackName"
    #echo "$stackCheckCommand"

    cmd=$(executeCommand "$stackCheckCommand")
    res=$?
    if [[ $res -ne 0 ]]; then
        logMsg "Stack check failed for $stackName.\n$cmd"
        return 1;
    else
        logMsg "Stack validation passed for $stackName."
        return 0;
    fi
}


function parseContentAndGetIP()
{
    propertyTag=$1
    tableContent=$2
    local res=0

    existingIPArray=()
    address="";
    while IFS= read -r line; do
        if (echo "$line" | grep -q "$propertyTag") ; then
            # Find which field has the IP address
            ipAddressTag=`echo $line | cut -d'|' -f 3 | awk -F "," '{print $1}'`; 
            if (echo "$ipAddressTag" | grep -q "$ipPropertyTag") ; then
                address=`echo $line | cut -d'|' -f 3 | awk -F "," '{print $1}' | awk -F "$ipPropertyTag" '{print $2}'`;
            else
                address=`echo $line | cut -d'|' -f 3 | awk -F "," '{print $2}' | awk -F "$ipPropertyTag" '{print $2}'`;
            fi
            address=`echo "$address" | egrep -o '([0-9]{1,3}\.){3}[0-9]{1,3}'`;
            existingIPArray+=`echo " $address "`
            continue;
        fi
        if [[ "$address" != "" ]]; then
            isAnotherIpPresent=`echo $line | cut -d'|' -f 2`;
            if [[ "$isAnotherIpPresent" == " " ]]; then
                if (echo "$line" | grep -q "$ipPropertyTag");then
                    # Find which field has the IP address
                    ipAddressTag=`echo $line | cut -d'|' -f 3 | awk -F "," '{print $1}'`;
                    if (echo "$ipAddressTag" | grep -q "$ipPropertyTag") ; then
                        address=`echo $line | cut -d'|' -f 3 | awk -F "," '{print $1}' | awk -F "$ipPropertyTag" '{print $2}'`;
                    else
                        address=`echo $line | cut -d'|' -f 3 | awk -F "," '{print $2}' | awk -F "$ipPropertyTag" '{print $2}'`;
                    fi
                    address=`echo "$address" | egrep -o '([0-9]{1,3}\.){3}[0-9]{1,3}'`;
                    existingIPArray+=`echo " $address "`
                fi
            else
                break
            fi
        fi
    done < <(echo "$tableContent")


    echo "${existingIPArray[@]}"

}


function getPortVipByTag()
{
    stackName=$1
    resourceName=$2
    res=0
    # Get Stack resource by the resource name defined by port
    timeOfExecution=`date +'%a %b %e %H:%M:%S %Z %Y'`
    echo -e "$timeOfExecution Executing: " >> rgUpdate.logs
    echo -e "openstack stack resource list $stackName | grep $resourceName | awk -F $resourceName '{print \$2}' | awk -F \"|\" '{print \$2}' | awk -F \" \" '{print \$1}'" >> rgUpdate.logs

    resourceId=`openstack stack resource list $stackName | grep $resourceName | awk -F $resourceName '{print \$2}' | awk -F "|" '{print \$2}' | awk -F " " '{print \$1}'`
    res=$?
    if [[ $res -ne 0 ]]; then
        logMsg "Unable to get $port in the stack. Is this stack launched with sbc template?\n$resourceId"
        return 1;
    fi

    # Get the current information of port.
    res=0
    portShowCommand="openstack port show $resourceId"
    portDetailsInJson=$(executeCommand "$portShowCommand")
    res=$?
    if [[ $res -ne 0 ]]; then
        logMsg "Unable to get $port in the stack. Is this stack launched with sbc template?"
        logMsg "Error: $res. $portDetailsInJson"
        return 2;
    fi

    # Get the Fixed IP addresses on this port.
    res=0
    virtualIPs=$(parseContentAndGetIP "$fixedIPProprtyTag" "$portDetailsInJson")
    res=$?
    if [[ $res -ne 0 ]]; then
        logMsg "Error parsing existing port details."
        logMsg "Error: $res. $virtualIPs"
        return 3;
    fi

    logMsg "VIP obtained from $stackName for $resourceName port : $virtualIPs"

    # Pass this IP address list back to the caller function
    echo " $virtualIPs " 
    return 0

}


function getRgVIPs()
{

    port=$1
    vipResourceTag="$1""_vip_port"
    local res=0

    vipIPList=()

    for activeStacks in "${activeStackName[@]}"; do
        vipIPList+=$(getPortVipByTag $activeStacks $vipResourceTag)
        res=$?
        if [[ $res -ne 0 ]]; then
            logMsg "Error in obtaining the virtual IPs for $port."
            logMsg "Error value $res.\n"
            exit 1;
        fi
    done

    logMsg "VIP list obtained for $port port : ${vipIPList[@]}"
    echo ${vipIPList[@]}
    return 0
    
}


function convertIPListToParsableString()
{
    local port=$1
    local ipList=$2
    local ipString=""
    local ipStringList=()

    echo -e "ipList = ${ipList[@]}" >>  logFileParent.txt

    IFS=' ' read -r -a ipArray <<< "$ipList"

    declare -A portToIpMap=( ["pkt0"]=" -e " ["pkt1"]=" -i " ["mgt0"]=" -m " )

    for ip in "${ipArray[@]}"; do
        ipString=" ${portToIpMap[$port]} $ip "
        ipStringList+=( "$ipString" ) 
    done

    echo "${ipStringList[@]}" >>  logFileParent.txt

    echo "${ipStringList[@]}"
    return 0
}

function removeConflictingIPs()
{
    consolidatedList=()
    IFS=' ' read -r -a ipListA <<< "$1"
    IFS=' ' read -r -a ipListB <<< "$2"

    ipListA+=(${ipListB[@]})
    consolidatedList=($(echo "${ipListA[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))
    #echo ${consolidatedList[@]}

    for val in "${consolidatedList[@]}"; do
        AAPValueSpec+=" $aapTag $ipTag=$val "
    done

    echo $AAPValueSpec
    return 0;
}


function updateAap()
{
    local res=0
    local stackIdentifier=$1
    local port=$2
    declare -A portToIpMap=( ["$pkt0PortResourceTag"]="${port_vip_List[pkt0]}" ["$pkt1PortResourceTag"]="${port_vip_List[pkt1]}" ["$mgt0PortResourceTag"]="${port_vip_List[mgt0]}" )

    # Get Stack resource by the resource name defined by port
    timeOfExecution=`date +'%a %b %e %H:%M:%S %Z %Y'`
    echo -e "$timeOfExecution Executing: " >> rgUpdate.logs
    echo -e "openstack stack resource list $stackIdentifier | grep $port | awk -F $port '{print \$2}' | awk -F \"|\" '{print \$2}' | awk -F \" \" '{print \$1}'" >> rgUpdate.logs

    resourceId=`openstack stack resource list $stackIdentifier | grep $port | awk -F $port '{print \$2}' | awk -F "|" '{print \$2}' | awk -F " " '{print \$1}'`
    res=$?
    if [[ $res -ne 0 ]]; then
        logMsg "Unable to get $port in the stack. Is this stack launched with sbc template?\n$resourceId"
        return 1;
    fi

    # Get the current information of port.
    res=0
    portShowCommand="openstack port show $resourceId "
    portDetailsInJson=$(executeCommand "$portShowCommand")
    res=$?
    if [[ $res -ne 0 ]]; then
        logMsg "\nUnable to get $port in the stack. Is this stack launched with sbc template?"
        logMsg "Error: $res. $portDetailsInJson"
        return 2;
    fi

    # Get existing allowed_address_pairs of the port.
    res=0
    parseExistingAapIPs=$(parseContentAndGetIP "$aapPropertyTag" "$portDetailsInJson")
    res=$?
    if [[ $res -ne 0 ]]; then
        logMsg "Error parsing existing port details."
        logMsg "Error: $res. $parseExistingAapIPs"
        return 3;
    fi

    # Remove conflicting IPs.
    res=0
    modifiedIPList=()
    modifiedIPList=$(removeConflictingIPs "$parseExistingAapIPs" "${portToIpMap[$port]}")
    res=$?
    if [[ $res -ne 0 ]]; then
        logMsg "Error constructing the consolidated allowed_address_pair IP list."
        logMsg "Error: $res. $parseExistingAapIPs"
        return 4;
    fi

    # Update port with consolidated IP list.
    res=0
    updateAapCommand="neutron port-update $resourceId ${modifiedIPList[@]}"

    updatePort=$(executeCommand "$updateAapCommand")
    res=$?

    if [[ $res -ne 0 ]]; then
        logMsg "Unable to update $port in the stack."
        logMsg "Error: $res. $updatePort"
        return 5;
    fi
}



main()
{

    local res1=0;

    logMsg "Validating the heat stacks created from active template"
    for activeStacks in "${activeStackName[@]}"; do
        res1=0
        validateStackExistance $activeStacks
        res1=$?
        if [[ $res1 -ne 0 ]]; then
            logMsg "Invalid stack name provided to the script for ACTIVE instance, or environment not set correctly."
            logMsg "Stack Name: $activeStacks"
            logMsg "Error value $res1.\n"
            logMsg "Please check if adminrc file is correct: \n"
            `cat $openrcFile`
            exit 1;
        fi
    done

    logMsg "Validating the heat stack created from standby template"
    validateStackExistance $standbyStackName
    local res2=$?
    if [[ $res2 -ne 0 ]]; then
        logMsg "Invalid stack name provided to the script for STANDBY instance, or environment not set correctly."
        logMsg "Stack Name: $standbyStackName"
        logMsg "Error value $res2.\n"
        exit 1;
    fi

    logMsg "Phase 1 : Obtain all the VIP addresses from the active stacks"
    local result=0
    for port in "${portsToUpdate[@]}"; do
        result=0
        port_vip_List["$port"]=$(getRgVIPs "$port") 
        result=$?
        if [[ $result -ne 0 ]]; then
            logMsg "Failed to get IPs on virtual port for $port"
            logMsg "Error value $result\n"
            exit 1;
        fi
        logMsg "Done populating VIP addresses for $port port"
        echo -e "port_vip_List[$port] - ${port_vip_List[$port]}"

    done

    logMsg "Phase 1 : Complete"

    logMsg "Phase 2 : Update the VIP obatined in the previous step in the port propertie of private ports"
    # update aap of pkt ports of active and standby instances.
    # Call the updateAAP script for all the stack in the list.
    allStacks=()
    allStacks+=( ${activeStackName[@]} )
    allStacks+=( $standbyStackName )

    for stack in "${allStacks[@]}"; do

        local result=0

        logMsg "Updating primary ports of $stack stack"
        for port in "${portsToUpdate[@]}"; do
            logMsg "Updating $port port of $stack stack"
            result=0
            retVal=$(updateAap "$stack" "${portResourceTag[$port]}" )
            result=$?

            if [[ $result -ne 0 ]]; then
                logMsg "\nFailed to update Allowed address pair for $port"
                logMsg "Error value $result.\n"
                exit 1
            fi
            logMsg "Done updating $port port of $stack stack"
        done
        logMsg "Done updating primary ports of $stack stack"
    done

    logMsg "Phase 2 : Complete"

    logMsg "VIP update procedure complete. Now the instance may be used for running calls"

}



usageFunction()
{
cat << EOF
usage: $0 options

OPTIONS:
  -p    Path to openrc file.
  -a    Active Stack name. Can be specified multiple times.
  -s    Standby Stack name.
  -t    Is the script executed from test framework, where the openrc file is already sourced: Supply this argument. Else, don't.
  -h    display this message.

FORMAT:

$0 -p </Path/to/openrc/file> -a <Active Stack name> -s <Standby Stack name>

EXAMPLE:
$0 -p /home/sipp/openrc.sh -a ActiveStack1 -a ActiveStack2 -a ActiveStack3 -a ActiveStack4 -s StandbyStack

$0 -p /home/sipp/openrc.sh -a ActiveStack1 -a ActiveStack2 -a ActiveStack3 -s StandbyStack

$0 -p /home/sipp/openrc.sh -a ActiveStack1 -a ActiveStack2 -s StandbyStack

$0 -a ActiveStack1 -a ActiveStack2 -s StandbyStack -t

EOF
exit 1;
}

isTestEnv=0

while getopts ":p:a:hts:" opt; do
  case $opt in

    p) openrcFile=${OPTARG} ;;

    a) activeStackName+=("$OPTARG") ;;

    s) standbyStackName=${OPTARG} ;;

    t) isTestEnv=1 ;;

    h) usageFunction ;;

    *) usageFunction ;;

  esac
done

shift $((OPTIND-1))
if [[ -z "$activeStackName" ]] || [[ -z "$standbyStackName" ]];then
    usageFunction
fi

if (( $isTestEnv == 0 )); then
    if [[ -z "$openrcFile" ]]; then
        echo -e "\nOpenrc file not provided. Exiting.\n"
        usageFunction
    fi
    source $openrcFile
else
    echo -e "\nRunning in test environment. Expecting the openrc file to be already sourced\n"
fi

pkt0PortVIPResourceTag="pkt0_vip_port"
pkt1PortVIPResourceTag="pkt1_vip_port"
mgt0PortVIPResourceTag="mgt0_vip_port"

pkt0PortResourceTag="pkt0_port1"
pkt1PortResourceTag="pkt1_port1"
mgt0PortResourceTag="mgt0_port1"

declare -gA portResourceTag=( ["pkt0"]="$pkt0PortResourceTag" ["pkt1"]="$pkt1PortResourceTag" ["mgt0"]="$mgt0PortResourceTag"  )
declare -a portsToUpdate=( "mgt0" "pkt0" "pkt1" )
declare -gA port_vip_List=( ["pkt0"]="" ["pkt1"]="" ["mgt0"]=""  )

aapTag="--allowed-address-pair"
ipTag="ip_address"
ipPropertyTag="ip_address"
aapPropertyTag="allowed_address_pairs"
fixedIPProprtyTag="fixed_ips"

echo "" > rgUpdate.logs

main
