#!/bin/bash

#######
####### to use this file, the caller must must define the following:
####### trapExit function
####### logMsg function
####### logFile variable
####### resultFile variable (set to "" if no result file is needed)
####### dbgInterfaceCfg
####### pktInterfaceCfg
####### hostType -- needs to be defined since may not run on a target sbc
####### AWK, BASH, CAT, ECHO, GREP, GETCONF, HOST_TYPE to mimc commandVariables
PROG_DIR=$(cd "$(/usr/bin/dirname $(readlink -f "${BASH_SOURCE[0]}" ))" && pwd)
if [ -e $PROG_DIR/personality.sh ]
then
    source $PROG_DIR/personality.sh
else
    $ECHO "Could not locate personality.sh Exiting..."
    exit 1
fi

if [ -e $PROG_DIR/commonTimeZones.sh ]
then
    source $PROG_DIR/commonTimeZones.sh
else
    $ECHO "Could not locate commonTimeZones.sh Exiting..."
    exit 1
fi

# provide defaults for geo-graphical HA mode for upgrades from versions
# that did not have the settings and for use in ISO app installs like
# isoSbx.  Originally we released one option called geoRedundHA.  That is
# now replaced by two options: leaderElect and bondMonitor.
leaderElect=0
bondMonitor=0
leaderElectOptions="0=standard, 1=enhanced"
bondMonitorOptions="0=direct-connect, 1=network-connect"

# Default tipcID in case not set in sbx.conf
tipcID=1500

# provide a default for FIPS mode that is used in upgrades from versions
# that did not have the setting
fipsMode="disabled"

# personality options: default to ERE
# to add a new personality, just add a new mode value and add it to the
# hash.  all other usage is taken care of by the hash and the option
# string created below.
# note: the following hash is used for a log statement.  the value is also
# lowercase'd to use as input for the call to updatePersonality.pl
# note: the option string is used to keep consistent output
# note: provide a default so we have a value on initial upgrades to a
# version containing the personality setting.  We only supported ERE
# prior to it existing, so we can use that as the default.

personality=$SbxErePersonality
bashVersion=`$BASH --version | $GREP "GNU bash, version" | $AWK -F, '{print $2}' |  $AWK '{print $2}' | $AWK -F. '{print $1}'`
if [ $bashVersion -ge 4 ]; then
   declare -A personalityMnemonic=( ["$SbxErePersonality"]="ERE" ["$SbxOamPersonality"]="OAM"  ["$SbxNoErePersonality"]="NO_ERE" )
else
   skipPersonalityCheck=true;
fi

personalityOptions=""
for personalityKey in "${!personalityMnemonic[@]}"
do
   if [ -n "$personalityOptions" ]; then
      personalityOptions="$personalityOptions "
   fi
   personalityOptions="${personalityOptions}${personalityKey}=${personalityMnemonic[$personalityKey]}"
done

# Test if an TIPC ID is valid:
# Usage:
#    valid_tipcID ID
#    if [[ $? -eq 0 ]]; then good; else bad; fi
# OR
#    if valid_tipcID ID; then good; else bad; fi
#
function valid_tipcID()
{
   local  id=$1
   local  stat=1

   # check for hostType file since we may be validating on a
   # non-SBC system.  in that case the hostType value has to be pre-defined
   # before calling validation.
   if [[ -f "$HOST_TYPE" ]]; then
       hostType=`$CAT $HOST_TYPE`
   fi
   if [[ $hostType == "ConnexIP5000" ]]; then
      if [[ $id -ge 1000 && $id -le 9999 ]] ; then
         stat=$?
      fi
   else
     if [[ $id =~ ^[0-9]{1,4}$ ]]; then
       ip=($ip)
       [[ ${id[0]} -ge 1025 && ${id[0]} -le 4095 ]]
       stat=$?
     fi
   fi
   return $stat
}

# Perform the validation that is performed when
# prompting the user on the passed in sbx.conf.
# The values have already been read, so we can
# just match the validation performed in
# promptForConfig.
validateConfig()
{
    # max size of host name, not including the null terminatore
    maxHostLength=
    if [[ -f $GETCONF ]]; then
        maxHostLength=`$GETCONF HOST_NAME_MAX`
    fi
    if [[ -z "$maxHostLength" ]]; then
        # most linuxes have 64 for the max value
        maxHostLength=64
    fi

    # max size of system name, not including the null terminator
    maxSystemLength=26

    if [ $role -ne 1 -a $role -ne 2 ]; then
       logMsg $logFile "ERROR: Configured role is invalid, Exiting !!!!"
       trapExit 1 $resultFile
    fi
    if [ -z "$systemName" ]; then
       logMsg $logFile "ERROR: Configured system name cannot be empty, Exiting !!!!"
       trapExit 1 $resultFile
    fi

    if [[ ! "$systemName" =~ ^[A-Za-z]{1}[-A-Za-z0-9.]*[A-Za-z0-9]{1}$ ]]; then
        logMsg $logFile "ERROR: Configured system name can only contain alphanumeric characters and a dash or period, must begin with a letter, and end with an alphanumeric character.  Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ ${#systemName} -gt $maxSystemLength ]; then
        logMsg $logFile "ERROR: Configured system name cannot be longer than $maxSystemLength characters, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ -z "$ceName" ]; then
        logMsg $logFile "ERROR: Configured CE name cannot be empty, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [[ "$ceName" =~ [_] ]]; then
        logMsg $logFile "ERROR: Configured CE name cannot contain an underscore, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [[ ! "$ceName" =~ ^[A-Za-z]{1}[-A-Za-z0-9.]*[A-Za-z0-9]{1}$ ]]; then
        logMsg $logFile "ERROR: Configured CE name can only contain alphanumeric characters and a dash or period, must begin with a letter, and end with an alphanumeric character.  Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ ${#ceName} -gt $maxHostLength ]; then
        logMsg $logFile "ERROR: Configured CE name cannot be longer than $maxHostLength characters, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ -z "$peerCeName" ]; then
        logMsg $logFile "ERROR: Configured peerCE name cannot be empty, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [[ "$peerCeName" =~ [_] ]]; then
        logMsg $logFile "ERROR: Configured peerCE name cannot contain an underscore, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [[ ! "$peerCeName" =~ ^[A-Za-z]{1}[-A-Za-z0-9.]*[A-Za-z0-9]{1}$ ]]; then
        logMsg $logFile "ERROR: Configured peerCE name can only contain alphanumeric characters and a dash or period, must begin with a letter, and end with an alphanumeric character.  Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ ${#peerCeName} -gt $maxHostLength ]; then
        logMsg $logFile "ERROR: Configured peerCE name cannot be longer than $maxHostLength characters, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ "$ceName" == "$peerCeName" -o "$ceName" == "$systemName" -o "$peerCeName" == "$systemName" ]; then
        logMsg $logFile "ERROR: Configured ceName, peerCeName, and systemName must be unique, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ "$dbgInterfaceCfg" == "y" ]; then
        if ! valid_ip $dbgIpaddr; then
            logMsg $logFile "ERROR: Configured debug IP is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if ! valid_ip $dbgNetmask; then
            logMsg $logFile "ERROR: Configured debug net mask is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if ! valid_ip $dbgGatewayIpaddr; then
            logMsg $logFile "ERROR: Configured debug gateway IP is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
    fi
    if [[ -n "$activePrimaryInterCeIpaddr" && "$activePrimaryInterCeIpaddr" == "$standbyPrimaryInterCeIpaddr" ]]; then
	    logMsg $logFile "ERROR: Configured active and standby IP addresses must be unique, Exiting !!!!"
	    trapExit 1 $resultFile
    fi

    if [ -z "$nif1Ipaddr" ]; then
        nif1Ipaddr="0.0.0.0"
    elif ! valid_ip $nif1Ipaddr; then
        logMsg $logFile "ERROR: Configured primary management IP is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ -z "$nif1Netmask" ]; then
        nif1Netmask="0.0.0.0"
        convNetmaskToNetprefix $nif1Netmask
        nif1Netprefix=$?
    elif ! valid_ip $nif1Netmask; then
        logMsg $logFile "ERROR: Configured primary management net mask is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ -z "$nif1GatewayIpaddr" ]; then
        nif1GatewayIpaddr="0.0.0.0"
    elif ! valid_ip $nif1GatewayIpaddr; then
        logMsg $logFile "ERROR: Configured primary management gateway IP is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [[ -z "$nif1IpaddrV6" || "$nif1IpaddrV6" == "0" ]]; then
        logMsg $logFile "Setting primary management IPv6 address from [$nif1IpaddrV6] to [::]"
        nif1IpaddrV6="::"
    elif ! valid_ipv6 $nif1IpaddrV6; then
        logMsg $logFile "ERROR: Configured primary management IPv6 is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ -z "$nif1NetprefixV6" ]; then
        nif1NetprefixV6="0"
    elif [[ 0 -gt $nif1NetprefixV6 || $nif1NetprefixV6 -gt 128 ]]; then
        logMsg $logFile "ERROR: Configured primary management IPv6 prefix length is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [[ -z "$nif1GatewayIpaddrV6" || "$nif1GatewayIpaddrV6" == "0" ]]; then
        logMsg $logFile "Setting primary management gateway IPv6 address from [$nif1GatewayIpaddrV6] to [::]"
        nif1GatewayIpaddrV6="::"
    elif ! valid_ipv6 $nif1GatewayIpaddrV6; then
        logMsg $logFile "ERROR: Configured primary management gateway IPv6 is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    
    if [[ "$hostType" == "ConnexIP5000" ]] && [[ "$hostSubType" != "virtualContainer" || "$POD_TYPE" == "OAM" ]]; then
        if [[ $nif1Ipaddr == "0.0.0.0" && $nif1IpaddrV6 == "::" ]]; then
            logMsg $logFile "ERROR: No Management IP Addresses detected, Exiting !!!!"
            trapExit 1 $resultFile
        fi
    fi

    if [ -z "$nif5Ipaddr" ]; then
        nif5Ipaddr="0.0.0.0"
    elif ! valid_ip $nif5Ipaddr; then
        logMsg $logFile "ERROR: Configured secondary management IP is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ -z "$nif5Netmask" ]; then
        nif5Netmask="0.0.0.0"
        convNetmaskToNetprefix $nif5Netmask
        nif5Netprefix=$?
    elif ! valid_ip $nif5Netmask; then
        logMsg $logFile "ERROR: Configured secondary management net mask is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ -z "$nif5GatewayIpaddr" ]; then
        nif5GatewayIpaddr="0.0.0.0"
    elif ! valid_ip $nif5GatewayIpaddr; then
        logMsg $logFile "ERROR: Configured secondary management gateway IP is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [[ -z "$nif5IpaddrV6" || "$nif5IpaddrV6" == "0" ]]; then
        logMsg $logFile "Setting secondary management IPv6 address from [$nif5IpaddrV6] to [::]"
        nif5IpaddrV6="::"
    elif ! valid_ipv6 $nif5IpaddrV6; then
        logMsg $logFile "ERROR: Configured secondary management IPv6 is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ -z "$nif5NetprefixV6" ]; then
        nif5NetprefixV6="0"
    elif [[ 0 -gt $nif5NetprefixV6 || $nif5NetprefixV6 -gt 128 ]]; then
        logMsg $logFile "ERROR: Configured secondary management IPv6 prefix length is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [[ -z "$nif5GatewayIpaddrV6" || "$nif5GatewayIpaddrV6" == "0" ]]; then
        logMsg $logFile "Setting secondary management gateway IPv6 address from [$nif5GatewayIpaddrV6] to [::]"
        nif5GatewayIpaddrV6="::"
    elif ! valid_ipv6 $nif5GatewayIpaddrV6; then
        logMsg $logFile "ERROR: Configured secondary management gateway IPv6 is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi

    if [[ "$hostType" != "SBC5400" ]] && [[ "$hostSubType" != "virtualContainer" || "$POD_TYPE" == "OAM" ]]; then
        if [[ $nif1Ipaddr == "0.0.0.0" && $nif5Ipaddr == "0.0.0.0" && $nif1IpaddrV6 == "::" && $nif5IpaddrV6 == "::" ]]; then
            logMsg $logFile "ERROR: No Management IP Addresses detected, Exiting !!!!"
            trapExit 1 $resultFile
        fi
    fi

    if [[ "$hostType" == "SBC5400" ]]; then
        if [ -z "$nif7Ipaddr" ]; then
            nif7Ipaddr="0.0.0.0"
        elif ! valid_ip $nif7Ipaddr; then
            logMsg $logFile "ERROR: Configured third management IP is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if [ -z "$nif7Netmask" ]; then
            nif7Netmask="0.0.0.0"
            convNetmaskToNetprefix $nif7Netmask
            nif7Netprefix=$?
        elif ! valid_ip $nif7Netmask; then
            logMsg $logFile "ERROR: Configured third management net mask is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if [ -z "$nif7GatewayIpaddr" ]; then
            nif7GatewayIpaddr="0.0.0.0"
        elif ! valid_ip $nif7GatewayIpaddr; then
            logMsg $logFile "ERROR: Configured third management gateway IP is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if [[ -z "$nif7IpaddrV6" || "$nif7IpaddrV6" == "0" ]]; then
            logMsg $logFile "Setting third management IPv6 address from [$nif7IpaddrV6] to [::]"
            nif7IpaddrV6="::"
        elif ! valid_ipv6 $nif7IpaddrV6; then
            logMsg $logFile "ERROR: Configured third management IPv6 is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if [ -z "$nif7NetprefixV6" ]; then
            nif7NetprefixV6="0"
        elif [[ 0 -gt $nif7NetprefixV6 || $nif7NetprefixV6 -gt 128 ]]; then
            logMsg $logFile "ERROR: Configured third management IPv6 prefix length is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if [[ -z "$nif7GatewayIpaddrV6" || "$nif7GatewayIpaddrV6" == "0" ]]; then
            logMsg $logFile "Setting third management gateway IPv6 address from [$nif7GatewayIpaddrV6] to [::]"
            nif7GatewayIpaddrV6="::"
        elif ! valid_ipv6 $nif7GatewayIpaddrV6; then
            logMsg $logFile "ERROR: Configured third management gateway IPv6 is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if [ -z "$nif8Ipaddr" ]; then
            nif8Ipaddr="0.0.0.0"
        elif ! valid_ip $nif8Ipaddr; then
            logMsg $logFile "ERROR: Configured fourth management IP is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if [ -z "$nif8Netmask" ]; then
            nif8Netmask="0.0.0.0"
            convNetmaskToNetprefix $nif8Netmask
            nif8Netprefix=$?
        elif ! valid_ip $nif8Netmask; then
            logMsg $logFile "ERROR: Configured fourth management net mask is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if [ -z "$nif8GatewayIpaddr" ]; then
            nif8GatewayIpaddr="0.0.0.0"
        elif ! valid_ip $nif8GatewayIpaddr; then
            logMsg $logFile "ERROR: Configured fourth management gateway IP is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if [[ -z "$nif8IpaddrV6" || "$nif8IpaddrV6" == "0" ]]; then
            logMsg $logFile "Setting fourth management IPv6 address from [$nif8IpaddrV6] to [::]"
            nif8IpaddrV6="::"
        elif ! valid_ipv6 $nif8IpaddrV6; then
            logMsg $logFile "ERROR: Configured fourth management IPv6 is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if [ -z "$nif8NetprefixV6" ]; then
            nif8NetprefixV6="0"
        elif [[ 0 -gt $nif8NetprefixV6 || $nif8NetprefixV6 -gt 128 ]]; then
            logMsg $logFile "ERROR: Configured fourth management IPv6 prefix length is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
        if [[ -z "$nif8GatewayIpaddrV6" || "$nif8GatewayIpaddrV6" == "0" ]]; then
            logMsg $logFile "Setting fourth management gateway IPv6 address from [$nif8GatewayIpaddrV6] to [::]"
            nif8GatewayIpaddrV6="::"
        elif ! valid_ipv6 $nif8GatewayIpaddrV6; then
            logMsg $logFile "ERROR: Configured fourth management gateway IPv6 is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi

        if [[ $nif1Ipaddr == "0.0.0.0" && $nif5Ipaddr == "0.0.0.0" && $nif7Ipaddr == "0.0.0.0" && $nif8Ipaddr == "0.0.0.0" && $nif1IpaddrV6 == "::" && $nif5IpaddrV6 == "::" && $nif7IpaddrV6 == "::" && $nif8IpaddrV6 == "::" ]]; then
            logMsg $logFile "ERROR: No Management IP Addresses detected, Exiting !!!!"
            trapExit 1 $resultFile
        fi
    fi
    if [ -z "$ntpServerIpaddr" ]; then
        ntpServerIpaddr="0.0.0.0"
    elif ! valid_ipv6 $ntpServerIpaddr && ! valid_ip $ntpServerIpaddr; then
            logMsg $logFile "ERROR: Configured NTP timer server IP is invalid, Exiting !!!!"
            trapExit 1 $resultFile
    fi
    if [[ $timeZoneIndex -ge ${#zoneName[@]} || $timeZoneIndex -le 0 ]]; then
        logMsg $logFile "ERROR: Configured timezone setting is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ -n "$allowSshAccess" ] && [[ "$allowSshAccess" != "y" && "$allowSshAccess" != "Y" && "$allowSshAccess" != "n" && "$allowSshAccess" != "N" ]]; then
        logMsg $logFile "ERROR: Configured SSH access is invalid, Exiting !!!!"
        trapExit 1 $resultFile
    fi
    if [ -n "$sbctype" ] && [[ "$sbctype" != "isbc" && "$sbctype" != "ssbc" && "$sbctype" != "msbc" && "$sbctype" != "mrfp" && "$sbctype" != "slb" ]]; then
        logMsg $logFile "ERROR: Configured SBC personality type is invalid, Exiting !!!"
        trapExit 1 $resultFile
    fi
    if [ "$skipPersonalityCheck" != "true" ];then
        if [ "${personalityMnemonic[$personality]+exists}" != "exists" ]; then
            logMsg $logFile "ERROR: Configured routing engine personality is invalid, Exiting !!!!"
            trapExit 1 $resultFile
        fi
    fi
    if [ "$pktInterfaceCfg" == "y" ]; then
        if [ -z "$nif3Ipaddr" ]; then
           nif3Ipaddr="0.0.0.0"
        fi
        if ! valid_ip $nif3Ipaddr; then
           logMsg $logFile "ERROR: Configured (port3) signaling/media IP is invalid, Exiting !!!!"
           trapExit 1 $resultFile
        fi
        if [ -z "$nif3Netmask" ]; then
           nif3Netmask="0.0.0.0"
        fi
        if ! valid_ip $nif3Netmask; then
           logMsg $logFile "ERROR: Configured (port3) signaling/media netmask is invalid, Exiting !!!!"
           trapExit 1 $resultFile
        fi
        if [ -z "$nif4Ipaddr" ]; then
           nif4Ipaddr="0.0.0.0"            
        fi
        if ! valid_ip $nif4Ipaddr; then
           logMsg $logFile "ERROR: Configured (port4) signaling/media IP is invalid, Exiting !!!!"
           trapExit 1 $resultFile
        fi
        if [ -z "$nif4Netmask" ]; then
           $nif4Netmask="0.0.0.0"
        fi
        if ! valid_ip $nif4Netmask; then
           logMsg $logFile "ERROR: Configured (port4) signaling/media netmask is invalid, Exiting !!!!"
           trapExit 1 $resultFile
        fi
    else
        nif3Ipaddr="0.0.0.0"
        nif3Netmask="0.0.0.0"
        nif4Ipaddr="0.0.0.0"
        nif4Netmask="0.0.0.0"        
    fi

    if ! valid_tipcID $tipcID; then
       logMsg $logFile "ERROR: Configured TIPC is invalid (should be 1000 - 9999 ), Exiting !!!!"
       trapExit 1 $resultFile
    fi

    if [ $leaderElect -ne 0 -a $leaderElect -ne 1 ]; then
       logMsg $logFile "ERROR: Configured leader election mode is invalid, Exiting !!!!"
       trapExit 1 $resultFile
    fi

    if [ $bondMonitor -ne 0 -a $bondMonitor -ne 1 ]; then
       logMsg $logFile "ERROR: Configured bond monitoring mode is invalid, Exiting !!!!"
       trapExit 1 $resultFile
    fi
    if [ -n "$haMode" ] && [[ "$haMode" != "Nto1" && "$haMode" != "1to1" ]]; then
        logMsg $logFile "ERROR: Configured SBC manangement mode is invalid, Exiting !!!"
        trapExit 1 $resultFile
    fi
}
