#!/bin/bash
#############################################################
#
# Copyright (c) 2013 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# Module Description:
#    Script to check for minimum disk performance.
#
# NOTE: the specific error message is looked for by other
# scripts, so do not change it without also updating them!
#############################################################

# Sourcing Command Variables
if [ -e /opt/sonus/staging/sonusCommands.sh ]
then
    source /opt/sonus/staging/sonusCommands.sh
elif [ -e /opt/sonus/bin/sonusCommands.sh ]
then
    source /opt/sonus/bin/sonusCommands.sh
else
    /bin/echo "Could not locate sonusCommands.sh Exiting..."
    exit 1
fi

if [ -e /opt/sonus/staging/sonusCommonFiles.sh ]
then
    source /opt/sonus/staging/sonusCommonFiles.sh
elif [ -e /opt/sonus/bin/sonusCommonFiles.sh ]
then
    source /opt/sonus/bin/sonusCommonFiles.sh
else
    $ECHO "Could not locate sonusCommonFiles.sh Exiting..."
    exit 1
fi

# Sourcing Common Utils
if [ -e $STAGING_SONUS_COMMON_UTILS_SH ]
then
    source $STAGING_SONUS_COMMON_UTILS_SH
elif [ -e $SONUS_COMMON_UTILS_SH ]
then
    source $SONUS_COMMON_UTILS_SH
else
    $ECHO "Could not locate sonusCommonUtils.sh Exiting..."
    exit 1
fi

# some globals:
# require 100MB/sec as a minimum read speed
requiredMinRead=100
requiredMinReadQual=MB
verboseMode="n"
tmpDir=$SONUS_TMP_DIR

# include sonusUtils - Assuming checkDiskPerformance.sh script is called from staging,
# make sure sonusUtils.sh script exists in the same path.
# If the path is changed later, need to make sure if the sonusUtils.sh is sourced
# from the right path.
installRoot=$(dirname $0)
. ${installRoot}/sonusUtils.sh

# cleans up and exits the script with the supplied value
function cleanup_exit()
{
    local exitNow=$1

    if [ -f $dstatRes ]; then
		$RM -f $dstatRes
    fi
    if [ -f $iotopRes ]; then
		$RM -f $iotopRes
    fi
	exit $exitNow
}

# check all disks for enough space
function checkDiskReadSpeed()
{
    readResult=$1
    diskReads=`$ECHO $readResult | $GREP "disk reads" | $AWK -F= '{print $2}'`
    readSpeed=`$ECHO $diskReads | $AWK '{print $1}' | $AWK -F. '{print $1}'`
    readQualifier=`$ECHO $diskReads | $AWK '{print $2}' | $AWK -F/ '{print $1}'`

    if [ $readSpeed -ge $requiredMinRead -a "$readQualifier" == "$requiredMinReadQual" ]; then
      if [ $verboseMode = "y" ]; then
        preInstallLogMsg "Disk read speed is greater than the minimum required ($requiredMinRead/$requiredMinReadQual)"
      fi
    else
      if [ $verboseMode = "y" ]; then
        preInstallLogMsg "Disk read speed ($readSpeed/$readQualifier) is less than the minimum required ($requiredMinRead/$requiredMinReadQual)" "user"
      fi
      if [[ ! -z $dstatRes ]] && [[ -e $dstatRes ]]; then
        $GREP -q "M" $dstatRes
        if [ $? -eq 0 ]; then
          preInstallLogMsg "Disk read speed failed minimum requirement test (reported $readSpeed/$readQualifier)" "user"
          preInstallLogMsg "Below is the current heavy I/O usage by processes or threads on the system(saved in the file $iotopRes)" "user"
          preInstallLogMsg "`$CAT $iotopRes | $GREP "M/s" | $SED "s/Total DISK READ/\nTotal DISK READ/"`" "user"
          preInstallLogMsg "ERROR: Some heavy disk I/O operations are running at this time. Please try installing again after sometime. If the problem persists, please contact Sonus support; Exiting..." "user"
          cleanup_exit 1
        else
          preInstallLogMsg "ERROR: Disk read speed failed minimum requirement test (reported $readSpeed/$readQualifier); Exiting..." "user"
          cleanup_exit 1
        fi
      else
        preInstallLogMsg "ERROR: Disk read speed failed minimum requirement test (reported $readSpeed/$readQualifier); Exiting..." "user"
        cleanup_exit 1 
      fi
    fi
}

usage()
{
$CAT << EOF
usage: `$BASENAME $0` [-h] [-m min_read] [-v]

OPTIONS:
   -h               Print this message.
   -m #             Absolute minimum MB read performance (default 100MB).
   -v               Verbose mode.
EOF
}

while getopts "hm:v" OPTION
do
   case $OPTION in
   h)
      usage
      exit 1
      ;;
   m)
      requiredMinRead=$OPTARG
      ;;
   v)
      verboseMode="y"
      ;;
   ?)
      usage
      exit 1
      ;;
   esac
done

shift $(($OPTIND - 1))

# Run dstat and iotop commands in background for 10 seconds(along with hdparm call) to retrieve the disk read/write speed and redirect to some outputFile.
# In case of low disk speeds,hdparm takes atleast 10 seconds to complete and then checkDiskReadSpeed displays the error message and current i/o operations from the outputFile.
if [ -e /usr/sbin/iotop ]; then
  dstatRes=$($MKTEMP --tmpdir=$tmpDir)
  iotopRes=$($MKTEMP --tmpdir=$tmpDir)
  $DSTAT --nocolor -dD ${primary_disk} 1 10 > $dstatRes 2> /dev/null &
  iotop -o -b -n 10 > $iotopRes 2> /dev/null &
fi

# retrieve the read results
# for now we are just getting disk reads. if we need to get cache reads
# too, just modify the hdparm call.  The disk read check greps out the line
# it wants just in case the output changes.

rootDisk=$(getRootDisk)

hostType=$(getHwType)

if [[ $hostType == "SBC7000" || $hostType == "SBC5400" ]]; then
  # /dev/sda8 has LVM partitions of our interest: debian-root1 debian-root2 debian-common
  hdparmRes=`$HDPARM -t --direct ${rootDisk}8`
else 
  # ConnexIP5000: /dev/vda5 has LVM partitions of our interest: debian-root debian-varlog
  # if external disk is provided for /var/log then /dev/vda5 will have only debian-root
  hdparmRes=`$HDPARM -t --direct ${rootDisk}5`
fi

if [ $verboseMode = "y" ]; then
  preInstallLogMsg "$hdparmRes\n"
fi

checkDiskReadSpeed "$hdparmRes"

cleanup_exit 0
