#!/usr/bin/python3
# -*- coding: utf-8 -*-
#title           :cloud_diags.py
#description     :This program allows diagnostics tests to be performed on Sonus cloud systems
#author          :Liaquat Ali
#date            :
#version         :0.3
#usage           :python cloud_diags.py
#notes           :
#python_version  :2.7
#=======================================================================
# version 0.1 - original version
# version 0.2 (12/07/2017) - fix for showPlatformInfo inline with platform change (getMetaData)
# version 0.3 (18/07/2017) - help text updated for showFullSystemInfo & startNetworkTest
#===========================================================================================
# Instructions for adding a new test/command to the tool
# ------------------------------------------------------
# The command menu system for this diagnostics tool is implemented using the Python
# standard 'cmd' library. The top level command processing is done in the 'diagsCommands'
# class, which includes an entry for each of the diagnostics command.
# Each entry should specify the help text associated with the command as well as the python
# function to be called to process the command.
# 
# Each command processing definition is passed a string input that may be used to carry the
# parameters required for that command.
# Processing of the string input to parse/extract and validate any required parameters will
# be the responsibility of each specific command processing section.
# 
# Thus, to add a new command to the tool the following steps are required -
#  
# 1. Add a new entry for the command in diagsCommands
#    e.g.  def do_startMessageSizeTest(self, line):
# 2. The entry should include help text for the command
# 3. If required, the entry should include calls to extract_options... function
#    and validateOptions function to parse and validate the command parameters.   
# 4. The entry should include call to the command processing function with dictionary of
#    parameters as input. The command processing function needs to implement the requirements
#    of the test and output any results info.
# 
# The following example is based on the addition of the do_startMessageSizeTest test
# 
#     def do_startMessageSizeTest(self, line):
#         
#         "\n    Start the Message Size Test on the client \
#          \n    Command Syntax: ........ etc\
#          \n"
#         
#         longOption_list = ['si=', 'sp=']
#         shortOption_list = 'p:'
# 
#         extracted_option = extract_option_messageSizeTest(shortOption_list, longOption_list, line)
#         
#         if len(extracted_option) >  0 and 'error' not in extracted_option:             
#             if validateMessageTestOptions(extracted_option):               
#                 udpStartMessageSizeTest(extracted_option)
#
#===========================================================================================
 
# Import the modules needed to run the script.
import sys
import re
import os
import cmd
import subprocess
import platform
import socket
import time
import _thread
from threading import Thread
import logging, logging.handlers
import getopt
from itertools import groupby

sys.path.append('/opt/sonus/bin')
from sonusCommands import *
from sonusCommonFiles import *

META_DATA_URI_OSTACK = "http://169.254.169.254/openstack/latest/meta_data.json"

META_DATA_FILE_SBX = CLOUD_META_DATA_JSON
META_DATA_FILE_EMS_PSX = CLOUD_META_DATA_EMS_PSX_JSON

# Following variable sets the debug mode to enable display of input parameters on some tests.
# By default its 0 (off), to turn it on set it to 1 (on)
param_debug_display = 0


########Parameter's for message size test - start#########

massageSizeTest_biggest_packet_tobe_sent = 10000

massageSizeTest_initial_packet_size_toStart = 100

massageSizeTest_packet_size_gets_incremented = 100

massageSizeTest_receiverThread_buffer_overhead = 10

massageSizeTest_forEach_PacketSize_totalMessageTobeSent = 10

########Parameter's for message size test - end##########


class Logger(object):
    """ logger object to allow sending of print output to a file """	
    def __init__(self, filename="Default.log"):
        self.terminal = sys.stdout
        self.log = open(filename, "a")

    def write(self, message):
        self.terminal.write(message)
        self.log.write(message)
        
class packetData(object):
    """ __init__() functions as the calls constructor"""
    def __init__(self, seqId=0, startTime=None, stopTime=None):

        self.seqId = seqId
        self.startTime = startTime
        self.stopTime = stopTime



class diagsCommands(cmd.Cmd):
    """command processor for diagnostics tests"""

    intro = "\nSonus test agent and diagnostics tool\n-------------------------------------\n"
    prompt = '(Enter command or ?) : '
    doc_header = 'Supported commands (type help <command_name>):'
    
    def do_showCpuInfo(self, line):
        "\n Show information on vCPU's, including clock speed, sockets, cores, threads etc \n"

        showCpuInfo(line)


    def do_showMemoryInfo(self, line):
        "\n Show information on system memory including available and free memory \n"       

        showMemoryInfo(line)


    def do_showStorageInfo(self, line):
        "\n Show information on system storage / disks including used and free space\n"

        showStorageInfo(line)


    def do_showPlatformInfo(self, line):
        "\n Show information on platform, includes node name, app type, linux platform info, \
		 \n architecture, dhcp use, sriov, config drive and metadata info  \n"

        showPlatformInfo(line)


    def do_showNetworkInfo(self, line):
        "\n Show information on Network Interfaces including configured IP addresses\n"

        showNetworkInfo(line)


    def do_showFullSystemInfo(self, line):
        "\n Show comprehensive system information including platform, CPU, \
         \n memory, storage, network interfaces and check disk IOPs \
         \n If the optional -f flag is specified, output from the command \
		 \n shall be written to a file named <applicationType>_<hostname>_fullsysinfo_YYYYMMDD-HHMMSS.txt \
         \n where YYYYMMDD-HHMMSS is the date-time when file is created \n"
		
        showFullSystemInfo(line)


    def do_checkDiskPerformance(self, line):
        "\n Perform basic check of the disk IO performance by reading and writing \
		 \n 10K data records 1,000 times from/to disk. Shows time taken to do read \
		 \n and write of data and read/write of data speed in MB/sec \n"

        checkDiskPerformance(line)

    def do_startNetworkTest(self, clientOptionList):
        "\n    Start the Network Test on the Client \
         \n     Command Syntax: \
         \n        startNetworkTest -p tcp  --si <remote_server_ip> [--sp <remote_server_port_no>]  --li <local_ip> [--lp <local_port_no>]  [-r <msg_rate_per_sec>]  [-m <message_size>] [-c <run_count>] \
         \n        startNetworkTest -p udp  --si <remote_server_ip> [--sp <remote_server_port_no>] [-r <msg_rate_per_sec>]  [-m <message_size>] [-c <run_count>]\
         \n    Mandatory Parameters: \
         \n        -p   <protocol_type> \
         \n        --si <remote_server_ip> \
         \n        --li <local_client_ip>       (only required for TCP) \
         \n    Optional parameters: \
         \n        --sp <remote_server_port_no> (default TCP=40000, UDP=5061) \
         \n        --lp <local_client_port_no>  (default TCP=20000, UDP=5061) \
         \n        -r   <msg_rate_per_sec>      (default = 1000) \
         \n        -m   <msg_size>              (default = 500) \
         \n        -c   <run_count>             (default = 1) \
         \n        -f                           (output test result to file named <applicationType>_<hostname>_performanceResult_YYYYMMDD-HHMMSS.txt) \
         \n    Example: \
         \n        startNetworkTest -p tcp --si 10.1.10.1 --sp 40000 --li 10.20.2.20 --lp 20000 -r 1000 -m 500 -c 1 -f \
         \n        startNetworkTest -p udp --si 10.1.10.1 --sp 5061 -r 10000 -m 250 -c 5 -f \
         \n       Note: Please ensure destination network server is running before running this test.\
         \n"

        shortOption_list = 'p:r:m:c:f'
        longOption_list = ['si=', 'sp=', 'li=', 'lp=']
        extracted_option = extract_option_client(shortOption_list, longOption_list, clientOptionList)
        
        ## below code will print the option value if the the value set to 1
        if param_debug_display == 1:
            print(extracted_option)

        if len(extracted_option) > 0 and 'error' not in extracted_option:
            # Validate parameter value inputs
            if validateClientOptions(extracted_option):

                protocolType = extracted_option['-p']

                if (protocolType == 'tcp'):
                    tcpStartNetworkTest(extracted_option)

                elif (protocolType == 'udp'):
                    udpStartNetworkTest(extracted_option)
        

    def do_startNetworkServer(self, serverOptionList):
        "\n    Start the Network Test on the Server \
         \n    Command Syntax: \
         \n        startNetworkServer -p <protocol_type> --li <servers_local_ip> [--lp <servers_local_port_no>] \
         \n    Mandatory Parameters: \
         \n        -p   <protocol_type> \
         \n        --li <local_server_ip> \
         \n    Optional parameter: \
         \n        --lp <local_server_port_no>  (default TCP=40000, UDP=5061) \
         \n    Example: \
         \n        startNetworkServer -p tcp --li 10.1.10.1 --lp 10000 \
         \n"

        shortOption_list = 'p:'
        longOption_list = ['li=', 'lp=', 'drop_start=', 'drop_count=', 'skip_count=', 'max_drops=']
        extracted_option = extract_option_server(shortOption_list, longOption_list, serverOptionList)
        

        ## below code will print the option value if the the value set to 1
        if param_debug_display == 1:
            print(extracted_option)

        if len(extracted_option) >  0 and 'error' not in extracted_option:
            # Validate parameter value inputs
            if validateServerOptions(extracted_option):

                protocolType = extracted_option['-p']

                if (protocolType == 'tcp'):
                    tcpStartNetworkServer(extracted_option)

                elif (protocolType == 'udp'):
                    udpStartNetworkServer(extracted_option)
    
    def do_startMessageSizeTest(self, line):
        
        "\n    Start the Message Size Test on the client \
         \n    Command Syntax: \
         \n        startMessageSizeTest -p <protocol_type> --si <remote_server_ip> [--sp <remote_server_port_no>] \
         \n    Mandatory Parameters: \
         \n        -p   <protocol_type> \
         \n        --si <remote_server_ip> \
         \n    Optional parameter: \
         \n        --sp <remote_server_port_no>  ( default UDP=5061) \
         \n    Example: \
         \n        startMessageSizeTest -p udp --si 10.1.10.1 --sp 5061 \
         \n       Note: Please ensure destination network server is running before running this test.\
         \n"
        
        # short and long options
        longOption_list = ['si=', 'sp=']
        shortOption_list = 'p:'

        # extracting options
        extracted_option = extract_option_messageSizeTest(shortOption_list, longOption_list, line)
        
        # if options extracted and no error then we proceed
        if len(extracted_option) >  0 and 'error' not in extracted_option: 
            
            # validation done on the extracted options for correct IP, range and so on
            # no else leg as error will be returned from validateMessageTestOptions itself
            if validateMessageTestOptions(extracted_option):
               
                udpStartMessageSizeTest(extracted_option)
        
    def do_quit(self, line): 
 	"\n Exit Program \n"
	return True

    def do_exit(self, line):
	"\n Exit Program \n"
        return True
    
    def do_EOF(self, line):
        "\n Exit Program \n"
        return True

    # over-ride default empty line function to not do anything
    def emptyline(self):
        pass



def showCpuInfo(paramLine):

    """
    Param: paramline (not used)
    
    Desc:  Collect and display systems CPU information
    
    Return: Nothing
    """
	
    print(" ")
    print(" System CPU info : ")

    # get CPU model info from /proc/cpuinfo (note - picks first model in the file, so assumes all CPUs will be same)
    with open('/proc/cpuinfo') as f:
        for line in f:
            if line.rstrip('\n').startswith('model name'):
                model_name = line.rstrip('\n').split(':')[1].strip()

    try:
        # execute linux lscpu command and save output as dictionary key/value pairs
        lsCpuOutput,err = subprocess.Popen(["lscpu"],
											stdout=subprocess.PIPE,
											stderr=subprocess.PIPE, encoding='utf-8').communicate()
    except subprocess.CalledProcessError as e:
        print(e)
        print(" ")
        return
    except OSError as e :
        print(e)
        print(" ")
        return
    if err != "":
        print(err)
        print(" ")
        return

    # Create dictionary from output
    lsCpuDict={}
    for i in lsCpuOutput.splitlines():
        splitted = i.split(":")  # split into two fields at :
        # remove leading & trailing spaces, add the two fields as key,value pair to dict
        lsCpuDict[splitted[0].strip()] = splitted[1].strip()

    numCpus = lsCpuDict["CPU(s)"]
    numSockets = lsCpuDict["Socket(s)"]
    numCores = lsCpuDict["Core(s) per socket"]
    numThreads = lsCpuDict["Thread(s) per core"]
    cpuClockSpeed = lsCpuDict["CPU MHz"]

    try:
        cpuVirt = lsCpuDict["Virtualization"]
    except:
        try:
            cpuVirt = lsCpuDict["Virtualization type"]
        except:	
            cpuVirt = "unknown"

    print("  CPU Model          = %s (%s MHz)" % (model_name, cpuClockSpeed))
    print("  Number of CPU's    = %s ( %s Sockets * %s Cores * %s Threads)" % (numCpus,numSockets,numCores,numThreads))
    print("  CPU Virtualization = %s" % cpuVirt)
    print(" ")

    return


def showMemoryInfo(paramLine):

    """
    Param: paramline (not used)
    
    Desc:  Collect and display systems memory (RAM) information, used/free etc
    
    Return: Nothing
    """
	
    # Create dictionary for meminfo
    memDict={}

    # read lines from meminfo and store as dictionary key/value pairs
    with open('/proc/meminfo') as f:
        for line in f:
            splitted = line.split(":")  # split into two fields at :
            # remove leading & trailing spaces, add the two fields as key,value pair to dict
            memDict[splitted[0].strip()] = splitted[1].strip()

    # get memory strings memifo
    memTotal = memDict["MemTotal"]
    memFree = memDict["MemFree"]
    memBuffers = memDict["Buffers"]
    memCached = memDict["Cached"]

    # convert string values numeric float values
    memTotalKB = float(memTotal.split(" ")[0].strip())
    memFreeKB = float(memFree.split(" ")[0].strip())
    memBuffersKB = float(memBuffers.split(" ")[0].strip())
    memCachedKB = float(memCached.split(" ")[0].strip())
    memFreeActualKB = memFreeKB + memBuffersKB + memCachedKB

    # output the memory values (x/1048576 converts from kB to GB)
    print(" ")
    print(" System memory info : ")
    print("  Total Memory = %.2f GB, Free = %.2f GB, Buffers = %.2f GB, Cached = %.2f GB" \
             % (memTotalKB/1048576, memFreeKB/1048576, memBuffersKB/1048576, memCachedKB/1048576))
    print("  Actual Free  = %.2f GB (Free+Buffers+Cached)" % (memFreeActualKB/1048576))
    print(" ")

    return


def showPlatformInfo(paramLine):

    """
    Param: paramline (not used)
    
    Desc:  Collect and display systems platform information, includes node name, app type,
	       linux platform info, archirecture, dhcp, sriov, config drive and metadata info.
    
    Return: Nothing
    """

    print(" ")
    print(" System platform info : ")

    platformInfo = platform.platform(aliased=0,terse=0)
    platformNode = platform.node()
    platformArch = platform.architecture()
    appType,appPackage = getApplication()
    print("   Node name        = %s" % platformNode)
    print("   Application type = %s, Package name = %s" % (appType, appPackage))
    print("   Platform Info    = %s, Architecture = %s" % (platformInfo,platformArch))

    # check for dhcp
    if checkDhcpUsed() == True:
        print("   DHCP             = DHCP used")
    else:
        print("   DHCP             = DHCP NOT used")
 

    # check for SR-IOV support (Single Root I/O Virtualization)
    print("   SR-IOV support   = " + checkSRIOVSupport())

    # check for config drive
    configDrive = checkConfigDrive()
    print("   Config Drive     = %s" % configDrive)

    # check for metadata
    print("   Checking for Metadata .....")

    # storing meta data information
    metaDataInfo = getMetaData(configDrive)

    if len(metaDataInfo) > 0:
        print("    Metadata - available at %s" % metaDataInfo)

    else:
        print("    Metadata - Not available on this platform")
    print(" ")

    return


def getMetaData(configDriveStr):
    """
    Param: configDriveStr - to determine whether or not config drive present

    Desc:  check meta data information avialable or not

    Return: meta data file path if available 

    """
    metaDataPath = ""

    # if config drive not mounted then we check meta server for this info
    # configDriveStr will be 'Not Found' or "" if not mounted

    if configDriveStr == 'Not Found' or configDriveStr == '' or len(configDriveStr) == 0:
        metaText = getRequest(META_DATA_URI_OSTACK)

        if metaText is not None:
            metaDataPath = META_DATA_URI_OSTACK

    # if config drive mounted then file location gets assigned
    else:
        if os.path.isfile(META_DATA_FILE_SBX) == True:
            metaDataFileSize = os.path.getsize(META_DATA_FILE_SBX)
            if metaDataFileSize > 0:
                metaDataPath = META_DATA_FILE_SBX
        elif os.path.isfile(META_DATA_FILE_EMS_PSX) == True:
            metaDataFileSize = os.path.getsize(META_DATA_FILE_EMS_PSX)
            if metaDataFileSize > 0:
                metaDataPath = META_DATA_FILE_EMS_PSX

    return metaDataPath


def checkSRIOVSupport():

    """
    Param: None
    
    Desc:  Check if SR-IOV (virtual i/o) supported on system
    
    Return: string containing sriovSupport info or  NotSupported or unknown  
    """
	
    sriovSupport = "unknown"

    # execute linux lspci command and look for virtual ethernet contoller
    try:        
        lsPciOutput,err = subprocess.Popen(["lspci"],
		 								   stdout=subprocess.PIPE,
			 							   stderr=subprocess.PIPE, encoding='utf-8').communicate()

    except subprocess.CalledProcessError as e:
        return sriovSupport
    except OSError as e :
        return sriovSupport
    if err != "":
        return sriovSupport

    sriovSupport = "Not Supported"
    for line in lsPciOutput.splitlines():
        if "Ethernet controller:" in line:
	    if "Virtual Function" in line:
                splitted = line.split("Ethernet controller:")
                sriovSupport = "Supported - " + splitted[1].strip()
                break

    return sriovSupport



def checkDhcpUsed():

    """
    Param: None
    
    Desc:  Check if DHCP used by any interfaces on the system
    
    Return: boolean, True if dhcp used, False if dhcp not used
    """
	
    dhcpFound = False

    if ((os.path.isfile("/var/lib/dhcp/dhclient.mgt0.leases") == True) or
		(os.path.isfile("/var/lib/dhcp/dhclient.pkt0.leases") == True) or
		(os.path.isfile("/var/lib/dhcp/dhclient.pkt1.leases") == True) or
		(os.path.isfile("/var/lib/dhclient/dhclient--eth0.lease") == True)):
        dhcpFound = True

    return dhcpFound


def getRequest(url):

    """
    Param: url string
    
    Desc:  Does wget to specified url/file
    
    Return: None if remote url/file not accesible or output from wget if accessible
    """

    try:
        wgetOutput1,wgetOutput2 = subprocess.Popen(["wget", "--spider", "-T 3", "-t 1", url],
										 stdout=subprocess.PIPE,
										 stderr=subprocess.PIPE,encoding='utf-8').communicate()
		
    except subprocess.CalledProcessError as e:
        return None
    except OSError as e :
        return None
	
	# Note - wget output is returned in stderr, hence it will be in wgetOutput2
    if ("Remote file exists" in wgetOutput2):
        return wgetOutput2
    else:
        return None



def checkConfigDrive():

    """
    Param: None
    
    Desc:  Check if config drive (config-2)exists on system
    
    Return: string containing config drive path, or Not Found 
    """
	
    configDrive = "/dev/disk/by-label/config-2"
    configDriveBlkid = True
    try:
        # execute linux ls command to check config-2
        lsOutput,err = subprocess.Popen(["ls", configDrive],
										stdout=subprocess.PIPE,
										stderr=subprocess.PIPE , encoding='utf-8').communicate()
    except subprocess.CalledProcessError as e:
        configDrive = "Not Found"
    except OSError as e :
        configDrive = "Not Found"
    if err != "":
        configDrive = "Not Found"
	
    if configDrive == "Not Found":
        try:
            blkidOutput,err = subprocess.Popen(["blkid", "-t", "LABEL=\"config-2\"", "-odevice"],
											   stdout=subprocess.PIPE,
											   stderr=subprocess.PIPE,encoding='utf-8').communicate()
        except subprocess.CalledProcessError as e:
            configDriveBlkid = False
        except OSError as e :
            configDriveBlkid = False
        if err != "":
            configDriveBlkid = False
		
        if configDriveBlkid == True:
            configDrive = blkidOutput

    return configDrive


def getApplication():

    """
    Param: None
    
    Desc:  get application name and package name of installed application
    
    Return: applicationType string containing Unknown or SBC or PSX or EMS
	        applicationPackage string containing Unknown or application package name e.g. sbc-V05.01.00-A622
    """
	
    rpmFound = True
    applicationType = "Unknown"
    applicationPackage = "Unknown"

    try:
        # execute linux rpm -qa command to check for sbx, psx or ems
        rpmOutput,err = subprocess.Popen(["rpm", "-qa"],
										 stdout=subprocess.PIPE,
										 stderr=subprocess.PIPE,encoding='utf-8').communicate()
    except subprocess.CalledProcessError as e:
        rpmFound = False
    except OSError as e :
        rpmFound = False
    if err != "":
        rpmFound = False
		
    if rpmFound == True:
        for rpmLine in rpmOutput.splitlines():
            if rpmLine.startswith("sbc-"):
                applicationType = "SBC"
                applicationPackage = rpmLine.rstrip('\n').split('.x86_64')[0].strip()
                break

            elif rpmLine.startswith("SONSss-"):
                applicationType = "PSX"
                applicationPackage = rpmLine
                break

            elif rpmLine.startswith("SONSems-"):
                applicationType = "EMS"
                applicationPackage = rpmLine
                break

    # if application type wasnt found via rpm -qa, do a simple directory structure check for sbx
    if applicationType == "Unknown":
        if os.path.isdir("/opt/sonus/sbx") == True:
            applicationType = "SBC"
            if os.path.isfile("/opt/sonus/staging/appInstall.out") == True: #try and get package name from appInstall.out
                with open('/opt/sonus/staging/appInstall.out') as f:
                    for line in f:
                        if line.rstrip('\n').startswith('sbc-V'):		
                            applicationPackage = line.rstrip('\n').split('.x86_64')[0].strip()

    return applicationType,applicationPackage


def showStorageInfo(paramLine):

    """
    Param: paramline (not used)
    
    Desc:  Collect and display systems storage (hard drive) information. Excludes tmpfs and only includes /dev devices
    
    Return: Nothing
    """
	
    print(" ")
    print(" System storage info : ")

    try:
        # invoke df command to get info on storage devices
        dfOutput,err = subprocess.Popen(["df", "-hT", "-x devtmpfs -x tmpfs -x vfat"],
										 stdout=subprocess.PIPE,
										 stderr=subprocess.PIPE , encoding='utf-8').communicate()

    except subprocess.CalledProcessError as e:
        print(e)
        print(" ")
        return
    except OSError as e:
        print(e)
        print(" ")
        return										
    if err != "":
        print(err)
        print(" ")
        return											

    for dfLine in dfOutput.splitlines():
        if dfLine.startswith("/dev/") or dfLine.startswith("Filesystem"):
            print("   %s" % dfLine)
    print(" ")

    return


def showNetworkInfo(paramLine):

    """
    Param: paramline (not used)
    
    Desc:  Collect and display systems network interfaces information. Includes eth, mgt and pkt interfaces
	       Excludes ppkt and pmgt interfaces that would not be used to send/rx packets by tool
    
    Return: Nothing
    """

    print(" ")
    print(" System network interfaces info : ")

    try:
        net_iface = os.listdir('/sys/class/net/') # get list of network interfaces
    except OSError as e:
        print(e)
        print(" ")
        return	        

    skip_if_list = ['ppkt', 'pmgt', 'int0', 'int1', 'lo', 'pnps']
    for x in net_iface:
        if not any(word in x for word in skip_if_list):            		
            try:
                # invoke ifconfig to get info on network interfaces
                ifConfigOutput,err = subprocess.Popen(["/sbin/ifconfig", x],
												  stdout=subprocess.PIPE,
												  stderr=subprocess.PIPE, encoding='utf-8').communicate()
            except subprocess.CalledProcessError as e:
                print(e)
                print(" ")
                return
            except OSError as e:
                print(e)
                print(" ")
                return										
            if err != "":
                print(err)
                print(" ")
                return

            for ifConfigLine in ifConfigOutput.splitlines():
                if ((x+":" in ifConfigLine) or
					("Ethernet" in ifConfigLine) or
					(ifConfigLine.strip().startswith("inet")) or
					(ifConfigLine.strip().startswith("UP")) or
					(ifConfigLine.strip().startswith("DOWN"))):
					
                    print("   %s" % ifConfigLine)
            print(" ")

    return


def showFullSystemInfo(paramLine):
    """
    Param: paramline, checked for -f flag, if set output written to a file as well
    
    Desc:  Invokes the various show and checkDiskPerformance commands to display all info on system
    
    Return: Nothing
    """
	
    # check if -f option is specified to write output to a file
    args_list = paramLine.split()
    try:
        opts, args = getopt.getopt(args_list, 'f')
    except getopt.GetoptError as err:
        print(str(err)) 
        return
	
    fileWrite = False
    for o, a in opts:
        if o == "-f":
            fileWrite=True

    #getting hostname
    nodeName = platform.node() if len(platform.node()) > 0 else 'Unknown'
    
    #getting application type i.e. SBX or PSX
    appType,appPackage = getApplication()
    
    appType = appType if len(appType) > 0 else 'Unknown'
    
    stdout = sys.stdout # save original stdout

    timestr = time.strftime("%Y%m%d-%H%M%S")
    logfile= appType + '_' + nodeName + '_' + "fullsysinfo_" + timestr + ".txt"

    if fileWrite == True:
        sys.stdout = Logger(logfile) # set stdout to logger which will write to terminal and specified log file

    print("")
    print(" Full System Info, collected on - " + time.strftime("%c"))

    showPlatformInfo(paramLine)
    showCpuInfo(paramLine)
    showMemoryInfo(paramLine)
    showStorageInfo(paramLine)
    showNetworkInfo(paramLine)
    checkDiskPerformance(paramLine)

    if fileWrite == True:
        sys.stdout = stdout # revert output back to original stdout
        print(" Test output also written to file  - " + logfile)
		
    return

def checkDiskPerformance(paramLine):
    """
    Param: paramline (not used)
    
    Desc:  Basic meaurement of read and write performance of hard drive by
	       -  Writing 10K data records 1,000 times to disk
		   -  Copying 10K data records 1,000 times from disk to null stream
		   Output shows time taken to do read and write of data and read/write of data speed in MB/sec
    
    Return: Nothing
    """

    print(" ")
    print(" Basic disk IO performance (may take a few secs to complete) : ")

    print("    Performing write test  - Writing 10K data records 1,000 times to disk : ")
    try:
        # invoke dd command to check disk write
        ddOutput1,ddOutput2 = subprocess.Popen(["dd", "if=/dev/zero", "of=testfile_dd", "bs=10K", "count=1000", "oflag=direct"],
											   stdout=subprocess.PIPE,
											   stderr=subprocess.PIPE, encoding='utf-8').communicate()
    except subprocess.CalledProcessError as e:
        print(e)
        print(" ")
        return
    except OSError as e:
        print(e)
        print(" ")
        return										

	# Note - dd output is returned in stderr, hence it will be in ddOutput2
    if ddOutput2 == "":
		print("            Error - No output from dd command")
    else:
        for ddLine in ddOutput2.splitlines():
            print("        %s" % ddLine)
        print(" ")

    print("    Performing read test  - Copying 10K data records 1,000 times from disk to null stream: ")
    try:
        # invoke dd command to check disk copy to null stream
        ddOutput1,ddOutput2 = subprocess.Popen(["dd", "if=testfile_dd", "of=/dev/null", "bs=10K", "count=1000"],
											   stdout=subprocess.PIPE,
											   stderr=subprocess.PIPE,encoding='utf-8').communicate()
    except subprocess.CalledProcessError as e:
        print(e)
        print(" ")
        return
    except OSError as e:
        print(e)
        print(" ")
        return
	
	# Note - dd output is returned in stderr, hence it will be in ddOutput2
    if ddOutput2 == "":
		print("            Error - No output from dd command")
    else:
        for ddLine in ddOutput2.splitlines():
            print("        %s" % ddLine)
        print(" ")


   # delete the test file
    try:
        dfOutput,err = subprocess.Popen(["rm", "-f", "testfile_dd"],
										stdout=subprocess.PIPE,
										stderr=subprocess.PIPE,encoding='utf-8').communicate()
    except subprocess.CalledProcessError as e:
        print(e)
        print(" ")
        return
    except OSError as e:
        print(e)
        print(" ")
        return										
    if err != "":
        print(err)
        print(" ")
        return	

    return



def extract_option_server(short_option, long_option, argv_list):

    """
    Param: Short (i.e. -p, -r, -m) & Long (i.e. --li, --si, --lp) options, Arguments passed from commnad line
    
    Desc:  Method will take above parameters and extract, short and long options. For any missing option - in case of an mandatory
    parameter will throw an error, otherwise fill it with a default value
    
    Return: A dictionary with options and their respective values i.e. {'-p': 'tcp', '--li': 'xxx.xxx.xxx.xxx' ...}
    """
    
    return_params = {}
    args_list = argv_list.split()
    localPort = 0
    
    try:
        opts, args = getopt.getopt(args_list, short_option, long_option)
        return_params = {x[0]:x[1] for x in opts}

        if '-p' in return_params and len(return_params['-p']) == 3 and return_params['-p'] in ['tcp', 'udp']:
            
            protocol_type = return_params['-p']
            
            # checking correct mandatory field exist
            if not ('-p' in return_params) or not('--li' in return_params):
                print('      Error: Protocol & Local IP are mandatory fields and required a value.')
                print('      Usage: -p value --li value')
                return_params['error'] = '1'
            #if mandatory field exist then we process
            else:
                if '--lp' not in return_params  and protocol_type == 'tcp':
                    localPort = '40000'
                elif '--lp' not in return_params and protocol_type == 'udp':
                    localPort = '5061'
                
                if localPort > 0:
                    return_params['--lp'] = localPort
                
                if '--drop_start' not in return_params :
                    return_params['--drop_start'] = '0'
                if '--drop_count' not in return_params :
                    return_params['--drop_count'] = '0'
                if '--skip_count' not in return_params :
                    return_params['--skip_count'] = '0'
                if '--max_drops' not in return_params :
                    return_params['--max_drops'] = '0'      
                # double checking if a option has a missing value
                tmp_dict = {key : val for key, val in return_params.items() if return_params[key] in ['', ' ']}
                
                # following code catching error where user entered an option but missing its value
                if len(tmp_dict) > 0:
                    print('      Error: Missing value for a option.')
                    print('      Usage: -p value --li value ....')
                    return_params['error'] = '1'
                  
         # if a -p value missing throw an error
        else:
            print('      Error: Protocol (-p) is a mandatory field and required a value.')
            print('      Usage: -p tcp or -p udp.')
            return_params['error'] = '1'
    
    except getopt.GetoptError as err:
        
        print(str(err))
  
    return return_params


def extract_option_client(short_option, long_option, argv_list):

    """
    Param: Short (i.e. -p, -r, -m) & Long (i.e. --li, --si, --lp) options, Arguments passed from commnad line
    
    Desc:  Method will take above parameters and extract, short and long options. For any missing option - in case of an mandatory
    parameter will throw an error, otherwise fill it with a default value
    
    Return: A dictionary with options and their respective values i.e. {'-p': 'tcp', '--li': 'xxx.xxx.xxx.xxx' ...}
    """

    return_params = {}
    args_list = argv_list.split()
    try:
        opts, args = getopt.getopt(args_list, short_option, long_option)
        return_params = {x[0]:x[1] for x in opts} # populating return params with argument list
        
        # if protocol in dictionary and has a value length of 3  and correct protocol then we process it
        if '-p' in return_params and len(return_params['-p']) == 3 and return_params['-p'] in ['tcp', 'udp']:
            protocol_type = return_params['-p']
            
            # checking correct mandatory field exist for tcp
            if protocol_type == 'tcp' and ('--si' not in return_params or '--li' not in return_params):
                print('      Error: Protocol, Server IP & Local IP are mandatory fields and required a value.')
                print('      Usage: -p value --si value --li value ')
                return_params['error'] = '1'
            
            # checking correct mandatory field exist for udp
            elif protocol_type == 'udp' and '--si' not in return_params:
                print('      Error: Protocol & Server IP are mandatory fields and required a value.')
                print('      Usage: -p value --si value ')
                return_params['error'] = '1'
            
            #if mandatory field exist then we process
            else:
                ############ Code Block to be update [start here] just uncomment below code upto update finish indicator
                if '--sp' not in return_params and protocol_type == 'tcp':
                    return_params['--sp'] = '40000'
                elif '--sp' not in return_params and protocol_type == 'udp':
                    return_params['--sp'] = '5061'

                if '--lp' not in return_params and protocol_type == 'tcp':
                    return_params['--lp'] = '20000'

                if '-r' not in return_params:
                    return_params['-r'] = '1000'

                if '-m' not in return_params:
                    return_params['-m'] = '500'

                if '-c' not in return_params:
                    return_params['-c'] = '1'

                ## double checking a missing option value 
                tmp_dict = {key : val for key, val in return_params.items() if return_params[key] in ['', ' '] and key not in '-f'}
                
                if len(tmp_dict) > 0:
                    print('      Error: Missing value for a mandatory option.')
                    print('      Usage: -p value --si value --li value ')
                    return_params['error'] = '1'

         # if a -p value missing throw an error
        else:
            print('      Error: Protocol (-p) is a mandatory field and required a value.')
            print('      Usage: -p tcp or -p udp.')
            return_params['error'] = '1'

    except getopt.GetoptError as err:
        
        print(err)          

    return return_params

def extract_option_messageSizeTest(short_opt, longOption_list, argv_list):

    """
    Param: Long (i.e. --si) options, Argv
    
    Desc:  Method will take above parameters and extract options. For any missing option - in case of an mandatory
    parameter will throw an error, otherwise fill it with a default value
    
    Return: A dictionary with options and their respective values i.e. {'-p': 'udp', '--si': 'xxx.xxx.xxx.xxx' ...}
    """

    return_params = {}
    args_list = argv_list.split()
    try:
        opts, args = getopt.getopt(args_list, short_opt, longOption_list)
        return_params = {x[0]:x[1] for x in opts} # populating return params with argument list
        
        # if protocol in dictionary and has a value length of 3  and correct protocol then we process it
        if '-p' in return_params and len(return_params['-p']) == 3 and return_params['-p'] in ['udp']:
            protocol_type = return_params['-p']
            
            # checking correct mandatory field exist for udp
            if protocol_type == 'udp' and '--si' not in return_params:
                print('      Error: Protocol & Server IP are mandatory fields and required a value.')
                print('      Usage: -p value --si value ')
                return_params['error'] = '1'
            
            #if mandatory field exist then we process
            else:
                
                if '--sp' not in return_params and protocol_type == 'udp':
                    return_params['--sp'] = '5061'


                ## double checking a missing option value 
                tmp_dict = {key : val for key, val in return_params.items() if return_params[key] in ['', ' ']}
                
                if len(tmp_dict) > 0:
                    print('      Error: Missing value for a mandatory option.')
                    print('      Usage: -p value --si value')
                    return_params['error'] = '1'

         # if a -p value missing throw an error
        else:
            print('      Error: Protocol (-p) is a mandatory field and required a value.')
            print('      Usage: -p udp ')
            print('      Note : ONLY UDP SUPPORTED FOR MESSAGE SIZE TEST.')
            return_params['error'] = '1'

    except getopt.GetoptError as err:
        
        print(err)          

    return return_params


def get_socket_options(packetPort):

    """
    Param: int packetPort
    Desc:  This method takes a packet port and extracts the Lif Group Index and Address Context for for that port. If not found
	       returns '0' for both variables
    Return: int lifgIndex, int address_context
	         
    """

    lifgIndex = 0
    addrContext = 0

    # Execute 'ip' command to extract the IP Link for pkt0
    ipLink,err = subprocess.Popen(['ip', 'link', 'show', 'dev', packetPort],
                                    stdout=subprocess.PIPE, 
                                    stderr=subprocess.PIPE, encoding='utf-8').communicate()
    lifGroup = ipLink.split(":")[0]         # Extract the first variable up to the first semi-colon

    try:
        with open('/proc/net/nrs') as nrs_info:
            for line in nrs_info:
                if line.startswith('LifgTable: ' + str(lifGroup)): # Find the correct Lif Group
                    lifgtable = line.split(' ')
                    lifgIndex = lifgtable[2]    # Extract the Lif Group Index
                    addrContext = lifgtable[3]  # Extract the Address Context
                    break

    except subprocess.CalledProcessError as e:
        print(e)
        print(" ")
    except OSError as e:
        print(e)
        print(" ")

    return lifgIndex, addrContext


def get_packet_route(serverIp):

    """
    Param: string serverIp
    Desc:  This method takes a serverIp and determines if this is packet port confiugred on the system. If not found
           returns string 'none'
    Return: int lifgIndex, int address_context
	         
    """

    #Execute 'ip route get <serverIp>' command to get the packet port to be used by the client
    packetPort = "none"

    try:
        packetRoute,err = subprocess.Popen(['ip', 'route', 'get', serverIp],
                                             stdout=subprocess.PIPE,
                                             stderr=subprocess.PIPE,encoding='utf-8').communicate()

        if ("pkt" in packetRoute):
            split_line = packetRoute.split(' ')
            for i in split_line:
                if ("pkt" in i):
                    packetPort = i
                    break

    except subprocess.CalledProcessError as e:
        print(e)
        print(" ")
    except OSError as e:
        print(e)
        print(" ")

    return packetPort


def extractIpType(ip_string): # this function will extract IP type
    
    """
    Param: ip string
    
    Desc:  Method will take a ip string and check if it is a ipv 4 or 6
    
    Return: a string '4' in case of ipv 4 or '6' in the case of ipv 6
    """

    ip_type = ''
    if validate_ipFour(ip_string):
        ip_type = '4'
    elif validate_ipSix(ip_string):
        ip_type = '6'

    return ip_type 


def validate_ip(ip_string): # this function will validate a given ip regardless of type
    
    """
    Param: ip string
    
    Desc:  Method will take a ip string and check if it is a valid ipv 4 or 6
    
    Return: return True in the case of either ip 4 or 6, False otherwise
    """

    valid_ip = False
    if validate_ipFour(ip_string):
        valid_ip = True
    elif validate_ipSix(ip_string):
        valid_ip = True
        
    return valid_ip


def validate_ipSix(ip_string): # this function validates IPv 6

    """
    Param: ip string
    
    Desc:  Method will take a ip string and check if it is a valid ipv 6 or not
    
    Return: return True if the given string is a valid ip6, otherwise return False
    """

    try:
        socket.inet_pton(socket.AF_INET6, ip_string)
    
    except socket.error:  # not a valid address
        
        return False
    
    return True   


def validate_ipFour(ip_string): # this function validates IPv 4
	# checks if a given IP address is in correct format or not

    """
    Param: IPv 4 string
    
    Desc:  Method will take a ip string and check if it is a valid ipv 4 or not
    
    Return: return True if the given string is a valid ip4, otherwise return False
    """
    splitted_ip_list = ip_string.split('.')
    if len(splitted_ip_list) != 4:
        return False
    for each_ip_part in splitted_ip_list:
        if not each_ip_part.isdigit():
            return False
        i = int(each_ip_part)
        if i < 0 or i > 255:
            return False
    return True


def expand_ipv6(ip):

    """
    Param: Valid IP 6
    
    Desc: This method works in two part 

        1. The first part expand a given ip 
            i.e XXXX::Y:Z to XXXX:XXXX:XXXX:XXXX:XXXX:XXXX:Y:Z
        2. The second part fully exapnd an IP that has been expanded in step 1 
            i.e XXXX:XXXX:XXXX:XXXX:XXXX:XXXX:Y:Z to XXXX:XXXX:XXXX:XXXX:XXXX:XXXX:000Y:0000Z
    
    Return: return a fully expanded IP 6 string
    """
    #below code will processed an ipv 6 fc00:0:0::0000:05::19 into 0000:0:0:0000:0000:05:0000:19
    ipString = ip
    location_list = [(k, len(list(g))) for k, g in groupby(ipString)]
    start_pointOfSplit = 0
    for k, v in location_list:
        
        if v == 2 and ':' in k:
            break
        else:
            start_pointOfSplit += v
    firstPart = ipString[0:start_pointOfSplit]
    lastPart = ipString[(start_pointOfSplit+2):]

    if ':' in firstPart:
        firstPart = firstPart.split(':')
    else:
        firstPart = list([firstPart])
        
    if ':' in lastPart:
        lastPart = lastPart.split(':')
    else:
        lastPart = list([lastPart])

    if not firstPart or '' in firstPart:
        total_val = len(lastPart)
    elif not lastPart or '' in lastPart:
        total_val = len(firstPart)
    else:
        total_val = len(firstPart) + len(lastPart)

    middleValToBeFilled = 8 - total_val
    count = 1
    middlePart=''
    while count <= middleValToBeFilled:
        middlePart = middlePart +':'+ '0000'
        count += 1
    firstPartStr = ''
    lastPartStr = ''
    for eachVal in firstPart:
        if '' not in firstPart:
            firstPartStr = firstPartStr + ':' + eachVal

        
    for eachVal in lastPart:
        if '' not in lastPart:
            lastPartStr = lastPartStr + ':' + eachVal


    expandedIP = firstPartStr + middlePart + lastPartStr

    expandedIP = expandedIP[1:]
    ## below code will processed an ipv 6 into 0000:0000:0000:0000:0000:0000:0000:0000
    ip_splitted = expandedIP.split(':')
    fully_expanded_addr = ''

    for value in ip_splitted:
        if len(value) < 4:
            tmp_length = len(value)
            while tmp_length < 4:
                value = '0' + value
                tmp_length += 1
        fully_expanded_addr = fully_expanded_addr +':'+value
    numTimeColonAppear = fully_expanded_addr.count(':')

    while numTimeColonAppear < 8:
        fully_expanded_addr = fully_expanded_addr +':'+'0000'
        numTimeColonAppear +=1
       
    return fully_expanded_addr[1:]
    


def ipAddressExist(input_ip, ipType):

    """
    Param: Two input parameters - IP string and IP type
    
    Desc:  This method will validates whether a given IP address exist on the system based on the type provided as input parameter
    
    Return: Returns '1' if exist otherwise '0' and 'pkt' if port type is PKT otherwise '0' 
    """

    exist = '0'
    pType = "" # this will return whether a port is pkt or mgt
    net_iface = os.listdir('/sys/class/net/') # get list of network interfaces
    
    if ipType == '4':
        for x in net_iface:
            f = os.popen("/sbin/ifconfig '"+x+"' | grep -w 'inet'")
            IPretrived = f.read()
            pattern = r"((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)([ (\[]?(\.|dot)[ )\]]?(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)){3})"
            ips_list = [match[0] for match in re.findall(pattern, IPretrived)]
            
            if len(ips_list) > 0:
                for ip in ips_list:
                    if ip == input_ip:
                        exist = '1'
                        pType = x
                        break

    if ipType == '6':

        for x in net_iface:

            f = os.popen('ip addr show '+ x +' | awk "/inet6/{print $3}"')
            grepLine = f.read()

            if '\n' in grepLine:
                grepLine = grepLine.split('\n') # from ip addr show there will be at least one newline

            grepLine = [k for k in grepLine if k not in ''] # removing any empty value exist from split

            count =0
            tmp = len(grepLine)
            while count < tmp:
                grepLine[count] = grepLine[count].strip()
                if '/' in grepLine[count]:
                    slashPosition = grepLine[count].find('/')
                    grepLine[count] = grepLine[count][6:slashPosition] # string will contain inet6 so this line extracrting everything in between
                    if (expand_ipv6(grepLine[count]) == expand_ipv6(input_ip)):
                        exist = '1'
                        pType = x

                        break 
                count += 1

    if 'pkt' not in pType:
        pType = '0'

    return exist, pType


def tcpReceiveThread(sock, total_messages, packet_size, packetListAll):

    """
    Param: socket sock, int total_messages, int packet_size, list packetListAll

    Desc:  This method starts a Client TCP thread to receive packets returned from the Server.
           When the packets are received a current timestamp is taken which is recorded in a
           list for that packet sequence number. The thread will exit when all packets are
           received for that test or on socket timeout (5 secs) if packes are dropped.

    Return: thread rx
	         
    """

    global received
    global terminateSocket

    received = 0
    terminateSocket = False

    packetListAllStop = []

    for eachPacket in range(total_messages):
                packetListAllStop.append(packetData(eachPacket, None, None))

    while received < total_messages:
        try:
            data = ""
            while len(data) < packet_size:
                packet = sock.recv(packet_size - len(data))
                if not packet:
                    terminateSocket = True
                    break
                data += packet

            # Generate the end time and add it to the packet
            ending = "%.6f" % time.time()

            # Extract the Sequence Number from the packet received
            tmpSeqNo = data.split(" ")[0]
            tmpSeqNo = int(tmpSeqNo) - 1

            # Add Stop Time to the List of Packets
            packetListAllStop[tmpSeqNo].stopTime = ending
            
            received = received + 1

        except socket.timeout:
            print("Socket Timeout")
            terminateSocket = True
            break

    if (received != 0):
        for eachPacket in packetListAllStop:
            seqNo = eachPacket.seqId
            packetListAll[seqNo].stopTime = eachPacket.stopTime

    sock.close()

    return


def udpReceiveThread(sock, total_messages, packet_size, packetListAll):

    """
    Param: socket sock, int total_messages, int packet_size, list packetListAll

    Desc:  This method starts a Client UDP thread to receive packets returned from the Server.
           When the packets are received a current timestamp is taken which is recorded in a
           list for that packet sequence number. The thread will exit when all packets are
           received for that test or on socket timeout (5 secs) if packes are dropped.

    Return: thread rx
	         
    """
	
    global received
    global terminateSocket

    received = 0
    terminateSocket = False
    packetListAllStop = []

    for eachPacket in range(total_messages):
                packetListAllStop.append(packetData(eachPacket, None, None))

    try:
        while received < total_messages:
            try:
                data, address = sock.recvfrom(packet_size)

                # Generate the end time and add it to the packet
                ending = "%.6f" % time.time()

                # Extract the Sequence Number from the packet received
                tmpSeqNo = data.split(" ")[0]
                tmpSeqNo = int(tmpSeqNo) - 1

                # Add Stop Time to the List of Packets
                packetListAllStop[tmpSeqNo].stopTime = ending
            
                received = received + 1

            except socket.timeout:
                print("Socket Timeout")
                terminateSocket = True
                break

    finally:
        if (received != 0):
            for eachPacket in packetListAllStop:
                seqNo = eachPacket.seqId
                packetListAll[seqNo].stopTime = eachPacket.stopTime

        sock.close()

    return


def tcpStartNetworkTest(clientOptions):

    """
    Param: dict clientOptions

    Desc:  This method runs the TCP Network Test from the Client. Having extracted the user
           options, a socket is setup for TCP and connects to the Server. Once connection is
           estabished the Client receive thread is called. The test starts by sending packets
           to the Server. Once the test is complete call the generate test output if required.

    Return: 
	         
    """

    generateStats = True
    repeatCount = 0
    packetListAll = []

    # Extract Parameter Options to run the Client
    protocolType = clientOptions['-p']

    # Extract Server info
    serverIp = clientOptions['--si']
    serverPort = int(clientOptions['--sp'])

    # Extract the IP type from the IP Address provided
    ipType = extractIpType(serverIp)

    # Extract Client info
    clientIp = clientOptions['--li']
    clientPort = int(clientOptions['--lp'])

    # Note the ipType of the client and server must be the same and validated at this point
    if (ipType == "4"):
        serverAddress = (serverIp, serverPort)
        clientAddress = (clientIp, clientPort)
    elif (ipType == "6"):
        serverAddress = (serverIp, serverPort, 0, 0)
        clientAddress = (clientIp, clientPort, 0, 0)

    # Extract the No of Messages 
    no_messages = int(clientOptions['-r'])

    # Extract the Packet Size
    packet_size = int(clientOptions['-m'])

    # Extract the Repeat Count
    repeatCount = int(clientOptions['-c'])

    try:
        if (ipType == "4"):
            #Create a TCP/IP socket
            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            
        elif (ipType == "6"):
            sock = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)

        sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)

        sock.settimeout(5.0)

        # Get the Application Type, if its not SBX then test on the Management Port
        appType, appPackage = getApplication()

        # Only set the socket options for the SBC for the Packet Ports
        if (appType == "SBC"):

            ipExist, pID = ipAddressExist(clientIp, ipType)

            if ("pkt" in pID):

                lifgIndex, addrContext = get_socket_options(pID)

                if (lifgIndex != 0 and addrContext != 0):
                    # Setup the following socket options to get the Linux Kernal binding correctly
                    # for the Clients LIF Group Index and the Address Context on the SBC
                    #
                    # LifgIndex - #define IP_LIFG    29
                    sock.setsockopt(socket.IPPROTO_IP, 29, int(lifgIndex))

                    # Address Context - #define IP_AC    30
                    sock.setsockopt(socket.IPPROTO_IP, 30, int(addrContext))
                else:
                    print("Error: Packet port not configured for LifgIndex or Address Context")
                    generateStats = False
                    return

        try:
        # Bind on Client Address
            sock.bind(clientAddress)
            try:
                sock.connect(serverAddress)

                # Use thread to concurrently run the Client Receiver (with the Client Sender)
                try:
                    rx = Thread(target=tcpReceiveThread, args=(sock, no_messages*repeatCount, packet_size, packetListAll))
                    rx.start()
        
                    pktSeqNo = 1
                    msgCount = 0

                    # Create a default packet size of Users request
                    packetFill = "x" * (packet_size)

                    # Repeat the test dependent on Users request - Defualt 1 
                    for repeat in range(repeatCount):
                        start_repeat_time = "%.6f" % time.time()

                        # Run the test for the number of messages requested and packet size
                        for msgCount in range(no_messages):

                            # Generate the Test Packet - Including Sequence Number and Start Timestamp
                            starting = "%.6f" % time.time()
                            pktSeqNoStr = str(pktSeqNo) + " "
                            pktSeqNoLen = len(pktSeqNoStr)
                
                            # Create a test packet by instering the Sequence Number in the created test packet
                            test_packet = packetFill[0:0] + pktSeqNoStr + packetFill[pktSeqNoLen+0:]

                            # Add Sequence Number and Start Time to Packet List 
                            packetListAll.append(packetData(pktSeqNo, starting, None))

                            sock.sendall(test_packet)

                            # Increment the Packet sequence number
                            pktSeqNo = pktSeqNo + 1

                            if ((pktSeqNo % 10000) == 0):
                                print(".", end=' ', file=sys.stderr)
                                
                        # Delay for the remainder of the second if the packets sent took less than 1 second
                        stop_repeat_time = "%.6f" % time.time()
                        repeat_process_time = float(stop_repeat_time) - float(start_repeat_time)
                        
                        if (float(repeat_process_time) < 1.0):
                            time.sleep(1.0 - float(repeat_process_time))

                    # Wait until all packets received
                    while received < (no_messages*repeatCount):
                        if terminateSocket == True:
                            if (received == 0):
                                generateStats = False
                            break
                        pass

                except:
                    print("Error: Unable to start Client receive thread")
                    generateStats = False

            except socket.error as msg:
                print("Socket Error: %s" % msg)
                generateStats = False 

        except socket.error as msg:
            print("Socket Error: %s" % msg)
            generateStats = False

    except socket.error as msg:
        print("Socket Error: %s" % msg)
        generateStats = False

    finally:
        if generateStats != False:
            #storing variables to an array for result generation
            userInputs = {}
            userInputs['protocolType'] = protocolType
            userInputs['serverIPAndPort'] = serverIp+' '+str(serverPort)
            
            userInputs['localIPAndPort'] = clientIp+' '+str(clientPort)
              
            dumpexist, portTyp = ipAddressExist(clientIp, ipType)
            userInputs['portType'] = portTyp

            # flag raised for result to be stored or not
            userInputs['resultToBeStored'] = '0'
            if '-f' in clientOptions:
                userInputs['resultToBeStored'] = '1'
                

            userInputs['no_messages'] = str(no_messages)
            userInputs['packet_size'] = str(packet_size)
            userInputs['repeatCount'] = str(repeatCount)
            
            generatePerfomanceStats((no_messages*repeatCount), packetListAll, userInputs)

    return


def udpStartNetworkTest(clientOptions):

    """
    Param: dict clientOptions

    Desc:  This method runs the UDP Network Test from the Client. Having extracted the user
           options, a socket is setup for UDP and the Client receive thread is called. The
           test starts by sending packets to the Server. Once the test is complete call the
           generate test output if required.

    Return: 
	         
    """
	
    generateStats = True
    repeatCount = 0
    packetListAll = []

    # Extract Parameter Options to run the Client
    protocolType = clientOptions['-p']

   # Extract Parameter Options to run the Client
    serverIp = clientOptions['--si']
    serverPort = int(clientOptions['--sp'])

    # Extract the IP type from the IP Address provided
    ipType = extractIpType(serverIp)

    if (ipType == "4"):
        serverAddress = (serverIp, serverPort)
    elif (ipType == "6"):
        serverAddress = (serverIp, serverPort, 0, 0)

    # Extract the No of Messages 
    no_messages = int(clientOptions['-r'])

    # Extract the Packet Size
    packet_size = int(clientOptions['-m'])

    # Extract the Repeat Count
    repeatCount = int(clientOptions['-c'])

    total_no_messages = no_messages*repeatCount

    try:
        if (ipType == "4"):
            # Create a UDP/IP socket
            sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        elif (ipType == "6"):
            sock = socket.socket(socket.AF_INET6, socket.SOCK_DGRAM)

        sock.settimeout(5.0)

        # Get the Application Type, if its not SBX then test on the Management Port
        appType, appPackage = getApplication()

        # Only set the socket options for the SBC for the Packet Ports
        if (appType == "SBC"):

            packetPort = get_packet_route(serverIp)

            if ("pkt" in packetPort):

                lifgIndex, addrContext = get_socket_options(packetPort)

                if (lifgIndex != 0 and addrContext != 0):
                    # Setup the following socket options to get the Linux Kernal binding correctly
                    # for the Clients LIF Group Index and the Address Context on the SBC
                    #
                    # LifgIndex - #define IP_LIFG    29
                    sock.setsockopt(socket.IPPROTO_IP, 29, int(lifgIndex))

                    # Address Context - #define IP_AC    30
                    sock.setsockopt(socket.IPPROTO_IP, 30, int(addrContext))
                else:
                    print("Error: Packet port not configured for LIF Group Index or Address Context")
                    return 

        # Use thread to concurrently run the Client Receiver (with the Client Sender)
        try:
            rx = Thread(target=udpReceiveThread, args=(sock, total_no_messages, packet_size, packetListAll))
            rx.start()

            pktSeqNo = 1
            msgCount = 0
            burstRange = 50
            messages_per_sec = 50
            burstRangeDelay = 0.0

            if (no_messages > burstRange):
                messages_per_sec = no_messages

            burstRangeDelay = float( float(1.0)/float(messages_per_sec) * float(burstRange))

            if (no_messages < burstRange):
                burstRange = no_messages

            # Create a default packet size of Users request
            packetFill = "x" * (packet_size)

            # Run the test for the number of messages requested and packet size
            for msgCount in range(total_no_messages):

                    # Generate the Test Packet - Including Sequence Number and Start Timestamp
                    starting = "%.6f" % time.time()
                    pktSeqNoStr = str(pktSeqNo) + " "
                    pktSeqNoLen = len(pktSeqNoStr)
                
                    # Create a test packet by instering the Sequence Number in the created test packet
                    test_packet = packetFill[0:0] + pktSeqNoStr + packetFill[pktSeqNoLen+0:]

                    # Add Sequence Number and Start Time to Packet List 
                    packetListAll.append(packetData(pktSeqNo, starting, None))

                    # Send test packet to Server
                    sent = sock.sendto(test_packet, serverAddress)

                    # Increment the Packet sequence number
                    pktSeqNo = pktSeqNo + 1

                    if ((pktSeqNo % 10000) == 0):
                        print(".", end=' ', file=sys.stderr)

                    if msgCount % burstRange == 0:
                        time.sleep(burstRangeDelay)

            while received < (total_no_messages):
                if terminateSocket == True:
                    if (received == 0):
                        generateStats = False
                    break
                pass

        except:
            print("Error: Unable to start Client receive thread")
            generateStats = False

    except socket.error as msg:
        print("Socket Error: %s" % msg)
        generateStats = False

    finally:

        if generateStats != False:
            #storing variables to an array for result generation
            userInputs = {}
            userInputs['protocolType'] = protocolType
            userInputs['serverIPAndPort'] = serverIp+' '+str(serverPort)
            
            packetPort = get_packet_route(serverIp)
            userInputs['portType'] = packetPort

            # flag raised for result to be stored or not
            userInputs['resultToBeStored'] = '0'
            if '-f' in clientOptions:
                userInputs['resultToBeStored'] = '1'
                
            userInputs['no_messages'] = str(no_messages)
            userInputs['packet_size'] = str(packet_size)
            userInputs['repeatCount'] = str(repeatCount)

            generatePerfomanceStats(total_no_messages, packetListAll, userInputs)

    return

def udpMessageSizeReceiveThread(sock):

    """
    Param: socket sock

    Desc:  This method starts a Client UDP thread to receive packets returned from the Server.
           When the packets are received packet size retrieved from packet fill and stored into a python set. 
           The thread will exit when all packets are
           received for that test or on socket time-out (5 secs) if packets are dropped.

    Return: thread rx
             
    """
    # terminateSocketPktTest to determine whether socket is closed or not
    global terminateSocketPktTest
    terminateSocketPktTest = False
    
    # pktSizeList is python set to hold packet size
    global pktSizeList
    pktSizeList = set()
    
    # this will set the buffer for recvfrom function below \
    buffer_size = massageSizeTest_biggest_packet_tobe_sent
    
    allowed_overhead = massageSizeTest_receiverThread_buffer_overhead
    
    # actual_buffer_size will store actual over head
    actual_buffer_size = buffer_size + (allowed_overhead * (buffer_size/100))
    
    # msgReceived - to keep track of number of message received
    msgReceived = 1
    
    # this variable stores number of expected message based on biggest packet size, \
    # packet size gets incremented and total messages sent for each run

    expected_number_of_message = massageSizeTest_forEach_PacketSize_totalMessageTobeSent * \
                                  (1000/massageSizeTest_packet_size_gets_incremented ) * \
                                  (massageSizeTest_biggest_packet_tobe_sent/1000)
    try:
        while terminateSocketPktTest == False:
            try:
                
                data, address = sock.recvfrom(actual_buffer_size)
                
                # below code extract packet size from message
                brktStart = data.find('[')
                brktFinish = data.find(']')
                pktSize = data[(brktStart+1):brktFinish]
                
                # packet size added to the set below
                pktSizeList.add(int(pktSize))
                
                # each successful message received msgReceived incremented
                msgReceived += 1
                
                if msgReceived > expected_number_of_message:
                    # if we received expected number of message we stop listening
                    terminateSocketPktTest = True

            except socket.timeout:
                terminateSocketPktTest = True
                break
            

    finally:

        sock.close()

    return


def udpStartMessageSizeTest(data):

    """
    Param: dict clientOptions

    Desc:  This method runs the UDP message size Test from the Client. Having extracted the user
           options, a socket is setup for UDP and the udpMessageSizeReceiveThread is called. The
           test starts by sending packets to the Server upto 10000. Once the test is complete (or socket timed out) 
           this method access global variable pktSizeList and display the maximum value.

    Return: 
             
    """
    
    # processing user input
    protocolType = data['-p']
    serverIP = data['--si']
    serverPort = int(data['--sp'])

    # Extract the IP type from the IP Address provided
    ipType = extractIpType(serverIP)

    if (ipType == "4"):
        serverAddress = (serverIP, serverPort)
    elif (ipType == "6"):
        serverAddress = (serverIP, serverPort, 0, 0)

    # setting biggest packet size
    biggest_packet = massageSizeTest_biggest_packet_tobe_sent

    # setting default packet size
    packet_size = massageSizeTest_initial_packet_size_toStart
    
    # sets message size to be incremented by amount
    incremented_by = massageSizeTest_packet_size_gets_incremented

    total_no_messages = massageSizeTest_forEach_PacketSize_totalMessageTobeSent
    
    # didTestRun - to determine if the test run or not \
    # this will help us to execute finally block
    didTestRun = True
    try:
        if (ipType == "4"):
            # Create a UDP/IP socket
            sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        elif (ipType == "6"):
            sock = socket.socket(socket.AF_INET6, socket.SOCK_DGRAM)

        sock.settimeout(5.0)

        # Get the Application Type, if its not SBX then test on the Management Port
        appType, appPackage = getApplication()

        # Only set the socket options for the SBC for the Packet Ports
        if (appType == "SBC"):

            packetPort = get_packet_route(serverIP)

            if ("pkt" in packetPort):

                lifgIndex, addrContext = get_socket_options(packetPort)

                if (lifgIndex != 0 and addrContext != 0):
                    # Setup the following socket options to get the Linux Kernal binding correctly
                    # for the Clients LIF Group Index and the Address Context on the SBC
                    #
                    # LifgIndex - #define IP_LIFG    29
                    sock.setsockopt(socket.IPPROTO_IP, 29, int(lifgIndex))

                    # Address Context - #define IP_AC    30
                    sock.setsockopt(socket.IPPROTO_IP, 30, int(addrContext))
                else:
                    print("Error: Packet port not configured for LIF Group Index or Address Context")
                    return 

        # Use thread to concurrently run the Client Receiver (with the Client Sender)
        try:
            rx = Thread(target=udpMessageSizeReceiveThread, args=(sock, )) # do not remove comma this is to help unpack variable passed
            rx.start()

            pktSeqNo = 1
            msgCount = 0
            sys.stdout.write("      Please wait, Test running ")
            
            # until we sent 10K we keep sending
            while packet_size <= biggest_packet:
                
                # printing a dot for eack 1K
                if packet_size % 1000 == 0:
                    sys.stdout.write('. ')
                    sys.stdout.flush()
                    
                # Create a default packet size of Users request
                packetFill = "x" * (packet_size)

                # Run the test for the number of messages requested and packet size

                for msgCount in range(total_no_messages):

                    # Generate the Test Packet - Including Sequence Number and Start Timestamp
                    pktSeqNoStr = str(pktSeqNo) + " "
                    pktSeqNoLen = len(pktSeqNoStr)
                
                    # Create a test packet by inserting the Sequence Number + PKT size in the created test packet
                    test_packet = packetFill[0:0] + pktSeqNoStr + '[' + str(packet_size) + ']'+packetFill[pktSeqNoLen+0:]
                    
                    # Send test packet to Server
                    sent = sock.sendto(test_packet, serverAddress)

                    # Increment the Packet sequence number
                    pktSeqNo = pktSeqNo + 1
                    
                packet_size += incremented_by
                
                # this sleep is to regulate PKT sending and avoid overflow\
                # in the receiver thread
                time.sleep(0.05)

        except:
            didTestRun = False
            print('\n')
            print("      Error: No packet received, Check server is running/IP address")

    except socket.error as msg:
        didTestRun = False
        print('\n')
        print("Socket Error: %s" % msg)

    finally:
        
        if didTestRun: # else leg not required
        
            # this while loop checks if received thread has stop receiving \
            # every 1 secs, so that we can access pktSizeList \
            # for biggest PKT received
            
            while terminateSocketPktTest == False:
                sys.stdout.write('. ')
                sys.stdout.flush()
                time.sleep(1.0)
            try:
                # if receiver thread stopped receiving and \
                # pktSizeList not empty then we process
                if len(pktSizeList) > 0 and terminateSocketPktTest :
                    print('\n')
                    print("        Largest packet size successfully transmitted and received: " + str(max(pktSizeList)))
                else:
                    print('\n')
                    print("        Error: No packet received, Check server is running/IP address")
            except NameError: # this line will catch error in case pktSizeList is empty
                print('\n')
                print("        Error: Error occurred, please try again.")
            

    return
    

def tcpStartNetworkServer(serverOptions):

    """
    Param: dict serverOptions

    Desc:  This method start the TCP Network Test Server. Having extracted the user options,
           a socket is setup for TCP which binds and waits, listening for packets. Packets
           reveived will be returned unchanged to the Client.

    Return: 
	         
    """

    serverIp = serverOptions['--li']
    serverPort = int(serverOptions['--lp'])

    # Extract the IP type from the IP Address provided
    ipType = extractIpType(serverIp)

    if (ipType == "4"):
        serverAddress = (serverIp, serverPort)
    elif (ipType == "6"):
        serverAddress = (serverIp, serverPort, 0, 0)

    #Create a TCP/IP socket for IPv4 or IPv6
    if (ipType == "4"):
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    elif (ipType == "6"):
        sock = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)

    # Get the Application Type, if its not SBX then test on the Management Port
    appType,appPackage = getApplication()

    # Only set the socket options for the SBC for the Packet Ports
    if (appType == "SBC"):

        ipExist, pID = ipAddressExist(serverIp, ipType)

        if ("pkt" in pID):

            lifgIndex, addrContext = get_socket_options(pID)

            if (lifgIndex or addrContext) != 0:
                # Setup the following socket options to get the Linux Kernal binding correctly
                # for the Clients LIF Group Index and the Address Context on the SBC
                #
                # LifgIndex - #define IP_LIFG    29
                sock.setsockopt(socket.IPPROTO_IP, 29, int(lifgIndex))

                # Address Context - #define IP_AC    30
                sock.setsockopt(socket.IPPROTO_IP, 30, int(addrContext))
            else:
                print("Error: Packet port not configured for LifgIndex or Address Context")
                return
			
    try:
        # Bind on Server Address
        sock.bind(serverAddress)

        # Listen for incoming connections
        sock.listen(1)
        print("This diagnostics tool is running as TCP Server to receive on IP = %s port = %s" % (serverIp, serverPort))
        print("Waiting ...")
        print("             Enter Ctrl-c to exit server mode and return to command prompt")

        try:
            while True:
                # Wait for a connection
                connection, clientAddress = sock.accept()

                # Receive the data with any size chunks and retransmit it
                while True:
                    data = connection.recv(10000)
                    if data:
                        connection.sendall(data)
                    else:
                        break

                # Clean up the connection
                connection.close()

        except socket.error as msg:
            print("Socket Error: %s" % msg)        
        except KeyboardInterrupt:				
            print('\n Exiting TCP server mode')

    except socket.error as msg:
        print("Socket Bind Error: %s" % msg)   

    return


def udpStartNetworkServer(serverOptions):

    """
    Param: dict serverOptions

    Desc:  This method start the UDP Network Test Server. Having extracted the user options,
           a socket is setup for UDP which binds and waits for packets. Packets reveived will
           be returned unchanged to the Client.

    Return: 
	         
    """

    serverIp = serverOptions['--li']
    serverPort = int(serverOptions['--lp'])

    # Extract the IP type from the IP Address provided
    ipType = extractIpType(serverIp)

    if (ipType == "4"):
        serverAddress = (serverIp, serverPort)
    elif (ipType == "6"):
        serverAddress = (serverIp, serverPort, 0, 0)

    try:
        if (ipType == "4"):
            # Create a UDP/IP socket
            sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        elif (ipType == "6"):
            sock = socket.socket(socket.AF_INET6, socket.SOCK_DGRAM)

        # Get the Application Type, if its not SBX then test on the Management Port
        appType,appPackage = getApplication()

        # Only set the socket options for the SBC for the Packet Ports
        if (appType == "SBC"):

            ipExist, pID = ipAddressExist(serverIp, ipType)
    
            if ("pkt" in pID):

                lifgIndex, addrContext = get_socket_options(pID)

                if (lifgIndex != 0 and addrContext != 0):
                    # Setup the following socket options to get the Linux Kernal binding correctly
                    # for the Clients LIF Group Index and the Address Context on the SBC
                    #
                    # LifgIndex - #define IP_LIFG    29
                    sock.setsockopt(socket.IPPROTO_IP, 29, int(lifgIndex))

                    # Address Context - #define IP_AC    30
                    sock.setsockopt(socket.IPPROTO_IP, 30, int(addrContext))
                else:
                    print("Error: Packet port not configured for LIF Group Index or Address Context")
                    return 

        try:
            # Bind on Server Address
            sock.bind(serverAddress)
            print("This diagnostics tool is running as UDP Server to receive on IP = %s port = %s" % (serverIp, serverPort))
            print("Waiting ...")
            print("             Enter Ctrl-c to exit server mode and return to command prompt")

            # Note - folowing need to be populated from input parameters
            start_drop =  int(serverOptions['--drop_start'])
            pkts_to_drop_in_a_block = int(serverOptions['--drop_count'])
            pkts_to_skip_in_a_block = int(serverOptions['--skip_count'])
            max_to_drop = int(serverOptions['--max_drops'])

            drop_and_skip_block_size = pkts_to_drop_in_a_block + pkts_to_skip_in_a_block
            pkts_dropped_count = 0;

            try:
                # Receive any size data chunks and retransmit it
                while True:
                    data, address = sock.recvfrom(10000)
                    if data:
                        rxSeqNo = int(data.split(" ")[0])
    				    # If there are packets to drop and we are in the drop range and havnt exceeded
    					# the drop count then, for each block of seq numbers in drop_and_skip_block_size
    					# check by doing a modulo to see if this pkt is in the pkts_to_drop section.
                        if ((pkts_to_drop_in_a_block > 0) and
    						(pkts_dropped_count < max_to_drop) and
    						(rxSeqNo >= start_drop) and
    						((rxSeqNo - start_drop) % drop_and_skip_block_size) < pkts_to_drop_in_a_block):
                            # this pkt is to be dropped
                            print("dropped pkt : %d" % rxSeqNo)							
                            pkts_dropped_count += 1
                        else:
                            # this pkt is sent back
                            sent = sock.sendto(data, address)
                    else:
                        print('Test complete')
                        break
    
            except socket.error as msg:
                print("Socket Error: %s" % msg)
            except KeyboardInterrupt:				
                print('\n Exiting UDP server mode')
				
        except socket.error as msg:
            print("Socket Bind Error: %s" % msg)

    finally:
        # Close the socket
        sock.close()

    return


def validateServerOptions(serverOptions):

    """
    Param: Server options and their respective values as Python dictionary i.e. {'-p': 'tcp' ....}
    
    Desc: This method expected to peform following validation on server option

        1. Each option has a value
        2. Number of option key and value not exceeded expected key value pair
        3. User entered only allowed protocol type or not misspelt
        4. Local ip is in valid format (IPv 4 or 6)
        5. IP address exist in test environment i.e. in SBX, PSX or EMS
        6. Port number entered is valid and allowed i.e. UDP - only 5060 or 5061 allowed
    
    Return: if all the validation pass returns true, otherwise false
    """

    # Validate the Server Options
    serverParamCheck = True

    for key, val in serverOptions.items():

        if val in ['-', '--']:
            print('    Error - Option missing a value.')
            print('    Usage - Do not enter an option, if unsure of its value')
            return serverParamCheck == False

    # Check minimum mandatory parameters
    if len(serverOptions) < 2:
        print('    Error - Does not have minimum mandatory parameters')
        print('    Usage - startNetworkServer -p <protocol_type> --li <servers_local_ip> [--lp <servers_local_port_no>] ')
        return serverParamCheck == False
	
    # Check max parameters
    if len(serverOptions) > 7:
        print('    Error - exceeded maximum parameters')
        print('    Usage - startNetworkServer -p <protocol_type> --li <servers_local_ip> [--lp <servers_local_port_no>] ')
        
        return serverParamCheck == False
        
    # Check protocol entered is correct
    if len(serverOptions['-p']) > 0:
        if not (str(serverOptions['-p'].strip()) in ('tcp', 'udp')):
                print('    Error - Incorrect protocol type')
                print('    Usage - Only tcp or udp allowed')
                
                return serverParamCheck == False
            
    else:
            print('    Error - Protocol not found')
            print('    Usage - Enter tcp or udp as protocol')
            
            return serverParamCheck == False
    
    # Check format of the IP Address is correct
    if len(serverOptions['--li']) > 0:
        ipType = extractIpType(serverOptions['--li'])
    else:
        print('    Error - Invalid IP address entered')
        print('    Correct Usage - Valid IPv 4 or 6 address')
        return serverParamCheck == False
      
        
    if not(validate_ip(serverOptions['--li'])):
        print('    Error - Invalid IP address entered')
        print('    Correct Usage - Valid IPv 4 or 6 address')
        return serverParamCheck == False
           
        # Check IP address exist with in the SBX interface(s)
    ipExist, pID = ipAddressExist(serverOptions['--li'], ipType)
    ipAddress = serverOptions['--li']
    if (ipExist != '1'):
        if(validate_ip(ipAddress)):
            print('    Error - Not a valid Server IP')
            print('    Correct Usage - Enter a valid Server IP')
            return serverParamCheck == False
                
    if not(1024 <= int(serverOptions['--lp']) <= 65535) and serverOptions['-p'] == 'tcp':
        print('    Error - Invalid port number')
        print('    Correct Usage - Port number in range 1024 - 65535')
        return serverParamCheck == False
    elif not(int(serverOptions['--lp']) == 5060) and not(int(serverOptions['--lp']) == 5061) and serverOptions['-p'] == 'udp':
        print('    Error - Invalid port number')
        print('    Correct Usage - ONLY 5060 or 5061 allowed for UDP')
        return serverParamCheck == False
    return serverParamCheck


def validateClientOptions(clientOptions):

    """
    Param: Client test options and their respective values as Python dictionary i.e. {'-p': 'tcp' ....}
    
    Desc: This method expected to peform following validation on local test option

        1. Each option has a value
        2. Number of option key and value not exceeded expected key value pair for different protocol i.e.
            TCP - requires 2 extra parameters than UDP
        3. User entered only allowed protocol type or not misspelt
        4. Server and Local ip are in valid format (IPv 4 or 6)
        5. Local IP address exist in test environment i.e. in SBX, PSX or EMS
        6. Port number entered (Both Server and Local) is a valid and allowed i.e. UDP - only 5060 or 5061 allowed
        7. Message rate per second has a numaric value and not exceed allowed limit
        8. Message size has a numaric value and not exceed allowed limit
        9. Run count has a numaric value and not exceed allowed number of repetation
        10. (Message rate per second * Message size) does not exceed allowed limit i.e on SBX 240 M/Bits allowed
    
    Return: if all the validation pass returns true, otherwise false
    """

    # Validate the Client Options
    clientParamCheck = True
    
    for key, val in clientOptions.items():

        if val in ['-', '--']:
            print('    Error - Option missing a value.')
            print('    Usage - Do not enter an option, if unsure of its value')
            return clientParamCheck == False
        
    if not clientOptions:
        print('    Error - No parameters specified')
        print('    Usage - startNetworkTest [-f] -p udp  --si <remote_server_ip> [--sp <remote_server_port_no>] [-r <msg_rate_per_sec>]  [-m <message_size>] [-c <run_count>]')
        print('    startNetworkTest [-f] -p tcp  --si <remote_server_ip> [--sp <remote_server_port_no>]  --li <local_ip> [--lp <local_port_no>]  [-r <msg_rate_per_sec>]  [-m <message_size>] [-c <run_count>]')
        return clientParamCheck == False

    protocolType = str(clientOptions['-p'].strip())
 
    # Check protocol entered is correct
    if len(protocolType) > 0:
        if not (protocolType in ('tcp', 'udp')):
                print('    Error - Incorrect protocol type')
                print('    Usage - Only tcp or udp allowed')
                
                return clientParamCheck == False
            
    else:
            print('    Error - Protocol not found')
            print('    Usage - Enter tcp or udp as protocol')
            
            return clientParamCheck == False
            
    # Since both TCP and UDP will have following parameters they will be processed below, before entering branch      
    if clientOptions['-r'].isdigit():
        numMessage = int(clientOptions['-r'])
        
        if not(1 <= numMessage <= 10000):
            print('    Error - Invalid message rate per second')
            print('    Usage - Message rate should be between 1 - 10000')
            
            return clientParamCheck == False
    else:
        print('    Error - Invalid message rate per second')
        print('    Usage - Message rate should be a number between 1 - 10000')
        
        return clientParamCheck == False
        
    if clientOptions['-m'].isdigit():
        messSize = int(clientOptions['-m'])
        
        if not(100 <= messSize <= 10000):
            print('    Error - Invalid Message Size')
            print('    Usage - Message size should be between 100 - 10000')
            
            return clientParamCheck == False
    else:
        print('    Error - Invalid Message Size')
        print('    Usage - Message size must be a number')
        
        return clientParamCheck == False
        
    if clientOptions['-c'].isdigit():
        repeatCount = int(clientOptions['-c'])
        
        if not(1 <= repeatCount <= 100):
            print('    Error - Invalid Run Count')
            print('    Usage - Run count should be 1 - 100')
            
            return clientParamCheck == False
    else:
        print('    Error - Invalid Run Count')
        print('    Usage - Run Count must be a number')
        
        return clientParamCheck == False  
    
    serverIP = clientOptions['--si'] # Processed here since protocol has not impact
    
    # Validate the Client Options for TCP
    if (protocolType == 'tcp'):
    
        if clientOptions['--sp'].isdigit():
            serverPort = int(clientOptions['--sp'])
            
            if not(1024 <= serverPort <= 65535):
                print('    Error - Invalid server port number')
                print('    Correct Usage - Port number in range 1024 - 65535')
               
                return clientParamCheck == False
        else:
            print('    Error - Invalid Server Port Value')
            print('    Usage - Port Value must be a number')
            
            return clientParamCheck == False
        
        localIP = clientOptions['--li']
        
        if clientOptions['--lp'].isdigit():
            localPort = int(clientOptions['--lp'])
            
            if not(1024 <= localPort <= 65535) and protocolType == 'tcp':
                print('    Error - Invalid port number')
                print('    Correct Usage - Port number in range 1024 - 65535')
                return clientParamCheck == False
        else:
            print('    Error - Invalid Local Port Value')
            print('    Usage - Port Value must be a number')
            
            return clientParamCheck == False
        
        # Check minimum mandatory parameters
        if len(clientOptions) < 3:
            print('    Error - Does not have minimum mandatory parameters')
            print('    Usage - startNetworkTest [-f] -p tcp  --si <remote_server_ip> [--sp <remote_server_port_no>]  --li <local_ip> [--lp <local_port_no>]  [-r <msg_rate_per_sec>]  [-m <message_size>] [-c <run_count>]')
            return clientParamCheck == False

        # Check max parameters
        if len(clientOptions) > 9:
            print('    Error - exceeded maximum parameters')
            print('    startNetworkTest [-f] -p tcp  --si <remote_server_ip> [--sp <remote_server_port_no>]  --li <local_ip> [--lp <local_port_no>]  [-r <msg_rate_per_sec>]  [-m <message_size>] [-c <run_count>]')
            return clientParamCheck == False
        
        # Check format of the IP Address is correct
        if(len(serverIP) > 0):
            serverIPType = extractIpType(serverIP)
        else:
            print('    Error - Invalid Server IP address entered')
            print('    Correct Usage - Valid IPv 4 or 6 address')
            
            return clientParamCheck == False

        if not(validate_ip(serverIP)):
            print('    Error - Invalid Server IP address entered')
            print('    Correct Usage - Valid IPv 4 or 6 address')
            
            return clientParamCheck == False
       
        localIPType = extractIpType(localIP)
        
        if not(validate_ip(localIP)):
            print('    Error - Invalid Client IP address entered')
            print('    Correct Usage - Valid IPv 4 or 6 address')
            return clientParamCheck == False
        
        ipExist, pID = ipAddressExist(localIP, localIPType)
        
        if (ipExist !='1'):
            if(validate_ip(localIP)):
                print('    Error - Client IP doesnt exist')
                print('    Correct Usage - Enter a existing Client IP')
                return clientParamCheck == False
                
        
        #####
        multiMessagePacket = numMessage * messSize # total size in bytes
        sizeInMBits = float(float(multiMessagePacket) / float(131072))  # 1 mbits = 131072 and 240 mbits allowed
        
        applicationType, packType = getApplication()
        ## setting allowed traffic limit for SBX and Non-SBX
        if applicationType == 'SBC':
            allowedLimit = float((10 *  (sizeInMBits/100)) + 240) ## to allow 10% overhead
            
            if(sizeInMBits > allowedLimit):
                print('    Error - Capacity Limit Exceed For Test')
                print('    Usage - Only 240 M/Bits Allowed for SBX port')
                return clientParamCheck == False
            
        else:
            sizeInGBits = float(sizeInMBits/1024)
            allowedLimit = float((10 *  (sizeInGBits/100)) + 1)
            if(sizeInGBits > allowedLimit):
                print('    Error - Capacity Limit Exceed For Test')
                print('    Usage - Only 1000 M/Bits Allowed for port')
                return clientParamCheck == False
 
    # Validate the Client Options for UDP
    
    if(protocolType == 'udp' and 6 <= len(clientOptions) <= 7):
        
       
        if clientOptions['--sp'].isdigit():
            serverPort = int(clientOptions['--sp'].strip())
            if not( serverPort == 5060 or serverPort == 5061):
                print('    Error - Invalid server port number')
                print('    Correct Usage - Only port 5060 or 5061 allowed')
                return clientParamCheck == False
        else:
            print('    Error - Invalid Server Port Value')
            print('    Usage - Port Value must be a number')
            
            return clientParamCheck == False

        serverIPType = extractIpType(serverIP)
       
        if not(validate_ip(serverIP)):
            print('    Error - Invalid Server IP address entered')
            print('    Correct Usage - Valid IPv 4 or 6 address')
            
            return clientParamCheck == False
            
        multiMessagePacket = numMessage * messSize # total size in bytes
        sizeInMBits = float(float(multiMessagePacket) / float(131072))  # 1 mbits = 131072 and 240 mbits allowed
        applicationType, packType = getApplication()
        ## setting allowed traffic limit for SBX and Non-SBX
        if applicationType == 'SBC':
            allowedLimit = float((10 *  (sizeInMBits/100)) + 240) ## to allow 10% overhead
            
            if(sizeInMBits > allowedLimit):
                print('    Error - Capacity Limit Exceed For Test')
                print('    Usage - Only 240 M/Bits Allowed for SBX port')
                return clientParamCheck == False
            
        else:
            sizeInGBits = float(sizeInMBits/1024)
            allowedLimit = float((10 *  (sizeInGBits/100)) + 1)
            if(sizeInGBits > allowedLimit):
                print('    Error - Capacity Limit Exceed For Test')
                print('    Usage - Only 1000 M/Bits Allowed for port')
                return clientParamCheck == False
        
    
    return clientParamCheck

def validateMessageTestOptions(data):

    """
    Param: message size test options and their respective values as Python dictionary i.e. {'-p': 'udp' ....}
    
    Desc: This method expected to peform following validation on local test option

        1. Each option has a value
        2. User entered only allowed protocol type or not misspelt
        3. Server and Local ip are in valid format (IPv 4 or 6)
        4. Port number entered is a valid and allowed i.e. UDP - only 5060 or 5061 allowed
    
    Return: if all the validation pass returns true, otherwise false
    """
    protocolType = data['-p']
    serverIP = data['--si']
    serverPort = data['--sp']
    messageTestValidationCheck = True

    # Check protocol entered is correct
    if len(protocolType) > 0:
        if not (protocolType in ('udp')):
            print('    Error - Incorrect protocol type')
            print('    Usage - Only udp allowed')
            
            return messageTestValidationCheck == False       
    else:
        print('    Error - Protocol not found')
        print('    Usage - Enter udp as protocol')
        
        return messageTestValidationCheck == False

    if not(validate_ip(serverIP)):
        print('    Error - Invalid IP address entered')
        print('    Correct Usage - Valid IPv 4 or 6 address')
        return messageTestValidationCheck == False

    if serverPort.isdigit():
        serverPort = int(serverPort.strip())
        if not( serverPort == 5060 or serverPort == 5061):
            print('    Error - Invalid server port number')
            print('    Correct Usage - Only port 5060 or 5061 allowed')
            return messageTestValidationCheck == False
    else:
        print('    Error - Invalid Server Port Value')
        print('    Usage - Port Value must be a number')
        
        return messageTestValidationCheck == False

    return messageTestValidationCheck


def generatePerfomanceStats(no_messages, packetListAll, inputParams):

    """
    Param: int no_messages, list packetListAll, dict inputParams

    Desc:  This method generates the performance statistics for the test being run.
           Displays the user inputs, IPs and ports used to run the test.
           Displays test results, for fastest, slowest packets and 5 equally sized buckets
           with bucket counts for each.
           Displays total test duration, along with packet loss count and list of up to 100
           packet numbers discarded.

    Return:
	         
    """

    # Initialise the range timers
    fastestRange1 = 0.0
    slowestRange1 = 0.0
    fastestRange2 = 0.0
    slowestRange2 = 0.0
    fastestRange3 = 0.0
    slowestRange3 = 0.0
    fastestRange4 = 0.0
    slowestRange4 = 0.0
    fastestRange5 = 0.0
    slowestRange5 = 0.0
    totalTime = 0.0

    # Initialise the 5 bucket counters
    bucketCount1 = 0
    bucketCount2 = 0
    bucketCount3 = 0
    bucketCount4 = 0
    bucketCount5 = 0
    bucketTotal = 0

    noReceivedMessage = 0
    packetsLost = 0

    # Initialise Lists
    startList = []
    stopList = []
    startStopList = []
    msgLatencyList = []
    lostPackSeqNo = []

    # Extract the Start and Stop times into two lists
    for packet in packetListAll:
        
        if not(packet.stopTime is None):  #packet.seqId
            noReceivedMessage = noReceivedMessage + 1
            startList.append(packet.startTime)
            stopList.append(packet.stopTime)
        else:
            packetsLost +=1
            lostPackSeqNo.append(packet.seqId)

    # Work out the Total Processing time for the test
    startTime = packetListAll[0].startTime

    stopTime = stopList[noReceivedMessage-1]
    totalTime = "%.6f" % (float(stopTime) - float(startTime))

    # Create a new list of all message round trip durations (latency).
    # Used to process latency times for all messages
    if (len(startList) == len(stopList)):
        for startTime, stopTime in zip(startList, stopList):
            tmpTime = "%.6f" % (float(stopTime) - float(startTime))
            msgLatencyList.append(float(tmpTime))

    # Sort the list to find fastest/slowest message duration
    msgLatencyList.sort()

    # Calculate Bucket Ranges in 5 buckets of equal sizes from Fastest to Slowest
    timeDiff = msgLatencyList[noReceivedMessage-1] - msgLatencyList[0]
    rangeGap = "%.6f" % float(timeDiff / 5.000000)
    rangeGap = float(rangeGap)
    fastest = float(msgLatencyList[0])
    slowest = float(msgLatencyList[noReceivedMessage-1])

    # Create Bucket Ranges
    fastestRange1 = float(fastest)
    slowestRange1 = float(fastest + rangeGap)
    fastestRange2 = float(slowestRange1)
    slowestRange2 = float(slowestRange1 + rangeGap)
    fastestRange3 = float(slowestRange2)
    slowestRange3 = float(slowestRange2 + rangeGap)
    fastestRange4 = float(slowestRange3)
    slowestRange4 = float(slowestRange3 + rangeGap)
    fastestRange5 = float(slowestRange4)
    slowestRange5 = float(slowest)

    # Fill the buckets!
    for latencyTime in msgLatencyList:
        bucketTotal = bucketTotal +1
        if ((fastestRange1 <= latencyTime) and (latencyTime <= slowestRange1)):
            bucketCount1 = bucketCount1 + 1
            continue
        elif ((fastestRange2 < latencyTime) and (latencyTime <= slowestRange2)):
            bucketCount2 = bucketCount2 + 1
            continue
        elif ((fastestRange3 < latencyTime) and (latencyTime <= slowestRange3)):
            bucketCount3 = bucketCount3 + 1
            continue
        elif ((fastestRange4 < latencyTime) and (latencyTime <= slowestRange4)):
               bucketCount4 = bucketCount4 + 1
               continue
        elif ((fastestRange5 < latencyTime) and (latencyTime <= slowestRange5)):
            bucketCount5 = bucketCount5 + 1
            continue

    ## below lines writing output to a txt file

    if (inputParams['resultToBeStored'] == '1'):
        nodeName = platform.node() if len(platform.node()) > 0 else 'Unknown'
    
        appType,appPackage = getApplication()
        
        appType = appType if len(appType) > 0 else 'Unknown'
        
        stdout = sys.stdout # save original stdout

        timestr = time.strftime("%Y%m%d-%H%M%S")
        logfile= appType + '_' + nodeName + '_' + "performanceResult_" + timestr + ".txt"

        sys.stdout = Logger(logfile)
    
    # input parameter(s) formatting [inputParams] 
   
    print("\n{0:<55s}".format("    Performance Measurement for " + inputParams['protocolType'].upper()) + "")
    
    print("      User Input(s)         |")
    
    print("{0:<28s}".format("          Server Address") + "|{0:<33}".format(" " + inputParams['serverIPAndPort']))
    
    
    if (inputParams['protocolType'] == 'tcp'):# TCP will have Client IP
    
        print("{0:<28s}".format("          Local Address") + "|{0:<33}".format(" " + inputParams['localIPAndPort']))
    elif(inputParams['protocolType'] == 'udp'):
        print("{0:<28s}".format("          Local Address") + "|{0:<33}".format(" " + "Not Applicable"))
        
    print("{0:<28s}".format("          Protocol Type") + "|{0:<33}".format(" " +inputParams['protocolType'].upper()))
   
    print("{0:<28s}".format("          Port Type") + "|{0:<33}".format(" " +inputParams['portType'].upper() if 'pkt' in inputParams['portType'] else ' MGMT'))
     
    print("{0:<28s}".format("          Number of Messages") + "|{0:<33}".format(" " +str(int(inputParams['no_messages'])*int(inputParams['repeatCount']))))
    
    print("{0:<28s}".format("          Message Size") + "|{0:<33}".format(" " +inputParams['packet_size']))
   
    
          

    # Display the Performance Test Measurement Outputs
    print("{0:<55s}".format("      Performance Test Results"))
    
    print("{0:<28s}".format("          Fastest") + "|{0:<33}".format(" " +str("{0:.6f}".format(0 if len(msgLatencyList) == 0 else msgLatencyList[0]) )))
    
    print("{0:<28s}".format("          Slowest") + "|{0:<33}".format(" " +str("{0:.6f}".format(0 if len(msgLatencyList) == 0 else msgLatencyList[noReceivedMessage-1]))))
    
    print("{0:<37s}".format("          Bucket 1 (Range "+str("{0:.6f}".format(fastestRange1))+" - "+str("{0:.6f}".format(slowestRange1))+")") + "|{0:<16}".format(" " +str(bucketCount1)))
    
    print("{0:<37s}".format("          Bucket 2 (Range "+str("{0:.6f}".format(fastestRange2))+" - "+str("{0:.6f}".format(slowestRange2))+")") + "|{0:<16}".format(" " +str(bucketCount2)))
    
    print("{0:<37s}".format("          Bucket 3 (Range "+str("{0:.6f}".format(fastestRange3))+" - "+str("{0:.6f}".format(slowestRange3))+")") + "|{0:<16}".format(" " +str(bucketCount3)))
    
    print("{0:<37s}".format("          Bucket 4 (Range "+str("{0:.6f}".format(fastestRange4))+" - "+str("{0:.6f}".format(slowestRange4))+")") + "|{0:<16}".format(" " +str(bucketCount4)))
   
    print("{0:<37s}".format("          Bucket 5 (Range "+str("{0:.6f}".format(fastestRange5))+" - "+str("{0:.6f}".format(slowestRange5))+")") + "|{0:<16}".format(" " +str(bucketCount5)))
    
    print("{0:<28s}".format("          Total Time") + "|{0:<33}".format(" " +str(totalTime)))
    if packetsLost == 0:
    
        print("{0:<28s}".format("          Packets Lost") + "|{0:<33}".format(" " +str(packetsLost)))
        
    else:

        lost_id = []
        numberOfLostPacket = len(lostPackSeqNo)
        print("{0:<28s}".format("          Packets Lost") + "|{0:<33}".format(" " + str(packetsLost)))
        
        if numberOfLostPacket > 100:
            lost_id = [lostPackSeqNo[i] for i in range (0, 100)]
            print("{0:<28s}".format("          Lost packet ID(s)") + "|{0:<33}".format(" " + indent(lost_id)))
        else:
            lost_id = [lostPackSeqNo[i] for i in range (0, numberOfLostPacket)]
            print("{0:<28s}".format("          Lost packet ID(s)") + "|{0:<33}".format(" " + indent(lost_id)))

    if inputParams['resultToBeStored'] == '1':
        sys.stdout = stdout # file writing finish here
        print("    Test output also written to file  - " + logfile)    
    
    return

# below method will indent the packet list into display

def indent(pkt_list):

    """
    Param: A python list that contain lost packet's ID numbers.
    
    Desc: This method will create newline by placing six lost packet ID numbers on each line.
        After, first line each line will have a paading of specified amount.
        
    Return: formated string
    """
    padding = 30 * ' '
    counter = 1
    string = ''
    for x in pkt_list:
        string = string + '{:>6s}'.format(str(x))
        if ( counter%6 == 0):
            string = string + '\n' + padding
        counter += 1
    return string


if __name__ == '__main__':

    if len(sys.argv) > 1:
        # if command line had arguments then just run the single 
        # command specified on the command line
        diagsCommands().onecmd(' '.join(sys.argv[1:]))
    else:
	    # no arguments on command line, so go into loop prompting for command input
        diagsCommands().cmdloop()


