#!/bin/bash

# Utility for configuring a grub password

# pickup common commands and variables
if [ -e /opt/sonus/staging/sonusCommands.sh ]; then
    source /opt/sonus/staging/sonusCommands.sh
elif [ -e /opt/sonus/bin/sonusCommands.sh ]; then
    source /opt/sonus/bin/sonusCommands.sh
else
   /bin/echo "Could not locate sonusCommands.sh Exiting..."
   exit 1
fi
if [ -e /opt/sonus/staging/sonusCommonFiles.sh ]; then
    source /opt/sonus/staging/sonusCommonFiles.sh
elif [ -e /opt/sonus/bin/sonusCommonFiles.sh ]; then
    source /opt/sonus/bin/sonusCommonFiles.sh
else
   /bin/echo "Could not locate sonusCommonFiles.sh Exiting..."
   exit 1
fi

# some helpful variables
PROG=${0##*/}

# command line option variables
gPassword=
gUser=
plainPwd=
pbkdf2Pwd=
grubFile=
skipGrubUpdate=0

usage()
{
$CAT << EOF
usage: $PROG [-h] [-u user] [-p pwd] [-P pwd] [-s] [-f grub_file]

Either add a configuration item for a grub menu entry or update the grub default file

OPTIONS:
   -h               Print this message
   -f               Grub file to modify (supports upgrade/revert functionality)
   -p               Plain text grub password -- used to generate a pbkdf2 password hash
   -P               PBKDF2 password hash (supports upgrade/revert functionality)
   -s               Skip update-grub call
   -u               Grub user

   If no user is specified, the current user is maintained

   Either -p or -P must be specified

   NOTE: -s should only be used if a call to update-grub will be subsequently made
EOF
}

configurePwd()
{
    if [ -n "$gUser" ]; then
        $ECHO "Configuring grub user..."
        $SED -i --follow-symlinks -e "/^SUPERUSER/s/=.*/=\"$gUser\"/" $grubFile
    fi

    $ECHO "Configuring grub password..."
    $SED -i --follow-symlinks -e "/^PASSWORD/s/=.*/=\"$gPassword\"/" $grubFile
}

#### process arguments and validate them

while getopts "hf:p:P:su:" OPTION
do
   case $OPTION in
   h)
      usage
      exit 0
      ;;
   f)
      # using -e since could be a file or could be a link. we just want
      # to know if it exists.
      if [ -e "$OPTARG" ]; then
          grubFile=$OPTARG
      else
          $ECHO "** file $OPTARG does not exist!"
          usage
          exit 1
      fi
      ;;
   p)
      plainPwd="$OPTARG"
      ;;
   P)
      pbkdf2Pwd="$OPTARG"
      ;;
   s)
      skipGrubUpdate=1
      ;;
   u)
      gUser="$OPTARG"
      ;;
   ?)
      $ECHO -e "$PROG: invalid argument specified\n"
      usage
      exit 1
      ;;
   esac
done

shift $(($OPTIND - 1))

if [[ $# -ne 0 ]]; then
    $ECHO "** extra arguments detected ($*)"
    usage
    exit 1
elif [[ -z "$plainPwd" && -z "$pbkdf2Pwd" ]]; then
    $ECHO "** Either -p or -P argument must be specified **"
    usage
    exit 1
elif [[ -n "$plainPwd" && -n "$pbkdf2Pwd" ]]; then
    $ECHO "** The -p and -P arguments are mutually exclusive **"
    usage
    exit 1
fi

# if file not specifieed, use the real location
if [ -z "$grubFile" ]; then
    grubFile=$SONUS_GRUB_MENU_PWD_FILE
fi

if [ -n "$plainPwd" ]; then
    # automate generation of the encrypted password
    # note: the utility is meant to be interactive, we work around that by
    # sending in the pwd twice and grabbing the appropriate ouput
    gPassword=$($ECHO -e "$plainPwd\n$plainPwd" | $GRUB_MKPASSWD_PBKDF2 | $AWK '/grub.pbkdf/{print $NF}')
else
    gPassword=$pbkdf2Pwd
fi

# either update default file or add a menu entry to the custom menu file
configurePwd

# the change is only official if we update grub.  when called from install
# and upgrade scripts, there is no reason to update grub here since it will
# be done by those scripts after making other changes
if [ $skipGrubUpdate -eq 0 ]; then
    $ECHO "Updating grub configuration (grub.cfg)..."
    $UPDATE_GRUB_SH
fi

exit 0
