#!/bin/bash

# * Script to partition the root(/) into 3 partitions P1(root1), P2(root2), P3(/home/common)
#   and change the boot partition to root1/root2
# * This script will be called only when we are booted from sda6/sda7  
# * Mount point to access the /root partition will be over /sda6 in case of iso and /drbd in case of update/LSWU

# Sourcing Command Variables
if [ -e /opt/sonus/staging/sonusCommonUtils.sh ]; then
   source /opt/sonus/staging/sonusCommonUtils.sh
elif [ -e /opt/sonus/bin/sonusCommonUtils.sh ];then
   source /opt/sonus/bin/sonusCommonUtils.sh
else
   /bin/echo "Could not locate sonusCommonUtils.sh Exiting..."
   exit 1
fi

NAME=partition_tool.sh
tempDrbd=/tmp/tempDrbd
common_dir=$tempDrbd/common
lv_root_dir=/tmp/root
lv_root2_dir=/tmp/root2
lv_common_dir=/tmp/common
lv_root=/dev/mapper/debian-root
lv_root1=/dev/mapper/debian-root1
lv_root2=/dev/mapper/debian-root2
lv_common=/dev/mapper/debian-common

logFile=/root/partition.log
gServiceName=sbx

# NOTE: use our own reboot command since sonusCommands/CommonFiles defines
# REBOOT to be the Sonus reboot.sh script.  We don't want the extra functionality
# that provides, nor is it available at this time.  We truly want to call the
# reboot command, and we want it to hang there till finished.
# NOTE: use a name other than REBOOT, since the call to switch_boot.sh will reset
# the value even if we set it to something else here
SBIN_REBOOT=/sbin/reboot

logMsg()
{
  local logFile=$1
  local logMsg=$2
  dateStr=`$DATE +"%Y-%m-%d %H:%M:%S"`
  $ECHO -e "$dateStr $logMsg" >> $logFile

  # uncoment the following for debug feedback within the java console
  #$ECHO "$dateStr $logMsg" | $TEE /dev/tty1

  $ECHO "Disk partition upgrade in progress. Please do not hit the power button..."| $TEE /dev/tty1
}

# in upgrade cases mount drbd instead on sda6 
mountDrbd()
{
  if [ -e /etc/init.d/drbd ]; then 

     $LSMOD | $GREP -q drbd 
     if [ $? -eq 0 ]; then 
        logMsg $logFile "drbd is active, may need to shutdown"
        $DRBDADM disconnect mirror
        $UMOUNT ${primary_device}6
        $SLEEP 1
        $DRBDADM secondary mirror
        $DRBDADM detach mirror
        $SERVICE drbd stop
     else
        $UMOUNT ${primary_device}6
        $SLEEP 1
        $SERVICE drbd stop
     fi

     $MODPROBE drbd
     $SERVICE drbd start
     dState=`$DRBDADM dstate mirror| $AWK -F "/" {'printf $1'}`

     # DRBD distinguishes between inconsistent and outdated data.  Inconsistent data
     # on a node indicates partially  up to date data ( disconnect during sync  )
     if [ "$dState" == "Inconsistent" ]; then
        logMsg $logFile "WARNING! overwriting drbd peer data.. This might result in drbd re-sync." 
        $DRBDADM -- --overwrite-data-of-peer primary mirror
     else 
        $DRBDADM primary mirror
     fi

     $MOUNT /dev/drbd0 $tempDrbd

     $MOUNT | $GREP drbd0 | $GREP -q -v grep 
      if [ $? -ne 0 ]; then
        logMsg $logFile "Could not mount drbd to $tempDir "
        logMsg $logFile "PARTITIONING could not be performed due to failure of drbd mount.."
        logMsg $logFile "Reverting back to older version.."
        logMsg $logFile "Please note that an editional standby server restart is required to bring cdbs in sync."
        logMsg $logFile "Please perform a manual 'service sbx restart' on standby."
        $MKDIR -p /tmp/recovery
        $MOUNT /dev/mapper/debian-root /tmp/recovery
        logMsg $logFile "Creating revertInProgress..."
        $ECHO  "userInitiated" > /tmp/recovery/opt/sonus/revertInProgress

        # if we were attempting an uprgade from grub-legacy, then we
        # need to restore the previous/unmodified /boot/grub and also
        # re-install grub-legacy.  If updating from a grub2 version,
        # we just need to put the previous menu config in place
        # and utilize it.
        if [ -d "$HOME_ETC_GRUBD_PREV_DIR" ]; then
            $RM -rf $HOME_ETC_GRUBD_DIR $HOME_ETC_DEFAULT_DIR
            $MV $HOME_ETC_GRUBD_PREV_DIR $HOME_ETC_GRUBD_DIR
            $MV $HOME_ETC_DEFAULT_PREV_DIR $HOME_ETC_DEFAULT_DIR

            grubDate=`$DATE +'%a %b %e %H:%M:%S %Z %Y'`
            $ECHO "$grubDate: calling update-grub" &>> $logFile
            $UPDATE_GRUB_SH &>> $logFile
        else
            # we reverted from a grub2 install to a grub install
            # note: this is deprecated functionality. needed until grub-legacy is
            # no longer a concern (we support direct upgrade for 2 releases, so after
            # after 10.x...)

            # restore the saved grub directory
            `$RM -fr /boot/grub`;
            `$MV /boot/grub.prev /boot/grub`;

            # re-install grub-legacy to the MBR
            $GRUB_INSTALL $primary_device &>> $logFile
        fi

        $CP -f $logFile /tmp/recovery/opt/sonus/staging/
        $UMOUNT /tmp/recovery
        $SBIN_REBOOT
      else
        logMsg $logFile "drbd is mounted on $tempDrbd"
      fi
  fi
}

# unmount drbd 
umountDrbd()
{
  $MOUNT | $GREP drbd0 | $GREP -q -v grep
  if [ $? -eq 0 ]; then
     $DRBDADM disconnect mirror
     $UMOUNT /dev/drbd0
     $SLEEP 1
     $DRBDADM secondary mirror
     $DRBDADM detach mirror
     $SERVICE drbd stop
  else
     $UMOUNT ${primary_device}6
  fi

}

# This function takes care special case handling required on backup partition as part of revert
updateBackupPart()
{
  # add marker file on the previous version, whenever booted from the previous version on Jviewer 
  # revert will be called automatically to previous version
  $ECHO "Marker created by $0 for Revert on $lv_root_dir " > $lv_root_dir/opt/sonus/lswuCommitPending
  # Remove leftover osUpgrade markers
  $RM -f $lv_root_dir/opt/sonus/.osUpgradeRevertStatus*

  # In some releases, parent of sbx on reboot is not /etc/init.d/rc, and we depend on it to run revert,
  # so augment the condition
  if [ -e $lv_root_dir/opt/sonus/sbx/scripts/sbxStart.sh ];then
      $SED -i 's,grep -q "/etc/init.d/rc",egrep -q "/etc/init.d/rc|startpar",' $lv_root_dir/opt/sonus/sbx/scripts/sbxStart.sh
  fi
}

start()
{
  $LOGGER -t $0 "Starting service $NAME"
  logMsg $logFile "Starting service $NAME"
  
  logMsg $logFile "Turning off ipmi watchdog timer"
  $IPMITOOL mc watchdog off

  # At times, SDD can become readonly. A reboot should recover this.
  $TOUCH /tmp/testing ;
  if [ $? -ne 0 ];then 
     $ECHO "File System got mounted in readonly mode.. Rebooting the server to recover.."
     $SBIN_REBOOT -f
  fi
  $RM -f /tmp/testing
 
  # Check if partitions are already created. Shouldnt be the case, but just being defensive.
  lvcount=$($LVSCAN|$WC -l)
  if [ $lvcount == 3 ]; then
     logMsg $logFile "Partitions are already created! Exiting.."
     # Remove self entry from rc.local to avoid reboot loop
     $SED -i "/partition_tool.sh/d" /etc/rc.local
     # Switch boot to root1
     $SH $SWITCH_BOOT_SH "root1" &>> $logFile
     $SBIN_REBOOT
     exit 1
  fi
  
  $MKDIR -p $tempDrbd

  upgradeCase="false"
  fsParam="-t ext4"
  if [[ -e /root/bootSda6 ]]; then 
   common_dir=/root/common
  elif [ -e /root/bootSda7 ]; then 
     # Partition tool called on upgrade, set the flag.
     upgradeCase="true"
     mountDrbd
  fi 

  $SLEEP 3;

  logMsg $logFile "Creating Directories $lv_root_dir,$lv_root2_dir,$lv_common_dir,$common_dir.."
  $MKDIR -p $lv_root_dir
  $MKDIR -p $lv_root2_dir
  $MKDIR -p $lv_common_dir
  $MKDIR -p $common_dir

  $MOUNT | $GREP $lv_root| $GREP -q -v grep
  if [ $? -eq 0 ]; then
     $UMOUNT $lv_root
  fi

  $MOUNT $lv_root $lv_root_dir

  # ASAN and TSAN builds create larger builds and need a larger root partition than
  # non ASAN builds
  if [[ -e $lv_root_dir/var/log/asanBuild || -e $lv_root_dir/var/log/tsanBuild ]]; then 
    lv_root_size=40  # In GBs
    lv_root_reduce_size=82  # In GBs
  else
	lv_root_size=10  # In GBs
	lv_root_reduce_size=22  # In GBs
  fi

  logMsg $logFile "Taking backup of directories which will go to P1,P2.."
  fileList=`$LS -d $lv_root_dir/* |$EGREP -v "/home|/opt|/var|$lv_root_dir/tmp|/proc|openhpidLog|/etc"`
  fileListSize=`$DU -csh $fileList |$GREP total |$AWK '{print $1}'` 
  logMsg $logFile "Copying root content except /home,/var,/opt,/proc,/etc. Size:$fileListSize"
  $DU -csh $fileList >> $logFile
  $CP -af $fileList $common_dir
  copiedFileListSize=`$DU -csh $common_dir |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Copy complete. Copied Size:$copiedFileListSize"
  filesToDelete=`$LS -d $lv_root_dir/* |$EGREP -v "/home|/opt|/var|/etc"`
  filesToDeleteSize=`$DU -csh $filesToDelete |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Deleting copied root content. Size:$filesToDeleteSize"
  $DU -csh $filesToDelete >> $logFile
  $RM -fr $filesToDelete
  logMsg $logFile "Deletion complete"

  $MKDIR -p $common_dir/tmp
  $CHMOD 1777 $common_dir/tmp
  $MKDIR -p $common_dir/proc
  $MKDIR -p $common_dir/opt/sonus/
  
  fileListOpt=`$LS -d $lv_root_dir/opt/sonus/* | $EGREP -v "/opt/sonus/external|/opt/sonus/staging"`
  fileListOptSize=`$DU -csh $fileListOpt |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Copying opt/sonus context except staging..Size:$fileListOptSize"
  $DU -csh $fileListOpt >> $logFile
  $CP -af $fileListOpt $common_dir/opt/sonus/
  copiedFileListOptSize=`$DU -csh $common_dir/opt/sonus/ |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Copy complete. Copied size:$copiedFileListOptSize"

  logMsg $logFile "Copying hidden opt/sonus content.."
# getting no such file or directory from ls, which also causes a cp error
  hiddenFileListOpt=`$LS -d $lv_root_dir/opt/sonus/.[^.]* 2> /dev/null | $EGREP -v "/opt/sonus/external|/opt/sonus/staging"`
  if [ -n "$hiddenFileListOpt" ]; then
      $CP -af $hiddenFileListOpt $common_dir/opt/sonus/
  fi

  logMsg $logFile "Removing copied opt/sonus content.."
  $RM -fr $fileListOpt
  $RM -fr $hiddenFileListOpt

  $MKDIR -p $common_dir/var/lib
  fileListVar=`$LS -d $lv_root_dir/var/* | $EGREP -v "$lv_root_dir/var/log|$lv_root_dir/var/lib"`
  fileListVarSize=`$DU -csh $fileListVar |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Copying var content except log and lib..Size:$fileListVarSize"
  $DU -csh $fileListVar >> $logFile
  $CP -af $fileListVar $common_dir/var/
  copiedFileListVarSize=`$DU -csh $common_dir/var/ |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Copy complete. Copied size:$copiedFileListVarSize"
  logMsg $logFile "Removing copied var content.."
  $RM -fr $fileListVar

  fileListVarLib=`$LS -d $lv_root_dir/var/lib/* |$EGREP -v "$lv_root_dir/var/lib/libvirt"`
  fileListVarLibSize=`$DU -csh $fileListVarLib |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Copying var/lib content except libvirt..Size:$fileListVarLibSize"
  $DU -csh $fileListVarLib >> $logFile
  $CP -af $fileListVarLib $common_dir/var/lib/
  copiedFileListVarLib=`$DU -csh $common_dir/var/lib/ |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Copy complete. Copied size:$copiedFileListVarLib"
  logMsg $logFile "Removing copied var/lib content.."
  $RM -fr $fileListVarLib

  $MKDIR -p $common_dir/etc
  fileListEtc=`$LS -d $lv_root_dir/etc/* |$EGREP -v "${lv_root_dir}${ETC_GRUBD_DIR}|$lv_root_dir/etc/default"`
  fileListEtcSize=`$DU -csh $fileListEtc |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Copying etc content except grub.d and default..Size:$fileListEtcSize"
  $DU -csh $fileListEtc >> $logFile
  $CP -af $fileListEtc $common_dir/etc
  copiedFileListEtc=`$DU -csh $common_dir/etc |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Copy complete. Copied size:$copiedFileListEtc"
  logMsg $logFile "Removing copied etc content.."
  $RM -fr $fileListEtc

  $MKDIR -p $common_dir/etc/default
  fileListEtcDefault=`$LS -d $lv_root_dir/etc/default/* |$EGREP -v "/etc/default/grub"`
  fileListEtcDefaultSize=`$DU -csh $fileListEtcDefault |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Copying etc default content except grub..Size:$fileListEtcDefaultSize"
  $DU -csh $fileListEtcDefault >> $logFile
  $CP -af $fileListEtcDefault $common_dir/etc/default
  copiedFileListEtcDefault=`$DU -csh $common_dir/etc/default/ |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Copy complete. Copied size:$copiedFileListEtcDefault"
  logMsg $logFile "Removing copied etc/default content.."
  $RM -fr $fileListEtcDefault

  # NOTE: we leave an etc dir under $lv_root_dir rather than moving its contents up
  # so that modifying grub.d doesn't require any special hooks for having contents
  # in a non-standard directory.  We therefore end up with /home/etc/grub.d and
  # /home/etc/default/grub once the partition renaming below is completed.
  if [ -e "$lv_root_dir/var/lib/libvirt" ]; then
      $MV -f $lv_root_dir/var/lib/libvirt $lv_root_dir/
  fi
  $MV -f $lv_root_dir/var/log/ $lv_root_dir/
  $MV -f $lv_root_dir/opt/sonus/* $lv_root_dir/
  $MV -f $lv_root_dir/home/* $lv_root_dir/ 

  $RM -fr $lv_root_dir/home/ $lv_root_dir/var $lv_root_dir/opt

  $MKDIR -p $lv_root_dir/sftproot
  $CHMOD 755 $lv_root_dir/sftproot
  $MV $lv_root_dir/external $lv_root_dir/staging $lv_root_dir/sftproot/

  logMsg $logFile "Unmounting $lv_root.."
  $UMOUNT $lv_root
  $LVCHANGE -a y $lv_root
  $E2FSCK -y -f $lv_root 2>> $logFile
  logMsg $logFile "Re-sizing $lv_root.."
  $RESIZE2FS -f -M $lv_root >> $logFile 2>&1
  logMsg $logFile "Reducing LV size of $lv_root by ${lv_root_reduce_size}G..."
  $LVREDUCE -L-${lv_root_reduce_size}G $lv_root -f >> $logFile 2>&1
  logMsg $logFile "Creating new LVMs root1 and root2 in debian VG"
  $LVCREATE -nroot1 -L${lv_root_size}G debian >> $logFile 2>&1
  $LVCREATE -nroot2 -L${lv_root_size}G debian >> $logFile 2>&1
  logMsg $logFile "Creating file system on $lv_root1 and $lv_root2.."
  $MKE2FS -F -j $fsParam $lv_root1 >> $logFile 2>&1
  $MKE2FS -F -j $fsParam $lv_root2 >> $logFile 2>&1
  logMsg $logFile "Extending $lv_root to occupy leftover space in debian VG.." 
  $LVEXTEND -f -l +100%FREE $lv_root >> $logFile 2>&1

  $E2FSCK -y -f $lv_root >> $logFile 2>&1
  $RESIZE2FS -f $lv_root >> $logFile 2>&1
  logMsg $logFile "Renaming $lv_root to $lv_common.."
  $LVRENAME /dev/debian/root /dev/debian/common
  $LVSCAN >> $logFile 2>&1

  
  $MOUNT $lv_common $lv_common_dir >> $logFile 2>&1
  $MOUNT $lv_root1 $lv_root_dir >> $logFile 2>&1
  $MOUNT $lv_root2 $lv_root2_dir >> $logFile 2>&1

  lv_common_id=$($BLKID $lv_common |$AWK {'print $2'}|$TR -d \")
  logMsg $logFile "Updating /etc/fstab with UUID of $lv_common=$lv_common_id"

  # Check if it is 5.0 ISO case. If yes, /home should be mounted as ext4 in case of ISO, upgrade case is taken care by postInstall.sh
  if [ "$upgradeCase" == "false" ];then 
     $ECHO "$lv_common_id /home ext4    defaults,noatime,nodev,data=ordered,barrier=0        0       2" >> $common_dir/etc/fstab
  else
     $ECHO "$lv_common_id /home ext3    defaults,noatime,nodev        0       2" >> $common_dir/etc/fstab
  fi

  logMsg $logFile "Creating symlinks.."
  $LN -fs $HOME_SFTPROOT/external $common_dir/opt/sonus/external
  $LN -fs /home/log $common_dir/var/log
  $LN -fs $HOME_SFTPROOT/staging $common_dir/opt/sonus/staging
  $LN -fs $HOME_ETC_GRUBD_DIR ${common_dir}${ETC_GRUBD_DIR}
  $LN -fs $HOME_ETC_DEFAULT_DIR/grub ${common_dir}${ETC_DEFAULT_GRUB}
  $LN -fs $HOME_ETC_DEFAULT_DIR/grub.d ${common_dir}${ETC_DEFAULT_GRUBD}
  $MKDIR -p $common_dir/home

  logMsg $logFile "Removing evlog details from fstab.."
  $SED -i '/evlog/d' $common_dir/etc/fstab

  backupSize=`$DU -csh $common_dir/ |$GREP total |$AWK '{print $1}'`
  logMsg $logFile "Restoring root backups to root1 and root2. Size:$backupSize"
  $CP -af $common_dir/* $lv_root_dir/  
  $CP -af $common_dir/* $lv_root2_dir/

  $SED -i "s/debian-root/debian-root1/" $lv_root_dir/etc/fstab
  $SED -i "s/debian-root/debian-root2/" $lv_root2_dir/etc/fstab
  
  # not only update currently used /etc/grub.d, but update the newly setup
  # common version that will be mounted as /home
  logMsg $logFile "Setting up boot menu to boot from $lv_root1 on reboot.."
  $SH $SWITCH_BOOT_SH "root1" $lv_common_dir &>> $logFile

  if [ "$upgradeCase" == "true" ];then 
     # Set marker to upgrade/install through root
     $ECHO "PLATFORM_MANAGER_KEY" > $lv_root_dir/opt/sonus/.parentScriptMarker
     logMsg $logFile "Copying new sbx Init script to $lv_root_dir/opt/sonus/etc/init.d/.."
     $RM -f $lv_root_dir/opt/sonus/etc/init.d/sbx
     $CP -f $lv_common_dir/sftproot/staging/sbxStart.sh $lv_root_dir/opt/sonus/etc/init.d/sbx 2>> $logFile

     # Copy revert script to older partition for Jviewer based reverts
     $CP -f $lv_common_dir/sftproot/staging/sbxRevert.pl  $lv_root2_dir/opt/sonus/ 
 
     $CP -f $lv_common_dir/sftproot/staging/sbxRevert.pl  $lv_root_dir/opt/sonus/  
  
     # Copy the latest checkDiskUsage.sh for sysdump backward compatiblity
     logMsg $logFile "Copying the latest checkDiskUsage.sh "
     $CP -f $lv_common_dir/sftproot/staging/checkDiskUsage.sh $lv_root_dir/opt/sonus/sbx/scripts/
     $CP -f $lv_common_dir/sftproot/staging/checkDiskUsage.sh $lv_root_dir/opt/sonus/

     $CP -f $lv_common_dir/sftproot/staging/checkDiskUsage.sh $lv_root2_dir/opt/sonus/sbx/scripts/ 
     $CP -f $lv_common_dir/sftproot/staging/checkDiskUsage.sh $lv_root2_dir/opt/sonus/ 

     # Create /var/log/sonus/install directory
     $MKDIR -p $lv_common_dir/log/sonus/install 
     $CHMOD 755 $lv_common_dir/log/sonus/install 
     $MV -n $lv_root_dir/root/*install-log* $lv_common_dir/log/sonus/install 

     # To take care of some corner cases in revert, need to create some markers and
     # edit some files on old partition
     updateBackupPart 
  fi

  logMsg $logFile "Copying $logFile to install log directory($lv_common_dir/log/sonus/install/) and rebooting.."
  logMsg $logFile "Stopping service $NAME"
  $CP -f $logFile $lv_common_dir/log/sonus/install

  $UMOUNT $lv_root1
  $UMOUNT $lv_root2
  $UMOUNT $lv_common
  $RM -fr $common_dir
  
  # Is this required?
  $E2FSCK -y -f $lv_root1
  $E2FSCK -y -f $lv_root2 
  $E2FSCK -y -f $lv_common

  if [ -e /root/bootSda7 ]; then 
     $ECHO -e " t\n7\n82\nw " | $FDISK $primary_device
     umountDrbd
  fi

  $ECHO "`$DATE +'%a %b %e %H:%M:%S %Z %Y'` Disk partition completed. System will reboot now..." | $TEE /dev/tty1
  $SBIN_REBOOT 2>&1 | $TEE /dev/tty1
}


stop()
{
  $LOGGER -t $0 "Stopping $NAME"
  $ECHO "Stopping $NAME"
}

#
# See how we were called...
#
case "$1" in
    start)
           start
        ;;

    stop)
          stop
        ;;
    status)
        ;;

    *)
	  $ECHO "Usage: $0 {start|stop}"
	  exit 1
esac

# to handle reboot, exit with return 1 
exit 1
