#!/usr/bin/perl -w

#############################################################
#
# Copyright (c) 2014 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# sbxTargetRevert.pl
#
# Gautham Nimmagadda
# 2/1/09
#
# Updated/Re-Designed 
# 16/07/14
#
# Module Description:
#
# This script is invoked by sbxStart.sh on a reverted partition. 
# This script mounts p2 and syncs current reverted data into p2.
# This also updates grub menu to point to the correct root.
#
# This script is copied to SBX tar.gz as sbxRevert.pl
#
#############################################################

BEGIN {
  # locate packages in places other than the standard perl path
  unshift(@INC,"/opt/sonus/bin");
  unshift(@INC,"/opt/sonus/staging");
  unshift(@INC,"/opt/sonus/revert");
}

use File::Copy;
use sonusCommands;
use sonusCommonFiles;
use sbxCommonFunctions;

no warnings 'once';

my $softwareUpgradeCommitPendingFile="$LSWU_COMMIT_PENDING";
if( ! -e "$softwareUpgradeCommitPendingFile" )
{
    $softwareUpgradeCommitPendingFile="$OLD_LSWU_COMMIT_PENDING";
}

my $rollingLogFile="$SBX_INSTALL_LOG";
if( ! -e "$rollingLogFile" )
{
    $rollingLogFile="/root/sbx-install-log";
}

my $mountedLV=`$DF -kh |$GREP debian-root |$HEAD -n1 | $AWK '{print \$1}'`;
chomp($mountedLV);

my $majorOsVersion="03";

if( $mountedLV eq "/dev/mapper/debian-root1" )
{
  $mountedLV="/dev/mapper/debian-root1";
  $targetLV="/dev/mapper/debian-root2";
  $nextBoot="root2";
  $currentBoot="root1";
}
else
{
  $mountedLV="/dev/mapper/debian-root2";
  $targetLV="/dev/mapper/debian-root1";
  $nextBoot="root1";
  $currentBoot="root2";
}

# if the service command wrapper exists then we use it.  if it doesn't
# exist we are reverting to a release PRIOR to supporting it and just
# use the service command directly.
my $serviceCmd="";
if( -e $SERVICE_SH )
{
  # the older software supports the wrapper
  $serviceCmd="$SERVICE_SH";
}
elsif( -e $STAGING_SERVICE_SH )
{
  # we can use the new one on the older software, given what the
  # wrapper does and how it does it.
  $serviceCmd="$STAGING_SERVICE_SH";
}
else
{
  $serviceCmd="service";
}

##############################################################
# Utility to log a message to STDOUT: prints time stamp also
##############################################################
sub logMsg
{
    my $msg = shift;
    my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime(time);
    my $timeDate = sprintf "%4d-%02d-%02d %02d:%02d:%02d",
                           $year+1900, $mon+1, $mday, $hour, $min, $sec;
    print(STDOUT "$timeDate: $msg\n");

    my $dateStr=`$DATE +"%b %d %T.%3N %Y"`;
    chomp($dateStr);
    `$ECHO -e "$dateStr: $msg" >> $rollingLogFile`;
}

###########################################################################################
# Utilities to log header and footer
###########################################################################################
sub logStartUpdate
{
  my $caller = shift;
  logMsg("==========================================================================");
  logMsg("Start: Sonus sbx Installation (Initiated through $caller)");
  logMsg("==========================================================================");
}

sub logEndUpdate
{
  if (-d "/root/.rpmdb")
  {
     $rpmDbPath = "--dbpath /root/.rpmdb";
  } 
  logMsg("==========================================================================");
  logMsg("End: Sonus sbx Installation");
  my $rbbnPkgs = "bios-update cnxipm cps ema lcamodules pamvalidator sbc sl";
  my $currentInstallations="";
  my $cps = `$DPKG -l cps 2>/dev/null | $GREP ^ii | $AWK '{ print \$2 }'`;
  if ($cps eq "")
  {
      $currentInstallations=`$RPM -qa --last $rpmDbPath`;
  } else {
      $currentInstallations=`$DPKG -l $rbbnPkgs 2>/dev/null | $GREP ^ii | $AWK 'BEGIN {  OFS = "_" }{ print \$2, \$3, \$4 }'`;     
  }
  logMsg("Updated Version(s): \n$currentInstallations");
  logMsg("==========================================================================");
}

#########################################################################
# Enable DB replication trigger for Oracle based release
#########################################################################
sub enableDbRepTrigger
{
    if ( ! -d "/home/postgresql" )
    {
        logMsg("Enabling Oracle DB replication triggers");
        system('su - oracle -c "echo \"alter trigger platform.sbx_dbrep_status_change enable; \n exit;\n \" | sqlplus \"/ as sysdba\"" ');
    }
}

#########################################################################
# Creating the extentions as part of the REVERT process.
#########################################################################
sub createExtentions
{
    if (( -e "/opt/sonus/sbx/psx/sql/rbbn_aes_encr.control" ) && ( -e "/opt/sonus/sbx/psx/sql/rbbn_aes_encr--0.0.1.sql" ) && ( -e "/opt/sonus/sbx/psx/lib/librbbn_aes_encr.so" ))
    {
        logMsg("Creating PostgreSQL Extensions");
        system("su - postgres -c \"psql -a -f $SONUS_PSX_SQL_DIR/CreateExtensions.sql\"");
        if( $? != 0)
        {
            logMsg("Failed to create the postgresql extensions.");
        }
        else
        {
            logMsg("Successfuly created the postgresql extensions.");
        }
    }
}

#########################################################################
# For a child and main process exits, so that the caller is not blocked
#########################################################################
sub runChild()
{
  my $pid;
  if ($pid = fork)
  {
    # Parent exits so that the caller can keep going, if they didn't used '&'
    exit 1;
  }
  elsif (defined $pid)
  {
    # everything good, return
    chdir "/"; # so that uninstall/install don't cause any cwd errors
    return;
  }
  else
  {
    logMsg("Unable to fork child");
    exit 1;
  }
}

#############################################################
# check and create marker for model revert
#############################################################
sub checkAndSetupModelRevert()
{
  my $modelMarkerFile="/mnt/target". "$PERFORM_MODEL_UPDATE";
  my $dynamicHANewCompsFile="/mnt/target". "$DYNAMIC_HA_NEW_COMPS";
  if( -e "$modelMarkerFile" )
  {
    my $token=`$CAT $modelMarkerFile`;
    chomp($token);
    logMsg("checkAndSetupModelRevert:Model Marker exists with token:$token");
    if ( "$token" eq "updateModel_dynha" || "$token" eq "updateComplete" )
    {
      if ( -e $dynamicHANewCompsFile )
      {
        logMsg("Reading Model Component list to be temporarily disabled..");
        open (my $handle,$dynamicHANewCompsFile);
        while(my $line = <$handle>)
        {
          chomp($line);
          logMsg("Stubbing new component: $line");
          # file content will be in the form: CE_Node1_2N_Su_Comp_Ccs=CE_Node1_2N_Su_Comp_Ccs
          my ($dummy,$comp)=split("=",$line);
          my $stubCmd="$LN -fs $SONUS_OPENCLOVIS_BIN_DIR/CE_2N_Comp_SmProcess.stub $SONUS_OPENCLOVIS_BIN_DIR/$comp";
          logMsg("Creating temporary link:$stubCmd\n");
          system("$stubCmd"); 
        }
        close $handle;
        
        #Create marker for model revert
        my $modelMarker="$PERFORM_MODEL_UPDATE";
        if ( ! -d "/opt/sonus/installUpgrade/log" )
        {
          $modelMarker="/opt/sonus/performModelUpdate";
        }
        else
        {
          # Clenaup 6.0 files left
          `$RM -f $SONUS_INSTALLUPGRADE_LOG_DIR/*`;
        }
        `$ECHO -n "updateModel_dynha" > $modelMarker`;
        `$CHOWN sonusadmin:sonus $modelMarker`;
      }
      else
      {
        # We can't do much without $dynamicHANewCompsFile file.
        logMsg("WARNING! $dynamicHANewCompsFile doesnt exist. Unable to detect and stub new components.");
      }
    }
  }
  else
  {
    #No-Op
    logMsg("checkAndSetupModelRevert:Model Marker doesnt exist. No action taken[INFO].");
  }
}

#############################################################
# Syncs p2 with self data
#############################################################
sub syncP2withSelf()
{
  # target is mounted in revert()
  system("$MKDIR -p /home/linuxadmin/source; $MOUNT $mountedLV /home/linuxadmin/source");
  logMsg("Destroying p2($targetLV) data and Updating with self data.. This may take 4-5 mins..");
  system("$RM -fr /mnt/target/*");
  `$CP -af /home/linuxadmin/source/* /mnt/target`;
  logMsg("Updating p2 /etc/fstab..");
  `$SED -i "s/$currentBoot/$nextBoot/" /mnt/target/etc/fstab`;
  my $p2ModelMarker="/mnt/target". "$PERFORM_MODEL_UPDATE";
  `$RM -f $p2ModelMarker`;
  `$UMOUNT /home/linuxadmin/source/ ; $UMOUNT /mnt/target`;
  `$RM -fr /home/linuxadmin/source`; 
}

#############################################################
# Update boot menu after revert
#############################################################
sub updateBootMenu()
{
  # how we update the menu depends if we went back to
  # a version supporting grub2 or grub-legacy.  once we get
  # 2 releases past grub2 support, we should be able to remove
  # the grub-legacy support from here.

  my $totalBootOpts=`$GREP -c -e sonus-curr-ver -e sonus-prev-ver $GRUB_CONFIG`;
  if( $totalBootOpts == 2) {
    if ( -d "$HOME_ETC_GRUBD_PREV_DIR" ) {
      # we reverted from a system supporting grub2
      # restore the saved grub configuration
      `$RM -rf $HOME_ETC_GRUBD_DIR $HOME_ETC_DEFAULT_DIR`;
      `$MV $HOME_ETC_GRUBD_PREV_DIR $HOME_ETC_GRUBD_DIR`;
      `$MV $HOME_ETC_DEFAULT_PREV_DIR $HOME_ETC_DEFAULT_DIR`;

      # remove the ability to revert again by removing the
      # entry for the previous version. we also have to remove
      # the base-os entry since the previous version may have
      # been the original ISO.
      # note: because we re-stored the previous config, everything
      # else is already correct (single user and current version)
      `$SED -i -e '/sonus-prev-ver/d' -e '/base-os/d' -e '/^prevKernelArgs/s/=.*/=""/' $SONUS_GRUB_MENU_SDA1_FILE`;

      # update grub.cfg
      `$UPDATE_GRUB_SH 2>&1 | $TEE -a $rollingLogFile`;

      # update the default entry -- current entry is last since prev
      # has been deleted
      my $lastOpt=`$GREP -c "^sonus_linux_entry " $SONUS_GRUB_MENU_SDA1_FILE`;
      chomp($lastOpt);
      $lastOpt--;
      `$GRUB_SET_DEFAULT $lastOpt`;
    } else {
      # we reverted from a grub2 install to a grub-legacy install
      # note: this is deprecated functionality. needed until grub-legacy is
      # no longer a concern (we support direct upgrade for 2 releases, so after
      # after 10.x...)

      # restore the saved grub configuration
      `$RM -fr /boot/grub`;
      `$MV /boot/grub.prev /boot/grub`;

      # remove /home/etc which we created only as part of the grub2
      # upgrade.  it didnt exist before that, and since we reverted back
      # to grub-legacy we don't want it to exist now.
      `$RM -fr /home/etc`;

      # We updated to grub2, so last used was not marked in menu.lst.  But
      # now the version we reverted to will look for last used in menu.lst.
      # So modify the menu.lst we just restored so that revert functionality
      # will realize revert already happened and cannot be done again.
      # note: Mimic what the upgrade script and revert script used to do:
      # 1) create a temporary file without the 2 SBC entries
      # 2) create a temporary file with the current entry and add last used
      #    (this is the actual entry since we didnt change menu.lst on upgrade)
      #    note: delete any last used first. if we upgrade, revert, upgrade, revert,
      #          we would get multiple last used statements
      # 3) add the last entry while changing the partition to the alternate one
      # 4) add the last entry
      # 5) add the end tag to the file
      # 6) manipulate default and root device the same as we used to
      my $tmpGrubConfig="$GRUB_MENU_LST.tmp";
      my $tmpGrubConfig2="$GRUB_MENU_LST.tmp2";
      my $timeStamp=`$DATE "+%FT%T%z"`;
      chomp($timeStamp);
      `$SED -e '/Sonus SBC/,\$d' $GRUB_MENU_LST > $tmpGrubConfig`;
      `$TAIL -n 7 $GRUB_MENU_LST | $SED -e '/^[ ]*\$/d' -e '/END DEBIAN/d' > $tmpGrubConfig2`;
      `$SED -i -e 's/ (Last Used:.*\$//' -e "/^title/s/\$/ (Last Used: $timeStamp)/" $tmpGrubConfig2`;
      `$ECHO >> $tmpGrubConfig`;
      `$CAT $tmpGrubConfig2 >> $tmpGrubConfig`;
      `$ECHO >> $tmpGrubConfig`;
      `$CAT $tmpGrubConfig2 >> $tmpGrubConfig`;
      `$TAIL -n 2 $GRUB_MENU_LST >> $tmpGrubConfig`;

      my $totalOpts=`$GREP "/vmlinuz" $tmpGrubConfig |$GREP -v "#" |$WC -l`;
      chomp($totalOpts);
      my $currOpt=`$EXPR $totalOpts - 1`;
      chomp($currOpt);
      `$SED -i "/^default/c\\default         $currOpt" $tmpGrubConfig`;
      `$SED -i "s|$targetLV|$mountedLV|" $tmpGrubConfig`;
      my $p2rootLine=`$GREP -n BOOTIF $tmpGrubConfig |$TAIL -n2 | $HEAD -n1  |$AWK '{ print \$1 }'| $TR -d ":kernel"`;
      my $p2root=`$GREP -n BOOTIF $tmpGrubConfig |$TAIL -n2 | $HEAD -n1 | $SED -r 's/^.*root=([^ ]+).*\$/\\1/'`;
      chomp($p2root);
      chomp($p2rootLine);
      `$SED -ie "\`$ECHO $p2rootLine\`s,$p2root,$targetLV," $tmpGrubConfig`;
      `$CP -f $tmpGrubConfig $GRUB_MENU_LST`;
      `$RM -f $tmpGrubConfig $tmpGrubConfig2`;

      # re-install grub-legacy to the MBR
       logMsg("grub2 currently installed: Re-installing grub-legacy to the MBR on ${sbxCommonFunctions::primary_device}...");
      `$GRUB_INSTALL $sbxCommonFunctions::primary_device 2>&1 | $TEE -a $rollingLogFile`;
    }
  }
}

#############################################################
# revert sonusDB
#############################################################
sub revertSonusDb()
{
  my $configureDBMode="NEW";

  # Case: upgrade of Db failed, do nothing and start db
  if ( ! -e "$DB_UPGRADE_MARKER")
  {
    if ( -d "/home/postgresql/current" )
    {
        logMsg("Reverting sonusDB..");
        $configureDBMode="REVERT";
    }
    # Case: If postgresql dbs are not backed up, so revert here means configuring it as NEW again and removing cdbs. 
    else
    {
       logMsg("Sonusdb will be configured as NEW. Removing CDBs to be in sync with sonusdb...");
       `$REMOVECDB_SH`; 
    }

    logMsg("Starting the postgresql...");
    `$SERVICE_SH postgresql start > /dev/null 2>&1`;

    createExtentions();
    #system("$SH -c \"cd $SONUS_PSX_SQL_DIR/; $PERL configureDB -install $configureDBMode -loglevel 2 -force Y -logdir /tmp\"");

    if( $? != 0)
    {
      logMsg("Failed to Revert Database..");
      exit 1;
    }

    logMsg("Enabling the replication triggers..");
    enableDbRepTrigger();
  }
  else
  {
    logMsg("DB Upgrade Marker present, No action taken.");
    `$RM -rf $DB_UPGRADE_MARKER`;
  }

  if ( -e "/home/admin/oracle/oracleDB.sha256sums" && ! -e "/home/oracle/oracleDB.sha256sums")
  {
    logMsg("Restoring oracleDB checksum file while reverting to old version");
    `$CP /home/admin/oracle/oracleDB.sha256sums /home/oracle/oracleDB.sha256sums`;
  }

  # Stop policy DB so that it gets started as part of SBX app after all the revert is completed
  logMsg("Stopping policy DB...");
  `$SERVICE_SH postgresql stop > /dev/null 2>&1`;

  logMsg("Checking the status of Oracle DB");
  `$ORA_START_SH status > /dev/null 2>&1`;
  if ( $? == 0 )
  {
    logMsg("Stopping Oracle DB...");
    `$ORA_START_SH stop  > /dev/null 2>&1`;
  }
}


#############################################################
# restore home and other backed up directories.
#############################################################

sub restoreBackup()
{
  my $HOME_BACKUP_TAR="$HOME_BACKUP_DIR/home_backup.tar.gz";

  if( -d "/opt/sonus/.rpmdb_bk" )
  {
    logMsg("Restoring rpmdb..");
    `$RM -fr /root/.rpmdb`;
    `$CP -ar /opt/sonus/.rpmdb_bk /root/.rpmdb`;
    `$RM -fr /opt/sonus/.rpmdb_bk`;
  }

  my $sbcMajorVersion=`$SWINFO_SH -l | $GREP SBC: | $AWK '{print \$2}' | $AWK -F. '{print \$1}' | $SED -e 's/V//'`;
  chomp($sbcMajorVersion);
  my $sbcMinorVersion=`$SWINFO_SH -l | $GREP SBC: | $AWK '{print \$2}' | $AWK -F. '{print \$2}'`;
  chomp($sbcMinorVersion);

  if( -e "$HOME_BACKUP_TAR" )
  {
    logMsg("Restoring /home..");
    `$RM -frv \`$LS -d /home/* | $EGREP -v "ora|common|libvirt|log|external|staging|sftproot|backup|etc"\``;
    `$TAR -xvzf $HOME_BACKUP_TAR -C /home/`;

    logMsg("Moving log directory to the right location as per the release");
    if ( $sbcMajorVersion eq "06" && ( $sbcMinorVersion eq "00" || $sbcMinorVersion eq "01" ) )
    {
      logMsg("Moving /home/log to /home/sftproot/log for $sbcMajorVersion.$sbcMinorVersion");
      `$MV /home/log /home/sftproot/log`;
      `$RM -f /var/log`;
      `$LN -s /home/sftproot/log /var/log`;
      `$RM -rf /home/sftproot/evlog`;
    }
    elsif ( $sbcMajorVersion eq "04" || $sbcMajorVersion eq "05" ) 
    {
      # it must be 05 release where log resides in /var
      logMsg("Moving /home/log to /var/log for $sbcMajorVersion.$sbcMinorVersion");
      `$RM -rf /var/log`;
      `$MV /home/log /var/log`;
    }
    `$MV -f $HOME_BACKUP_TAR $HOME_BACKUP_TAR.reverted;`;
  }
  else
  {
    logMsg("$HOME_BACKUP_TAR doesnt exist! Failed to restore /home.. Cannot Revert..");
    exit 1;
  }

  my $permFile="$HOME_BACKUP_DIR/common_dir_perms";
  if( -e "$permFile" )
  {
    logMsg("Restoring common dir permissions..");
    open(my $handle,$permFile);
    while( my $line = <$handle>)
    {
      my($name,$owner,$group,$perm)=split(" ",$line);
      logMsg("Directory Name=$name,owner=$owner,group=$group,perm=$perm\n");
      if ( $name eq $SONUS_LOG_EVLOG_DIR && ( $sbcMajorVersion eq "04" || $sbcMajorVersion eq "05" || ( $sbcMajorVersion eq "06" && ( $sbcMinorVersion eq "00" || $sbcMinorVersion eq "01" ) ) ) )
      {
           logMsg("Removing link to $SONUS_LOG_EVLOG_DIR as it will be the mountpoint");
           `$RM -f $SONUS_LOG_EVLOG_DIR`;
           `$MKDIR -p $SONUS_LOG_EVLOG_DIR`;
           `$CHOWN -R $owner:$group $name`;
      }
      else
      {
          if ( -d "$name" )
          {
            `$CHOWN -R $owner:$group $name`;
            `$CHMOD -R $perm $name`;
          }
      }
    }
    close $handle;

    my $currentSonusDirs=`$FIND /var/log/sonus/ -type d -print`;
    chomp($currentSonusDirs);
    my @arr=split /^/,$currentSonusDirs;
    foreach(@arr)
    {
      my $currentDir=$_;
      chomp($currentDir);
      `$GREP -q "$currentDir" $permFile`;
      if ( $? ne 0 )
      {
        logMsg("Removing dir $currentDir...");
        `$RM -rf $currentDir`;
      }
    }
    `$MV -f $permFile $permFile.reverted;`;

    if ( -e "$SONUS_LOG_CNXIPM_DIR" )
    {
      # ownership on cnxipm dir and files beneath can be different. A safe way to resolve
      # ownership issues is to set www-data ownership to dir and  PM files.
      logMsg("Setting www-data:www-data ownership to cnxipm dir..");
      `$CHOWN -R www-data:www-data $SONUS_LOG_CNXIPM_DIR`;
    }

    if( -e "$SONUS_LOG_EVLOG_DIR/faillog" )
    {
	`$CHOWN root:root $SONUS_LOG_EVLOG_DIR/faillog`;
	`$CHMOD 600 $SONUS_LOG_EVLOG_DIR/faillog/*`;
    }
    if( -e "$SONUS_LOG_EVLOG_DIR/faillogOS" )
    {
	`$CHOWN root:root $SONUS_LOG_EVLOG_DIR/faillogOS`;
	`$CHMOD 600 $SONUS_LOG_EVLOG_DIR/faillogOS/*`;
    }
    if( $sbcMajorVersion eq "06" && ( $sbcMinorVersion eq "00" || $sbcMinorVersion eq "01" ))
    {
        # In these releases we have $HOME_SFTPROOT/var/log a link to ../log
        `$RM -rf $HOME_SFTPROOT/var`; 
        `$MKDIR -p $HOME_SFTPROOT/var`;
        `$CHGRP sftponly $HOME_SFTPROOT/var`;
        `$CHMOD 750 $HOME_SFTPROOT/var`;
        `$LN -s ../log $HOME_SFTPROOT/var/log`;
    }
  }
  else
  {
    logMsg("$permFile doesnt exist!");
    `$CHMOD -R 777 $SONUS_STAGING_DIR/ $SONUS_EXTERNAL_DIR/"`;
    `$CHOWN -R root:root $SONUS_STAGING_DIR/ $SONUS_EXTERNAL_DIR/"`;
  }

  if( -e "$HOME_BACKUP_DIR/sonuscert.p12" )
  {
    logMsg("Restoring certification file sonuscert.p12...");
    `$MV -f $HOME_BACKUP_DIR/sonuscert.p12 $SONUS_EXTERNAL_DIR/`;
  }

  if(getgrnam("sftponly"))
  {
    `$CHMOD 755 $HOME_SFTPROOT`;
    `$CHOWN root:sftponly $HOME_SFTPROOT`;
  }
  else
  {
    logMsg("group sftponly doesnt exist.");
    # Set /home permissions since this dir will be equivalent to /home
    # w.r.t being a parent directory for log, staging, external etc.
    `$CHMOD --reference=/home $HOME_SFTPROOT`;
    `$CHOWN --reference=/home $HOME_SFTPROOT`;
  }
  # log files should have write permissions for sonusadmin user
  `$CHMOD g+w $SONUS_STAGING_DIR/*.out 2>/dev/null`;

  if ( -e "$POSTGRES_LOG" )
  {
    logMsg("Updating permissions in postgresql log dir.");
    `$CHOWN postgres:postgres $POSTGRES_LOG/postgresql.log*`;
    `$CHOWN postgres:adm $POSTGRES_LOG/postgresql*main*`;
  }
} 

#############################################################
# This function is used for reset/restart DRBD partition and start sync.
# This should be run from both the sides
# E.g. moving from DRBD 8.4 back to 8.3 as part of revert
#############################################################
sub reconfig83xDrbd()
{
  my $logFile = "$SONUS_STAGING_DIR/reconfig83xDrbd.log";
  `$ECHO "Starting reconfig83xDrbd()" >> $logFile`;

  my $sbxConfFile="$SBXCONF_FILE";
  if( ! -e "$sbxConfFile" )
  {
    $sbxConfFile="$SBX_CONF";
  }

  my $role=`$GREP role $sbxConfFile | $AWK -F "="  '{ print \$2 }' | $AWK '{ print \$1 }'`;
  chomp($role);
  # Chose different port numbers for active and standby so that they don't connect.
  my $drbdPort="7788";
  if( "$role" eq "2" )
  {
    $drbdPort="7787";
  }
  logMsg("reconfig83xDrbd: temp DRBD port is:$drbdPort");

  my $initdDrbd="$SERVICE_SH drbd";
  `$initdDrbd stop >> $logFile 2>&1`;
  `$SED  -i 's/:7789/:$drbdPort/g' $ETC_DRBD_CONF >> $logFile 2>&1`;
  `$MODPROBE drbd >> $logFile 2>&1`;
  `$DD if=/dev/zero of=${sbxCommonFunctions::primary_device}6 bs=10M count=1; sync >> $logFile 2>&1`;
  `$ECHO "yes" | $DRBDADM create-md mirror >> $logFile 2>&1`;
   sleep(2);
  `$BLKID ${sbxCommonFunctions::primary_device}6  >> $logFile`;
  `$initdDrbd start >> $logFile 2>&1`;
  `$DRBDADM secondary mirror >> $logFile 2>&1`;
  `$DRBDADM disconnect mirror >> $logFile 2>&1`;
  `$CAT /proc/drbd >> $logFile 2>&1`;
  `$DRBDADM -- --overwrite-data-of-peer primary mirror >> $logFile 2>&1`;
  `$MKE2FS -j /dev/drbd0 >> $logFile 2>&1`;
  `$DRBDADM -- --clear-bitmap new-current-uuid  mirror >> $logFile 2>&1`;
  `$DRBDADM new-current-uuid mirror >> $logFile 2>&1`;
  # redirecting error because these are really not required but just in case
  `$DRBDADM secondary mirror >> $logFile 2> /dev/null`;
  `$DRBDADM detach mirror >> $logFile 2> /dev/null`;
  `$initdDrbd stop >> $logFile 2>&1`;
  `$ECHO yes | $DRBDADM  -- 0:0:0:0:1 set-gi mirror >> $logFile 2>&1`;
   sleep(1);
  `$initdDrbd status >> $logFile 2>&1`;
  `$CAT /proc/drbd >> $logFile 2>&1`;
  if( "$role" eq "2" )
  {
    # Since revert happens simultaneously and independently on both nodes, allow one of the nodes
    # to start first. Simultaneous start seems to cause issues with drbd while startup.
    # Wait for 15 minutes and then start anyway.
    logMsg("reconfig83xDrbd: Role is:$role. Waiting in a loop for active to come up before continuing..");
    my $count=0;
    while (sleep 30)
    {
      my $peerState=`$SWINFO_SH | $GREP Current| $TAIL -n1 | $AWK -F: '{print \$2}'`;
      chomp($peerState);
      if ($peerState =~ /active/)
      {
          logMsg("reconfig83xDrbd: peerState:$peerState. Count:$count. Ending loop and continuing with revert process..");
          last;
      }
      elsif ( $count eq 30 )
      {
          logMsg("reconfig83xDrbd: Waited for 15 minutes for peer to become active. But peerState:$peerState. Ending loop and continuing with revert process..");
          last;
      }
      else
      {
          logMsg("reconfig83xDrbd: peerState:$peerState. Count:$count. Waiting for peer to become active..");
          $count ++;
      }
    }
  }
  `$SED  -i 's/:$drbdPort/:7789/g' $ETC_DRBD_CONF >> $logFile 2>&1`;
  `$initdDrbd start >> $logFile 2>&1`;
  `$ECHO "drbdadm show-gi mirror:" >>  $logFile`;
  `$DRBDADM show-gi mirror >> $logFile 2>&1`;
  `$ECHO "cat /proc/drbd:" >> $logFile`;
  `$CAT /proc/drbd >> $logFile 2>&1`;
  `$ECHO "Finished reconfig83xDrbd()" >> $logFile`;
}

#############################################################
# post revert updates are neccessory to make sure sbx works fine
# even with re-partitioned disk.
#############################################################
sub postRevert()
{
  logMsg("Updating sbx startup script..");
  # modifications to sbx startup script
  `$SED -i --follow-symlinks 's,$FSCK,$E2FSCK,' $ETC_INITD_SBX`;
  `$SED -i --follow-symlinks 's,$SONUS_LOG_EVLOG_DIR,$HOME_SFTPROOT/log/sonus/evlog,' $ETC_INITD_SBX`;

  # Update sbxCleanup.sh to unmount drbd
  my $sbxCleanupFile="$SBX_CLEANUP_SH";
  if ( -f $sbxCleanupFile )
  {
     logMsg("Updating sbxCleanup.sh..");
     `$SED -i '/stop)/a \\\\t#Unmount drbd, it will be mounted on startup as and when required\\n\\t$UMOUNT /dev/drbd0' $sbxCleanupFile`;
     `$SED -i '/zap)/a \\\\t#Unmount drbd, it will be mounted on startup as and when required\\n\\t$UMOUNT /dev/drbd0' $sbxCleanupFile`;
  }

  # Remove openclovis var. It will be freshly created on startup.
  logMsg("Saving pre-revert openclovis logs..");
  `$SAVE_OC_LOGS_SH`;
  logMsg("Removing openclovis var dir.."); 
  `$RM -fr $SONUS_OPENCLOVIS_VAR_DIR`;

  # Reset SoftwareUpgradeStatus file if present
  my $softwareUpgradeStatusFile="$SOFTWARE_UPGRADE_STATUS";
  my $peerScript="$SONUS_PEER_CNX_EXPECT";

  if( -e "$softwareUpgradeStatusFile" )
  {
      # Cleanup marker keys if any
      `$RM -f $SONUS_LOG_UPGRADE_DIR/*.key`;
      logMsg("Resetting UpgradeStatus to ServerReverted in softwareUpgradeStatus file..");
      `$SED -i "s/^UpgradeStatus.*/UpgradeStatus=ServerReverted/" $softwareUpgradeStatusFile`;
      `$SED -i "s/^primaryUpgradeStatus.*/primaryUpgradeStatus=reverted/" $softwareUpgradeStatusFile`;
      `$SED -i "s/^secondaryUpgradeStatus.*/secondaryUpgradeStatus=reverted/" $softwareUpgradeStatusFile`;
      if ( -e "$peerScript" )
      {
          `$peerScript $RM -f $SONUS_LOG_UPGRADE_DIR/*.key`;
          logMsg("Resetting UpgradeStatus to ServerReverted on peer..");
          `$peerScript $SED -i "s/^UpgradeStatus.*/UpgradeStatus=ServerReverted/" $softwareUpgradeStatusFile`;
          `$peerScript $SED -i "s/^primaryUpgradeStatus.*/primaryUpgradeStatus=reverted/" $softwareUpgradeStatusFile`;
          `$peerScript $SED -i "s/^secondaryUpgradeStatus.*/secondaryUpgradeStatus=reverted/" $softwareUpgradeStatusFile`;
      }
  }

  # Remove leftover PM markers if any
  `$RM -fr $SONUS_LOG_CNXIPM_DIR/admin/tmp/* > /dev/null 2>&1`;
 
  # Update OS version file to make sure upgrade is not attempted to intermediate version.
  my $osVersionFile="$CONNEXIP_OS_VERSION";
  my $connexIPOSVersion=`$CAT $osVersionFile | $AWK '{print \$3}' | $SED 's/^v//'`;
  my @values = split('\.', $connexIPOSVersion);
  $majorOsVersion=$values[0];
 
  # Identify if current version is 02.0x
  if ( $majorOsVersion =~ /02/ || $majorOsVersion =~ /2/)
  {
     my $originalString=`$CAT $osVersionFile | $AWK '{print \$3}'`;
     chomp($originalString);
     if ( $originalString !~ /Reverted/)
     {
          copy($osVersionFile,$osVersionFile."_swinfo");
          # Eg: Original Version=02.02.00-R000
          ###  New Version=02.Reverted02.00-R000
          
          $values[1] = "Reverted".$values[1] ;
          $newString = join ( '.',@values);
          chomp($newString);
          logMsg("Updating OS Version from $originalString to $newString..");
          `$SED -i "s/$originalString/$newString/" $osVersionFile`;

          # update swinfo to show OS version as below:
          # Eg: Original display:                   V02.02.00-R000
          ###  Current Display:                    V02.Reverted02.00-R000
          ###  New display(based on SVT feedback): V02.02.00-R000 (Reverted)
          my $file="$SWINFO_SH";
          `$SED -i 's,/etc/connexip_os_version,/etc/connexip_os_version_swinfo,' $file`;
          `$SED -i 's,V\$connexipVersion,V\$connexipVersion (Reverted),' $file`;
          `$SED -i 's, \$connexipVersion, \$connexipVersion (Reverted),' $file`; # To cover 3.x
      }
      else
      {
          logMsg("Not updating OS Version since previous revert has already updated it..");
      }
  }

  # Remove faillock files
  `$RM -fr /var/log/sonus/evlog/faillog`;
  `$RM -fr /var/log/sonus/sbx/linux/faillogOS`;

  # Starting 6.x, we need to have s-bit set on this file
  if( -e "/home/oracle/product/11.2.0/bin/oracle" )
  {
    logMsg("Setting s-bit on /home/oracle/product/11.2.0/bin/oracle...");
    `$CHMOD ug+s /home/oracle/product/11.2.0/bin/oracle`;
  }

  if( -e "/home/oracle/product/12.1.0/bin/oracle" )
  {
    logMsg("Setting s-bit on /home/oracle/product/12.1.0/bin/oracle...");
    `$CHMOD ug+s /home/oracle/product/12.1.0/bin/oracle`;
  }

  my $kernelVer = `$UNAME -r`;
  chomp($kernelVer);
  my $drbdVersion = `/sbin/modinfo /lib/modules/$kernelVer/kernel/drivers/block/drbd/drbd.ko| $GREP "^version:" | $AWK '{print \$2}'`;
  chomp($drbdVersion);
  if ($drbdVersion =~ /^8.3./)
  {
    logMsg("Reverting to a release with DRBD version: $drbdVersion, resetting DRBD...");
    reconfig83xDrbd();
  }
  else
  {
    logMsg("DRBD version is: $drbdVersion. [INFO]");
  }

  # remove any knowledge of what leadership algorithm was in use
  `$RM -f $SBX_PEER_SPLIT_BRAIN_MODE_MARKER`;
}

##############################################################
# Revert driver to:
# Revert sonusDB
# Sync p2 with current partition.
# Update Grub boot menu
# Post Revert changes
#############################################################
sub revert()
{
 
 # Check if personality is set to ePSX.
 my $sbxConfFile="$SBXCONF_FILE";
 if( ! -e "$sbxConfFile" )
 {
   $sbxConfFile="$SBX_CONF";
 }
 my $personality=`$CAT $sbxConfFile |$GREP personality |$AWK -F "=" '{print \$2}'| $AWK '{print \$1}'`;
 chomp($personality);
 if( "$personality" eq "2" )
 {
   logMsg("ERROR! Revert not supported to software installed with ePSX. Exiting..");
   exit(1);
 }

 system("$MKDIR -p /mnt/target; $MOUNT $targetLV /mnt/target");

 # Remove marker file
 logMsg("removing Marker file..");
 `$RM -f $softwareUpgradeCommitPendingFile`;

 logMsg("Calling restoreBackup()..");
 restoreBackup();

 logMsg("Calling revertSonusDb()..");
 revertSonusDb();

 logMsg("Calling checkAndSetupModelRevert()..");
 checkAndSetupModelRevert();

 logMsg("Calling syncP2withSelf()..");
 syncP2withSelf();

 logMsg("Calling updateBootMenu()..");
 updateBootMenu();

 logMsg("Calling postRevert()..");
 postRevert();

 # Add marker for LSWU status
 logMsg("Adding LSWU marker file..");
 `$ECHO "userInitiated" > $REVERT_IN_PROGRESS`;
 `$ECHO "userInitiated" > $OLD_REVERT_IN_PROGRESS`;
 `$CHMOD 744 $REVERT_IN_PROGRESS`;
 `$CHMOD 744 $OLD_REVERT_IN_PROGRESS`;

 logMsg("Starting SBX service..");
 system("$serviceCmd sbx start");

 my $sbcMajorVersion=`$SWINFO_SH -l | $GREP SBC: | $AWK '{print \$2}' | $AWK -F. '{print \$1}' | $SED -e 's/V//'`;
 chomp($sbcMajorVersion);
 # evlog partition may not have proper permissions. We need to wait for application to come up and mount drbd to make changes.
 # Changes are done as follows:
 # major version 4: sonusadmin:sonus to all, 777 for config dir and 775 for rest
 # major version 5: sonusadmin:sonus to all, 777 for config dir and 775 for rest
 # major version 6: sonusadmin:upload to config, evlog, statistics, templates, announcements, sonusadmin:sonus to rest, 775 for all
 while (sleep 10)
 {
    my $state = `$SWINFO_SH -l|$GREP Current|$AWK -F: '{print \$2}'`;
    if ($state =~ /active/)
    {
       my $ret=system("$GREP -q evlog /etc/mtab"); 
       my $exitCode=$ret >> 8;
       if ( $exitCode == 0 )
       {
          logMsg("Service is up as active and evlog mounted. Updating mirror directory permissions..\(sbcMajorVersion=$sbcMajorVersion\)");
          my $evlogMountPoint="$SONUS_LOG_EVLOG_DIR";
          if ( $sbcMajorVersion eq "04" || $sbcMajorVersion eq "05" )
          {
            `$CHOWN -R sonusadmin:sonus $evlogMountPoint/*`;
            `$CHMOD -R 775 $evlogMountPoint/*`;
            if ( -d "$SONUS_LOG_EVLOG_CONFIG_DIR")
            {
              `$CHMOD -R 777 $SONUS_LOG_EVLOG_CONFIG_DIR`;
            }

          }
          elsif ( $sbcMajorVersion eq "06" )
          {
            `$CHOWN -R sonusadmin:sonus $evlogMountPoint/*`;
            `$CHMOD -R 775 $evlogMountPoint/*`;
            if ( -d "$SONUS_LOG_EVLOG_CONFIG_DIR")
            {
              `$CHOWN -R sonusadmin:upload $SONUS_LOG_EVLOG_CONFIG_DIR`;
            }
            if ( -d "$evlogMountPoint/evlog")
            {
              `$CHOWN -R sonusadmin:upload $evlogMountPoint/evlog`;
            }
            if ( -d "$evlogMountPoint/statistics")
            {
              `$CHOWN -R sonusadmin:upload $evlogMountPoint/statistics`;
            }
            if ( -d "$evlogMountPoint/templates")
            {
              `$CHOWN -R sonusadmin:upload $evlogMountPoint/templates`;
            }
            if ( -d "$evlogMountPoint/announcements")
            {
              `$CHOWN -R sonusadmin:upload $evlogMountPoint/announcements`;
            }
            if ( -d "$evlogMountPoint/config")
            {
              `$CHOWN -R sonusadmin:upload $evlogMountPoint/config`;
            }
            if ( -d "$evlogMountPoint/config/ema")
            {
              `$CHOWN -R sonusadmin:upload $evlogMountPoint/config/ema`;
            }
          }
          if ( -e "$OPERATOR_ACCESS_CONF")
          {
             `$CHOWN www-data:www-data $OPERATOR_ACCESS_CONF`;
          }
       }
       else
       {
         logMsg("WARNING: service is up as active but evlog not mounted! Unable to update mirror directory permissions..\(ret=$ret,exitCode=$exitCode\)");
       }
       last;
    }
    elsif ($state =~ /standby/)
    {
       # Restart if coming up as standby on revert
       # In case of revert, CDBs may not get connected due to LSWU states and that
       # requires restart of standby server. But there is a bug in that logic in pre-5.0.
       # Below code is to work around the issue
       # We can't make this based on timer value because of different size configs and h/w types
       # in the worst case when the service doesn't come up at all, this may run forever
       if ( $majorOsVersion =~ /02/ || $majorOsVersion =~ /2/ )
       {
          logMsg("Restarting SBX service (for CDB connection)..");
          system("$serviceCmd sbx restart");
          last;
       }
    }
 }
}

runChild();

logStartUpdate("LSWU-revert");

revert();

logEndUpdate();
exit 0;
