#!/bin/bash
#############################################################
#
# Copyright (c) 2009 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# sbxUpdate.sh
#
# Mark St. Pierre
# 6/1/09
#
# Module Description:
# Script to update application service
# Example: 
#
#    sh sbxUpdate.sh
#    
#############################################################

# Extracting Sonus command files from SBC tarball & sourcing
# DO NOT ADD ANYTHING ABOVE THIS POINT!
# This needs to be the first commands executed!
/bin/tar -zxvf /opt/sonus/staging/sbc*.tar.gz sonusCommonUtils.sh sonusCommands.sh sonusCommands.py sonusCommands.expect sonusCommands.pm sonusCommonFiles.sh sonusCommonFiles.py sonusCommonFiles.pm sonusCommonFiles.expect > /dev/null
if [ -e /opt/sonus/staging/sonusCommands.sh ] && [ -e /opt/sonus/staging/sonusCommonFiles.sh ]
then
    source /opt/sonus/staging/sonusCommands.sh
    source /opt/sonus/staging/sonusCommonFiles.sh
else
    /bin/echo "sonusCommands.sh and/or sonusCommonFiles.sh not found. Exiting..."
fi

# Globals
E_BADARGS=65
gServiceName=sbx
sonusRoot=$SONUS_DIR
sonusStaging=$SONUS_STAGING_DIR
evLogDir=evlog
preInstall=$PRE_INSTALL_CHECK_SH
# this is the file name that SM uses when it calls the script 
logFile=$SBX_UPDATE_OUT

# Flag to check whether to reset log file
resetLogFile="true"

# Flag to check whether to skip preInstall checks
skipPreInstallChecks="false"

fileName=`$BASENAME $0`

# Revert feature related Markers
$MKDIR -p $HOME_COMMON

# PM marker file
pmMarkerFile="$SBX_INSTALL_UPGRADE_TXT"

dbUpgradeMarker=$DB_UPGRADE_MARKER

pre50MarkerFile="$PRE50_MARKER_TXT"

DoDUpgradeMarker="/home/admin/DoDUpgradeMarker"

sbxImageUpgrader="$SBX_IMAGE_BASED_UPGRADE_SH";
sbxHwImageUpgrader="$SBX_HW_IMAGE_BASED_UPGRADE_SH";

p2gSbxInstallUpgradeCommand="$fileName $@ -q"
p2gSbxInstallUpgradeCommand="$sonusStaging/$p2gSbxInstallUpgradeCommand"

# pickup some common utilities. 
if [ -e $STAGING_SONUS_UTILS_SH ]; then
   . $STAGING_SONUS_UTILS_SH
else
   $ECHO "ERROR: Missing utilities file: `$BASENAME $STAGING_SONUS_UTILS_SH`; Exiting..."
   exit 1
fi

declare -a targetFiles=("revertUtil.sh" "switch_boot.sh" "partition_tool.sh" "configureGrub.sh" "configureGrubPwd.sh")

for i in "${targetFiles[@]}"
do
   if [ -e $sonusStaging/$i ]; then 
      $CP -f $sonusStaging/$i /usr/local/bin/
      $CHMOD 0775 /usr/local/bin/$i
   else
      $ECHO "ERROR: Missing utilities file: $i ; Exiting..."
      exit 1
   fi
done

if [ -e $STAGING_SBX_REVERT_PL ]; then
   if [ -d $SONUS_INSTALLUPGRADE_DIR ]; then
      $CP -fp $STAGING_SBX_REVERT_PL $SONUS_INSTALLUPGRADE_DIR/
      $CHMOD 0775 $SBX_REVERT_PL
   else
      $CP -fp $STAGING_SBX_REVERT_PL $sonusRoot/
      $CHMOD 0775 $OLD_SBX_REVERT_PL
   fi
   if [ -d $SONUS_REVERT_DIR ];then
      $RM -fr $SONUS_REVERT_DIR/* 
   else
      $MKDIR -p $SONUS_REVERT_DIR
      $CHMOD 0775 $SONUS_REVERT_DIR 
   fi
   $LOGGER -t $0 "Copying sonusCommands to revert dir(only to be used by sbxRevert.pl after revert)"
   $CP $SONUS_STAGING_DIR/sonusCommands.pm $SONUS_STAGING_DIR/sonusCommonFiles.pm $SONUS_STAGING_DIR/sbxCommonFunctions.pm $SONUS_REVERT_DIR/
else
    $ECHO "ERROR: Missing utilities file: `$BASENAME $STAGING_SBX_REVERT_PL`; Exiting..."
    exit 1
fi
  
hostType=$(getHwType)

if [ -f $SBXCONF_FILE ]; then
   sonusConfigFile=$SBXCONF_FILE
else
   sonusConfigFile="$SBX_CONF"
fi

# Check if update is possible or not
checkUpdatePossibleInCurrentPeerState()
{
  # For the update to be possible:
  # One of the following cases should be met
  #   1. Application is not running on peer
  #   2. Application is running on the peer and is of same version as of the version
  #      being installed on this server

  # First time install after ISO, can go ahead and install/update
  # because there is no reliable way to get to peer server without any 
  # info/connectivity etc
  okToProceed=0
  peerName="none"

  # If present use the existing conf file...
  if [ -f $sonusConfigFile ]; then
     peerName=`$GREP peerCeName $sonusConfigFile | $AWK  '{print $1}'| $AWK -F "="  '{print $2}'`
  fi

  if [ "$peerName" != "none" ]; then
     swinfoOut=`$SWINFO_SH`
     $ECHO "$swinfoOut" | $GREP -iq "login failed"
     ableToLoginToPeer=$?

     $ECHO "$swinfoOut" | $GREP -iE "Current.*host.*role.*:"|$TAIL -1|$GREP -iEq "active|standby"
     remoteNotActiveOrStandby=$?

     if [ $ableToLoginToPeer -eq 0 ]; then # can't login to peer, go ahead with install/upgrade
        okToProceed=1
     elif [ $remoteNotActiveOrStandby -ne 0 ]; then # not active or standby
        okToProceed=1
     fi
  else 
     okToProceed=1
  fi

  if [ $okToProceed -ne 1 ]; then
    # Seems like the peer is up and running, check if it is running the same version as
    # we are going to install
    #
    # cat /opt/sonus/installUpgrade/.buildinfo
    # Build View: gnimmagadda.sbx40
    # Build Time: Fri May 17 08:38:34 IST 2013  
    # Build Host: HP-BLADE1-VM2.in.sonusnet.com
    # Required BMC  Version: v2.4.1
    # Required BIOS Version: v2.1.2
    # Required OS   Version: 02.00.02-A010

    #
    # Compare the buildinfo of peer and from package that is about to be installed, 
    # if they are same, can go ahead with installation while peer is still running.
    # The compare is only concerned with the Build Time and Build Host. 
    #
    # While upgrading from 2.0 to 4.0 we might not have sonusPeerCnx.sh file on the peer CE.
    # So if its not available then treat it as a valid case to proceed with upgrade. 
    
    remoteBuildInfoCmd="$SONUS_PEER_CNX_EXPECT $CAT $BUILDINFO"
    remoteBuildInfo=$($SONUS_PEER_CNX_EXPECT $LS $BUILDINFO)
    if [[ $remoteBuildInfo == *"No such file or directory"* ]];then
        remoteBuildInfoCmd="$SONUS_PEER_CNX_EXPECT $CAT $sonusRoot/.buildinfo"
    fi
    $remoteBuildInfoCmd 
    if [ $? -eq 0 ]; then
      `$remoteBuildInfoCmd | $GREP -A 1 "Build Time: "|$TAIL -2 |$SED 's/\r$//' >$SONUS_TMP_DIR/.remoteBuildInfo`
      $SED -i  's/^M$//g'  $SONUS_TMP_DIR/.remoteBuildInfo

      $CAT $STAGING_BUILDINFO | $GREP -A 1 "Build Time: "|$TAIL -2 > $SONUS_TMP_DIR/.localBuildInfo
      $SED -i 's///g' $SONUS_TMP_DIR/.localBuildInfo

      if $CMP -s $SONUS_TMP_DIR/.localBuildInfo $SONUS_TMP_DIR/.remoteBuildInfo ; then
        okToProceed=1
      fi
    else
      okToProceed=1 
    fi
  fi

  if [ $okToProceed -ne 1 ]; then
    logMsg $logFile "ERROR: Can't upgrade the application while a different version of software is running on peer server, Exiting..."
    remoteBuildInfo=`$CAT $SONUS_TMP_DIR/.remoteBuildInfo`
    logMsg $logFile "Peer Build Info: \n$remoteBuildInfo"
    logMsg $logFile "Please stop application on peer server(Warning: this will be service impacting) and retry the upgrade."
    exit 1
  fi
  $RM -f $SONUS_TMP_DIR/.remoteBuildInfo $SONUS_TMP_DIR/.localBuildInfo
}


usage()
{
    # don't use logMsg here since no reason to have a timestamp on the usage
$ECHO "usage: `$BASENAME $0` [-hdo] -f <application DEB file> -D <database RPM file> 

OPTIONS:
   -h Print this message.
   -d Do not restore databases as part of update.
   -f <application DEB file>.
   -D <database RPM file>.
   -o skip OS Upgrade check." | $TEE -a $logFile
}

# Validate the execution of this script
validateExecution $PPID "upgrade" || exit 1

# Set program...
program=`$BASENAME $0`

# Do not allow runnning from the evlog directory since the terminal
# will get ripped out from under us, stopping the script prematurely.
# also look for /var/log/sonus/evlog or /var/log/sonus/mirror.
if [[ "$hostSubType" != "virtualCloud" ]]; then
  baseDir=`$BASENAME $PWD`
  mountDir=`$ECHO $baseDir | $AWK -F/ '{print $5}'`
  if [ "$baseDir" = "$evLogDir" -o "$mountDir" = "evlog" -o "$mountDir" = "mirror" ]; then
      logMsg $logFile "ERROR: `$BASENAME $0` not allowed from a drbd mounted directory.  Please leave the directory and try again."
      exit 1
  fi
fi

# handle upgrade from versions prior to systemd integration and after the integration
if [[ ! -e /etc/init.d/$gServiceName && ! -e $SONUS_ETC_INITD_DIR/$gServiceName ]]; then
   logMsg $logFile "ERROR: $gServiceName service not installed;  Install required instead of upgrade."
   exit 1
fi

while getopts "hf:dD:soq" OPTION
do
   case $OPTION in
   h)
      usage
      exit 1
      ;;
   d)
      ;;
   D)
      ;;
   f)
      ;;
   # -s option is deprecated but keeping it for backward compatibility
   # until all the customer boxes get updated to this release.
   s)
      # No action needed
      ;;   
   o)
      # No action needed
      # -o option is deprecated but keeping it for backward compatibility
      #  for upgrades from releases prior to 6.0
      ;;
   q)
      $ECHO "Upgrade in progress... " 2>/dev/null > $pmMarkerFile
      $ECHO "Upgrade from pre-5.0..." 2>/dev/null > $pre50MarkerFile
      skipPreInstallChecks="true"
      resetLogFile="false"
      ;;
   ?)
      usage
      exit $E_BADARGS
      ;;
   esac
done

shift $(($OPTIND - 1))

if [ $resetLogFile == "true" ]
then
   # reset the log file
   $CAT /dev/null > $logFile
fi

# guarantee we are copied to/running from the correct location
# note: checkLocation exits on error...
checkLocation $0 $sonusStaging $logFile

# Check if hwSubType is virtualCloud. It has image based upgrade mechanism. 
if [[ "$hostSubType" == "virtualCloud" ]]; then
    if [ -f $sbxImageUpgrader ];then
        logMsg $logFile "Calling $sbxImageUpgrader ..."
        $CHMOD +x $sbxImageUpgrader
        $sbxImageUpgrader -L $logFile
        exit 0;
    else
        logMsg $logFile "ERROR: $sbxImageUpgrader doesnt exist. hostSubType=$hostSubType."
        exit 1;
    fi
fi

logStartUpdate $logFile

checkDiags $logFile

if [ $skipPreInstallChecks != "true" ]; then
   # Check if update is possible
   checkUpdatePossibleInCurrentPeerState

   # Add a dummy log for PM to detect and proceed to the next step on PM screen
   logMsg $logFile "Installing $SONUS_STAGING_DIR/sonusdb- (INFO)"
   # Check if qcow2 image is present in external directory
   if [ ! -e $SONUS_EXTERNAL_DIR/*.qcow2 ]; then
      logMsg $logFile "ERROR: qcow2 image is missing. Please copy the image to external directory. Exiting..."
      exit 1
   fi

   # verify that it is ok to continue with the upgrade
   logMsg $logFile "Performing pre-install checks..."
   # append '-c' and '-l' options to preInstall script to skip creation of LSWU status file and to run localOnly checks respectively.
   preInstallErr=`$preInstall -c -l 2>> ${sonusStaging}/preUpgradeCheck.log`
   if [ $? -ne 0 ]; then
      logMsg $logFile "$preInstallErr"
      logMsg $logFile "ERROR : pre-install checks failed....Exiting !!!!"
      exit 1
   fi

   # Remove performModelUpdate/dynamicHANewComps files if any left out from previous LSWU.
   # If present, will cause issues with bring up after offline upgrade since dynamic HA
   # model change is not applicable for offline upgrades.
   if [ -e $PERFORM_MODEL_UPDATE ]; then
      logMsg $logFile "Removing performModelUpdate file left out from previous upgrade."
      $RM -f $PERFORM_MODEL_UPDATE
   fi

   if [ -e $DYNAMIC_HA_NEW_COMPS ]; then
      logMsg $logFile "Removing dynamicHANewComps file left out from previous upgrade."
      $RM -f $DYNAMIC_HA_NEW_COMPS
   fi

   # store the original sbx version
   origMajorVersion=$(getSWVersion local)
   logMsg $logFile "origMajorVersion is $origMajorVersion"
   if [[ $origMajorVersion -lt 5 ]]; then
     # Create pre50Marker.txt marker file
     logMsg $logFile "Creating pre50Marker.txt marker file..."
     $ECHO "Upgrade from pre-5.0..." 2>/dev/null > $pre50MarkerFile

     # Install required qemu-utils packages for upgrades from 4.x
     logMsg $logFile "Installing required qemu-utils packages..."
     $DPKG -i $sonusStaging/libiscsi*.deb
     $DPKG -i $sonusStaging/qemu-utils*.deb
   fi

   # Disable fstrim for Tuna.
   # By default fstrim is enabled
   # When fstrim.timer is enabled symlink is created.
   if [[ "$hostType" == "SBC7000" || "$hostType" == "SBC5400" ]]; then
     logMsg $logFile "Disabling fstrim before upgrade..."
     linkFile="/etc/systemd/system/timers.target.wants/fstrim.timer"
     if [ -h "$linkFile" ]; then  
       logMsg $logFile "Fstrim is enabled.  Disabling..."
       $SYSTEMCTL stop fstrim.timer
       $SYSTEMCTL disable fstrim.timer
       if [ ! -h "$linkFile" ]; then
         logMsg $logFile "Fstrim is disabled."
       fi
     fi
   fi 

   # Check for DoD mode
   $HOST_JITC_MODE_SH status 2>/dev/null | $GREP -q "DoD mode enabled"
   if [ $? -eq 0 ];then
      logMsg $logFile "DoD mode Enabled. Setting DoD upgrade flag.."
      $ECHO "Upgrading in DoD Mode.." > $DoDUpgradeMarker
      pmAccess=$($HOST_JITC_MODE_SH pmAccessStatus)
      cliAccess=$($HOST_JITC_MODE_SH cliAccessStatus)
      $ECHO "PM=$pmAccess" >> $DoDUpgradeMarker
      $ECHO "CLI=$cliAccess" >> $DoDUpgradeMarker
   fi

   logMsg $logFile "Saving current configuration and databases..."
   $SAVE_CDB_SH

   # Save checksum file while upgrading from old oracle version to new
   # this is required to support revert in fips mode
   fipsModeEnabled="$($GREP "fipsMode=enabled" $SBXCONF_FILE)"
   if [ -n "$fipsModeEnabled" ]; then
       oracleDbCheckSumFile="/home/oracle/oracleDB.sha256sums"
       savedOracleDbCheckSumFile="/home/admin/oracle/oracleDB.sha256sums"
       if [ -f $oracleDbCheckSumFile ] && [ ! -f $savedOracleDbCheckSumFile ]; then
           logMsg $logFile "Copying oracleDB checksum file to $savedOracleDbCheckSumFile to support revert in fips mode"
           $MKDIR -p /home/admin/oracle/
           $CP $oracleDbCheckSumFile $savedOracleDbCheckSumFile
       fi
   fi

   if [ -e /opt/sonus/staging/exportDB ]; then
       # Add oracle/postgres users to sonus group, as they refer to files belonging to sonus group.
       $_GROUPS oracle | $GREP -q sonus
       if [ $? -ne 0 ]; then
         logMsg $logFile "Adding oracle to sonus group"
         $USERMOD -a -G sonus oracle
       fi
       $GETENT passwd postgres 2&> /dev/null
       if [ $? -eq 0 ]; then
         $_GROUPS postgres | $GREP -q sonus
         if [ $? -ne 0 ]; then
           logMsg $logFile "Adding postgres to sonus group"
           $USERMOD -a -G sonus postgres
         fi
       fi
       $CHMOD 775 /opt/sonus/staging/exportDB
       SBC_VERSION="`/opt/sonus/sbx/scripts/swinfo.sh|$AWK -F: '/SBC:/{print $2}'| $HEAD -1| $TR -d ' '`"
       curTime="`date +%y%m%d%H%M%S`"
       ssdbBackupFile="/home/admin/sonusdb_${SBC_VERSION}_${curTime}_for_upgrade.dmp"
       ssdbPgBackupFile="/home/admin/pg_sonusdb_${SBC_VERSION}_${curTime}_for_upgrade.dmp"
       $RM -f /home/admin/pg_sonusdb_*_for_upgrade.dmp
       /opt/sonus/staging/exportDB -file $ssdbBackupFile
       if [ $? != 0 ] || [ ! -e $ssdbPgBackupFile ]; then
        logMsg $logFile "Operation Failed! /opt/sonus/staging/exportDB -file $ssdbBackupFile"
        exit 1
       else
	      logMsg $logFile "Database backup files is $ssdbBackupFile"
       fi
   else
       logMsg $logFile "Cannot find /opt/sonus/staging/exportDB..."
       exit 1
   fi

   # Stop sbx service...
   stopService $logFile

   #save all the permissions in a file.
   logMsg $logFile "Started sbxPermissionCheck.sh script"
   $sonusStaging/sbxPermissionCheck.sh beforeUpgrade
   logMsg $logFile "Completed sbxPermissionCheck.sh script"

   # Remove ePSX if installed at the beginning of upgrade.
   if [ -e $VIRSH ]; then
      $VIRSH list --all | $GREP -q ePSX
      if [ $? -eq 0 ]; then
          vmDeletionLogs=$($OVFINSTALL_SH delete ePSX)
          logMsg $logFile "ePSX deletion logs:$vmDeletionLogs"
      fi
   fi
else
   # create symbolic link ETC_INITD_SBX
   logMsg $logFile "Creating sbx symbolic link..."
   $RM -f $ETC_INITD_SBX
   $LN -fs $SBX_START_SH $ETC_INITD_SBX
fi 

revertCapable
if [ $? -eq 0 ]; then
   lvcount=$($LVSCAN 2>/dev/null |$WC -l 2>/dev/null)
   if [ $lvcount == 1 ]; then
      logMsg $logFile "Calling checkDiskforRevert to scan disk to allow Upgrade..p2gSbxInstallUpgradeCommand=$p2gSbxInstallUpgradeCommand "
      #prepareDiskforRevertSupport $logFile
      checkDiskforRevert "UPGRADE" "$p2gSbxInstallUpgradeCommand" $logFile || exit 0
   elif [ $lvcount == 3 ]; then
      # Backup home directory
      logMsg $logFile "Taking /home backup..."
      $MKDIR -p $HOME_BACKUP_DIR
      backupHomeDir

      saveCommonDirPerms

      $ECHO " DB upgrade required " >  $dbUpgradeMarker

      logMsg $logFile "Calling $sbxHwImageUpgrader ..."
      $CHMOD +x $sbxHwImageUpgrader
      $sbxHwImageUpgrader -L $logFile
   else
      logMsg $logFile "Cannot proceed with install/upgrade.. Number of LVMs=$lvcount"
      exit 0
   fi
else
   $ECHO "Upgrade in progress... " 2>/dev/null > $pmMarkerFile
   logMsg $logFile "Saving openclovis and tailf logs..."
   $SAVE_OC_LOGS_SH
   $SAVE_TAILF_LOGS_SH

   if [ -f $sbxImageUpgrader ];then
      logMsg $logFile "Calling $sbxImageUpgrader ..."
      $CHMOD +x $sbxImageUpgrader
      $sbxImageUpgrader -L $logFile
   else
      logMsg $logFile "ERROR: $sbxImageUpgrader doesnt exist."
      exit 1;
   fi
fi

exit 0;
