#!/bin/bash
#############################################################
#
# Copyright (c) 2016 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# Module Description:
#    Wrapper script for sudo commands.
#    Limits functionality exposed to sudo users
#############################################################
# Sourcing Command Variables
# NOTE: need to check staging directory first for upgrades
# from 5.0.x or earlier.
if [ -e /opt/sonus/staging/sonusCommands.sh ];then
    source /opt/sonus/staging/sonusCommands.sh
elif [ -e /opt/sonus/bin/sonusCommands.sh ];then
    source /opt/sonus/bin/sonusCommands.sh
else
    /bin/echo "Could not locate sonusCommands.sh Exiting..."
    exit 1
fi

if [ -e /opt/sonus/staging/sonusCommonFiles.sh ];then
    source /opt/sonus/staging/sonusCommonFiles.sh
elif [ -e /opt/sonus/bin/sonusCommonFiles.sh ];then
    source /opt/sonus/bin/sonusCommonFiles.sh
else
    /bin/echo "Could not locate sonusCommonFiles.sh Exiting..."
    exit 1
fi

PROG=${0##*/}

svcResult=1

usage()
{
    $ECHO "usage: $PROG [-h] <service> <serviceAction>"
    $ECHO "  Manage system services"
    $ECHO "  service and available serviceAction: "
    $ECHO "      auditd              [start|stop|status]"
    $ECHO "      ntp                 [start|stop|restart]"
    $ECHO "      openhpid            [start|stop|restart]"
    $ECHO "      cron                [restart]"
    $ECHO "      ssh                 [start|stop|restart|reload]"
    $ECHO "      ssh-cli             [start|stop|reload]"
    $ECHO "      ssh-netconf         [start|stop|reload]"
    $ECHO "      networking          [start|stop]"
    $ECHO "      sbx                 [start|stop|restart|status]"
    $ECHO "      manageNetworkAccess [start|stop|enable-cli|disable-cli|enable-pm|disable-pm]"
    $ECHO "      drbd                [start|stop|status|reload|restart]"
    $ECHO "      apache2             [stop|reload|restart]"
    $ECHO "      serf                [start|stop|status]"
    $ECHO "      serfMembership      [start|stop|status]"
    $ECHO "      lca                 [stop]"
    $ECHO "      checkbonding        [status]"
    $ECHO "      cps                 [start|restart|status]"
    $ECHO "      sbxirqbalance       [start]"
    $ECHO "      lwresd              [stop]"
    $ECHO "      monit               [restart]"
    $ECHO "      postgresql          [start|stop|status|active|standby|switchover]"
    $ECHO "      dataagent           [start|stop|monitor|unmonitor]"
    $ECHO "      bfd                 [start|stop]"
    $ECHO "      trc-dtmf-anonymization [enable|disable|start|stop|restart|status]"
    $ECHO ""
    $ECHO "  -h : This usage help."
    $ECHO ""
    $ECHO " e.g.: $PROG openhpid restart"
    $ECHO " e.g.: $PROG sbx start"
    exit
}

error()
{
    $LOGGER -t $PROG "error: $1"
    exit 1
}


while getopts h o
do
    case $o in
        h) usage ;;
        \?) usage "Unrecognized switch"
    esac
done

shift $(expr $OPTIND - 1)

[[ $# -ne 2 ]] && usage

service=$1
serviceAction=$2

case $service in
    "auditd") 
        case $serviceAction in
            "status")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            "start")
                /etc/init.d/$service $serviceAction && /sbin/augenrules --load > /dev/null
                svcResult=$?
                ;;
            "stop")
                /etc/init.d/$service $serviceAction && /sbin/auditctl -e 0 > /dev/null
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "ntp") 
        case $serviceAction in
            "start"|"stop"|"restart")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "openhpid") 
        case $serviceAction in
            "start"|"stop"|"restart")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "ssh") 
        case $serviceAction in
            "start"|"stop"|"restart"|"reload")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "ssh-cli") 
        case $serviceAction in
            "start"|"stop"|"reload")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "ssh-netconf") 
        case $serviceAction in
            "start"|"stop"|"reload")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "sbx") 
        case $serviceAction in
            "start"|"stop"|"restart"|"status")
                $ETC_INITD_SBX $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "networking") 
        case $serviceAction in
            "start"|"stop")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "cron") 
        case $serviceAction in
            "restart")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "manageNetworkAccess")
        case $serviceAction in
            "start"|"stop"|"enable-cli"|"disable-cli"|"enable-pm"|"disable-pm")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "drbd") 
        case $serviceAction in
            "start"|"stop"|"status"|"reload"|"restart")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "apache2") 
        case $serviceAction in
            "stop"|"reload"|"restart")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "serf") 
        case $serviceAction in
            "start"|"stop"|"status"|"restart")
                $ETC_INITD_SERF $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "serfMembership") 
        case $serviceAction in
            "start"|"stop"|"status"|"restart")
                $ETC_INITD_SERFMEMBERSHIP $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "lca") 
        case $serviceAction in
            "stop")
                $ETC_INITD_LCA $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "checkbonding") 
        case $serviceAction in
            "status")
                $ETC_INITD_CHECKBONDING $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "cps") 
        case $serviceAction in
            "start"|"restart"|"status")
                $ETC_INITD_CPS $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "trc-dtmf-anonymization") 
        case $serviceAction in
            "enable"|"disable"|"start"|"restart"|"status"|"stop")
                /bin/systemctl $serviceAction $service
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "sbxirqbalance") 
        case $serviceAction in
            "start")
                $ETC_INITD_SBXIRQBALANCE $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "lwresd") 
        case $serviceAction in
            "stop")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "monit") 
        case $serviceAction in
            "restart")
                /etc/init.d/$service $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "postgresql") 
        case $serviceAction in
            "start"|"stop"|"status"|"active"|"standby"|"switchover")
                $START_PGDB_SH $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    "dataagent")
        case $serviceAction in
            "start"|"stop")
                /bin/systemctl $serviceAction dataagent
                svcResult=$?
                ;;
            "monitor"|"unmonitor")
                /usr/bin/monit $serviceAction dataagent
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
        
    "bfd") 
        case $serviceAction in
            "start"|"stop")
                $ETC_INITD_BFD $serviceAction
                svcResult=$?
                ;;
            *) error "Invalid service action $serviceAction for $service";;
        esac
        ;;
    *) usage ;;
esac

exit $svcResult
