#!/bin/bash
#############################################################
#
# Copyright (c) 2016 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# sonusCommonUtils.sh
#
# Module Description:
# Common Sonus bash utilities for parameter validation.
#    
##############################################################################

if [ -n "$_SONUS_COMMON_UTILS" ]
then
    return 0
fi

_SONUS_COMMON_UTILS=defined

# Set to ":" to suppress debug output
# or
# DEBUG="echo -e DEBUG:"
if [ -z $DEBUG ]
then
    DEBUG=":"
fi

if [ -e /opt/sonus/staging/sonusCommands.sh ]
then
    source /opt/sonus/staging/sonusCommands.sh
elif [ -e /opt/sonus/bin/sonusCommands.sh ]
then
    source /opt/sonus/bin/sonusCommands.sh
else
    /usr/bin/logger -s -t sonusCommonUtils.sh "Could not locate sonusCommands.sh"
    exit 1
fi

if [ -e /opt/sonus/staging/sonusCommonFiles.sh ]
then
    source /opt/sonus/staging/sonusCommonFiles.sh
elif [ -e /opt/sonus/bin/sonusCommonFiles.sh ]
then
    source /opt/sonus/bin/sonusCommonFiles.sh
else
    /usr/bin/logger -s -t sonusCommonUtils.sh "Could not locate sonusCommonFiles.sh"
    exit 1
fi

# source functions that were split out for more general usage
CMN_PROG_DIR=$(cd "$($DIRNAME "${BASH_SOURCE[0]}" )" && pwd)
source $CMN_PROG_DIR/commonCfgValidation.sh

function genDmidecodeOut()
{
  # Create SONUS_CONF_DIR as necessary
  if [ ! -d $SONUS_CONF_DIR ]; then
     $MKDIR -p $SONUS_CONF_DIR
     $CHOWN sonusadmin:sonus $SONUS_CONF_DIR/
     $CHMOD 775 $SONUS_CONF_DIR/
  fi

  # Save DMIDECODE info to a file
  # note: some dmidecode versions output a 'invalid entry length' error,
  # so redirect stderr to avoid seeing that output
  $DMIDECODE > $DMIDECODE_OUT 2> /dev/null
  $SED -i 's/\(UUID: \)\(.*\)/\1\U\2/' $DMIDECODE_OUT 2> /dev/null
  $CHOWN sonusadmin:sonus $DMIDECODE_OUT
  $CHMOD 640 $DMIDECODE_OUT
}

# Functionality related to dmidecode output has been relocated here for access
# from both the OS and app layers.  However, this file is also sourced during
# ISO installation and at that time postInstall.sh provides its own implementation
# of the functionality as it is needed prior to users/groups being added.  Since
# this is sourced prior to postInstall.sh writing dmidecode.out, and also prior
# to the sonusadmin user being created, we need to skip the dmidecode functionality
# that postInstall.sh handles on its own.  Thankfully we can easily tell if we are
# called from postInstall.sh due to the non-existence of user sonusadmin.
if $ID sonusadmin &> /dev/null; then
  # Create dmidecode output file if not present, as it is required for upgrades
  # from 6.0(where this file doesn't exist).
  if [ ! -e $DMIDECODE_OUT ];then
    genDmidecodeOut
  fi

  # note: prodString/hwType/hname get defined seperately in postInstall.sh during install
  # and therefore should also be skipped when this file is sourced from postInstall.sh
  hwType=`$GREP "Product Name" $DMIDECODE_OUT | $HEAD -1 | $AWK '{print $3}'`
  prodString=`$GREP "String 4: Type:" $DMIDECODE_OUT | $AWK -F : '{ print $3 }'`
  hname=`$GREP "Product Name" $DMIDECODE_OUT | $HEAD -1 | $AWK -F: '{print $2}'`
fi

# Global variable to get primary device and disk
# primary disk name : vda - OpenStack, hda/xvda for AWS, sda/sdc for hw
primary_disk=`$GREP -vE '.*[0-9]$' /proc/partitions|$GREP -v major|$GREP -vE ^$|$HEAD -n 1|$AWK '{print $4}' | $GREP -v db$`
primary_device=/dev/$primary_disk

# check_unsigned_integer()
# $1 is the value to check
# $2 is the number of bits in the number
# returns $?, zero if the value is an unsigned integer in the range
# used by other integer validation functions

function check_unsigned_integer()
{
  local limit="$((1<<$2))"
  [[ "$1" =~ ^[[:digit:]]+$ ]] ||
    { $ECHO "Invalid number"; return 1; }
  [[ "$1" -ge "0" && "$1" -lt "$limit" ]] ||
    { $ECHO "Value out of range"; return 1; }
  return 0
}

# check_signed_integer()
# $1 is the value to check
# $2 is the number of bits in the number
# returns $?, zero if the value is a signed integer in the range
# used by other integer validation functions

function check_signed_integer()
{
  local limit="$((1<<$2))"
  [[ "$1" =~ ^-?[[:digit:]]+$ ]] ||
    { $ECHO "Invalid number"; return 1; }
  [[ "$1" -ge "$((-$limit))" && "$1" -lt "$limit" ]] ||
    { $ECHO "Value out of range"; return 1; }
  return 0
}

# validate_uint8()
# returns $?, zero if the entire string is decimal digits in the range

function validate_uint8 () {
  $DEBUG "validate_uint8($1,$2)"
  check_unsigned_integer "$2" 8
}

# validate_uint16()
# returns $?, zero if the entire string is decimal digits in the range

function validate_uint16 () {
  $DEBUG "validate_uint16($1,$2)"
  check_unsigned_integer "$2" 16
}

# validate_uint32()
# returns $?, zero if the entire string is decimal digits in the range

function validate_uint32 () {
  $DEBUG "validate_uint32($1,$2)"
  check_unsigned_integer "$2" 32
}

# validate_uint64()
# returns $?, zero if the entire string is decimal digits

function validate_uint64 () {
  $DEBUG "validate_uint64($1,$2)"
  [[ "$2" =~ ^[[:digit:]]+$ ]] ||
    { $ECHO "Invalid number"; return 1; }
  return 0
}

# validate_sint8()
# returns $?, zero if the entire string is decimal digits in the range

function validate_sint8 () {
  $DEBUG "validate_sint8($1,$2)"
  check_signed_integer "$2" 8
}

# validate_sint16()
# returns $?, zero if the entire string is decimal digits in the range

function validate_sint16 () {
  $DEBUG "validate_sint16($1,$2)"
  check_signed_integer "$2" 16
}

# validate_sint32()
# returns $?, zero if the entire string is decimal digits in the range

function validate_sint32 () {
  $DEBUG "validate_sint32($1,$2)"
  check_signed_integer "$2" 32
}

# validate_sint64()
# returns $?, zero if the entire string is decimal digits

function validate_sint64 () {
  $DEBUG "validate_sint64($1,$2)"
  [[ -n "$2" && "$2" =~ ^-?[[:digit:]]+$ ]] ||
    { $ECHO "Invalid number"; return 1; }
  return 0
}

# validate_boolean()
# returns $?, zero if the value is a valid boolean value
# converts the answer to "true" or "false" and returns in $REPLY

function validate_boolean () {
  $DEBUG "validate_boolean($1,$2)"
  local ans="${2,,}"
  case "$ans" in
  true|yes|y|1|good|ok)
    REPLY="true"
    ;;
  false|no|n|0|bad)
    REPLY="false"
    ;;
  *)
    $ECHO "Not a boolean value"
    return 1
  esac
  return 0
}

# validate_netprefix()
# returns $?, zero if the string is a decimal number [0..128]
# corresponding to the number of bits in the IP network prefix.
# Blank values are validated in cross check

function validate_netprefix () {
  $DEBUG "validate_netprefix($1,$2)"
  if [ -z "$2" ]; then
    return 0
  fi
  [[ -n "$2" && "$2" =~ ^[[:digit:]]+$ ]] ||
    { $ECHO "Invalid number"; return 1; }
  if [ "$2" -lt 0 -o "$2" -gt 128 ]; then
    $ECHO "Network prefix out of range IPv4:[0..32], IPv6:[0..128]"
    return 1
  fi
  return 0
}

# validate_netprefix_optional()
# must be a valid netprefix or empty

function validate_netprefix_optional () {
  [[ -z "$2" ]] || validate_netprefix "$1" "$2"
}

# validate_ipv4addr()
# Must be a 4-tuple of 1-3 digits each, separated by '.'
# Each number must be in the range [0..255]
# returns $?, zero if the value is a legal IPv4 address
# Blank values are validated in cross check

function validate_ipv4addr () {
  $DEBUG "validate_ipv4addr($1,$2)"
  local addr="$2"
  local N
  if [ -z "$addr" ]; then
    return 0
  fi
  if [[ "$addr" =~ ^([[:digit:]]{1,3}\.){3}[[:digit:]]{1,3}$ ]]; then
    for N in $($ECHO "$addr" | $SED 's/\./ /g'); do
      if [ $N -gt 255 ]; then
        $ECHO "Invalid value for IPv4 address"
        return 1
      fi
    done
  else
    $ECHO "Invalid format for IPv4 address"
    return 2
  fi
}

# validate_ipv6addr()
# Must be a valid IPv6 address but not the loopback or a multicast address
# as reported by the "validate" program. Note that the program returns
# non-zero on successful validation.
# Blank values are validated in cross check

function validate_ipv6addr () {
  $DEBUG "validate_ipv6addr($1,$2)"
  if [ -z "$2" ]; then
    return 0
  fi
  if /opt/sonus/bin/validateV6addr "$2"; then
    $ECHO "Invalid value for IPv6 address"
    return 1
  else
    return 0
  fi
}

# validate_ipaddr()
# Must be a valid IPv4 or IPv6 address
# Blank values are validated in cross check

function validate_ipaddr () {
  if [ -z "$2" ]; then
    return 0
  fi
  if validate_ipv4addr "$1" "$2" >/dev/null || validate_ipv6addr "$1" "$2" >/dev/null; then
    return 0
  else
    $ECHO "Invalid value for IP address"
    return 1
  fi
}

# validate_ipaddr_optional()
# Must be a valid IPv4 or IPv6 address, or empty

function validate_ipaddr_optional () {
  [[ -z "$2" ]] || validate_ipaddr "$1" "$2"
}

# validate_vlan()
# Must be an unsigned integer in the range [0..4095]

function validate_vlan () {
  $DEBUG "validate_vlan($1,$2)"
  check_unsigned_integer "$2" 12
}

# validate_timezone()
# Must be present in /usr/share/zoneinfo/$timezone

function validate_timezone() {
  $DEBUG "validate_timezone($1,$2)"
  [[ -f "/usr/share/zoneinfo/$2" ]] && return 0
  $ECHO "Invalid timezone $2"
  return 1
}

# Test for IPv6 being a specified address
function specified_ipv6()
{
	local ip=$1
	local retVal=1

	if [[ $ip =~ [1-9,a-f,A-F] ]]; then
		retVal=0
	fi
	return $retVal
}

# Reads only config parameter assignments
# Filtering out other shell commands
function readConfigFile()
{
    local configFile="$1"
    local secureTmpFile=/tmp/.sonusTmpConfig$$
    if [ -r $configFile ]
    then
        $RM -f $secureTmpFile
        $EGREP '^#|^[^ ]*=[^;&]*'  "$configFile" > $secureTmpFile
        . $secureTmpFile
        $RM -f $secureTmpFile
    fi
}

# Common function for telling if the system is shutting down
function isSystemShuttingDown()
{
    # note: shutdown will be active for reboot and poweroff. need to check
    # those to see which is happening.
    local count=`$SYSTEMCTL list-jobs | $EGREP -c 'shutdown.target.*start'`
    $ECHO $count
}

# Common function for telling if the system is rebooting
function isSystemRebooting()
{
    local count=`$SYSTEMCTL list-jobs | $EGREP -c 'reboot.target.*start'`
    $ECHO $count
}

# Common function for telling if the system is halting
function isSystemHalting()
{
    local count=`$SYSTEMCTL list-jobs | $EGREP -c 'halt.target.*start'`
    $ECHO $count
}

# Common function for telling if the system is powering off
function isSystemPoweringOff()
{
    local count=`$SYSTEMCTL list-jobs | $EGREP -c 'poweroff.target.*start'`
    $ECHO $count
}

# ifconfig output format changed between Debian 8 and Debian 9.  In addition
# net-tools is being deprecated in favor of iproute2 tools. Provide
# a subroutine to determine the hw address of an interface to keep
# the implementation in one place to make it easier to switch.
function getHwAddr()
{
  local iface=$1
  local mac=`$_IP link show $iface | $GREP ether | $AWK '{print $2}' | $AWK {'print tolower($_)'}`
  $ECHO $mac
}

function getIpAddr()
{
  local iface=$1
  local addr

  if [ -n "$iface" ]; then
      addr=`$_IP addr show $iface 2> /dev/null | $GREP "inet " | $AWK '{print $2}' | $AWK -F/ '{print $1}'`
  else
      addr=`$_IP addr | $GREP "inet " | $GREP eth | $AWK '{print $2}' | $AWK -F/ '{print $1}'`
  fi
  $ECHO $addr
}

function checkIface()
{
  local iface=$1
  $_IP addr show $iface &> /dev/null
  $ECHO $?
}

function getNetPrefix()
{
  local iface=$1
  local prefix

  if [ -n "$iface" ]; then
      prefix=`$_IP addr show $iface | $GREP "brd " | $GREP "inet " | $AWK '{print $2}' | $CUT -d'/' -f2`
  else
      prefix=`$_IP addr | $GREP "brd " | $GREP "inet " | $AWK '{print $2}' | $CUT -d'/' -f2`
  fi
  $ECHO $prefix
}

function getMask()
{
  prefix=$1
  mask="0"
  while [ $prefix -gt 0 ]
  do
    shiftValue=$((8-prefix))
    prefix=$[$prefix-1]
    tmpMask=$((1<<shiftValue))
    mask=$((mask |=tmpMask))	
  done
  eval ${2}=$mask
}

function convNetprefixToNetmask()
{
  prefix=$1
  oct1="0"
  oct2="0"
  oct3="0"
  oct4="0"
  if [ $prefix -ge 8 ];then
    oct1="255"
    if [ $prefix -ge 16 ];then
      oct2="255"
      if [ $prefix -ge 24 ];then
        oct3="255"
        if [ $prefix -ge 32 ];then
          oct4="255"
        else
          tmpPrefix=$[prefix-24]
          getMask $tmpPrefix oct4
        fi
      else 
        tmpPrefix=$[prefix-16]
        getMask $tmpPrefix oct3
      fi
    else
      tmpPrefix=$[prefix-8]
      getMask $tmpPrefix oct2
    fi
  else
    tmpPrefix=$[prefix]
    getMask $tmpPrefix oct1
  fi
  netMask="$oct1.$oct2.$oct3.$oct4"

  eval ${2}=$netMask
}

function setIfaceAddress()
{
  local device=$1
  local address=$2
  local mask=$3
  local broadcast=${4-""}
  local log=${5-/dev/null}

  local prefix=$(convNetmaskToNetprefix $mask 1)

  if [ -z "$broadcast" ]; then
      $_IP addr add $address/$prefix dev $device &>> $log
  else
      $_IP addr add $address/$prefix broadcast $broadcast dev $device &>> $log
  fi
}

function clearIfaceAddress()
{
  local device=$1
  local log=${2-/dev/null}

  $_IP addr flush dev $device &>> $log
}

function bringUpIface()
{
  local device=$1
  local log=${2-/dev/null}

  $_IP link set $device up &>> $log
}

function bringDownIface()
{
  local device=$1
  local log=${2-/dev/null}

  $_IP link set $device down &>> $log
}

function getAllSerfParams()
{
  local json=`$PYTHON3 $SERF_GET_SELF_PARAMS_PY allParams`
  local arr=("$(jq -r '. | to_entries | .[] | "[\"" + .key + "\"]=" + (.value | @sh)' <<< "$json")")
  local cmd="${1}=($arr)"
  eval "$cmd"
}

# There is a bug with lvm2 that causes update-grub to complain about non-existent drives
# info can be found here: https://bugs.launchpad.net/ubuntu/+source/lvm2/+bug/1834250
# This results in seeing errors like the following when update-grub is run:
# /dev/sdb: open failed: No medium found
# /dev/sdc: open failed: No medium found
# To work around this, we can set a global filter in the lvm configuration file
function updateLvmConf()
{
  mode=$1

  # note: indentation on added lines matches lvm.conf, don't change it!
  if [[ $mode == "hw" ]]; then
    $SED -i.orig -e '/# global_filter = /a \
        #\
        # RIBBON MODIFICATION\
        # Ribbon workaround for lvm2 bug reporting "No medium found" on non-existent drives\
        global_filter = [ "r|/dev/sdb|", "r|/dev/sdc|" ]\
        # RIBBON MODIFICATION\
        #' /etc/lvm/lvm.conf
  fi

  # TBD: for virtual we may need to check if have a second drive, it could be vdb, ...
}

# Initialize hardware type
getHwType()
{
   hostType=$hwType

   if [[ $prodString == "5100" || $prodString == "5110" ]]; then
      hostType="ConnexIP5100"
   else 
      if [ -e $HOST_TYPE ]; then 
         hostType=`$CAT $HOST_TYPE`
      fi
   fi

   # The path shall match with from where SmaLoad method reads
   # note: when doing an ISO install (isoSbx) the path might not exist yet
   $ECHO $hostType
}

getVirtualType()
{
   virtType=Unknown
   local hostType=`$ECHO $hname | $AWK '{print tolower($0)}'`

   if [[ "$hostType" =~ ^hvm  || "$hostType" =~ ^c5 || "$hostType" =~ ^m5 || "$hostType" =~ ^c6 ]]; then
       virtType="AWS"
   elif [[ "$hostType" =~ "openstack" ]]; then
       virtType="OpenStack"
   elif [[ "$hostType" =~ "google" ]]; then
       virtType="GCE"
   elif [[ "$hostType" =~ "virtual" ]]; then
       azCheck=`$GREP -m 1 "Manufacturer" $DMIDECODE_OUT | $AWK -F: '{print $2}'`
       if [[ "$azCheck" =~ "Microsoft" ]]; then
           virtType="AZ"
       fi
   fi

   $ECHO $virtType
}

function get_log_dev()
{
    # Check if the current instance is launched in UEFI mode, in which case, disk label is gpt
    isUEFIMode=`$FDISK -l | $GREP "Disklabel type: gpt" | $WC -l`
    if [[ $isUEFIMode -eq 1 ]]; then
        $ECHO "/dev/sda3"
        return
    fi

    if [[ $($LS /dev/[sv]da6 2> /dev/null) ]]; then
        # drbd /dev/[sv]da6 is never an lvm partition
        $PVS | $GREP -q /dev/[sv]da6 2> /dev/null 
        if [[ $? -ne 0 ]]; then
            log_dev=$($LS /dev/[sv]da6)
            $ECHO $log_dev
            return
        fi
    fi

    if [[ $hostType == "ConnexIP5000" ]]; then
        local hwType=`$GREP "Product Name" $DMIDECODE_OUT | $HEAD -1 | $AWK '{print $3}'`
        if [[ "$hwType" == "KVM" ]]; then
            local mirrorDriveTag=RIBBON_MIRROR_DRIVE
            mirrorDrive=`$READLINK -e /dev/disk/by-id/*$mirrorDriveTag`
            if [[ "$mirrorDrive" != "" ]]; then 
                log_dev=$mirrorDrive
            fi
        # Hardcoding the disk sdb or vdb as secondary disk for VMWare/RHEV setup.
        # In VMWare/RHEV setup, unable to identify the secondary disk by label/id inside the instance.
        elif [[ "$hwType" == "VMware" || "$hwType" == "RHEV" ]]; then
            if [[ $($LS /dev/[sv]db) ]]; then
                log_dev=$($LS /dev/[sv]db)
            fi
        # For AWS M5, C5 and C6 instances we have nvme disks
        elif [[ "$hwType" =~ ^m5 || "$hwType" =~ ^c5 || "$hwType" =~ ^c6 ]]; then
            local tmp=$($LSBLK -l | $GREP "boot" | $AWK '{ print $1}')
            local rootDisk=${tmp::-2}
            local baseDev=$($LSBLK | $GREP "^nvme" | $GREP -v "$rootDisk" | $AWK '{print $1}')
            if [ ! -z "$baseDev" ]; then
                log_dev=/dev/$baseDev
            fi
        # For AWS HVM instances we have xvd* disks
        elif [[ "$hwType" == "HVM" ]]; then
            local tmp=$($LSBLK -l | $GREP "boot" | $AWK '{ print $1}')
            local rootDisk=${tmp::-1}
            local baseDev=$($LSBLK | $GREP "^xvd" | $GREP -v "$rootDisk" | $AWK '{print $1}')
            if [ ! -z "$baseDev" ]; then
                log_dev=/dev/$baseDev
            fi
        fi
    fi
    $ECHO $log_dev
}

# Update /etc/security/limits.conf file		 
# $1 - String to search for the line		 
# $2 - Line to update
function updateLimitsDotConfFile()
{
   local ulimitFile="/etc/security/limits.conf"
   local searchString="$1"
   local value="$2"
   $GREP "^$searchString" $ulimitFile &> /dev/null
   if [ $? -eq 0 ]; then
      $SED -i "/$searchString/s/$searchString/$value/" $ulimitFile
   else
      $ECHO "$value" >> $ulimitFile
   fi
}

# $1 string to search/update, $2 value to set, $3 comment (could be blank)
function updateSysctlConf()
{
   local string="$1"
   local value="$2"
   local comment="$3"
   $GREP -v $string $ETC_SYSCTL_CONF > /tmp/sysctl.conf
   if [ "$comment" != "" ]; then
      if ! $GREP -q $string "$ETC_SYSCTL_CONF" ; then
         $ECHO -e "# $comment" >> /tmp/sysctl.conf
      fi
   fi
   $ECHO -e "$string=$value" >> /tmp/sysctl.conf
   $MV -f /tmp/sysctl.conf $ETC_SYSCTL_CONF

   $SYSCTL -e -p $ETC_SYSCTL_CONF > /dev/null 2>&1
}

# in a chroot, Only enable/disable systemctl options work.  All others are
# NO-OPS.  To mask/unmask which we want to do, we need to add or remove
# the /dev/null link manually.
chrootSystemctl()
{
  local baseDir=$1
  local action=$2
  local service=$3
  local logfile=$4

  # service files are located in /lib/systemd/system
  local libServiceFile="$baseDir/lib/systemd/system/$service"
  # when masking a service the link is put in /etc/systemd/system
  local etcServiceFile="$baseDir/etc/systemd/system/$service"

  # simulate the following:
  # 1) $SYSTEMCTL mask $service: create link to /dev/null
  # 2) $SYSTEMCTL unmask $service: remove link to /dev/null
  # 3) $SYSTEMCTL enable $service: create links to wanted targets
  # 4) $SYSTEMCTL disable $service: remove links from wanted targets

  # for enable and disable, we need to determine the target(s) that is specified in
  # the service file. if the service file exists under /lib/systemd/system in the
  # chroot path, grep the value from there.  if it doesn't exist there, then it is
  # probably a generated file and isn't a service. in that case systemctl shouldn't
  # be getting used.
  local targets=""
  local targetFile=""
  if [[ "$action" == "enable" || "$action" == "disable" ]] && [[ -f "$libServiceFile" ]]; then
      targets=$($GREP WantedBy $libServiceFile | $AWK -F= '{print $2}')
  fi

  # perform the necessary action
  if [ "$action" == "mask" ]; then
      $LN -sf /dev/null $etcServiceFile
      if [ -n "$logfile" ]; then
         $ECHO "Created symlink $etcServiceFile -> /dev/null." | $TEE -a $logfile
      fi
  elif [ "$action" == "unmask" ]; then
      if [[ -L $etcServiceFile && $(readlink -f $etcServiceFile) == "/dev/null" ]]; then
        $RM -f $etcServiceFile
        if [ -n "$logfile" ]; then
           $ECHO "Removed $etcServiceFile." | $TEE -a $logfile
        fi
      elif [ -n "$logfile" ]; then
        $ECHO "$service is not currently masked" | $TEE -a $logfile
      fi
  elif [[ "$action" == "enable" && -n "$targets" ]]; then
      for tgt in $targets; do
          targetFile="$baseDir/etc/systemd/system/${tgt}.wants/$service"
          $LN -sf $libServiceFile $targetFile
          if [ -n "$logfile" ]; then
             $ECHO "Created symlink $targetFile -> $libServiceFile." | $TEE -a $logfile
          fi
      done
  elif [[ "$action" == "disable" && -n "$targets" ]]; then
      for tgt in $targets; do
          # note: /lib is a link to /usr/lib, so use readlink on both paths to make
          # sure we match
          targetFile="$baseDir/etc/systemd/system/${tgt}.wants/$service"
          if [[ -L $targetFile && $(readlink -f $targetFile) == $(readlink -f "$libServiceFile") ]]; then
            $RM -f $targetFile
            if [ -n "$logfile" ]; then
               $ECHO "Removed $targetFile." | $TEE -a $logfile
            fi
          elif [ -n "$logfile" ]; then
            $ECHO "$service is not currently enabled" | $TEE -a $logfile
          fi  
      done
  else
      # can't perform the requested action in a chroot session
      if [ -n "$logfile" ]; then
          $ECHO "Invalid action/service combination for chrootSystemctl: action=$action service=$service" | $TEE -a $logfile
      else
          $ECHO "Invalid action/service combination for chrootSystemctl: action=$action service=$service"
      fi
  fi
}

#    0 if not container env
#    1 if running in docker environment
#    2 if running in k8s environment
DOCKER_ENV=1
KUBE_ENV=2

function running_in_docker() {
  var=$(grep "docker" /proc/self/cgroup | wc -l)
  if [ $var -ne 0 ] || [ -f /.dockerenv ]; then
    echo 1
  else
    echo 0
  fi
}

function running_in_k8() {
  var=$(grep "kubepods" /proc/self/cgroup | wc -l)
  if [ $var -ne 0 ] || [ ! -z $KUBERNETES_SERVICE_HOST ]; then
    echo 1
  else
    echo 0
  fi
}

function isCnf
{
  ret=0
  if [ $(running_in_k8) -ne 0 ]; then 
    ret=$KUBE_ENV
  elif [ $(running_in_docker) -ne 0 ]; then 
    ret=$DOCKER_ENV
  fi
  echo $ret
}

function getActualCeName()
{
    local actualCeName=""

    if [ -f $ACTUAL_NAME_CONF ]; then
       actualCeName=`$GREP actualCeName $ACTUAL_NAME_CONF | $AWK -F "="  '{print $2}'`
    else
       actualCeName=`$GREP ceName $SBXCONF_FILE | $AWK -F "=" '{print $2}'`
    fi

    $ECHO $actualCeName
}

function getActualSystemName()
{
    local actualSystemName=""

    if [ -f $ACTUAL_NAME_CONF ]; then
       actualSystemName=`$GREP actualSystemName $ACTUAL_NAME_CONF | $AWK -F "=" '{print $2}'`
    else
       actualSystemName=`$GREP systemName $SBXCONF_FILE | $AWK -F "=" '{print $2}'`
    fi

    $ECHO $actualSystemName
}

function getRunAsUserId()
{
  echo $($PS -p 1 -o uid -h)
}

function isRunAsRandomUser()
{
  runAsUserId=$(getRunAsUserId)
  if [[ ($runAsUserId -ne 0) && ($runAsUserId -ne $($ID -u sonusadmin)) ]]; then
    # running as a non-root non-sonusadmin, ie random user
    echo 1
  else
    echo 0
  fi
}

function groupfix()
{
  # /etc/group+ if present, needs to be manually written to /etc/group
  if [ ! -r /etc/group+ ]; then
    $LOGGER -s -t $FUNCNAME "Fix for group+ not needed, file not present"
  else
    $CP -f /etc/group+ /etc/group && $RM -f /etc/group+
    if [[ $? -ne 0 ]]; then
      $LOGGER -s -t $FUNCNAME "Fix for group+ failed"
      return 1
    fi
    $LOGGER -s -t $FUNCNAME "Fix for group+ successful"
  fi
}
# This fix is required in CNF if the pod is running as a non-root, non-sonusadmin, ie random user
# Takes optional variables
# CNF_FIX_USER : username, eg: emssftp
# CNF_FIX_PASSHASH : encrypted pass hash, eg : '$6$...', '*'
# CNF_FIX_NOPASS : 1 or unset
# CNF_FIX_CREATE_GROUP : 1 or unset
# CNF_FIX_DELETE_GROUP : 1 or unset
# CNF_FIX_SEC_GROUPS : Confd,sftponly,upload,newuser
function cnfNonRootPasswdFix()
{

  isCnfFlag=$(isCnf)
  runAsUserId=$(getRunAsUserId)

  if [[ ( $isCnfFlag == 0 ) || ( $(isRunAsRandomUser) != 1 ) ]]; then
    # No fix is needed
    return 0
  fi

  $LOGGER -s -t $FUNCNAME "Attempting fix, isCnf $isCnfFlag, runAsUserId $(getRunAsUserId), id $($ID -u)"

  # /etc/passwd+ if present, needs to be manually written to /etc/passwd
  if [ ! -r /etc/passwd+ ]; then
    $LOGGER -s -t $FUNCNAME "Fix for passwd+ not needed, file not present"
  else
    $CP -f /etc/passwd+ /etc/passwd && $RM -f /etc/passwd+
    if [[ $? -ne 0 ]]; then
      $LOGGER -s -t $FUNCNAME "Fix for passwd+ failed"
      return 1
    fi
    $LOGGER -s -t $FUNCNAME "Fix for passwd+ successful"
  fi
  groupfix
  # If specified further fixes may be needed for user
  if [[ ( -z "$CNF_FIX_USER" ) ]]; then
    $LOGGER -s -t $FUNCNAME "No more fixes needed"
  else

    # If provided, need to add password via chpasswd
    if [[ ( ! -z "$CNF_FIX_PASSHASH" ) || ( ! -z "$CNF_FIX_NOPASS" ) ]]; then
      if [[ ! -z "$CNF_FIX_NOPASS" ]]; then
        $ECHO $CNF_FIX_USER:'*' | $CHPASSWD -e
      else	
        $ECHO $CNF_FIX_USER:$CNF_FIX_PASSHASH | $CHPASSWD -e
      fi
      if [[ $? -ne 0 ]]; then
        $LOGGER -s -t $FUNCNAME "Updating pass hash failed for user $CNF_FIX_USER, with nopass=$CNF_FIX_NOPASS."
        return 1
      fi
      $LOGGER -s -t $FUNCNAME "Updating hash succeeded for user $CNF_FIX_USER, with nopass=$CNF_FIX_NOPASS."
    fi

    # If specified, need to create new group
    if [[ ( ! -z "$CNF_FIX_CREATE_GROUP" ) ]]; then
      new_user_gid=$($ID -u $CNF_FIX_USER)
      $GROUPADD -g $new_user_gid $CNF_FIX_USER
      if [[ $? -ne 0 ]]; then
          # /etc/group+ if present, needs to be manually written to /etc/group
          if [ ! -r /etc/group+ ]; then
            $LOGGER -s -t $FUNCNAME "Adding group $new_user_gid failed for user $CNF_FIX_USER"
            return 1  
          fi
          groupfix
      fi
      $LOGGER -s -t $FUNCNAME "Adding group $new_user_gid succeeded for user $CNF_FIX_USER"
    fi

    # If specified, delete group from /etc/group
    if [[ ( ! -z "$CNF_FIX_DELETE_GROUP" ) ]]; then
      if $GETENT group $CNF_FIX_USER > /dev/null; then
        # Remove the user from secondary groups
        $GREP $CNF_FIX_USER /etc/group | $CUT -d ":" -f 1 | $XARGS -n 1 $GPASSWD -d $CNF_FIX_USER

        # Remove user primary group
        $GROUPDEL $CNF_FIX_USER
        if [[ $? -ne 0 ]]; then
          # /etc/group+ if present, needs to be manually written to /etc/group
          if [ ! -r /etc/group+ ]; then
            $LOGGER -s -t $FUNCNAME "Deleting group failed for user $CNF_FIX_USER"
            return 1  
          fi
          groupfix
        fi
        $LOGGER -s -t $FUNCNAME "Deleting group succeeded for user $CNF_FIX_USER"    
      else
        $LOGGER -s -t $FUNCNAME "Group does not exist for user $CNF_FIX_USER"    
      fi
    fi

    # If specified, add secondary groups
    if [[ ( ! -z "$CNF_FIX_SEC_GROUPS" ) ]]; then
      for g in $( $ECHO $CNF_FIX_SEC_GROUPS | $TR ',' ' ' ); do
        $GPASSWD -a $CNF_FIX_USER $g
        if [[ $? -ne 0 ]]; then
          $LOGGER -s -t $FUNCNAME "Adding to group $g failed for user $CNF_FIX_USER"
          return 1
        fi
      done
      $LOGGER -s -t $FUNCNAME "Added user $CNF_FIX_USER to groups $CNF_FIX_SEC_GROUPS"
    fi

  fi

  # This is necessary to keep passwd and shadow db in sync
  # userdel fails to delete entry from /etc/shadow
  $LOGGER -s -t $FUNCNAME "Syncing db"
  $PWCONV

  return 0
}

if [[ "$#" -ge 1 ]]; then
  case $1 in
    "cnfNonRootPasswdFix")
      cnfNonRootPasswdFix
    ;;
  esac
fi