#!/usr/bin/env python3

###############################################################################
#
# Copyright (c) 2019-2020 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# updateCallmix.py
#
# Module Description:
#
# This script modifies the file 
# /opt/sonus/conf/swe/capacityEstimates/.activeCallMixInput.json such that upon 
# upgrade, the callmix abides by the rules introduced in 8.1 release in Hybrid 
# transcoding.
# 
# This script can be run as a standalone script as well, so that, 
# .activeCallMixInput.json can be modified before performing upgrade in 
# pre-8.1.0R2 release, where this upgrade mechanism is absent.
#
# Usage for running the script as standalone script : 
# ./updateCallmix.py standalone
#
# NOTE     : ANY CHANGES MADE HERE WOULD REQUIRE CORRESPONDING CHANGES IN  
#            /hornet/cpx/src/CpxSwe.cpp
# FUNCTION : CpxSysSWeModifyCodecProfileTableOnUpgrade
################################################################################

"""
Procedure : 

1. If default profile is activated, the .activeCallMixInput.json file won't be
   present. Skip the procedure.

2. If the file .activeCallMixInput.json is not in json format, print a suitable
   message and skip.

3. Parse the .activeCallMixInput.json file and get the corresponding transcode callmix.

4. If the codecMix already contains G7112G711, skip the procedure.

5. Calculate the sum of all codec percentages. Sum of G711%, sum of non-G711%.

6. If G711% <= non-G711%, skip the procedure.

7. Else loop through the transcode codec profile entries and get the modified
   callmix.
   Overview of logic :
   for entry in transcode codec entries:
       if non-G711% == 0:
           if entry.codec == G711:
               replace current G711 codec entry with G7112G711.
       elif non-G711% < G711%:
           if entry.codec == G711:
               if non-G711% >= entry.codec%:
                   non-G711% -= entry.codec%
                   continue
               if non-G711% < entry.codec%:
                   G7112G711% = entry.codec - non-G711%
                   create a new G7112G711 entry with same ptime and G7112G711%.
                   set the current G711 entry's percentage to non-G711%
                   non-G711% = 0

8.  Update the callmix file with the new updated transcode codec mix.

9.  Update .activeCallMixInput.json with this content.

10. Any changes made here, needs to be replicated in the function 
    CpxSysSWeModifyCodecProfileTableOnUpgrade in /hornet/cpx/src/CpxSwe.cpp 
    
"""
import os
import sys
import logging
import json
import shutil
from collections import namedtuple

sys.path.append('/opt/sonus/staging/')
sys.path.append('/opt/sonus/bin/')
from sonusCommonFiles import *
from sonusCommands import *


log = logging.getLogger('updateCallmix')
log.addHandler(logging.NullHandler())


CodecMixItem = namedtuple('CodecMixItem', ['name', 'ptime', 'percent'])

def usage():
    """
    Help function
    """
    file_log_info("Usage : ")
    file_log_info("./updateCallmix.py <standalone|upgrade> <base_path>\n")
    file_log_info("Example for running the script standalone post upgrade - ")
    file_log_info("./updateCallmix.py standalone\n")
    file_log_info("Example for running the script from the upgrade scripts - ")
    file_log_info("./updateCallmix.py upgrade /src \n")
    file_log_info("Log file : %s" % CORE_PARTITION_LOG)
    

def touch_file(fname):
    """
    Param   : File name
    Fn Desc : Opens a file for editing. Returns error if not able to open/edit 
    the file.
    Return  : True/False
    """
    try:
        with open(fname, 'a') as f:
            f.write("\n--CORE-PARTITION UPGRADE LOGS--\n")
        return True
    except:
        print("error setting up logging")
        return False


def file_log_info(msg):
    print(msg)
    log.info(msg)


def file_log_error(msg):
    print(msg)
    log.error(msg)


def setup_logging():
    """
    Setup logging
    """
    log_file = CORE_PARTITION_LOG
    if touch_file(log_file):
        logging.basicConfig(level = logging.INFO,
                            format = "[%(asctime)-15s %(name)s %(levelname)s] - %(message)s",
                            filename = log_file)
    else:
        print("error setting up logging")
        logging.basicConfig(level = logging.DEBUG,
                            format = "[%(asctime)-15s %(name)s %(levelname)s] - %(message)s")


def does_file_exist(file):
    """
    Param   : File Name
    Fn Desc : Checks for the file's existance
    Return  : True/False
    """
    if not os.path.exists(file):
        log.error("The file " + file + " does not exist.")
        return False
    return True


def read_file(file):
    """
    Param   : File Name
    Fn Desc : Reads the content of the file
    Return  : File content.
    """
    try:
        with open(file) as f:
            return f.read()
    except (OSError, IOError):
        log.info("Cannot read %s" % file)
        return None


def load_json_content(file):
    """
    Param   : File Name
    Fn Desc : Reads the json content in the file
    Return  : json content read from the file.
    """
    data = read_file(file)
    if data == None:
        return None

    try:
        json_data = json.loads(data)
    except ValueError:
        log.exception("error parsing %s" % file)
        return None
    return json_data


def get_xcode_details(callmix):
    """
    Param   : callmix
    Fn Desc : Populates the transcode callmix tuple.
    Return  : Transcode callmix tuple.
    """
    try: 
        xcode_details = [CodecMixItem(k['codec'], int(k['ptime']), float(k['percent']))
                                     for k in callmix['transcodeCodecProfile']]
    except Exception as e:
        log.error("Exception encountered while obtaining transcode codec mix details.")
        log.error("Exception : " + str(e))
        return None
    return xcode_details


def get_codec_dict(name, ptime, percent):
    """
    Param   : Transcode callmix elements : codec name, ptime and codec 
    percentage.
    Fn Desc : Populates the transcode codecMix dictionary.
    Return  : Transcode codecMix dictionary.
    """
    c_dict = {"codec" : name}
    c_dict.update({"ptime" : ptime})
    c_dict.update({"percent" : percent})
    log.info("Adding %s %d %f entry" % (name, ptime, percent))
    return c_dict


def get_modified_xcode_details(non_g711p, g711p, xcode_details):
    """
    Params : 
    non_g711p : non G711 percentage
    g711p     : G711 percentage
    xcode_details : Transcode callmix tuple.

    Fn Desc   : Modifies the codec mix based on the new set of rules.
    return    : Modified callmix.
    """
    transcode_codec_mix = []
    for entry in xcode_details:
        if entry.name != "G711":
            codec_dict = get_codec_dict(entry.name, entry.ptime, entry.percent)
            transcode_codec_mix.append(codec_dict)

    for entry in xcode_details:
        if non_g711p == 0:
            if entry.name == "G711":
                codec_dict = get_codec_dict("G7112G711", entry.ptime, entry.percent)
                transcode_codec_mix.append(codec_dict)

        elif non_g711p >= g711p:
            if entry.name == "G711":
                codec_dict = get_codec_dict("G711", entry.ptime, entry.percent)
                transcode_codec_mix.append(codec_dict)
        elif non_g711p < g711p:
            if entry.name == "G711":
                if non_g711p >= entry.percent:
                    codec_dict = get_codec_dict("G711", entry.ptime, entry.percent)
                    transcode_codec_mix.append(codec_dict)
                    non_g711p -= entry.percent
                else:
                    g7112g711_percent = entry.percent - non_g711p
                    codec_dict = get_codec_dict("G7112G711", entry.ptime, g7112g711_percent)
                    transcode_codec_mix.append(codec_dict)

                    codec_dict = get_codec_dict("G711", entry.ptime, non_g711p)
                    transcode_codec_mix.append(codec_dict)

                    non_g711p = 0

    return transcode_codec_mix


def dump_callmix_to_file(callmix, file):
    """
    Param   : Modified callmix dictionary, file to dump the data.
    Fn Desc : Populates the modified callmix into the file.
    Return  : Status of writing json into the file : True/False.
    """
    try:
        with open(file, 'wb') as f:
            json_data = json.dumps(callmix) + "\n"
            f.write(json_data)
    except Exception as e:
        log.error("Exception encountered while dumping the callmix content.")
        return False
    return True


def update_callmix_file(xcode_codec_mix, file_to_update, callmix_content):
    """
    Params  : 
    1. modified transcode codec mix.  
    2. file to update the transcode callmix.
    3. Entire callmix content of the current activated profile.
    Fn Desc : 
    1. Take backup of existing file.
    2. Update json with xcode codec mix.
    3. Update the json file.
    Return  : Status of writing json into the file : True/False.
    """
    shutil.copyfile(file_to_update, file_to_update + '_orig')
    callmix_content['transcodeCodecProfile'] = xcode_codec_mix
    ret_status = dump_callmix_to_file(callmix_content, file_to_update)
    return ret_status


def main():

    try :
        setup_logging()
    except Exception as e:
        file_log_error("Exception encoutered during setup of the log file")
        return False

    if len(sys.argv) < 2:
        file_log_error("Insufficient arguments! Exiting")
        usage()
        return False

    allowed_arguments = ['standalone', 'upgrade']
    cmd               = sys.argv[1]

    if not cmd in allowed_arguments:
        log.error("unrecognized command %s", cmd)
        return False
    else:
        if cmd == 'standalone':
            callmix_file = ACT_CALLMIX_JSON
        else:
            upgrade_base_path = sys.argv[2]
            callmix_file = upgrade_base_path + ACT_CALLMIX_JSON

    log.info("Using " + callmix_file + " for further operations.")

    if not does_file_exist(callmix_file):
        log.info("Assuming default profile.")
        log.info("No operation required to be performed. Exiting.")
        return True


    callmix_content = load_json_content(callmix_file)
    if callmix_content == None:
        log.error("Callmix file exists, but doesn't contain json data! Exiting.")
        return False

    xcode_details = get_xcode_details(callmix_content)
    if xcode_details == None or xcode_details == []:
        log.info("Transcode codec mix is either Null or not present! Exiting.")
        return True

    # If G7112G711 is already present in the callmix, then do not perform any
    # modifications in the callmix.

    sum_g7112g711 = sum([k.percent for k in xcode_details if k.name == 'G7112G711'])
    if sum_g7112g711 > 0 :
        log.info("The callmix already contains data in new format! Exiting.")
        return True

    # Get sum of G711 and non-G711 percentages.
    sum_non_g711 = sum([k.percent for k in xcode_details if k.name != 'G711' ])
    sum_g711     = sum([k.percent for k in xcode_details if k.name == 'G711' ])
    sum_of_codec_percents = sum_g711 + sum_non_g711 

    if sum_of_codec_percents != 100:
        log.error("Sum of codec percentages do not add to 100. Exiting!")
        return False

    if sum_non_g711 >= sum_g711:
        log.info("No need of any modification in callmix. Skipping further procedure")
        return True

    try:
        modified_callmix = get_modified_xcode_details(sum_non_g711, sum_g711, xcode_details)
    except Exception as e:
        log.error("Exception encountered while getting the modified callmix.")
        log.error("Exception : " + str(e))
        return False

    status = update_callmix_file(modified_callmix, callmix_file, callmix_content)
    
    if not status:
        log.error("Unable to write content into the callmix file. Exiting")
        return False
    else:
        log.info("Callmix upgrade operation completed.")

    return True

if __name__ == "__main__":

    status = main()
    if not status:
        sys.exit(-1)

