#!/bin/bash
#############################################################
#
# Copyright (c) 2010 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# Module Description:
#    Script to install application.
#    
#############################################################

# Extracting Sonus command files from SBC tarball & sourcing
# DO NOT ADD ANYTHING ABOVE THIS POINT! 
# This needs to be the first commands executed!
/bin/tar -zxvf /opt/sonus/staging/sbc*.tar.gz sonusCommands.sh sonusCommands.py sonusCommands.expect sonusCommands.pm sonusCommonFiles.sh sonusCommonFiles.pm sonusCommonFiles.py sonusCommonFiles.expect > /dev/null
/bin/chown root:upgrade sonusCommands.sh sonusCommands.py sonusCommands.expect sonusCommands.pm sonusCommonFiles.sh sonusCommonFiles.pm sonusCommonFiles.py sonusCommonFiles.expect
/bin/chmod 755 sonusCommands.sh sonusCommands.py sonusCommands.expect sonusCommands.pm sonusCommonFiles.sh sonusCommonFiles.pm sonusCommonFiles.py sonusCommonFiles.expect
if [ -e /opt/sonus/staging/sonusCommands.sh ] && [ -e /opt/sonus/staging/sonusCommonFiles.sh ]
then
    source /opt/sonus/staging/sonusCommands.sh
    source /opt/sonus/staging/sonusCommonFiles.sh
else
    /bin/echo "sonusCommands.sh and/or sonusCommonFiles.sh not found. Exiting..."
fi

# Copying Sonus commands file to /opt/sonus/bin
$MKDIR -p $SONUS_BIN_DIR
$CHMOD 755 $SONUS_BIN_DIR
$CP $SONUS_STAGING_DIR/sonusCommands.sh $SONUS_STAGING_DIR/sonusCommands.py $SONUS_STAGING_DIR/sonusCommands.expect $SONUS_STAGING_DIR/sonusCommands.pm $SONUS_BIN_DIR/
$CP $SONUS_STAGING_DIR/sonusCommonFiles.sh $SONUS_STAGING_DIR/sonusCommonFiles.py $SONUS_STAGING_DIR/sonusCommonFiles.expect $SONUS_STAGING_DIR/sonusCommonFiles.pm $SONUS_STAGING_DIR/sbxPeerExpect* $SONUS_BIN_DIR/
$CHMOD 755 $SONUS_BIN_DIR/sonusCommands.* $SONUS_BIN_DIR/sonusCommonFiles.* $SONUS_STAGING_DIR/sbxPeerExpect*

stagingDir=$SONUS_STAGING_DIR
logFile=$stagingDir/appInstall.out
installCommand="$SBX_INSTALL_SH -L $logFile -f sbc*.deb"
preInstallArg=""

resetLogFile="true"

# Flag to check whether to skip preInstall checks
skipPreInstallChecks="false"

# Flag to check whether to skip SBC service start at the end of the installation (Default : false)
skipServiceStart="false"

# Save config file to /home/common which will go to P3 and will be accessible by both P1 and P2
fileName=`$BASENAME $0`
p2Install="false"
p2Base="$HOME_COMMON/"
p2ConfigFile="$HOME_COMMON/installSBX.conf"
p2gSbxInstallUpgradeCommand="$fileName $@ -R -q"
p2gSbxInstallUpgradeCommand="$stagingDir/$p2gSbxInstallUpgradeCommand"
pmMarkerFile="$SBX_INSTALL_UPGRADE_TXT"

$MKDIR -p $p2Base

# New config file location
newConfigFile=$stagingDir/.newSbx.conf

# output an error and exit the script
function error_exit()
{
    local exitNow=$1
    local errorStr=$2

    logMsg $logFile "$errorStr"

    if [ $exitNow -eq 1 ]; then
       logMsg $logFile "Exiting..."
       exit 1
    fi
}

# output an error and exit the script
# don't use logMsg since we are called before the utilities are loaded.  We need
# to check first so we can give out reasonable error messages.
# NOTE: mimic what logMsg does....
function early_error_exit()
{
    local errorStr=$1

    dateStr=`$DATE +"%Y-%m-%d %H:%M:%S"`
    $ECHO -e "$errorStr"
    if [ -n "$logFile" ]; then
       $ECHO -e "$errorStr" >> $logFile
    fi
    if [ ! -z "$gRollingAppInstallLogFile" ]; then
       dateStr=`$DATE +"%b %d %T"`
       $ECHO -e "$dateStr $errorStr" >> $gRollingAppInstallLogFile
    fi

    exit 1
}

# make sure that the install is running from the staging directory
function check_location() 
{
   local dirName=`$DIRNAME $0`
   if [ "$dirName" = "." ]; then
      dirName=$PWD
   fi
   if [ "$dirName" != "$stagingDir" ]; then
      early_error_exit "ERROR: Installation/update files should be installed to and run from $stagingDir; Exiting..."
   fi
}



# prevent starting if diagnostics are running
checkDiags()
{
    pid=`$PIDOF testAppDiagConnexip5200`
    if [ -n "$pid" ]; then
        $ECHO "ERROR: Install cannot be started while diagnostics are running; Exiting..."
        exit 1
    fi
}

# Check for existance of only one package of the requested kind
# and its validy against the Checksum file
function check_package_errors()
{
  local package=$1
  local checkChecksum=$2
  errorString1="ERROR: More than one version found at: $PWD"
  errorString2="ERROR: Please remove files so only target install files are located at: $PWD"
  errorString3="ERROR: Package checksum mismatch for: $package"
  errorString4="ERROR: Package $package not found at: $PWD"

  # Check if we have multiple packages in the current directory...
  packageCount=`$LS $package 2> /dev/null | $WC -l`
  if [[ $packageCount -gt 1 ]]; then
     error_exit 0 "\n$errorString1"
     error_exit 0 "\n`$LS -la $package`"
     error_exit 1 "\n$errorString2"
  fi

  if [[ $packageCount -lt 1 ]]; then
     error_exit 0 "\n$errorString4"
     error_exit 1 "\n`$LS -la $package`"
  fi

  if [ "$checkChecksum" = "y" ]; then
     packageToBeInstalled=`$LS $package 2> /dev/null`

     # Assume it be .deb
     checkSum="$MD5SUM"
     File=`$ECHO $packageToBeInstalled | $SED -e 's/.deb/.md5/'`

     # Check if .tar.gz
     if $ECHO $packageToBeInstalled | $GREP "\.tar\.gz"; then
       checkSum="$SHA256SUM"
       File=`$ECHO $packageToBeInstalled | $SED -e 's/.tar.gz/.sha256/'`
     fi

     # Remove "-INSTALL" if present in file content
     $SED -i 's/-INSTALL//' $File

     if [ ! -e $File ]; then
       error_exit 0 "\nChecksum file $File not found at: $PWD"
       error_exit 1 "\n`$LS -la $File`"
     fi

     # Compute checksum
     # Make sure that this matches with corresponding buildDEB script
     computedChecksumFile=$($MKTEMP --tmpdir=$SONUS_TMP_DIR)
     packageToBeInstalledFileName=`$BASENAME $packageToBeInstalled`

     $checkSum $packageToBeInstalledFileName > $computedChecksumFile

     $DIFF $File $computedChecksumFile > /dev/null 2>&1
	 
     if [ $? -ne 0 ]; then
       error_exit 0 "\n$errorString3"
       error_exit 0 "Invalid checksum, expected: "
       error_exit 0 "\t\t `$CAT $File`"
       error_exit 0 "\t\t computed: "
       error_exit 1 "\t\t `$CAT $computedChecksumFile`"
     fi
	 $RM -f $computedChecksumFile
  fi
}

# Check for all the packages required for installation
function check_packages()
{
  check_package_errors "./sbc*.tar.gz" "y"
  sbcpackage=`$LS sbc*.tar.gz`
  $TAR -xvzf $sbcpackage 

}

# Install the packages
function install_packages()
{
  # If using an existing config file, setup the option
  if [ -n "$configFile" ]; then
     installCommand="$installCommand -c $configFile"
  fi

  # Extract the contents of the package and verify it
  logMsg $logFile "INMESSAGE: Extracting package contents..."
  local output=`$TAR -zxvf sbc*.tar.gz`
  logMsg $logFile $output
  $CHOWN root:sonus * 2>/dev/null
  $CHGRP upgrade $stagingScripts $stagingFiles 2>/dev/null
  $CHMOD 0770 $stagingScripts
  $CHMOD 0660 $stagingFiles

  check_package_errors "./sbc*.deb" "n"

  # Check for OS version 
  getConnexIPOSVersion
  getReqOSVersion
  if [[ "$connexIPOSVersion" != "$reqOSVersion" ]]; then
     error_exit 1 "ERROR: Invalid OS version: $connexIPOSVersion, Please re-iso to $reqOSVersion and then install the app; Exiting..."
  else
     logMsg $logFile "Current OS version [$connexIPOSVersion] and required OS version [$reqOSVersion] are same."
  fi

  # Install NBS application...
  logMsg $logFile "INMESSAGE: Installing sbx service..."
  $installCommand 2>> $logFile
  if [ $? -ne 0 ]; then
     error_exit 1 "ERROR: installation failure in sbxInstall.sh"
  fi
}

usage()
{
$CAT << EOF
usage: `$BASENAME $0` [-h] [-c <configuration file>] [-R] [-o]

OPTIONS:
   -h               Print this message.
   -c <config file> Use configuration file as input.
   -R               Confirm reboot if a reboot is necessary.
   -o               Skip OS Upgrade check.
   -n               Skip SBC service start after installation is complete.
EOF
}

# Set program...
program=`$BASENAME $0`

# change the current working directory to /opt/sonus/staging
cd $stagingDir

# verify that the install is running from the correct location
check_location

# Extract the sonusUtils -- required by installation scripts
stagingScripts="sonusCommonUtils.sh sonusUtils.sh sbxStart.sh checkDiskUsage.sh checkDiskPerformance.sh preInstallCheck.sh appPreUpgradeChecks.sh sbxPeerExpect.pl sbxPeerExpectUtils.pm revertUtil.sh switch_boot.sh partition_tool.sh sbxRevert.pl HostCheck sbxVerifyDrbdPartition.sh configureGrub.sh configureGrubPwd.sh cfgValidation.sh commonTimeZones.sh commonCfgValidation.sh personality.sh personality.pm"
stagingFiles=".buildinfo"
$TAR -zxvf sbc*.tar.gz $stagingScripts $stagingFiles > /dev/null
$CHOWN root $stagingScripts $stagingFiles 2>/dev/null
$CHGRP upgrade $stagingScripts $stagingFiles 2>/dev/null
if [ $? -ne 0 ]; then
  # If upgrade group not available, set it to sonus
  $CHGRP sonus $stagingScripts $stagingFiles 2>/dev/null
fi
$CHMOD 0770 $stagingScripts
$CHMOD 0660 $stagingFiles

if [ -e $STAGING_SONUS_UTILS_SH ]; then
   . $STAGING_SONUS_UTILS_SH
else
   early_error_exit "ERROR: $program Unable to extract `$BASENAME $STAGING_SONUS_UTILS_SH`, Exiting..."
fi

hostSubType=`$CAT $HOST_SUB_TYPE`

if [[ "$hostSubType" == "virtualCloud" || "$hostSubType" == "virtualContainer" ]] && [[ -e $SBX_START_SH ]]; then
   # This is a case of appInstall on cloud when an application is already installed. This is not suported on cloud, so just error out.
   early_error_exit "ERROR: Cannot install application on hostSubType=$hostSubType when an app is already installed. You can either upgrade or create a new instance. Exiting..." 
fi

while getopts "hc:Roqn" OPTION
do
   case $OPTION in
   h)
      usage
      exit 1
      ;;
   c)
      configFile="$OPTARG"
      if [ -z $configFile ]
      then
         error_exit 1 "Invalid configuration file, Exiting..."
      else
        # remove leading whitespace
        configFile=`$ECHO $configFile|$SED -e 's/^[ \t]*//'`
      fi
      configFileTemp=`$ECHO $configFile|$SED -e 's|^/||'`
      if [ $configFile = $configFileTemp ]; then
         logMsg $logFile "Using configuration file $PWD/$configFile..."
      else
         logMsg $logFile "Using configuration file $configFile..."
         $CP -pf $configFile $p2ConfigFile 
      fi
       

      # make sure the input is all valid
      . $configFile
      validateConfig

      # skip config file validation in preInstall since we already did it
      preInstallArg="-s -f $configFile"
      ;;
   R)
     installCommand="$installCommand -R"
     ;;
   q) 
      # set p2Install flag
      p2Install="true"
      # Add marker for PM
      $ECHO "Install in progress..." 2>/dev/null > $pmMarkerFile 
      ;;
   n)
     skipServiceStart="true"
     ;;
   ?)
      usage
      exit 1
      ;;
   esac
done

shift $(($OPTIND - 1))

declare -a targetFiles=("sonusCommonUtils.sh" "checkDiskUsage.sh" "checkDiskPerformance.sh" "preInstallCheck.sh" "appPreUpgradeChecks.sh" "sbxPeerExpect.pl" "sbxPeerExpectUtils.pm" "sbxStart.sh" "revertUtil.sh" "switch_boot.sh" "partition_tool.sh" "sbxRevert.pl" "configureGrub.sh" "configureGrubPwd.sh" "personality.sh" "personality.pm" )

for i in "${targetFiles[@]}"
do
   if [ ! -e $stagingDir/$i ]; then
      early_error_exit "ERROR: Unable to extract $stagingDir/$i, Exiting..."
   fi
done

# Create installUpgrade Directory
$MKDIR -p $SONUS_INSTALLUPGRADE_DIR
$CHOWN root:upgrade $SONUS_INSTALLUPGRADE_DIR
$CHMOD 750 $SONUS_INSTALLUPGRADE_DIR

# To support revert to older version
$CP -f sbxRevert.pl $SONUS_INSTALLUPGRADE_DIR/

$CP -f revertUtil.sh switch_boot.sh partition_tool.sh configureGrub.sh configureGrubPwd.sh /usr/local/bin/
$CHMOD 0775 $REVERT_UTIL_SH $SWITCH_BOOT_SH $PARTITION_TOOL_SH $SBX_REVERT_PL $CONFIGURE_GRUB_SH $CONFIGURE_GRUB_PWD_SH

# Validate the execution of this script
validateExecution $PPID "install" || exit 1

# reset the log file only if flag is set
if [ $resetLogFile == "true" ]
then
  # reset the log file
  $CAT /dev/null > $logFile
fi

logStartUpdate $logFile

hostType=$(getHwType)

checkDiags

# Disable audit system to prevent high auditd activity
logMsg $logFile "Disabling audit system..."
if ! $( $AUDITCTL -e 0 > /dev/null 2>&1 && $SERVICE auditd stop > /dev/null 2>&1); then
   # Could not disable audit system
   logMsg $logFile "Could not disable audit system..."
fi

if [[ "$p2Install" != "true" ]];then
   if [ $skipPreInstallChecks != "true" ]; then
      # verify that it is ok to continue with the install
      logMsg $logFile "Performing pre-install checks..."
      # append '-c' and '-l' options to preInstall script to skip creation of LSWU status file and to run localOnly checks respectively.
      # In failure case log stdout to appInstall log file and always log stderr to preInstall log file
      preInstallErr=`$PRE_INSTALL_CHECK_SH $preInstallArg -c -l 2>> $SONUS_STAGING_DIR/preUpgradeCheck.log`
      if [ $? -ne 0 ]; then
         error_exit 1 "$preInstallErr"
      fi
   fi

   # verify all files neeeded are in place
   check_packages
   
   # Checking if revert possible
   revertCapable
   if [ $? -eq 0 ]; then
      p2gSbxInstallUpgradeCommand=${p2gSbxInstallUpgradeCommand/$configFile/$p2ConfigFile}
      logMsg $logFile "Calling checkDiskforRevert to scan disk to allow Install..p2gSbxInstallUpgradeCommand=$p2gSbxInstallUpgradeCommand "
      checkDiskforRevert "INSTALL" "$p2gSbxInstallUpgradeCommand" $logFile || exit 0
   else
      $ECHO "Install in progress..." 2>/dev/null > $pmMarkerFile
   fi
fi

# perform final checks and install the software
# note: install does pre-installation checks prior to modifying the system,
# and will perform removal of the old service and install of the new
# service.
install_packages

logEndUpdate $logFile

# Remove SBX install/upgrade in progress marker file.
removeMarkerFile $sbxInstallUpgradeInProgressKey

# Check if reboot marker exists
if [ -e /tmp/rebootInProgressKey.key ];then
  logMsg $logFile ""
  logMsg $logFile "INMESSAGE: Rebooting application service..."
  logMsg $logFile ""
else
  if [ "$skipServiceStart" != "true" ]; then
    # Start SBX service
    $SERVICE_SH sbx start | $TEE -a $logFile
  else
    logMsg $logFile ""
    logMsg $logFile "Skipping SBC service start..."
    logMsg $logFile ""
  fi
  logMsg $logFile ""
  logMsg $logFile "INMESSAGE: SBC Installation Complete"
  logMsg $logFile ""
fi

exit 0
