#!/bin/bash
#############################################################
#
# Copyright (c) 2016 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# appPreUpgradeChecks.sh
#
# 08/01/16
#
# Module Description:
#
# This script is invoked by preInstallCheck.sh
# This script performs different app level preUpgrade Checks and reports
# errors/factual information(like current callCount etc) to $preUpgradeCheckLogs
#
# Note 1: $preUpgradeCheckLogs and $ceName should be exported before calling this script.
# Note 2: Script should use preInstallLogMsg() API to log the data.
# Note 3: returns 0 on success, 1 on failure.
#
#############################################################

# Sourcing Command Variables
if [ -e /opt/sonus/staging/sonusCommands.sh ]
then
    source /opt/sonus/staging/sonusCommands.sh
elif [ -e /opt/sonus/bin/sonusCommands.sh ]
then
    source /opt/sonus/bin/sonusCommands.sh
else
    /bin/echo "Could not locate sonusCommands.sh Exiting..."
    exit 1
fi

if [ -e /opt/sonus/staging/sonusCommonFiles.sh ]
then
    source /opt/sonus/staging/sonusCommonFiles.sh
elif [ -e /opt/sonus/bin/sonusCommonFiles.sh ]
then
    source /opt/sonus/bin/sonusCommonFiles.sh
else
    $ECHO "Could not locate sonusCommonFiles.sh Exiting..."
    exit 1
fi

installRoot=$(dirname $0)
. ${installRoot}/sonusUtils.sh

error_code=0

# Check that all users/groups are following correct new naming convention. Fail pre-upgrade checks if invalid user/group name(s) present.
preInstallLogMsg "Checking if all users/groups are following the correct new naming convention." "user"

# ANY CHANGES IN BELOW LIST OF INVALID NAMES MUST ALSO BE UPDATED IN $HORNET_ROOT/cpx/src/CpxRegex.cpp
invalidNamesPattern="^tty$\|^disk$\|^kmem$\|^dialout$\|^fax$\|^voice$\|^cdrom$\|^floppy$\|^tape$\|^sudo$\|^audio$\|^dip$\|^src$\|^utmp$\|^video$\|^sasl$\|^plugdev$\|^staff$\|^users$\|^nogroup$"

validateNames(){
        type=$1
        invalidNames=

        if [ "$type" == "user" ]
        then
            numberofInstances=`$CONFD_CMD -c "num_instances /aaa/authentication/users/user"`
        else
            numberofInstances=`$CONFD_CMD -c "num_instances /nacm:nacm/groups/group"`
        fi

        for ((i=0;i<$numberofInstances;i+=1))
        do

            if [ "$type" == "user" ]
            then
                 instanceName=`$CONFD_CMD -c "get /aaa/authentication/users/user[$i]/name"`
            else
                 instanceName=`$CONFD_CMD -c "get /nacm:nacm/groups/group[$i]/name"`
            fi

            if [ $? -ne 0 ]; then
                preInstallLogMsg "Failed to fetch user/group name from CDB" "user"
                continue
            fi

            `$ECHO $instanceName | $GREP -q $invalidNamesPattern`
            res=$?

            if [[ ! "$instanceName" =~ ^[A-Za-z_][-.A-Za-z_0-9]*$ ]] || [[ $res -eq 0 ]]; then
                invalidNames="$invalidNames $instanceName"
            fi
        done

        if [[ -n "$invalidNames" ]]; then
            preInstallLogMsg "ERROR: Following $type(s) were found not conforming to the new naming convention:$invalidNames" "user"
            preInstallLogMsg "ERROR: Please remove the invalid $type(s) and retry." "user"
            error_code=1
        else
            preInstallLogMsg "No invalid $type names found." "user"
        fi
}

productName=$($CAT $DMIDECODE_OUT | $GREP -m 1 "Product Name" | $AWK '{ print $NF }')
if [[ ( `$CAT /etc/hostSubType` == "virtual" ) && ( $productName == "VMware" ) ]] && [[ -e /usr/bin/vmware-toolbox-cmd ]]; then
    memReservation=`vmware-toolbox-cmd stat memres| $AWK '{print $1}'`
    requiredRAM=6
    minReservation=$((1000*$requiredRAM))
    #check memoryReservation
    if [[ $memReservation -lt $minReservation ]]; then
        preInstallLogMsg "Error: Insufficient MemoryReservation: $memReservation. Need atleast $minReservation" "user"
        preInstallLogMsg "Error: Please increase MemoryReservation and retry." "user"
        error_code=1
    fi
fi

fipsModeEnabled="$($GREP "fipsMode=enabled" $SBXCONF_FILE)"
if [ -n "$fipsModeEnabled" ]; then
    $ECHO -e "set screen width 512\n show table profiles security ipsecProtectionProfile" | $CONFD_CLI -u admin --noaaa --noninteractive | $EGREP '3DesCbc|null|hmacMd5' > /dev/null
    if [ $? -eq 0 ]; then
        preInstallLogMsg "Error: Unsupported encryption/integrity algorithm used in ipsecProtectionProfile" "user"
        preInstallLogMsg "WARNING: Unsupported encryption/integrity algorithm 3DesCbc/null/hmacMd5 used ipsecProtectionProfile." "user"
        error_code=1
    fi  
    
    $ECHO -e "set screen width 512\n show table profiles security ikeProtectionProfile" | $CONFD_CLI -u admin --noaaa --noninteractive | $EGREP '3DesCbc|hmacMd5|modp768|modp1024|modp1536' > /dev/null
    if [ $? -eq 0 ]; then
        preInstallLogMsg "Error: Unsupported encryption/integrity algorithm or DH group used in ikeProtectionProfile" "user"
        preInstallLogMsg "WARNING: Unsupported encryption/integrity algorithm 3DesCbc/hmacMd5 or DH group modp768/modp1024/modp1536used ikeProtectionProfile." "user"
        error_code=1
    fi  

    $ECHO -e "set screen width 512\n show table profiles security dtlsProfile" | $CONFD_CLI -u admin --noaaa --noninteractive | $EGREP 'rsa-with-null-sha|rsa-with-3des-ede-cbc-sha|rsa-with-aes-128-cbc-sha|rsa-with-aes-128-cbc-sha-256|rsa-with-aes-256-cbc-sha|rsa-with-aes-256-cbc-sha-256|tls_rsa_with_aes_128_gcm_sha256|tls_rsa_with_aes_256_gcm_sha384|tls_chacha20_poly1305_sha256' > /dev/null
    if [ $? -eq 0 ]; then
        preInstallLogMsg "Error: Unsupported DTLS ciphersuite used in dtlsProfile" "user"
        preInstallLogMsg "WARNING: Unsupported DTLS ciphersuite rsa algorithms used in dtlsProfile." "user"
        error_code=1
    fi  
    
    $ECHO -e "set screen width 512\n show table profiles security EmaTlsProfile ciphersuite" | $CONFD_CLI -u admin --noaaa --noninteractive | $EGREP 'AES256_SHA256|AES256_SHA|AES256_GCM_SHA384|AES128_SHA256|AES128_GCM_SHA256|AES128_SHA' > /dev/null
    if [ $? -eq 0 ]; then
        preInstallLogMsg "Error: Unsupported TLS ciphersuite used in EmatlsProfile" "user"
        preInstallLogMsg "WARNING: System is in FIPS mode, These EmaTlsProfile ciphersuites containing RSA key transport are not allowed AES256_SHA256, AES256_SHA, AES256_GCM_SHA384, AES128_SHA256, AES128_GCM_SHA256, AES128_SHA" "user"
        error_code=1
    fi

    $ECHO -e "set screen width 512\n show table profiles security tlsProfile" | $CONFD_CLI -u admin --noaaa --noninteractive | $EGREP 'rsa-with-null-sha|rsa-with-3des-ede-cbc-sha|rsa-with-aes-128-cbc-sha|rsa-with-aes-128-cbc-sha-256|rsa-with-aes-256-cbc-sha|rsa-with-aes-256-cbc-sha-256|tls_rsa_with_aes_128_gcm_sha256|tls_rsa_with_aes_256_gcm_sha384|tls_chacha20_poly1305_sha256' > /dev/null
    if [ $? -eq 0 ]; then
        preInstallLogMsg "Error: Unsupported TLS ciphersuite used in tlsProfile" "user"
        preInstallLogMsg "WARNING: Unsupported TLS ciphersuite rsa algorithms used in tlsProfile." "user"
        error_code=1
    fi

    checkForSHA1inApacheServer=$($OPENSSL_RBBN x509 -in /opt/sonus/ema/apache/server.pem -noout -text | $GREP -i "Signature Algorithm:" | $GREP "sha1")
    checkForSHA1inApacheCaCert=$($OPENSSL_RBBN x509 -in /opt/sonus/ema/apache/caCert.pem -noout -text | $GREP -i "Signature Algorithm:" | $GREP "sha1")
    if [ -n "$checkForSHA1inApacheServer" ] || [ -n "$checkForSHA1inApacheCaCert" ]; then
        preInstallLogMsg "Error: SHA1 is unsupported for FIPS" "user"
        preInstallLogMsg "WARNING: Unsupported hashing algorithm SHA1 used for signature in Apache2 EMA certificates in FIPS mode" "user"
        error_code=1
    fi
fi

validateNames "group"
validateNames "user"

$ECHO -e "set screen width 512\n show table alarms currentStatus" | $CONFD_CLI -u admin --noaaa --noninteractive | $EGREP 'Critical|Major' > /dev/null
if [ $? -eq 0 ]; then
  preInstallLogMsg "WARNING: Critical/Major alarms are present. Please check the current alarms on active server EMA to ensure they are non-impacting." "user"
fi

$ECHO -e "set screen width 512\n show table profiles security tlsProfile" | $CONFD_CLI -u admin --noaaa --noninteractive | $EGREP 'tls_ecdh_ecdsa_with_aes_256_cbc_sha384' > /dev/null
if [ $? -eq 0 ]; then
   preInstallLogMsg "Error: Unsupported cipher suite used" "user"
   preInstallLogMsg "WARNING: Unsupported cipher tls_ecdh_ecdsa_with_aes_256_cbc_sha384 used." "user"
   error_code=1
fi

$ECHO -e "set screen width 512\n show table profiles security tlsProfile" | $CONFD_CLI -u admin --noaaa --noninteractive | $EGREP 'tls_ecdh_ecdsa_with_aes_256_gcm_sha384' > /dev/null
if [ $? -eq 0 ]; then
   preInstallLogMsg "Error: Unsupported cipher suite used" "user"
   preInstallLogMsg "WARNING: Unsupported cipher tls_ecdh_ecdsa_with_aes_256_gcm_sha384 used." "user"
   error_code=1
fi

preInstallLogMsg "Status of Pre-Upgrade check: $error_code"
exit $error_code
