#!/bin/bash

# Utility for configuring /etc/default/grub and for creating
# Ribbon specific grub menu entries.  This utility is utilized both
# on install and upgrade.

# pickup common commands and variables
if [ -e /opt/sonus/staging/sonusCommands.sh ]; then
    source /opt/sonus/staging/sonusCommands.sh
elif [ -e /opt/sonus/bin/sonusCommands.sh ]; then
    source /opt/sonus/bin/sonusCommands.sh
else
   /bin/echo "Could not locate sonusCommands.sh Exiting..."
   exit 1
fi
if [ -e /opt/sonus/staging/sonusCommonFiles.sh ]; then
    source /opt/sonus/staging/sonusCommonFiles.sh
elif [ -e /opt/sonus/bin/sonusCommonFiles.sh ]; then
    source /opt/sonus/bin/sonusCommonFiles.sh
else
   /bin/echo "Could not locate sonusCommonFiles.sh Exiting..."
   exit 1
fi

# some helpful variables
PROG=${0##*/}
hostType=`cat $HOST_TYPE`
hostSubType=`cat $HOST_SUB_TYPE`

# command line option variables
kernelVersion=
upgrade=0
quiet=0
forceFsck=
consoleTtyS0=
singleUser=0
device=
sbcVersion=
menuID=
title=
grubDefault=0
menuOptions=0
grubFile=
timeStamp=
skipKernelArgs=0
previousKernelArgs=


usage()
{
$CAT << EOF
usage: $PROG [-h] [-c add|remove] [-C add|remove] [-a add|remove] [-u] [-q] [-s] [-g] [-P kernel_args] [-T timestamp] -k kernel_ver -d device -v sbcVersion -t title -i id -f grub_file

Either add a configuration item for a grub menu entry or update the grub default file

OPTIONS:
   -h               Print this message
   -c add|remove    Add or remove fsck.mode=force kernel parameter
   -C add|remove    Add or remove console=ttyS0 kernel parameter
   -a add|remove    Add or remove AppArmor
   -d device        Root device
   -f               Grub file in which to place the entry
   -g               Update /etc/default/grub file
   -i menu id       Identifier to associate with the menu entry
   -k version       Kernel version
   -P               Kernel args to use instead of the default args (used for previous entry, i.e. revert)
   -q               Add 'quiet' kernel parameter
   -s               Single user mode
   -t title         Menu title displayed by grub
   -T timestamp     Last used timestamp
   -v version       SBC version
   -u               Add 'upgrade=true' to kernel command line

    -c is mutually exclusive from all others except -f
    -C is mutually exclusive from all others except -f
    -a is mutually exclusive from all others except -f
    -g is mutually exclusive from all others except -f
    -f is mandatory for all modes of operation
EOF
}

updateGrubDefault()
{
    # set common options, and then modify for each machine
    # NOTE: bullseye has switched to cgroupv2 but not everything supports it. revert to cgroupv1
    # until the issues are fixed (until cgroup-tools for bullseye supports cgroupv2)
    local commonKernelOptions="ro panic=1 idle=halt rootflags=data=ordered cgroup_enable=memory intel_pstate=disable clocksource=tsc fsck.repair=yes security=yama audit=1 audit_backlog_limit=8192 systemd.unified_cgroup_hierarchy=false"
    local consoleOptions
    local kernelOptions

    if [ "$hostType" == "ConnexIP5000" ]; then
        consoleOptions="console=tty1"
        kernelOptions="$commonKernelOptions nopku"
    else
        consoleOptions="console=tty0 console=ttyS0,115200"
        kernelOptions="$commonKernelOptions nohz=off"
        spectreOptions="spectre_v2=off pti=off kpti=off spectre_v2_user=off spec_store_bypass_disable=off"
        if [[ "$hostType" != "SBC7000" && "$hostType" != "SBC5400" ]]; then
            kernelOptions="$kernelOptions $spectreOptions"
        else
            kernelOptions="$kernelOptions i2c-i801.disable_features=0x10"
        fi
    fi

    # don't use uuid in entries
    $SED -i --follow-symlinks \
            -e "/GRUB_CMDLINE_LINUX_DEFAULT=/s/=.*/=\"$consoleOptions\"/" \
            -e "/GRUB_CMDLINE_LINUX=/s/=.*/=\"$kernelOptions\"/" \
            -e '/GRUB_DISABLE_LINUX_UUID/s/^#//' \
            -e '/GRUB_DEFAULT=/s/=.*/=saved/' $grubFile
}

addSonusLinuxEntry()
{
    local extra_args=

    if [ $skipKernelArgs -eq 1 ]; then
        # add kernel args passed in instead of the defaults. used for maintaining
        # options for the previous version (needed for revert)
        extra_args=
        $SED -i --follow-symlinks -e "s/prevKernelArgs=.*/prevKernelArgs=\"$previousKernelArgs\"/" $grubFile
    fi
    if [ $quiet -eq 1 ]; then
        extra_args="$extra_args quiet"
    fi
    if [ $singleUser -eq 1 ]; then
        extra_args="$extra_args single"
    fi
    if [ $upgrade -eq 1 ]; then
        # note: there is a hook in initrrd that utilizes this
        extra_args="$extra_args upgrade=true"
    fi

    # add the sbc version to kernel version to differentiate kernels completely
    # multiple releases could use the same kernel
    local sonusKernelVersion="$kernelVersion-amd64-sonus-$sbcVersion"

    # remove potential beginning space in variable extra_args (readability...)
    extra_args=`$ECHO $extra_args | $SED -e 's/^ //'`

    # call sonus_linux_entry to generate the menu item
    # refer to 50_ribbon_sda file for what is needed to call sonus_linux_entry
    # note: need to double quote those that could have spaces in them
    if [ $skipKernelArgs -eq 1 ]; then
        $ECHO "sonus_linux_entry $sonusKernelVersion $device \"\$prevKernelArgs\" \"$title\" \"$menuID\" \"$timeStamp\" $skipKernelArgs" >> $grubFile
    else
        $ECHO "sonus_linux_entry $sonusKernelVersion $device \"$extra_args\" \"$title\" \"$menuID\" \"$timeStamp\" $skipKernelArgs" >> $grubFile
    fi
}

modifyFsckParam()
{
    if [ $forceFsck == "add" ]; then
        $GREP sonus-curr-ver $grubFile | $GREP fsck\.mode=force
        if [ $? -ne 0 ]; then
            # for nice spacing purposes see if a param already exists
            $GREP sonus-curr-ver $grubFile | $GREP '/dev/mapper/[^ ]* ""'
            if [ $? -eq 0 ]; then
                #replacing "" with "var"
                $SED -i --follow-symlinks '/sonus-curr-ver/s|/dev/mapper/\([^ ]*\) ""|/dev/mapper/\1 "fsck.mode=force"|'  $grubFile
            else
                #replacing " with "var<space>
                $SED -i --follow-symlinks '/sonus-curr-ver/s|/dev/mapper/\([^ ]*\) "|/dev/mapper/\1 "fsck.mode=force |' $grubFile
            fi
        fi
    elif [ $forceFsck == "remove" ]; then
        $GREP sonus-curr-ver $grubFile | $GREP fsck\.mode=force
        if [ $? -eq 0 ]; then
            $SED -i --follow-symlinks '/sonus-curr-ver/s/fsck.mode=force[ ]*//' $grubFile
        fi
    fi
}

modifyConsoleTtyS0Param()
{
    if [ $consoleTtyS0 == "add" ]; then
        $GREP sonus-curr-ver $grubFile | $GREP console=ttyS0
        if [ $? -ne 0 ]; then
            # for nice spacing purposes see if a param already exists
            $GREP sonus-curr-ver $grubFile | $GREP '/dev/mapper/[^ ]* ""'
            if [ $? -eq 0 ]; then
                #replacing "" with "var"
                $SED -i --follow-symlinks '/sonus-curr-ver/s|/dev/mapper/\([^ ]*\) ""|/dev/mapper/\1 "console=ttyS0"|'  $grubFile
            else
                #replacing " with "var<space>
                $SED -i --follow-symlinks '/sonus-curr-ver/s|/dev/mapper/\([^ ]*\) "|/dev/mapper/\1 "console=ttyS0 |' $grubFile
            fi
            $GREP single-user $grubFile | $GREP '/dev/mapper/[^ ]* ""'
            if [ $? -eq 0 ]; then
                #replacing "" with "var"
                $SED -i --follow-symlinks '/single-user/s|/dev/mapper/\([^ ]*\) ""|/dev/mapper/\1 "console=ttyS0"|'  $grubFile
            else
                #replacing " with "var<space>
                $SED -i --follow-symlinks '/single-user/s|/dev/mapper/\([^ ]*\) "|/dev/mapper/\1 "console=ttyS0 |' $grubFile
            fi
        fi
    elif [ $consoleTtyS0 == "remove" ]; then
        $GREP sonus-curr-ver $grubFile | $GREP console=ttyS0
        if [ $? -eq 0 ]; then
            $SED -i --follow-symlinks '/single-user/s/console=ttyS0[ ]*//' $grubFile
            $SED -i --follow-symlinks '/sonus-curr-ver/s/console=ttyS0[ ]*//' $grubFile
        fi
    fi
}

modifyAppArmorSettings()
{   
    if [ $apparmor == "add" ]; then
        $ECHO 'GRUB_CMDLINE_LINUX_DEFAULT="$GRUB_CMDLINE_LINUX_DEFAULT apparmor=1 security=apparmor"' > $grubFile
    elif [ $apparmor == "remove" ]; then
        $ECHO 'GRUB_CMDLINE_LINUX_DEFAULT="$GRUB_CMDLINE_LINUX_DEFAULT apparmor=0"' > $grubFile
    fi
}

#### process arguments and validate them

while getopts "hbc:C:a:d:f:gi:k:P:qst:T:uv:" OPTION
do
   case $OPTION in
   h)
      usage
      exit 0
      ;;
   c)
      forceFsck="$OPTARG"
      if [[ $forceFsck != "add" && $forceFsck != "remove" ]]; then
          $ECHO "Invalid value '$forceFsck' for option 'c'"
          usage
          exit 1
      fi
      ;;
   C)
      consoleTtyS0="$OPTARG"
      if [[ $consoleTtyS0 != "add" && $consoleTtyS0 != "remove" ]]; then
          $ECHO "Invalid value '$consoleTtyS0' for option 'C'"
          usage
          exit 1
      fi
      ;;
   a)
      apparmor="$OPTARG"
      if [[ $apparmor != "add" && $apparmor != "remove" ]]; then
          $ECHO "Invalid value '$apparmor' for option 'a'"
          usage
          exit 1
      fi
      ;;

   d)
      device="$OPTARG"
      menuOptions=1
      ;;
   f)
      # using -e since could be a file or could be a link. we just want
      # to know if it exists.
      if [ -e "$OPTARG" ]; then
          grubFile=$OPTARG
      else
          $ECHO "** file $OPTARG does not exist!"
          usage
          exit 1
      fi
      ;;
   g)
      grubDefault=1
      ;;
   i)
      menuID="$OPTARG"
      menuOptions=1
      ;;
   k)
      kernelVersion="$OPTARG"
      menuOptions=1
      ;;
   P)
      previousKernelArgs="$OPTARG"
      skipKernelArgs=1
      menuOptions=1
      ;;
   q)
      quiet=1
      menuOptions=1
      ;;
   s)
      singleUser=1
      menuOptions=1
      ;;
   t)
      title="$OPTARG"
      menuOptions=1
      ;;
   T)
      timeStamp="$OPTARG"
      menuOptions=1
      ;;
   u)
      upgrade=1
      menuOptions=1
      ;;
   v)
      sbcVersion="$OPTARG"
      menuOptions=1
      ;;
   ?)
      $ECHO -e "$PROG: invalid argument specified\n"
      usage
      exit 1
      ;;
   esac
done

shift $(($OPTIND - 1))

if [[ $# -ne 0 ]]; then
    $ECHO "** extra arguments detected ($*)"
    usage
    exit 1
elif [[ $grubDefault -eq 1 && $menuOptions -eq 1 ]]; then
    $ECHO "** updating /etc/default/grub and adding a menu entry are mutually exclusive"
    usage
    exit 1
elif [[ ! -z "$forceFsck" && $grubDefault -eq 1 ]]; then
    $ECHO "** updating force fsck and adding /etc/default/grub are mutually exclusive"
    usage
    exit 1
elif [[ ! -z "$forceFsck" && $menuOptions -eq 1 ]]; then
    $ECHO "** updating force fsck and adding a menu entry are mutually exclusive"
    usage
    exit 1
elif [[ ! -z "$consoleTtyS0" && $grubDefault -eq 1 ]]; then
    $ECHO "** removing console=ttyS0 and adding /etc/default/grub are mutually exclusive"
    usage
    exit 1
elif [[ ! -z "$consoleTtyS0" && $menuOptions -eq 1 ]]; then
    $ECHO "** removing console=ttyS0 and adding a menu entry are mutually exclusive"
    usage
    exit 1
elif [[ ! -z "$apparmor" &&  $menuOptions -eq 1 ]]; then
    $ECHO "** updating apparmor and adding a menu entry are mutually exclusive"
    usage
    exit 1
elif [[ ! -z "$apparmor" && $grubDefault -eq 1 ]]; then
    $ECHO "** updating apparmor and adding /etc/default/grub are mutually exclusive"
    usage
    exit 1
elif [[ ! -z "$consoleTtyS0" && ! -z "$forceFsck" ]]; then
    $ECHO "** updating console=ttyS0 and updating force fsck are mutually exclusive"
    usage
    exit 1
elif [[ ! -z "$apparmor" && ! -z "$forceFsck" ]]; then
    $ECHO "** updating apparmor and updating force fsck are mutually exclusive"
    usage
    exit 1
elif [[ ! -z "$apparmor" && ! -z "$consoleTtyS0" ]]; then
    $ECHO "** updating apparmor and updating console=ttyS0 are mutually exclusive"
    usage
    exit 1
elif [ -z "$grubFile" ]; then
    $ECHO "** grub file must be specified"
    usage
    exit 1
elif [[ $grubDefault -eq 0 && -z "$forceFsck" && -z "$consoleTtyS0" && -z "$apparmor" ]]; then
    if [ -z "$kernelVersion" ]; then
        $ECHO "** kernelVersion must be specified"
        usage
        exit 1
    elif [ -z "$device" ]; then
        $ECHO "** root device must be specified"
        usage
        exit 1
    elif [ -z "$sbcVersion" ]; then
        $ECHO "** SBC version must be specified"
        usage
        exit 1
    elif [ -z "$menuID" ]; then
        $ECHO "** menu ID must be specified"
        usage
        exit 1
    elif [ -z "$title" ]; then
        $ECHO "** title must be specified"
        usage
        exit 1
    elif [ $# -ne 0 ]; then
        $ECHO "** extra arguments specified"
        usage
        exit 1
    fi
fi

# either update default file or add a menu entry to the custom menu file
# or modify fsck param or modify console=ttyS0
if [ $grubDefault -eq 1 ]; then
    updateGrubDefault
elif [ $menuOptions -eq 1 ]; then
    addSonusLinuxEntry
elif [ ! -z "$forceFsck" ]; then
    modifyFsckParam
elif [ ! -z "$consoleTtyS0" ]; then
    modifyConsoleTtyS0Param
elif [ ! -z "$apparmor" ]; then
    modifyAppArmorSettings
fi

exit 0
