#!/usr/bin/env python3

#================================================================================
# script : upgradeMop.py
# Usage  : upgradeMop.py <operation>
# Description: This script is invoked by VNFCs to perform MOP operations
#              as part of applyMop operaton from IaC, VNFR REST API is invoked
#              which will result in calling this script with appropriate arguments,
#              e.g. preUpgradeChecks, preUpgradeActions. This script looks for
#              directory name with operation and calls scripts in .list file
#              with in that directory.
#================================================================================
import os
import re
import sys
import time
import json
import subprocess
import argparse
import yaml

class ArgumentParserError(Exception): pass

class ThrowingArgumentParser(argparse.ArgumentParser):
    def error(self, message):
        raise ArgumentParserError(message)

def runOperation(operation, getTimeFlag):
    # Removing leading and trailing spaces.
    # So that we can use operation to get operation directory and operation list file
    operation.strip()

    operationDir         = os.path.abspath(os.path.dirname(os.path.realpath(__file__)) + '/' + operation) + '/'
    operationListFile    = operationDir+'/'+operation+'.list'
    operationCommandList = []
    mopCmdResArr         = []

    if not os.path.isdir(operationDir):
        return(False,"Directory for the operation does not exist : " + operation)

    if not os.path.exists(operationListFile):
        return(False,"Operation list file does not exist : " + operation)

    # Load the list yaml file to json
    with open(operationListFile) as fOperationListYaml:
        operationListYaml = yaml.safe_load(fOperationListYaml)

    operationListJson = json.loads(json.dumps(operationListYaml))

    if 'CommandList' in list(operationListJson.keys()):
        operationCommandList = operationListJson['CommandList']
    else:
        return False, ("commandList does not exists in " + operationListFile)

    totalTimeout = 0

    if not operationCommandList:
        commandResp               = {'command': 'n/a', 'retryafter': 'n/a'}
        commandResp['operation']  = 'n/a'
        commandResp['result']     = 'n/a'
        commandResp['reason']     = 'n/a'
        mopCmdResArr.append(commandResp)
        if getTimeFlag != True:
            return True, mopCmdResArr
        else:
            return False, mopCmdResArr

    for operation in operationCommandList:
        if 'timeout' in list(operation.keys()):
            totalTimeout = totalTimeout + int(operation['timeout'])
        else:
            if 'command' in list(operation.keys()):
                return False, ("timeout was not provided for operation : " + operation['command'])
            else:
                return False, ("One or more commands not formatted well in " + operationListFile)

    if getTimeFlag == True:
        return True, totalTimeout

    for command in operationCommandList:
        commandRespString   = ''
        commandErrorString  = ''
        commandResp         = {'command':command['command'], 'retryafter': '0'}
        commandToRun        = operationDir + command['command'].strip()

        if not os.path.isfile(commandToRun):
            commandToRun = command['command'].strip()

        try:
            subpro = subprocess.Popen(commandToRun.split(' '), stdout=subprocess.PIPE, stderr=subprocess.PIPE,encoding='utf-8')

            t = 0
            while t < int(command['timeout']) and subpro.poll() is None:
                time.sleep(1)  # (comment 1)
                t += 1
            if subpro.poll() is None:
                subpro.terminate()
                commandResp['operation']  = str(command['command'])
                commandResp['result']     = 'failure'
                commandResp['reason']     = 'timeout reach for command'
                mopCmdResArr.append(commandResp)
                return False, mopCmdResArr

            commandRespString, commandErrorString = subpro.communicate()

        except Exception as e:
            commandResp['operation'] = 'n/a'
            commandResp['result']    = 'failure'
            commandResp['reason']    = '\'' + str(e) + '\''
            mopCmdResArr.append(commandResp)
            return False, mopCmdResArr

        try:
            commandRespFinalString   = (commandRespString + commandErrorString).strip()
            responseLines = commandRespFinalString.splitlines()
            for lineNum in range(0, len(responseLines)):
                key = responseLines[lineNum].strip().split(' ')[0]
                keyCase = key.lower()
                if keyCase == 'operation:' or keyCase == 'result:' or keyCase == 'reason:' or keyCase == 'retryafter:':
                    commandResp[keyCase[:-1]] = responseLines[lineNum].split(key)[1].strip()
                if keyCase == 'reason:' and lineNum < len(responseLines)-1:
                    for x in range(lineNum+1, len(responseLines)):
                        firstWordLine = responseLines[x].strip().split(' ')[0]
                        if firstWordLine.lower() == 'operation:' or firstWordLine.lower() == 'result:' or firstWordLine.lower() == 'retryafter:':
                            break
                        commandResp[keyCase[:-1]] = commandResp[keyCase[:-1]] + '\n' + responseLines[x].strip()
                        lineNum = lineNum+1
            mopCmdResArr.append(commandResp)
            if commandErrorString != '' or commandResp['result'] == 'failure':
                return False, mopCmdResArr
        except Exception as e:
            print('An error occured : %s: %s' %(e.__class__, e))
            commandResp['operation']  = 'n/a'
            commandResp['result']     = 'failure'
            commandResp['reason']     = 'Response format error'
            mopCmdResArr.append(commandResp)
            return False, mopCmdResArr

    return True, mopCmdResArr

def parse_arguments():
    try:
        parser = ThrowingArgumentParser(
                    description='Runs specified upgrade mop operations')

        subparsers = parser.add_subparsers(dest = 'operation')

        #####################
        # Upgrade operations
        #####################
        preUpgradeChecksParser   = subparsers.add_parser('preUpgradeChecks')
        preUpgradeChecksParser.add_argument('-t', '--getTimeout', action='store_true', help='get total time for this operation')

        preUpgradeActionsParser  = subparsers.add_parser('preUpgradeActions')
        preUpgradeActionsParser.add_argument('-t', '--getTimeout', action='store_true', help='get total time for this operation')

        postUpgradeChecksParser  = subparsers.add_parser('postUpgradeChecks')
        postUpgradeChecksParser.add_argument('-t', '--getTimeout', action='store_true', help='get total time for this operation')

        postUpgradeActionsParser = subparsers.add_parser('postUpgradeActions')
        postUpgradeActionsParser.add_argument('-t', '--getTimeout', action='store_true', help='get total time for this operation')

        upgradeStartParser       = subparsers.add_parser('upgradeStart')
        upgradeStartParser.add_argument('-t', '--getTimeout', action='store_true', help='get total time for this operation')

        upgradeStopParser        = subparsers.add_parser('upgradeStop')
        upgradeStopParser.add_argument('-t', '--getTimeout', action='store_true', help='get total time for this operation')

        #####################
        # Revert operations
        #####################
        preRevertChecksParser   = subparsers.add_parser('preRevertChecks')
        preRevertChecksParser.add_argument('-t', '--getTimeout', action='store_true', help='get total time for this operation')

        preRevertActionsParser  = subparsers.add_parser('preRevertActions')
        preRevertActionsParser.add_argument('-t', '--getTimeout', action='store_true', help='get total time for this operation')

        postRevertChecksParser  = subparsers.add_parser('postRevertChecks')
        postRevertChecksParser.add_argument('-t', '--getTimeout', action='store_true', help='get total time for this operation')

        postRevertActionsParser = subparsers.add_parser('postRevertActions')
        postRevertActionsParser.add_argument('-t', '--getTimeout', action='store_true', help='get total time for this operation')

        revertStartParser       = subparsers.add_parser('revertStart')
        revertStartParser.add_argument('-t', '--getTimeout', action='store_true', help='get total time for this operation')

        revertStopParser        = subparsers.add_parser('revertStop')
        revertStopParser.add_argument('-t', '--getTimeout', action='store_true', help='get total time for this operation')

        return parser.parse_args()
    except Exception as e:
        print('%s' %(e))
        exit(1)

if __name__ == '__main__':
    args = parse_arguments()

    ret, mopApplyResultArr = runOperation(args.operation, args.getTimeout)
    response = { 'mopApplyResult' : mopApplyResultArr}
    print(json.dumps(response))
    if not ret:
        exit(1)
    exit(0)
