#!/bin/bash

if [ -e /opt/sonus/staging/sonusCommonUtils.sh ]
then
    source /opt/sonus/staging/sonusCommonUtils.sh
elif [ -e /opt/sonus/bin/sonusCommonUtils.sh ]
then 
    source /opt/sonus/bin/sonusCommonUtils.sh
else
    /bin/echo "Could not locate sonusCommonUtils.sh Exiting..."
    exit 1
fi
if [ -e /opt/sonus/staging/personality.sh ]
then
    source /opt/sonus/staging/personality.sh
elif [ -e /opt/sonus/sbx/scripts/personality.sh ]
then
    source /opt/sonus/sbx/scripts/personality.sh
else
    $ECHO "Could not locate personality.sh Exiting..."
    exit 1
fi

# Make sure we have proper ownership and permission staging
$CHOWN -R $USER:sonus $SONUS_STAGING_DIR/
$CHMOD 775 $SONUS_STAGING_DIR/*sh $SONUS_STAGING_DIR/*pl $SONUS_STAGING_DIR/*pm $SONUS_STAGING_DIR/*expect $SONUS_STAGING_DIR/*.out 2>/dev/null

# Globals
installRoot=$($DIRNAME $0)
exitRequired=0
index=0

# include sonusUtils - Assuming preInstallCheck.sh script is called from staging, 
# make sure sonusUtils.sh script exists in the same path.
# If the path is changed later, need to make sure if sonusUtils.sh is sourced
# from the right path.
. ${installRoot}/sonusUtils.sh
hostType=$(getHwType);

# Do not run peer server. By default preInstall checks will be run on peer as well.
localOnly=0

# checkOnly, do not create status files but give a dry report to UI or however asked.
checkOnly=0

# default config file if not specified
if [ -f $SBXCONF_FILE ]; then
   sonusConfigFile=$SBXCONF_FILE
else
   sonusConfigFile="$SBX_CONF"
fi

if [ -e $sonusConfigFile ];then
   ceName=`$GREP ceName $sonusConfigFile | $AWK -F "="  '{print $2}'`;
   peerCeName=`$GREP peerCeName $sonusConfigFile | $AWK -F "="  '{print $2}'`;
   role=`$GREP role $sonusConfigFile |$AWK -F "="  '{print $2}' |$AWK '{print $1}'`
   if [ "$role" == "1" ];then
      primaryCeName=$ceName
      secondaryCeName=$peerCeName
   else
      primaryCeName=$peerCeName
      secondaryCeName=$ceName
   fi
else
   ceName=connexip
   peerCeName=unknown
   primaryCeName=$ceName
   secondaryCeName=$peerCeName
fi

startTime=`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`
upgradeStartTime=`$DATE "+%FT%T%z"`
upgradeBaseDir="$SONUS_LOG_UPGRADE_DIR";
upgradeDir="$SONUS_LOG_UPGRADE_CURRENT_DIR";
upgradeCheckDir="$SONUS_LOG_UPGRADE_PRECHECKS_DIR";
preUpgradeCheckLogs="$PRE_UPGRADE_CHECK_LOG";
statusUpder=$STATUS_UPDATER_PL;
packageContents=$PACKAGE_CONTENTS;
sonusTmpDir=$SONUS_TMP_DIR;

if [ ! -d $upgradeCheckDir ]; then
    $MKDIR -p $upgradeCheckDir;
fi
# upgrade group not defined in older release
# give write permission to world.
# will be reverted after new OS is installed.
if upgradeGroupExists; then
    $CHMOD 770 $upgradeCheckDir;
else
    $CHMOD 777 $upgradeCheckDir;
fi

preUpgradeCheckMarker=$upgradeCheckDir/preUpgradeCheckStatus

# Create sonus tmp directory
sonusTmpDir="$SONUS_TMP_DIR"
if [ ! -d $sonusTmpDir ]; then
    $MKDIR -m 770 -p $sonusTmpDir
    $CHOWN sonusadmin:sonus $sonusTmpDir
fi

usage()
{
$CAT << EOF
usage: `$BASENAME $0` [-h] [-s] [-l] [-u] [-f sonusConfigFile]

OPTIONS:
   -h               Print this message.
   -s               Skip configuration file validation.
   -l               Run the checks only on local server.	
   -f               Specify an alternate configuration file.
   -u               LSWU, source sonusCommands.sh from staging.
EOF
}

lvcount=$($LVSCAN 2>/dev/null | $WC -l)

createUpgradeFile()
{
  stepNameTbl=("" PreUpgradeChecks DiskPartition ImageUpgrade BIOSUpgrade Reboot DBRestore AppBringUp PostUpgrade)

  stepUpgradeCmplTbl=("" ",EstTime=20,StartTime=$startTime,EndTime=$startTime,Status=complete,Reason=Succesful_Completion" ",EstTime=600,StartTime=none,EndTime=none,Status=none,Reason=none" ",EstTime=240,StartTime=none,EndTime=none,Status=none,Reason=none" ",EstTime=120,StartTime=none,EndTime=none,Status=none,Reason=none" ",EstTime=180,StartTime=none,EndTime=none,Status=none,Reason=none" ",EstTime=300,StartTime=none,EndTime=none,Status=none,Reason=none" ",EstTime=180,StartTime=none,EndTime=none,Status=none,Reason=none" ",EstTime=180,StartTime=none,EndTime=none,Status=none,Reason=none")

  local createCheckFile=$1
  stepNo=0
  stepIndex=1
  activeCeList=
  standbyCeList=

  if [ "$createCheckFile" == "true" ]; then
     outPutFile=$preUpgradeCheckMarker
     unset stepUpgradeCmplTbl
  else
     fileData="TargetRelease=$TargetRelease\nCurrentRelease=$CurrentRelease\nNumberOfSteps="
     postHeader="\nCurrentStep=1\nUpgradeStartTime=$upgradeStartTime\nUpgradeCompleteTime=none\nUpgradeStatus=none\nprimaryUpgradeStatus=$primaryUpgradeStatus\nsecondaryUpgradeStatus=$secondaryUpgradeStatus\nReason=none"
     outPutFile="$SOFTWARE_UPGRADE_STATUS"
  fi

  maxStepCount=`$ECHO ${#stepNameTbl[@]}`

  if [[ $lvcount -eq 3 || "$prodString" == "5100" || "$hostType" == "ConnexIP5000" ]];then
     unset stepNameTbl[2]
  fi

  # Remove the bios upgrade steps if the version are the same
  if [ "$TargetRelease" == "$CurrentRelease" ];then
     # remove BIOSUpgrade
     unset stepNameTbl[4]
  else
     biosUpdateNeeded
     if [ $? -eq 0 ];then
        # bios version is uptodate remove BIOSUpgrade
        unset stepNameTbl[4]
     fi
  fi

  # Remove DBRestore for the 5100 and the swe platform.
  if [[ "$prodString" == "5100" || "$hostType" == "ConnexIP5000" ]];then
     # remove DBRestore 
     unset stepNameTbl[6]
  fi

  activeStepOffset=`$ECHO ${#stepNameTbl[@]}`
  activeStepOffset=$(($activeStepOffset-1))

  while [ $stepIndex -lt $maxStepCount ]
     do
        currentStep=`$ECHO ${stepNameTbl[$stepIndex]}`
        if [ -n "$currentStep" ]; then
           currentCplt=`$ECHO ${stepUpgradeCmplTbl[$stepIndex]}`
           stepNo=`expr $stepNo + 1`
           activeCeList="$activeCeList\nceName=$activeCeName,StepNo=`expr $stepNo + $activeStepOffset`,Step=$currentStep$currentCplt"
           standbyCeList="$standbyCeList\nceName=$standbyCeName,StepNo=$stepNo,Step=$currentStep$currentCplt"
        fi
        stepIndex=`expr $stepIndex + 1`
     done
     if [ "$createCheckFile" == "true" ]; then
        $ECHO  -e "$standbyCeList $activeCeList" >> $outPutFile
        totalSteps=`expr $stepNo + $activeStepOffset`
        $SED -i "s/NumberOfSteps=.*/NumberOfSteps=$totalSteps/" $outPutFile
        $SED -i "/^\s*$/d" $outPutFile
     else
        $ECHO  -e "$fileData$(($stepNo + $activeStepOffset))$postHeader $standbyCeList $activeCeList" > $outPutFile
     fi
     if upgradeGroupExists; then
         $CHMOD 770 $outPutFile;
     else
         $CHMOD 777 $outPutFile;
     fi
}

skipConfigVal=0
# Below flag is not used, but keeping it since during LSWU SM in few version calls preChecks with -u option.
lswuFlag=0

while getopts "hslcf:u" OPTION
do
   case $OPTION in
   h)
      usage
      exit 1
      ;;
   s)
      skipConfigVal=1
      ;;
   l) localOnly=1
      ;; 
   c) # Install, PM upgrade and preChecks will call preInstallChecks with -c argument to make sure script doesnt create live upgrade status file.
      checkOnly=1
      ;; 
   f)
      sonusConfigFile="$OPTARG"
      ;;

   u)
      lswuFlag=1   
      ;;
   ?)
      usage
      exit 1
      ;;
   esac
done
commandArgs=$@
shift $(($OPTIND - 1))

# Function to update status files and exit
errorExit()
{
  local reason=`$ECHO $1 | $SED 's/__$//'`
  if [ $skipStatusFileCreation == "false" ];then
     $PERL $statusUpder -c -r preChecks -s "CEcheckStatus" -v "checkFailed"
     $PERL $statusUpder -c -r preChecks -s "EndTime" -v "`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`"
     if [ ! -z "$reason" ];then
        $PERL $statusUpder -c -s "Reason" -v "$reason";
        $PERL $statusUpder -c -r preChecks -s "Reason" -v "$reason";
     fi
     $PERL $statusUpder -c -s "CheckStatus" -v "checkFailed"
  fi
  if [ -e $packageContents ];then
      # Cleanup staging directory.
      currDir=`pwd`
      cd $SONUS_STAGING_DIR
      $RM -f `$CAT $packageContents`
      # Adding more files to be removed in case multiple installs/upgrades occur before starting SBX (like in case of ePSX).
      $RM -f sbc*.tar.gz *.md5 *.sha256 appInstall*.sh connexip*.deb ema*.deb sbc*.deb sbc*.signature
      $RM -f $packageContents
      cd $currDir
  fi

  # SM in few newer versions uses mktmp for log file. Find logFile and process it for proper o/p.
  preCheckOutFile=`$LS $SONUS_TMP_DIR/preInstallCheck.out* 2>&1`
  if [[ $preCheckOutFile == *"No such file or directory"* ]];then
      preCheckOutFile=`$LS /tmp/preInstallCheck.out* 2>&1`
      if [[ $preCheckOutFile == *"No such file or directory"* ]];then
          preCheckOutFile=$SONUS_STAGING_DIR/preInstallCheck.out
      fi
  fi
  $MV $preCheckOutFile $SONUS_STAGING_DIR/preInstallCheck.out.full

  # print only errors or warnings.
  $CAT $SONUS_STAGING_DIR/preInstallCheck.out.full | $EGREP 'ERROR|WARNING' > $preCheckOutFile
  $CAT $SONUS_STAGING_DIR/preInstallCheck.out.full | $EGREP 'ERROR|WARNING' > $SONUS_STAGING_DIR/preInstallCheck.out
  $CHOWN sonusadmin:sonus $preCheckOutFile 
  $CHMOD 760 $preCheckOutFile 
  $CHOWN sonusadmin:sonus $SONUS_STAGING_DIR/preInstallCheck.out*
  $CHMOD 760 $SONUS_STAGING_DIR/preInstallCheck.out*

  # echo "errorExit: Exiting preInstallChecks on error.." 
  $RM -f $upgradeBaseDir/preUpgradeCheckActive.key 
  preInstallLogMsg "Exiting preInstallChecks on error.."  "user"
  exit 1
}

if [[ $hostType != "ConnexIP5000" && $hostType != "SBC7000" && $hostType != "SBC5400" ]]; then
    preInstallLogMsg "================================================================" "user"
    preInstallLogMsg "= FATAL: This is an EOL Platform.                              =" "user"
    preInstallLogMsg "= FATAL: Not supported to install SBC version 11.0 or greater. =" "user"
    preInstallLogMsg "================================================================" "user"
    errorExit InstallOnEOLPlatform
fi

# We should create preUpgradeStatusFile only in case of LSWU or preChecks and not in case of PM install/Upgrade.

# If checkOnly=1 and if there is no active token, it is a case of PM Install/Upgrade. Because calling preCheck API from UMD always creates active token.
skipStatusFileCreation=false
if [[ $checkOnly == "1" && ! -e $upgradeBaseDir/preUpgradeCheckActive.key ]];then
   skipStatusFileCreation=true
   preUpgradeCheckLogs=$SONUS_STAGING_DIR/preUpgradeCheck.log
fi

$TOUCH $preUpgradeCheckLogs
$CHMOD 770 $preUpgradeCheckLogs

preInstallLogMsg "=================================="
preInstallLogMsg "$0 called with args($commandArgs)"
preInstallLogMsg "=================================="
preInstallLogMsg "skipStatusFileCreation=$skipStatusFileCreation"

$DPKG -l 2>/dev/null | $GREP -q "libexpect-perl"
if [ $? -eq 0 ]; then
   sonusPeerCnxExpectScript="$SONUS_STAGING_DIR/sonusPeerCnx.expect"
else
   sonusPeerCnxExpectScript="$SONUS_PEER_CNX_EXPECT"
fi
preInstallLogMsg "sonusPeerCnxExpectScript=$sonusPeerCnxExpectScript"

# When we switched from sysvinit to embracing systemd, we moved some scripts around.  the
# sbx script is checked during upgrade however prior to using new files.  For that reason,
# lets create a link between the new and old locations in case we are upgrading from
# a release with sysvinit support to a release with systemd support.
# note: the directory already exists as cps has always used it
if [[ -e "/etc/init.d/sbx" && ! -f $ETC_INITD_SBX ]]; then
    preInstallLogMsg "Upgrading from sysvinit to systemd: temporarily linking old and new script locations"
    $LN -s -f /etc/init.d/sbx $ETC_INITD_SBX
fi

# Skip preInstallChecks after standby upgrade in the middle of LSWU
if [ -e $SOFTWARE_UPGRADE_STATUS ];then
   upgradeStatus=$($CAT $SOFTWARE_UPGRADE_STATUS |$GREP ^UpgradeStatus |$AWK -F "=" '{print $2}')
   if [[ "$upgradeStatus" == "postStandbyUpgradeChecks" || "$upgradeStatus" == "upgradingStandbyDone" ]];then 
      preInstallLogMsg "Skipping preInstallChecks since we are in the middle of LSWU! Exiting.."  
      exit 0
   elif [[ $checkOnly == "0" ]];then
      # Need to check if this is a case of manual CLI upgrade after an failed upgrade.
      primaryUpgradeStatus=$($CAT $SOFTWARE_UPGRADE_STATUS |$GREP ^primaryUpgradeStatus |$AWK -F "=" '{print $2}')
      secondaryUpgradeStatus=$($CAT $SOFTWARE_UPGRADE_STATUS |$GREP ^secondaryUpgradeStatus |$AWK -F "=" '{print $2}')
      if [[ $primaryUpgradeStatus =~ .*upgraded.* && $secondaryUpgradeStatus =~ .*pendingUpgrade.* || $primaryUpgradeStatus =~ .*pendingUpgrade.* && $secondaryUpgradeStatus =~ .*upgraded.* || $primaryUpgradeStatus =~ .*upgrading.* && $secondaryUpgradeStatus =~ .*pendingUpgrade.* || $primaryUpgradeStatus =~ .*pendingUpgrade.* && $secondaryUpgradeStatus =~ .*upgrading.* || $primaryUpgradeStatus =~ .*reverted.* && $secondaryUpgradeStatus =~ .*upgraded.* || $primaryUpgradeStatus =~ .*upgraded.* && $secondaryUpgradeStatus =~ .*reverted.* ]];then
          preInstallLogMsg "previous UpgradeState is $upgradeStatus. primaryUpgradeStatus=$primaryUpgradeStatus , secondaryUpgradeStatus$secondaryUpgradeStatus. This is a manual recovery upgrade. Exiting preInstallChecks..."
          # Need to set upgradeStatus to something else. Or else PM and other presentation layers will think upgrade has failed. Also checkStatus should be made success to make SM happy.
          $PERL $statusUpder -c -s "CheckStatus" -v "checkComplete"
          $PERL $statusUpder -s "UpgradeStatus" -v "manualUpgrade"
          $PERL $statusUpder -s "Reason" -v "manualUpgradeInProgress"
          preInstallLogMsg "Removing preUpgradeCheckActive.key..."
          $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
          exit 0
      fi
   fi
fi

if [ $skipStatusFileCreation == "false" ];then
  # Get target and current release
  TargetRelease=$($GREP "SBC Version:" $STAGING_BUILDINFO | $AWK '{print $3}')
  isDebPkgMgmt
  if [ $? -ne 0 ]; then
    CurrentRelease=$($RPM -qa |$GREP sbc |$AWK -F "sbc-" '{print $2}' |$SED 's/.x86_64.*//')
  else
    CurrentRelease=$($DPKG -l sbc | $GREP ^ii | $AWK '{print $3}' | $SED 's/^/V/')
  fi
  # Create status file
  $ECHO -e "TargetRelease=$TargetRelease
CheckStartTime=`$DATE +'%a %b %e %H:%M:%S %Z %Y'`
CheckCompleteTime=none
CheckStatus=checkInProgress
NumberOfSteps=0
Reason=none
ceName=$ceName,CEcheckStatus=none,StartTime=none,EndTime=none,Reason=none
ceName=$peerCeName,CEcheckStatus=none,StartTime=none,EndTime=none,Reason=none " > $preUpgradeCheckMarker

  if upgradeGroupExists; then
      $CHMOD 770 $preUpgradeCheckMarker;
  else
      $CHMOD 777 $preUpgradeCheckMarker;
  fi

  # Update Check Status on current server '-c' arg to statusUpdater indicates preCheck status
  $PERL $statusUpder -c -r preChecks -s "CEcheckStatus" -v "checkInProgress"
  currTime=`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`;
  $PERL $statusUpder -c -r preChecks -s "StartTime" -v "$currTime"
fi

# This script is run as root (invoked with sudo) during install/upgrade. So it should always have read permission on external directory.
# Adding an extra check here just in case (to handle scenarios if ever run as a different user)..
if [ ! -r "$SONUS_EXTERNAL_DIR" ]; then
    userName=`$ID -un`
    preInstallLogMsg "ERROR: User $userName doesn't have read permission for $SONUS_EXTERNAL_DIR directory. Exiting..." "user"
    errorExit NoUserReadPermissionOnExternalDir
fi

# Check to ensure external directory group is upload and has read permission set.
groupReadBit=`$STAT -c %A $SONUS_EXTERNAL_DIR/ | $CUT -c 5`
groupName=`$STAT -c %G $SONUS_EXTERNAL_DIR/`
if [[ "$groupName" != "upload" || "$groupReadBit" != "r" ]]; then
    preInstallLogMsg "ERROR: Directory $SONUS_EXTERNAL_DIR does not have upload group read permission set. Exiting..." "user"
    errorExit NoUploadGroupReadPermissionOnExternalDir
fi

if [ $localOnly == "0" ]; then

    # cnxExt is set to { sh | expect } and the variable peerServer is set to { "" | "peer ip address" }
    peerCfg=1
    peerIpAddr=""
    cnxExt="expect"
    haPortName="bond0"

    peerCfg=$(checkIface $haPortName)
    if [ $peerCfg -ne 0 ]; then
       haPortName="ha0"
       peerCfg=$(checkIface $haPortName)
    fi

    if [ -e  "$sonusConfigFile" ]; then
       $CAT $sonusConfigFile | $GREP ^peerCeName | $GREP -q none$
       if [ $? -eq 0 ]; then
          # this is a stand alone system
          peerCfg=1
       fi
    fi

    if [ $peerCfg -eq 0 ]; then
       if [ -e  $SONUS_PEER_CNX_SH ]; then
          cnxExt="sh"
          tmpPeerServer=$(getIpAddr $haPortName)
          if [[ $tmpPeerServer == "169.254.99.1" ]]; then
             peerIpAddr="169.254.88.1"
          elif [[ $tmpPeerServer == "169.254.88.1" ]]; then
             peerIpAddr="169.254.99.1"
          else
             # could not resolve peer ip address
             peerCfg=1
          fi
       elif [ ! -e  $SONUS_PEER_CNX_EXPECT ]; then
          # could not find the peer connect script
          peerCfg=1
       fi
    fi
    preInstallLogMsg "peerCfg=$peerCfg haPortName=$haPortName cnxExt=$cnxExt peerIpAddr=$peerIpAddr"
    if [ $peerCfg -ne 0 ]; then
       preInstallLogMsg "ERROR: Could not find the peer connect info. Exiting..." "user"
       errorExit CouldNotFindPeerConnectInfoOn_${ceName}
    fi

    # Check if qcow2 image exists on current server and perform sha256 validation in case of LSWU/preUpgradeChecks.
    getReqOSVersion
    TargetQcow=`$ECHO $TargetRelease | $SED 's/-//'`
    qcow2ImageSHA256=`$LS $SONUS_STAGING_DIR/sbc-${TargetQcow}-connexip-os_${reqOSVersion}_*amd64.qcow2.sha256`
    qcow2Name=$($BASENAME `$ECHO $qcow2ImageSHA256 | $SED 's/.sha256//'`)
    qcow2SubDir=`$ECHO $qcow2Name | $SED 's/sbc-V/sbc_V/'`

    if [ -e $SONUS_EXTERNAL_DIR/$qcow2Name ]; then
       preInstallLogMsg "Image $qcow2Name exists in external directory...[INFO]"
       qcow2Image=$SONUS_EXTERNAL_DIR/$qcow2Name
    elif [ -e $SONUS_EXTERNAL_DIR/$qcow2SubDir/$qcow2Name ]; then
       preInstallLogMsg "Image $qcow2Name exists in subDir...[INFO]"
       qcow2Image=$SONUS_EXTERNAL_DIR/$qcow2SubDir/$qcow2Name
    else
       $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
       preInstallLogMsg "ERROR: TargetQcow2Image=$qcow2Name doesnt exist on $ceName . Please upload the qcow2 image on both servers before starting LSWU." "user"
       array_exitReason[index++]=TargetQcow2ImageFileDoesntExistsOn_${ceName}
       exitRequired=1
    fi

    # Exit if the qcow image is missing on current server.
    if [ $exitRequired == "1" ]; then
       exitreason=$(printf '%s__' "${array_exitReason[@]}")
       errorExit $exitreason
    fi
    
    if [ $($CAT $qcow2ImageSHA256 | $AWK '{print $1}'| $TR -d '[[:space:]]') != $($SHA256SUM $qcow2Image | $AWK '{print $1}'| $TR -d '[[:space:]]') ];then
       $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
       preInstallLogMsg "ERROR: qcow2 image sha256sum checksum verification failed on $ceName !.. Please manually check the integrity of the qcow2 image you have uploaded." "user"
       array_exitReason[index++]=TargetQcow2ImageSHA256ChecksumValidationFailedOn_${ceName}
       exitRequired=1
    else
       preInstallLogMsg "qcow2 image sha256sum checksum verification is successfull on $ceName"
    fi

    # get package location on current server.
    if [ -e $SONUS_EXTERNAL_DIR/sbc-${TargetRelease}.x86_64.tar.gz ];then
       package=$SONUS_EXTERNAL_DIR/sbc-${TargetRelease}.x86_64.tar.gz
    elif [ -e $SONUS_EXTERNAL_DIR/sbc_${TargetRelease}/sbc-${TargetRelease}.x86_64.tar.gz ];then
       package=$SONUS_EXTERNAL_DIR/sbc_${TargetRelease}/sbc-${TargetRelease}.x86_64.tar.gz
    else
       $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
       preInstallLogMsg "ERROR: TargetPackage=sbc-${TargetRelease}.x86_64.tar.gz doesnt exist on $ceName . Please upload the packages on both servers before starting LSWU." "user"
       array_exitReason[index++]=TargetPackageDoesntExistsOn_${ceName}
       exitRequired=1
    fi

    checksum="md5";
    CHECKSUM=$MD5SUM;
    if [ -e $SONUS_EXTERNAL_DIR/sbc_${TargetRelease}/sbc-${TargetRelease}.x86_64.sha256 ]; then
        checksum="sha256";
        CHECKSUM=$SHA256SUM;
    fi

    preInstallLogMsg "Using ${checksum} as checksum [INFO]"
    
    if [ -e $SONUS_EXTERNAL_DIR/sbc-${TargetRelease}.x86_64.${checksum} ];then
       packageChecksum=$SONUS_EXTERNAL_DIR/sbc-${TargetRelease}.x86_64.${checksum}
    elif [ -e $SONUS_EXTERNAL_DIR/sbc_${TargetRelease}/sbc-${TargetRelease}.x86_64.${checksum} ];then
       packageChecksum=$SONUS_EXTERNAL_DIR/sbc_${TargetRelease}/sbc-${TargetRelease}.x86_64.${checksum}
    else
       $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
       preInstallLogMsg "ERROR: TargetPackageChecksum=sbc-${TargetRelease}.x86_64.(md5|sha256) doesnt exist on $ceName . Please upload package md5/sha256 files on both servers before starting LSWU." "user"
       array_exitReason[index++]=TargetPackageChecksumFileDoesntExistsOn_${ceName}
       exitRequired=1
    fi
    if [ -e $SONUS_EXTERNAL_DIR/sbc-${TargetRelease}.x86_64.signature ];then
       packageSignature=$SONUS_EXTERNAL_DIR/sbc-${TargetRelease}.x86_64.signature
    elif [ -e $SONUS_EXTERNAL_DIR/sbc_${TargetRelease}/sbc-${TargetRelease}.x86_64.signature ];then
       packageSignature=$SONUS_EXTERNAL_DIR/sbc_${TargetRelease}/sbc-${TargetRelease}.x86_64.signature
    else
       $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
       preInstallLogMsg "ERROR: TargetPackageSignature=sbc-${TargetRelease}.x86_64.signature doesnt exist on $ceName . Please upload package signature files on both servers before starting LSWU." "user"
       array_exitReason[index++]=TargetPackageSignatureFileDoesntExistsOn_${ceName}
       exitRequired=1
    fi

    # Check if package exists on peer. Extract the package.
    peerPackage=$($sonusPeerCnxExpectScript $peerIpAddr $LS $SONUS_EXTERNAL_DIR/sbc-${TargetRelease}.x86_64.tar.gz)
    peerPackageChecksum=$($sonusPeerCnxExpectScript $peerIpAddr $LS $SONUS_EXTERNAL_DIR/sbc-${TargetRelease}.x86_64.${checksum})
    peerPackageSignature=$($sonusPeerCnxExpectScript $peerIpAddr $LS $SONUS_EXTERNAL_DIR/sbc-${TargetRelease}.x86_64.signature)
    if [[ $peerPackage == *"No such file or directory"* ]];then
       # Check subDir
       peerPackage=$($sonusPeerCnxExpectScript $peerIpAddr $LS $SONUS_EXTERNAL_DIR/sbc_${TargetRelease}/sbc-${TargetRelease}.x86_64.tar.gz)
       peerPackageChecksum=$($sonusPeerCnxExpectScript $peerIpAddr $LS $SONUS_EXTERNAL_DIR/sbc_${TargetRelease}/sbc-${TargetRelease}.x86_64.${checksum})
       if [[ $peerPackage == *"No such file or directory"* ]];then 
          $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key 
          preInstallLogMsg "ERROR: TargetPackage=sbc-${TargetRelease}.x86_64.tar.gz doesnt exist on $peerCeName . Please upload the packages on both servers before starting LSWU."  "user"
          array_exitReason[index++]=TargetPackageDoesntExistsOn_${peerCeName}
          exitRequired=1
       elif [[ $peerPackageChecksum == *"No such file or directory"* ]];then 
          $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key 
          preInstallLogMsg "ERROR: TargetPackageChecksum=sbc-${TargetRelease}.x86_64.(md5|sha256) doesnt exist on $peerCeName . Please upload package md5/sha256 files on both servers before starting LSWU."  "user"
          array_exitReason[index++]=TargetPackageChecksumFileDoesntExistsOn_${peerCeName}
          exitRequired=1
       fi
    elif [[ $peerPackageChecksum == *"No such file or directory"* ]];then 
       $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key 
       preInstallLogMsg "ERROR: TargetPackageChecksum=sbc-${TargetRelease}.x86_64.(md5|sha256) doesnt exist on $peerCeName . Please upload package md5/sha256 files on both servers before starting LSWU."  "user"
       array_exitReason[index++]=TargetPackageChecksumFileDoesntExistsOn_${peerCeName}
       exitRequired=1
    fi
    if [[ $peerPackageSignature == *"No such file or directory"* ]];then
       # Signature file can be in either external dir or subDir. It depends on which version we are upgrading from. 
       peerPackageSignature=$($sonusPeerCnxExpectScript $peerIpAddr $LS $SONUS_EXTERNAL_DIR/sbc_${TargetRelease}/sbc-${TargetRelease}.x86_64.signature)
       if [[ $peerPackageSignature == *"No such file or directory"* ]];then
           $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
           preInstallLogMsg "ERROR: TargetPackageSignature=sbc-${TargetRelease}.x86_64.signature doesnt exist on $peerCeName . Please upload package signature files on both servers before starting LSWU." "user"
           array_exitReason[index++]=TargetPackageSignatureFileDoesntExistsOn_${peerCeName}
           exitRequired=1
       fi
    fi

    # Exit if any one the packages sbc-*.tar.gz/.${checksum}/.signature is missing on current/peer server.
    if [ $exitRequired == "1" ]; then
       exitreason=$(printf '%s__' "${array_exitReason[@]}")
       errorExit $exitreason
    fi

    # Remove any trailing spaces
    peerPackage=`$ECHO $peerPackage | $TR -d '[[:space:]]'`
    peerPackageChecksum=`$ECHO $peerPackageChecksum | $TR -d '[[:space:]]'`
    peerPackageSignature=`$ECHO $peerPackageSignature | $TR -d '[[:space:]]'`

    # Validate ${checksum} and signature files are same on both servers.
    if [[ $($CAT $packageChecksum |$AWK '{print $1}' |$TR -d '[[:space:]]') != $($sonusPeerCnxExpectScript $peerIpAddr $CAT $peerPackageChecksum |$AWK '{print $1}'|$TR -d '[[:space:]]') ]];then
        $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
        preInstallLogMsg "packageChecksum=$packageChecksum peerPackageChecksum=$peerPackageChecksum" "user"
        preInstallLogMsg "ERROR: TargetPackageChecksums are different on active and standby.. Please check the packages you have uploaded." "user"
        array_exitReason[index++]=TargetPackageChecksumMisMatchOnActiveAndStandby
        exitRequired=1
    fi
    if [[ $($SHA256SUM $packageSignature |$AWK '{print $1}' |$TR -d '[[:space:]]') != $($sonusPeerCnxExpectScript $peerIpAddr $SHA256SUM $peerPackageSignature |$AWK '{print $1}'|$TR -d '[[:space:]]') ]];then
        $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
        preInstallLogMsg "ERROR: TargetPackageSignatures are different on active and standby.. Please check the packages and signature files  you have uploaded." "user"
        array_exitReason[index++]=TargetPackageSignatureMisMatchOnActiveAndStandby
        exitRequired=1
    fi
    
    # preChecks should perform ${checksum} validation of packages. 
    if [ "$checkOnly" == "1" ];then
       if [ $($CAT $packageChecksum |$AWK '{print $1}') != $($CHECKSUM $package |$AWK '{print $1}') ];then
          $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
          preInstallLogMsg "ERROR: Checksum verification failed on $ceName !.. Please manually check the integrity of the package you have uploaded." "user"
          array_exitReason[index++]=TargetPackageSHA256ChecksumValidationFailedOn_${ceName}
          exitRequired=1
       else
          preInstallLogMsg "Package Checksum checksum verification is successfull on $ceName"
       fi 
    fi
    if [[ $($sonusPeerCnxExpectScript $peerIpAddr $CAT $peerPackageChecksum |$AWK '{print $1}'|tr -d '[[:space:]]') != $($sonusPeerCnxExpectScript $peerIpAddr $CHECKSUM $peerPackage | $AWK '{print $1}'|$TR -d '[[:space:]]') ]];then
       $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
       preInstallLogMsg "ERROR: Checksum checksum verification failed on $peerCeName !.. Please manually check the integrity of the package you have uploaded." "user"
       array_exitReason[index++]=TargetPackageSHA256ChecksumValidationFailedOn_${peerCeName}
       exitRequired=1
    else
       preInstallLogMsg "Package Checksum checksum verification is successfull on $peerCeName"
    fi

    # preChecks should perform versionCheck
    if [ "$checkOnly" == "1" ];then
       isDebPkgMgmt
       if [ $? -ne 0 ]; then
         currVer=$(rpm -qa |grep sbc |$AWK -F "sbc-" '{print $2}' |sed 's/.x86_64.*//' | sed 's/-//')
       else
         currVer=$($DPKG -l sbc | $GREP ^ii | $AWK '{print $3}' | $SED 's/^/V/' | sed 's/-//')
       fi
       newVer=$($GREP "SBC Version:" $STAGING_BUILDINFO | $AWK '{print $3}' | sed 's/-//') 
       # Make use of the base version checkLswu binary, since new version will have compiler incompatibilities.
       if [ -f /opt/sonus/installUpgrade/checkLswu ]; then
          checkLswuBinary=/opt/sonus/installUpgrade/checkLswu
       else
          checkLswuBinary=/opt/sonus/checkLswu
       fi
       $checkLswuBinary -f $currVer -t $newVer -m $SONUS_STAGING_DIR > /dev/null 2>&1
       if [ $? != "0" ]; then
          preInstallLogMsg "ERROR: versionCheck failed..LSWU not supported, currVer: $currVer, newVer: $newVer" "user"
          array_exitReason[index++]=TargetPackageVersionIncompatibleOnActiveAndStandby
          exitRequired=1
       else
          preInstallLogMsg "versionCheck is successfull,current version: $currVer, new version: $newVer" "user"
       fi
    fi
    
    $sonusPeerCnxExpectScript $peerIpAddr $MKDIR -p $upgradeBaseDir

    # upgrade group not defined in older release
    # give write permission to world.
    # will be reverted after new OS is installed.
    peerUpgradeGroup=$($sonusPeerCnxExpectScript $peerIpAddr $GREP upgrade /etc/group |$TR -d '[[:space:]]')
    if [ -z $peerUpgradeGroup ]; then
        $sonusPeerCnxExpectScript $peerIpAddr $CHMOD 777 $upgradeBaseDir
    else
        $sonusPeerCnxExpectScript $peerIpAddr $CHMOD 770 $upgradeBaseDir
    fi

    $sonusPeerCnxExpectScript $peerIpAddr $TOUCH $upgradeBaseDir/preUpgradeCheckActive.key 2>/dev/null

    preInstallLogMsg "Extracting package on $peerCeName.."  
    if [ -e $SONUS_MODIFY_STAGING_SH ];then
       preInstallLogMsg "Using sonusModifyStaging.."
       $sonusPeerCnxExpectScript $peerIpAddr $SUDO $SONUS_MODIFY_STAGING_SH untar $peerPackage
       $sonusPeerCnxExpectScript $peerIpAddr $SUDO $SONUS_MODIFY_STAGING_SH chmodUM 777
    else
       $sonusPeerCnxExpectScript $peerIpAddr $TAR -xzf $peerPackage -C $SONUS_STAGING_DIR/
       $sonusPeerCnxExpectScript $peerIpAddr $CHMOD 777 $statusUpder
       $sonusPeerCnxExpectScript $peerIpAddr $CHMOD 777 $SONUS_STAGING_DIR/sbxCommonFunctions.pm
    fi

    result=$($sonusPeerCnxExpectScript $peerIpAddr $LS $0)
    if [[ $result == *"No such file or directory"* ]];then
       $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
       preInstallLogMsg "ERROR: Could not extract PreInstallCheckScript on $peerCeName" "user"
       array_exitReason[index++]=CouldNotExtractPreInstallCheckScriptOn_${peerCeName}
       exitRequired=1
    fi
    # Check if qcow2 image exists on peer.
    peerQcow2ImageSHA256=$($sonusPeerCnxExpectScript $peerIpAddr $LS $SONUS_STAGING_DIR/sbc-${TargetQcow}-connexip-os_${reqOSVersion}_*amd64.qcow2.sha256)
    peerQcow2ImageSHA256=`$ECHO $peerQcow2ImageSHA256 | $TR -d '[[:space:]]'`
    peerQcow2Name=$($BASENAME `$ECHO $peerQcow2ImageSHA256 | $SED 's/.sha256//'`)
    peerQcow2SubDir=`$ECHO $peerQcow2Name | $SED 's/sbc-V/sbc_V/'`

    peerQcow2Image=$($sonusPeerCnxExpectScript $peerIpAddr $LS $SONUS_EXTERNAL_DIR/$peerQcow2Name)
    if [[ $peerQcow2Image == *"No such file or directory"* ]];then
       preInstallLogMsg "Image $peerQcow2Name doesn't exist in external on $peerCeName.Checking subDir..." "user"
       # Check subDir
       peerQcow2Image=$($sonusPeerCnxExpectScript $peerIpAddr $LS $SONUS_EXTERNAL_DIR/$peerQcow2SubDir/$peerQcow2Name)
       if [[ $peerQcow2Image == *"No such file or directory"* ]];then
          $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
          preInstallLogMsg "ERROR: TargetQcow2Image=$peerQcow2Name doesnt exist on $peerCeName . Please upload the qcow2 image on both servers before starting LSWU."  "user"
          array_exitReason[index++]=TargetQcow2ImageFileDoesntExistsOn_${peerCeName}
          exitRequired=1
       fi
    fi

    # Exit if the qcow image is missing on peer.
    if [ $exitRequired == "1" ]; then
       exitreason=$(printf '%s__' "${array_exitReason[@]}")
       errorExit $exitreason
    fi

    # Remove any trailing spaces
    peerQcow2Image=`$ECHO $peerQcow2Image | $TR -d '[[:space:]]'`

    # Perform qcow2 image sha256 validation on peer
    if [[ $($sonusPeerCnxExpectScript $peerIpAddr $CAT $peerQcow2ImageSHA256 |$AWK '{print $1}'|$TR -d '[[:space:]]') != $($sonusPeerCnxExpectScript $peerIpAddr /usr/bin/sha256sum $peerQcow2Image |$AWK '{print $1}'|$TR -d '[[:space:]]') ]];then
       $sonusPeerCnxExpectScript $peerIpAddr /bin/rm -f $upgradeBaseDir/preUpgradeCheckActive.key
       preInstallLogMsg "ERROR: qcow2 image sha256sum checksum verification failed on $peerCeName !.. Please manually check the integrity of the qcow2 image you have uploaded." "user"
       array_exitReason[index++]=TargetQcow2ImageSHA256ChecksumValidationFailedOn_${peerCeName}
       exitRequired=1
    else
       preInstallLogMsg "qcow2 image sha256sum checksum verification is successfull on $peerCeName"
    fi

    # Validate qcow2 sha256 files are same on both servers.
    if [[ $($CAT $qcow2ImageSHA256 |$AWK '{print $1}' |$TR -d '[[:space:]]') != $($sonusPeerCnxExpectScript $peerIpAddr $CAT $peerQcow2ImageSHA256 |$AWK '{print $1}'|$TR -d '[[:space:]]') ]];then
        $sonusPeerCnxExpectScript $peerIpAddr $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
        preInstallLogMsg "ERROR: TargetQcow2ImageSHA256s are different on active and standby.. Please check the qcow2 images you have uploaded." "user"
        array_exitReason[index++]=TargetQcow2ImageSHA256MisMatchOnActiveAndStandby
        exitRequired=1
    else
        preInstallLogMsg "qcow2 SHA256 files are same on both servers..."
    fi

    if [ $exitRequired == "1" ]; then
       exitreason=$(printf '%s__' "${array_exitReason[@]}")
       errorExit $exitreason
    else
       # call preInstall Checks on peer , call it asynchronously to make sure both scripts run independently. Status of checks can be determined from the status file.
       preInstallLogMsg "Calling $0 on peer..Command is=$sonusPeerCnxExpectScript $peerIpAddr $SUDO $0 $commandArgs -l" # sudoers-cnxipmadmin
       ($NOHUP $sonusPeerCnxExpectScript $peerIpAddr $SUDO $0 $commandArgs -l )& # sudoers-cnxipmadmin
       $SLEEP 2
    fi
fi

if [ $skipStatusFileCreation == "false" ];then
   # Update CEcheckStatus again to make sure it is updated on both sides.
   $PERL $statusUpder -c -r preChecks -s "CEcheckStatus" -v "checkInProgress"
   $PERL $statusUpder -c -r preChecks -s "StartTime" -v "$currTime"
fi

getConnexIPOSVersion
if [[ "$connexIPOSVersion" < "02.00.00" ]]; then
   preInstallLogMsg "ERROR: Direct upgrade from SBC 3.0.x to SBC 5.x is NOT a supported upgrade path.\nPlease first upgrade to SBC 3.1.x or higher release, then upgrade to SBC 5.x.  Exiting installation..."  "user"
   array_exitReason[index++]=unsupportedUpgradePathOn_${ceName}
   exitRequired=1
fi

# Verify the OS version before installation
IFS='.' read -a majorOSversion <<<"${connexIPOSVersion}"

if [[ "${majorOSversion[0]}" < "03" && ! -e $ETC_INITD_SBX ]]; then 
   preInstallLogMsg "ERROR: SBC Installation on current OS version is not supported. Kindly ISO with SBC 5.x and continue the installation. Exiting installation..."  "user"
   array_exitReason[index++]=incompatibleOSFoundOn_${ceName}
   exitRequired=1
fi

# Check dmesg for FS errors
timeDiff=1800    #30 Mins
if [ -e $DMESG_ANALYZER_SH ]; then
    if [ -e $DMESG_ERROR_MARKER ]; then
        curTime=$($CAT /proc/uptime | $AWK '{print int($1)}')
        latestFsErr=$($EGREP -i "EXT.-fs error" $DMESG_ERROR_MARKER | $TAIL -1)    # Only need last FS error message
        if [ ! -z "$latestFsErr" ]; then
            fsErrTs=$($ECHO $latestFsErr | $AWK -F'[][]' '{print int($2)}')
            if [ $((curTime - fsErrTs)) -le $timeDiff ]; then
                preInstallLogMsg "ERROR: File System errors detected in dmesg in the last 30 minutes. Exiting installation..." "user"
                array_exitReason[index++]=fsErrorsFoundOn_${ceName}
                exitRequired=1
            fi
        fi
    fi
else   # Handle upgrading from pre 10.0.0
    log="/var/log/kern.log"
    curTime=$($DATE +%s)
    fsErrFound=0
    while read logFile && [ $fsErrFound -eq 0 ]; do
        while read res && [ $fsErrFound -eq 0 ]; do
            fsErrTs=$($DATE -d $($ECHO $res | $AWK '{print $3}') +%s)
            if [ $((curTime - fsErrTs)) -le $timeDiff ]; then
                preInstallLogMsg "ERROR: File System errors detected in kernel log in the last 30 minutes. Exiting installation..." "user"
                array_exitReason[index++]=fsErrorsFoundOn_${ceName}
                exitRequired=1
                fsErrFound=1
            else
                # We don't need to worry about older logs
                break
            fi
        done < <($EGREP -i "EXT.-fs error" $logFile | $TAC)
    done < <($LS $log* | $GREP -v gz)
fi

# Remove packages from earlier release to reclaim some space
pushd $SONUS_DIR > /dev/null 2>&1
$RM -fr *.rpm *.deb *.md5 *.sha256 os_up*
popd > /dev/null 2>&1

export preUpgradeCheckLogs
export ceName
export peerCeName

# Verify we have enough disk space available (per our desired limits)
$installRoot/checkDiskUsage.sh
result=${PIPESTATUS[0]}
if [ $result -ne 0 ]; then
   # note: the script outputs the error, no need to say anything here...
   # sbxLinuxCmd traps the output error...
   array_exitReason[index++]=diskUsagecheckFailedOn_${ceName}
   exitRequired=1
fi

# perform appPreUpradeChecks from active server if app is installed.
if [ -d "$SONUS_SBX_DIR/" ]; then
   getCurrentRole
   if [ "$currentRole" == "active" ];then
      $installRoot/appPreUpgradeChecks.sh
      result=${PIPESTATUS[0]}
      if [ $result -ne 0 ]; then
         # note: the script outputs the error, no need to say anything here... 
         array_exitReason[index++]=appPreUpgradeChecksFailedOn$ceName
         exitRequired=1
      fi
   else
      preInstallLogMsg "Skipping appPreUpradeChecks since currentRole is not active."
   fi
else
   preInstallLogMsg "Skipping appPreUpradeChecks since app is not installed."
fi


# Verify if the temporary OS will fit in sda7
if [ $lvcount -ne 3 ]; then
   if [[ "$hostType" != "ConnexIP5000" && $prodString != "5100"  ]]; then
      sda7Size=$($GREP -e sda7 /proc/partitions | $AWK '{print $3}')
      rootFiles=$($DU -cs `$LS -d /*|$EGREP -v "proc|root|tmp|lost|home|opt|var|usr|*.log" ` | $AWK '{print $1 }' | $TAIL -1)
      usrFiles=$($DU -cs `$LS -d /usr/* | $EGREP -v "/usr/share"` | $AWK '{print $1 }' | $TAIL -1)
      varFiles=$($DU -cs `$LS -d /var/* | $EGREP -v "/var/lib|/var/log"` | $AWK '{print $1 }' | $TAIL -1)
      varLibFiles=$($DU -cs `$LS -d /var/lib/* | $EGREP -v "/var/lib/libvirt"` | $AWK '{print $1 }' | $TAIL -1)
      osSize=` $EXPR $rootFiles + $usrFiles + $varFiles + $varLibFiles ` 

      if [ $osSize -gt $sda7Size ]; then 
         preInstallLogMsg "ERROR! Limited swap space available. Not able to support upgrade.."  "user"
         array_exitReason[index++]=limitedSwapSpaceFoundOn_${ceName}
         exitRequired=1
      fi
  
      # Calculate size of root(/) that we'll copy to root partition
      # new root partitions (P1/P2) are 10GB, so reject the command if 
      # contents is > 8GB 
      rootSize=$($DU -cs `$LS -d /* | $EGREP -v "/home|/opt|/var|/tmp|/proc"` | $AWK '{print $1 }' | $TAIL -1)
      optSonusSize=$($DU -cs `$LS -d $SONUS_DIR/* | $EGREP -v "$SONUS_EXTERNAL_DIR|$SONUS_STAGING_DIR"` | $AWK '{print $1 }' | $TAIL -1)
      totalSize=` $EXPR $rootSize + $optSonusSize + $varFiles + $varLibFiles `

      if [ $totalSize -gt 8388608 ]; then 
          varFileTotalSize=`$EXPR $varFiles + $varLibFiles`
          preInstallLogMsg "ERROR: Contents to copy to root partition is too large . Please clean up /($rootSize KB), $SONUS_DIR($optSonusSize KB), /var($varFileTotalSize KB)" "user"
          array_exitReason[index++]=Contents_to_copy_to_root_partition_is_too_largeOn_${ceName}
          exitRequired=1
      else
          preInstallLogMsg "Calculated size of root partition to copy: $totalSize KB"  "user"
      fi 
   fi
fi

# Check the /home size
if [[ "$hostType" != "ConnexIP5000" && $prodString != "5100"  ]]; then
   # this is a preinstall script so /var/log/sonus/tmp may not exist. if not then create it
   if [ ! -d  $sonusTmpDir ]
   then
      $MKDIR -m 770 -p $sonusTmpDir
      $CHOWN sonusadmin:sonus $sonusTmpDir
   fi
   hSz=$($MKTEMP --tmpdir=$SONUS_TMP_DIR)
   pushd /home > /dev/null 2>&1
   $DU -s `$LS -d * | $EGREP -v "ora|common|libvirt|log|external|staging|sftproot|postgres"` | $AWK '{print $1}' > $hSz
   file=$($AWK '{ sum += $1 } END { print sum }' $hSz)
   if [ $file -gt 1000000 ]; then 
       preInstallLogMsg "ERROR: /home directory is too large . Please clean up the following files . " "user"
       $DU -s `$LS -d * | $EGREP -v "ora|common|libvirt|log|external|staging|sftproot|postgres"`  > $hSz
       largeFiles=`$CAT $hSz | $AWK '{ if ($1 >500000)  {print $1, $2}} '`
       preInstallLogMsg "ERROR: $largeFiles" "user"
       popd > /dev/null 2>&1
       array_exitReason[index++]=homeDirTooLargeOn_${ceName}
       exitRequired=1
   fi 
   popd > /dev/null 2>&1
   $RM -f $hSz
fi

# Skip disk speed check in case of LSWU as it is included in the upgrade script
if [ $skipStatusFileCreation == "true" ];then
  if [[ "$hostType" != "ConnexIP5000" ]]; then
    # When drbd sync is under progress the disk read performance will fall below desired threshold.
    # Hence we pause drbd sync when drbd connection status is not in 'Connected' state
    # assuming sync is under progress.
    # During upgrade case drbdadm would fail, hence checking for /proc/drbd
    syncFlag=0
    if [[ -e "/proc/drbd" ]]; then
      if [[ `$DRBDADM cstate mirror` != "Connected" ]]; then 
        syncFlag=1 
        $DRBDADM pause-sync mirror > /dev/null 2>&1
        $SLEEP 1
      fi
    fi

    # Verify we have good enough disk read performance (per our desired threshold)
    $installRoot/checkDiskPerformance.sh |$TEE -a $preUpgradeCheckLogs
    result=${PIPESTATUS[0]}
    if [ $result -ne 0 ]; then
      # resume sync if paused
      if [[ $syncFlag == 1 ]]; then
        $DRBDADM resume-sync mirror > /dev/null 2>&1
      fi
      # note: the script outputs the error, no need to say anything here...
      # sbxLinuxCmd traps the output error...
      array_exitReason[index++]=diskPerformanceTestFailedOn_${ceName}
      exitRequired=1
    fi

    # resume sync if paused
    if [[ $syncFlag == 1 ]]; then
      $DRBDADM resume-sync mirror > /dev/null 2>&1
    fi
  fi
fi

# Make sure that the minimum required platform components are there(OS/BMC/BIOS etc).
checkPlatformMinVersions ${installRoot}/preInstallCheck.result $skipStatusFileCreation $checkOnly

if [[ "$hostType" != "ConnexIP5000" ]]; then
  # verify that we have a valid base MAC address
  # note: upcase the values for comparison
  mac=`$IPMITOOL lan print | $GREP "^MAC Address"  | $AWK  '{print toupper($4)}'`
  if [ "$mac" == "00:00:00:00:00:00" ]; then
    preInstallLogMsg "ERROR: Base MAC address $mac is invalid; Exiting..."  "user"
    array_exitReason[index++]=invalidBaseMACAddressFoundOn_${ceName}
    exitRequired=1
  fi
fi

# Temporary Host Check functionality. Check
#    No. of available vCPUs = 2 and type = Intel
#    Min vRAM size = 5 GB
#    Min. hard disk size on VM = 65 GB
#    Min. no. of vNICs = 4
$CHMOD +x ${installRoot}/HostCheck
if [[ "$hostType" == "ConnexIP5000" ]]; then
   cpu_model=`$CAT /proc/cpuinfo | $GREP 'model' |$GREP -v 'model name' | $GREP -Eo '[0-9]{1,4}' | $UNIQ`
   cpu_family=`$CAT /proc/cpuinfo | $GREP 'family' | $GREP -Eo '[0-9]{1,4}' | $UNIQ`
   cpu_vendor=`$CAT /proc/cpuinfo | $GREP 'vendor_id' | $AWK '{print $3}' | $UNIQ`
   if [ "$cpu_vendor" == "GenuineIntel" ] && [ "$cpu_family" -eq 6 ] && [ "$cpu_model" -ge 26 ]; then
      $ECHO "Installing SWe on Intel, as the processor is suitable family..."
   elif [ "$cpu_vendor" == "AuthenticAMD" ] && [ "$cpu_family" -ge 21 ] ; then
      $ECHO "Installing SWe on AMD, as the processor is suitable family..."
   else
      preInstallLogMsg "ERROR: SWe cannot be run as the processor does not meet the requirements"  "user"
      $SED -i 's/CheckStatus=.*/CheckStatus=checkFailed/' $preUpgradeCheckMarker
      $SED -i 's/Reason=.*/Reason= SWe cannot be run as the processor does not meet the requirements/' $preUpgradeCheckMarker
      $RM -f $upgradeBaseDir/preUpgradeCheckActive.key
      array_exitReason[index++]=processorOn_${ceName}DoesntMeetSWERequirements
      exitRequired=1
   fi
fi

# make sure that we are installing on supported hardware
checkPlatformHarwareType $preUpgradeCheckLogs ${installRoot}/preInstallCheck.result

# verify drbd and backing device sizes.
if [[ "$hostSubType" != "virtualCloud" && "$hostSubType" != "virtualContainer" ]] ; then
    $installRoot/sbxVerifyDrbdPartition.sh |$TEE -a $preUpgradeCheckLogs
    result=${PIPESTATUS[0]}
    if [ $result -ne 0 ]; then
       # note: the script outputs the error, no need to say anything here...
       # sbxLinuxCmd traps the output error...
       array_exitReason[index++]=sbxVerifyDrbdPartitionFailedOn_${ceName}
       exitRequired=1
    fi
fi

# Installs and upgrades in ePSX mode require
# minimum memory sizes.
if [ -e "$sonusConfigFile" ]; then
  loadConfig $sonusConfigFile
  if [ "$personality" -eq $SbxEpsxPersonality ]; then
    # Installs and upgrades in ePSX mode are not supported for 5400
    if [ "$hostType" == "SBC5400" ]; then
      preInstallLogMsg "ERROR: ePSX mode not allowed for host type $hostType"
      exit 1
    fi
  fi

  # validate the current configuration if doing an upgrade.  we now enforce naming
  # conventions and some machines may have a system name and hostname that are
  # the same.  we need to catch it early rather than catching it after starting
  # the upgrade.
  # NOTE: logFile must be set due to how validateConfig works
  # NOTE: validateConfig calls exit with an error value if it detects an issue
  if [ $skipConfigVal -eq 0 ]; then
    logFile=$preUpgradeCheckLogs
    loadConfig $sonusConfigFile
    # skip personality check in case of LSWU or PM upgrade for ePSX installations 
    if [ "$personality" -eq $SbxEpsxPersonality ]; then
       skipPersonalityCheck=true;
    fi
    validateConfig
  fi
fi

if [ $exitRequired == "1" ]; then
  reasonNum=1
  preInstallLogMsg "Pre Checks failed with the following errors:" "user"
  for element in "${array_exitReason[@]}"
  do
  preInstallLogMsg "$reasonNum. $element" "user"
    ((reasonNum+=1))
  done
  exitreason=$(printf '%s__' "${array_exitReason[@]}")
  errorExit $exitreason
fi

if [ $skipStatusFileCreation == "false" ];then

  # Get current role
  getCurrentRole
  if [ "$currentRole" == "active" ];then
     activeCeName=$ceName
     standbyCeName=$peerCeName
  else
     activeCeName=$peerCeName
     standbyCeName=$ceName
  fi
  if [ "$activeCeName" == "$primaryCeName" ];then
     primaryUpgradeStatus="pendingUpgrade"
     secondaryUpgradeStatus="upgrading"
  else
     primaryUpgradeStatus="upgrading"
     secondaryUpgradeStatus="pendingUpgrade"
  fi
  $PERL $statusUpder -c -r preChecks -s "EndTime" -v "`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`"
  $PERL $statusUpder -c -r preChecks -s "Reason" -v "Successful_Completion"
  $PERL $statusUpder -c -r preChecks -s "CEcheckStatus" -v "checkComplete"
fi

# If current server is active, poll for peer servers CheckStatus
if [ "$currentRole" == "active" ];then
  if [ $skipStatusFileCreation == "false" ];then
     preInstallLogMsg "Pre Checks completed on $ceName. Waiting for Pre Checks to complete on peer:$peerCeName.."
     count=0 
     while($SLEEP 2)
     do
       peerState=`$PERL $statusUpder -c -r preChecks -g CEcheckStatus -p`;
       if [ "$peerState" == "checkComplete" ];then
          preInstallLogMsg "Pre Checks completed on $peerCeName.. Updating overall Status to complete.." 
          $PERL $statusUpder -c -s "CheckStatus" -v "checkComplete"
          $PERL $statusUpder -c -s "Reason" -v "Successful_Completion"
          $PERL $statusUpder -c -s "CheckCompleteTime" -v "`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`"
          break;
       elif [ "$peerState" == "checkFailed" ];then
          preInstallLogMsg "ERROR: Pre Checks failed on Server:$peerCeName! Calling errorExit().."  "user"
          sleep 2 
          reason=`$statusUpder -g Reason`
          if [ "$reason" == "none" ];then
             errorExit preChecksFailedOn_${peerCeName}
          else
             # Do not overwrite the failure reason filled by peer.
             errorExit
          fi
       elif [ "$peerState" == "checkInProgress" ];then
           if [ $count == 90 ];then
               preInstallLogMsg "ERROR: Waited for 3 minutes, But peerState=inProgress. Manual Intervention required. Setting state to failed and Exiting.." "user"
               errorExit timedOutWhileWaitingForPreCheckCompletionOn_${peerCeName}
           fi
          count=$[$count +1] 
          preInstallLogMsg "Pre Checks are in progress on peer server:$peerCeName. Continue Waiting.." 
       elif [ "$peerState" == "none"  ];then
          if [ $count == 90 ];then
             preInstallLogMsg "ERROR: Waited for 3 minutes, But peerState=none. Manual Intervention required. Setting state to failed and Exiting.." "user"
             errorExit timedOutWhileWaitingForPreCheckStartOn_${peerCeName}
          fi
          count=$[$count +1]
          preInstallLogMsg "Pre Checks not yet started on peer server:$peerCeName!!" 
       else
          if [ $count == 90 ];then
             preInstallLogMsg "ERROR: Waited for 3 minutes, But peerState=$peerState. Manual Intervention required. Setting state to failed and Exiting.." "user"
             errorExit timedOut_InvalidPreCheckStatusOn_${peerCeName}
          fi
          count=$[$count +1]
          preInstallLogMsg "Invalid PreCheck state detected on peer:$peerCeName. State=$peerState.."  "user"
       fi
     done
  fi
elif [[ "$currentRole" == "standby" && $skipStatusFileCreation == "false" && $localOnly == "0" ]];then
     #This is a case of preInstallChecks getting called on standby during LSWU. Standby should wait until active completes to catch active preCheck failure(if any).
     preInstallLogMsg "Pre Checks completed on $ceName. Waiting for overall status to become checkComplete.."
     count=0
     while($SLEEP 2)
     do
         overallState=`$PERL $statusUpder -c -g "CheckStatus"`
         if [ "$overallState" == "checkComplete" ];then
            preInstallLogMsg "Pre Checks completed on active server: $peerCeName[INFO]."
            break;
         elif [ "$overallState" == "checkFailed" ];then
            preInstallLogMsg "ERROR: Pre Checks failed on Server:$peerCeName! Calling errorExit().."  "user"
            sleep 2 
            reason=`$statusUpder -g Reason`
            if [ "$reason" == "none" ];then
               errorExit preChecksFailedOn_${peerCeName}
            else
               # Do not overwrite the failure reason filled by peer.
               errorExit
            fi
         elif [ "$overallState" == "checkInProgress" ];then
            if [ $count == 90 ];then
               preInstallLogMsg "ERROR: Waited for 3 minutes, But overallState=checkInProgress. Manual Intervention required. Setting state to failed and Exiting.." "user"
               errorExit timedOutWhileWaitingForPreCheckCompletionOn_${peerCeName}
            fi
            count=$[$count +1]
            preInstallLogMsg "Pre Checks are in progress on peer server:$peerCeName. Continue Waiting.."
         elif [ "$overallState" == "none"  ];then
            if [ $count == 90 ];then
               preInstallLogMsg "ERROR: Waited for 3 minutes, But overallState=none. Manual Intervention required. Setting state to failed and Exiting.." "user"
               errorExit timedOutWhileWaitingForPreCheckStartOn_${peerCeName}
            fi
            count=$[$count +1]
            preInstallLogMsg "Pre Checks not yet started on peer server:$peerCeName!!"
         else
            if [ $count == 90 ];then
               preInstallLogMsg "ERROR: Waited for 3 minutes, But overallState=$overallState. Manual Intervention required. Setting state to failed and Exiting.." "user"
               errorExit timedOut_InvalidPreCheckStatusOn_${peerCeName}
            fi
            count=$[$count +1]
            preInstallLogMsg "Invalid PreCheck state detected. State=$overallState.."  "user"
         fi

         # Check for the role on peer and if not active, exit with preCheck failure
         peerRole=`/opt/sonus/sbx/scripts/swinfo.sh | $AWK '/Current/{iter++}iter==2{print; exit}' | $AWK '{print $4}'`
         peerRole=`$ECHO $peerRole | $TR -d '[[:space:]]'`

         if [ "$peerRole" != "active" ]; then
            preInstallLogMsg "ERROR: Invalid role detected on $peerCeName. peerRole=$peerRole. Manual Intervention required. Setting state to failed and Exiting.."  "user"
            errorExit invalidRoleDetectedOn_${peerCeName}
         fi
     done
fi

if [ $skipStatusFileCreation == "false" ];then
   createUpgradeFile true
fi

if [ "$checkOnly" == "0" ]; then
  preInstallLogMsg "Creating upgrade steps/status file.." 

  # Disable Oracle replication status trigger if Oracle based release.
  # pre-install check is only executed after peers are in sync.
  if [ ! -d ${HOME}/postgresql ]; then
      preInstallLogMsg "Disabling Oracle replication status trigger"
      su - oracle -c "echo \"alter trigger platform.sbx_dbrep_status_change disable; \n exit;\n \" | sqlplus \"/ as sysdba\""
      $ECHO "0~syncCompleted" > ${HOME}/oraadmin/SSDB/udump/.PolicyDBSyncStatus
  fi

  if [ ! -d $upgradeDir ]; then
    $MKDIR -p $upgradeDir;
  fi
  # upgrade group not defined in older release
  # will be reverted after new OS is installed.
  # Also need to relax permissions on staging. 
  # These will be reversed at the end of upgrade by reconfigHw.pl. 
  if upgradeGroupExists; then
    $CHMOD 770 $upgradeDir;
    $CHMOD g+rw $SONUS_STAGING_DIR/
  else
    $CHMOD 777 $upgradeDir;
    $CHMOD 777 $SONUS_STAGING_DIR/
  fi

  createUpgradeFile false

  $RM -f $upgradeBaseDir/latest 2>/dev/null
  $LN -s current $upgradeBaseDir/latest
else
  preInstallLogMsg "CheckOnly=$checkOnly, not creating status files..."  
  if [[ $skipStatusFileCreation == "false" && -e $packageContents ]];then
     # This is a case of preChecks, so cleanup staging directory.
     currDir=`pwd`
     cd $SONUS_STAGING_DIR 
     $RM -f `$CAT $packageContents`
     # Adding more files to be removed in case multiple installs/upgrades occur before starting SBX (like in case of ePSX).
     $RM -f sbc*.tar.gz *.md5 *.sha256 appInstall*.sh connexip*.deb ema*.rpm sbc*.rpm ema*.deb sbc*.deb sbc*.signature
     $RM -f $packageContents
     cd $currDir
  fi
 
fi
preInstallLogMsg "Removing preUpgradeCheckActive.key..." 
$RM -f $upgradeBaseDir/preUpgradeCheckActive.key 

exit 0
