#!/usr/bin/perl
##############################################################
##
## Copyright (c) 2014 Sonus Networks, Inc.
##
## All Rights Reserved.
## Confidential and Proprietary.
##
## Module Description:
##
##
##
##############################################################

BEGIN {
  # locate packages in places other than the standard perl path
  unshift(@INC,"/opt/sonus/bin");
}

use Getopt::Std;
use strict;
use lib '/opt/sonus/bin';
use lib "/opt/sonus/staging";
no warnings 'once';

require "sbxUpgradeUtils.pl";
use sbxCommonFunctions;

if ( -e "/opt/sonus/bin/sonusCommands.pm" )
{
    use sonusCommands;
}
else
{
    logmsg("Could not locate sonusCommands.pm. Exiting...");
    exit 1;
}

if ( -e "/opt/sonus/bin/sonusCommonFiles.pm" )
{
    use sonusCommonFiles;
}
else
{
    logmsg("Could not locate sonusCommonFiles.pm. Exiting...");
    exit 1;
}

my $progname=`$BASENAME $0`;
my $sonusRoot = "$SONUS_DIR";
my $sonusStaging="$SONUS_STAGING_DIR";
my $pre50MarkerFile="$PRE50_MARKER_TXT";
my $dynamicHAScript="$CHECK_DYNAMIC_HA_PL";
my $statusUpdater="$STATUS_UPDATER_PL";
my $upgradeBaseDir="$SONUS_LOG_UPGRADE_DIR";
my $origMajorVersion=5;
my $upgradeType="unknown";
my $logFile="unknown";
my $sbxInitArgs="unknown";
my $status="unknown";
my $time="unknown";
my $currentLv;
my $prevLv;
my $prodString;
my $gRpm=`$LS $sonusStaging/sbc-*rpm`;
chomp($gRpm);
my $gPackage="unknown";
my $DoDUpgradeMarker="/home/admin/DoDUpgradeMarker";

if ( -e "$gRpm" )
{
   $gPackage = $gRpm;
   $gPackage =~ s/rpm$/tar.gz/;
}

my $hostType=`$CAT $HOST_TYPE`;
my $hostSubType=`$CAT $HOST_SUB_TYPE`;
chomp($hostType);
chomp($hostSubType);

if ( -e "$STAGING_SONUS_UTILS_SH" )
{
   $prodString = `source $STAGING_SONUS_UTILS_SH ; $ECHO \$prodString`;
}
else
{
   $prodString = `source $SONUS_UTILS_SH ; $ECHO \$prodString`;
}
chomp($prodString);

################################################################
## SBC version 5.0 contains an NP re-architecture.  When updating
## to version 5.0 via LSWU, we need to use the legacy mode until
## both nodes are upgraded and the LSWU is complete.  Then and
## only then can we switch modes, and we do so via a switchover.
## Create a marker file, if neccessary, to indicate that we
## need to swith the NP mode.
## NOTE: the file name and contents are shared with SM and MUST
## NOT CHANGE without the subsequent code changes.
#################################################################
sub checkNPRearch()
{
  my $npRearchMarkerFile="$NP_MODE_UPDATE";
  my $hostType=`$CAT $HOST_TYPE`;
  chomp($hostType);

  if ( -e $pre50MarkerFile )
  {
    $origMajorVersion=4;
  }

  if( ! -e "$npRearchMarkerFile" && "$hostType" ne "SBC7000")
  {
    logMsg("Performing NP re-architecture check...");

    # the marker does not exist.  if we are upgrading from a version
    # prior to 5.0 then we need to create the marker.
    # NOTE: the marker shouldn't exist, but for debug/troubleshooting purposes
    # allow it to exist already
    if( $origMajorVersion < 5 ) {
      # phase1 means this is the first machine to upgrade to a minimum of 5.0
      my $contents="phase1";
      logMsg("Preparing for NP-Rearchitecture switch after LSWU completes");

      my $majVersionRemote=getSWVersion("remote");
      logMsg("Remote version is $majVersionRemote");
      if( $majVersionRemote >= 5 ) {
        # phase3 means this is the second machine to upgrade to a minumum of 5.0
        $contents="phase3"
      }

      # create the marker file so that service startup does the correct thing
      my $ret = createMarkerFile("$npRearchMarkerFile", $contents);
      if ($ret ne "success") {
        logMsg("FAILED to create NP re-architecture marker file with contents \"$contents\"");
      }
    } else {
      logMsg("New NP-architecture already in use");
    }
  }
  elsif(  "$hostType" eq "SBC7000" ) {
    logMsg("SBX7000: Skipping NP re-architecture check...");
  }
  else {
    logMsg("NP re-architecture marker found, check being skipped...");
  }
}

sub main()
{
  getopts('u:');

  my $upgradeType = $Getopt::Std::opt_u; 

  if ( $upgradeType eq "LSWU" )
  {
    # Since sbxbond may be missing in case of qcow, setup sbxbond here
    # temporarily before running sbxInit.  If it is already started it
    # will just ignore the request.
    logMsg("Setting up sbxbond temporarily");
    system("$SYSTEMCTL start sbxbond");

    openPeerHandle();

    `$PERL $statusUpdater -r Reboot -s Status -v "Complete"`;
    $time=getTime();
    `$PERL $statusUpdater -r Reboot -s EndTime -v "$time"`;
    `$PERL $statusUpdater -r Reboot -s Reason -v "Successful_Completion"`;

    $logFile="$UPGRADE_OUT";
    $sbxInitArgs="-u -p -l -R -f $SBXCONF_FILE -L $logFile";
  }
  else
  {
      $logFile="$SBX_UPDATE_OUT";
      $sbxInitArgs="-u -R -f $SBXCONF_FILE -L $logFile";
  }

  if ( $hostType ne "ConnexIP5000" )
  {
     # Install Securelink package
     `$RPM -e --force-debian --allmatches --dbpath /root/.rpmdb sl > /dev/null 2>&1`;
     `$DPKG -P sl > /dev/null 2>&1`;
     my $slDeb=`$LS $SONUS_SECURELINK_DIR/sl_*.deb | $CUT -d"/" -f5`;
     chomp($slDeb);
     logMsg("Installing SecureLink package $slDeb...");
     `$DPKG -i $SONUS_SECURELINK_DIR/$slDeb >> $logFile 2>&1`;

     # Need root ownership since it is in sudo path. Group sonus allow access by PM
     `$CHOWN -R root:sonus $SONUS_SECURELINK_DIR`;
     `$CHMOD -R 0750 $SONUS_SECURELINK_DIR $SONUS_SECURELINK_DIR/*sh $SL_ACL`;
     `$CHMOD -R g+r $SONUS_SECURELINK_DIR`;

     # Create directory to save securelink logs (if not present)
     `$MKDIR -p /var/log/sonus/securelink`;
     `$CHOWN root:root /var/log/sonus/securelink`;
     `$CHMOD 755 /var/log/sonus/securelink`;
  }

  # Invoke enableFIPS.sh if FIPS mode is enabled
  my $fipsMode=`$GREP fipsMode $SBXCONF_FILE | $AWK -F= '{print \$2}'`;
  chomp($fipsMode);
  logMsg("Checking the FIPS mode status... Current mode: $fipsMode");
  if ( "$fipsMode" eq "enabled" )
  {
      logMsg("Executing enableFIPS.sh as FIPS mode is enabled");
      `sh $ENABLE_FIPS_SH`;
  }

  # Remove staging/mop directory if present
  logMsg("Checking for the existence of staging/mop dir and deleting it if present...");
  if ( -d "$SONUS_STAGING_DIR/mop")
  {
      logMsg("Removing $SONUS_STAGING_DIR/mop during upgrade...");
      my $delRes = system("rm -rf '$SONUS_STAGING_DIR/mop'");
      if ( $delRes != 0 )
      {
        logMsg("Failed to remove $SONUS_STAGING_DIR/mop...");
      }	
  }

  # Removing s-bit from statusUpdater.pl
  `$CHMOD g-s $STATUS_UPDATER_PL`;

  # Run sbxInit.sh and redirect output to null to avoid redundant logs as sbxInit script by itself does the logging.
  my $sbxInit=`$BASENAME $SBX_INIT_SH`;
  my $sbxInitOut=`$BASENAME $SBX_INIT_OUT`;
  chomp($sbxInit);
  chomp($sbxInitOut);
  logMsg("Initializing sbx service...");
  logMsg("Calling sbxInit to regenerate conf files.. Command=$sbxInit $sbxInitArgs");
  my $res=system("$SBX_INIT_SH $sbxInitArgs >/dev/null");
  if ( $res != 0 )
  {
      logMsg("sbx Initialization failed during upgrade! Refer $sbxInitOut for details...");
      exit(1);
  }

  if( !(($hostType =~ m/ConnexIP5000/ ) || ( $prodString =~ m/5100/ )))
  {
     $currentLv=`$DF -kh |$GREP debian-root |$HEAD -n1 | $AWK '{print \$1}'`;
     chomp($currentLv);
     if( $currentLv eq "/dev/mapper/debian-root2" )
     {
        $prevLv="/dev/mapper/debian-root1";
     }
     else
     {
        $prevLv="/dev/mapper/debian-root2";
     }
     if (! -d "/mnt/target")
     {
        `$MKDIR -p /mnt/target`;
     }
     `$MOUNT $prevLv /mnt/target`;

     if ( -e $DB5_3_RECOVER )
     {
        logMsg("Backing up $prevLv rpmdb...");
        `$RM -fr /mnt/target/opt/sonus/.rpmdb_bk`;
        `$CP -ar /mnt/target/root/.rpmdb /mnt/target/opt/sonus/.rpmdb_bk`;
        my $p2RpmDb="/mnt/target/root/.rpmdb";
        logMsg("Running db5.3_recover on p2RpmDb...");
        system("pushd $p2RpmDb; $DB5_3_RECOVER; popd");
     }
     else 
     {
        logMsg("Package db5.3-util not installed! Not updating rpmdb on p2...");
     }
     `$UMOUNT /mnt/target`;
  }

  logMsg("Restoring saved configuration...");
  `sh $RESTORE_CDB_SH`; 
  if( ("$hostType" eq "ConnexIP5000" ) || ( $prodString =~ m/5100/ ))
  {
      logMsg("Restoring policy database...");
      chomp(my $upgBackupFile=`$LS -rt /home/admin/pg_sonusdb_*_for_upgrade.dmp 2>/dev/null | $TAIL -1`);
      if (("$upgBackupFile" eq "" ))
      {
          logMsg("Policy data backup file not found. Exiting...");
          exit 1;
      }

     `cd $SONUS_PSX_SQL_DIR; $PERL configureDB -install UPGRADE -upgBackupFile $upgBackupFile -loglevel 1 -force Y -logdir /tmp >> $logFile 2>&1`;
      if ($? != 0)
      {
         logMsg("Failed to restore policy database. Exiting...");
         exit 1;
      }
  }

  if ( -e "$PSX_GENERIC_KEY" )
  {
      # Update LI encryption key file permissions
      logMsg("Update LI encryption key file permissions...");
     `$CHMOD 600 $PSX_GENERIC_KEY`;
     `$CHOWN root:root $PSX_GENERIC_KEY`;
  }

  # Creating new peer ssh handle here as restoreSonusDb takes longer and timeouts
  if ( $upgradeType eq "LSWU" )
  { 
     openPeerHandle();
  }

  logMsg("Restoring saved confd.conf file...");
  `sh $RESTORE_CONFD_CONFIG_SH`;

  if ( "$upgradeType" eq "LSWU" )
  {
     # make sure we are properly setup for dynamic AMF model changes
     logMsg("Invoking $dynamicHAScript...");
     system($dynamicHAScript);

     logMsg("Invoking checkNPRearch...");
     checkNPRearch();
  }

  # Remove any upgrade packages
  `$RM -rf $SONUS_EXTERNAL_DIR/sbc_V*`;
  `$RM -rf $SONUS_EXTERNAL_DIR/sbc-V*`;
  `$RM -rf $SONUS_EXTERNAL_DIR/*.qcow2`;
  `$RM -f $sonusStaging/qemu-utils*.deb`;
  `$RM -f $sonusStaging/libiscsi*.deb`;

  # Remove pre-upgrade shadow file if any
  `$RM -f /home/admin/shadow.preUpgrade`;

  if( ("$hostType" eq "ConnexIP5000" ) || ( $prodString =~ m/5100/ ))
  {
     logMsg("Updating grub entries...");
     `$SED -i '/sonus-curr-ver/s/[ ]*upgrade=true[ ]*//' $SONUS_GRUB_MENU_SDA1_FILE`;
     `$UPDATE_GRUB_SH >> $logFile 2>&1`;
  }

  logMsg("Adding acl rules to upload directory...");
  `$SETFACL -R -d -m u::rwx $HOME_SFTPROOT/external/ >> $logFile 2>&1 || $ECHO "ERROR: failed to set user acl on external"  >> $logFile`;
  `$SETFACL -R -d -m g::rwx $HOME_SFTPROOT/external/ >> $logFile 2>&1 || $ECHO "ERROR: failed to set group acl on external" >> $logFile`;
  `$SETFACL -R -d -m o::000 $HOME_SFTPROOT/external/ >> $logFile 2>&1 || $ECHO "ERROR: failed to set world acl on external" >> $logFile`;

  if(  "$hostType" ne "SBC7000" && "$hostType" ne "SBC5400" ) {
    # remove unused software so there are no ldd reported errors
    `$RM -fr $SONUS_SBX_DIR/utilities/cavcreek`;
  }

  if( "$hostType" ne "ConnexIP5000" ) {
     # Keeping just the ipsec-stat binary on H/W. since, 
     # it is useful to track the packets transferred through ipsec tunnel.
     `$CP $SONUS_BIN_NP_SWE_DIR/ipsec-stat $SONUS_SBX_UTILITIES_DIR`;
     `$RM -fr $SONUS_BIN_NP_SWE_DIR`;
  }
  else
  {
     `$TOUCH $POST_CLOUD_UPGRADE_OPERATIONS_MARKER`;
  }

  # start fstrim service timer to start on Monday 00:00:00Hrs
  logMsg("Enabling fstrim before starting sbc app...");
  system("$SED -i '/ExecStart/s/\\/etc\\/fstab://' /lib/systemd/system/fstrim.service");
  # Change 'Persistent' field value to 'false' so that timer does not get triggered on running 'systemctl daemon-reload'
  system("$SED -i 's/Persistent=.*\$/Persistent=false/' /lib/systemd/system/fstrim.timer");
  system("$SYSTEMCTL enable --now fstrim.timer");
  
  logMsg("REBOOT: Skipping reboot for image based upgrade and starting SBC app...");
  # restart SBX
  if ( "$upgradeType" eq "LSWU" )
  {
     logMsg("--------Starting APP_BRINGUP--------");
     `$PERL $statusUpdater -r AppBringUp -s Status -v "inProgress"`;
     $time=getTime();
     `$PERL $statusUpdater -r AppBringUp -s StartTime -v "$time"`;
  } 
  my $ret = startSbxService();
  if ($ret ne "success")
  {
     logMsg("Unable to start SBX, ret: $ret Exiting....");
     if ( "$upgradeType" eq "LSWU" )
     {
        `$PERL $statusUpdater -r AppBringUp -s Reason -v "Unable_to_start_SBX"`;
        `$PERL $statusUpdater -s Reason -v "Unable_to_start_SBX.Possible_Recovery_Action_is:Manual_Retry_to_start_or_Revert."`;
        `$PERL $statusUpdater -r AppBringUp -s Status -v "failed"`;
        $time=getTime();
        `$PERL $statusUpdater -r AppBringUp -s EndTime -v "$time"`;
        logMsg("Unable to start SBX, ret: $ret reverting....");
     }
     exit 1;
  }
  else
  {
    logMsg("SBC Upgrade Complete...");
    if ( "$upgradeType" eq "LSWU" )
    { 
       `$PERL $statusUpdater -r AppBringUp -s Status -v "Complete"`;
       `$PERL $statusUpdater -r AppBringUp -s Reason -v "Successful_Completion"`;
       $time=getTime();
       `$PERL $statusUpdater -r AppBringUp -s EndTime -v "$time"`;
       logMsg("--------Completed APP_BRINGUP--------");
    }
  }

  if ( "$upgradeType" eq "LSWU" )
  {
     $status=`$PERL $statusUpdater -g UpgradeStatus`;
     chomp($status);
     if ($ret eq "success")
     {
        logMsg("--------Starting POSTUPGRADE--------");
        $time=getTime();
        `$PERL $statusUpdater -r PostUpgrade -s StartTime -v "$time"`;
        `$PERL $statusUpdater -s Status -v "inProgress" -r PostUpgrade`;
        if ( "$status" =~ /upgradingStandby/ )
        {
           postUpgradeChecks("standby");
           `$PERL $statusUpdater -s UpgradeStatus -v "upgradingStandbyDone"`;
           logMsg("--------Completed STANDBY UPGRADE--------");
           logMsg("Starting active upgrade monitor in background..");
           system("$PERL $UPGRADE_MANAGER_PL startUpgrade $gPackage activeMonitor \&");
        }
        elsif ( "$status" =~ /upgradingActive/ || "$status" =~ /manualUpgrade/ )
        {
           postUpgradeChecks("active");
           `$PERL $statusUpdater -s UpgradeStatus -v "upgradingActiveDone"`;
           `$PERL $statusUpdater -s UpgradeStatus -v "upgradeComplete"`;
           $time=`date "+%FT%T%z"`;
           chomp($time);
           `$PERL $statusUpdater -s UpgradeCompleteTime -v "$time"`;
           `$PERL $statusUpdater -s Reason -v "Successful_Completion"`;
           logMsg("--------Completed ACTIVE UPGRADE--------");
           logMsg("Archiving log and status files..");
           sbxCommonFunctions::archiveUMLogFiles;
           `$RM -f $upgradeBaseDir/liveUpgradeActive.key`;
           my $cmd="$RM -f $upgradeBaseDir/liveUpgradeActive.key";
           peerCommand ($cmd);
        }
     }
     else
     {
        if ( "$status" =~ /upgradingStandby/ )
        {
           `$PERL $statusUpdater -s Reason -v "Standby_Server_Sbxstart_Failed.Possible_Recovery_Action_is:To_start_SBX_Manually_or_To_Contact_Sonus_support."`;
        }
        elsif ( "$status" =~ /upgradingActive/ )
        {
           `$PERL $statusUpdater -s Reason -v "Active_Server_Sbxstart_Failed.Possible_Recovery_Action_is:To_start_SBX_Manually_or_To_Contact_Sonus_support."`;
        }
        `$PERL $statusUpdater -s UpgradeStatus -v "upgradeFailed"`;
     }
  }

  if ( -e $pre50MarkerFile)
  {
    `$RM -f $pre50MarkerFile.backup`;
    `$MV -f $pre50MarkerFile $pre50MarkerFile.backup`;
  }
  
  # Update staging dir permissions
  `$CHMOD 755 $sonusStaging/`;
  `$CHMOD g+w $sonusStaging/upgrade.out 2>/dev/null`;

  # Check if we upgraded in DoD mode
  if( -e "$DoDUpgradeMarker" )
  {
    logMsg("Upgraded in DoD mode.. Setting up DoD environment..");
    my $pmStatus=`$GREP "PM=" $DoDUpgradeMarker |$AWK -F= '{print \$2}'`;
    chomp($pmStatus);
    my $cliStatus=`$GREP "CLI=" $DoDUpgradeMarker |$AWK -F= '{print \$2}'`;
    chomp($cliStatus);

    `$RM -f $DoDUpgradeMarker`;
    `$HOST_JITC_MODE_SH enable`;
    if ( "$pmStatus" eq "Enabled" )
    {
      logMsg("Enabling PM in DoD mode..");
      `$HOST_JITC_MODE_SH enable-pm`;
    }
    else
    {
      logMsg("Disabling PM in DoD mode..");
      `$HOST_JITC_MODE_SH disable-pm`;
    }
    if ( "$cliStatus" eq "Enabled" )
    {
      logMsg("Enabling CLI in DoD mode..");
      `$HOST_JITC_MODE_SH enable-cli`;
    }
    else
    {
      logMsg("Disabling CLI in DoD mode..");
      `$HOST_JITC_MODE_SH disable-cli`;
    }
  }

  my $homeperm = `stat -c "%a" /home`;
  chomp($homeperm);
  if ($homeperm ne "755")
  {
    logMsg("/home permission is not 755 and found to be $homeperm. Changing it back to 755.");
    `$CHMOD 755 /home`;
  }
  my $slashperm = `stat -c "%a" /`;
  chomp($slashperm);
  if ($slashperm ne "755")
  {
    logMsg("/ permission is not 755 and found to be $slashperm. Changing it back to 755.");
    `$CHMOD 755 /`;
  }

  #check if some file permissions changed after upgrade
  logMsg("Starting permission check after upgrade");
  if(-e "/var/log/permission/permission.version")
  {
    system("$SBX_PERMISSION_CHECK_SH afterUpgrade");
    logMsg("Finished permission check after upgrade");
  }
  else
  {
    logMsg("Could not find permission.version file.")
  }
}

main();

logEndUpdate();

exit(0);

