#!/bin/bash
#############################################################
#
# Copyright (c) 2016 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# sbxImageBasedUpgrade.sh
#
# 04/28/2016
#
# Module Description:
# Script to facilitate qcow2 image based in place upgrade,
# primarily to be used for SBX5100/SWe where there is no re-partitioning.
#
# Example:
#
#    sh sbxImageBasedUpgrade.sh -L /opt/sonus/staging/upgrade.out
#
#############################################################

# Source sonus commands
if [ -e /opt/sonus/staging/sonusCommands.sh ] && [ -e /opt/sonus/staging/sonusCommonFiles.sh ]
then
    source /opt/sonus/staging/sonusCommands.sh
    source /opt/sonus/staging/sonusCommonFiles.sh
else
    /bin/echo "sonusCommands.sh and/or sonusCommonFiles.sh not found. Exiting..."
    exit 1
fi

basePath=/src
logFile=
lswu="n"
sonusRoot=$SONUS_DIR
sonusStaging=$SONUS_STAGING_DIR
hostTypeFile=$HOST_TYPE
hostSubTypeFile=$HOST_SUB_TYPE
statusUpdater=$STATUS_UPDATER_PL
sftpChrootDir=$HOME_SFTPROOT
srcSftpChrootDir=$basePath/home/sftproot
srcUpgradeBaseDir=$basePath/home/log/sonus/upgrade

if [ -f $STAGING_SONUS_UTILS_SH ]; then
   . $STAGING_SONUS_UTILS_SH
else
   . $SONUS_UTILS_SH
fi

if [ -f $SBXCONF_FILE ]; then
   sonusConfigFile=$SBXCONF_FILE
else
   sonusConfigFile=$SBX_CONF
fi

ceName=`$GREP ceName $sonusConfigFile | $AWK -F "="  '{print $2}'`

# Note: Kernel versions are pulled from the kernel environment during packaging
kernelVersion="5.10.234"
rescueVersion=""

usage()
{
  $CAT << EOF
  usage: $0 options

  OPTIONS:
     -h     Print this message.
     -L     logFile name
     -l     flag to indicate lswu.
EOF
}

while getopts "hL:l" OPTION
do
  case $OPTION in
  h)
     usage
     exit 1
     ;;
  L)
     logFile=$OPTARG
     ;;
  l)
     lswu="y"
     ;;

  ?)
     usage
     exit $E_BADARGS
     ;;
  esac
done

if [ -f $hostTypeFile ]; then
   hostType=`$CAT $hostTypeFile`
else
   # Use prodString from sonusUtils
   if [[ $prodString == "5100" || $prodString == "5110" ]]; then
       hostType="ConnexIP5100"
   else
       hostType="ConnexIP5000"
   fi
   logMsg $logFile "File $hostTypeFile does not exist..hostType is set to $hostType"
fi

getReqOSVersion
targetVersion=$($GREP "SBC Version:" $STAGING_BUILDINFO | $AWK '{print $3}')
targetQcow=`$ECHO $targetVersion | $SED 's/-//'`
qcowImageName=$($BASENAME `$LS $sonusStaging/sbc-${targetQcow}-connexip-os_${reqOSVersion}_*amd64.qcow2.sha256 | $SED 's/.sha256//'`)
qcowSubDir=`$ECHO $qcowImageName | $SED 's/sbc-V/sbc_V/'`

if [ -f $SONUS_EXTERNAL_DIR/$qcowImageName ]; then
   qcowImagePath=$SONUS_EXTERNAL_DIR/$qcowImageName
elif [ -f $SONUS_EXTERNAL_DIR/$qcowSubDir/$qcowImageName ]; then
   # For upgrades from old versions like 4.0/4.1, qcow2 image can be present in sub-dir
   qcowImagePath=$SONUS_EXTERNAL_DIR/$qcowSubDir/$qcowImageName
else
   logMsg $logFile "ERROR: qcow2 image $qcowImageName not found in $SONUS_EXTERNAL_DIR and sub-dir $SONUS_EXTERNAL_DIR/$qcowSubDir. Exiting..."
   exit 1;
fi

if [ "$lswu" == "y" ]; then
   logMsg $logFile "--------Starting IMAGE UPGRADE--------"
   $PERL $statusUpdater -r ImageUpgrade -s Status -v "inProgress"
   $PERL $statusUpdater -r ImageUpgrade -s StartTime -v "`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`"
fi

# Mount the qocw2 image on /tmp/root and /tmp/boot
performQcow2Mount $logFile $qcowImagePath

# Copy qcow2 contents (/tmp/root) to basePath
performQcow2Copy $logFile $basePath

# Retain passwdadmin for encryptedStore to work
if [ -d /opt/sonus/shared/passwdadmin ]; then
   $RM -rf $basePath/opt/sonus/shared/passwdadmin/
   basePathPasswdadminID=$($GREP 'passwdadmin' $basePath/etc/passwd | $CUT -f 3 -d ':')
   $CP -af /opt/sonus/shared/passwdadmin $basePath/opt/sonus/shared/passwdadmin/
   $CHOWN -R $basePathPasswdadminID:$basePathPasswdadminID $basePath/opt/sonus/shared/passwdadmin/
   $CHMOD --reference=/tmp/root/opt/sonus/shared/passwdadmin $basePath/opt/sonus/shared/passwdadmin
   
else
   $RM -rf $basePath/opt/sonus/shared/passwdadmin/
   basePathPasswdadminID=$($GREP 'passwdadmin' $basePath/etc/passwd | $CUT -f 3 -d ':')
   $CP -af /home/passwdadmin $basePath/opt/sonus/shared/passwdadmin/
   $CHOWN -R $basePathPasswdadminID:$basePathPasswdadminID $basePath/opt/sonus/shared/passwdadmin/
   $CHMOD --reference=/tmp/root/home/passwdadmin $basePath/opt/sonus/shared/passwdadmin/
  
fi

if [ -d /var/log/permission ]; then
    $RM -rf $basePath/var/log/permission/
    $CP -af /var/log/permission $basePath/var/log/permission/
fi

# Retain cnxipmadmin .ssh directory for inter CE communication using public key to work
if [ -e /home/cnxipmadmin/.ssh/authorized_keys ]; then
   logMsg $logFile "Found /home/cnxipmadmin/.ssh/authorized_keys"
   $RM -rf $basePath/home/cnxipmadmin/.ssh/
   $CP -af /home/cnxipmadmin/.ssh $basePath/home/cnxipmadmin/.ssh/
   $CHOWN -R --reference=/tmp/root/home/cnxipmadmin/.ssh $basePath/home/cnxipmadmin/.ssh/
   $CHMOD --reference=/tmp/root/home/cnxipmadmin/.ssh $basePath/home/cnxipmadmin/.ssh/
   logMsg $logFile "Backed up cnxipmadmin authorized keys to $basePath..."
else
   logMsg $logFile "Not found: /home/cnxipmadmin/.ssh/authorized_keys (INFO)"
fi

# Also retain root .ssh directory for inter CE communication using public key to work
if [ -e /root/.ssh/id_rsa ]; then
   logMsg $logFile "Found /root/.ssh/id_rsa"
   $RM -rf $basePath/root/.ssh/
   $CP -af /root/.ssh $basePath/root/.ssh/
   $CHOWN -R --reference=/tmp/root/root/.ssh $basePath/root/.ssh/
   $CHMOD --reference=/tmp/root/root/.ssh $basePath/root/.ssh/
   logMsg $logFile "Backed up root ssh keys to $basePath..."
else
   logMsg $logFile "Not found: /root/.ssh/id_rsa (INFO)"
fi

# Copy sonusMetadata.xml
$CP -pf $basePath/opt/sonus/sbx/tailf/var/confd/cdb/sonusMetadata.xml.orig $basePath/opt/sonus/sbx/tailf/var/confd/cdb/sonusMetadata.xml

checkForNBDError $logFile

# Save off qcow2 boot dir contents
$RM -rf /tmp/qcow2BootDir
$MKDIR -p /tmp/qcow2BootDir
$CP -af /tmp/boot/ /tmp/qcow2BootDir

performQcow2Unmount $logFile $qcowImagePath

# NOTE: reference the grub configuration file in basePath so that we use
# the new version of the configuration
if [ -d /tmp/qcow2BootDir ]; then
   logMsg $logFile "Setting up grub to boot from new kernel/initrd..."

   # remove current/former images from the boot directory
   cleanupBootFiles /tmp/qcow2BootDir/boot

   # Save off the previous ribbon custom menu file before making changes so we
   # have it for reference purposes.  However, make sure to remove the execute
   # permission so that update-grub doesn't generate menu items from it.
   if [[ -e "$SONUS_GRUB_MENU_SDA1_FILE" ]]; then
      $CP -f $SONUS_GRUB_MENU_SDA1_FILE ${basePath}${SONUS_GRUB_MENU_SDA1_FILE}.prev
      $CHMOD 644 ${basePath}${SONUS_GRUB_MENU_SDA1_FILE}.prev
   fi

   # maintain the existing grub2 password
   # note: get the user/password from grub.cfg so keep all platforms the same
   # note: the password hash will generate differently each time the utility
   # grub-mkpasswd-pbkdf2 is run.  there is therefore no reason to check if
   # it is different.  just perform the update with the original value.
   cfgUserArg=""
   origGrubUser=$($GREP "set superusers" $GRUB_CONFIG | $AWK -F'"' '{print $2}')
   origGrubPwd=$($GREP "password_pbkdf2" $GRUB_CONFIG | $AWK '{print $3}')
   newGrubUser=$($GREP "^SUPERUSER" ${basePath}${SONUS_GRUB_MENU_PWD_FILE} | $AWK -F'"' '{print $2}')

   if [[ -n "$origGrubUser" && "$origGrubUser" != "$newGrubUser" ]]; then
       cfgUserArg="-u $origGrubUser"
   fi
   if [ -n "$origGrubPwd" ]; then
       # -s option species skip calling update-grub.  no need to call it when
       # configuring the password since that will happen below
       ${basePath}${CONFIGURE_GRUB_PWD_SH} -s $cfgUserArg -P $origGrubPwd -f ${basePath}${SONUS_GRUB_MENU_PWD_FILE} 2>&1 | $TEE -a $logFile
   fi

   # udpate the grub default file so kernel parameters properly match the platform
   ${basePath}${CONFIGURE_GRUB_SH} -g -f ${basePath}${ETC_DEFAULT_GRUB}

   # Delete the existing SBC version menu
   # note: recall the file we are dealing with here is the qcow file
   $SED -i -e '/sonus-curr-ver/d' ${basePath}${SONUS_GRUB_MENU_SDA1_FILE}

   # Add the new menu entry
   # note: upgrade is a hook into initrd for swe/5100 upgrades
   ${basePath}${CONFIGURE_GRUB_SH} -u -k $kernelVersion -d "/dev/mapper/debian-root" -v $targetVersion -t "Sonus SBC $targetVersion" -i "sonus-curr-ver" -f ${basePath}${SONUS_GRUB_MENU_SDA1_FILE}

   # On KVM and openstack we need to add console=ttyS0
   # ttyS0 is used by virsh console on KVM
   productType=`$GREP -m 1 "Product Name" $DMIDECODE_OUT | $AWK '{print $3}'`
   if [[ $productType == "KVM" ]];then
      ${basePath}${CONFIGURE_GRUB_SH} -C add -f ${basePath}${SONUS_GRUB_MENU_SDA1_FILE}
   fi

   # Need to mount a few items in the target path in order to generate the
   # new grub.cfg in a chroot session
   $MOUNT -t devtmpfs none $basePath/dev
   $MOUNT -t proc none $basePath/proc
   $MOUNT --bind /sys  $basePath/sys
   $MOUNT --bind /boot $basePath/boot
   $MOUNT --bind /home $basePath/home
   # To fix an lvm2 bug
   $MOUNT --bind /run  $basePath/run

   # install grub2 to the MBR if not already installed
   if [[ -n "$($DPKG -l | $GREP grub-legacy)" ]]; then
       logMsg $logFile "grub-legacy currently installed: Installing grub2 to the MBR on ${primary_device}..."
       $CHROOT $basePath $GRUB_INSTALL $primary_device 2>&1 | $TEE -a $logFile
       $RM -f ${GRUB_MENU_LST}* /boot/grub/{{xfs,reiserfs,e2fs,fat,jfs,minix}_stage1_5,stage{1,2}}
   fi

   if [[ "$hostType" != "ConnexIP5000" ]]; then
       # Work around an lvm2 bug causing error output during update-grub command
       # Refer to the function for more info
       updateLvmConf "hw"
   fi

   # copying apparmor.cfg
   if [ -f $APPARMOR_GRUB ] ; then
       logMsg $logFile "copying Apparmor.cfg..."
       $CP -pf $APPARMOR_GRUB $basePath/etc/default/grub.d
   fi
   
   # generate a new grub.cfg
   # note: update-grub spits a lot of junk to stderr; lets trash it...
   logMsg $logFile "Generating new grub config..."
   $CHROOT $basePath $UPDATE_GRUB_SH 2> /dev/null

   # update the default entry
   grubEntries=$($GREP -c "^sonus_linux_entry " ${basePath}${SONUS_GRUB_MENU_SDA1_FILE})
   newGrubDefault=$((grubEntries-1))
   logMsg $logFile "Updating grub default to menu entry $newGrubDefault..."
   $CHROOT $basePath $GRUB_SET_DEFAULT $newGrubDefault

   # unmount the temporary mounts we created
   $UMOUNT $basePath/dev $basePath/proc $basePath/sys $basePath/boot $basePath/home

   # udpate the real grub config dir incase a different call to update grub
   # is made before we reboot and start running the new OS/app
   # (not necessary in h/w since grub.d is a link to /home/etc/grub.d so
   # we are always dealing with the right file)
   # note: this will/does happen for adding the fsck parameter when
   # shutting down/rebooting as connexip-os updates grub when shutting down.
   $CP -f ${basePath}${SONUS_GRUB_MENU_SDA1_FILE} ${SONUS_GRUB_MENU_SDA1_FILE}

   $RM -rf /tmp/qcow2BootDir
else
   logMsg $logFile "Failed to find qcow2 image boot directory. Exiting..."
   exit 1;
fi

$ECHO "$hostType" > $basePath/etc/hostType
if [ -e $hostSubTypeFile ]; then
   hostSubType=`$CAT $hostSubTypeFile`
   $CP -pf $hostSubTypeFile $basePath/etc/hostSubType
else
   if [[ "$hostType" == "ConnexIP5000" ]]; then
      # SWe case
      hostSubType='virtual'
   else
      # For 5100
      hostSubType=$prodString
   fi
   logMsg $logFile "File $hostSubTypeFile does not exist..hostSubType is set to $hostSubType"
   $ECHO "$hostSubType" > $basePath/etc/hostSubType
fi

$RM -f $basePath/opt/sonus/sbx/reconfigHaMarker
$RM -f $basePath/opt/sonus/sbx/tailf/var/confd/sonusCdbState.conf
$RM -fr $basePath/opt/sonus/staging/*

# Perform all hostType specific updates
performHostSpecificUpdatesForUpgrade $logFile $basePath $lswu $hostType $kernelVersion

if [[ -d /home/Administrator && ! -L /home/Administrator ]]; then
   # Retain files from /home/Administrator if dir not empty
   if [ "$($LS -A /home/Administrator)" ]; then
      $MKDIR -p $basePath/home/Administrator 
      $CP -Rpf /home/Administrator/* $basePath/home/Administrator/
   fi
fi

if [[ "$hostType" == "ConnexIP5000" ]]; then
    logMsg $logFile "Checking for SWe_NP conf files"
    if [ -d $SONUS_CONF_SWE_DIR/ ]; then
        logMsg $logFile "SWe conf dir exists, copying files from capacityEstimates"
        if [ -e $ACT_CORE_PARTITION ]; then
            $CP -pf $ACT_CORE_PARTITION $basePath/$SONUS_CONF_SWE_DIR/
        fi
        if [ -e $ACT_CALLMIX_JSON ]; then
            $CP -pf $ACT_CALLMIX_JSON $basePath/$SONUS_CONF_SWE_DIR/
            # Take a backup of the activeCallMixInput.json file.
            # Modify the activeCallMixInput.json's transcode callmix to abide by the new
            # rules of codecmix (i.e., having G7112G711 when G711% > non-G711%)

            # PLEASE NOTE : The underlying python script will currently run on
            # the python version installed in the older version SBC!

            $STAGING_UPDATE_SWE_CALLMIX_PY upgrade $basePath
            retCode=$?
            if [[ $retCode -ne 0 ]]; then
                logMsg $logFile "Failed to modify the content of $ACT_CALLMIX_JSON"
            else
                logMsg $logFile "The $ACT_CALLMIX_JSON update process completed successfully."
            fi

            # Saving the corepartition logs.
            $CP -pf $CORE_PARTITION_LOG $basePath/$CORE_PARTITION_LOG

            logMsg $logFile "Please refer to $CORE_PARTITION_LOG for further details"
        fi

        logMsg $logFile "Major version comparison for SWe upgrade"
        srcVersion=`$SWINFO_SH -l | $GREP SBC: | $AWK '{print $2}' | $SED -e 's/sbc-//' | $AWK -F. '{print $1}' | $SED -e 's/V//'`
        targetVersion=$($GREP "SBC Version:" $STAGING_BUILDINFO | $AWK '{print $3}' | $SED 's/-//' | $AWK -F. '{print $1}' | $SED -e 's/V//')
        logMsg $logFile "srcVersion=$srcVersion, targetVersion=$targetVersion"
        if [ -e $INDEX_TXT ]; then
            logMsg $logFile "Retaining $INDEX_TXT and $INDEX_MARKER"
            $CP -pf $INDEX_TXT $basePath/$SONUS_CONF_SWE_DIR/
            $CP -pf $INDEX_MARKER $basePath/$SONUS_CONF_SWE_DIR/
        else
            logMsg $logFile "$INDEX_TXT does not exist, $INDEX_TXT will be regenerated."
        fi
    else
        logMsg $logFile "SWe conf dir does not exist, copying files from np-log path"
        $CP -pf $SONUS_LOG_NP_DIR/.* $basePath/$SONUS_CONF_SWE_DIR/
    fi

    if [ -e $SWE_CFG_PROFILE_TXT ]; then
        logMsg $logFile "SWe: $SWE_CFG_PROFILE_TXT exists, copying it"
        $CP -pf $SWE_CFG_PROFILE_TXT $basePath/$SONUS_CONF_DIR/swe/
    else
        sweTotalMem=`$GREP "MemTotal" /proc/meminfo | $AWK '{print $2}'`
        #for applying large profile min RAM should be 18GB.
        reqMemLargeCfgProfile="18308136"

        if [ $sweTotalMem -lt $reqMemLargeCfgProfile ]; then
           logMsg $logFile "SWe: writing configProfile=0 to $SWE_CFG_PROFILE_TXT"
           $ECHO "configProfile=0" > $basePath/$SWE_CFG_PROFILE_TXT
        else
           logMsg $logFile "SWe: writing configProfile=1 to $SWE_CFG_PROFILE_TXT"
           $ECHO "configProfile=1" > $basePath/$SWE_CFG_PROFILE_TXT
        fi 
    fi

    # Retaining 70-persistent-net.rules and interface_mapping.json file, as a
    # custom interface ordering configuration as part of previous release has to be maintained.
    if [ -e $UDEV_RULES_FILE ]; then
        logMsg $logFile "SWe: $UDEV_RULES_FILE file exists, copying it"
        $CP -pf $UDEV_RULES_FILE $basePath/$UDEV_RULES_FILE
    else
        logMsg $logFile "SWe: $UDEV_RULES_FILE file doesn't exist..."
    fi

    if [ -e $SONUSUDEV_RULES_JSON ]; then
        logMsg $logFile "SWe: $SONUSUDEV_RULES_JSON file exists, copying it"
        $CP -pf $SONUSUDEV_RULES_JSON $basePath/$SONUSUDEV_RULES_JSON
    else
        logMsg $logFile "SWe: $SONUSUDEV_RULES_JSON file doesn't exist..."
    fi
fi

logMsg $logFile "Saving off most recent pre-upgrade logs..."
fileList="/opt/sonus/sbx/openclovis/var/log /opt/sonus/sbx/tailf/var/confd/log /var/log/auth.log /var/log/daemon.log /var/log/kern.log /var/log/messages /var/log/monit.log /var/log/remoteExecution.log /var/log/syslog /var/log/user.log /var/log/sonus/sbxPerf"

isDebPkgMgmt
if [ $? -ne 0 ]; then
    currentRelease=$($RPM -qa |$GREP sbc |$AWK -F "sbc-" '{print $2}' |$SED 's/.x86_64.*//')
else
    currentRelease=$($DPKG -l sbc | $GREP ^ii | $AWK '{print $3}' | $SED 's/^/V/')
fi
$MKDIR -p $basePath/opt/sonus/staging/preUpgradeLogs
$TAR -czf $basePath/opt/sonus/staging/preUpgradeLogs/logsFromVersion-sbc-${currentRelease}.tar.gz $fileList
logMsg $logFile "Saving off logs done..."

# Move dirs and Create links
$CP -rpf $sonusStaging/* $basePath/opt/sonus/staging/
$CP -af $sonusStaging/.[^.]* $basePath/opt/sonus/staging/
if [ -d $basePath/opt/sonus/external ]; then
   # Retain old external dir contents without overwriting existing files
   logMsg $logFile "Retaining external directory contents..."
   $MV -n /opt/sonus/external/* $srcSftpChrootDir/external/
   $RM -rf $basePath/opt/sonus/external
   $LN -s $sftpChrootDir/external $basePath/opt/sonus/external
fi
if [ -d $basePath/opt/sonus/staging ]; then
   $MV $basePath/opt/sonus/staging $srcSftpChrootDir/
   $LN -s $sftpChrootDir/staging $basePath/opt/sonus/staging
fi

# Move log back to /home
if [ -d $basePath/var/log ]; then
   logMsg $logFile "Moving $basePath/var/log to $basePath/home/"
   $MV $basePath/var/log $basePath/home
   $LN -s /home/log $basePath/var/log
fi

$CHOWN -R --reference=$srcSftpChrootDir/external/ $srcSftpChrootDir/external
$CHMOD -R --reference=$srcSftpChrootDir/external/ $srcSftpChrootDir/external
$CHOWN -R --reference=$srcSftpChrootDir/staging/ $srcSftpChrootDir/staging/
$CHMOD -R --reference=$srcSftpChrootDir/staging/ $srcSftpChrootDir/staging/

if [ "$lswu" == "y" ]; then
   $PERL $statusUpdater -r ImageUpgrade -s Status -v "Complete"
   $PERL $statusUpdater -r ImageUpgrade -s EndTime -v "`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`"
   $PERL $statusUpdater -r ImageUpgrade -s Reason -v "Successful_Completion"
   logMsg $logFile "--------Completed IMAGE UPGRADE--------"
fi

# Copy new grub default file, update-grub and make sure kernel
# options are applied on boot.
logMsg $logFile "Copying default grub file and generating new grub config for next boot.."
$CP -f ${basePath}${ETC_DEFAULT_GRUB} $ETC_DEFAULT_GRUB
$UPDATE_GRUB_SH 2> /dev/null

logMsg $logFile "Rebooting to perform image based upgrade..."

if [ "$lswu" == "y" ]; then
   logMsg $logFile "--------Starting REBOOT--------"
   $PERL $statusUpdater -r Reboot -s Status -v "inProgress"
   $PERL $statusUpdater -r Reboot -s StartTime -v "`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`"
fi

# Update /home/log/sonus/upgrade to preserve the logs
# NOTE: must be done after statusUpdater calls so we capture those entries
$RM -fr $srcUpgradeBaseDir/*
$CP -af $SONUS_LOG_UPGRADE_DIR/* $srcUpgradeBaseDir/
$CHOWN -R --reference=$srcUpgradeBaseDir $srcUpgradeBaseDir/
$CHMOD -R --reference=$srcUpgradeBaseDir $srcUpgradeBaseDir/

# 6.2 jessie changes (systemd) caused a change to how we handle our
# custom reboot script.  in upgrading from a release prior to 6.2,
# the new version of the custom script won't exist.  for that
# case we need to revert to using the previous override script.
if [ -e $REBOOT ]; then
   logMsg $logFile "Rebooting via $REBOOT..."
   $REBOOT 2>&1 | tee -a $logFile
else
   logMsg $logFile "Rebooting via /sbin/reboot..."
   /sbin/reboot 2>&1 | tee -a $logFile
fi

exit 0
