#!/bin/bash
#############################################################
#
# Copyright (c) 2009 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# sbxInstall.sh
#
# Mark St. Pierre
# 6/1/09
#
# Module Description:
# Script to install application service.
#    
#############################################################

# Sourcing sonus commands
source /opt/sonus/bin/sonusCommands.sh
source /opt/sonus/bin/sonusCommonFiles.sh

# Globals
E_BADARGS=65
sbxDebFile="."
oracleXeFile=""
emaDebFile="."
gServiceName=sbx
sonusRoot=$SONUS_DIR
sonusStaging=$SONUS_STAGING_DIR
sbxRoot=$SONUS_SBX_DIR
evLogDir=evlog
sbxInitResultFile=$sonusStaging/.sbxInit_result
initCommand="$SBX_INIT_SH -E $sbxInitResultFile"
sbctype="isbc"
logFile=
hmacEncoded="cmJibgo="
hmacKey=$(echo $hmacEncoded | base64 -d)

# Revert feature related marker
$MKDIR -p /home/common
updateP2RpmdbMarker=$SONUS_INSTALLUPGRADE_LOG_DIR/.updateP2RpmdbMarker
p2Base=/mnt/target

# pickup some common utilities. 
if [ -e $STAGING_SONUS_UTILS_SH ]; then
   . $STAGING_SONUS_UTILS_SH
else
   $ECHO "ERROR: Missing utilities file: `$BASENAME $STAGING_SONUS_UTILS_SH`; Exiting..."
   exit 1
fi

# A list of the Debian MD5 checksum files to be omitted from SHA256 calculations.
# Exclude these digests from the list of digests we are interested in.
function debian_excluded_digests () {
cat <<EOF
lwresd.md5sums
gdbserver.md5sums
apache2.2-bin.md5sums
libapache2-mod-php7.0.md5sums
libvirt0.md5sums
libvirt-bin.md5sums
openssh-client.md5sums
openssh-server.md5sums
apache2.2-common.md5sums
gnupg.md5sums
monit.md5sums
fips.md5sums
usbutils.md5sums
EOF
}

# A list of specific Debian files to be omitted.
# These files may be altered intentionally during installation.
function debian_excluded_files () {
cat <<EOF
usr/lib/python3/dist-packages/cloudinit/sources/DataSourceEc2.py
usr/lib/python3/dist-packages/cloudinit/sources/DataSourceGCE.py
usr/lib/python3/dist-packages/cloudinit/stages.py
usr/lib/python3/dist-packages/cloudinit/net/__init__.py
EOF
}

# A list of the interesting directories in the Debian packages.
# Note that a few MD5 files contain pathnames prefixed with './', dammit.
function debian_digest_directories () {
cat <<EOF
 bin
 usr/bin
 ./usr/bin
 usr/local/bin
 usr/local/sbin
 sbin
 usr/sbin
 usr/lib
 ./usr/lib
 opt/sonus/bin
 etc/init.d/sfips-post
EOF
}

# Generate file names based on the checksums in Debian packages.
function gen_connexip_digest_names () {
  ls *.md5sums | grep -v -F "$(debian_excluded_digests)" |
  while read f; do
    grep -F "$(debian_digest_directories)" "$f" | awk '{print $2}'
  done |
  grep -v -F "$(debian_excluded_files)" |
  sed 's#^#/#;s#//#/#g;s#^/\./#/#' |
  sort
}

# Generate HMAC SHA256 digests based on the MD5 checksums in Debian packages.
function gen_connexip_digests () {
  logMsg $logFile "INMESSAGE: Generating connexip HMAC SHA256 digests"
  pushd /var/lib/dpkg/info &>/dev/null
  if [ -f connexip.hmacSha256sums ]; then
    logMsg $logFile "INMESSAGE: Digests already present for $(wc -l connexip.hmacSha256sums) files"
  else
    for i in $(gen_connexip_digest_names);
    do
        if [ -e "$i" ];then
           dig=$(openssl sha256 -hmac $hmacKey $i | awk -F "=" '{print $2}' |$TR -d '[[:space:]]')
           echo "$dig  $i" >> connexip.hmacSha256sums
        fi
    done
    logMsg $logFile "INMESSAGE: Digests generated for $(wc -l connexip.hmacSha256sums) files"
  fi
  popd &>/dev/null
}

# main function.
main()
{
   ROOT_UID=0
   if [ "$UID" -ne "$ROOT_UID" ]; then
      logMsg $logFile "ERROR: must be root to run this script."
      exit 1
   fi

   # Stop sbx service...
   stopService $logFile

   # Remove current EMA RPM...
   emaPkg=`$RPM --dbpath /root/.rpmdb -qa | $GREP ema`
   if [ -n "$emaPkg" ]; then
      logMsg $logFile "Removing current EMA package: $emaPkg"
      executeCommand "$RPM --dbpath /root/.rpmdb -e --allmatches --force-debian ema" 0 $logFile "Failed to remove rpm $emaPkg"
   fi

   # Remove current EMA DEB...
   emaPkg=`$DPKG -l ema 2>/dev/null | $GREP ^ii | $AWK '{ print $2 }'`
   if [ -n "$emaPkg" ]; then
      logMsg $logFile "Removing current EMA package: $emaPkg"
      executeCommand "$DPKG -P ema" 0 $logFile "Failed to remove deb $emaPkg"
   fi

   # Remove current application RPM...
   # note: on first install we could have app and app1 empty, leading to a grep -e error
   if [[ $app != "" || $app1 != "" ]]; then
      sbxPkg=`$RPM --dbpath /root/.rpmdb -qa | $GREP -e $app $app1`
      if [ -n "$sbxPkg" ]; then
         logMsg $logFile "Removing current `$ECHO $app | $AWK '{print toupper($1)}'` package: $sbxPkg"
         mnemonics=`$ECHO $sbxPkg | $SED -e 's/-V[0-9][0-9]\.[0-9][0-9]\.[0-9][0-9]-[A-Z][0-9][0-9][0-9]//g'`
         executeCommand "$RPM --dbpath /root/.rpmdb -e --allmatches --force-debian $mnemonics" 0 $logFile "Failed to remove rpm $sbxPkg"
      fi
   fi

   # Remove current application DEB...
   sbxPkg=`$DPKG -l sbc 2>/dev/null | $GREP ^ii | $AWK '{print $2 }'`
   if [ -n "$sbxPkg" ]; then
      logMsg $logFile "Removing current SBC package: $sbxPkg"
      executeCommand "$DPKG -P sbc" 0 $logFile "Failed to remove deb $sbxPkg"
   fi

   # If the configuration file option was not specified, remove any existing
   # application startup config file.  Otherwise make sure we pass the option to
   # the initialization script.
   if [ -z $configFile ]; then
      $RM -f $SBXCONF_FILE
   else
      initCommand="$initCommand -f $configFile"
      tmpsbctype=`$CAT $configFile | $GREP 'sbctype' | $AWK -F '=' '{print $2}'`
      if [[ "$tmpsbctype" == "isbc" || "$tmpsbctype" == "ssbc" || "$tmpsbctype" == "msbc" || "$tmpsbctype" == "mrfp" ||  "$tmpsbctype" == "slb" ]]; then
          sbctype=$tmpsbctype
      fi
      logMsg $logFile "Using Configuration file: $configFile"
   fi
   logMsg $logFile "DSBC Personality in config file: $sbctype"

   # Install new database RPMs...
   # No need to check diskspace since we just removed the old rpm
   if [ -e $updateP2RpmdbMarker ] ;then
      # Update p2 rpm db with sonusdb
      targetLV=$($CAT $updateP2RpmdbMarker |$AWK -F "root=" '{print $2}')
      logMsg $logFile "Updating rpm db on $targetLV.." 
      $MOUNT |$GREP $p2Base &>/dev/null

      if [ $? != 0 ]; then
         $MKDIR -p $p2Base
	     $MOUNT $targetLV $p2Base
	     if [ $? != 0 ]; then
	        logMsg $logFile "Cannot mount $targetLV on $p2Base to update p2 rpm database.."
	        exit 1
         fi
      fi

      $MV -f $updateP2RpmdbMarker $updateP2RpmdbMarker.backup
   fi
   

   # Install new application DEBs...
   # No need to check diskspace since we just removed the old deb
   logMsg $logFile "Installing $sbxDebFile from path ($PWD)"
   executeCommand "$DPKG -i $sbxDebFile" 0 $logFile "Failed to install deb package $sbxDebFile"
   if [ "$?" != "0" ]; then
      $ECHO "Cannot install SBX package"
      exit $?
   fi

   # Install new EMA DEB...
   # No need to check diskspace since we just removed the old deb
   logMsg $logFile "Installing $emaDebFile from path ($PWD)"
   executeCommand "$DPKG -i $emaDebFile" 0 $logFile "Failed to install deb package $emaDebFile"
   if [ "$?" != "0" ]; then
      $ECHO "Cannot install EMA package"
      exit $?
   fi

   # If the SHA256 digest for Debian-supplied files does not exist already,
   # generate it to replace Debian's MD5 sums. Used by FIPS software verification.
   [[ -f /var/lib/dpkg/info/connexip.hmacSha256sums ]] || gen_connexip_digests

   # Export the digests so that the file can be used to update the ISO later.
   # This is part of QCOW generation. If the labeled filesystem does not exist,
   # then this is not part of QCOW generation and the export is skipped.
   local buildoutput=$(mktemp -d)
   mount LABEL=buildoutput $buildoutput && {
     logMsg $logFile "Exporting connexip.hmacSha256sums"
     cp /var/lib/dpkg/info/connexip.hmacSha256sums $buildoutput
     umount $buildoutput
     logMsg $logFile "Exported connexip.hmacSha256sums"
   }
   rmdir $buildoutput

   # Run initialization script...
   # NOTE: using the result file is a holdover from working around tee with script.
   # Since then sbxInit was changed to do its own log capture, but there was no
   # reason to bother with removing all of the resultFile work.
   # We do have to capture stderr however.  Unfortunately we cannot both capture it
   # and have it show on the terminal since any use of tee (process substitution and
   # tee) won't work as the tee command inherits any fds that are started and therefore
   # does not close properly, even though this script moves on (seen in install case
   # as shutdown debugging showed appInstall gone by sbxInstall still running with the
   # tee as a child of it).  Do the only thing we can: capture it in the log but
   # not on the terminal.
   logMsg $logFile "INMESSAGE: Initializing $gServiceName service..."
   $initCommand 2>> $logFile
   sbxInitRes=`$CAT $sbxInitResultFile`
   if [ $sbxInitRes -ne 0 ]; then
      logMsg $logFile "Failed initialization during execution of \"$initCommand\""
      exit $sbxInitRes
   fi
}

usage()
{
    # don't use logMsg here since no reason to have a timestamp on the usage
$ECHO "usage: `$BASENAME $0` [-h] -f <application DEB file> [-d <database rpm>] [-c <configuration file>] [-L <log file>] [-R]

OPTIONS:
   -h                 Print this message.
   -f <app DEB>       Application DEB file.
   -d <database rpm>  Database RPM file.
   -c <config file>   Use configuration file as input.
   -L <log file>      Log file to capture script output.
   -R                 Confirm reboot if a reboot is necessary." | $TEE -a $logFile
}


# Do not allow runnning from the evlog directory since the terminal
# will get ripped out from under us, stopping the script prematurely.
# also look for /var/log/sonus/evlog or /var/log/sonus/mirror.
baseDir=`$BASENAME $PWD`
mountDir=`$ECHO $baseDir | $AWK -F/ '{print $5}'`
if [ "$baseDir" = "$evLogDir" -o "$mountDir" = "evlog" -o "$mountDir" = "mirror" ]; then
    logMsg $logFile "ERROR: `$BASENAME $0` not allowed from a drbd mounted directory.  Please leave the directory and try again."
    exit 1
fi

# Set program...
program=`$BASENAME $0`

while getopts "hf:d:x:c:L:R" OPTION
do
   case $OPTION in
   h)
      usage
      exit 1
      ;;
   f)
      sbxDebFile="$OPTARG"
      emaDebFile=`$ECHO $OPTARG | $SED 's/sbc/ema/'`
      ;;
   d)
      dbRpmFile="$OPTARG"
      ;;
   x)
      oracleXeFile="$OPTARG"
      ;;
   c)
      configFile="$OPTARG"
      if [ -z $configFile ]
      then
         logMsg $logFile "ERROR: Invalid configuration file, exiting..."
         exit 1
      else
        # remove leading whitespace
        configFile=`$ECHO $configFile|$SED -e 's/^[ \t]*//'`
      fi
      configFileTemp=`$ECHO $configFile|$SED -e 's|^/||'`
      configFileMsg=
      if [ $configFile = $configFileTemp ]; then
         configFileMsg="Using configuration file $PWD/$configFile..."
      else
         configFileMsg="Using configuration file $configFile..."
      fi
      # use double quote on logFile so even if not set it is a param...
      logMsg "$logFile" "$configFileMsg"
      ;;
   L)
      logFile="$OPTARG"
      ;;
   R)
     initCommand="$initCommand -R"
     ;;
   ?)
      usage
      exit $E_BADARGS
      ;;
   esac
done

shift $(($OPTIND - 1))

# default logFile name if not set. if passed in, the caller
# is responsible for resetting it...
if [ -z "$logFile" ]; then
   logFile=$sonusStaging/sbxInstall.out
   $CAT /dev/null > $logFile
fi

initCommand="$initCommand -L $logFile"

# guarantee we are copied to/running from the correct location
# note: checkLocation exits on error...
checkLocation $0 $sonusStaging $logFile

checkDiags $logFile

# Validate DEB files...
if [ ! -f $sbxDebFile ]; then
   logMsg $logFile "ERROR: $sbxDebFile not found."
   usage
   exit $E_BADARGS
fi
if [ ! -f $emaDebFile ]; then
   logMsg $logFile "ERROR: EMA package not found."
   usage
   exit $E_BADARGS
fi

# Figure out application package type...
app=`dpkg -l sbc 2> /dev/null | grep ^ii | awk '{ print $2 }'`
app1=""
if [[ $app == "sbc" ]]; then
   app1="-e sbx"
fi

# Start main() script...
main

exit 0
