#!/bin/bash

#############################################################
#
# Copyright (c) 2014 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# sbxTargetRevert.sh
#
# 29/07/14
#
# Module Description:
#
# This script is invoked by Platform Manager.
# This script mounts p2 and checks for possible Revert. 
# If Secondary version is lower than the current, it sets up Revert script 
# and marker on p2, sets boot root as p2 and reboots.
# 
# This script is copied to SBX tar.gz as sbxRevert.sh
#
#############################################################

# Sourcing Command Variables
source /opt/sonus/bin/sonusCommands.sh
source /opt/sonus/bin/sonusCommonFiles.sh
source /opt/sonus/sbx/scripts/personality.sh

sonusConfigFile=$SBXCONF_FILE
localOnly="n"
localServer=`$GREP ceName $sonusConfigFile | $AWK -F "="  '{print \$2}'`
peerServer=`$GREP peerCeName $sonusConfigFile | $AWK -F "="  '{print \$2}'`
revertLog=$SONUS_STAGING_DIR/revert.out

if [ -f $STAGING_SONUS_UTILS_SH ]; then
   . $STAGING_SONUS_UTILS_SH
else
   . $SONUS_UTILS_SH
fi

# Clean up earlier revert log
$ECHO > $revertLog

usage()
{
    logMsg $revertLog "Wrong argument. Usage: $0 Version"
    logMsg $revertLog "Example: $0 $secVersion"
}

while getopts "lh" OPTION
do
   case $OPTION in
   l)
        localOnly="y"
        ;;
   h)
        usage
        exit 1;
        ;;
   ?)
        usage
        exit 1;
        ;;
   esac
done


shift $(($OPTIND - 1))


if [ -e $LVINFO_PL  ]; then
   currentVersion=$($LVINFO_PL |$GREP SBC | $GREP -v "CE Name" |$HEAD -n1 |$AWK -F ":V" '{print $2}' | $AWK '{print $1}')
   secVersion=$($LVINFO_PL |$GREP SBC | $GREP -v "CE Name" | $TAIL -n1 |$AWK -F ":V" '{print $2}' | $AWK '{print $1}')
   currentPath=$($LVINFO_PL |grep "Current Path" |awk -F ":" '{print $2}')
else
   logMsg $revertLog "$LVINFO_PL doesnt exist! Unable to identify backed up Version. exiting..."
   exit 1
fi

if [ -z "$1" ]; then 
   logMsg $revertLog "Invalid selection. exiting..."
   exit 1
else
   arg=$1
fi

logMsg $revertLog "Initiating revert to $1 from $localServer"

if [ $peerServer == "none" ];then
   logMsg $revertLog "CE mode=Standalone"
   localOnly="y";
fi 

# Check if preUpgrade version had ePSX installed.

if [ -e /mnt/target/opt/sonus/sbx.conf ];then
   mntSbxDotConf="/mnt/target/opt/sonus/sbx.conf"
else 
   if [ -e /mnt/target/opt/sonus/conf/sbx.conf ];then
      mntSbxDotConf="/mnt/target/opt/sonus/conf/sbx.conf"
   else
      logMsg $revertLog "sbx.conf does not exist in /mnt directory."
   fi
fi

if [ -e "$mntSbxDotConf" ];then
   EPSX_MODE=2
   personality=`$CAT $mntSbxDotConf |$GREP personality |$AWK -F "=" '{print $2}'| $AWK '{print $1}'`
   if [ $personality -eq $SbxEpsxPersonality ];then
      logMsg $revertLog "preUpgrade version has ePSX personality! Revert not supported to software installed with ePSX.."
      exit 1
   fi
fi

if [ $localOnly == "n" ]; then 
   # call revert on the peer server also, call it asynchronously to make sure both scripts run independently. 
   ($SONUS_PEER_CNX_EXPECT $SUDO $SBX_REVERT_SH -l "$arg"  > /dev/null)& # sudoers-cnxipmadmin
fi


if [[ "$arg" != "$currentVersion" && "$arg" != "$secVersion" ]]; then
   usage
   exit 1;
fi

revertTosecondary()
{
    logMsg $revertLog "Checking if Revert is possible on $localServer.."
    if [[ ! -e "$HOME_BACKUP_DIR/home_backup.tar.gz" ]]; then
       if [[ ! -e "$DB_UPGRADE_MARKER" ]]; then
           logMsg $revertLog "Revert Software cannot be performed on a fresh installed system."
           logMsg $revertLog "REVERT NOT POSSIBLE!"
           exit 1
       fi
    fi

    lvcount=$($LVSCAN 2>/dev/null |$WC -l 2>/dev/null)
    if [ $lvcount == 3 ]; then
        # Find current mounted partition
        mountedLV=$($DF -kh |$GREP debian-root |$HEAD -n1 | $AWK '{print $1}')
        if [[ $mountedLV == "/dev/mapper/debian-root1" ]]; then
            next=root2
        else
            next=root1
        fi
    else
        logMsg $revertLog "Cannot revert while running with single LV..!!"
        exit 1
    fi

    if [[ -z "$($GREP sonus-prev-ver $GRUB_CONFIG)" ]]; then 
       logMsg $revertLog "Already reverted. Cannot revert again."
       exit 1 
    fi

    logMsg $revertLog "Revert is possible.."
    logMsg $revertLog "Current running Version=$currentVersion"
    logMsg $revertLog "Reverting to $secVersion"
    logMsg $revertLog "Copying Revert script to Secondary.."
    if [ -e $SBX_REVERT_PL ] ;then
       version=$( $ECHO $secVersion | $CUT -d'.' -f 1)
       if [ $version -le 05 ]; then
         $CP -f $SBX_REVERT_PL /mnt/target/opt/sonus/sbxRevert.pl
       else
         $CP -f $SBX_REVERT_PL /mnt/target/opt/sonus/installUpgrade/sbxRevert.pl
       fi
    else
        logMsg $revertLog "Revert Script not found! Revert not possible"
        exit 1
    fi

    logMsg $revertLog "Updating boot menu (nextBoot=$next).."
    $SWITCH_BOOT_SH "$next" &>> $revertLog

    if [ -d /mnt/target/opt/sonus/installUpgrade/log/ ];then
       $ECHO "Marker created by $0 for Revert on $mountedLV " > /mnt/target/opt/sonus/installUpgrade/log/lswuCommitPending
    else
       $ECHO "Marker created by $0 for Revert on $mountedLV " > /mnt/target/opt/sonus/lswuCommitPending
    fi

    # remove any knowledge about what leadership algorithm was in use
    $RM -f $SBX_PEER_SPLIT_BRAIN_MODE_MARKER

    logMsg $revertLog "Rebooting the box and Reverting to previous version($secVersion).."
    $ECHO "About to reboot" > /tmp/rebootInProgressKey.key; $CHMOD 777 /tmp/rebootInProgressKey.key
    $SYSTEMCTL --no-block reboot
}

currentPath=`$ECHO $currentPath | $TR -d '[[:space:]]'`
if [ "$currentPath" != "updated" ];then
    logMsg $revertLog "currentPath is: $currentPath .. Not taking any action on current server."
    exit 0
fi

if [ "$arg" == "$secVersion" ];then
   revertTosecondary
elif [ "$arg" == "$currentVersion" ];then
   logMsg $revertLog "User selected version is same as current running version. No action taken." 
else
   logMsg $revertLog "$arg and $secVersion are not same. Revert not possible with the current selection..."
fi

exit 0

