#!/usr/bin/perl -w

#############################################################
#
# Copyright (c) 2009 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# sbxTargetUpgradeUtils.pl
#
# Gautham Nimmagadda
# 6/1/09
#
# Module Description:
#
# This script contains utilities to be used during SBX
# live software upgrade process. This file is included by
# other higher level scripts such as sbxTargetUpgrade.pl
# and sbxTargetRevert.pl.
#
# This script is copied to SBX tar.gz as sbxUpgradeUtils.pl
#
# This script doesn't contain main() and shouldn't be invoked on it's own.
#
# TBD: check errors from fcopy() and dircopy()
#############################################################

BEGIN {
  # locate packages in places other than the standard perl path
  unshift(@INC,"/opt/sonus/bin");
  unshift(@INC,"/opt/sonus/staging");
}

use FileHandle;
use File::Path;
use Getopt::Std;
use File::Copy::Recursive qw(fcopy rcopy dircopy fmove rmove dirmove);
use File::Basename;
use Cwd;
use sonusCommands;
use sonusCommonFiles;
use sbxCommonFunctions;
###
### global variables
###
my $rollingLogFile="$SBX_INSTALL_LOG";
if (! -d "$SONUS_LOG_INSTALL_DIR") {
    $rollingLogFile="/root/sbx-install-log";
}
# SBX install/upgrade in progress key file
# Using /tmp for creating marker file so that it gets cleared on system reboot.
my $sbxInstallUpgradeInProgressKey="$SBX_INSTALL_UPGRADE_IN_PROGRESS";
my $stepScript="$STEP_COUNTER_SH";
my $maxStep="";
my $stagingDir="$SONUS_STAGING_DIR";
my $statusUpdater="$STATUS_UPDATER_PL";

my $upgradeBaseDir="$SONUS_LOG_UPGRADE_DIR";
my $upgradeDir="$SONUS_LOG_UPGRADE_CURRENT_DIR";
my $upgradeCheckDir="$SONUS_LOG_UPGRADE_PRECHECKS_DIR";
my $lswuLogFile="$SOFTWARE_UPGRADE_LOG";

if( -d $upgradeDir && ! -e $lswuLogFile )
{
    `touch $lswuLogFile`;
    `chmod 777 $lswuLogFile`;
}

# revert feature related files
my $gServiceName="sbx"; 
my $installUpgradeMarker="$INSTALL_UPGRADE_MARKER";

my $ceName=sbxCommonFunctions::getInstalledceName();
my $peerCeName=sbxCommonFunctions::getInstalledpeerCeName();

my $remoteLog="/var/log/remoteExecution.log";
my $shellHandle ="";
my $packageInstalled = 0;
$packageInstalled =  eval { require Expect };
if ( defined($packageInstalled) )
{
   require sbxPeerExpectUtils;
}

##############################################################################
#
# Open shell handle to peer server.
#
##############################################################################
sub openPeerHandle
{
    $packageInstalled =  eval { require Expect };
    if ( defined($packageInstalled) )
    {
      require sbxPeerExpectUtils;
      $shellHandle = sbxPeerExpectUtils::connectToShell($peerCeName, "2024" );
      if ($shellHandle =~ /failure/)
      {
        logMsg("Unable to login to shell...shellHandle=$shellHandle\n");
      }
    }
}

##############################################################################
# Directories to backup during LSWU
# this global array is visible and can be accessed
# by other top level scripts.
# Intially tried to take backup of "/opt/sonus/sbx" and "/var/log/sonus/sbx"
# during LSWU but that didn't work very well because of soft links that
# created trouble during restore (in case of revert)
##############################################################################
%dirBackup = ("$SONUS_TAILF_DIR",          "$HOME_COMMON/tailf.prev",
              "$SONUS_OPENCLOVIS_VAR_DIR", "$HOME_COMMON/openclovisVar.prev",
              "$SONUS_EMA_APACHE_DIR", "$HOME_COMMON/ema.prev"
             );

%dirPreRevertSave = ("$SONUS_TAILF_DIR",          "$SONUS_DIR/tailf.preRevert",
                     "$SONUS_OPENCLOVIS_VAR_DIR", "$SONUS_DIR/openclovisVar.preRevert",
                     "$SONUS_EMA_APACHE_DIR", "$SONUS_DIR/ema.preRevert"
                    );

##############################################################
# Function to run the service wrapper script.  We need to run
# it from the staging directory first.  But once we unpack the
# deb it is in the scripts dir.  And once we start the sbx it
# it removed from the staging dir.  Call it from the staging
# dir until it no longer exists there.
##############################################################
sub runServiceCmd
{
    my $svcName = shift;
    my $svcCmd = shift;
    my $useSystem = shift;
    my $background = shift;
    my $result;
    my $cmd;

    if ( -e $STAGING_SERVICE_SH )
    {
      $cmd = $STAGING_SERVICE_SH;
    }
    elsif ( -e $SERVICE_SH )
    {
      $cmd = $SERVICE_SH;
    }
    else
    {
      logMsg("Unable to find required service command wrapper");
      exit 1;
    }

    if( $useSystem )
    {
      if( $background )
      {
        system("$cmd $svcName $svcCmd &");
      }
      else
      {
        system("$cmd $svcName $svcCmd");
      }
    }
    else
    {
      $result = `$cmd $svcName $svcCmd`;
      chomp($result);
    }

    return "$result";
}

##############################################################
# Utility to log a message to STDOUT: prints time stamp also
##############################################################
sub logMsg
{
    my $msg = shift;
    my $skipRemote = 0;
    ($skipRemote = 1) if ( $shellHandle eq "" || $shellHandle =~ /failure/ );

    my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime(time);
    my $timeDate = sprintf "%4d-%02d-%02d %02d:%02d:%02d", 
                           $year+1900, $mon+1, $mday, $hour, $min, $sec;
    print(STDOUT "$timeDate: $msg\n");

    my $dateStr=`$DATE +"%b %d %T.%3N %Y"`;
    chomp($dateStr);
    `$ECHO -e "$dateStr: $msg" >> $rollingLogFile`;
    if ( -e $lswuLogFile )
    {
       `$ECHO -e "$dateStr $ceName: $msg" >> $lswuLogFile`;
       if ( defined($packageInstalled) && $packageInstalled eq "1" && $skipRemote eq 0 )
       {
          sbxPeerExpectUtils::executeCommand($shellHandle, "sh -c \"echo '$dateStr $ceName: $msg' >> $lswuLogFile\"");
       }
    }
}

###########################################################################################
# Utilities to log header and footer
# Similar 'shell' utilities are present in orca/install/sonusUtils.sh
# When making any changes to these methods, please update same methods in sonusUtils.sh
###########################################################################################
sub logStartUpdate
{
  my $caller = shift;
  my $rpmDbPath = "";

  if (-d "/root/.rpmdb")
  {
     $rpmDbPath = "--dbpath /root/.rpmdb";
  }
  
  logMsg("==========================================================================");
  logMsg("Start: Sonus sbx Installation (Initiated through $caller)");
  my $rbbnPkgs = "bios-update cnxipm cps ema lcamodules pamvalidator sbc sl";
  my $currentInstallations="";
  my $cps = `$DPKG -l cps 2>/dev/null | $GREP ^ii | $AWK '{ print \$2 }'`;
  if ($cps eq "")
  {
      $currentInstallations=`$RPM -qa --last $rpmDbPath`;
  } else {
      $currentInstallations=`$DPKG -l $rbbnPkgs 2>/dev/null | $GREP ^ii | $AWK 'BEGIN {  OFS = "_" }{ print \$2, \$3, \$4 }'`;     
  }
  logMsg("Currently Installed Version(s): \n$currentInstallations");
  logMsg("==========================================================================");
}

sub logEndUpdate
{
  my $rpmDbPath = "";

  if (-d "/root/.rpmdb")
  {
     $rpmDbPath = "--dbpath /root/.rpmdb";
  }

  logMsg("==========================================================================");
  logMsg("End: Sonus sbx Installation");
  my $rbbnPkgs = "bios-update cnxipm cps ema lcamodules pamvalidator sbc sl";
  my $currentInstallations="";
  my $cps = `$DPKG -l cps 2>/dev/null | $GREP ^ii | $AWK '{ print \$2 }'`;
  if ($cps eq "")
  {
      $currentInstallations=`$RPM -qa --last $rpmDbPath`;
  } else {
      $currentInstallations=`$DPKG -l $rbbnPkgs 2>/dev/null | $GREP ^ii | $AWK 'BEGIN {  OFS = "_" }{ print \$2, \$3, \$4 }'`;     
  }
  logMsg("Updated Version(s): \n$currentInstallations");
  logMsg("==========================================================================");
}

#########################################################################
# getTime: returns current time in hh:mm:ss without timeZone. 
#########################################################################
sub getTime
{
  my $time=`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print \$4}'`;
  chomp($time);
  return($time);
}

#########################################################################
# For a child and main process exits, so that the caller is not blocked
#########################################################################
sub runChild()
{
  my $pid;
  if ($pid = fork) 
  {
    # Parent exits so that the caller can keep going, if they didn't used '&'
    exit 1;
  } 
  elsif (defined $pid) 
  {
    # everything good, return
    logMsg("Child spawned, Creating new peer ssh handle.."); 
    openPeerHandle();
    chdir "/"; # so that uninstall/install don't cause any cwd errors
    return;
  }
  else
  {
    logMsg("Unable to fork child");
    exit 1;
  }
}


sub logUMMessage
{
    my $msg = shift;

    my $currStep = 0;

    # Towards the end of the upgrade, script gets moved from 
    # /opt/sonus/staging to /opt/sonus, so check before invoking the script
    if ( -e $stepScript ) {
      $currStep = `$stepScript -i`;
    }
    else {
      $currStep = `$SONUS_SBX_SCRIPT_DIR/stepCounter.sh -i`;
    }

    chomp($currStep);
    logMsg("UMMESSAGE:$currStep:$maxStep:$msg");
}

############################
# Make sure running as root
############################
sub checkUser()
{
    # Create the counter file and set max steps 
    $maxStep=`$GREP -r "logUMMessage" $SONUS_DIR/{staging,sbx/scripts}/* | $WC -l`;
    chomp($maxStep); 
    logMsg("got maxStep: $maxStep");
    `$stepScript -c -m $maxStep`;

    logUMMessage("Checking User...");

    if ($< != 0)
    {
        logMsg("Not running as user");
        exit 1;
    }
}

#####################################################################
# Stop SBX: "service sbx stop" and makes sure everything is stopped
# Exits, if not stopped -- but should never happen
#####################################################################
sub stopSbxService()
{

  # Remove SBX install/upgrade in progress marker file if existing.
  removeMarkerFile("$sbxInstallUpgradeInProgressKey");

  my $initServiceStatus = runServiceCmd("sbx","status",0,0);

  # Do not stop the service if it is not running.
  # This may cause a RID issue if sbxCleanup is called
  # when the os has been upgraded and the application has not.

  if ($initServiceStatus !~ /not running/)
  {
      my $sbcVer = "";
      my $cps = `$DPKG -l cps 2>/dev/null | $GREP ^ii | $AWK '{ print \$2 }'`;
      if ($cps eq "")
      {
          $sbcVer = `$RPM --dbpath /root/.rpmdb -qa | $GREP -e sbx -e sbc `;
          $sbcVer =~ s/^sb[cx]-//;
          $sbcVer =~ s/.x86_64//;
      } else {
          $sbcVer = `$DPKG -l sbc | $GREP ^ii | $AWK '{ print \$3 }'`;
          $sbcVer =~ s/^/V/;
      }
      chomp($sbcVer);

      if ($sbcVer lt "V03.00.00-R000" && $initServiceStatus =~ /not ready/)
      {
         ;
      }
      else
      {
         # capture the output to the log file (everything is redirected to a log)
         logMsg("Stopping SBX service...");
         runServiceCmd("sbx","stop",0,0);

         my $status = runServiceCmd("sbx","status",0,0);
         if ($status !~ /not running/)
         {
            logMsg("Unable to stop sbx service:\n$status");
            exit 1;
         }
      }
  }

  # Create SBX install/upgrade in progress marker file.
  createMarkerFile("$sbxInstallUpgradeInProgressKey", "$0");
}

##############################################################
# Creates a marker file
##############################################################
sub createMarkerFile
{
  my $markerFile = shift;
  my $keyWord = shift;
  my $fh = new FileHandle;

  logMsg("Creating marker file $markerFile with \"$keyWord\"...");

  if ($fh->open(">$markerFile"))
  {
    print $fh "$keyWord\n"; 
    close $fh;
  }
  else
  {
    logMsg("Unable to open marker file: $markerFile");
    return "failure";
  }
  return "success";
}

############################
# Cleans up marker file
############################
sub removeMarkerFile
{
  my $markerFile = shift;
  unlink($markerFile);
}

#################################################################
# Removes SBX deb/package, should be invoked after stopping SBX
#################################################################
sub removeSbxPackage()
{
  # Take a backup of drbd.conf
  # During LSWU, we need to maintain the same settings so that
  # sync continues
  fcopy("$ETC_DRBD_CONF", "$ETC_DRBD_CONF.preserve");

  my $sbxPkg = `$RPM --dbpath /root/.rpmdb -qa | $GREP -e sbx -e sbc`;
  chomp($sbxPkg);

  my $emaPkg = $sbxPkg;
  $emaPkg =~ s/^sb[cx]/ema/;
  logUMMessage("Removing package: $sbxPkg...");
  `$RPM --dbpath /root/.rpmdb -e --allmatches --force-debian sbx > /dev/null 2>&1`;
  `$RPM --dbpath /root/.rpmdb -e --allmatches --force-debian sbc > /dev/null 2>&1`;
  `$DPKG -P sbc > /dev/null 2>&1`;

  logUMMessage("Removing package: $emaPkg...");
  `$RPM --dbpath /root/.rpmdb -e --allmatches --force-debian ema > /dev/null 2>&1`;
  `$DPKG -P ema > /dev/null 2>&1`;

  $sbxPkg = `$RPM --dbpath /root/.rpmdb -qa | $GREP -e sbc -e sbx`;
  chomp($sbxPkg);
  $emaPkg = `$RPM --dbpath /root/.rpmdb -qa | $GREP ema`;
  chomp($emaPkg);

  my $sbcDebPkgName = `$DPKG -l sbc 2>/dev/null | $GREP ^ii | $AWK '{ print \$2 }'`;
  my $emaDebPkgName = `$DPKG -l ema 2>/dev/null | $GREP ^ii | $AWK '{ print \$2 }'`;
  if ($sbxPkg ne "" || $emaPkg ne "")
  {
     logMsg("Unable to remove sbc package(s): $sbxPkg and/or $emaPkg");
     return "failure";
  }
  if ($sbcDebPkgName ne "" || $emaDebPkgName ne "")
  {
     logMsg("Unable to remove sbc package(s): $sbcDebPkgName and/or $emaDebPkgName");
     return "failure";
  }
  return "success";
}

##############################################################
# Installs SBX deb/package: 
# should be invoked after removing old packages
##############################################################
sub installSbxPackage
{
  my $package = shift;
  $package =~ s/sbc-V/sbc_/;
  $package =~ s/.x86_64.tar.gz$/_amd64.deb/;

  # /opt/sonus/staging/sbc_10.01.00-A001_amd64.deb
  my $emaPackage = $package;
  $emaPackage =~ s/sbc_/ema_/;

  logUMMessage("Installing package: $package...");
  `$DPKG -i $package`;

  logUMMessage("Installing package: $emaPackage...");
  `$DPKG -i $emaPackage`;

  my $sbcPkgName = `$DPKG -l sbc 2>/dev/null | $GREP ^ii | $AWK '{ print \$2 }'`;
  if ($sbcPkgName eq "")
  {
     logMsg("Unable to install sbc package: $sbcPkgName");
     return "failure";
  }

  my $emaPkgName = `$DPKG -l ema 2>/dev/null | $GREP ^ii | $AWK '{ print \$2 }'`;
  if ($emaPkgName eq "")
  {
     logMsg("Unable to install ema package: $emaPkgName");
     return "failure";
  }
  return "success";
}

##############################################################
# Applies the settings of /opt/sonus/conf/sbx.conf
# Typically not required as part of LSWU but just in case
##############################################################
sub reInitSbxConf
{
  my $options = shift;
  logMsg("Calling sbxInit with options $options...");
  if (-e "$SBXCONF_FILE")
  {
    # don't redirect the output of sbxInit.sh, it is doing all logging on its own...
    # We do have to capture stderr however.  Unfortunately we cannot both capture it
    # and have it show on the terminal since any use of tee (process substitution and
    # tee) won't work as the tee command inherits any fds that are started and therefore
    # does not close properly, even though this script moves on (seen in install case
    # as shutdown debugging showed appInstall gone by sbxInstall still running with the
    # tee as a child of it).  Do the only thing we can: capture it in the log but
    # not on the terminal.
    my $result = system("$SBX_INIT_SH -f $SBXCONF_FILE $options 2>> $rollingLogFile)");
    if ( $result != 0 ) {
      return "failure";
    }
  }
  else
  {
    logMsg("Not found $SBXCONF_FILE, proceeding...");
  }
  # Don't need to do anything otherwise

  return "success";
}

##############################################################
# Copies the saved SBX CDB files to original place
# This should be invoked:
#   after installing SBX package/deb
#   and before starting SBX
#
# TBD: define the paths at a common place
##############################################################
sub restoreCdbFiles()
{
  my $oldPath = "$HOME_COMMON/tailf.prev/var/confd/cdb";
  my $newPath = "$SONUS_TAILF_CDB_DIR";

  logMsg("Restoring CDB files, oldPath: $oldPath, newPath: $newPath...");

  fcopy("$oldPath/A.cdb", "$newPath/A.cdb");
  fcopy("$oldPath/C.cdb", "$newPath/C.cdb");
  fcopy("$oldPath/O.cdb", "$newPath/O.cdb");

  # remove world permission
  `chmod o-rwx $newPath/A.cdb $newPath/C.cdb $newPath/O.cdb`;

  # restore confd configuration 
  system("$RESTORE_CONFD_CONFIG_SH");

  #my $buff = `ls -l $oldPath/A.cdb`;
  #logMsg($buff);
  #$buff = `ls -l $newPath/A.cdb`;
  #logMsg($buff);
}

##############################################################
# Copies the saved EMA files to original place
# This should be invoked:
#   after installing SBX package/deb
#   and before starting SBX
##############################################################
sub restoreEmaFiles()
{
  my $oldPath = "$HOME_COMMON/ema.prev";
  my $newPath = "$SONUS_EMA_APACHE_DIR";

  logMsg("Restoring EMA files, oldPath: $oldPath, newPath: $newPath...");

  fcopy("$oldPath/server.pem", "$newPath/server.pem");
  fcopy("$oldPath/server.key", "$newPath/server.key");
  fcopy("$oldPath/caCert.pem", "$newPath/caCert.pem");

  logMsg("Reload apache2 configuration...");
  runServiceCmd("apache2","reload",0,0);
}

##############################################################
# Invokes individual methods to restore data after installing
# new SBX package
##############################################################
sub restoreData()
{
  logUMMessage("Restoring Data...");
  # restore CDB files
  restoreCdbFiles();

  # restore EMA files
  restoreEmaFiles();
}


##############################################################
# Returns pid of the requested process, 
# 0 if the process doesn't exist
# parses output of 'service sbx status'
##############################################################
sub getProcessPid
{
  my $name = shift;
  my @status = runServiceCmd("sbx","status",0,0);
  my $pid = 0;

  foreach $process (@status)
  {
    if ($process =~ /$name/ && $process =~ /running/)
    {
      ($blah, $namePidRest) = split(/ \(/, $process);
      ($namePid, $blah) = split(/\) /, $namePidRest);
      ($blah, $pid) = split(/\s+/, $namePid);
      return $pid;
    }
  }
  return $pid;
}

##############################################################
# Starts SBX and makes sure that it is in stable state
# Stability is confirmed by checking for CHM process's pid
##############################################################
sub startSbxService()
{

  # Remove SBX install/upgrade in progress marker file before starting SBX.
  removeMarkerFile("$sbxInstallUpgradeInProgressKey");

  # Restart CPS if needed.
  my $cpsStatus = runServiceCmd("cps","status",0,0);
  chomp($cpsStatus);
  if ($cpsStatus !~ /is initialized/)
  {
    logMsg("restarting cps service");
    runServiceCmd("cps","start",1,1);
    sleep(2);
  }

  logMsg("Starting SBX service...");
  # capture the output to the log file (everything is redirected to a log)
  # Using system() call instead of backticks. Backticks opens a pipe and causes defunct child process
  # on specific scenarios like service cps restart.

  runServiceCmd("sbx","start",1,0);

  # Get CHM process's pid, sleep for another 10 sec, get it again and compare
  # if matching, everything is fine, if not getting restarted, so report and revert
  sleep(80);
  my $firstPid = getProcessPid("ChmProcess");
  sleep(10);
  my $secondPid = getProcessPid("ChmProcess");
  sleep(10);
  my $thirdPid = getProcessPid("ChmProcess");

  logMsg("Checking pids..., firstPid: $firstPid, secondPid: $secondPid, thirdPid: $thirdPid");
  if ($firstPid == 0 || $firstPid != $secondPid || $secondPid != $thirdPid)
  {
    logMsg("Failed to start SBX service, firstPid: $firstPid, secondPid: $secondPid, thirdPid: $thirdPid");
    return "failure";
  }

  return "success";
}

##############################################################
# Perform pre-installation/pre-upgrade checks
##############################################################
sub preInstallCheck()
{
  logMsg("Running pre-installation checks...");

  my $result=system("$PRE_INSTALL_CHECK_SH");
  if( $result != 0 ) {
    return "failure";
   }

   return "success";
}
##############################################################
# Perform pre-installation/pre-upgrade checks
##############################################################
sub checkDiskforRevert()
{
  my $case = shift;
  my $command = shift;
  my $lvcount = shift;
  my $cnt = chown $uid, $gid, 'root', 'root';

  logMsg("running checkDiskforRevert()..");
  # Create Install/Upgrade marker, it will be used by sbx init script after reboot.
  # Check whether /home/common directory exists
  if ( ! -d "$HOME_COMMON" ) {
	  logMsg("creating /home/common directory");
	  mkdir -p $HOME_COMMON;
  } 
  `$ECHO "case=$case" > $installUpgradeMarker`;
  `$ECHO "command=$command" >> $installUpgradeMarker`;
  chown $uid, $gid, $installUpgradeMarker;
  logMsg("created and changed ownership of installUpgradeMarker file");

  if( $lvcount == 1)
  {
    prepareDiskforRevertSupport();
  } 
  else
  {
    logMsg("ERROR! Wrong number of LVMs detected! Manual Intervention required..Number of LVMs=$lvcount");
    exit 1; 
  }
}

sub prepareDiskforRevertSupport()
{
  system("pushd $SONUS_DIR; $RM -fr *.rpm *.deb *.md5 *.sha256 os_up*; popd");
  system("$REVERT_UTIL_SH \"sda7\"");

  if( -e $lswuLogFile )
  {
    system("$SWITCH_BOOT_SH \"sda7\" &>> $lswuLogFile");
  }
  else
  {
    system("$SWITCH_BOOT_SH \"sda7\"");
  }

  logMsg("\n");
  logMsg("********************************************************************************************************\n");
  logMsg("PLEASE NOTE:\n");
  logMsg("\n");
  logMsg("THIS SERVER WILL BE TAKEN OUT OF SERVICE AND REBOOTED TO PERFORM LIVE SOFTWARE UPGRADE.\n");
  logMsg("\n");
  logMsg("********************************************************************************************************\n");
  logMsg("\n");

  stopSbxService();

  # Find state of drbd, if it is Inconsistent, wait for some time to complete partial sync(if any).
  my $count=0;
  while ($count < 3)
  {
    my $dState=`$DRBDADM dstate mirror| $AWK -F "/" {'printf \$1'}`;
    my $drbdStatus = `$CAT /proc/drbd`;
    if ( $dState =~ /Inconsistent/ )
    {
      logMsg("DRBD mirror state is $dState, waiting for 5 secs...\n"); 
      logMsg("DRBD status (waiting in transient state): $drbdStatus");
      sleep(5);
    }
    else
    {
      logMsg("DRBD mirror state is $dState\n");
      logMsg("DRBD status (about to stop DRBD): $drbdStatus");
      `$DRBDADM disconnect mirror`;
      sleep(1);
      `$DRBDADM secondary mirror`;
      `$DRBDADM detach mirror`;
      runServiceCmd("drbd","stop",0,0);
      last;
    } 
    $count++;
  }

  if ( $count eq 3 )
  {
    logMsg("Warning! DRBD mirror state is not UpToDate");
  }
  `$PERL $statusUpdater -r Reboot -s Status -v "inProgress"`;
  my $time=getTime();
  `$PERL $statusUpdater -r Reboot -s StartTime -v "$time"`;
  `$PERL $statusUpdater -r Reboot -s Reason -v "Rebooting_server_for_partitioning"`;

  # Do NOT change /sbin/reboot to $REBOOT as sonusCommands in staging refers to new reboot script which doesn't exist yet
  logMsg("Rebooting via /sbin/reboot..");
  system("/sbin/reboot");
  exit 0; 
}

sub backupHomeDir()
{
  # Take backup of /home, this should exclude oracle and other common directories.
  logMsg("Taking backup of /home(to be used while reverting)..");
  my $curDir=getcwd;
  chdir("/home/");
  `$TAR -czf home_backup.tar.gz \`$LS -d * | $EGREP -v "ora|common|libvirt|log|external|staging|sftproot|etc"\``;
  if ( -e "$SONUS_DIR/home_backup.tar.gz" )
  {
     `$RM -f $SONUS_DIR/home_backup.tar.gz`;
  }
  if ( -e "$SONUS_BACKUP_DIR/home_backup.tar.gz" )
  {
     `$RM -f $SONUS_BACKUP_DIR/home_backup.tar.gz`;
  }
  `$MV -f home_backup.tar.gz $HOME_BACKUP_DIR/`;
  `$CHMOD 640 $HOME_BACKUP_DIR/home_backup.tar.gz`;

  # Backup sonuscert.p12 cerfication file to be used during revert
  if ( -e "$SONUS_EXTERNAL_DIR/sonuscert.p12" )
  {
     `$CP -pf $SONUS_EXTERNAL_DIR/sonuscert.p12 $HOME_BACKUP_DIR/`;
  }
  chdir($curDir);
}

sub saveCommonDirPerms()
{
  if ( -e "$HOME_BACKUP_DIR/common_dir_perms" )
  {
     `$RM -f $HOME_BACKUP_DIR/common_dir_perms`;
  }
  my @dirArray=("$SONUS_EXTERNAL_DIR/", "$SONUS_STAGING_DIR/", "$HOME_ORACLE/", "/home/orasql/", "/home/oraadmin/", "/home/oradata/", "/home/oraidx/", "/home/orasys/" );
  foreach(@dirArray)
  {
    my $dir=$_;
    if( -d "$dir")
    {
      logMsg("Saving ownership/permission info of $dir..");
      `$STAT -c "%n %u %g %a" $dir >> $HOME_BACKUP_DIR/common_dir_perms`;
    }
    else
    {
      logMsg("$dir doesnt exist!");
    }
  }
  # /var/log/sonus/* permissions
  my $sonusDirList=`$FIND $SONUS_LOG_DIR/ -type d -print`;
  chomp($sonusDirList);
  my @arr=split /^/,$sonusDirList;
  foreach(@arr)
  {
    my $dir=$_;
    chomp($dir);
    if( -d "$dir")
    {
      `$STAT -c "%n %u %g %a" $dir >> $HOME_BACKUP_DIR/common_dir_perms`;
    }
  }
  # Save the permissions only if the dir exists.
  if ( -d "$HOME_SFTPROOT")
  {
    # Save the ownership & permissions for confd users.
    # Exclude itself, dirs which are already added like external, staging or dirs containing system logs like log dir.
    $sonusDirList=`$FIND $HOME_SFTPROOT/* -type d ! \\( -path $HOME_SFTPROOT/external -prune -o -path $HOME_SFTPROOT/staging -prune -o -path $HOME_SFTPROOT/log -prune \\) -print`;
    chomp($sonusDirList);
    @arr=split /^/,$sonusDirList;
    foreach(@arr)
    {
      my $dir=$_;
      chomp($dir);
      if( -d "$dir")
      {
        `$STAT -c "%n %u %g %a" $dir >> $HOME_BACKUP_DIR/common_dir_perms`;
      }
    }
  }
}

################################################################
## subroutine to determine the major version of either the local
## or remote software.  NOTE: it gets the version from swinfo, so
## the result depends on whether called before or after the new
## software is installed.
## NOTE: older versions of software report the version as
## sbc-V##......  We need to handle deleting the 'sbc-' in case
## it is there.
#################################################################
sub getSWVersion()
{
  my $localRemote=shift;
  my $swinfoFlag=($localRemote eq "local") ? "-l" : "-R";
  my $version=`$SWINFO_SH $swinfoFlag | $GREP SBC: | $AWK '{print \$2}' | $SED -e 's/sbc-//' | $AWK -F. '{print \$1}' | $SED -e 's/V//'`;
  chomp($version);
  # add 0 to force perl to treat it numerically rather than as a string (e.g. 5 instead of 05)
  return $version+0;
}


##############################################################
# Check syncStatus, returns good if syncCompleted.
##############################################################

sub checkSyncStatus
{
  my $syncStatus="good";
  my $cmd = "show table system syncStatus";
  my $out = sbxPeerExpectUtils::runCliCommand($shellHandle, "$cmd");
  my @lines = split /\n/, $out;
  foreach my $line (@lines)
  {
    if ( $line =~ /Policy Data/ && $line !~ /syncCompleted/)
    {
       logMsg("checkSyncStatus: Policy Data is not in Sync..");
       $syncStatus="bad";
    }
    elsif ( $line =~ /Disk Mirroring/ && $line !~ /syncCompleted/)
    {
       logMsg("checkSyncStatus: Disk Mirroring is not in Sync..");
       $syncStatus="bad";
    }
    elsif ( $line =~ /Configuration Data/ && $line !~ /syncCompleted/)
    {
       logMsg("checkSyncStatus: Configuration Data is not in Sync..");
       $syncStatus="bad";
    }
    elsif ( $line =~ /Registration Data/ && $line !~ /syncCompleted/)
    {
       logMsg("checkSyncStatus: Call/Registration Data is not in Sync..");
       $syncStatus="bad";
    }
  }
  return($syncStatus);
}

##############################################################
# postUpgradeChecks: run after standby and active upgrade. 
##############################################################
sub postUpgradeChecks
{
  my $role=shift;
  my $regex="";
  my $count=0;
  my $syncStatus="unknown";
  my $installedRole=`$GREP role $SBXCONF_FILE | $AWK -F "=" '{print \$2}'|$AWK '{print \$1}'`;
  chomp($installedRole);
  if( $installedRole eq "2" )
  {
    $regex="secondaryUpgradeStatus";
  }
  else
  {
    $regex="primaryUpgradeStatus";
  }

  logMsg("In postUpgradeChecks(), role=$role, installedRole=$installedRole,regex=$regex");

  if( $role eq  "standby")
  {
    `$PERL $statusUpdater -s UpgradeStatus -v "postStandbyUpgradeChecks"`;
  }
  elsif( $role eq  "active")
  {
    `$PERL $statusUpdater -s UpgradeStatus -v "postActiveUpgradeChecks"`;
  }
  openPeerHandle();
  logMsg("Starting monitor loop..");
  my $cliStateGood="false";
  while(1)
  {
    if ( $cliStateGood  =~ /false/ )
    {
      logMsg("Checking CLI upgrade status..");
      my $cmd = "show table system softwareUpgradeStatus";
      my $out = sbxPeerExpectUtils::runCliCommand($shellHandle, "$cmd");
      chomp($out);
      my @lines = split /\n/, $out;
      foreach my $line (@lines)
      {
        if ( $line =~ /$regex/ && $line !~ /upgraded/)
        {
           logMsg("postUpgradeChecks: regex=$regex, line=$line");
           if ( $count eq "80")
           {
             logMsg("postUpgradeChecks: Waited for 40 minutes but CLI $regex is still not set to 'upgraded'. Failing upgrade and exiting.."); 
             $syncStatus=checkSyncStatus();
             if ( $syncStatus =~ /good/ )
             {
                 logMsg("postUpgradeChecks: All modules are in Sync..");
                 `$PERL $statusUpdater -r PostUpgrade -s Reason -v "Timedout_waiting_for_CLI_upgrade_state_change"`;
                 `$PERL $statusUpdater -s Reason -v "Timedout_waiting_for_CLI_upgrade_state_change.Possible_Recovery_Action_is:To_Revert_or_To_contact_Sonus_support."`;
             }
             else
             {
                 logMsg("postUpgradeChecks: CLI SyncStatus is not set to 'SyncCompleted'..");
                 `$PERL $statusUpdater -r PostUpgrade -s Reason -v "Timedout_waiting_for_CLI_SyncCompletion"`;
                 `$PERL $statusUpdater -s Reason -v "Timedout_waiting_for_CLI_SyncCompletion.Possible_Recovery_Action_is:To_Revert_or_To_contact_Sonus_support."`;
             }
             my $time=getTime();
             `$PERL $statusUpdater -r PostUpgrade -s EndTime -v "$time"`;
             `$PERL $statusUpdater -r PostUpgrade -s Status -v "failed"`;
             exit(1);
           }
           else
           {
             logMsg("postUpgradeChecks: count=$count. CLI upgrade state is not updated yet.. Continue monitoring..");
           }
        }
        elsif($line =~ /$regex/ && $line =~ /upgraded/)
        {
           logMsg("postUpgradeChecks: regex=$regex, line=$line");
           logMsg("postUpgradeChecks: CLI upgrade state is good.");
           $cliStateGood="true";
        }
      }
    }
    if ( $cliStateGood  =~ /true/ )
    {
      logMsg("postUpgradeChecks: Checking syncStatus..");
      $syncStatus=checkSyncStatus();
      if ( $syncStatus =~ /good/ )
      {
        logMsg("postUpgradeChecks: All entities are in Sync..");
        # exit loop
        last;
      }
      else
      {
        if ( $count eq "100")
        {
           logMsg("postUpgradeChecks: Waited for 50 minutes but CLI SyncStatus is still not set to 'SyncCompleted' Failing upgrade and exiting..");
           my $time=getTime();
           `$PERL $statusUpdater -r PostUpgrade -s EndTime -v "$time"`;
           `$PERL $statusUpdater -r PostUpgrade -s Reason -v "Timedout_waiting_for_CLI_SyncCompletion"`;
           `$PERL $statusUpdater -s Reason -v "Timedout_waiting_for_CLI_SyncCompletion.Possible_Recovery_Action_is:To_Revert_or_To_contact_Sonus_support."`;
           `$PERL $statusUpdater -r PostUpgrade -s Status -v "failed"`;
           exit(1);
        }
        else
        {
           logMsg("postUpgradeChecks: count=$count. One or more entities are not in sync.");
        }
      }
    }
    if ( $count eq "120")
    {
	    logMsg("postUpgradeChecks: Waited for 60 minutes but either upgrade status is not updated or syncStatus is not good. Failing upgrade and exiting..");
        my $time=getTime();
        `$PERL $statusUpdater -r PostUpgrade -s EndTime -v "$time"`;
        `$PERL $statusUpdater -r PostUpgrade -s Reason -v "Timedout_waiting_for_postUpgradeChecks_toComplete"`;
        `$PERL $statusUpdater -s Reason -v "Timedout_waiting_for_postUpgradeChecks_toComplete.Possible_Recovery_Action_is:To_Revert_or_To_contact_Sonus_support."`;
        `$PERL $statusUpdater -r PostUpgrade -s Status -v "failed"`;
        exit(1);
    }
    $count ++;
    logMsg("postUpgradeChecks(): Continue monitoring(wait period=30s)..");
    sleep(30);
  }
  logMsg("postUpgradeChecks: Complete.");
  `$PERL $statusUpdater -r PostUpgrade -s Status -v "Complete"`;
  my $time=getTime();
  `$PERL $statusUpdater -r PostUpgrade -s EndTime -v "$time"`;
  `$PERL $statusUpdater -r PostUpgrade -s Reason -v "Successful_Completion"`;
  logMsg("--------Completed POST-UPGRADE--------");
}

sub peerCommand
{
    $command=$_[0];
    $peerCfg=1;
    $peerIpAddr="";
    $cnxExt="expect";
    $haPortName="bond0";
    my $cfgScript=$SBXCONF_FILE;

    if( ! -e "$cfgScript" )
    {
      $cfgScript="$SBX_CONF";
    }

    `$_IP addr show $haPortName 2> /dev/null | $GREP -q "inet " `;
    $peerCfg=$?;

    if ( $peerCfg != 0 )
    {
        $haPortName="ha0";
        `$_IP addr show $haPortName  2> /dev/null | $GREP -q "inet "`;
        $peerCfg=$?;
    }
    if ( -e $cfgScript )
    {
        `$CAT $cfgScript | $GREP "^peerCeName" | $GREP -q "none\$"`;
        if ( $? == 0 )
        {
            # this is a stand alone system
            $peerCfg=1;
        }
        if ( $peerCfg == 0 )
        {
            if ( -e  "$SONUS_PEER_CNX_SH" )
            {
                $cnxExt="sh";
                $tmpPeerServer=`$_IP addr show $haPortName | $GREP "inet " | $AWK '{print \$2}' | $AWK -F/ '{print \$1}'`;
                chomp $tmpPeerServer;
                if ( $tmpPeerServer eq "169.254.99.1" )
                {
                    $peerIpAddr = "169.254.88.1";
                }
                elsif ( $tmpPeerServer eq "169.254.88.1" )
                {
                    $peerIpAddr="169.254.99.1";
                }
                else
                {
                    # could not resolve peer ip address
                    $peerCfg=1;
                }
            }
            elsif ( ! -e  "$SONUS_PEER_CNX_EXPECT" )
            {
                # could not find the peer connect script
                $peerCfg=1;
            }
        }
    }
    system("$SONUS_SBX_SCRIPT_DIR/sonusPeerCnx.$cnxExt $peerIpAddr $command");
}
