#!/bin/bash
#############################################################
#
# Copyright (c) 2009 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# sonusUtils.sh
#
# Mark St. Pierre
# 12/4/09
#
# Module Description:
# Common Sonus bash script utilities.
#    
#############################################################

# Sourcing Command Variables
if [ -e /opt/sonus/staging/sonusCommands.sh ]
then
    source /opt/sonus/staging/sonusCommands.sh
elif [ -e /opt/sonus/bin/sonusCommands.sh ]
then
    source /opt/sonus/bin/sonusCommands.sh
else
    /bin/echo "Could not locate sonusCommands.sh Exiting..."
    exit 1
fi

if [ -e /opt/sonus/staging/sonusCommonFiles.sh ]
then
    source /opt/sonus/staging/sonusCommonFiles.sh
elif [ -e /opt/sonus/bin/sonusCommonFiles.sh ]
then
    source /opt/sonus/bin/sonusCommonFiles.sh
else
    $ECHO "Could not locate sonusCommonFiles.sh Exiting..."
    exit 1
fi
if [ -e /opt/sonus/staging/personality.sh ]
then
    source /opt/sonus/staging/personality.sh
elif [ -e /opt/sonus/sbx/scripts/personality.sh ]
then
    source /opt/sonus/sbx/scripts/personality.sh
else
    $ECHO  "Could not locate personality.sh Exiting..."
    exit 1
fi

log_message() {
    $ECHO "$(date +'%b %d %H:%M:%S.%3N %Y') $1"
}

sonusRoot=$SONUS_DIR
gServiceName=sbx
sbxInstallRoot=$SONUS_SBX_DIR
stagingDir=$SONUS_STAGING_DIR
updateLogFile=$stagingDir/update.out                                    # To redirect logs from stdout/stderr to log file
cpsIntfFile=$CPS_INTERFACES

if [[ $POD_TYPE == "ISBC" ]]; then
   defaultMgt='isbc_default_mgt0'
   mgtIpForPod='MANAGEMENT_IP_FOR_ISBC_PODS'
else
   defaultMgt='oam_default_mgt0'
   mgtIpForPod='MANAGEMENT_IP_FOR_OAM_PODS'
fi
# Sourcing Common Utils for Validation
# Must be builtin pwd instead of /bin/pwd
PROG_DIR=$(cd "$($DIRNAME "${BASH_SOURCE[0]}" )" && pwd)
if [ $PROG_DIR = "$stagingDir" ]
then
    source $STAGING_SONUS_COMMON_UTILS_SH
else
    source $SONUS_COMMON_UTILS_SH
fi

# source functions that were split out for more general usage
source $PROG_DIR/cfgValidation.sh

sonusConfigFile=$SBXCONF_FILE

# Marker file for appInstall.sh, sbxUpdate.sh script execution
scriptMarkerFile=$SONUS_INSTALLUPGRADE_LOG_DIR/.parentScriptMarker
maxStep=""
stepScript=$STEP_COUNTER_SH

# Revert related stuff
installUpgradeMarker=$INSTALL_UPGRADE_MARKER
updateP2RpmdbMarker=$SONUS_INSTALLUPGRADE_LOG_DIR/.updateP2RpmdbMarker
pmMarkerFile="$SBX_INSTALL_UPGRADE_TXT"

upgradeBaseDir=$SONUS_LOG_UPGRADE_DIR
upgradeCheckDir="$SONUS_LOG_UPGRADE_PRECHECKS_DIR";
preUpgradeCheckMarker=$upgradeCheckDir/preUpgradeCheckStatus
statusUpdater=$STATUS_UPDATER_PL
# Usage of PROG is to print only the filename along with log messages
PROG=${0##*/}

if [[ -e $SBX_PEER_EXPECT_PL ]];then
    sbxPeerExpect=$SBX_PEER_EXPECT_PL
else
    sbxPeerExpect=$SONUS_BIN_SBX_PEER_EXPECT_PL
fi

lswuLogFile=$upgradeBaseDir/latest/softwareUpgrade.log

ret=$(isCnf)
if [ $ret == $KUBE_ENV ]; then
    prodString=""
    hwType=""
    uuid=" "`$CAT /etc/podinfo/uid`
    cset_prefix=$($ENV | $GREP "\<MY_POD_NAME\>" | $AWK -F= '{print $2}')
else
    uuid=`$GREP "UUID" $DMIDECODE_OUT | $AWK -F: '{print$2}' | $TR [:lower:] [:upper:]`
fi

if [ $ret == $DOCKER_ENV ]; then
    hname=" DOCKER"
elif [ $ret == $KUBE_ENV ]; then
    hname=" KUBERNETES"
else
    hname=`$GREP "Product Name" $DMIDECODE_OUT | $HEAD -1 | $AWK -F: '{print $2}'`
fi 

if [[ $hwType != "ConnexIP" && $hwType != "ConnexIP5200" && $hwType != "ConnexIP5100" && $hwType != "SBC5400" && $hwType != "SBC7000" ]]; then
    hwType="ConnexIP5000"
fi

# Initialize hardware sub type
getHwSubType()
{
   hostSubType=Unknown
   if [ -e $HOST_SUB_TYPE ]; then
     hostSubType=`$CAT $HOST_SUB_TYPE`
   fi
   $ECHO $hostSubType
}

hostSubType=$(getHwSubType)

# Returns the "application" used by this SBC.
function getApplicationType
{
    # Default to SBC type
    appType="sbc"
    if [ -e $ETC_APPLICATION ]; then
        appType=`$CAT $ETC_APPLICATION | $TR -d '\r' | $TR -d '\n' | $TR [:upper:] [:lower:]`
    fi
    $ECHO $appType
}

# Variables used to get current role for the node
currentRole=

# SBX install/upgrade in progress key file
# Using /tmp for creating marker file so that it gets cleared on system reboot.
sbxInstallUpgradeInProgressKey=$SBX_INSTALL_UPGRADE_IN_PROGRESS


if [ -d $SONUS_LOG_INSTALL_DIR/ ]
then
    gRollingAppInstallLogFile="$SBX_INSTALL_LOG"
else
    gRollingAppInstallLogFile="/root/$gServiceName-install-log"
fi

# needed for upgrades from 1.8.x os's
rpmDbPath=""  
if [ -d /root/.rpmdb ]; then
   rpmDbPath="--dbpath /root/.rpmdb"
fi

# Source the process_names.sh script (if present) to get LAST_CSI variable
# which is used to get current role of the node.
if [ -f $PROCESS_NAMES_SH ]; then
  . $PROCESS_NAMES_SH
fi

# Functions
upgradeGroupExists()
{
    if $GREP -q upgrade /etc/group; then
        return 0;
    fi
    return 1
}

testLinuxSshPort()
{
   #START !/usr/local/bin/expect
   expect - << EndMark
   spawn ssh -p $sshLinuxPort $user@$host
   set timeout 15
   # move along, answering "yes" and giving password
   set needrepeat 1
   sleep 1
   while { \$needrepeat != 0 } {
      set needrepeat 0
      expect {
         "assword" {
            send "$pass\r"
         }
         "re you sure you want" {
            send "yes\r"
            set needrepeat 1
         }
         "Connection refused" {
            send \003
            exit 3
         }
         timeout {
            send_user "\n$program failed to connect!\n"
            exit 1
         }
      }
   }

   # Now check if server is CLI or Linux...
   expect {
      "]# " {
         send \003
         exit 0
      }
      "Permission denied" {
         send \003
         exit 3
      }
      timeout {
         send \003
         exit 1
      }
   }
EndMark
   #END !/usr/local/bin/expect
}

sbxLinuxCmd()
{
   # param 1 is the Linux command arg

   #START !/usr/local/bin/expect
   expect - << EndMark
   spawn ssh -p $sshLinuxPort $user@$host
   set timeout 1200
   # move along, answering "yes" and giving password
   set needrepeat 1
   sleep 1
   while { \$needrepeat != 0 } {
      set needrepeat 0
      expect {
         "assword" {
            send "$pass\r"
         }
         "re you sure you want" {
            send "yes\r"
            set needrepeat 1
         }
         timeout {
            send_user "\n$program failed to connect!\n"
            exit 1
         }
      }
   }
   
   # Perform service sbx action
   expect "]# "
   send -- "$1\n"
   expect {
      "error: Failed dependencies:" {
          exit 1
      }
      "ERROR: Not enough available disk space" {
          exit 1
      }
      "ERROR: Disk read speed failed" {
          exit 1
      }
      "No such file or directory" {
          exit 1
      }
      "error: " {
          exit 1
      }
      "ERROR: Base MAC address" {
          exit 1
      }
      "]# " {
         exit 0
      }
   }
EndMark
  #END !/usr/local/bin/expect
  return $?;
}

scpDir()
{
   # param 1 is source dir full path name
   # param 2 is destination dir full path name
   #START !/usr/local/bin/expect
   expect - << EndMark
   spawn scp -P $sshLinuxPort -r $1 $user@$host:$2
   set timeout 1200
   # move SCP along, answering "yes" and giving password
   set needrepeat 1
   sleep 1
   while { \$needrepeat != 0 } {
        set needrepeat 0
        expect {
                "assword" {
                        send "$pass\r"
                }
                "re you sure you want" {
                        send "yes\r"
                        set needrepeat 1
                }
                timeout {
                        send_user "\n$program failed to connect!\n"
                        exit 1
                }
        }
   }
   expect eof
EndMark
   #END !/usr/local/bin/expect
}

scpFile()
{
   # param 1 is source file full path name
   # param 2 is destination file full path name

   #START !/usr/local/bin/expect
   expect - << EndMark

   set fileargs [split {$1} ]
   set filelist [list]
   foreach globpat \$fileargs {
      set globlist [glob \$globpat]
      foreach globfile \$globlist {
        lappend filelist \$globfile
      }
   }

   set spawncmd "spawn scp -P $sshLinuxPort [join \$filelist] $user@$host:$2"
   set result [eval \$spawncmd]
   if { \$result == 0 } {
    send_user "\nSCP program $scpprog failed to start!\n"
    exit 1
   }

   set timeout 1200
   # move SCP along, answering "yes" and giving password
   set needrepeat 1
   sleep 1
   while { \$needrepeat != 0 } {
        set needrepeat 0
        expect {
                "assword" {
                        send "$pass\r"
                }
                "re you sure you want" {
                        send "yes\r"
                        set needrepeat 1
                }
                timeout {
                        send_user "\n$program failed to connect!\n"
          exit 1
                }
        }
   }
   expect eof
EndMark
   #END !/usr/local/bin/expect
}

###############################################################
#
# rmtCmd()
#   run command on remote sbx system
#
###############################################################
rmtCmd()
{
   # param 1 is the Linux command arg

   #START !/usr/local/bin/expect
   expect - << EndMark
   spawn ssh -p $sshLinuxPort $user@$host
   set timeout 15
   # move along, answering "yes" and giving password
   set needrepeat 1
   sleep 1
   while { \$needrepeat != 0 } {
      set needrepeat 0
      expect {
         "assword" {
            send "$pass\r"
         }
         "re you sure you want" {
            send "yes\r"
            set needrepeat 1
         }
         timeout {
            send_user "\n$program failed to connect!\n"
            exit 1
         }
      }
   }
   
   # Perform service sbx action
   expect "]# "
   send -- "$1\n"
   expect "]# "
   #expect eof
EndMark
   #END !/usr/local/bin/expect
}

validateHW5000()
{
   minMemory="12326812"

   if [[ $hostType == "ConnexIP5000" ]]; then
      minMemory="4000000"
   fi

   # Validate total memory...
   ret=$(isCnf)
   if [ $ret == $DOCKER_ENV ]; then
       totalMem=`$CAT /etc/podinfo/memory`
   else
       totalMem=`$CAT /proc/meminfo | $GREP MemTotal | $AWK '{print $2}'`
   fi
   if [ $totalMem -lt $minMemory ]; then
      $ECHO -e "Total Memory = $totalMem, minimum expected = $minMemory"
      return 1
   fi
   return 0
}

# function to remove all markers while exiting on error.
cleanupAllMarkers()
{
  log=$stagingDir/cleaupMarkers.log
  $ECHO > $log
  markerArray="$SBX_INSTALL_UPGRADE_IN_PROGRESS $upgradeBaseDir/preUpgradeCheckActive.key $LIVE_UPGRADE_ACTIVE_KEY $SONUS_INSTALLUPGRADE_LOG_DIR/.osUpgradeRevertStatus $NP_MODE_UPDATE $PERFORM_MODEL_UPDATE $LSWU_IN_PROGRESS $SBX_INSTALL_UPGRADE_TXT $INSTALL_UPGRADE_MARKER $SONUS_INSTALLUPGRADE_LOG_DIR/.updateP2RpmdbMarker"
  for f in $markerArray
  do
    if [ -e $f ];then
       $ECHO "cleanupAllMarkers(): cat $f" >> $log
       $CAT $f >> $log 
       $ECHO "" >> $log
       $ECHO "cleanupAllMarkers(): Removing $f.." >> $log
       $RM -fr $f
    fi
  done
}

# provide an exit function so all exits use
# common functionality
# Usage:
#    trapExit exitValue
# param 1: result
# param 2: result file
trapExit()
{
    result=$1
    resultFile=$2
    if [ -n "$resultFile" ]; then
       $ECHO -n $result > $resultFile
    fi
    if [ $result != 0 ];then
       # Assumption non-zero exits as error exits.
       # Cleanup markers if any
       cleanupAllMarkers

       # If it is a case of LSWU, we need to update status file with failure.
       if [ -e "$LIVE_UPGRADE_ACTIVE_KEY" ];then
          currStepNo=$($PERL $statusUpdater -g CurrentStep)
          currStep=$($PERL $statusUpdater -r $currStepNo -g Step)
          $PERL $statusUpdater -r $currStep -s Status -v "failed"
       fi
    fi
    exit $result
}

# log a message to both stdout and a log file: prints time stamp also
# don't put the timestamp on stdout....
# param 1: log file
# param 2: the message to log
# NOTE: a log file name of UNSET means that we should NOT write to
# the log file...
logMsg()
{
    local logFile=$1
    local logMsg=${@:2}

    dateStr=`$DATE +"%b %d %T.%3N %Y"`
    $ECHO -e "$logMsg"
    if [ -n "$logFile" ]; then
       $ECHO -e "$dateStr: $logMsg" >> $logFile
    fi

    if [ ! -z "$gRollingAppInstallLogFile" ]; then
       $ECHO -e "$dateStr: $logMsg" >> $gRollingAppInstallLogFile
    fi
    if [ -e "$LIVE_UPGRADE_ACTIVE_KEY" ];then
       # Set the permissions for lswuLogFile
       if [ ! -e "$lswuLogFile" ];then
          $TOUCH $lswuLogFile
          $CHMOD g+w  $lswuLogFile
          $PERL $sbxPeerExpect "$TOUCH $lswuLogFile" "shell"
       fi
       if upgradeGroupExists; then
          $CHMOD g+w  $lswuLogFile
          $PERL $sbxPeerExpect "$CHMOD g+w $lswuLogFile" "shell"
       else
          $CHMOD 777 $lswuLogFile
       fi
       $ECHO "$dateStr $ceName: $logMsg" >> $lswuLogFile
       $PERL $sbxPeerExpect "$ECHO -e '$dateStr $ceName: $logMsg' >> $lswuLogFile" "shell"
    fi
}

# Utility to log preInstallCheck events.
preInstallLogMsg()
{
    local logMsg=$1
    local userMsg=$2
    dateStr=`$DATE +"%b %d %T.%3N %Y"`

    # echo the message, this is redirected by SM/caller to capture and display to user
    if [[ "$userMsg" == "user" ]]; then
       $ECHO -e "$dateStr $ceName: $logMsg" 
    fi

    if [ -z ${preUpgradeCheckLogs+x} ]; then
       $ECHO -e "preUpgradeCheckLogs is not set!"
       preUpgradeCheckLogs=$stagingDir/preUpgradeCheck.log
    fi
 
    # Send full logs to staging/preInstall
    $ECHO -e "$dateStr $ceName: $logMsg" >> $preUpgradeCheckLogs

    peerCeName=`$CAT $sonusConfigFile |$GREP peerCeName |$AWK -F "=" '{print \$2}'| $AWK '{print \$1}'`
    if [[ "$peerCeName" != "none" ]]; then
       $PERL $sbxPeerExpect "$ECHO -e '$dateStr $ceName: $logMsg' >> $preUpgradeCheckLogs" "shell" > /dev/null 2>&1
    fi

}

# Utility functions to check platform version
# and if that is compatible with application's version

# OS/BMC/Platform related variable that decide the min version required for this version of app
connexIPOSVersion=Unknown
bmcVersion=Unknown
minBmcVersion=Unknown
bmcUpgradeRequired=Unknown
reqOSVersion=Unknown
biosVersion=Unknown
reqBiosVersion=Unknown
debianRelease=/etc/debian_version

# Get OS version
getConnexIPOSVersion()
{
   if [ -f $debianRelease ]; then
      osType="debian"
      moduleFile=/etc/modprobe.d/aliases.conf
   fi

   # With older release of connecip(1.7.x) we have a different format for OS version. In order to
   # make the OS version consistent we need to change the existing old version same as the latest
   # OS version format available in 4.x.
   #     Old version in 1.7.5 - ConnexIP OS v1.7.5 - Mon Oct 17 17:02:39 EDT 2011
   #     Old version in 1.8.0 - ConnexIP OS v1.8.0-A1 - Wed Dec 14 15:11:05 EST 2011 
   #     Old version in 1.8.3 - ConnexIP OS 01.08.03-R001 - Mon Jun 23 09:52:49 EDT 2014
   #     New version in 4.x   - ConnexIP OS 02.00.02-A036 - Tue Nov 12 18:04:44 IST 2013
   #     convert  Old formats to new format v1.7.5 to the format of 01.07.05

   connexIPOSVersion=`$CAT $CONNEXIP_OS_VERSION | $AWK '{print $3}' | $SED 's/^v//'`
   # If it is in old format, convert to new format because this utility expects it in new format
   # for comparing the version strings
   # e.g. Convert 1.8.0-A8 to 01.08.00-A008, 1.8.0-A10 to 01.08.00-A010
   connexIPOSVersion=`$ECHO $connexIPOSVersion | $SED -e 's/\([0-9]\).\([0-9]\).\([0-9]\)-A\([0-9]\)/0\1.0\2.0\3-A00\4/' -e 's/-A0010/-A010/' -e 's/\([0-9]\).\([0-9]\).\([0-9]\)/0\1.0\2.0\3/'`
}

# Get Required OS version
getReqOSVersion()
{
   if [ -e "$STAGING_BUILDINFO" ];
   then
      reqOSVersion=`$CAT $STAGING_BUILDINFO | $GREP "Required OS   Version" | $AWK '{print $4}'`
   fi
}

# Get the required and current Bios Version
# check to see if they are the same / need to be updated. 
biosUpdateNeeded()
{
   # return 0 - bios does not need to be updated 
   # return 1 - bois needs to be updated.
   rtnVal=0
   if [ -e "$STAGING_BUILDINFO" ];
   then
      # get BIOS version based on harware platform...
      if [ "$hostType" == "SBC7000" ]; then
         biosVersion=$($PRINTF "v%d.%d.%d" `$GREP "SBX7K_BIOS_V" $DMIDECODE_OUT | $AWK '{print $3}' | $SED 's/SBX7K_BIOS_V//' | $AWK -F . '{print $1,$2,$3}'`)
         reqBiosVersion=`$CAT $STAGING_BUILDINFO  | $GREP " BIOS Version:" | $GREP Blue | $AWK -F ": " '{printf $2}'`
      elif [ "$hostType" != "ConnexIP5000" ]; then 
         biosVersion=$($PRINTF "v%d.%d.%d" `$GREP "BIOS V" $DMIDECODE_OUT | $AWK '{print $5}' | $SED 's/SBX5K_v//' | $AWK -F . '{print $1,$2,$3}'`)
         reqBiosVersion=`$CAT $STAGING_BUILDINFO  | $GREP " BIOS Version:" |  $GREP -v Blue | $AWK -F ": " '{printf $2}'`
      fi

      if [[ "$biosVersion" != "$reqBiosVersion" && "$hostType" != "ConnexIP5000" ]]; then
         retVal=1
      fi
   fi
   return $retVal
}

# Get BMC version
getBmcVersion()
{
    # output of ipmi command: Firmware Revision         : 2.0
    bmcVersionBase=`$IPMITOOL mc info | $GREP "Firmware Revision" | $AWK '{print $4}'`
    bmcVersionExt=`$IPMITOOL mc info | $GREP -A 1 Aux | $GREP 0x | $AWK {'print $1'}  | xargs $PRINTF "%d\n"`
    bmcVersion=$($PRINTF "%2.2d.%2.2d.%2.2d" `$ECHO "$bmcVersionBase.$bmcVersionExt" | $AWK -F . '{print $1,$2,$3}'`)
    minRawCmdVer="03.15.00"

    if [[ "$bmcVersion" == "$minRawCmdVer" || "$bmcVersion" > "$minRawCmdVer" ]]; then
        local bmcVersionInHex=`$IPMITOOL raw 0x00 0x11`
        local hexArray=($bmcVersionInHex)

        if [[ "${hexArray[0]}" != "00" ]]; then
            bmcVersion=Unknown
            return
        fi
        bmcVersion=""
        for i in "${hexArray[@]:1}"
        do
            if [[ "$i" != "00" ]]; then
                char=$(echo -e "\x$i")
                bmcVersion=${bmcVersion}${char}
            else
                break
            fi
        done
    fi
}

# Get Min Required BMC version
getMinReqBmcVersion()
{
   if [ -e "$STAGING_BUILDINFO" ];
   then
      if [[ $hwType == "SBC7000" ]]; then
         minBmcVersion=$($CAT $STAGING_BUILDINFO | $GREP "Required Bluefin BMC  Version" | $AWK '{print $5}' | $CUT -f2 -d"v")
      elif [[ $hwType == "SBC5400" ]]; then
         minBmcVersion=$($CAT $STAGING_BUILDINFO | $GREP "Required Yellowfin BMC  Version" | $AWK '{print $5}' | $CUT -f2 -d"v")
      else
         minBmcVersion=$($CAT $STAGING_BUILDINFO | $GREP "Required BMC  Version" | $AWK '{print $4}' | $CUT -f2 -d"v")
      fi
   fi
}

isBmcUpgradeRequired()
{
    getBmcVersion
    getMinReqBmcVersion

    local bmcBaseVersion=$($ECHO $bmcVersion | $AWK -F'-' '{ print $1 }')
    local minBmcBaseVersion=$($ECHO $minBmcVersion | $AWK -F'-' '{ print $1 }')

    if [[ "$bmcBaseVersion" < "03.15.00" || "$bmcBaseVersion" < "$minBmcBaseVersion" ]]; then
        bmcUpgradeRequired=true
        return
    fi

    if [[ "$bmcBaseVersion" == "$minBmcBaseVersion" ]]; then
        local bmcSubVersion=$($ECHO $bmcVersion | $AWK -F'-' '{ print $2 }')
        local minBmcSubVersion=$($ECHO $minBmcVersion | $AWK -F'-' '{ print $2 }')
        if [[ "$bmcSubVersion" < "$minBmcSubVersion" ]]; then
            bmcUpgradeRequired=true
        fi
    fi
}


# Check if the platform is running min rev required for this release
checkPlatformMinVersions()
{
   local resultFile=$1
   local skipStatusFileCreation=$2
   local checkOnly=$3
   getConnexIPOSVersion
   getReqOSVersion

   # only check the OS version if not doing an uprade
   # note: when we switched from sysvinit to systemd the location of the sbx script
   # changed. we therefore need to check both the current and potentially old location.
   if [[ ! -e $ETC_INITD_SBX && ! -e /etc/init.d/sbx ]]; then
      if [[ "$connexIPOSVersion" != "$reqOSVersion" ]]; then
         preInstallLogMsg "ERROR: Invalid OS version: $connexIPOSVersion, Please re-iso to $reqOSVersion and then install the app; Exiting..." "user"
         if [ $skipStatusFileCreation == "false" ];then 
            $PERL $statusUpdater -c -r preChecks -s "CEcheckStatus" -v "checkFailed"
            $PERL $statusUpdater -c -r preChecks -s "EndTime" -v "`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`"	
            $PERL $statusUpdater -c -s "Reason" -v "InvalidOsVersion_${connexIPOSVersion}_On_${ceName}.Possible_Recovery_Action_is:To_reIso_to_${reqOSVersion}."
            $RM -f $upgradeBaseDir/preUpgradeCheckActive.key 2>/dev/null
            $PERL $statusUpdater -c -s "CheckStatus" -v "checkFailed"
         fi 
      	 if [ -e $SONUS_TMP_DIR/preInstallCheck.out ];then
            # print only 5 lines.
            $MV -f $SONUS_TMP_DIR/preInstallCheck.out $SONUS_TMP_DIR/preInstallCheck.out.full
            $CAT $SONUS_TMP_DIR/preInstallCheck.out.full  |$TAIL -n5 > $SONUS_TMP_DIR/preInstallCheck.out
         elif [ -e /tmp/preInstallCheck.out ];then
            # print only 5 lines.
            $MV -f /tmp/preInstallCheck.out /tmp/preInstallCheck.out.full
            $CAT /tmp/preInstallCheck.out.full  |$TAIL -n5 > /tmp/preInstallCheck.out
         fi

         if [ -n "$resultFile" ]; then
            trapExit 2 $resultFile
         else
            exit 2
         fi
      fi
   fi

   # On target hardware, check BMC and BIOS versions
   if [[ $hwType == "ConnexIP" || $hwType == "ConnexIP5200" || $hwType == "ConnexIP5100" || $hwType == "SBC7000" || $hwType == "SBC5400" ]]; then
     isBmcUpgradeRequired
     if [[ "$bmcUpgradeRequired" == "true" ]]; then
        if [[ $checkOnly == "1" && -e $upgradeBaseDir/preUpgradeCheckActive.key ]];then
           # This is a case of pre-upgrade checks. So just log a warning and do not fail pre-upgrade checks.
           preInstallLogMsg "WARNING: Invalid BMC version: $bmcVersion, need at least: $minBmcVersion" "user"
        else
           preInstallLogMsg "ERROR: Invalid BMC version: $bmcVersion, need at least: $minBmcVersion" "user"
           array_exitReason[index++]=InvalidBMCVersionOn${ceName}.Possible_Recovery_Action_is:To_upgradeBMC.
           exitRequired=1
        fi
     fi
     # We do not need to check BIOS version as we are mandating BIOS
     # upgrade while upgrading OS.
   else
     hwType=`$CAT $HOST_TYPE`
     if [[ $hwType == "ConnexIP5000" ]]; then
        preInstallLogMsg "Virtual SBC."
     else
        preInstallLogMsg "Not connexIP hardware, skipping BMC version check"
     fi
   fi
}

# check the return value of a command run via executeCommand
checkResult()
{
    local result=$1
    local expectedRes=$2
    local logFile=$3
    local error=$4
    local resultFile=$5

    if [ $result -ne $expectedRes ]; then
       # enable the following for debug purposes
       #echo "*** executeCommmand error ***: Failed to execute \"$cmd\""
       if [ -e "$LIVE_UPGRADE_ACTIVE_KEY" ];then
          # Its a LSWU case. Find current step and update status file with failure.
          # currently statusUpdater doesnt handle whitespaces and backslashes. So replace ' ' with  '_' and '/' with 'or'.
          myReason=$($ECHO $error |$SED 's/ /_/g' | $SED 's/\//_or_/g' )
          currStepNo=$($PERL $statusUpdater -g CurrentStep)
          currStep=$($PERL $statusUpdater -r $currStepNo -g Step)
          $PERL $statusUpdater -r $currStep -s Reason -v "$myReason"  
          $PERL $statusUpdater -r $currStep -s Status -v "failed"
          $PERL $statusUpdater -s Reason -v "$myReason" 
       fi
       if [ -n "$logFile" ]; then
          logMsg $logFile "ERROR: $error"
       fi
       if [ -n "$resultFile" ]; then
          trapExit $result $resultFile
       else
          cleanupAllMarkers
          exit $result
       fi
    fi
}

# execute a command and check its exit status
# param 1: the command
# param 2: the expected result
# param 3: log file to capture stderr and stdio
# param 4: error msg upon failure
# param 5: result file
# NOTE: this function captures stdio and stderr too
executeCommand()
{
    local cmd=$1
    local expectedRes=$2
    local logFile=$3
    local error=$4
    local resultFile=$5

    # enable the following line to show where we are taking all of the time...
    # logMsg $logFile "\tExecuting \"$cmd\"..."

    if [ -n "$logFile" ]; then
       # pipe the command to tee to capture the output
       # note: need to check PIPESTATUS rather than $? so we check the
       #       $cmd result and not the result of tee...
       $cmd 2>&1 | $TEE -a $logFile
       local result=${PIPESTATUS[0]}
       checkResult $result $expectedRes $logFile "$error" $resultFile
    else
       # not capturing output to a log, so just execute it...
       $cmd
       checkResult $? $expectedRes $logFile "$error" $resultFile
    fi

    # enable the following line to show where we are taking all of the time...
    # logMsg $logFile "\tFinished executing \"$cmd\"..."
}

#############################################################
# Function to run the service wrapper script.  We need to run
# it from the staging directory first.  But once we unpack the
# deb it is in the scripts dir.  And once we start the sbx it
# it removed from the staging dir.  Call it from the staging
# dir until it no longer exists there.
##############################################################
runServiceCmd()
{
   local svcName=$1
   local svcCmd=$2
   local background=$3

   if [ -e "$STAGING_SERVICE_SH" ];
   then
      cmd=$STAGING_SERVICE_SH
   elif [ -e "$SERVICE_SH" ];
   then
      cmd=$SERVICE_SH
   else
      $ECHO "Unable to find service script. Exiting..."
      exit 1
   fi
   if [ $background == 1 ]; then
      $cmd $svcName $svcCmd &
   else
      $cmd $svcName $svcCmd
   fi
}

# stop the service
# param1: the log file
stopService()
{
   local logFile=$1

   # Stop sbx service...
   if [ -e "$ETC_INITD_SBX" ];
   then
      # Remove SBX install/upgrade in progress marker file if existing.
      removeMarkerFile $sbxInstallUpgradeInProgressKey

      # Do not stop the service if it is not running.
      # This may cause a RID issue if sbxCleanup is called
      # when the os has been upgraded and the application has not.
      runServiceCmd "sbx" "status" 0 | $GREP -q "Service not running"
      if [ $? -eq 1 ]; then 
         logMsg $logFile "Stopping application service..."
         executeCommand "runServiceCmd "sbx" "stop" 0" 0 $logFile "Failed to stop the application service"
      fi
   fi

   # Create SBX install/upgrade in progress marker file.
   createMarkerFile $sbxInstallUpgradeInProgressKey "$0"
}

# make sure that the install is running from the staging directory
# param1: the caller (i.e. $0 that called us)
# param2: the staging directory
# param3: log file
# param4: result file
checkLocation() 
{
   local dirName=`$DIRNAME $1`
   local stagingDir=$2
   local logFile=$3
   local resultFile=$4
   if [ "$dirName" = "." ]; then
      dirName=$PWD
   fi
   if [ "$dirName" != "$stagingDir" ]; then
      logMsg $logFile "ERROR: Installation/update files should be installed to and run from $stagingDir; Exiting..."
      if [ -n "$resultFile" ]; then
         trapExit 1 $resultFile
      else
         exit 1
      fi
   fi
}

# prevent the install from starting if diagnostics are running
# param1: the log file
checkDiags()
{
   local logFile=$1

    pid=`$PIDOF testAppDiagConnexip5200`
    if [ -n "$pid" ]; then
        logMsg $logFile "ERROR: Install cannot be started while diagnostics are running; Exiting..."
        exit 1
    fi
}

# To create a marker file with a given keyword
createMarkerFile()
{
    markerFile=$1
    keyWord=$2

    logMsg $logFile "Creating marker file $markerFile with \"$keyWord\"..."
    $ECHO "$keyWord" > $markerFile
}

# To remove a marker file
removeMarkerFile()
{
    markerFile=$1

    logMsg $logFile "Removing marker file $markerFile ..."
    $RM -f $markerFile
    
}

# This function validates whether the script should be allowed to be executed.
# It should meet one of the two conditions:
# 1. This script should either be started by Platform Manager
# 2. Or the parent process should have created a given marker file.
# Param 1: Parent process ID
# Param 2: Operation being performed (install/upgrade)
validateExecution () 
{
    local parentProcessId=$1
    local operation=$2

    isExecutionValid="true"
    parentProcessCmd=`$PS -ww -o command= -p $parentProcessId | $AWK '{print $1, $2, $3}'`
    platformMgrCmd=
    pltfMgrLogFile=
    isPltMgrInstalled="true"

    if [ "$operation" == "install" ]
    then
        # Changing the order of checking so that in case the file 'runAppInstall.exp' is present at both the locations
        # (i.e /var/www and /var/www/scripts directory), the one located at /var/www/scripts directory will only be used.
        if [ -f /var/www/scripts/runAppInstall.exp ]
        then
            platformMgrCmd="$EXPECT -f /var/www/scripts/runAppInstall.exp"
        elif [ -f /var/www/runAppInstall.exp ] 
        then
       
            platformMgrCmd="$EXPECT -f /var/www/runAppInstall.exp"
        else
            isPltMgrInstalled="false"
        fi
        
        pltfMgrLogFile="$SONUS_LOG_CNXIPM_DIR/installStatus.log"

    elif [ "$operation" == "upgrade" ]
    then
        # Changing the order of checking so that in case the file 'runAppUpgrade.exp' is present at both the locations
        # (i.e /var/www and /var/www/scripts directory), the one located at /var/www/scripts directory will only be used.
        if [ -f /var/www/scripts/runAppUpgrade.exp ]
        then
            platformMgrCmd="$EXPECT -f /var/www/scripts/runAppUpgrade.exp"
        elif [ -f /var/www/runAppUpgrade.exp ]
        then
            platformMgrCmd="$EXPECT -f /var/www/runAppUpgrade.exp"
        else
            isPltMgrInstalled="false"
        fi
        pltfMgrLogFile="$SONUS_LOG_CNXIPM_DIR/upgradeStatus.log"
    fi

    # Check whether parent process command is that of Platform Manager
    if [ "$parentProcessCmd" == "$platformMgrCmd" ]
    then
        # Set the update log file same as Platform Manager log file
        # so that logs can be redirected to Platform Manager log file.
        updateLogFile=$pltfMgrLogFile
    else
        if [ $isPltMgrInstalled == "true" ]
        then
            isExecutionValid="false"
        fi
        # Check for the contents of the marker file
        if [ -f $scriptMarkerFile ]
        then
            key=`$CAT $scriptMarkerFile`
            key=`$ECHO "$key" | $TR -d ' '`

            # Check for Platform Manager key in marker file.
            if [ "$key" == "PLATFORM_MANAGER_KEY" ]
            then
                isExecutionValid="true"
            fi

            # Move the marker file (Need this for debugging purposes)
            $MV -f $scriptMarkerFile $scriptMarkerFile.backup
        else
            # Check for the marker in $sonusRoot/ dir in case of upgrades from 4.x
            if [ -f $sonusRoot/.parentScriptMarker ]
            then
                key=`$CAT $sonusRoot/.parentScriptMarker`
                key=`$ECHO "$key" | $TR -d ' '`

                # Check for Platform Manager key in marker file.
                if [ "$key" == "PLATFORM_MANAGER_KEY" ]
                then
                    isExecutionValid="true"
                fi

                # Move the marker file (Need this for debugging purposes)
                $MV -f $sonusRoot/.parentScriptMarker $sonusRoot/.parentScriptMarker.backup
            fi
        fi
    fi

    if [ $isExecutionValid == "false" ]
    then
        # Script installation only allowed for vSBC. Regular SBC *MUST* use
        # PM for installation
        if [[ "$hostSubType" == "virtualCloud" ]]; then
            logMsg $logFile "virtualCloud package: allowing script from shell installation..."
        else
            logMsg $logFile "Running the script from shell is no longer supported !!!!"
            return 1
        fi
    fi

    return 0
}

# display the manually entered config before prompting for confirmation
displayEnteredConfig()
{
   local hostType=$(getHwType)
   $ECHO ""
   $ECHO "===================================================================="
   $ECHO "role=$role                  # 1=Active, 2=Standby"
   $ECHO "systemName=$systemName"
   $ECHO "hostName=$ceName"
   $ECHO "peerHostName=$peerCeName"
   $ECHO "mgt0Ipaddr=$nif1Ipaddr"
   $ECHO "mgt0Netmask=$nif1Netmask"
   $ECHO "mgt0GatewayIpaddr=$nif1GatewayIpaddr"
   $ECHO "mgt0IpaddrV6=$nif1IpaddrV6"
   $ECHO "mgt0NetprefixV6=$nif1NetprefixV6"
   $ECHO "mgt0GatewayIpaddrV6=$nif1GatewayIpaddrV6"
   $ECHO "mgt1Ipaddr=$nif5Ipaddr"
   $ECHO "mgt1Netmask=$nif5Netmask"
   $ECHO "mgt1GatewayIpaddr=$nif5GatewayIpaddr"
   $ECHO "mgt1IpaddrV6=$nif5IpaddrV6"
   $ECHO "mgt1NetprefixV6=$nif5NetprefixV6"
   $ECHO "mgt1GatewayIpaddrV6=$nif5GatewayIpaddrV6"
   $ECHO "ntpServerIpaddr=$ntpServerIpaddr"
   $ECHO "timeZoneIndex=$timeZoneIndex"
   $ECHO "allowSshAccess=$allowSshAccess"
   $ECHO "sbctype=$sbctype"
   $ECHO "personality=$personality            # $personalityOptions"
   $ECHO "geoRedundHA=$geoRedundHA"
   $ECHO "leaderElect=$leaderElect            # $leaderElectOptions"
   if [[ $hostType != "ConnexIP5000" ]]; then
      $ECHO "bondMonitor=$bondMonitor            # $bondMonitorOptions"
   fi
   if [[ "$hostType" == "SBC5400" ]]; then
      $ECHO "mgt2Ipaddr=$nif7Ipaddr"
      $ECHO "mgt2Netmask=$nif7Netmask"
      $ECHO "mgt2GatewayIpaddr=$nif7GatewayIpaddr"
      $ECHO "mgt2IpaddrV6=$nif7IpaddrV6"
      $ECHO "mgt2NetprefixV6=$nif7NetprefixV6"
      $ECHO "mgt2GatewayIpaddrV6=$nif7GatewayIpaddrV6"
      $ECHO "mgt3Ipaddr=$nif8Ipaddr"
      $ECHO "mgt3Netmask=$nif8Netmask"
      $ECHO "mgt3GatewayIpaddr=$nif8GatewayIpaddr"
      $ECHO "mgt3IpaddrV6=$nif8IpaddrV6"
      $ECHO "mgt3NetprefixV6=$nif8NetprefixV6"
      $ECHO "mgt3GatewayIpaddrV6=$nif8GatewayIpaddrV6"
   fi
   if [[ $hostType == "ConnexIP5000" ]]; then
      $ECHO "haMode=$haMode"
   fi
   if [[ $hostType == "ConnexIP5000" ]]; then
      $ECHO "rgIp=$rgIp"
   fi
   $ECHO "===================================================================="
}

getConfigFromUser()
{
  while [ 1 ]
  do
    promptForConfig

    # Ask if configuration should be applied to server?
    displayEnteredConfig
    read -p "Press <ENTER> to apply configuration or R to re-enter? <ENTER/R> " prompt
    # Check if we should restart?
    if [[ $prompt != "r" && $prompt != "R" ]]; then 
       break
    fi
  done
}

getRootDisk()
{
  rootDisk=`$CAT /proc/partitions 2> /dev/null | $AWK '$2 == 0 { print $NF }' | $GREP -Ev "dm-0|sr0|drbd0|fd0"`
  rootDisk="/dev/${rootDisk}"
  $ECHO $rootDisk
}

getLogDisk() 
{
  rootDisk=$(getRootDisk)
  local tmpHwType=`$ECHO $hname | $AWK '{print tolower($0)}'`

  if [[ "$tmpHwType" =~ ^hvm  || "$tmpHwType" =~ ^c5 || "$tmpHwType" =~ ^m5 || "$tmpHwType" =~ ^c6 ]]; then
    logDisk="${rootDisk}5"
  else
    logDisk="${rootDisk}6"
  fi

  $ECHO $logDisk
}

# Returns if running on Public Cloud
# As we support more clouds, we need to extend the list
isPublicCloud()
{
    local isPC=0
    local virtType=$(getVirtualType)
    if [[ "$virtType" == "AWS" || "$virtType" == "GCE" || "$virtType" == "AZ" ]]; then
        isPC=1
    fi
    $ECHO $isPC
}

isDSP2x()
{
   local isDSP2xFlag=0

   if [[ "$prodString" == "5110" || "$prodString" == "5210" || "$prodString" == "5400" ]]; then
     isDSP2xFlag=1
   fi
   return $isDSP2xFlag
}

isDSPLC()
{
  local isDSPLCFlag=0
  local hostType=$(getHwType)
  
  if [[ "$hostType" == "SBC7000" ]]; then
    isDSPLCFlag=1
  fi
  return $isDSPLCFlag
}

isBfdRunning()
{
   local pid=$($PIDOF bfdd)
   return $pid
}

function openhpidRestart()
{
   local logFile=$1
   local doLog=$2

   [[ "$doLog" != "skip" ]] && logMsg $logFile "Stopping openhpid..."
   runServiceCmd "openhpid" "stop" 0 >> $logFile 2>&1

   if [ "$hostType" != "ConnexIP5000" ]; then 
       [[ "$doLog" != "skip" ]] && logMsg $logFile "Taking backup of SEL entries..."
       $IPMITOOL sel info >> $logFile 2>&1
       $IPMITOOL sel elist >> $logFile 2>&1

       [[ "$doLog" != "skip" ]] && logMsg $logFile "Clearing SEL entries..."
       $IPMITOOL sel clear >> $logFile 2>&1
   fi

   [[ "$doLog" != "skip" ]] && logMsg $logFile "Starting openhpid..."
   runServiceCmd "openhpid" "start" 0 >> $logFile 2>&1

   # If hpi doesn't start in 90 seconds, we continue anyway. Let the
   # calling script decide to try and restart the service or exit.
   if [ "$hostType" != "ConnexIP5000" ]; then
       openhpidCheck 90 1 $logFile
   else
       openhpidCheck 90 1 /dev/null
   fi
   hpiStatus=$?

   if [ $hpiStatus -ne 0 ] ; then
     [[ "$doLog" != "skip" ]] && logMsg $logFile "openhpid service failed to restart..."
   fi
   return $hpiStatus
}

function openhpidCheck()
{
   # the daemon takes some time to startup and be ready to produce
   # requested information.  use the hpiinv command to determine when
   # openhpid is ready.  in testing we typically have 2 hpiinv attempts
   # that return right away before openhpid is ready.  the third attempt
   # is accepted but takes a few seconds to complete.  using this mechanism
   # is much better than having a hard-coded arbitrary wait.
   local loopCountMax=$1
   local sleepTime=$2
   local logFile=$3
   local loopCount=0
   local notReady=1

   while [ $notReady -eq 1 -a $loopCount -lt $loopCountMax ]
   do
     loopCount=$((loopCount+1))
     $HPIINV >> $logFile 2>&1 
     if [ $? -ne 0 ]; then
       $SLEEP $sleepTime
     else
       notReady=0
     fi
   done
   return $notReady
}

# Initialize modular type
getModularType()
{
   # Get if the system is of modular type: e.g. 5110, 5210.
   modularType=Unknown
   modularTypeStr=$prodString
   if [[ "$modularTypeStr" == "5210" || "$modularTypeStr" == "5110"  || "$modularTypeStr" == "7000" || "$modularTypeStr" == "5400" ]]; then
      modularType=true
   else
      modularType=false
   fi

    $ECHO $modularType
}

# Determine the assigned role of the given CSI
getAssignedRole()
{
   csiToCheck=$1

   # Determine which component entry (cenode1 or cenode2) should be checked. 
   # Installed active is CE_Node2, standby CE_Node1.
   installRole=`$SED -n -e '/role/s/role=\([12]\).*/\1/p' $sonusConfigFile`
   compPrefix=CE_Node
   if [ $installRole -eq 1 ]; then
      compPrefix="${compPrefix}2"
   else
      compPrefix="${compPrefix}1"
   fi
   compPrefix="${compPrefix}_2N_Su_Comp_"

   # Use asp_console to dump the desired CSI entity and retrieve the HA state
   # of the node's component. to get just the piece we want, process the CSI
   # status using sed with the no-print option and find the correct component
   # prefix.  Then read the next line which is its HA state, and pull just
   # the state.  To be compatible with previous versions of the status display,
   # display the state in lowercase letters.
   currentRole=`$ECHO -e "setc master\nsetc cpm\namsEntityPrint CSI $csiToCheck\nquit\n" | \
                $ASP_CONSOLE | \
                $SED -n -e '/'$compPrefix'/ {
                    n
                    s/.*HA State: //p
                }' | \
                $TR [:upper:] [:lower:]`
}

# Get the current role of the node from OpenClovis
getCurrentRole()
{
   getAssignedRole $LAST_CSI

   if [[ $currentRole == "active" && "$haMode" == "Nto1" ]]; then
     currentRole=`$PYTHON3 /opt/sonus/sbx/serf/serfGetSelfParams.py currentRole`
   fi
}

# Load initialization config parameters from file.
loadConfig()
{
    # Load config...
    local configFile=$1
    if [ -e $configFile ];then
        . $configFile
    else 
        . $sonusConfigFile
    fi

    # new parameters should be defaulted.  in the case of timezone,
    # the default has to be read from the linux setup.
    if [ -z $timeZoneIndex ]; then
        # Read the /etc/timezone file to retrieve the timezone
        zoneIndex=1
        while [ $zoneIndex -lt ${#zoneByArea[@]} ]
        do
            grep -q ${zoneByArea[$zoneIndex]} /etc/timezone
            if [ $? -eq 0 ]; then
                timeZoneIndex=$zoneIndex;
                break;
            fi
            zoneIndex=`expr $zoneIndex + 1`
        done
    fi

    # originally GR-HA was one option.  now it is two options and
    # we changed the names in the sbx.conf file.  If the old name is
    # set, set both of the new options.  The old name can ONLY be set
    # for those few customers where the GR-HA MOP released with 5.0.1R1
    # was run.  In most cases the variable won't even be defined.
    if [ ! -z ${geoRedundHA+x} ]; then
        if [ $geoRedundHA -eq 1 ]; then
            leaderElect=1
            bondMonitor=1
        fi
    fi

    # A new paramater sbctype is added to sbx.conf in 7.2, 
    # which needs to be fetched from personalitytype.txt 
    # for upgrades from pre 7.2. For upgrades from 7.2/later,
    # this will be set as part of sourcing config file.
    # NOTE: at least one config file had the entry but it was empty,
    # and this will cause install to fail.  We need to default it
    # if that is the case. (fails in building sonusSystem.xml)
    $GREP -q sbctype $configFile
    if [[ $? -ne 0 || -z "$sbctype" ]]; then
       if [ -f "$PERSONALITYTYPE_TXT" ]; then
          logMsg $logFile "Loading personality parameter sbctype."
          sbctype=`$CAT $PERSONALITYTYPE_TXT`
       else
          logMsg $logFile "Defaulting sbctype to isbc."
          sbctype=isbc
          $ECHO $sbctype > $PERSONALITYTYPE_TXT
       fi
    elif [ ! -f "$PERSONALITYTYPE_TXT" ]; then
       $ECHO $sbctype > $PERSONALITYTYPE_TXT
    fi

    # A new paramater haMode is added to sbx.conf in 8.1,
    # for cloud, it should come from meta/user/instance data and should be populated into sbx.conf by LCA
    # for h/w and SWe, it may not be there in upgrade cases, default it to 1:1/1to1
    $GREP -q haMode $configFile
    if [ $? -ne 0 ]; then
       logMsg $logFile "Defaulting haMode to 1to1."
       haMode=1to1
    fi

    # A new paramater rgIp is added to sbx.conf in 8.1 for SWe,
    # Default it to peer's HA IP if not set.
    if [[ $hostType == "ConnexIP5000" ]]; then
       $GREP -q rgIp $configFile
       if [ $? -ne 0 ]; then
          if [ $role -eq "1" ];then
             logMsg $logFile "Defaulting rgIp to $standbyPrimaryInterCeIpaddr."
             rgIp="$standbyPrimaryInterCeIpaddr"
          else
             logMsg $logFile "Defaulting rgIp to $activePrimaryInterCeIpaddr."
             rgIp="$activePrimaryInterCeIpaddr"
          fi
       fi
    fi

    # New parameters enableREST, enableCoreEma. enableTS are added to sbx.conf in 9.0 
    # Default all the three to enable if not set.
	$GREP -q enableREST $configFile
    if [ $? -ne 0 ]; then
        enableREST=enabled
    fi

    $GREP -q enableCoreEMA $configFile
    if [ $? -ne 0 ]; then
        enableCoreEMA=enabled
    fi

    $GREP -q enableTS $configFile
    if [ $? -ne 0 ]; then
        enableTS=enabled
    fi

    # A bug in GR-HA functionality added in version 10.1.x caused new interCe HA
    # port addresses to be written out incorrectly.  If the variables are not set
    # properly do it now.  Since this is just for h/w and we don't change them, we
    # can just do it without thought to special values.
    if [[ -z $activeInterCeHaPort1Ipaddr ]]; then
        activeInterCeHaPort1Ipaddr="169.254.99.3"
    fi
    if [[ -z $activeInterCeHaPort2Ipaddr ]]; then
        activeInterCeHaPort2Ipaddr="169.254.99.4"
    fi
    if [[ -z $standbyInterCeHaPort1Ipaddr ]]; then
        standbyInterCeHaPort1Ipaddr="169.254.88.3"
    fi
    if [[ -z $standbyInterCeHaPort2Ipaddr ]]; then
        standbyInterCeHaPort2Ipaddr="169.254.88.4"
    fi
}

# Prompt user for initialization config parameters.
# NOTE: this part is interactive so do not log it.
# NOTE: ******************************************************
#       IF YOU ADD A FIELD TO THIS ROUTINE, PLEASE MAKE
#       SURE TO ALSO UPDATE THE ROUTINE validateConfig
#       IN cfgValidation.sh WITH THE SAME VALIDATION CODE!!!!!
# NOTE: ******************************************************
promptForConfig()
{
   # For vSBC, most of these answers are hard-coded so skip them.
   if [[ "$hostSubType" == "virtualCloud" ]] ; then
       echo "virtualCloud installation. Skipping n/a configurables..."
   fi

   # Prompt for local host role...
   while [ 1 ]
   do
      if [[ "$hostSubType" == "virtualCloud" ]] ; then
          role=1
          break
      fi

      $ECHO -n "   Local host role is the initial install host 
   role and must be set opposite from the peer. To check
   the installed role use 'swinfo'.  Options are:
       1. Active
       2. Standby
Enter local host role: "
      read role
      case $role in
      1)
         break
         ;;
      2)
         break
         ;;
      *)
         $ECHO -e "Invalid role. Please re-enter."
         ;;
      esac
   done

   # Prompt for names...
   while [ 1 ]
   do
      if [[ "$hostSubType" == "virtualCloud" ]] ; then
          systemName=vsbc
          break
      fi

      $ECHO -n "Enter system name: "
      read systemName
      if [ -n "$systemName" ]; then
         break
      else
         $ECHO -e "Invalid system name. Please re-enter."
      fi
   done
   while [ 1 ]
   do
      if [[ "$hostSubType" == "virtualCloud" ]] ; then
          ceName=vsbc
          break
      fi

      $ECHO -n "Enter local host name: "
      read ceName
      if [ -n "$ceName" ]; then
        if [[ "$ceName" =~ [_] ]]; then
         $ECHO -e "The CE name can not contain an underscore. Please re-enter."
        else
           break
        fi
      else
         $ECHO -e "Invalid local host name. Please re-enter."
      fi

   done
   while [ 1 ]
   do
      if [[ "$hostSubType" == "virtualCloud" ]] ; then
          peerCeName=none
          break
      fi

      $ECHO -n "Enter peer host name (enter 'none' if single node system): "
      read peerCeName
 
      if [ -n "$peerCeName" -a "$peerCeName" != "$ceName" ]; then
        if [[ "$peerCeName" =~ [_] ]]; then
         $ECHO -e "The peer CE name can not contain an underscore. Please re-enter."
        else
           break
        fi
      else
         $ECHO -e "Invalid peer host name. Please re-enter."
      fi
   done

   # Only prompt for debug interface config if -d has been specified... 
   if [ "$dbgInterfaceCfg" == "y" ]; then
      # Prompt for debug interface configuration...
      while [ 1 ]
      do
         $ECHO -n "Enter debug IP (aaa.bbb.ccc.ddd): "
         read dbgIpaddr
         if valid_ip $dbgIpaddr; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter debug netmask (aaa.bbb.ccc.ddd): "
         read dbgNetmask
         if valid_ip $dbgNetmask; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter debug gateway IP (aaa.bbb.ccc.ddd): "
         read dbgGatewayIpaddr
         if valid_ip $dbgGatewayIpaddr; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
   else
      dbgIpaddr="0.0.0.0"
      dbgNetmask="0.0.0.0"
      dbgGatewayIpaddr="0.0.0.0"
   fi

   # Use values from the cps-interfaces file 
   local hostType=$(getHwType)
   if [ -e $cpsIntfFile ] && [[ "$hostType" == "ConnexIP5200" || "$hostType" == "ConnexIP5100" || "$hostType" == "ConnexIP5000" || "$hostType" == "SBC7000" || "$hostType" == "SBC5400" ]]; then
      if [ ! -f $cpsIntfFile.bkp ]
      then
        # Back up the CPS_INTERFACES file
        $CP -f $cpsIntfFile $cpsIntfFile.bkp
      fi

      # Modifying the CPS_INTERFACES file to change lif name from 'mgmtIntf*' to 'mgtIntf*'
      $SED -i "s/name = mgmtIntf/name = mgtIntf/g" $cpsIntfFile
   
      # Find the current owner
      cpsOwner=`$CPSI -g lif_owner mgmt_lif owner | $AWK -F ':' '{printf "%s", $2}'`
      
      nif1Ipaddr=`$CPSI -g lif mgtIntf1 v4_address owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
      if [ ! -n "$nif1Ipaddr" ]; then
        nif1Ipaddr="0.0.0.0"
      fi

      nif1Netprefix=`$CPSI -g lif mgtIntf1 v4_prefix owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
      if [ ! -n "$nif1Netprefix" ]; then
        nif1Netprefix="0"
      fi
      convNetprefixToNetmask $nif1Netprefix nif1Netmask

      nif1GatewayIpaddr=`$CPSI -g lif mgtIntf1 v4_gateway owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
      if [ ! -n "$nif1GatewayIpaddr" ]; then
        nif1GatewayIpaddr="0.0.0.0"
      fi

      nif1IpaddrV6=`$CPSI -g lif mgtIntf1 v6_address owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
      if [ ! -n "$nif1IpaddrV6" ]; then
        nif1IpaddrV6="::"
      fi

      nif1NetprefixV6=`$CPSI -g lif mgtIntf1 v6_prefix owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
      if [ ! -n "$nif1NetprefixV6" ]; then
        nif1NetprefixV6="0"
      fi

      nif1GatewayIpaddrV6=`$CPSI -g lif mgtIntf1 v6_gateway owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
      if [ ! -n "$nif1GatewayIpaddrV6" ]; then
        nif1GatewayIpaddrV6="::"
      fi

      ### Following is for HW Platforms (non softSBC)
      if [[ "$hostType" != "ConnexIP5000" ]]; then
        nif5Ipaddr=`$CPSI -g lif mgtIntf2 v4_address owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
        if [ ! -n "$nif5Ipaddr" ]; then
          nif5Ipaddr="0.0.0.0"
        fi

        nif5Netprefix=`$CPSI -g lif mgtIntf2 v4_prefix owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
        if [ ! -n "$nif5Netprefix" ]; then
          nif5Netprefix="0"
        fi
        convNetprefixToNetmask $nif5Netprefix nif5Netmask

        nif5GatewayIpaddr=`$CPSI -g lif mgtIntf2 v4_gateway owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
        if [ ! -n "$nif5GatewayIpaddr" ]; then
          nif5GatewayIpaddr="0.0.0.0"
        fi

        nif5IpaddrV6=`$CPSI -g lif mgtIntf2 v6_address owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
        if [ ! -n "$nif5IpaddrV6" ]; then
          nif5IpaddrV6="::"
        fi

        nif5NetprefixV6=`$CPSI -g lif mgtIntf2 v6_prefix owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
        if [ ! -n "$nif5NetprefixV6" ]; then
          nif5NetprefixV6="0"
        fi

        nif5GatewayIpaddrV6=`$CPSI -g lif mgtIntf2 v6_gateway owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
        if [ ! -n "$nif5GatewayIpaddrV6" ]; then
          nif5GatewayIpaddrV6="::"
        fi

        ### Following is for 5xx0 platforms 
	if [[ "$hostType" != "SBC5400" ]] && [[ "$hostSubType" != "virtualContainer" || $POD_TYPE == "OAM" ]]; then
          if [[ $nif1Ipaddr == "0.0.0.0" && $nif5Ipaddr == "0.0.0.0" && $nif1IpaddrV6 == "::" && $nif5IpaddrV6 == "::" ]]; then
            logMsg $logFile "ERROR: `basename $0` No Management IP Addresses detected, exiting"
            trapExit 1 $resultFile
          fi
        fi

        ### Following is for Yellowfin
        if [[ "$hostType" == "SBC5400" ]]; then
          nif7Ipaddr=`$CPSI -g lif mgtIntf3 v4_address owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
          if [ ! -n "$nif7Ipaddr" ]; then
            nif7Ipaddr="0.0.0.0"
          fi

          nif7Netprefix=`$CPSI -g lif mgtIntf3 v4_prefix owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
          if [ ! -n "$nif7Netprefix" ]; then
            nif7Netprefix="0"
          fi
          convNetprefixToNetmask $nif7Netprefix nif7Netmask

          nif7GatewayIpaddr=`$CPSI -g lif mgtIntf3 v4_gateway owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
          if [ ! -n "$nif7GatewayIpaddr" ]; then
            nif7GatewayIpaddr="0.0.0.0"
          fi
  
          nif7IpaddrV6=`$CPSI -g lif mgtIntf3 v6_address owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
          if [ ! -n "$nif7IpaddrV6" ]; then
            nif7IpaddrV6="::"
          fi

          nif7NetprefixV6=`$CPSI -g lif mgtIntf3 v6_prefix owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
          if [ ! -n "$nif7NetprefixV6" ]; then
            nif7NetprefixV6="0"
          fi

          nif7GatewayIpaddrV6=`$CPSI -g lif mgtIntf3 v6_gateway owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
          if [ ! -n "$nif7GatewayIpaddrV6" ]; then
            nif7GatewayIpaddrV6="::"
          fi
        
          nif8Ipaddr=`$CPSI -g lif mgtIntf4 v4_address owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
          if [ ! -n "$nif8Ipaddr" ]; then
            nif8Ipaddr="0.0.0.0"
          fi
  
          nif8Netprefix=`$CPSI -g lif mgtIntf4 v4_prefix owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
          if [ ! -n "$nif8Netprefix" ]; then
            nif8Netprefix="0"
          fi
          convNetprefixToNetmask $nif8Netprefix nif8Netmask
  
          nif8GatewayIpaddr=`$CPSI -g lif mgtIntf4 v4_gateway owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
          if [ ! -n "$nif8GatewayIpaddr" ]; then
            nif8GatewayIpaddr="0.0.0.0"
          fi

          nif8IpaddrV6=`$CPSI -g lif mgtIntf4 v6_address owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
          if [ ! -n "$nif8IpaddrV6" ]; then
            nif8IpaddrV6="::"
          fi
  
          nif8NetprefixV6=`$CPSI -g lif mgtIntf4 v6_prefix owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
          if [ ! -n "$nif8NetprefixV6" ]; then
            nif8NetprefixV6="0"
          fi

          nif8GatewayIpaddrV6=`$CPSI -g lif mgtIntf4 v6_gateway owner $cpsOwner | $AWK -F 'value:' '{printf "%s", $2}'`
          if [ ! -n "$nif8GatewayIpaddrV6" ]; then
            nif8GatewayIpaddrV6="::"
          fi

          if [[ $nif1Ipaddr == "0.0.0.0" && $nif5Ipaddr == "0.0.0.0" && $nif7Ipaddr == "0.0.0.0" && $nif8Ipaddr == "0.0.0.0" && $nif1IpaddrV6 == "::" && $nif5IpaddrV6 == "::" && $nif7IpaddrV6 == "::" && $nif8IpaddrV6 == "::" ]]; then
              logMsg $logFile "ERROR: `basename $0` No Management IP Addresses detected, exiting"
              trapExit 1 $resultFile
          fi
        fi # End for 5400
      elif [[ "$hostSubType" != "virtualContainer" || $POD_TYPE == "OAM" ]]; then
        if [[ $nif1Ipaddr == "0.0.0.0" && $nif1IpaddrV6 == "::" ]]; then
           logMsg $logFile "ERROR: $BASENAME $0 No Management IP Addresses detected, exiting"
           trapExit 1 $resultFile
        fi
      fi
   else
      # Prompt for primary IPv4 management interface configuration...
      while [ 1 ]
      do
         $ECHO -n "Enter primary management IP (aaa.bbb.ccc.ddd, '0.0.0.0' if not used): "
         read nif1Ipaddr
         if valid_ip $nif1Ipaddr; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter primary management netmask (aaa.bbb.ccc.ddd): "
         read nif1Netmask
         if valid_ip $nif1Netmask; then
            convNetmaskToNetprefix $nif1Netmask
            nif1Netprefix=$?
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter primary management gateway IP (aaa.bbb.ccc.ddd, '0.0.0.0' if not used): "
         read nif1GatewayIpaddr
            if valid_ip $nif1GatewayIpaddr; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
      # Prompt for primary IPv6 management interface configuration...
      while [ 1 ]
      do
         $ECHO -n "Enter primary management IPv6 (aaaa:bbbb:cccc:dddd:eeee:ffff:gggg:hhhh, '::' if not used): "
         read nif1IpaddrV6
         if [ -n "$nif1IpaddrV6" ] && valid_ipv6 $nif1IpaddrV6; then
            break
         else
            $ECHO -e "Invalid IPv6 address. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter primary management IPv6 prefix length[0-128]: "
         read nif1NetprefixV6
         if [ -n "$nif1NetprefixV6" ] && [[ 0 -le $nif1NetprefixV6 && $nif1NetprefixV6 -le 128 ]]; then
            break
         else
            $ECHO -e "Invalid IPv6 prefix length. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter primary management gateway IPv6 (aaaa:bbbb:cccc:dddd:eeee:ffff:gggg:hhhh, '::' if not used): "
         read nif1GatewayIpaddrV6
         if [ -n "$nif1GatewayIpaddrV6" ] && valid_ipv6 $nif1GatewayIpaddrV6; then
            break
         else
            $ECHO -e "Invalid IPv6 address. Please re-enter."
         fi
      done

      # Prompt for secondary IPv4 management interface configuration...
      while [ 1 ]
      do
         $ECHO -n "Enter secondary management IP (aaa.bbb.ccc.ddd, '0.0.0.0' if not used): "
         read nif5Ipaddr
         if  valid_ip $nif5Ipaddr && [[ "$nif5Ipaddr" == "0.0.0.0" || "$nif5Ipaddr" != "$nif1Ipaddr" ]]; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter secondary management netmask (aaa.bbb.ccc.ddd, '0.0.0.0' if not used): "
         read nif5Netmask
         if valid_ip $nif5Netmask; then
            convNetmaskToNetprefix $nif5Netmask
            nif5Netprefix=$?
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter secondary management gateway IP (aaa.bbb.ccc.ddd, '0.0.0.0' if not used): "
         read nif5GatewayIpaddr
         if valid_ip $nif5GatewayIpaddr; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
      # Prompt for secondary IPv6 management interface configuration...
      while [ 1 ]
      do
         $ECHO -n "Enter secondary management IPv6 (aaaa:bbbb:cccc:dddd:eeee:ffff:gggg:hhhh, '::' if not used): "
         read nif5IpaddrV6
         if  [ -n "$nif5IpaddrV6" ] && valid_ipv6 $nif5IpaddrV6; then
			if specified_ipv6 $nif5IpaddrV6 && [ "$nif5IpaddrV6" == "nif5IpaddrV6" ] ; then
            	$ECHO -e "Invalid IPv6 address. Please re-enter."
			else
				break
			fi
         else
            $ECHO -e "Invalid IPv6 address. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter secondary management IPv6 prefix length[0-128]: "
         read nif5NetprefixV6
         if [ -n "$nif5NetprefixV6" ] && [[ 0 -le $nif5NetprefixV6 && $nif5NetprefixV6 -le 128 ]]; then
            break
         else
            $ECHO -e "Invalid IPv6 prefix length. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter secondary management gateway IPv6 (aaaa:bbbb:cccc:dddd:eeee:ffff:gggg:hhhh, '::' if not used): "
         read nif5GatewayIpaddrV6
         if [ -n "$nif5IpaddrV6" ] && valid_ipv6 $nif5GatewayIpaddrV6; then
            break
         else
            $ECHO -e "Invalid IPv6 address. Please re-enter."
         fi
      done
   fi


      # Prompt for NTP time server IP address...
      while [ 1 ]
      do
         $ECHO -n "Enter NTP time server IP (aaa.bbb.ccc.ddd or aaaa:bbbb:cccc:dddd:eeee:ffff:gggg:hhhh): "
         read ntpServerIpaddr
         if [ -n "$ntpServerIpaddr" ] && valid_ipv6 $ntpServerIpaddr; then
            break
         elif valid_ip $ntpServerIpaddr; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done

      while [ 1 ]
      do  
         i=1
         spaces="                                  "
         while [ $i -lt ${#zoneName[@]} ]
            do  
               index=$i
               digSpace=" "
               if [ $index -lt 10 ]; then
                  digSpace="   "
               elif [ $index -lt 100 ]; then
                  digSpace="  "
               fi
               nextIndex=`$EXPR $i + 1`
               if [ $nextIndex -lt ${#zoneName[@]} ]; then
                  firstName=${zoneName[$index]}
                  $PRINTF '%d) %s%s%s %d) %s\n' $index "$firstName" "$digSpace" "${spaces:${#firstName}}" $nextIndex "${zoneName[$nextIndex]}"
               else
                  $ECHO -e "$index) ${zoneName[$i]}"
               fi
               i=`$EXPR $i + 2`
            done

         $ECHO -n "Enter index of the time zone from options above: "
            read timeZoneIndex
            if [[ $timeZoneIndex -lt ${#zoneName[@]} && $timeZoneIndex -gt 0 ]]; then
               break
            else
               $ECHO -e "Invalid option. Please re-enter."
            fi
      done
  # Prompt to allow Linux shell access...
   while [ 1 ]
   do
      if [[ "$hostSubType" == "virtualCloud" ]] ; then
          allowSshAccess=y
          break
      fi

      $ECHO -n "   Allowing users to access the Linux shell remotely thorugh SSH may have security implications. 
   It is recommended by Sonus to keep SSH access disabled.
Allow Linux ssh access (y/[n]): "
      read allowSshAccess
      case $allowSshAccess in
      y|Y)
         allowSshAccess="y"
         break
         ;;
      n|N|"")
         allowSshAccess="n"
         break
         ;;
      *)
         $ECHO -n "Invalid input. Please enter (y/[n]): "
         ;;
      esac
   done

   # Prompt for sbctype...
   while [ 1 ]
   do
      $ECHO -n "Enter the SBC personality type (isbc/ssbc/msbc/mrfp/slb): " 
      read sbctype
      case $sbctype in
      isbc)
         sbctype="isbc"
         break
         ;;
      ssbc)
         sbctype="ssbc"
         break
         ;;
      msbc)
         sbctype="msbc"
         break
         ;;
      mrfp)
         sbctype="mrfp"
         break
         ;;
      slb)
         sbctype="slb"
         break
         ;;
      *)
         $ECHO -n "Invalid input. Please re-enter (isbc/ssbc/msbc/mrfp/slb): "
         ;;
      esac
   done

   # Prompt for personality...
   # note: ${personalityMnemonic[$personality]+exists} returns exists if the key
   # exists and is empty otherwise
   while [ 1 ]
   do
      if [[ "$hostSubType" == "virtualCloud" ]] ; then
          personality=$SbxErePersonality
          break
      fi
      #the NO_ERE personality is only supported on the cloud
      
      declare -A personalityMnemonicNoEre
      personalityOptions=""
      for personalityKey in "${!personalityMnemonic[@]}"
      do
         if [ $personalityKey -ne $SbxNoErePersonality ]
         then
            personalityMnemonicNoEre[$personalityKey]=${personalityMnemonic[$personalityKey]}
           if [ -n "$personalityOptions" ]; then
             personalityOptions="$personalityOptions "
           fi
           personalityOptions="${personalityOptions}${personalityKey}=${personalityMnemonicNoEre[$personalityKey]}"
        fi
      done


      $ECHO -n "Enter routing engine personality ($personalityOptions): "
      read personality
      if [ "${personalityMnemonicNoEre[$personality]+exists}" ]; then
         break
      else
         $ECHO -e "Invalid personality. Please re-enter."
      fi
   done

   # Prompt for leader election mode
   while [ 1 ]
   do
      $ECHO -n "Enter leader election mode ($leaderElectOptions): "
      read leaderElect
      case $leaderElect in
      0|1)
         break
         ;;
      *)
         $ECHO -e "Invalid leader election setting. Please re-enter."
         ;;
      esac
   done

   # Prompt for bond device monitoring mode
   if [[ "$hostType" != "ConnexIP5000" ]]; then
      while [ 1 ]
      do
         $ECHO -n "Enter bond device monitoring mode ($bondMonitorOptions): "
         read bondMonitor
         case $bondMonitor in
         0|1)
            break
            ;;
         *)
            $ECHO -e "Invalid bond device monitoring setting. Please re-enter."
            ;;
         esac
      done
   fi

   if [[ "$hostType" == "ConnexIP5000" ]]; then
      # Prompt for haMode...
      while [ 1 ]
      do
         $ECHO -n "Enter the SBC HA mode (Nto1/1to1): "
         read haMode
         case $haMode in
         Nto1)
            haMode="Nto1"
            break
            ;;
         1to1)
            haMode="1to1"
            break
            ;;
         *)
            $ECHO -n "Invalid input. Please re-enter (Nto1/1to1): "
            ;;
         esac
      done

      # Prompt for rgIp.
      while [ 1 ]
      do
         $ECHO -n "Enter the SBC RG IP.(Peer's HA IP/Default-169.254.88.1):"
         read rgIp
         if valid_ip $rgIp; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
   fi

   # Only prompt for packet interface config if -d has been specified... 
   if [ "$pktInterfaceCfg" == "y" ]; then
      # Prompt for signaling/media interface configuration...
      while [ 1 ]
      do
         $ECHO -n "Enter $nif3Name (port 3) signaling/media IP (aaa.bbb.ccc.ddd, '0.0.0.0' if not used): "
         read nif3Ipaddr
         if  valid_ip $nif3Ipaddr; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter $nif3Name (port 3) signaling/media netmask (aaa.bbb.ccc.ddd, '0.0.0.0' if not used): "
         read nif3Netmask
         if valid_ip $nif3Netmask; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter $nif4Name (port 4) signaling/media IP (aaa.bbb.ccc.ddd, '0.0.0.0' if not used): "
         read nif4Ipaddr
         if  valid_ip $nif4Ipaddr; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
      while [ 1 ]
      do
         $ECHO -n "Enter $nif4Name (port 4) signaling/media netmask (aaa.bbb.ccc.ddd, '0.0.0.0' if not used): "
         read nif4Netmask
         if valid_ip $nif4Netmask; then
            break
         else
            $ECHO -e "Invalid address. Please re-enter."
         fi
      done
   else
      nif3Ipaddr="0.0.0.0"
      nif3Netmask="0.0.0.0"
      nif4Ipaddr="0.0.0.0"
      nif4Netmask="0.0.0.0"
   fi

   # Prompt for redundancy setup
   if [[ $hostType == "ConnexIP5000" ]]; then 
       while [ 1 ] 
       do
           if [[ "$hostSubType" == "virtualCloud" ]] ; then
               tcon="n"
               break
           fi

           $ECHO -n "   Do you want to enter TIPC ID configuration. 
           Options are: 
           Y. Yes 
           N. No
           Enter your choice: " 
           read tcon 
           case $tcon in 
           y|Y) 
               tcon="y"
               break 
               ;; 
           n|N) 
               tcon="n"
               break 
               ;; 
           *) 
               $ECHO -e "Invalid choice. Please re-enter." 
               ;; 
           esac 
           if [[ $tcon == "y" ]] ; then
               $ECHO -n "Enter unique TIPC_NETID for your system: " 
               read tipcID
               if valid_tipcID $tipcID; then 
                   break 
               else 
                   $ECHO -e "Invalid TIPC_NETID. Please re-enter." 
               fi 
           fi
       done

       while [ 1 ] 
       do
          if [[ "$hostSubType" == "virtualCloud" ]] ; then
              cho="N"
              break
          fi
          $ECHO -n "   Do you want to enter redundancy configuration. 
          Options are: 
          Y. Yes 
          N. No
Enter your choice: " 
          read cho 
          case $cho in 
          y|Y) 
               cho="Y"
            break 
            ;; 
          n|N) 
               cho="N"
            break 
            ;; 
          *) 
            $ECHO -e "Invalid choice. Please re-enter." 
            ;; 
          esac 
       done
       if [[ $cho == "Y" ]]; then 
           $ECHO -n "Inter-CE IP address configuration"
           # enter activePrimaryInterCeIpaddr 
           while [ 1 ]
           do
               $ECHO -n "Enter activePrimaryInterCeIpaddr (169.254.99.1): "
               read activePrimaryInterCeIpaddr
               if valid_ip "$activePrimaryInterCeIpaddr"; then
                   break
               else
                   $ECHO -e "invalid activePrimaryInterCeIpaddr. Please reenter"
               fi
           done
           # enter activeSecondaryInterCeIpaddr 
           while [ 1 ]
           do
               $ECHO -n "Enter activeSecondaryInterCeIpaddr (169.254.99.2): "
               read activeSecondaryInterCeIpaddr
               if valid_ip "$activeSecondaryInterCeIpaddr"; then
                   break
               else
                   $ECHO -e "invalid activeSecondaryInterCeIpaddr. Please reenter"
               fi
           done
           # enter activeInterCeHaPort1Ipaddr
           while [ 1 ]
           do
               $ECHO -n "Enter activeInterCeHaPort1Ipaddr (169.254.99.3): "
               read activeInterCeHaPort1Ipaddr
               if valid_ip "$activeInterCeHaPort1Ipaddr"; then
                   break
               else
                   $ECHO -e "invalid activeInterCeHaPort1Ipaddr. Please reenter"
               fi
           done
           # enter activeInterCeHaPort2Ipaddr
           while [ 1 ]
           do
               $ECHO -n "Enter activeInterCeHaPort2Ipaddr (169.254.99.4): "
               read activeInterCeHaPort2Ipaddr
               if valid_ip "$activeInterCeHaPort2Ipaddr"; then
                   break
               else
                   $ECHO -e "invalid activeInterCeHaPort2Ipaddr. Please reenter"
               fi
           done
           # enter standbyPrimaryInterCeIpaddr 
           while [ 1 ]
           do
               $ECHO -n "Enter standbyPrimaryInterCeIpaddr (169.254.88.1): "
               read standbyPrimaryInterCeIpaddr
               if valid_ip "$standbyPrimaryInterCeIpaddr"; then
                   break
               else
                   $ECHO -e "invalid standbyPrimaryInterCeIpaddr. Please reenter"
               fi
           done
           # enter standbySecondaryInterCeIpaddr 
           while [ 1 ]
           do
               $ECHO -n "Enter standbySecondaryInterCeIpaddr (169.254.88.2): "
               read standbySecondaryInterCeIpaddr
               if valid_ip "$standbySecondaryInterCeIpaddr"; then
                   break
               else
                   $ECHO -e "invalid standbySecondaryInterCeIpaddr. Please reenter"
               fi
           done 
           # enter standbyInterCeHaPort1Ipaddr
           while [ 1 ]
           do
               $ECHO -n "Enter standbyInterCeHaPort1Ipaddr (169.254.88.3): "
               read standbyInterCeHaPort1Ipaddr
               if valid_ip "$standbyInterCeHaPort1Ipaddr"; then
                   break
               else
                   $ECHO -e "invalid standbyInterCeHaPort1Ipaddr. Please reenter"
               fi
           done
           # enter standbyInterCeHaPort2Ipaddr
           while [ 1 ]
           do
               $ECHO -n "Enter standbyInterCeHaPort2Ipaddr (169.254.88.4): "
               read standbyInterCeHaPort2Ipaddr
               if valid_ip "$standbyInterCeHaPort2Ipaddr"; then
                   break
               else
                   $ECHO -e "invalid standbyInterCeHaPort2Ipaddr. Please reenter"
               fi
           done
           while [ 1 ] 
           do 
               $ECHO -n "Enter Default InterCe netmask (255.255.0.0): " 
               read defaultInterCeNetmask 
               if valid_ip $defaultInterCeNetmask; then 
                   break 
               else 
                   $ECHO -e "Invalid address. Please re-enter." 
               fi 
           done
           while [ 1 ] 
           do 
               $ECHO -n "Enter Default netmask (255.255.0.0): " 
               read defaultNetmask 
               if valid_ip $defaultNetmask; then 
                   break 
               else 
                   $ECHO -e "Invalid address. Please re-enter." 
               fi 
           done
           defaultPrefix="32"
       else
           $ECHO "Using default options for redundancy" 
           activePrimaryInterCeIpaddr="169.254.99.1" 
           activeSecondaryInterCeIpaddr="169.254.99.2" 
           activeInterCeHaPort1Ipaddr="169.254.99.3"
           activeInterCeHaPort2Ipaddr="169.254.99.4"
           standbyPrimaryInterCeIpaddr="169.254.88.1" 
           standbySecondaryInterCeIpaddr="169.254.88.2" 
           standbyInterCeHaPort1Ipaddr="169.254.88.3"
           standbyInterCeHaPort2Ipaddr="169.254.88.4"
           defaultInterCeNetmask="255.255.0.0" 
           defaultNetmask="255.255.0.0"
           defaultPrefix="32"
       fi
   else 
       activePrimaryInterCeIpaddr="169.254.99.1" 
       activeSecondaryInterCeIpaddr="169.254.99.2" 
       activeInterCeHaPort1Ipaddr="169.254.99.3"
       activeInterCeHaPort2Ipaddr="169.254.99.4"
       standbyPrimaryInterCeIpaddr="169.254.88.1" 
       standbySecondaryInterCeIpaddr="169.254.88.2" 
       standbyInterCeHaPort1Ipaddr="169.254.88.3"
       standbyInterCeHaPort2Ipaddr="169.254.88.4"
       defaultInterCeNetmask="255.255.0.0" 
       defaultNetmask="255.255.0.0" 
       defaultPrefix="32"
   fi

   # Build and write the boot configuration file...
   logMsg $logFile "Building $sonusConfigFile..."
   buildConfigFile
}

buildConfigFile()
{
   $ECHO -e "# $product startup configuration.
#
# Copyright (c) 2010 Sonus Networks, Inc.
# All rights reserved.
#
# Configuration generated by $program:
# `$DATE +'%a %b %e %H:%M:%S %Z %Y'`" > $sonusConfigFile

   $ECHO -e "role=$role             # 1=Active, 2=Standby" >> $sonusConfigFile
   $ECHO -e "systemName=$systemName" >> $sonusConfigFile
   $ECHO -e "ceName=$ceName" >> $sonusConfigFile
   $ECHO -e "peerCeName=$peerCeName" >> $sonusConfigFile
   $ECHO -e "dbgIpaddr=$dbgIpaddr" >> $sonusConfigFile
   $ECHO -e "dbgNetmask=$dbgNetmask" >> $sonusConfigFile
   $ECHO -e "dbgGatewayIpaddr=$dbgGatewayIpaddr" >> $sonusConfigFile
   $ECHO -e "nif1Ipaddr=$nif1Ipaddr" >> $sonusConfigFile
   $ECHO -e "nif1Netmask=$nif1Netmask" >> $sonusConfigFile
   $ECHO -e "nif1GatewayIpaddr=$nif1GatewayIpaddr" >> $sonusConfigFile
   $ECHO -e "nif1IpaddrV6=$nif1IpaddrV6" >> $sonusConfigFile
   $ECHO -e "nif1NetprefixV6=$nif1NetprefixV6" >> $sonusConfigFile
   $ECHO -e "nif1GatewayIpaddrV6=$nif1GatewayIpaddrV6" >> $sonusConfigFile
  
 local hostType=$(getHwType)
   if [[ "$hostType" != "ConnexIP5000" ]];then
       $ECHO -e "nif5Ipaddr=$nif5Ipaddr" >> $sonusConfigFile
       $ECHO -e "nif5Netmask=$nif5Netmask" >> $sonusConfigFile
       $ECHO -e "nif5GatewayIpaddr=$nif5GatewayIpaddr" >> $sonusConfigFile
       $ECHO -e "nif5IpaddrV6=$nif5IpaddrV6" >> $sonusConfigFile
       $ECHO -e "nif5NetprefixV6=$nif5NetprefixV6" >> $sonusConfigFile
       $ECHO -e "nif5GatewayIpaddrV6=$nif5GatewayIpaddrV6" >> $sonusConfigFile
   fi
   if [[ "$hostType" == "SBC5400" ]];then
      $ECHO -e "nif7Ipaddr=$nif7Ipaddr" >> $sonusConfigFile
      $ECHO -e "nif7Netmask=$nif7Netmask" >> $sonusConfigFile
      $ECHO -e "nif7GatewayIpaddr=$nif7GatewayIpaddr" >> $sonusConfigFile
      $ECHO -e "nif7IpaddrV6=$nif7IpaddrV6" >> $sonusConfigFile
      $ECHO -e "nif7NetprefixV6=$nif7NetprefixV6" >> $sonusConfigFile
      $ECHO -e "nif7GatewayIpaddrV6=$nif7GatewayIpaddrV6" >> $sonusConfigFile
      $ECHO -e "nif8Ipaddr=$nif8Ipaddr" >> $sonusConfigFile
      $ECHO -e "nif8Netmask=$nif8Netmask" >> $sonusConfigFile
      $ECHO -e "nif8GatewayIpaddr=$nif8GatewayIpaddr" >> $sonusConfigFile
      $ECHO -e "nif8IpaddrV6=$nif8IpaddrV6" >> $sonusConfigFile
      $ECHO -e "nif8NetprefixV6=$nif8NetprefixV6" >> $sonusConfigFile
      $ECHO -e "nif8GatewayIpaddrV6=$nif8GatewayIpaddrV6" >> $sonusConfigFile
   fi
   $ECHO -e "nif3Ipaddr=$nif3Ipaddr" >> $sonusConfigFile
   $ECHO -e "nif3Netmask=$nif3Netmask" >> $sonusConfigFile
   $ECHO -e "nif4Ipaddr=$nif4Ipaddr" >> $sonusConfigFile
   $ECHO -e "nif4Netmask=$nif4Netmask" >> $sonusConfigFile
   $ECHO -e "ntpServerIpaddr=$ntpServerIpaddr" >> $sonusConfigFile
   $ECHO -e "activePrimaryInterCeIpaddr=$activePrimaryInterCeIpaddr" >> $sonusConfigFile
   $ECHO -e "activeSecondaryInterCeIpaddr=$activeSecondaryInterCeIpaddr" >> $sonusConfigFile
   $ECHO -e "activeInterCeHaPort1Ipaddr=$activeInterCeHaPort1Ipaddr" >> $sonusConfigFile
   $ECHO -e "activeInterCeHaPort2Ipaddr=$activeInterCeHaPort2Ipaddr" >> $sonusConfigFile
   $ECHO -e "standbyPrimaryInterCeIpaddr=$standbyPrimaryInterCeIpaddr" >> $sonusConfigFile
   $ECHO -e "standbySecondaryInterCeIpaddr=$standbySecondaryInterCeIpaddr" >> $sonusConfigFile
   $ECHO -e "standbyInterCeHaPort1Ipaddr=$standbyInterCeHaPort1Ipaddr" >> $sonusConfigFile
   $ECHO -e "standbyInterCeHaPort2Ipaddr=$standbyInterCeHaPort2Ipaddr" >> $sonusConfigFile
   $ECHO -e "defaultInterCeNetmask=$defaultInterCeNetmask" >> $sonusConfigFile
   $ECHO -e "defaultNetmask=$defaultNetmask" >> $sonusConfigFile
   $ECHO -e "timeZoneIndex=$timeZoneIndex" >> $sonusConfigFile
   $ECHO -e "allowSshAccess=$allowSshAccess" >> $sonusConfigFile
   $ECHO -e "sbctype=$sbctype" >> $sonusConfigFile
   $ECHO -e "personality=$personality          # $personalityOptions" >> $sonusConfigFile
   $ECHO -e "fipsMode=$fipsMode" >> $sonusConfigFile
   $ECHO -e "tipcID=$tipcID" >> $sonusConfigFile
   $ECHO -e "leaderElect=$leaderElect" >> $sonusConfigFile
   #added for EMA Componentization enhancement 
   $ECHO -e "enableREST=$enableREST" >> $sonusConfigFile
   $ECHO -e "enableCoreEMA=$enableCoreEMA" >> $sonusConfigFile
   $ECHO -e "enableTS=$enableTS" >> $sonusConfigFile

   if [[ "$hostType" != "ConnexIP5000" ]];then
       $ECHO -e "bondMonitor=$bondMonitor" >> $sonusConfigFile
   fi
   $ECHO -e "haMode=$haMode" >> $sonusConfigFile
   if [[ "$hostType" == "ConnexIP5000" ]];then
       $ECHO -e "rgIp=$rgIp" >> $sonusConfigFile
   fi

   $CHOWN sonusadmin:sonus $sonusConfigFile
   $CHMOD 640 $sonusConfigFile
}

# Utilities to log header/footer for the update/install cases
# Similar 'perl' utilities are present in orca/install/sbxTargetUpgradeUtils.pl
# When making any changes to these methods, please update same methods in sbxTargetUpgradeUtils.pl
rbbnPkgs="bios-update cnxipm cps ema lcamodules pamvalidator sbc sl"
logStartUpdate()
{
  local logfile=$1
  local sbcPkg=""
  logMsg $logFile "=========================================================================="
  logMsg $logFile "Start: Sonus $gServiceName Installation (Initiated through $program)"
  isDebPkgMgmt
  if [ $? -ne 0 ]; then
    currentInstallations=`$RPM -qa --last $rpmDbPath`
  else
    currentInstallations=`$DPKG -l $rbbnPkgs 2>/dev/null | $GREP ^ii | $AWK 'BEGIN {  OFS = "_" }{ print $2, $3, $4 }'`
  fi
  logMsg $logFile "Currently Installed Version(s): \n$currentInstallations"
  logMsg $logFile "=========================================================================="
}

logEndUpdate()
{
  local logfile=$1

  logMsg $logFile "=========================================================================="
  logMsg $logFile "End: Sonus $gServiceName Installation"
  currentInstallations=`$DPKG -l $rbbnPkgs 2>/dev/null | $GREP ^ii | $AWK 'BEGIN {  OFS = "_" }{ print $2, $3, $4 }'`
  logMsg $logFile "Updated Version(s): \n$currentInstallations"
  logMsg $logFile "=========================================================================="
}

# Test for Supported hardware types
checkPlatformHarwareType()
{
   local logFile=$1
   local resultFile=$2
   hwTypeArray=("5100" "5200" "5110" "5210" "7000" "5400")

   $GREP "Manufacturer" $DMIDECODE_OUT | $GREP -q -m 1 "Sonus" > /dev/null 2>&1
   manufacturer=$?   
   if [ $manufacturer -eq 0 ]; then
      hwType=$prodString

      for hwTypeElem in "${hwTypeArray[@]}"
      do
       if [[ "$hwType" == "$hwTypeElem" ]]; then
          return 0
       fi
      done

      logMsg $logFile "ERROR: The version being installed/updated is not supported on $hwType platforms; Possible Recovery Action is to contact Sonus support. Exiting..."
      $PERL $statusUpdater -c -s "CEcheckStatus" -v "checkFailed"
      $PERL $statusUpdater -c -s "Reason" -v "UnsupportedHwTypeOn_${ceName}.Possible_Recovery_Action_is:To_Contact_Sonus_support."
      $RM -f $upgradeBaseDir/preUpgradeCheckActive.key 2>/dev/null
      $PERL $statusUpdater -c -s "CheckStatus" -v "checkFailed"
      if [ -n "$resultFile" ]; then
         trapExit 2 $resultFile
      else
         exit 2
      fi
   else
       logMsg $logFile "Not a connexIP hardware platform, skipping hardware type check...."
   fi 
}

# Scan disk to find and evaluate LVMs
# Needed for Revert support
checkDiskforRevert()
{
   local case=$1
   local command=$2
   local logFile=$3
   lvcount=$($LVSCAN 2>/dev/null |$WC -l 2>/dev/null)
   
   # Create common directory 
   $MKDIR -p /home/common/

   # Create Install/Upgrade marker, it will be used by sbx init script after reboot.
   $ECHO "case=$case" > $installUpgradeMarker
   $ECHO "command=$command" >> $installUpgradeMarker 
   $CHOWN root:root $installUpgradeMarker
   logMsg $logFile "Created and changed ownership of installUpgradeMarker file"
   
   if [ $lvcount == 1 ]; then
      prepareDiskforRevertSupport $logFile
   elif [ $lvcount == 3 ]; then
      # Check if it is install case and installation should be on P1.
      if [ $case == "INSTALL" ]; then
         # Find current mounted partition
         mountedLV=$($DF -kh |$GREP debian-root |$HEAD -n1 | $AWK '{print $1}')
         if [[ $mountedLV == "/dev/mapper/debian-root1" ]]; then
            targetLV=/dev/mapper/debian-root2
         else
	        targetLV=/dev/mapper/debian-root1
         fi

         logMsg $logFile "Exiting checkDiskforRevert() and continuing with Installation on $mountedLV.."
         $RM -f $installUpgradeMarker 
         # Lets create a marker to indicate sbxInstall to update p2 rpm db with sonusdb rpm.
         $ECHO "Marker created by $0, to update rpmdb on root=$targetLV" > $updateP2RpmdbMarker
         $ECHO "Upgrade in progress..." 2>/dev/null > $pmMarkerFile
         return 0
      fi
   else
      logMsg $logFile "Cannot proceed with install/upgrade.. Number of LVMs=$lvcount"
      return 1	 
   fi
   return 0
}

# Check for revert possibility
revertCapable()
{
  if [[ "$hwType" != "ConnexIP5000" && "$prodString" != "5100" ]]; then
      return 0
  else
      return 1
  fi
}

prepareDiskforRevertSupport()
{
  local logFile=$1

  # need to add Checks to see if we are good for partitioning..
  pushd $sonusRoot > /dev/null
  $RM -fr *.rpm *.deb *.md5 os_up*
  popd > /dev/null

  $REVERT_UTIL_SH ${primary_disk}7 &>> $logFile
  $SWITCH_BOOT_SH ${primary_disk}7 &>> $logFile
  logMsg $logFile ""
  logMsg $logFile "********************************************************************************************************"
  logMsg $logFile "PLEASE NOTE:"
  logMsg $logFile ""
  logMsg $logFile "THIS SERVER WILL BE TAKEN OUT OF SERVICE AND REBOOTED TO PERFORM INSTALL/UPGRADE."
  logMsg $logFile "PLEASE CLOSE YOUR BROWSER SESSION AND RE-LOGIN AFTER A FEW MINUTES !!!!"
  logMsg $logFile "********************************************************************************************************"
  logMsg $logFile ""

  runServiceCmd "sbx" "stop" 0

  # Find state of drbd, if it is Inconsistent, wait for some time to complete partial sync(if any).
  count=0
  while [ $count -lt 3 ]
  do
     dState=`$DRBDADM dstate mirror| $AWK -F "/" {'printf $1'}`
     drbdStatus=`$CAT /proc/drbd`
     if [ "$dState" == "Inconsistent" ]; then
        logMsg $logFile "DRBD mirror state is $dState, waiting for 5 secs.."
        logMsg $logFile "DRBD status (waiting in transient state): $drbdStatus"
        $SLEEP 5
     else
        logMsg $logFile "DRBD mirror state is $dState"
        logMsg $logFile "DRBD status (about to stop DRBD): $drbdStatus"
        $DRBDADM disconnect mirror >/dev/null 2>&1 
        $SLEEP 1
        $DRBDADM secondary mirror >/dev/null 2>&1 
        $DRBDADM detach mirror >/dev/null 2>&1 
        runServiceCmd "drbd" "stop" 0 >/dev/null 2>&1 
        break 
     fi
     count=`$EXPR $count + 1`
  done

  if [ $count -eq 3 ]; then
     logMsg $logFile "Warning! DRBD mirror state is not UpToDate"
  fi

  # change the sbxStart to avoid start by PM/incorrect run levels
  PROG=${0##*/}
  $SED -i.preUpdate --follow-symlinks '/^\s*start)/s/$/\n       if [ -e \/tmp\/rebootInProgressKey.key ];then \n           $LOGGER -t \$PROG "Called with start and \/tmp\/rebootInProgressKey.key exists, returning..." \n           exit 0 \n        fi/' $ETC_INITD_SBX
  $LOGGER -t $PROG "Updated SBX script for reboot case to check if rebootInprogressKey exists..."

  $ECHO "About to reboot" > /tmp/rebootInProgressKey.key; $CHMOD 640 /tmp/rebootInProgressKey.key

  # Below log is necessory since older PMs fail the install if below strings are not found in log.
  logMsg $logFile "After reboot, Upgrading and Restarting SBX service..."

  # Do NOT change /sbin/reboot to $REBOOT as sonusCommands in staging refers to new reboot script which doesn't exist yet 
  /sbin/reboot 

  exit 0
}

# Take backup of home
backupHomeDir()
{
  logMsg $logFile "Taking backup of /home(to be used while reverting).."
  pushd /home > /dev/null 
  # Including Oracle/Postgres folders in the backup. This backup would be used
  # by revert.
  $TAR -czf home_backup.tar.gz `$LS -d * | $EGREP -v "ora|common|libvirt|log|external|staging|sftproot|etc"`

  if [ -e $sonusRoot/home_backup.tar.gz ]
  then
    $RM -f $sonusRoot/home_backup.tar.gz
  fi 
  if [ -e $SONUS_BACKUP_DIR/home_backup.tar.gz ]
  then
    $RM -f $SONUS_BACKUP_DIR/home_backup.tar.gz
  fi

  $MV -f home_backup.tar.gz $HOME_BACKUP_DIR/
  $CHMOD 640 $HOME_BACKUP_DIR/home_backup.tar.gz

  if [ -e $SONUS_EXTERNAL_DIR/sonuscert.p12 ]
  then
    $CP -pf $SONUS_EXTERNAL_DIR/sonuscert.p12 $HOME_BACKUP_DIR/
  fi

  popd > /dev/null
}

# Save ownership/permission info of staging/external dirs
saveCommonDirPerms()
{
  if [ -e $HOME_BACKUP_DIR/common_dir_perms ];then
     $RM -f $HOME_BACKUP_DIR/common_dir_perms
  fi

  dirs=($SONUS_EXTERNAL_DIR/ $stagingDir/ $HOME_ORACLE/ /home/orasql/ /home/oraadmin/ /home/oradata/ /home/oraidx/ /home/orasys/ )
  for dir in "${dirs[@]}"
  do
    if [ -d $dir ];then
       logMsg $logFile "Saving ownership/permission info of $dir.."
       $STAT -c "%n %u %g %a" $dir >> $HOME_BACKUP_DIR/common_dir_perms
    else
       logMsg $logFile "$dir doesnt exist!"
    fi
  done
  for sonusDir in `$FIND $SONUS_LOG_DIR/ -type d -print`
  do
    if [ -d $sonusDir ];then
        $STAT  -c "%n %u %g %a" $sonusDir >> $HOME_BACKUP_DIR/common_dir_perms
    fi
  done
  # Save the permissions only if the dir exists.
  if [ -d "$HOME_SFTPROOT" ]; then
    # Save the ownership & permissions for confd users.
    # Exclude itself, dirs which are already added like external, staging or dirs containing system logs like log dir.
    for sonusDir in `$FIND $HOME_SFTPROOT/* -type d ! \( -path $HOME_SFTPROOT/external -prune -o -path $HOME_SFTPROOT/staging -prune -o -path $HOME_SFTPROOT/log -prune \) -print`
    do
        $STAT  -c "%n %u %g %a" $sonusDir >> $HOME_BACKUP_DIR/common_dir_perms
    done
  fi
}

# Assign root ownership to passed dir
assignRootOwnerToCode()
{
    local dir=$1
    $FIND $dir -maxdepth 1 -type f |while read filename
    do
        $FILE -b $filename |$GREP -qEw "executable|shared object|ELF|script|a\.out"
        if [ $? = 0 ]
        then
            $CHOWN root $filename
            $CHMOD go-w $filename
  fi
    done
}

################################################################
## subroutine to determine the major version of either the local
## or remote software.  NOTE: it gets the version from swinfo, so
## the result depends on whether called before or after the new
## software is installed.
## NOTE: older versions of software report the version as
## sbc-V##......  We need to handle deleting the 'sbc-' in case
## it is there.
#################################################################
getSWVersion()
{
  local localRemote=$1
  if [[ "$localRemote" == "local" ]]; then
     swinfoFlag="-l"
  elif [[ "$localRemote" == "remote" ]]; then
     swinfoFlag="-R"
  fi
  version=`$SWINFO_SH $swinfoFlag | $GREP SBC: | $AWK '{print $2}' | $SED -e 's/sbc-//' | $AWK -F. '{print $1}' | $SED -e 's/V//'`
  $ECHO $version
}
# Get sbc config dirver path from serial passed at virt-install
function GetSbcConfigDrive()
{
    local serial=RIBBON_CONFIG_DRIVE
    $READLINK -f /dev/disk/by-id/*$serial
}

createSerfConfigFile()
{
  . $sonusConfigFile
  if [ $role -eq "1" ];then
     ceRole="ACTIVE"
     selfHAIp="$activePrimaryInterCeIpaddr"
     peerHAIp="$standbyPrimaryInterCeIpaddr"
  else
     ceRole="STANDBY"
     selfHAIp="$standbyPrimaryInterCeIpaddr"
     peerHAIp="$activePrimaryInterCeIpaddr"
  fi

  # This needs to be taken care of when swe_nto1 stream is merged.
  if [[ -z "$haMode" && "$hostSubType" == "virtual" ]]; then
     haMode="1to1"
  fi

  # Create a json formatted string since py script takes json dictionary as input.
  inputDict=`printf '{"CEName":"%s","PeerCEName":"%s","ClusterIp":"%s","CERole":"%s","haIp":"%s","SbcHaMode":"%s"}' "$ceName" "$peerCeName" "$peerHAIp" "$ceRole" "$selfHAIp" "$haMode"`
  logMsg $logFile "Calling createSerfConfigFile.py with param:$inputDict"
  $PYTHON3 $CREATE_SERF_CONFIG_FILE_PY $inputDict
}

getHaMode()
{
  # note: this is called by checkDiskUsage.sh which is called preInstallCheck.sh
  # in addition to other places.  When called by preInstallCheck.sh for a new install,
  # the config file won't yet be in place.  Avoid getting a grep 'No such file...' error
  # by making sure the file exists
  if [ -f "$sonusConfigFile" ]; then
     haMode=`$GREP "haMode=" $sonusConfigFile | $AWK -F\= '{print $2}'`
  fi

  # In case haMode is null, default it.
  if [[ "$haMode" == "null" || "$haMode" == "" ]]; then
     if [[ "$hostSubType" == "virtualCloud" ]]; then
        haMode="Nto1"
     else
        haMode="1to1"
     fi
  fi
  $ECHO $haMode
}

function updateDrbdStartupSequence()
{
   logMsg $logFile "Updating DRBD startup sequence..."

   # although should already be disabled for startup, lets make sure
   $UPDATE_RC_D -f drbd remove

   local hostType=$(getHwType)
   # If not 7k or swe, we need to make drbd dependent on the sbxbond
   # service since that is where the HA interface is created.
   # The 7k was already modified to make drbd dependent on cps, and swe
   # has the interface created by the network subsystem.
   if [[ "$hostType" != "SBC7000" && "$hostType" != "ConnexIP5000" ]]; then
       $GREP "Required\-Start:" $ETC_INITD_DRBD | $GREP -q sbxbond
       if [ $? -ne 0 ]; then
           $SED -i -e '/Required-Start/s/$/ sbxbond/' $ETC_INITD_DRBD
           $SYSTEMCTL daemon-reload
           $UPDATE_RC_D drbd defaults
       fi
   fi

   # although the service is disabled, systemd will still try to control
   # its shutdown.  so lets make sure it only stops after sonusInitdSystemDSync,
   # since the app will stop it before that.
   $GREP "X\-Start\-Before:" $ETC_INITD_DRBD | $GREP -q sonusInitdSystemDSync
   if [ $? -ne 0 ]; then
       $SED -i -e '/X-Start-Before/s/$/ sonusInitdSystemDSync/' $ETC_INITD_DRBD
       $SYSTEMCTL daemon-reload
       $UPDATE_RC_D drbd defaults
   fi
   $GREP "X\-Stop\-After:" $ETC_INITD_DRBD | $GREP -q sonusInitdSystemDSync
   if [ $? -ne 0 ]; then
       $SED -i -e '/X-Stop-After/s/$/ sonusInitdSystemDSync/' $ETC_INITD_DRBD
       $SYSTEMCTL daemon-reload
       $UPDATE_RC_D drbd defaults
   fi 
}

function stopDrbdService()
{
   logMsg $logFile "Stopping DRBD and unloading DRBD services..."
   $MOUNT >> $logFile 
   log_dev=$(get_log_dev)
   if [ -z "$log_dev" ]; then
      return -1
   fi
   # Make sure that /dev/sda6 is not mounted from a reboot.
   #  Found sd6 mounted:
   #  first install after an iso
   #  linux shell upgrade from 2.0.x to 4.0.1.x
   $GREP -q $log_dev /etc/mtab
   if [ $? -eq 0 ]; then
      $UMOUNT $log_dev   >> $logFile 2>&1
   fi
 
   # check drbd status, a return of 3 indicates not loaded
   $SERVICE_SH drbd status > /dev/null 2>&1
   if [ $? -ne 3 ]; then

      # check drbd config status
      # note: with image based upgrades, it won't be configured yet and running the
      # the disconnect, secondary and detach commands puts errors in the logs
      $DRBD_OVERVIEW 2>&1 | $GREP -q Unconfigured
      if [ $? -ne 0 ]; then
         # Pause sync, if sync is active
         $DRBDADM cstate mirror 2>&1 | $GREP -qe "SyncSource" -qe "SyncTarget" > /dev/null 
         if [ $? -eq 0 ]; then
            $DRBDADM pause-sync mirror
            $SLEEP 1
         fi

         # kill any process using the drbd mounted file system
         # and then unmount it
         $MOUNT | $EGREP "drbd" > /dev/null 2>&1
         if [ $? -eq 0 ]; then
            drbdDir=`$MOUNT | $EGREP "drbd" | $AWK {'printf $3'}`
            $FUSER -m $drbdDir -n file -s -k
            $SLEEP 1 
         fi

         # Stop DRBD service and unmount /drbd... 
         # if drb.conf is zero length do not issue down.
         # drbd prompts with no configuration.
         if [ -s $ETC_DRBD_CONF ]; then
            $DRBDADM disconnect mirror >> $logFile 2>&1
            $SLEEP 1 
         fi

         unmountMirrorRoot
         $SLEEP 1

         if [ -s $ETC_DRBD_CONF ]; then 
            $DRBDADM secondary mirror >> $logFile 2>&1
            $SLEEP 1
            $DRBDADM detach mirror  >> $logFile 2>&1
         fi
      fi

      # Issue a service stop before the restart. 
      # The stop can report an in use error:
      # Stopping all DRBD resourcesERROR: Module drbd is in use
      $SERVICE_SH drbd stop >> $logFile 2>&1
      $SLEEP 1

      # check drbd status, a return of 3 indicates not loaded
      $SERVICE_SH drbd status > /dev/null 2>&1
      if [ $? -ne 3 ]; then
         $RMMOD drbd >> $logFile
      fi

      # check drbd status, a return of 3 indicates not loaded
      $SERVICE_SH drbd status > /dev/null 2>&1 
      if [ $? -ne 3 ]; then
         $SERVICE_SH drbd stop >> $logFile 2>&1 
         $SLEEP 1
         $RMMOD -s drbd >> $logFile
      fi
   fi
}

function createDrbdConfFile()
{
   local hostType=$(getHwType)
   # Setup DRBD configuration...
   $CP -f $SONUS_SBX_SCRIPT_DIR/drbd.conf /tmp/drbd.conf
   log_dev=$(get_log_dev)
   if [ -z "$log_dev" ]; then
      return -1
   fi
   # In case of LSWU -- upgrade, keep the existing replication mode
   # to keep the active/standby syncing during LSWU without which LSWU
   # won't proceed. Once the upgrade is done, application takes care of
   # updating the mode and do a switchover to get to new mode -- if required.
   local drbdReplicationMode="    protocol A;" # Default replication mode unless in upgrade mode

   if [[ "$preserveDRBDprotocol" == "y" ]]; then
      drbdReplicationMode=`$GREP -E "protocol .;" $ETC_DRBD_CONF.preserve`
   fi

   local installRole=`$SED -n -e '/role/s/role=\([12]\).*/\1/p' $sonusConfigFile`
   local ceName=`$SED -n -e '/ceName/s/ceName=\(.*\)/\1/p' $sonusConfigFile`
   local peerCeName=`$SED -n -e '/peerCeName/s/peerCeName=\(.*\)/\1/p' $sonusConfigFile`

   if [ $installRole -eq 1 ]; then
     localCeIpAddr=`$SED -n -e '/activePrimaryInterCeIpaddr/s/activePrimaryInterCeIpaddr=\(.*\)/\1/p' $sonusConfigFile`
     peerCeIpAddr=`$SED -n -e '/standbyPrimaryInterCeIpaddr/s/standbyPrimaryInterCeIpaddr=\(.*\)/\1/p' $sonusConfigFile`
   else
     localCeIpAddr=`$SED -n -e '/standbyPrimaryInterCeIpaddr/s/standbyPrimaryInterCeIpaddr=\(.*\)/\1/p' $sonusConfigFile`
     peerCeIpAddr=`$SED -n -e '/activePrimaryInterCeIpaddr/s/activePrimaryInterCeIpaddr=\(.*\)/\1/p' $sonusConfigFile`
   fi

   $SED -i -e"s/activePlaceHolderTemp /$ceName /" /tmp/drbd.conf
   $SED -i -e"s/standbyPlaceHolderTemp /$peerCeName /" /tmp/drbd.conf
   $SED -i -e"s/activeIP/$localCeIpAddr/" /tmp/drbd.conf
   $SED -i -e"s/standbyIP/$peerCeIpAddr/" /tmp/drbd.conf
   $SED -i -e"s:MirrorDisk:$log_dev:" /tmp/drbd.conf
   $SED -i -e"s:${primary_device}6:$log_dev:" /tmp/drbd.conf
   $SED -i -e"s/^[ ]*protocol .;/$drbdReplicationMode/" /tmp/drbd.conf
   $SED -i -e"s/  net {/  net { \n    sndbuf-size 0; # allow the kernel to autotune the tcp buffer size/" /tmp/drbd.conf
   $SED -i -e"s/  protocol A;/  protocol A; \n    ko-count          0; #Disabling ko-count feature/" /tmp/drbd.conf

   if [[ "$hostType" == "ConnexIP5000" ]]; then
      $SED -i -e "s/^    rate 128M; /    rate 29M; /" /tmp/drbd.conf
      $SED -i -e "s/^    c-min-rate 128M;/    c-min-rate 29M;/" /tmp/drbd.conf
   fi

   if [ ! -e $ETC_DRBD_CONF.orig ]; then
     $CP -f $ETC_DRBD_CONF $ETC_DRBD_CONF.orig
   fi
   $CP -f /tmp/drbd.conf $ETC_DRBD_CONF
}

function unmountMirrorRoot()
{
  # In some error cases and switching back and forth between old/new
  # seeing that either one of these could be mounted, so try unmounting both
  # NOTE: purposefully not checking error status (i.e. not using executeCommand)
  $UMOUNT /dev/drbd0 2> /dev/null
}

function removeSda6FromFstab()
{
  $GREP -q "$SONUS_LOG_EVLOG_DIR" /etc/fstab 
  if [ $? -eq 0 ]; then 
     $SED  -i".orig" -e "/var\/log\/sonus\/evlog/,2d"  /etc/fstab
  fi 
}

function disableCheckbonding()
{
  $ECHO "Disabling checkBonding service..."

  # remove the checkbonding order dependency in the sbxbond service
  $SED -i -e '/checkbonding/s/^Before/#Before/' $LIB_SYSTEMD_SYSTEM/sbxbond.service
  $SYSTEMCTL daemon-reload

  # stop and mask the checkbonding service so it can't be started
  $SYSTEMCTL stop checkbonding
  $SYSTEMCTL mask checkbonding
}

function enableCheckbonding()
{
  $ECHO "Enabling checkBonding service..."

  # make sure the sbxbond service starts before the checkbonding service
  $SED -i -e '/checkbonding/s/^#Before/Before/' $LIB_SYSTEMD_SYSTEM/sbxbond.service
  $SYSTEMCTL daemon-reload

  # unmask and enable the checkbonding script
  $SYSTEMCTL unmask checkbonding
  $SYSTEMCTL enable checkbonding
}

function performHostSpecificUpdatesForUpgrade()
{
   local logFile=$1
   local basePath=$2
   local lswu=$3
   local hostType=$4
   local kernelVersion=$5

   logMsg $logFile "Updating hostname..."
   hostName=`$CAT /etc/hostname`
   $CHROOT $basePath $HOSTNAME $hostName 2>&1 | $TEE -a $logFile

   if [[ -f $SONUS_EMA_APACHE_DIR/EMA ]]; then
      logMsg $logFile "Saving current EMA configuration file..."
      $CP -pf $SONUS_EMA_APACHE_DIR/EMA $basePath/$SONUS_EMA_APACHE_DIR/EMA
   fi
   $CP -pf /etc/network/interfaces $basePath/etc/network/interfaces

   if [[ -f /opt/sonus/sbx/tailf/state/global.data ]]; then
       logMsg $logFile "Saving current tailf data from /opt/sonus/sbx/tailf/state/global.data..."
       $CP -pf /opt/sonus/sbx/tailf/state/global.data $basePath/opt/sonus/sbx/tailf/state/global.data
   fi
   
   # Extracting run mount details from the new qcow2
   runSearch="run tmpfs"
   if [ -f "$basePath/etc/fstab" ]; then
       logMsg $logFile "Extracting run mount details from  $basePath/etc/fstab..."
       while IFS= read -r line; do
           if [[ $line == *"$runSearch"* ]]; then
               if [[ $line != *nodev* && $line == *nosuid* ]]; then
                   line="${line/nosuid/nodev,nosuid}"
               fi
               runmountLine="$line"
               break
           fi
       done < "$basePath/etc/fstab"
   fi

   # Copy fstab before making any changes to it
   $CP -pf /etc/fstab $basePath/etc/fstab

   if [[ "$hostType" != "ConnexIP5000" ]]; then
      # Start ipmi driver on boot...
      # update /etc/modules on preseed and update
      $SED -i -e "/postInstall/ d" -e "/ipmi_watchdog/ d"  -e "/ipmi_poweroff/ d"  -e "/ipmi_devintf/ d" -e "/ipmi_si/ d" -e "/ipmi_msghandler/ d" -e "/i2c_dev/ d" -e "/ioatdma/ d" $basePath/etc/modules
      $ECHO -n "   # Added by $0.
      ipmi_watchdog
      ipmi_poweroff
      ipmi_devintf
      ipmi_si
      ipmi_msghandler
      i2c_dev
      ioatdma
      " >> $basePath/etc/modules

      # Enable watchdog with 60 second interval
      $SED -i -e '/RuntimeWatchdogSec/s/^#//' -e '/RuntimeWatchdogSec/s/=0/=60/' $basePath/etc/systemd/system.conf

      # Blacklist ioatdma driver and postpone the load to
      # /etc/modules to overcome race condition with igb driver
      logMsg $logFile "Blacklisting ioatdma module..."
      $ECHO "blacklist ioatdma" > $basePath/etc/modprobe.d/ioatdma.conf

      # In jessie the blacklisted modules are not loaded by systemd-modules-load.service,
      # so adding it to rc.local until we find dependencies or other ways to address this
      $SED -i '/^exit 0/i modprobe ioatdma' $basePath/etc/rc.local

      $SED -i '/blacklist nouveau\|blacklist nvidia/d' $basePath/etc/modprobe.d/nvidia.conf

      # Disable sr_mod driver by blacklisting it. With Jessie, random periodic iowaits
      # are observed due to some of the kworkers trying to access this device.
      # Also make sure there is no cdrom entry in /etc/fstab.
      # Not disabling this on ConnexIP5000 since Cloud needs this for configdrive to work.
      logMsg $logFile "Blacklisting sr_mod module..."
      $ECHO "blacklist sr_mod" > $basePath/etc/modprobe.d/sr_mod.conf
      $SED -i -e '/^\/dev\/sr0/d' $basePath/etc/fstab

      # Blacklist LiquidIO.  LiquidIO is enabled starting with our 4.9 kernel in
      # case we need it for SWe.  But we don't want it on h/w since we need our
      # Octeon drivers to manage the octeon chips.
      logMsg $logFile "Blacklisting liquidio module..."
      $ECHO "blacklist liquidio" > $basePath/etc/modprobe.d/liquidio.conf

      # Remove swe_np monit conf file
      $RM -f $basePath/$MONIT_CONFD_DIR/swe_np

      $CP -pf /etc/dhcp/dhclient.conf $basePath/etc/dhcp/dhclient.conf

      # enable serial port for h/w (don't bother logging from the function)
      logMsg $logFile "Unmasking serial-getty@ service..."
      chrootSystemctl $basePath unmask serial-getty@.service

      #ANSSI related changes
      logMsg $logFile "Masking open-vm-tools service..."
      chrootSystemctl $basePath mask open-vm-tools.service $logFile
   fi

   # Check if current openhpi.conf file has the string 'OPENHPI_UNCONFIGURED' indicating new format
   $GREP -q "OPENHPI_UNCONFIGURED" /etc/openhpi/openhpi.conf
   if [ $? -eq 0 ]; then
      logMsg $logFile "Retaining current openhpi conf file..."
      $CP -f /etc/openhpi/openhpi.conf $basePath/etc/openhpi/openhpi.conf
      # due to security concerns the config file cannot have group/world permissions
      $CHMOD 600 $basePath/etc/openhpi/openhpi.conf
   else
      logMsg $logFile "Retaining openhpi conf file from the qcow2 image..."
   fi

   if [ -e $PERSONALITYTYPE_TXT ]; then
      logMsg $logFile "Retaining SBC personality type..."
      $CP -pf $PERSONALITYTYPE_TXT $basePath/$PERSONALITYTYPE_TXT
   else
      logMsg $logFile "File personalitytype.txt does not exist..."
   fi

   $CP -pf /etc/hosts $basePath/etc/hosts
   $CP -pf /etc/hostname $basePath/etc/hostname
   $CP -pf $ETC_DRBD_CONF $basePath$ETC_DRBD_CONF
   $CP -pf $ETC_DRBD_CONF $basePath$ETC_DRBD_CONF.preserve

   # Retain the softlink localtime
   $CP -Pf /etc/localtime $basePath/etc/
   $CP -pf /etc/ntp.conf $basePath/etc/
   $CP -pf /etc/timezone $basePath/etc/

   $CP -pf $CONFD_CONF /home/admin/

   #If upgrading from 9.0, in sbx.conf enableREST may be appended in the same line as other
   #Putting a new line character to separate so that LSWU does not fail.
   if [ -e $sonusConfigFile ]; then
      $GREP -q [a-zA-Z0-9]enableREST $sonusConfigFile
      # Fixing sbx.conf if appended "enableREST" is found
      if [[ $? == 0 ]]; then
         logMsg $logFile "Correcting sbx.conf as enableREST is appended to other line."
         $SED -i 's/[a-zA-Z0-9]nableREST/\'$'\nenableREST/g' $sonusConfigFile
      fi
   fi

   if [ -f $sonusConfigFile ]; then
      $CP -pf $sonusConfigFile $basePath/opt/sonus/conf/
   else
      $CP -pf $SBX_CONF $basePath/opt/sonus/conf/
   fi
   
   # If FIPS mode is enabled, handle fips related changes during upgrade
   fipsModeConf=`grep fipsMode $basePath/opt/sonus/conf/sbx.conf | awk -F"=" '{print $2}'`
   logMsg $logFile "Current fips mode from sbx.conf $fipsModeConf"
   if [ "$fipsModeConf" == "enabled" ]; then
      # Make necessary changes when FIPS mode is enabled during upgrade
      logMsg $logFile "Updating OSSL_MODULES during upgrade..."
      echo export OPENSSL_MODULES="/usr/lib/x86_64-linux-gnu/ossl-modules" >> $basePath/etc/profile.d/sonusEnv.sh
      echo export OPENSSL_MODULES="/usr/lib/x86_64-linux-gnu/ossl-modules" >> $basePath/etc/environment

      # Update openssl.cnf with fips mode related changes
      logMsg $logFile "Updating openssl.cnf with FIPS related changes during upgrade"
      if [ -e $basePath/etc/ssl/openssl.cnf ]; then
          $SED -i --follow-symlinks \
          -e 's/openssl_conf = default_conf/#openssl_conf = default_conf/' \
          -e '/providers = provider_sect/s/^# providers/providers/' \
          -e 's/^# default = default_sect/default = default_sect\nrbbnEntropy = rbbnEntropy_sect\nbase = base_sect/' \
          -e '/fips = fips_sect/s/^# fips/fips/' \
          -e '/include fipsmodule.cnf/{s/^# //;s@fipsmodule@/usr/local/ssl/fipsmodule@}' \
          -e '/^# \[provider_sect/s/^# \[provider_sect/ \[provider_sect/' \
          -e '/^# \[default_sect]/{s/^# \[default_sect/\[default_sect/;n;s/^# activate/activate/}' \
          -e '/^activate = 1/a\[rbbnEntropy_sect\]\nactivate = 1\n\[base_sect\]\nactivate = 1' \
          -e '/^providers = provider_sect/arandom = random_sect\nalg_section = evp_properties\n\n\[random_sect\]\nseed = rbbnEntropy\n\n[evp_properties\]\n# Ensure FIPS non-approved algorithms in the FIPS module are suppressed\ndefault_properties = "fips=yes"' \
          $basePath/etc/ssl/openssl.cnf

      else
          logMsg $logFile "$basePath/etc/ssl/openssl.cnf not found"
      fi   
   fi

   # Timezones Asia/Riyadh87 Asia/Riyadh88 and Asia/Riyadh89 don't exist now.
   # Upon upgrade if these timezones are found, we are replacing with Asia-Riyadh
   $SED -i '/timeZoneIndex=/s/\(=312\s*$\|=364\s*$\|=375\s*$\)/=301/' $basePath/opt/sonus/conf/sbx.conf

   # Remove qcow2 swe marker used for replacement install/upgrade
   logMsg $logFile "Removing swe_reconfig_pending marker"
   $RM -f $basePath/$SWE_SBX_RECONFIG_PENDING

   # when we embraced systemd, the location of our service scripts moved.  make sure we properly
   # handle the location change.  additionally, there is no longer any updated need to the script.
   # it hasnt needed updating for a while, but leaving the old code won't hurt as it just won't
   # do anything but be a cp if nothing needs changing.
   # note: we put the old one in place rather than the one from the qcow until sbxInit re-runs
   # as part of reconfig at first app startup.
   if [ "$hostType" != "ConnexIP5000" ]; then
      bondMonitor=`$GREP bondMonitor $basePath/opt/sonus/conf/sbx.conf | $AWK '{print $1}' | $AWK -F"=" '{print $2}'`
      oldEtcInitdSbxBond="/etc/init.d/sbxbond"
      if [ -e $oldEtcInitdSbxBond ];then
         # copy the old sbxbond script to use it temporarily prior to sbxInit.sh
         # setting up the new one.  we need to update the location of sbx.conf
         # within it as it moved during 6.x development.
         $SED -e '/confFile=\/opt\/sonus\/sbx.conf/s|/opt/sonus/sbx.conf|/opt/sonus/conf/sbx.conf|' \
              -e '/SERVICE sbx/s|$SERVICE sbx|/etc/init.d/sbx|' $oldEtcInitdSbxBond > $basePath/$ETC_INITD_SBXBOND
         $CHMOD 755 $basePath/$ETC_INITD_SBXBOND

         # we are upgrading from a release prior to our switching to systemd. make sure that if we are using
         # network-connect bond monitoring that we properly remove the checkbonding dependency - due to a bug
         # we were not previously disabling it.
         # note: stop the currently running service as well
         if [ $bondMonitor -eq 1 ]; then
             logMsg $logFile "Stop and disable still running checkbonding service (due to being in network-connect mode)..."
             /etc/init.d/checkbonding stop
             $SED -i --follow-symlinks -e '/X-Start-Before/s/ checkbonding//' -e '/X-Stop-After/s/ checkbonding//' $basePath/$ETC_INITD_SBXBOND
             $CHROOT $basePath $UPDATE_RC_D -f checkbonding remove
             $SED -i -e '/Default-Start:/s/2 3 4 5//' $basePath/$ETC_INITD_CHECKBONDING
         fi
      elif [ -e $ETC_INITD_SBXBOND ]; then
         # the checkbonding dependency is only in the service file so no manipulation of it here
         $CP -f $ETC_INITD_SBXBOND $basePath/$ETC_INITD_SBXBOND
         $CHMOD 755 $basePath/$ETC_INITD_SBXBOND
      fi

      # re-enable the checkbonding service which the qcow has disabled or leave it masked
      # but remove the default depdency on it from the bond service
      if [ $bondMonitor -eq 0 ]; then
         logMsg $logFile "Unmasking and enabling checkbonding service (direct-connect mode)..."
         chrootSystemctl $basePath unmask checkbonding.service $logFile
         chrootSystemctl $basePath enable checkbonding.service $logFile
      else 
         logMsg $logFile "Removing checkbonding dependency from bond service (network-connect mode)..."
         $SED -i -e '/checkbonding/s/^Before/#Before/' $basePath/$LIB_SYSTEMD_SYSTEM/sbxbond.service
      fi

      # re-enable the bond service which the qcow has disabled
      logMsg $logFile "Unmasking and enabling sbxbond service..."
      chrootSystemctl $basePath unmask sbxbond.service $logFile
      chrootSystemctl $basePath enable sbxbond.service $logFile
   fi

   if [ -e /opt/sonus/.sbxCompList ];then
      logMsg $logFile "Moving sbxCompList to target..."
      $MV -f  /opt/sonus/.sbxCompList $basePath/opt/sonus/
   fi

   # Check and update personality to 1 if set to 2.
   personality=`$CAT $basePath/opt/sonus/conf/sbx.conf |$GREP personality |$AWK -F "=" '{print \$2}'| $AWK '{print \$1}'`
   if [ $personality -eq 2 ]; then
      logMsg $logFile "Changing personality from $personality to 1.."
      $SED -i 's/personality=2/personality=1/' $basePath/opt/sonus/conf/sbx.conf
   fi

   if [ -e $PERFORM_MODEL_UPDATE ]; then
      $MV -f $PERFORM_MODEL_UPDATE $basePath/opt/sonus/installUpgrade/log/
   else
      if [ -e $sonusRoot/performModelUpdate ]; then
         $MV -f $sonusRoot/performModelUpdate $basePath/opt/sonus/installUpgrade/log/
      fi
   fi

   if [ -e $NP_MODE_UPDATE ]; then
      $MV -f $NP_MODE_UPDATE $basePath/opt/sonus/installUpgrade/log/
   else
      if [ -e $sonusRoot/npModeUpdate ]; then
         $MV -f $sonusRoot/npModeUpdate $basePath/opt/sonus/installUpgrade/log/
      fi
   fi

   # Restore User Activity log settings
   rotate=`$GREP -A 1 "/var/log/sonus/cnxipm/admin/userActivity.log {*" /etc/logrotate.d/platformManager | $GREP rotate`
   if [ $? -eq 0 ];then
      numDays=`$ECHO $rotate | $AWK '{print $2}'`
      $SED -i "0,/ rotate.*/s// rotate $numDays/" $basePath/etc/logrotate.d/platformManager
   fi
   if [ -e /opt/sonus/cnxipm/conf/pmLog.conf ]; then
      $MV -f /opt/sonus/cnxipm/conf/pmLog.conf $basePath/opt/sonus/cnxipm/conf/pmLog.conf
   fi

   if [[ $hostType == "SBC7000" ]]; then
      # In SBC7000, ixgbe driver should be loaded as part of CPS service startup and
      # should be after BCM driver loading.
      # So, removing it from the list of modules loaded as part of booting up.
      logMsg $logFile "Blacklisting ixgbe module"
      $ECHO "blacklist ixgbe" > $basePath/etc/modprobe.d/ixgbe.conf
      $CHROOT $basePath $DEPMOD -a $kernelVersion

      # For the 7k we need drbd to be dependent on cps since that starts the network (bcmsdk)
      if [ -e "${basePath}${DRBD_SBX_SERVICE_FILE}" ]; then
         $GREP "After=" ${basePath}${DRBD_SBX_SERVICE_FILE} | $GREP -q "cps.service"
         if [ $? -ne 0 ]; then
            $SED -i -e '/After=/s/$/ cps.service/' ${basePath}${DRBD_SBX_SERVICE_FILE}
         fi
      else
         $GREP "Required-Start" ${basePath}${ETC_INITD_DRBD} | $GREP -q "cps"
         if [ $? -ne 0 ]; then
            $SED -i -e '/Required-Start/s/$/ cps/' ${basePath}${ETC_INITD_DRBD}
         fi
      fi

      $RM -rf $basePath/lib/modules/$kernelVersion/kernel/extra/sonus-bcm-drv-dummy.ko
      $RM -rf $basePath/lib/modules/$kernelVersion/kernel/extra/octeon_drv.ko
   else
      $RM -rf $basePath/lib/modules/$kernelVersion/kernel/extra/sonus-bcm-drv.ko
      if [[ "$hostType" == "SBC5400" ]]; then
         $RM -rf $basePath/lib/modules/$kernelVersion/kernel/extra/octeon_drv.ko
      else
         $RM -rf $basePath/lib/modules/$kernelVersion/kernel/extra/octeon68xx_drv.ko
      fi

      if [ "$hostType" != "ConnexIP5000" ]; then
         # If not 7k or swe, we need drbd to be dependent on sbxbond since that starts the HA interface
         if [ -e "${basePath}${DRBD_SBX_SERVICE_FILE}" ]; then
            $GREP "After=" ${basePath}${DRBD_SBX_SERVICE_FILE} | $GREP -q "sbxbond.service"
            if [ $? -ne 0 ]; then
               $SED -i -e '/After=/s/$/ sbxbond.service/' ${basePath}${DRBD_SBX_SERVICE_FILE}
            fi
         else
            $GREP "Required-Start" ${basePath}${ETC_INITD_DRBD} | $GREP -q "sbxbond"
            if [ $? -ne 0 ]; then
               $SED -i -e '/Required-Start/s/$/ sbxbond/' ${basePath}${ETC_INITD_DRBD}
            fi
         fi
      fi
   fi

   # Modify fstab for mounting the FAT partition in 775 permission.
   # Upgrade case: replace defaults,umask=0000 with defaults,umask=0002
   $SED -i '/bios/s/\ defaults,umask=0000\ /\ defaults,umask=0002\ /' $basePath/etc/fstab
   $SED -i -e 's/ext3/ext4/' $basePath/etc/fstab
   $GREP -q "noatime" $basePath/etc/fstab
   if [ $? -ne 0 ]; then
      $SED -i 's/errors=remount/noatime,errors=remount/' $basePath/etc/fstab
      $SED -i 's/ext4    defaults/ext4    defaults,noatime/' $basePath/etc/fstab
   fi

   $GREP -q "data=writeback,barrier" $basePath/etc/fstab
   if [ $? -eq 0 ]; then
      $SED -i -e 's/data=writeback/data=ordered/' $basePath/etc/fstab
   fi

   $GREP -q "data=ordered,barrier" $basePath/etc/fstab
   if [ $? -ne 0 ]; then
      $SED -i -e '/ext4/s/noatime/noatime,data=ordered,barrier=0/' $basePath/etc/fstab
   fi

   $GREP -q "evlog" $basePath/etc/fstab
   if [ $? -eq 0 ];then
      logMsg $logFile "Removing evlog details from fstab"
      $SED -i -e '/evlog/d' $basePath/etc/fstab
   fi

   $GREP -q "swap" $basePath/etc/fstab
   if [ $? -eq 0 ];then
      logMsg $logFile "Removing swap entry from fstab"
      $SED -i -e '/swap/d' $basePath/etc/fstab
   fi

   if [[ "$hwType" != "ConnexIP5000" && "$prodString" != "5100" ]]; then
      $GREP -q "debian-root1" $basePath/etc/fstab
      if [ $? == 0 ]; then
         $SED -i 's/debian-root1/debian-root2/' $basePath/etc/fstab
      else
         $SED -i 's/debian-root2/debian-root1/' $basePath/etc/fstab
      fi
   fi

   # Add nodev to /boot and /run partition in new /etc/fstab
   if ! $GREP -q "/boot.*nodev" $basePath/etc/fstab; then
       logMsg $logFile "Found existence of boot in /etc/fstab.. Adding nodev option"
       $SED -i "/\/boot/s/\(defaults\)/\1,nodev/" $basePath/etc/fstab
   fi
   
   # Add run mount details to the new /etc/fstab
   if [ -n "$runmountLine" ] ; then
       logMsg $logFile "Adding new run mount details to $basePath/etc/fstab"
       # Remove run mount entry from /etc/fstab if already exists as the already existing entry will have 
       # either no nodev option or a fixed size from previous installation.
       $SED -i -e "/$runSearch/ d" $basePath/etc/fstab
       echo "$runmountLine" | $TEE -a $basePath/etc/fstab > /dev/null
       # Removing size parameter to pickup default size alloted by Debian
       $SED -i '/^tmpfs \/run tmpfs/s/,size=[0-9]*k//' $basePath/etc/fstab
   fi

   filesystems=( $($MOUNT | $EGREP -w 'ext3|ext4' | $GREP -wv 'acl' | $AWK '{print $1}') )
   if [[ ${#filesystems[@]} -gt 0 ]]; then
      for fs in "${filesystems[@]}"; do
         logMsg $logFile "Running tune2fs to set acl for filesystem $fs..."
         $TUNE2FS -o acl $fs >> $logFile 2>&1 || logMsg $logFile "ERROR: failed to add acl to $fs"
      done
   fi

   # Uninstall LCA
   isDebPkgMgmt
   if [ $? -ne 0 ]; then
       $CHROOT $basePath $RPM -e --allmatches --nodeps --force-debian --dbpath /root/.rpmdb lcamodules > /dev/null 2>&1
   else
       $CHROOT $basePath $DPKG -P lcamodules > /dev/null 2>&1
   fi
   logMsg $logFile "Masking lca service..."
   chrootSystemctl $basePath mask lca.service $logFile

   # Hardware/SWe and non-cloud qcow cases, disable cloud-init and all the related services.
   # Leaving these enabled is causing 2+ mins of delay in bootup and errors.
   # note: mask them instead of disable so they cannot be enabled even if someone depends on them.
   logMsg $logFile "Masking cloud-init and related services..."
   $SED -i '/cloud-init/s/^After/#After/' ${basePath}${CPS_SERVICE_FILE}
   $CHROOT $basePath $RM -fr /etc/systemd/system/cloud-init.target.wants /etc/systemd/system/cloud-init.service
   chrootSystemctl $basePath mask cloud-final.service $logFile
   chrootSystemctl $basePath mask cloud-config.service $logFile
   chrootSystemctl $basePath mask cloud-init.service $logFile
   chrootSystemctl $basePath mask cloud-init-local.service $logFile

   # Cleanup cloud related files
   $RM -f $basePath/etc/profile.d/Z99-cloud-locale-test.sh
   $RM -f $CLOUD_DIAGS_PY 
   
   # Cleanup oracle related files
   logMsg $logFile "Cleanup oracle related files in home path..."
   $RM -fr /home/ora*

   # Mask intervalstats service for Hardware/SWe and non-cloud qcow.
   logMsg $logFile "Masking intervalstats service..."
   chrootSystemctl $basePath mask intervalstats.service $logFile

   # Mask trc-anonymization service for Hardware/SWe and non-cloud qcow.
   logMsg $logFile "Masking trc-anonymization service..."
   chrootSystemctl $basePath mask trc-anonymization.service $logFile
   
   # Setup sbx service
   logMsg $logFile "Enabling sbx service..."
   chrootSystemctl $basePath enable sbx.service $logFile

   # serf and serfEventProcessor do not run on Hardware and Swe 1:1, so removing monit files for serf and serfEventProcssor
   # to prevent monit from trying to invoke serf/ serfEventProcessor in case of hardware or Swe 1:1
   # Remove serf monit conf file
   $RM -f $basePath/$MONIT_CONFD_DIR/serf

   # Remove serfEventProcessor monit conf file
   $RM -f $basePath/$MONIT_CONFD_DIR/serfEventProcessor

   if [ "$hostType" != "ConnexIP5000" ]; then
      # Uninstall sonusudev
      logMsg $logFile "Masking sonusudev service..."
      $SED -i "/After=sonusudev/s/^After/#After/" ${basePath}${CPS_SERVICE_FILE}
      chrootSystemctl $basePath mask sonusudev.service $logFile

      # Re-enable openipmi service
      logMsg $logFile "Creating the openipmi service init.d links..."
      $SED -i "/After=openipmi/s/^#After/After/" ${basePath}${CPS_SERVICE_FILE}
      $CHROOT $basePath $UPDATE_RC_D openipmi defaults 2>&1 | $TEE -a $logFile

      # If it is not Swe 1:1, then serfMembership does not run either, so removing serfLogRotate and monit file for serfMembership
      # Remove serfLogRotate file
      $RM -f $basePath/etc/logrotate.d/serfLogRotate

      # Remove serfMembership monit conf file
      $RM -f $basePath/$MONIT_CONFD_DIR/serfMembership

      if [ -f $basePath/etc/udev/rules.d/70-persistent-net.rules ]; then
         $MV -f $basePath/etc/udev/rules.d/70-persistent-net.rules $basePath/etc/udev/rules.d/70-persistent-net.rules.orig
      fi

      # Setup sbxirqbalance service
      logMsg $logFile "Enabling sbxirqbalance service..."
      chrootSystemctl $basePath enable sbxirqbalance.service $logFile

      # Setup sbxcgroups service
      logMsg $logFile "Enabling sbxcgroups service..."
      chrootSystemctl $basePath enable sbxcgroups.service $logFile

      # Setup watchdog service
      logMsg $logFile "Creating watchdog service init.d links..."
      $CHROOT $basePath $UPDATE_RC_D watchdog defaults 2>&1 | $TEE -a $logFile

      # Install bios-update
      if [ "$hostType" == "ConnexIP" -o "$hostType" == "ConnexIP5200" -o "$hostType" == "ConnexIP5100" ]; then

         if [ "$lswu" == "y" ]; then
            $PERL $statusUpdater -r BIOSUpgrade -s Status -v "inProgress"
            $PERL $statusUpdater -r BIOSUpgrade -s StartTime -v "`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`"
         fi

         # note: need to pass current bios version to script since it cannot run dmidecode
         logMsg $logFile "Installing bios-update package..."
         isDebPkgMgmt
         if [ $? -ne 0 ]; then
             $CHROOT $basePath $RPM -e --force-debian --allmatches --dbpath /root/.rpmdb bios-update > /dev/null 2>&1
         else
             $CHROOT $basePath $DPKG -P bios-update > /dev/null 2>&1
         fi
         biosDeb=$(basename $($LS $basePath/opt/sonus/bios-update/bios-update*.deb))
         logMsg $logFile "Installing ${biosDeb}..."
         $CHROOT $basePath $DPKG -i $sonusRoot/bios-update/$biosDeb > /dev/null 2>&1
         currBiosVer=$($DMIDECODE | $GREP "BIOS Ver" | $AWK '{print $5}' | $AWK -F _v '{print $2}')
         # flashrom (BIOS upgrade tool) need /dev/mem and /proc/pci files to update
         # BIOS, if these files are not there it will fail to upgrade BIOS
         $CHROOT $basePath $FINDMNT -n -l -t proc &> /dev/null
         if [ $? -ne 0 ]; then
             logMsg $logFile "Temporarily mounting proc $basePath/proc..."
             $MOUNT -n -t proc none $basePath/proc
             procMount=1
         else
             procMount=0
         fi
         $CHROOT $basePath $FINDMNT -n -l -t devtmpfs &> /dev/null
         if [ $? -ne 0 ]; then 
             logMsg $logFile "Temporarily mounting devtmpfs $basePath/dev..."
             $MOUNT -n -t devtmpfs none $basePath/dev
             devMount=1
         else
             devMount=0
         fi
         $CHROOT $basePath $sonusRoot/bios-update/bios-update.sh start $currBiosVer 2>&1 | $TEE -a $logFile
         local result=${PIPESTATUS[0]}
         if [ $procMount -eq 1 ]; then
             logMsg $logFile "Unmounting temporary $basePath/proc..."
             $UMOUNT -n $basePath/proc
         fi
         if [ $devMount -eq 1 ]; then
             logMsg $logFile "Unmounting temporary $basePath/dev..."
             $UMOUNT -n $basePath/dev
         fi
         if [ $result -ne 0 ]; then
            logMsg $logFile "Installation of bios-update package failed; Exiting..."
            if [ "$lswu" == "y" ]; then
               $PERL $statusUpdater -r BIOSUpgrade -s Status -v "failed"
               $PERL $statusUpdater -r BIOSUpgrade -s Reason -v "Installation_of_bios-update_package_failed"
            fi
            exit 1
         else
            if [ "$lswu" == "y" ]; then
               $PERL $statusUpdater -r BIOSUpgrade -s Status -v "Complete"
               $PERL $statusUpdater -r BIOSUpgrade -s Reason -v "Successful_Completion"
               $PERL $statusUpdater -r BIOSUpgrade -s EndTime -v "`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`"
            fi
         fi
      fi

      # Remove mpstat warning cron job as it is not needed  in the hardware based systems.
      if [ "$hostType" != "ConnexIP5000" ]; then
         $RM -rf $basePath/etc/cron.hourly/mpstatStealWarning
      fi
   fi

   # NP Cleanup for Upgrade
   if [ -e ${basePath}${CLEANUP_NP_SH} ]; then
      $SH ${basePath}${CLEANUP_NP_SH} $hostType $kernelVersion $basePath $logFile
      if [ $? -ne 0 ]; then
         logMsg $logFile "Failed to cleanup NP utils..."
      fi
   else
      logMsg $logFile "Script ${basePath}${CLEANUP_NP_SH} does not exist"
   fi

   # Restore passwords for root and other non-cdb users.
   userArray=( "root" "linuxadmin" "oracle" "cnxipmadmin" )
   for userElem in "${userArray[@]}"
   do
      newPass=$($GREP "^\b$userElem\b[:]" $basePath/etc/shadow |cut -d: -f 2)
      currentPass=$($GREP "^\b$userElem\b[:]" /etc/shadow |cut -d: -f 2)

      if [ "$newPass" == "$currentPass" ]; then
         logMsg $logFile "No password change is required for $userElem"
      else
         logMsg $logFile "Restoring password for $userElem..."
         $CHROOT $basePath $USERMOD -p $currentPass $userElem
      fi
   done

   if [ "$lswu" == "y" ]; then
      $ECHO "case=LSWU" > $basePath/opt/sonus/staging/upgradeMarker
   else
      $ECHO "case=OFFLINE_UPGRADE" > $basePath/opt/sonus/staging/upgradeMarker
   fi

   if [ -f $PSX_GENERIC_KEY ]; then
      logMsg $logFile "Retaining current LI encryption key file..."
      $CP -p $PSX_GENERIC_KEY $basePath/$PSX_GENERIC_KEY
   fi

   # in 11.1 and above pam_tally was changed to pam_faillock
   # pam_tally2 used a file to store the login attempts, but pam_faillock uses a directory
   # If the SONUS_FAILLOG and SONUS_FAILLOG_OS files exist, delete them and create directories
   # with the same names

   if [ -f $SONUS_FAILLOG ]; then
       logMsg $logFilelogMsg $logFile "Found existing file $SONUS_FAILLOG. Deleting it..."
       $RM -f $SONUS_FAILLOG
   fi
   if [ ! -d $SONUS_FAILLOG ]; then
       $MKDIR $SONUS_FAILLOG
   fi
   $CHOWN root:root $SONUS_FAILLOG
   $CHMOD 700 $SONUS_FAILLOG
   
   if [ -f $SONUS_FAILLOG_OS ]; then
       $LOGGER "Found existingfile $SONUS_FAILLOG_OS. Deleting it..."
       $RM -f $SONUS_FAILLOG_OS
   fi
   if [ ! -d $SONUS_FAILLOG_OS ]; then
       $MKDIR $SONUS_FAILLOG_OS
   fi
   
   $CHOWN root:root $SONUS_FAILLOG_OS
   $CHMOD 700 $SONUS_FAILLOG_OS
}

# when running lvm commands for the upgrade, ignore drbd since it will
# not be mounted here
function lvmIgnoreDrbd()
{
    logMsg $logFile "Modifying lvm.conf to temporarily skip device drbd0..."
    $SED -i -e '/ Configuration option devices\/global_filter/i\
\tfilter = [ "r|/dev/drbd0|" ]
' /etc/lvm/lvm.conf
}
function lvmAcknowledgeDrbd()
{
    logMsg $logFile "Reverting temporary change to lvm.conf"
    $SED -i -e '/r|\/dev\/drbd0/d' /etc/lvm/lvm.conf
}

function exitOnQcowMountFailure()
{
  local logFile=$1
  $LVSCAN 2>&1 | $TEE -a $logFile
  $DF -kh 2>&1 | $TEE -a $logFile
  $LVCHANGE -a n /dev/debian/root
  $LVCHANGE -a n /dev/debian/varlog
  $QEMU_NBD -d /dev/nbd0
  $SLEEP 2
  $RM -rf /dev/debian
  $VGRENAME debian_current debian
  $MODPROBE -r nbd
  lvmAcknowledgeDrbd
  exit 1;
}

function checkForNBDError()
{
  local logFile=$1
  $LVSCAN > /tmp/lvscan 2>&1
  $GREP -E 'Input/output error' /tmp/lvscan | $GREP -E 'nbd'
  if [ $? == "0" ];then
    logMsg $logFile "ERROR: Found nbd input/output errors. Exiting..."
    $CAT /tmp/lvscan >> $logFile
    $RM -fr /tmp/lvscan
    $UMOUNT /tmp/root
    $UMOUNT /tmp/varlog
    $LVCHANGE -a n /dev/debian/root
    $LVCHANGE -a n /dev/debian/varlog
    $QEMU_NBD -d /dev/nbd0
    $RM -rf /dev/debian
    $VGRENAME debian_current debian
    $MODPROBE -r nbd
    lvmAcknowledgeDrbd
    exit 1;
  fi
}

function performQcow2Mount()
{
    local logFile=$1
    local qcow2Path=$2

    # skip drbd0 since it won't be mounted to this machine
    lvmIgnoreDrbd

    logMsg $logFile "Mounting qcow2 image $qcow2Path..."
    $MODPROBE nbd max_part=8 nbds_max=1
    $VGRENAME debian debian_current
    $SLEEP 2
    if [ -d /dev/debian_current ]; then
      logMsg $logFile "Renamed VG debian to debian_current."
      if [[ "$hwType" != "ConnexIP5000" && "$prodString" != "5100" ]]; then
        if [ -L /dev/debian/root1 ]; then
          logMsg $logFile "Link /dev/debian/root1 still exists. Removing it."
          $RM -f /dev/debian/root1
        fi
        if [ -L /dev/debian/root2 ]; then
          logMsg $logFile "Link /dev/debian/root2 still exists. Removing it."
          $RM -f /dev/debian/root2
        fi
      else
        if [ -L /dev/debian/root ]; then
          logMsg $logFile "Link /dev/debian/root still exists. Removing it."
          $RM -f /dev/debian/root
        fi
        if [ -L /dev/debian/varlog ]; then
          logMsg $logFile "Link /dev/debian/varlog still exists. Removing it."
          $RM -f /dev/debian/varlog
        fi
      fi
    else
      logMsg $logFile "Directory /dev/debian_current doesn't exists. VG rename failed. Exiting..."
      $LVSCAN 2>&1 | $TEE -a $logFile
      $MODPROBE -r nbd
      exit 1;
    fi

    logMsg $logFile "Connecting the qcow2 image to NBD device nbd0..."
    $QEMU_NBD --connect=/dev/nbd0 -v --port=65300 $qcow2Path > $SONUS_LOG_UPGRADE_DIR/nbd.log 2>&1 &

    $MKDIR -p /tmp/root/
    $MKDIR -p /tmp/varlog/
    $MKDIR -p /tmp/boot/
    $SLEEP 5
    checkForNBDError $logFile

    #request to update partition table in kernel
    $PARTPROBE /dev/nbd0

    $LVSCAN | $GREP -qw "/dev/debian/root"
    if [ $? -eq 0 ]; then
      logMsg $logFile "LV /dev/debian/root found..."
    else
      count=0
      while($SLEEP 2)
      do
        $LVSCAN | $GREP -qw "/dev/debian/root"
        lvRes=$?
        if [ $count == 30 ]; then
          if [ $lvRes -eq 0 ]; then
            logMsg $logFile "LV /dev/debian/root found at count $count..."
            break;
          else
            logMsg $logFile "Waited for a minute, but LV /dev/debian/root not found, Exiting..."
            exitOnQcowMountFailure $logFile
          fi
        elif [ $lvRes -eq 0 ]; then
          logMsg $logFile "LV /dev/debian/root found at count $count..."
          break;
        elif [ $lvRes -ne 0 ]; then
          logMsg $logFile "LV /dev/debian/root not found yet, count=$count. Continue waiting..."
          count=$[$count +1]
        fi
      done
    fi

    logMsg $logFile "Activating the LV /dev/debian/root..."
    $LVCHANGE -a y /dev/debian/root
    $SLEEP 2

    logMsg $logFile "Activating the LV /dev/debian/varlog..."
    $LVCHANGE -a y /dev/debian/varlog
    $SLEEP 2

    $GREP -q ext4 /proc/filesystems
    if [ $? -ne 0 ]; then
      logMsg $logFile "Mounting /dev/debian/root to /tmp/root with FS type EXT3..."
      $MOUNT /dev/debian/root /tmp/root/ -t ext3 -o ro
      if [ $? -ne 0 ];then
        logMsg $logFile "ERROR: Mounting /dev/debian/root failed. Exiting..."
        exitOnQcowMountFailure $logFile
      fi

      logMsg $logFile "Mounting /dev/debian/varlog to /tmp/varlog with FS type EXT3..."
      $MOUNT /dev/debian/varlog /tmp/varlog/ -t ext3 -o ro
      if [ $? -ne 0 ];then
        logMsg $logFile "ERROR: Mounting /dev/debian/varlog failed. Exiting..."
        exitOnQcowMountFailure $logFile
      fi

      logMsg $logFile "Mounting /dev/nbd0p1 to /tmp/boot with FS type EXT3..."
      $MOUNT /dev/nbd0p1 /tmp/boot -t ext3 -o ro
      if [ $? -ne 0 ];then
        logMsg $logFile "ERROR: Mounting /dev/nbd0p1 failed. Exiting..."
        exitOnQcowMountFailure $logFile
      fi
    else
      logMsg $logFile "Mounting /dev/debian/root to /tmp/root with FS type EXT4..."
      $MOUNT /dev/debian/root /tmp/root/ -t ext4 -o ro -o noload
      if [ $? -ne 0 ];then
        logMsg $logFile "ERROR: Mounting /dev/debian/root failed. Exiting..."
        exitOnQcowMountFailure $logFile
      fi

      logMsg $logFile "Mounting /dev/debian/varlog to /tmp/varlog with FS type EXT4..."
      $MOUNT /dev/debian/varlog /tmp/varlog/ -t ext4 -o ro -o noload
      if [ $? -ne 0 ];then
        logMsg $logFile "ERROR: Mounting /dev/debian/varlog failed. Exiting..."
        exitOnQcowMountFailure $logFile
      fi

      logMsg $logFile "Mounting /dev/nbd0p1 to /tmp/boot with FS type EXT4..."
      $MOUNT /dev/nbd0p1 /tmp/boot -t ext4 -o ro -o noload
      if [ $? -ne 0 ];then
        logMsg $logFile "ERROR: Mounting /dev/nbd0p1 failed. Exiting..."
        exitOnQcowMountFailure $logFile
      fi
    fi

    $SLEEP 5
    $GREP -q /tmp/root /etc/mtab
    if [ $? -ne 0 ]; then
      logMsg $logFile "/tmp/root not mounted. Exiting..."
      exitOnQcowMountFailure $logFile
    fi
    $GREP -q /tmp/varlog /etc/mtab
    if [ $? -ne 0 ]; then
      logMsg $logFile "/tmp/varlog not mounted. Exiting..."
      exitOnQcowMountFailure $logFile
    fi
    $GREP -q /tmp/boot /etc/mtab
    if [ $? -ne 0 ]; then
      logMsg $logFile "/tmp/boot not mounted. Exiting..."
      exitOnQcowMountFailure $logFile
    fi
}

function performQcow2Copy()
{
    local logFile=$1
    local basePath=$2

    # basePath is the mount point. we want to delete everything
    # in the mount point, not the mount point itself.
    $MKDIR -p $basePath
    $RM -fr $basePath/*
    logMsg $logFile "Copying contents of qcow (/tmp/root) to $basePath..."

    if [[ "$hwType" != "ConnexIP5000" && "$prodString" != "5100" ]]; then
      qArray=( $($LS -d /tmp/root/* |$GREP -v "/tmp/root/home") )
    else
      qArray=( $($LS -d /tmp/root/*) )
    fi

    for qElem in "${qArray[@]}"
    do
      $CP -af $qElem $basePath/
      sync
    done
    
    logMsg $logFile "Copying contents of qcow (/tmp/varlog) to $basePath/var/log..."
    vlArray=( $($LS -d /tmp/varlog/*) )
    $MKDIR -p $basePath/var/log/
    for vlElem in "${vlArray[@]}"
    do
      $CP -af $vlElem $basePath/var/log/
      sync
    done
}

function performQcow2Unmount()
{
    local logFile=$1
    local qcow2Path=$2

    $SLEEP 5
    # Umount qcow2
    logMsg $logFile "Unmounting qcow2 image $qcow2Path..."
    $UMOUNT /tmp/root
    $UMOUNT /tmp/varlog
    $UMOUNT /tmp/boot
    logMsg $logFile "Unmount done"

    $LVCHANGE -a n /dev/debian/root
    $LVCHANGE -a n /dev/debian/varlog
    logMsg $logFile "LV deactivation done"
    logMsg $logFile "Disconnecting NBD device"
    $QEMU_NBD -d /dev/nbd0
    logMsg $logFile "Disconnect NBD done"
    $SLEEP 10
    logMsg $logFile "Removing /dev/debian dir..."
    $RM -rf /dev/debian/
    logMsg $logFile "Renaming VG debian_current back to debian..."
    $VGRENAME debian_current debian
    logMsg $logFile "Removing NBD module"
    $MODPROBE -r nbd

    lvmAcknowledgeDrbd
}

# Depending upon changeType increases or decreases the count by 1 in file pointed by path
# It also creates a file with 1 as count when changeType is inc and no file exists
# And also removes the file is count is 1 and changeType is dec
function alterCount()
{
    file=$1
    changeType=$2
    count=1

    if [[ -e $file ]]
    then
        read count < $file

        if [ "$count" = "1" ] && [ "$changeType" = "dec" ]
        then
            $RM -f $file
        else
            if [ "$changeType" = "inc" ]; then
                count=$(($count+1))
            elif [ "$changeType" = "dec" ]; then
                count=$(($count-1))
            fi

            $ECHO $count > $file
            $CHOWN sonusadmin:sonus $file
        fi
    elif [[ "$changeType" = "inc" ]]
    then
        $ECHO $count > $file
        $CHOWN sonusadmin:sonus $file
    fi
}

## wrapper to change count by 1 or delete files
function alterRebootAndSoftRestartCount()
{
    changeType="$1"
    countFileName="$2"
    countFilePath=""

    if [ "$countFileName" == "rebootCountMarkerFile" ]; then
        countFilePath=$REBOOT_COUNT
    elif [ "$countFileName" == "softRestartCountMarkerFile" ]; then
        countFilePath=$SOFT_RESTART_COUNT
    elif [ "$countFileName" == "" ]; then
        countFilePath=""
    else
        countFileName="error"
    fi

    if [ "$changeType" = "rm" ]; then
        if [ "$countFilePath" != "" ] && [ -e "$countFilePath" ]; then
            $RM -f $countFilePath;
        fi
    elif [ "$countFileName" == "error" ]; then
        $ECHO "Invalid file name"
    elif [ "$countFilePath" != "" ]; then
        alterCount $countFilePath $changeType
    elif [ "$changeType" == "inc" ] || [ "$changeType" == "dec" ]; then
        alterCount $REBOOT_COUNT $changeType
        alterCount $SOFT_RESTART_COUNT $changeType
    fi
}

# Function to check whether second management port is configured in the system or not.
function isSecondMgmtPortConfigured()
{
    result=`$CAT /proc/net/dev | $GREP "mgt1" | $WC -l`

    if [ $result -eq 1 ]; then
        return 1
    else
        return 0
    fi
}

function cleanupBootFiles()
{
   local bootDir=$1
   local currSbcVer=""

   # remove all but the current image from the boot directory
    if [[ "$hwType" != "ConnexIP5000" && "$prodString" != "5100" ]]; then
        currSbcVer=$($SWINFO_SH -l | $GREP "SBC:" | $AWK '{print $2}')
        $RM -f $($LS /boot/initrd.img* | $GREP -v "$currSbcVer")
        $RM -f $($LS /boot/vmlinuz* | $GREP -v "$currSbcVer")
        $RM -f $($LS /boot/System.map* | $GREP -v "$currSbcVer")
        $RM -f $($LS /boot/config-* | $GREP -v "$currSbcVer")
    else
        $RM -f $($LS /boot/initrd.img*)
        $RM -f $($LS /boot/vmlinuz*)
        $RM -f $($LS /boot/System.map*)
        $RM -f $($LS /boot/config-*)
    fi

   # Copy the new images to the boot directory
   local initrd=$($BASENAME `$LS $bootDir/initrd.img* | $GREP amd64-sonus`)
   local vmlinuz=$($BASENAME `$LS $bootDir/vmlinuz* | $GREP amd64-sonus`)
   local sysMap=$($BASENAME `$LS $bootDir/System.map* | $GREP amd64-sonus`)
   local kConfig=$($BASENAME `$LS $bootDir/config-* | $GREP amd64-sonus`)
   $CP -pf $bootDir/$initrd /boot
   $CP -pf $bootDir/$vmlinuz /boot
   $CP -pf $bootDir/$sysMap /boot
   $CP -pf $bootDir/$kConfig /boot
}

function isDebPkgMgmt()
{
  local cpsPkg=""
  cpsPkg=$($DPKG -l cps 2>/dev/null | $GREP ^ii | $AWK '{ print $2 }')
  if [ "$cpsPkg" == "cps" ]; then
    return 0
  else
    return 1
  fi
}

function isVnfmDeployment()
{
    local isVnfm=0
    if [ -e $CLOUD_USER_DATA_JSON ]; then
        $GREP -q 'NFV:VNFM:ORCHESTRATION' $CLOUD_USER_DATA_JSON
        if [ $? -eq 0 ]; then
            isVnfm=1
        fi
    fi
    $ECHO $isVnfm
}

function validateNetworkInterfaces() 
{
    LOG=$1

    while [[ ! -f $POD_NETWORK_ANNOTATION ]]; do
        log_message "Waiting for existence of $POD_NETWORK_ANNOTATION file." | $TEE -a $LOG
        sleep 1
    done
    log_message "$POD_NETWORK_ANNOTATION file exists." | $TEE -a $LOG

    while [[ ! -s $POD_NETWORK_ANNOTATION  ]]; do
        log_message "$POD_NETWORK_ANNOTATION file is empty. Waiting." | $TEE -a $LOG
        sleep 1
    done
    log_message "$POD_NETWORK_ANNOTATION file has content, moving forward" | $TEE -a $LOG

    # Get the number of PCI devices as seen from the network annotations
    fromAnnotations=$($CAT $POD_NETWORK_ANNOTATION | $JQ '.[] | with_entries(select(.key|contains("device-info")))' | $GREP -v {} | $JQ '.|select(."device-info"."type" == "pci") | length' | $AWK '{s+=$1} END {print s}')

    log_message "Number of PCI devices slot IDs from annoations : $fromAnnotations" | $TEE -a $LOG

    # Get the number of PCI devices as seen from the environment variables
    # populated by the kubernetes.
    fromEnvVariable=$($ENV | grep "PCIDEVICE_" | $GREP -v INFO | $AWK -F '=' '{print $2}' | $TR ',' '\n' | $WC -l)

    log_message "Number of PCI devices slot IDs from environment variable : $fromEnvVariable" | $TEE -a $LOG

    # Check if both the values are same.
    if [[ $fromAnnotations -ne $fromEnvVariable ]]; then
        log_message "Network resource allocation for $POD_TYPE seems incorrect.  Redeploy the helm with correct networks" | $TEE -a $LOG
        return 1
    else
        log_message "Network validation was successful" | $TEE -a $LOG
        return 0
    fi
}

function configureOamMgt0V6() 
{
    role=$1
    LOG=$2
    confFile=$SBXCONF_FILE

    if [[ $role -eq 1 ]];then
        mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv6.primaryMgmtV6[0] $NS_CONFIG_JSON_FILE)
        $ECHO "Obtained mgt0 ip [$mgt0IP] for active OAM from helm charts" | $TEE -a $LOG
    elif [[ $role -eq 2 ]];then
        mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv6.secondaryMgmtV6[0] $NS_CONFIG_JSON_FILE)
        $ECHO "Obtained mgt0 ip [$mgt0IP] for standby OAM from helm charts" | $TEE -a $LOG
    fi

    $ECHO "Flushing all V6 addresses from Mgt0 interface" | $TEE -a $LOG
    $_IP -6 addr flush dev mgt0
    if [[ $? -ne 0 ]]; then
        $ECHO "Failed to flush the IPV6 address from Mgt0 interface" | $TEE -a $LOG
        return 1
    fi

    gwV6=$($JQ -r .$defaultMgt.COMMON_CONFIG.GWV6 $NS_CONFIG_JSON_FILE)
    prefixV6=$($JQ -r .$defaultMgt.COMMON_CONFIG.PrefixV6 $NS_CONFIG_JSON_FILE)
    $ECHO "Allocating $mgt0IP/$prefixV6 on mgt0 interface" | $TEE -a $LOG
    $_IP -6 addr add $mgt0IP/$prefixV6 dev mgt0
    if [[ $? -ne 0 ]]; then
        $ECHO "Failed to add $mgt0IP IPv6 address on mgt0 interface."  | $TEE -a $LOG
        return 2
    fi

    return 0
}

function addStaticRoute()
{
    LOG=$1
    mgt0NetworkFamily=$($JQ -r .mgt0NetworkFamily $SBC_CONFIGMAP_JSON)

    route_count=$($JQ '.routes | length' $SBC_CONFIGMAP_JSON)
    if [[ $route_count -eq 0 ]]; then
        log_message "No static routes defined in config." | $TEE -a $LOG
        return
    fi

    for (( i=0; i<route_count; i++ )); do
        destIp=$($JQ -r ".routes[$i].destIp" $SBC_CONFIGMAP_JSON)
        destPrefix=$($JQ -r ".routes[$i].destPrefix" $SBC_CONFIGMAP_JSON)

        if [[ ! -z "$destIp" ]] && [[ ! -z "$destPrefix" ]]; then
            if [[ $mgt0NetworkFamily == "v6" ]]; then
                gwV6=$($JQ -r .$defaultMgt.COMMON_CONFIG.GWV6 $NS_CONFIG_JSON_FILE)
                log_message "Adding new route [ip -6 route add $destIp/$destPrefix dev mgt0 via $gwV6]" | $TEE -a $LOG
                $_IP -6 route add $destIp/$destPrefix via $gwV6 dev mgt0
            else
                gwV4=$($JQ -r .$defaultMgt.COMMON_CONFIG.GWV4 $NS_CONFIG_JSON_FILE)
                log_message "Adding new route [ip route add $destIp/$destPrefix dev mgt0 via $gwV4]" | $TEE -a $LOG
                $_IP route add $destIp/$destPrefix via $gwV4 dev mgt0
            fi

            if [[ $? -ne 0 ]]; then
                log_message "Failed to add route to $destIp/$destPrefix on mgt0 interface." | $TEE -a $LOG
            fi
        else
            log_message "Skipping invalid route: destIp=$destIp, destPrefix=$destPrefix" | $TEE -a $LOG
        fi
    done
}

function delSourceBasedRoute()
{
    LOG=$1
    role=$2

    mgt0NetworkFamily=$($JQ -r .mgt0NetworkFamily $SBC_CONFIGMAP_JSON)
    table_id="100"
    interace="mgt0"

    # Del the file entry,rule,route for mgt0_table entry
    log_message "SourceBasedRoute: Deleting the mgt0_table entry role[$role]" | $TEE -a $LOG

    if [[ $mgt0NetworkFamily == "v6" ]]; then
        if [[ $role -eq 1 ]];then
            mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv6.primaryMgmtV6[0] $NS_CONFIG_JSON_FILE)
            log_message "Obtained mgt0 ip [$mgt0IP] for active OAM from helm charts" | $TEE -a $LOG
        elif [[ $role -eq 2 ]];then
            mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv6.secondaryMgmtV6[0] $NS_CONFIG_JSON_FILE)
            log_message "Obtained mgt0 ip [$mgt0IP] for standby OAM from helm charts" | $TEE -a $LOG
        fi
        table_name="mgt0_table_v6"
        cmd="$_IP -6"
        gw=$($JQ -r .$defaultMgt.COMMON_CONFIG.GWV6 $NS_CONFIG_JSON_FILE)
    else
        if [[ $role -eq 1 ]];then
            mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv4.primaryMgmtV4[0] $NS_CONFIG_JSON_FILE)
            log_message "Obtained mgt0 ip [$mgt0IP] for active OAM from helm charts" | $TEE -a $LOG
        elif [[ $role -eq 2 ]];then
            mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv4.secondaryMgmtV4[0] $NS_CONFIG_JSON_FILE)
            log_message "Obtained mgt0 ip [$mgt0IP] for standby OAM from helm charts" | $TEE -a $LOG
        fi
        table_name="mgt0_table_v4"
        cmd="$_IP"
        gw=$($JQ -r .$defaultMgt.COMMON_CONFIG.GWV4 $NS_CONFIG_JSON_FILE)
    fi

    # Route would already be deleted by now , hence skipping that 
    # Check if the policy rule exists, if yes delete
    if $cmd rule show | grep -q "from $mgt0IP lookup $table_name"; then
        $cmd rule del from $mgt0IP table $table_name
        if [[ $? -ne 0 ]]; then
            log_message "SourceBasedRoute:Failed to del rule [$cmd rule del from $mgt0IP table $table_name]" | $TEE -a $LOG
        else
            log_message "SourceBasedRoute:Deleted rule  [$cmd rule del from $mgt0IP table $table_name]" | $TEE -a $LOG
        fi
    fi

    # Check if entry exists in rt_tables, if yes delete 
    if grep -q "$table_id $table_name" /etc/iproute2/rt_tables; then
	$SED -i "/$table_id $table_name/d" /etc/iproute2/rt_tables
        log_message "SourceBasedRoute: Deleted table $table_name from rt_tables"
    fi
    return 0
}

function addSourceBasedRoute()
{
    LOG=$1
    role=$2

    mgt0NetworkFamily=$($JQ -r .mgt0NetworkFamily $SBC_CONFIGMAP_JSON)
    table_id="100"
    interace="mgt0"

    # Add the file,rule,route for mgt0_table entry
    log_message "SourceBasedRoute: Updating the mgt0_table entry role[$role]" | $TEE -a $LOG

    if [[ $mgt0NetworkFamily == "v6" ]]; then
        if [[ $role -eq 1 ]];then
            mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv6.primaryMgmtV6[0] $NS_CONFIG_JSON_FILE)
            log_message "Obtained mgt0 ip [$mgt0IP] for active OAM from helm charts" | $TEE -a $LOG
        elif [[ $role -eq 2 ]];then
            mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv6.secondaryMgmtV6[0] $NS_CONFIG_JSON_FILE)
            log_message "Obtained mgt0 ip [$mgt0IP] for standby OAM from helm charts" | $TEE -a $LOG
        fi
        table_name="mgt0_table_v6"
        cmd="$_IP -6"
        gw=$($JQ -r .$defaultMgt.COMMON_CONFIG.GWV6 $NS_CONFIG_JSON_FILE)
    else
        if [[ $role -eq 1 ]];then
            mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv4.primaryMgmtV4[0] $NS_CONFIG_JSON_FILE)
            log_message "Obtained mgt0 ip [$mgt0IP] for active OAM from helm charts" | $TEE -a $LOG
        elif [[ $role -eq 2 ]];then
            mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv4.secondaryMgmtV4[0] $NS_CONFIG_JSON_FILE)
            log_message "Obtained mgt0 ip [$mgt0IP] for standby OAM from helm charts" | $TEE -a $LOG
        fi
        table_name="mgt0_table_v4"
        cmd="$_IP"
        gw=$($JQ -r .$defaultMgt.COMMON_CONFIG.GWV4 $NS_CONFIG_JSON_FILE)
    fi


    # Check if entry exists in rt_tables, if not add
    if ! grep -q "$table_id $table_name" /etc/iproute2/rt_tables; then
        $ECHO "$table_id $table_name" >> /etc/iproute2/rt_tables
        log_message "SourceBasedRoute: Added table $table_name from rt_tables"
    fi

    # Check if route exists in the table, if not add
    if ! $cmd route show table $table_name | grep -q "default via $gw dev mgt0"; then
        log_message "SourceBasedRoute: [$cmd route add default via $gw dev mgt0 table $table_name]" | $TEE -a $LOG
        $cmd route add default via $gw dev mgt0 table $table_name
        if [[ $? -ne 0 ]]; then
            log_message "SourceBasedRoute:Failed to add route [$cmd route add default via $gw dev mgt0 table $table_name]"  | $TEE -a $LOG
            #return 1 Dont exit the startup processa
	fi
    else
        log_message "SourceBasedRoute:Already present route [$cmd route add default via $gw dev mgt0 table $table_name]"  | $TEE -a $LOG
    fi

    # Check if the policy rule exists, if not add 
    if ! $cmd rule show | grep -q "from $mgt0IP lookup $table_name"; then
        log_message "SourceBasedRoute: [$cmd rule add from $mgt0IP table $table_name]" | $TEE -a $LOG
        $cmd rule add from $mgt0IP table $table_name
        if [[ $? -ne 0 ]]; then
            log_message "SourceBasedRoute:Failed to add rule [$cmd rule add from $mgt0IP table $table_name]" | $TEE -a $LOG
            #return 1 Dont exit the startup process
        fi
    else
        log_message "SourceBasedRoute:Already present rule [$cmd rule add from $mgt0IP table $table_name]"  | $TEE -a $LOG
    fi
    return 0
}



function updateNifInfo()
{
    role=$1
    LOG=$2
    mgt0NetworkFamily=$($JQ -r .mgt0NetworkFamily $SBC_CONFIGMAP_JSON)
    
    if [[ $mgt0NetworkFamily == "v6" ]]; then
        if [[ $role -eq 1 ]];then
            mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv6.primaryMgmtV6[0] $NS_CONFIG_JSON_FILE)
            log_message "Obtained mgt0 ip [$mgt0IP] for active OAM from helm charts" | $TEE -a $LOG
        elif [[ $role -eq 2 ]];then
            mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv6.secondaryMgmtV6[0] $NS_CONFIG_JSON_FILE)
            log_message "Obtained mgt0 ip [$mgt0IP] for standby OAM from helm charts" | $TEE -a $LOG
        fi
        # Update the nif1IpaddrV6, nif1NetprefixV6 and nif1GatewayIpaddrV6 here as 
        # it will be needed for cfgValidations
        gwV6=$($JQ -r .$defaultMgt.COMMON_CONFIG.GWV6 $NS_CONFIG_JSON_FILE)
        prefixV6=$($JQ -r .$defaultMgt.COMMON_CONFIG.PrefixV6 $NS_CONFIG_JSON_FILE)
        $SED -i "s/\(nif1IpaddrV6=\).*$/\1$mgt0IP/" $confFile
        $SED -i "s/\(nif1NetprefixV6=\).*$/\1$prefixV6/" $confFile
        $SED -i "s/\(nif1GatewayIpaddrV6=\).*$/\1$gwV6/" $confFile
        # reset ipv4 default nif1Ipaddress and netmask values
        dummyIp="0.0.0.0"
        $SED -i "s/\(nif1Ipaddr=\).*$/\1$dummyIp/" $confFile
	$SED -i "s/\(nif1Netmask=\).*$/\1$dummyIp/" $confFile
    else
        if [[ $role -eq 1 ]];then
            mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv4.primaryMgmtV4[0] $NS_CONFIG_JSON_FILE)
            log_message "Obtained mgt0 ip [$mgt0IP] for active OAM from helm charts" | $TEE -a $LOG
        elif [[ $role -eq 2 ]];then
            mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv4.secondaryMgmtV4[0] $NS_CONFIG_JSON_FILE)
            log_message "Obtained mgt0 ip [$mgt0IP] for standby OAM from helm charts" | $TEE -a $LOG
        fi
        gwV4=$($JQ -r .$defaultMgt.COMMON_CONFIG.GWV4 $NS_CONFIG_JSON_FILE)
        prefixV4=$($JQ -r .$defaultMgt.COMMON_CONFIG.PrefixV4 $NS_CONFIG_JSON_FILE)
        # Update the nif1Ipaddr and nif1Netmask mask here as it will be needed for 
        # cfgValidations
        $SED -i "s/\(nif1Ipaddr=\).*$/\1$mgt0IP/" $confFile
        mask=$(( 0xffffffff ^ ((1 << (32-prefixV4)) -1) ))
        netmask_t=$(( (mask>>24) & 0xff )).$(( (mask>>16) & 0xff )).$(( (mask>>8) & 0xff )).$(( mask & 0xff ))
        $SED -i "s/\(nif1Netmask=\).*$/\1$netmask_t/" $confFile
        $SED -i "s/\(nif1GatewayIpaddr=\).*$/\1$gwV4/" $confFile
    fi
}
function configureOamMgt0V4() 
{
    role=$1
    LOG=$2
    confFile=$SBXCONF_FILE

    if [[ $role -eq 1 ]];then
        mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv4.primaryMgmtV4[0] $NS_CONFIG_JSON_FILE)
        log_message "Obtained mgt0 ip [$mgt0IP] for active OAM from helm charts" | $TEE -a $LOG
    elif [[ $role -eq 2 ]];then
        mgt0IP=$($JQ -r .$defaultMgt.$mgtIpForPod.IPv4.secondaryMgmtV4[0] $NS_CONFIG_JSON_FILE)
        log_message "Obtained mgt0 ip [$mgt0IP] for standby OAM from helm charts" | $TEE -a $LOG
    fi

    log_message "Flushing all V4 addresses from Mgt0 interface" | $TEE -a $LOG
    $_IP -4 addr flush dev mgt0
    if [[ $? -ne 0 ]]; then
        log_message "Failed to flush the IPV4 address from Mgt0 interface" | $TEE -a $LOG
        return 1
    fi

    gwV4=$($JQ -r .$defaultMgt.COMMON_CONFIG.GWV4 $NS_CONFIG_JSON_FILE)
    prefixV4=$($JQ -r .$defaultMgt.COMMON_CONFIG.PrefixV4 $NS_CONFIG_JSON_FILE)
    log_message "Allocating $mgt0IP/$prefixV4 on mgt0 interface" | $TEE -a $LOG
    $_IP addr add $mgt0IP/$prefixV4 dev mgt0
    if [[ $? -ne 0 ]]; then
        log_message "Failed to add $mgt0IP IPv4 address on mgt0 interface." | $TEE -a $LOG
        return 2
    fi

    return 0
}

function configureOamMgt0() 
{
    isMgmtSriov=$1
    LOG=$2
    role=$3

    if [[ $isMgmtSriov == 'true' ]]; then
        $ECHO "Configuring Mgt0 interface of OAM in SRIOV scenario." | $TEE -a $LOG
    else
        $ECHO "Configuring Mgt0 interface of OAM in AF_PACKET scenario." | $TEE -a $LOG
    fi
    if [[ $role -eq 0 ]];then
        $ECHO "Getting the application role of the OAM pod." | $TEE -a $LOG
        $PYTHON3 $SBC_POD_ROLE_SELECTOR
        role=$?
        if [[ $role -eq 0 ]];then
            $ECHO "Failed to get installed role of the pod. Exiting!" | $TEE -a $LOG
            return 1
        elif [[ $role -eq 1 ]];then
            $ECHO "Obtained the installed role as active."  | $TEE -a $LOG
        elif [[ $role -eq 2 ]];then
            $ECHO "Obtained the installed role as standby."  | $TEE -a $LOG
        else
            $ECHO "Obtained unknown installed role $role . Exiting!"  | $TEE -a $LOG
            return 2
        fi
    else
        $ECHO "Obtained the role of OAM pod as $role ." | $TEE -a $LOG
    fi

    mgt0NetworkFamily=$($JQ -r .mgt0NetworkFamily $SBC_CONFIGMAP_JSON)

    if [[ $mgt0NetworkFamily == "v6" ]]; then
        $ECHO "Mgt0 belongs to IPV6 family" | $TEE -a $LOG
        configureOamMgt0V6 $role $LOG
        retCode=$?
        if [[ $retCode -ne 0 ]]; then
            $ECHO "Failed to add IPv6 address on Mgt0 interface of OAM." | $TEE -a $LOG
            $ECHO "Return code : $retCode" | $TEE -a $LOG
            return 3
        fi
    elif [[ $mgt0NetworkFamily == "v4" ]]; then
        $ECHO "Mgt0 belongs to IPV4 family" | $TEE -a $LOG
        configureOamMgt0V4 $role $LOG
        retCode=$?
        if [[ $retCode -ne 0 ]]; then
            $ECHO "Failed to add IPv4 address on Mgt0 interface of OAM." | $TEE -a $LOG
            $ECHO "Return code : $retCode" | $TEE -a $LOG
            return 4
        fi
    else
        $ECHO "Invalid value provided for Mgt0 interface : $mgt0NetworkFamily" | $TEE -a $LOG
        return 5
    fi
}

function isModuleLoaded()
{
    local LOG=$1
    local module=$2

    $LSMOD | $GREP -wq "$module"
    if [[ $? -ne 0 ]]; then
        log_message "$module is not loaded!" | $TEE -a $LOG
        return 1
    fi
    return 0
}

function validateTunDeviceCapability()
{
    local LOG=$1
    
    if [[ $IS_MKNOD_CAPABILITY_ENABLED == "false" ]]; then
        if [[ ! -c /dev/net/tun ]]; then
            log_message "/dev/net/tun device doesn't exist. Redeploy helm chart with MKNOD capability enabled." | $TEE -a $LOG
            return 1
        fi
    fi

    if [[ ! -c /dev/net/tun ]]; then
        # If the /dev/net/tun device doesn't exist, create it.
        $SUDO $MKDIR -p /dev/net
        if [[ $? -ne 0 ]]; then
            log_message "Failed to create /dev/net directory" | $TEE -a $LOG
            return 1
        fi
        $SUDO $CHMOD 0755 /dev/net/
        if [[ $? -ne 0 ]]; then
            log_message "Failed to change the permission of /dev/net directory" | $TEE -a $LOG
            return 1
        fi
        $SUDO $MKNOD /dev/net/tun c 10 200
        if [[ $? -ne 0 ]]; then
            log_message "Failed to create /dev/net/tun device" | $TEE -a $LOG
            return 1
        fi
        $SUDO $CHMOD 666 /dev/net/tun
        if [[ $? -ne 0 ]]; then
            log_message "Failed to change the permission of /dev/net/tun device." | $TEE -a $LOG
            return 1
        fi
    fi
    return 0
}

function validateDedicatedCpu()
{
    LOG=$1

    # Get the content of /etc/podinfo/cpu - No. of allotted CPUs
    numCpu=$($CAT $SWE_CPU_INFO_K8S)
    if [[ $? -ne 0 ]]; then
        $ECHO "Failed to get the number of CPU allocated to the container" | $TEE -a $LOG
        return 1
    fi
    $ECHO "Number of vCPUs alloted to the container : $numCpu" | $TEE -a $LOG

    # Get the content of /opt/sonus/conf/swe/.cpulist_k8s - The CPUs allocated 
    # to the container
    cpuList=$($CAT $SWE_CPU_LIST_K8S | $AWK '{n=split($0, array, ",")} END{print n}')
    if [[ $? -ne 0 ]]; then
        $ECHO "Failed to get list of CPU allocated to the container" | $TEE -a $LOG
        return 2
    fi
    $ECHO "vCPUs actually alloted to the container : $cpuList" | $TEE -a $LOG

    if [[ $numCpu -ne $cpuList ]];then
        $ECHO "Dedicated vCPUs are not allotted to the SBC container." | $TEE -a $LOG
        $ECHO "Configure CPU policy manager to have dedicated vCPUs." | $TEE -a $LOG
        return 3
    else
        $ECHO "Dedicated vCPUs are allocated to the SBC container." | $TEE -a $LOG
        return 0
    fi
}

function perfCollectionProcedure() 
{
    # Procedure for mpstat and pidstat collection.
    LOG=$1
    mpstatLog="$SONUS_LOG_SBXPERF/mpstat.log"
    pidstatLog="$SONUS_LOG_SBXPERF/pidstat.log"
    
    # Get the CPUs allocated to the container
    cpuList=$($CAT $SWE_CPU_LIST_K8S)
    if [[ $? -ne 0 ]]; then
        $ECHO "Failed to get list of CPU allocated to the container" | $TEE -a $LOG
        return 1
    fi

    # Check if mpstat is already running
    mpstatPid=$($PGREP mpstat)
    if [[ $? -eq 0 ]]; then
        $ECHO "The mpstat collection procedure is already running." | $TEE -a $LOG
        $ECHO "PID of mpstat collection procedure is $mpstatPid." | $TEE -a $LOG
    elif [[ $? -eq 1 ]]; then
        if [[ $mpstatPid == "" ]]; then
            # Run mpstat every 5 seconds.
            $ECHO "Starting mpstat collection." | $TEE -a $LOG
            $MPSTAT -P $cpuList 5 >> $mpstatLog &
            if [[ $? -ne 0 ]]; then
                $ECHO "Failed to start mpstat collection." | $TEE -a $LOG
                return 2
            fi
        else
            $ECHO "The mpstat collection procedure is already running." | $TEE -a $LOG
            $ECHO "PID of mpstat collection procedure is $mpstatPid." | $TEE -a $LOG
        fi
    else
        $ECHO "Failed to get the infomation about the running mpstat command" | $TEE -a $LOG
        return 3
    fi

    # Check if pidstat is already running
    pidstatPid=$($PGREP pidstat)
    if [[ $? -eq 0 ]]; then
        $ECHO "The pidstat collection procedure is already running." | $TEE -a $LOG
        $ECHO "PID of pidstat collection procedure is $pidstatPid." | $TEE -a $LOG
    elif [[ $? -eq 1 ]]; then
        if [[ $pidstatPid == "" ]]; then
            # Run pidstat every 5 seconds.
            $ECHO "Starting pidstat collection." | $TEE -a $LOG
            $PIDSTAT -ruh -p ALL 5 >> $pidstatLog &
            if [[ $? -ne 0 ]]; then
                $ECHO "Failed to start pidstat collection." | $TEE -a $LOG
                return 4
            fi
        else
            $ECHO "The pidstat collection procedure is already running." | $TEE -a $LOG
            $ECHO "PID of pidstat collection procedure is $pidstatPid." | $TEE -a $LOG
        fi
    else
        $ECHO "Failed to get the infomation about the running pidstat command" | $TEE -a $LOG
        return 5
    fi

    $ECHO "Started performance statistics collection procedures." | $TEE -a $LOG
    return 0
}


function is_functional_test_deployment()
{
    # This function is applicable only for kubernetes environment.
    # FUNC_TEST is an environment variable passed to the SBC pods.
    # Return 0 if functional test mode of deployment.
    # Return 1 if performance mode of deployment.
    if [[ $FUNC_TEST == "true" ]]; then
        return 0
    else
        return 1
    fi
}


function sanitize_distribution()
{
    LOG_FILE=$1

    # Check for presence of the file.
    if [ ! -f $SWE_CORE_DIST_K8S ]; then
        $ECHO "Core-distribution file $SWE_CORE_DIST_K8S doesn't exist!" | $TEE -a $LOG_FILE
        return 1
    fi
    # Convert single quote to double quote to make the file content a proper 
    # json.
    $SED -i "s/'/\"/g" $SWE_CORE_DIST_K8S
    if [ $? -ne 0 ]; then
        $ECHO "Encountered issue while replacing single quotes with double quotes in $SWE_CORE_DIST_K8S" | $TEE -a $LOG_FILE
        return 2
    fi
    $ECHO "Done sanitizing $SWE_CORE_DIST_K8S file" >> $LOG_FILE
    return 0
}


function get_sig_cores()
{
    # Get the Signaling cores from the $SWE_CORE_DIST_K8S file in container 
    # environment.
    LOG_FILE=$1

    sanitize_distribution $LOG_FILE
    if [[ $? -ne 0 ]]; then
       $ECHO "Failed to sanitize $SWE_CORE_DIST_K8S file." | $TEE -a $LOG_FILE
       return 1
    fi

    # Get sig cores, and remove the quotes.
    sig_cores=$($CAT $SWE_CORE_DIST_K8S | $JQ '.sig' | $SED "s/\"//g")
    if [ $? -ne 0 ]; then
        $ECHO "Encountered issues while parsing $SWE_CORE_DIST_K8S file for sig core" | $TEE -a $LOG_FILE
        return 2
    fi
    $ECHO $sig_cores
    return 0
}


function get_mgmt_cores()
{
    # Get the Management cores from the $SWE_CORE_DIST_K8S file in container 
    # environment.
    LOG_FILE=$1

    sanitize_distribution $LOG_FILE
    if [[ $? -ne 0 ]]; then
       $ECHO "Failed to sanitize $SWE_CORE_DIST_K8S file." | $TEE -a $LOG_FILE
       return 1
    fi

    # Get mgmt cores, and remove the quotes.
    mgmt_cores=$($CAT $SWE_CORE_DIST_K8S | $JQ '.mgmt' | $SED "s/\"//g")
    if [ $? -ne 0 ]; then
        $ECHO "Encountered issues while parsing $SWE_CORE_DIST_K8S file for mgmt core" | $TEE -a $LOG_FILE
        return 2
    fi
    $ECHO $mgmt_cores
    return 0
}


function pin_application_processes()
{
    # Pin application processes to the cores allocated to sig cset.
    # UXPAD/TPAD and NP processes will be already pinned to appropriate cores.
 
    LOG_FILE=$1
 
    # Get sig cores
    sig_cores=$(get_sig_cores $LOG_FILE)
    if [ $? -ne 0 ]; then
        $ECHO "Failed to get the signaling cores from $SWE_CORE_DIST_K8S file" | $TEE -a $LOG_FILE
        return 1
    fi
    $ECHO "Obtained the sig core as $sig_cores" | $TEE -a $LOG_FILE
 
    # Get mgmt cores
    mgmt_cores=$(get_mgmt_cores $LOG_FILE)
    if [ $? -ne 0 ]; then
        $ECHO "Failed to get the mgmt cores from $SWE_CORE_DIST_K8S file" | $TEE -a $LOG_FILE
        return 1
    fi
    $ECHO "Obtained the mgmt core as $mgmt_cores" | $TEE -a $LOG_FILE
 
    # Empty-out the tmp file.
    $ECHO > /tmp/taskset.log
 
    $ECHO "Moving all the signaling processes to SIG CPU set" | $TEE -a $LOG_FILE 
    for proc in $($PS -elfaT | $GREP -vE "SWe_|NP" | $AWK -F " " '{print $5}' | $GREP -v SPID); do
 
        # Check if the process is already pinned to mgmt core. 
        # If yes, then no need to move the corresponding process to the signaling core.
        process_affinity=$($GREP "Cpus_allowed_list" "/proc/$proc/status" | $AWK '{print $2}')
        if [[ $process_affinity == $mgmt_cores ]]; then
            $ECHO "Not moving $proc to sig CPU set"  >> /tmp/taskset.log
        else
            # Move to the signaling CPU set.
            $TASKSET -acp $sig_cores $proc >> /tmp/taskset.log
            if [ $? -ne 0 ]; then
                $ECHO "Failed to set the taskset for pid $proc" | $TEE -a $LOG_FILE
                # Not returning from here intentionally.
            fi
        fi
    done
 
    $ECHO "Done performing pinning of signaling processes to specific cores." | $TEE -a $LOG_FILE
    return 0

}

function isHaMode()
{
    # Set HA mode for Cloud
    haMode=""
    if [[ "$hwType" == "ConnexIP5000" ]]; then
       # Check for the json file before using it as it won't be present in SWe in case of ISO and App installation.
       if [ -e "$CLOUD_INSTANCE_LCA_DATA_JSON" ]; then
           haMode=`$JQ -r '.SbcHaMode' $CLOUD_INSTANCE_LCA_DATA_JSON`
       fi

       # In case haMode is empty or SbcHaMode is not provided in the user data, default it to 'Nto1'/'1to1' for cloud/Swe respectively.
       if [[ "$haMode" == "" || "$haMode" == "null" ]]; then
           if [[ "$hwSubType" == "virtualCloud" ]]; then
               haMode="Nto1"
           else
               haMode="1to1"
           fi
       fi
    else
       haMode="1to1"
    fi
    echo "$haMode"
}

function getContainerId()
{
    # To be called only in kubernetes environment.
    # This function expects the following environment variables defined for the
    # container : 
    # KUBERNETES_SERVICE_HOST
    # POD_RBAC_PATH
    # MY_POD_NAME
    # MY_CONTAINER_NAME

    logFile=$1
    namespace=$($CAT ${POD_RBAC_PATH}/namespace)
    token=$($CAT ${POD_RBAC_PATH}/token)
    cacert=${POD_RBAC_PATH}/ca.crt
    podName=$MY_POD_NAME
    containerName=$MY_CONTAINER_NAME
    retryCounter=10

    if [[ $KUBERNETES_SERVICE_HOST =~ ^[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+$ ]]; then
        apiserver="https://$KUBERNETES_SERVICE_HOST"
    else
        apiserver="https://[$KUBERNETES_SERVICE_HOST]"
    fi

    while [[ $retryCounter > 0 ]]; do
        containerId=$($CURL -s -g --max-time 10 --connect-timeout 2  --interface eth0 --cacert ${cacert} --header "Authorization: Bearer ${token}" -X GET ${apiserver}/api/v1/namespaces/$namespace/pods/$podName | $JQ .status.containerStatuses | $JQ -r ".[] | select(.name == \"$containerName\") | .containerID")
        retCode=$?
        if [[ $retCode -ne 0 ]]; then
            $ECHO "Failed to retrieve the container ID, error code : $retCode" | $TEE -a $logFile
            retryCounter=$((retryCounter - 1))
            $SLEEP 1
        elif [[ $containerId == "" ]]; then
            $ECHO "Obtained null value for the container ID" | $TEE -a $logFile
            retryCounter=$((retryCounter - 1))
            $SLEEP 1
        else
            # Got the container ID.
            break
        fi
    done

    if [[ $containerId == "" ]]; then
        $ECHO "Failed to get the container ID even after 10 retries for $containerName of $podName in $namespace"  | $TEE -a $logFile
        return 1
    fi

    $ECHO $containerId | $AWK -F "//" '{print $2}'
    return 0
}
