#!/usr/bin/perl -w

#############################################################
#
# Copyright (c) 2016 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# upgradeManager.pl
#
# Dhananjaya Ajeru
# 22/1/2016
#
# Module Description:
#
# This is the driver script for Upgrade Manager feature and 
# provides interfaces  to presentation layer to:
#
# 1. Start preUpgradeChecks
# 2. Get Status of preUpgradeChecks
# 3. Start LSWU
# 4. Get Status of LSWU
#
# This script is responsible to start active upgrade automatically after
# standby upgrade completes.
#
#############################################################

BEGIN {
  # locate packages in places other than the standard perl path
  unshift(@INC,"/opt/sonus/bin");
  unshift(@INC,"/opt/sonus/staging");
}

no warnings 'once';

my $stagingDir="/opt/sonus/staging";
my $externalDir="/opt/sonus/external";
my $preUpgradeScript="";
my $statusUpdater="$stagingDir/statusUpdater.pl";
my $sbxPeerExpect="$stagingDir/sbxPeerExpect.pl";
my $goAway = 0;
my $monitorInterval = 30;
use Data::Dumper;
use File::Basename;
use File::Copy qw(copy);
use sonusCommands;
use sonusCommonFiles;
use sbxCommonFunctions;

my $updateRcLocal="$SUDO $stagingDir/updateRcLocal.sh"; # sudoers-www-data
if ( ! -e "$stagingDir/updateRcLocal.sh" )
{
  # The file may have already been removed from staging
  $updateRcLocal="$SUDO /opt/sonus/sbx/scripts/updateRcLocal.sh"; # sudoers-www-data
}

my $upgradeBaseDir="/var/log/sonus/upgrade";
my $upgradeDir="$upgradeBaseDir/current";
my $upgradeCheckDir="$upgradeBaseDir/preChecks";
my $preUpgradeCheckMarker="$upgradeCheckDir/preUpgradeCheckStatus";
`$MKDIR -p $upgradeBaseDir`;

END {
   copy("$SONUS_CNXIPM_CONF_DIR/logrotate.conf", $ETC_LOGROTATE_PM);
}

# upgrade group not defined in older release
# give write permission to world.
# will be reverted after new OS is installed
if(!getgrnam("upgrade"))
{
    `$CHMOD -R 777 $upgradeBaseDir`;
}
else
{
    `$CHMOD -R g+w $upgradeBaseDir`;
}

my $umdLogFile="$upgradeBaseDir/umd.log";
my $lswuLogFile="$upgradeBaseDir/latest/softwareUpgrade.log";
my $preUpgradeCheckLogs="$upgradeCheckDir/preUpgradeCheck.log";

my $ceName=sbxCommonFunctions::getInstalledceName();
my $peerCeName=sbxCommonFunctions::getInstalledpeerCeName();
my $installedRole=sbxCommonFunctions::getInstalledRole();

my $miscArg ="none";
my $packageName = "unknown";

sub sigTermHandler
{
  print("Received SIGTERM, exiting...");
  $goAway = 1
};

# TBD: Detect any previous instances

$SIG{TERM} = \&sigTermHandler;


sub printUsage()
{
  logUMDMsg("UMD: Usage: $0: startUpgradeCheck  <Package Name>\n upgradeCheckStatus\n startUpgrade <Package Name>\n upgradeStatus\n Cancel<Package Name>\n");
  print("UMD: Usage: $0:\n startUpgradeCheck  <Package Name>\n upgradeCheckStatus\n startUpgrade <Package Name>\n upgradeStatus\n Cancel<Package Name>\n");
  exit(1);
}

#########################################################################
# For a child and main process exits, so that the caller is not blocked
#########################################################################
sub runChild()
{
  my $pid;
  if ($pid = fork)
  {
    # Parent exits so that the caller can keep going, if they didn't used '&'
    exit 1;
  }
  elsif (defined $pid)
  {
    # everything good, return
    chdir "/"; # so that uninstall/install don't cause any cwd errors
    return;
  }
  else
  {
    logMsg("Unable to fork child");
    exit 1;
  }
}

##############################################################
# Utility to log a message to a logFile: logs time stamp also
##############################################################
sub logUMDMsg
{
    my $msg = shift;
    my $dateStr=`$DATE +"%b %d %T"`;
    chomp($dateStr);
    `$ECHO -e "$dateStr: $msg" >> $umdLogFile`;
}

##############################################################
# Utility to log a message to a logFile: logs time stamp also
##############################################################
sub runCliCommandLocal
{
  my $cmd = shift;
  my $out;

  if ( $cmd =~ /startSoftwareUpgrade/ )
  {
    # Running the startSoftwareUpgrade request command in configure mode, without which confd_cli returns error.
    $out=`($SLEEP 1; $ECHO -e "set screen width 512\n configure\n $cmd \n") | $SUDO /opt/sonus/sbx/tailf/bin/confd_cli -u admin --noaaa --noninteractive`; # sudoers-www-data
  }
  else
  {
    $out=`($SLEEP 1; $ECHO -e "set screen width 512\n $cmd \n") | $SUDO /opt/sonus/sbx/tailf/bin/confd_cli -u admin --noaaa --noninteractive`; # sudoers-www-data
  }
  return($out);
}

##############################################################
# Creates staging area to keep status files 
##############################################################
sub setupUpgradeArea
{
  my $operation=shift;
  logUMDMsg("In setupUpgradeArea(), operation=$operation");
  if ( $operation =~ /preChecks/ )
  {
    `$RM -fr $upgradeCheckDir`;
    `$MKDIR -p $upgradeCheckDir`;
    `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $RM -fr $upgradeCheckDir`;
    `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $MKDIR -p $upgradeCheckDir`; 
  }
  elsif ( $operation =~ /upgrade/ )
  {
    `$RM -fr $upgradeDir`;   
    `$MKDIR -p $upgradeDir`;
    `$RM -f $upgradeBaseDir/latest`;
    `$LN -s current $upgradeBaseDir/latest`;
    `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $RM -fr $upgradeDir`;
    `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $MKDIR -p $upgradeDir`;
    `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $RM -f $upgradeBaseDir/latest`;
    `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $LN -s current $upgradeBaseDir/latest`;
  }

  if(!getgrnam("upgrade"))
  {
      `$CHMOD -R 777 $upgradeBaseDir`;
  }
  else
  {
      `$CHMOD -R g+w $upgradeBaseDir`;
  }
  my $peerUpgradeGroup=`/opt/sonus/sbx/scripts/sonusPeerCnx.expect $GREP upgrade /etc/group`;
  if ($peerUpgradeGroup eq "")
  {
    `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $CHMOD -R 777 $upgradeBaseDir`;
  }
}

##############################################################
# Check syncStatus, returns good if syncCompleted. 
##############################################################
sub checkSyncStatus
{
  my $syncStatus="good";
  my $syncEntity="";
  my $cmd = "show table system syncStatus";
  my $out = runCliCommandLocal("$cmd");
  my @lines = split /\n/, $out;
  foreach my $line (@lines)
  {
    if ( $line =~ /Policy Data/ && $line !~ /syncCompleted/)
    {
       logUMDMsg("UMD: Policy Data is not in Sync..!");
       $syncStatus="bad";
       $syncEntity="$syncEntity Policy_Data";
    }
    elsif ( $line =~ /Disk Mirroring/ && $line !~ /syncCompleted/)
    {
       logUMDMsg("UMD: Disk Mirroring is not in Sync..!");
       $syncStatus="bad";
       $syncEntity="$syncEntity Disk_Mirroring";
    }
    elsif ( $line =~ /Configuration Data/ && $line !~ /syncCompleted/)
    {
       logUMDMsg("UMD: Configuration Data is not in Sync..!");
       $syncEntity="$syncEntity Configuration_Data";
       $syncStatus="bad";
    }
    elsif ( $line =~ /Registration Data/ && $line !~ /syncCompleted/)
    {
       logUMDMsg("UMD: Call/Registration Data is not in Sync..!");
       $syncStatus="bad";
       $syncEntity="$syncEntity Call/Registration_Data";
    }
  } 
  return($syncStatus,$syncEntity);
}

##############################################################
# earlyExit: exits the program with a dummy status file
# indicating failure 
##############################################################
sub earlyExit
{
  my $operation=shift;
  my $msg=shift;
  my $dateStr=`$DATE +"%b %d %T"`;
  chomp($dateStr);
  logUMDMsg("In earlyExit(): Operation=$operation");
  if( ! -e "$sbxPeerExpect" )
  {
     `$TAR -C $stagingDir/ -xzf $packageName sbxPeerExpect.pl sbxPeerExpectUtils.pm sbxCommonFunctions.pm`;
  }

  if ( $operation =~ /preChecks/ )
  {
    logUMDMsg("Updating status file with failure..");
    `$ECHO "CheckStatus=checkFailed" > $upgradeCheckDir/preUpgradeCheckStatus`;
    `$ECHO "Reason=$msg" >> $upgradeCheckDir/preUpgradeCheckStatus`;
    `$ECHO -e "$dateStr: $msg" >> $preUpgradeCheckLogs`;
    `$CHMOD go+w $upgradeCheckDir/preUpgradeCheckStatus`;
    `$CHMOD go+w $preUpgradeCheckLogs`;
    
    sleep(10);
    logUMDMsg("Removing preUpgradeCheckActive.key..");
    `$RM -f $upgradeBaseDir/preUpgradeCheckActive.key`;
    if ( $peerCeName !~ /none/)
    {
      `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $RM -f $upgradeBaseDir/preUpgradeCheckActive.key`;
      `$sbxPeerExpect "$ECHO 'CheckStatus=checkFailed' > $upgradeCheckDir/preUpgradeCheckStatus" "shell"`;
      `$sbxPeerExpect "$ECHO 'Reason=$msg' >> $upgradeCheckDir/preUpgradeCheckStatus" "shell"`;
      `$sbxPeerExpect "$ECHO -e '$dateStr: $msg' >> $preUpgradeCheckLogs" "shell"`;
      `$sbxPeerExpect "$CHMOD go+w $upgradeCheckDir/preUpgradeCheckStatus" "shell"`;
      `$sbxPeerExpect "$CHMOD go+w $preUpgradeCheckLogs" "shell"`;
    }
  }
  elsif ( $operation =~ /upgrade/ )
  {
    logUMDMsg("Updating upgrade status file with failure..");
    my $version=`$BASENAME $packageName |$AWK -F "sbc-" '{print \$2}' |$SED 's/.x86_64.*//'`;
    chomp($version);
    my $upgradeStartTime=`$DATE "+%FT%T%z"`;
    chomp($upgradeStartTime);
    my $CurrentRelease=`$SUDO /opt/sonus/sbx/scripts/swinfo.sh -l|$GREP "SBC:"|$AWK '{print \$2}'`; # sudoers-www-data
    chomp($CurrentRelease);
    my( $primary,$secondary)=checkCliUpgradeStatus();
    `$ECHO 'TargetRelease=$version'             >  $upgradeDir/softwareUpgradeStatus`;
    `$ECHO 'CurrentRelease=$CurrentRelease'     >> $upgradeDir/softwareUpgradeStatus`;
    `$ECHO 'UpgradeStartTime=$upgradeStartTime' >> $upgradeDir/softwareUpgradeStatus`;
    `$ECHO 'UpgradeCompleteTime=$upgradeStartTime' >> $upgradeDir/softwareUpgradeStatus`;
    `$ECHO 'UpgradeStatus=upgradeFailed'        >> $upgradeDir/softwareUpgradeStatus`;
    `$ECHO 'primaryUpgradeStatus=$primary'       >> $upgradeDir/softwareUpgradeStatus`;
    `$ECHO 'secondaryUpgradeStatus=$secondary'  >> $upgradeDir/softwareUpgradeStatus`;
    `$ECHO 'Reason=$msg'                        >> $upgradeDir/softwareUpgradeStatus`;	
    `$RM -fr $upgradeBaseDir/latest`;  
    `$LN -s current $upgradeBaseDir/latest`;
    `$ECHO -e '$dateStr: $msg'                  >> $lswuLogFile`; 
    `$CHMOD go+w $upgradeDir/softwareUpgradeStatus`;
    `$CHMOD go+w $lswuLogFile`;
    
    sleep(10);
    logUMDMsg("Removing liveUpgradeActive.key"); 
    `$RM -f $upgradeBaseDir/liveUpgradeActive.key`;
    if ( $peerCeName !~ /none/)
    {
      `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $RM -f $upgradeBaseDir/liveUpgradeActive.key`;
      `$sbxPeerExpect "$ECHO 'TargetRelease=$version'             >  $upgradeDir/softwareUpgradeStatus" "shell"`;
      `$sbxPeerExpect "$ECHO 'CurrentRelease=$CurrentRelease'     >> $upgradeDir/softwareUpgradeStatus" "shell"`;
      `$sbxPeerExpect "$ECHO 'UpgradeStartTime=$upgradeStartTime' >> $upgradeDir/softwareUpgradeStatus" "shell"`;
      `$sbxPeerExpect "$ECHO 'UpgradeCompleteTime=$upgradeStartTime' >> $upgradeDir/softwareUpgradeStatus" "shell"`;
      `$sbxPeerExpect "$ECHO 'UpgradeStatus=upgradeFailed'        >> $upgradeDir/softwareUpgradeStatus" "shell"`;
      `$sbxPeerExpect "$ECHO 'primaryUpgradeStatus=$primary'       >> $upgradeDir/softwareUpgradeStatus" "shell"`;
      `$sbxPeerExpect "$ECHO 'secondaryUpgradeStatus=$secondary'  >> $upgradeDir/softwareUpgradeStatus" "shell"`;
      `$sbxPeerExpect "$ECHO 'Reason=$msg'                        >> $upgradeDir/softwareUpgradeStatus" "shell"`;
      `$sbxPeerExpect "$RM -fr $upgradeBaseDir/latest" "shell"`;
      `$sbxPeerExpect "$LN -s current $upgradeBaseDir/latest" "shell"`;
      `$sbxPeerExpect "$ECHO -e '$dateStr: $msg'                  >> $lswuLogFile" "shell"`;
      `$sbxPeerExpect "$CHMOD go+w $upgradeDir/softwareUpgradeStatus" "shell"`;
      `$sbxPeerExpect "$CHMOD go+w $lswuLogFile" "shell"`;
    }
    `$updateRcLocal -c`;
  }
  exit(1); 

}

##############################################################
# checkCliUpgradeStatus: returns primary and secondary UpgradeStatus
# to the caller.
##############################################################
sub checkCliUpgradeStatus
{
  my $command = "show table system softwareUpgradeStatus";
  my $result = runCliCommandLocal("$command");
  my @lines = split /\n/, $result;
  my $ret1="none";
  my $ret2="none";
  foreach $line (@lines)
  {
   if ( $line =~ /primaryUpgradeStatus/ )
   {
      (my $dummy,$ret1) = split(" ", $line);
   }
   elsif( $line =~ /secondaryUpgradeStatus/ )
   {
     (my $dummy,$ret2) = split(" ", $line);
   }
  }
  #Remove trailing semicolon.
  $ret1 =~ s/\;//g;
  $ret2 =~ s/\;//g;
  return($ret1,$ret2);
}

##############################################################
# startMonitoring() Returns only after issuing LSWU on active
##############################################################
sub startMonitoring
{
  my $package = shift;
  my $integrityCheck = shift;
  my $miscArg = shift;
  my $count=0;
  my $standbyTimeout=0;
  my $activeTimeout=0;
  for (;;)
  {
    if( -e "$statusUpdater" )
    {
      my $upgradeStatus = `$PERL $statusUpdater -g UpgradeStatus`;
      chomp($upgradeStatus);
      if ( $upgradeStatus =~ /none/ )
      {
         logUMDMsg("UMD: No Upgrade is in progress or upgrade not yet started on standby..!");
         # Is this state possible?
      }
      elsif( $upgradeStatus eq "upgradingStandby" )
      {
         if ( $standbyTimeout eq "240")
         {
           logUMDMsg("UMD: Waited for 2 hours for standby upgrade to complete, but standby upgrade state is still inProgress. Manual Intervention required. Failing upgrade and exiting..");
	       `$PERL $statusUpdater -s Reason -v "TimedoutWaitingForStandbyUpgradeToComplete.Possible_Recovery_Action_is:To_contact_Sonus_support."`;
           `$PERL $statusUpdater -s UpgradeStatus -v "upgradeFailed"`; 
           `$RM -f $upgradeBaseDir/liveUpgradeActive.key`;
           `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $RM -f $upgradeBaseDir/liveUpgradeActive.key`;
           `$updateRcLocal -c`;
           exit(1); 
         }
         else
         {
           $standbyTimeout ++;
         } 
         my $currStep=`$PERL $statusUpdater -g CurrentStep`;
         chomp($currStep);
         logUMDMsg("UMD: Upgrade is in progress on peer server..CurrentStep=$currStep");
         
      }
      elsif( $upgradeStatus eq "postStandbyUpgradeChecks" )
      {
         logUMDMsg("UMD: PostUpgradeChecks are in progress on Peer server..");
      } 
      elsif( $upgradeStatus eq "upgradingStandbyDone" )
      {
         if( $miscArg ne "activeMonitor" )
         {
            # sleep for 10 seconds so that SM messaging etc completes.
            sleep(10);
            logUMDMsg("UMD: Starting Upgrade on active server..");  
            my $cliCmd = "request system serverAdmin $ceName startSoftwareUpgrade package $package integrityCheck $integrityCheck";  
            logUMDMsg("UMD: Calling CLI upgrade command: $cliCmd");  
            my $result = runCliCommandLocal("$cliCmd");
            $result =~ s/`//g;
            $result =~ s/'//g;
            if ( $result eq "" )
            {
                logUMDMsg("UMD: runCliCommandLocal returned NULL!");
                earlyExit("upgrade", "CLI command to start LSWU returned NULL. Unable to continue. Possible_Recovery_Action_is: Check and make sure both the servers of HA are up. Try active upgrade manually.");
            }
            my @lines = split /\n/, $result;
            foreach $line (@lines)
            {
              if ( $line =~ /result failure/ )
              {
                 logUMDMsg("UMD: CLI command to start LSWU has failed!");
                 my $failReason= join("\n ", @lines);
                 logUMDMsg("UMD: Detailed CLI output:\n $failReason");
                 open(my $fh, '>>', $lswuLogFile);
                 print $fh "CLI command to start LSWU has failed!. Detailed CLI output:\n" ;
                 print $fh $failReason;
                 close $fh;
                 `$PERL $statusUpdater -s Reason -v "CLI_command_to_start_upgrade_failed_on_active.Possible_Recovery_Action_is:To_try_CLI_based_upgrade_on_active."`;
                 `$PERL $statusUpdater -s UpgradeStatus -v "upgradeFailed"`; 	
              }
              elsif ( $line =~ /reason/)
              {
                 logUMDMsg("UMD: $line");
              }
              elsif( $line =~ /syntax error/ )
              {
                 logUMDMsg("UMD: CLI command execution failed!");
                 logUMDMsg("UMD: $line");
                 `$PERL $statusUpdater -s UpgradeStatus -v "upgradeFailed"`; 	
                 `$RM -f $upgradeBaseDir/liveUpgradeActive.key`;
                 `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $RM -f $upgradeBaseDir/liveUpgradeActive.key`;
              }
	          elsif( $line =~ /Action Command Timeout/ )
	          {
	             logUMDMsg("UMD: CLI request command timed out!");
	             logUMDMsg("UMD: $line");
	             `$PERL $statusUpdater -s Reason -v "CLI_request_command_timed_out_on_active.Possible_Recovery_Action_is:To_Try_CLI_based_LSWU_for_active."`;
                 `$RM -f $upgradeBaseDir/liveUpgradeActive.key`;
  	             `$PERL $statusUpdater -s UpgradeStatus -v "upgradeFailed"`;		 
	          }
            }
            logUMDMsg("UMD: Stoping UMD..");
            `$updateRcLocal -c`;
            exit(0);
         }
         else
         {
            logUMDMsg("UMD: Waiting for active upgrade to start..miscArg=$miscArg");
         }
      }
      elsif( $upgradeStatus eq "upgradingActive")
      {
         if ( $activeTimeout eq "240")
         {
             logUMDMsg("UMD: Waited for 2 hours for active upgrade to complete, but active upgrade state is still inProgress. Manual Intervention required. Failing upgrade and exiting..");
             `$PERL $statusUpdater -s Reason -v "TimedoutWaitingForActiveUpgradeToComplete.Possible_Recovery_Action_is:To_contact_Sonus_support."`;
             `$PERL $statusUpdater -s UpgradeStatus -v "upgradeFailed"`;
             `$RM -f $upgradeBaseDir/liveUpgradeActive.key`;
             `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $RM -f $upgradeBaseDir/liveUpgradeActive.key`;
             `$updateRcLocal -c`;
             exit(1);
         }
         else
         {
             $activeTimeout ++;
         }
         my $currStep=`$PERL $statusUpdater -g CurrentStep`;
         chomp($currStep);
         logUMDMsg("UMD: Original Active Upgrade is in progress..CurrentStep=$currStep");
      }
      elsif( $upgradeStatus eq "postActiveUpgradeChecks" )
      {
         logUMDMsg("UMD: PostUpgradeChecks are in progress on original active..");
      }
      elsif( $upgradeStatus eq "upgradingActiveDone" )
      {
         logUMDMsg("UMD: Upgrade is complete on original active..");
      }
      elsif( $upgradeStatus eq "upgradeFailed" )
      {
         logUMDMsg("UMD: Upgrade has failed on peer Server..");
         logUMDMsg("UMD: Failure detected.. Stoping UMD..");
         `$updateRcLocal -c`;
         `$RM -f $upgradeBaseDir/liveUpgradeActive.key`;
	     exit(0);
      }
      elsif( $upgradeStatus eq "upgradeComplete" )
      {
         logUMDMsg("UMD: Upgrade Completed..Nothing to do. Exiting..");
         `$updateRcLocal -c`;
         `$RM -f $upgradeBaseDir/liveUpgradeActive.key`;
         `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $RM -f $upgradeBaseDir/liveUpgradeActive.key`;
         exit(0);
      } 
      else
      {
         logUMDMsg("UMD: Unable to handle upgrade state! upgradeStatus='$upgradeStatus' waiting for possible auto-recovery.."); 
         if ( $count eq "60")
         {
           logUMDMsg("UMD: Could not recover!, Failing upgrade and exiting..");
           `$PERL $statusUpdater -s UpgradeStatus -v "upgradeFailed"`; 
           `$RM -f $upgradeBaseDir/liveUpgradeActive.key`;
           `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $RM -f $upgradeBaseDir/liveUpgradeActive.key`;
           `$updateRcLocal -c`;
           exit(1); 
         }
         else
         {
           $count ++;
         } 
      } 
    }
    else
    {
      logUMDMsg("UMD: $statusUpdater tool doesn't exist!");
      logUMDMsg("UMD: Failing update and exiting UMD..");
      # Some sed command to change state as failed.
      exit(0);
    } 
    if ($goAway != 0)
    {
      logUMDMsg("UMD: goAway=$goAway. Stopping UMD..");
      exit($goAway);
    }

    sleep($monitorInterval); 
  } 

}

my %hash;
if( -e "$preUpgradeCheckMarker" )
{
   open (MYFILE, "$preUpgradeCheckMarker");
   while (<MYFILE>)
   {
      if ( $_ !~ /StepNo/ )
      {
        chomp;
        my ($key, $val) = split /=/;
        $hash{$key} .= exists $hash{$key} ? "$val" : $val;
      }
   }
   close (MYFILE);
}

sub startUpgradeCheck
{
  my $package= shift;
  my $localOnly="0";
 
  if( -e "$preUpgradeCheckMarker" )
  {
    if ("$hash{'CheckStatus'}" =~ /checkInProgress/ )
    {
      logUMDMsg("UMD: preUpgradeCheck is in progress");
      exit 1;
    } 
  }
  if( -e "$upgradeBaseDir/preUpgradeCheckActive.key" )
  {
     logUMDMsg("UMD: New request rejected since preUpgradeCheckActive.key key is present!");
     exit 1;
  } 
  setupUpgradeArea("preChecks");
  `$ECHO "preUpradeCheck started" > $upgradeBaseDir/preUpgradeCheckActive.key`;
  if ( -e "$upgradeBaseDir/liveUpgradeActive.key" )
  {
     logUMDMsg("UMD: Cannot start upgradeCheck when LSWU is in progress!");
     earlyExit("preChecks" ,"preUpgradeChecks cannot be started since LSWU is in progress."); 
  }
  if ( $peerCeName =~ /none/ )
  {
    logUMDMsg("UMD: preUpgradeChecks cannot be run on a standalone server!");
    earlyExit("preChecks" ,"preUpgradeChecks cannot be run on a standalone server");
  }
  my $role=sbxCommonFunctions::getCurrentRole("self");
  if( $role !~ /active/)
  {
    logUMDMsg("UMD: Current redundancy role is not active! UMD can only be started from active server.");
    earlyExit("preChecks", "UM can only be performed from active node. Current server role=$role");
  }
  $role=sbxCommonFunctions::getCurrentRole("peer");
  if( $role !~ /standby/)
  {
    logUMDMsg("UMD: Peer redundancy role is not standby! Check if peer server is down. Exiting.");
    earlyExit("preChecks", "Peer redundancy role is not standby. Peer server=$peerCeName, role=$role");
  }

  my( $ret1,$ret2)=checkCliUpgradeStatus();
  if ( $ret1 =~ /upgraded/ && $ret2 =~ /pendingUpgrade/ || $ret1 =~ /pendingUpgrade/ && $ret2 =~ /upgraded/ || $ret1 =~ /upgrading/ && $ret2 =~ /pendingUpgrade/ || $ret1 =~ /pendingUpgrade/ && $ret2 =~  /upgrading/ || $ret1 =~ /upgrading/ && $ret2 =~ /upgraded/ ||  $ret1 =~ /upgraded/ && $ret2 =~ /upgrading/ )
  { 
     logUMDMsg("UMD: CLI upgrade states are not good! UM based upgrade cannot be performed. primaryUpgradeStatus=$ret1,secondaryUpgradeStatus=$ret2");
     earlyExit("preChecks", "CLI upgrade states are not good! UM based upgrade cannot be performed. primaryUpgradeStatus=$ret1,secondaryUpgradeStatus=$ret2");  
  }
 
  `/opt/sonus/sbx/scripts/sonusPeerCnx.expect $TOUCH $upgradeBaseDir/preUpgradeCheckActive.key`;
 
  my ($syncStatus,$syncEntity)=checkSyncStatus();
  if ( $syncStatus =~ /bad/ )
  {
     logUMDMsg("UMD: Sync is in progress!!. Please run the preChecks after sync completion.");
     earlyExit("preChecks", "Redundancy Sync is in progress. Entities not in Sync are: $syncEntity . Please run the preChecks after sync completion.");
  }

  # Extract and run preUpgradeChecks.
  # Error handling 
  `[[ ! -z $stagingDir ]] && $TAR -xzf $package -C $stagingDir`;
  `[[ ! -z $stagingDir ]] && $CHGRP -R upgrade $stagingDir/ 2>/dev/null`;
  `[[ ! -z $stagingDir ]] && $CHMOD 0770 $stagingDir/*sh $stagingDir/*pl $stagingDir/*pm 2>/dev/null`;

  # Peer package extraction is done in preInstallCheck script. 
  my $result1=`$LS $stagingDir/preInstallCheck.sh`;
  chomp($result1);  
  if ( $result1 =~ /No such file or directory/ )
  {
    logUMDMsg("UMD: Could not extract preInstallCheck script! Unable to run preUpgradeChecks.");
    earlyExit("preChecks" ,"Could not extract preInstallCheck script! Unable to run preUpgradeChecks.");
  }
  else
  { 
    `$SUDO $stagingDir/preInstallCheck.sh -c`; # sudoers-www-data
  }
  return 0;

}

sub upgradeCheckStatus
{
  if( ! -e "$preUpgradeCheckMarker" )
  {
    if( -e "$upgradeBaseDir/preUpgradeCheckActive.key" )
    {
      logUMDMsg("UMD: preUpgradeChecks are in progress, Status file is not created yet.");
      print("checkInProgress\n"); 
      return 0;
    }
    else
    {
      logUMDMsg("UMD: preUpgradeChecks are not in progress");
      print("checkNotRun\n");
      return 1;
    }
  } 
  else 
  {
    # Get the content in a hash and check status

    if ("$hash{'CheckStatus'}" =~ /checkComplete/ ) 
    { 
      logUMDMsg("UMD: preUpgradeChecks are complete");
      print("checkComplete\n");
      print Dumper(\%hash);
      my $steps=`$CAT $preUpgradeCheckMarker |$GREP StepNo`;
      print("$steps");
      return 0;
    }
    elsif ("$hash{'CheckStatus'}" =~ /checkInProgress/ ) 
    {
      logUMDMsg("UMD: preUpgradeChecks are in progress");
      print("checkInProgress\n");
      return 1; 
    }
    elsif ("$hash{'CheckStatus'}" =~ /checkFailed/ )
    {
      logUMDMsg("UMD: preUpgradeChecks have failed!");
      logUMDMsg("UMD: Reason=$hash{'Reason'}");
      print("checkFailed\n");
      print("Reason=$hash{'Reason'}\n");
      return 1;
    }
    else
    {
      logUMDMsg("UMD: preUpgradeCheck state=$hash{'CheckStatus'}");
      print("$hash{'CheckStatus'}\n");
    }
  }
  return 1;
}

sub startUpgrade
{
  my $package = shift;
  my $miscArg = shift;
  my $baseName = basename($package);
  my $upgradeInProgress = "false";
  my $integrityCheck = "skip"; 
  my $startUpgradeFailed="false";
  my $failReason=""; 

  # Check if server is active. If not, exit from here.
 
  logUMDMsg("UMD: ==========================================================================");
  logUMDMsg("UMD: Starting UMD startUpgrade()");
  logUMDMsg("UMD: Calling runChild()"); 
  runChild();

  # Remove logrotation for platformManager to avoid apache getting restarted in the middle of upgrade
  # apache restart kills this script as it is invoked through PM. A change is made to use "NOHUP" in PM code
  # but until we upgrade to that version of PM, need this change
  # Not adding this in BEGIN because of dependencies
  # This is here because we need to run as part of child, parent exits and calls END block
  unlink $ETC_LOGROTATE_PM;

  ##
  ## Perform checks to know if upgrade is already in progress
  ## Calling startUpgrade() when upgrade is already in progress would mean daemon got killed abruptly.
  ## In this case UMD will go directly to monitor mode.
  ##

  if ( $miscArg eq "rc_local" )
  {
    logUMDMsg("UMD: Daemon got recovered after a reboot.. going to monitor mode..");
    $upgradeInProgress="true";
  }
  elsif ( $miscArg eq "activeMonitor")
  {
    logUMDMsg("UMD: Daemon started from standby to monitor orginal active upgrade.. going to monitor mode..");
    $upgradeInProgress="true";
    `$updateRcLocal $0 $package`;
  }
  else
  {
    logUMDMsg("UMD: Setting up daemon in rc.local to handle reboots..");
    # print("Command is='$0 startUpgrade $package rc_local'\n");
    `$updateRcLocal $0 $package`;
  }

  if( -e "$upgradeBaseDir/liveUpgradeActive.key" && $upgradeInProgress ne "true" )
  {
    logUMDMsg("UMD: Upgrade already in progress.. New LSWU request rejected!");
    exit(1);
  } 
  `$ECHO "Upgrade started.." >> $upgradeBaseDir/liveUpgradeActive.key`;
  if ( -e "$upgradeBaseDir/preUpgradeCheckActive.key" )
  {
    logUMDMsg("UMD: preUpgradeChecks are in progress.. LSWU request rejected!"); 
    earlyExit("upgrade", "LSWU request rejected since preUpgradeChecks are in progress. Possible_Recovery_Action_is:Wait for preUpgradeCheck completion.");
  }  

  if ( $upgradeInProgress eq "false" )
  {
    sbxCommonFunctions::archiveUMLogFiles;
    setupUpgradeArea("upgrade");
    # Check if running on standalone
    if ( $peerCeName =~ /none/ )
    {
      logUMDMsg("UMD: LSWU cannot be run on a standalone server!");
      earlyExit("upgrade" ,"LSWU cannot be run on a standalone server. Possible_Recovery_Action_is:Nothing to Recover.");
    }
    # check syncStatus and run other pre-upgrade checklist
    my $role=sbxCommonFunctions::getCurrentRole("self");
    if( $role !~ /active/) 
    {
       logUMDMsg("UMD: Current redundancy role is not active! UMD can only be started from active server."); 
       earlyExit("upgrade", "UM can only be performed from active node. Current server role=$role . Possible_Recovery_Action_is:Start upgrade from active node.");
    }
    $role=sbxCommonFunctions::getCurrentRole("peer");
    if( $role !~ /standby/)
    {
       logUMDMsg("UMD: Peer redundancy role is not standby! Check if peer server is down. Exiting.");
       earlyExit("upgrade", "Peer redundancy role is not standby. Peer server=$peerCeName, role=$role. Possible_Recovery_Action_is:Make sure both servers of HA are up and running.");
    } 
    my( $ret1,$ret2)=checkCliUpgradeStatus();
    if ( $ret1 =~ /upgraded/ && $ret2 =~ /pendingUpgrade/ || $ret1 =~ /pendingUpgrade/ && $ret2 =~ /upgraded/ || $ret1 =~ /upgrading/ && $ret2 =~ /pendingUpgrade/ || $ret1 =~ /pendingUpgrade/ && $ret2 =~  /upgrading/ || $ret1 =~ /upgrading/ && $ret2 =~ /upgraded/ ||  $ret1 =~ /upgraded/ && $ret2 =~ /upgrading/ )
    { 
       logUMDMsg("UMD: CLI upgrade states are not good! UM based upgrade cannot be performed. primaryUpgradeStatus=$ret1,secondaryUpgradeStatus=$ret2");
       earlyExit("upgrade", "CLI upgrade states are not good! UM based upgrade cannot be performed. primaryUpgradeStatus=$ret1,secondaryUpgradeStatus=$ret2. Possible_Recovery_Action_is: Try CLI based LSWU (or) Contact Sonus support.");  
    }


    my ($syncStatus,$syncEntity)=checkSyncStatus();
    if ( $syncStatus =~ /good/ )
    {
      # Run disk usage check before unpacking full package
      `$TAR -C $stagingDir -xzf $package sonusUtils.sh checkDiskUsage.sh sonusCommonUtils.sh`; 
      my $ret=`$SUDO $stagingDir/checkDiskUsage.sh`; # sudoers-www-data
      if ( $ret != 0 )
      {
          logUMDMsg("UMD: Disk Usage checks on active server failed! Exiting.");
          earlyExit("upgrade", "disk usage checks failed on active server. Possible_Recovery_Action_is:To free up space on active.");
      }
      # Need to untar package on current active server.
      logUMDMsg("UMD: untaring package on active server");
      `[[ ! -z $stagingDir ]] && $TAR -xzf $package -C $stagingDir`;
      `[[ ! -z $stagingDir ]] && $CHGRP -R sonus $stagingDir/ 2>/dev/null`;
      `[[ ! -z $stagingDir ]] && $CHGRP -R upgrade $stagingDir/ 2>/dev/null`;
      `[[ ! -z $stagingDir ]] && $CHMOD 0770 $stagingDir/*sh $stagingDir/*pl $stagingDir/*pm 2>/dev/null`;

      my $cmd = "request system serverAdmin $peerCeName startSoftwareUpgrade package $baseName integrityCheck $integrityCheck";
      logUMDMsg("UMD: Calling CLI upgrade command: $cmd");
      my $result = runCliCommandLocal("$cmd");
      $result =~ s/`//g;
      $result =~ s/'//g;
      if ( $result eq "" )
      {
          logUMDMsg("UMD: runCliCommandLocal returned NULL!");
          earlyExit("upgrade", "CLI command to start LSWU returned NULL. Unable to continue. Possible_Recovery_Action_is: Check and make sure both the servers of HA are up.");
      }

      my @lines = split /\n/, $result;
      foreach $line (@lines)
      {
        if ( $line =~ /result failure/ )
        {
          logUMDMsg("UMD: CLI command to start LSWU has failed!");
          $startUpgradeFailed="true"; 
          foreach my $userLine (@lines)
          {
            if ($userLine !~ /result failure/ && $userLine !~ /^\s*$/)
            {
              $failReason .= $userLine;
            }
          }
          last;
        }
        elsif ( $line =~ /reason/)
        {
          logUMDMsg("UMD: $line");
          $failReason=$line;
        }
        elsif( $line =~ /syntax error/ )
        {
          logUMDMsg("UMD: CLI command execution failed!");
          logUMDMsg("UMD: $line");
          $startUpgradeFailed="true";
          foreach my $userLine (@lines)
          {
            if ($userLine !~ /result failure/ && $userLine !~ /^\s*$/)
            {
              $failReason .= $userLine;
            }
          }
          last;
        }
	    elsif( $line =~ /Action Command Timeout/ )
        {
	      logUMDMsg("UMD: CLI action command timed out!");
	      logUMDMsg("UMD: $line");
	      $failReason=$line;
          $startUpgradeFailed="true";
	    }	      
      }
    }
    else
    {
      logUMDMsg("UMD: ERROR! Cannot start upgrade while sync is in progress!!");
      earlyExit("upgrade", "LSWU cannot be started while sync is in progress. Entities not in Sync are: $syncEntity . Possible_Recovery_Action_is: Wait for Sync to complete and Retry after sync completion.");
    }
  }
  if ( $startUpgradeFailed eq "false" )
  {
    # Go to monitor mode
    logUMDMsg("UMD: Starting startMonitoring()");
    startMonitoring("$baseName", "$integrityCheck","$miscArg" );
  }
  else
  {
    # Exit
    logUMDMsg("UMD: Failure detected.. Stopping UMD..");
    earlyExit("upgrade", "CLI command to start LSWU has failed. Detailed CLI output:$failReason. Possible_Recovery_Action_is: Check logs and Contact Sonus Support.");
  }

}
    
my $fun = "";
if( $ARGV[0] ne "" )
{
  $fun = $ARGV[0];
}
else
{
  printUsage();
}
 
if ( $fun eq "startUpgradeCheck" )
{
  $packageName = $ARGV[1];
  print("fun is=$fun\npackage is=$packageName\n");
  if ( !defined $packageName )
  {
    printUsage();
  }
  elsif ( ! -e "$packageName" )
  {
    logUMDMsg("UMD: Supplied package doesn't exist!");
    exit(1);
  }
  startUpgradeCheck("$packageName");
}
elsif ( $fun eq "startUpgrade" )
{
  $packageName = $ARGV[1];
  if ( !defined $packageName )
  {
    printUsage();
  }
  if( defined($ARGV[2]) && $ARGV[2] ne "" )
  {
    $miscArg = $ARGV[2];
  }
  if ( ! -e "$packageName" && ($miscArg ne "rc_local" && $miscArg ne "activeMonitor"))
  {
    logUMDMsg("UMD: Supplied package doesn't exist!");
    exit(1);
  }
  startUpgrade("$packageName", $miscArg);
}
elsif ( $fun eq "upgradeCheckStatus" )
{
  upgradeCheckStatus();
}
elsif ( $fun eq "upgradeStatus" )
{
  upgradeStatus();
}
elsif ( $fun eq "Cancel" )
{
  $packageName = $ARGV[1];
  if ( !defined $packageName )
  {
    printUsage();
  }
  elsif ( -e "$packageName" )
  {
    logUMDMsg("UMD: Supplied package doesn't exist!");
    exit(1);
  }
  CancelUpgrade("$packageName");
}
else
{
  logUMDMsg("UMD: Unsupported argument! Argument supplied=$fun\n");
  printUsage();
}
exit(0);  

