#!/bin/bash

# exit on errors
set -e

trap 'res=$? && test $res -ne 0 && echo "command ($BASH_COMMAND) return $res"' EXIT

tarFile=$1
isoFile=$2
targetDir=$3
systemIP=$4

CHECKSUM="md5sum"
isoFileChecksum="$isoFile.md5"

bootDir='/boot'
grubDir="$bootDir/grub"
grubCfgDir="/etc/grub.d"
grubMenuFile="$grubDir/menu.lst"
grubISOCfg="$grubCfgDir/51_ribbon_iso"
grubDefaultFile="/etc/default/grub"

# functions for generating the netmask from the prefix that 'ip' returns
function getMask()
{
  prefix=$1
  mask="0"
  while [ $prefix -gt 0 ]
  do
    shiftValue=$((8-prefix))
    prefix=$[$prefix-1]
    tmpMask=$((1<<shiftValue))
    mask=$((mask |=tmpMask))
  done
  eval ${2}=$mask
}

function convNetprefixToNetmask()
{
  prefix=$1
  oct1="0"
  oct2="0"
  oct3="0"
  oct4="0"
  if [ $prefix -ge 8 ];then
    oct1="255"
    if [ $prefix -ge 16 ];then
      oct2="255"
      if [ $prefix -ge 24 ];then
        oct3="255"
        if [ $prefix -ge 32 ];then
          oct4="255"
        else
          tmpPrefix=$[prefix-24]
          getMask $tmpPrefix oct4
        fi
      else
        tmpPrefix=$[prefix-16]
        getMask $tmpPrefix oct3
      fi
    else
      tmpPrefix=$[prefix-8]
      getMask $tmpPrefix oct2
    fi
  else
    tmpPrefix=$[prefix]
    getMask $tmpPrefix oct1
  fi
  netMask="$oct1.$oct2.$oct3.$oct4"

  eval ${2}=$netMask
}

#
# Determine the linux flavor and if the system is running grub-legacy or grub2
#
# NOTE: cannot use grep to check the currently installed grub since we are
# using 'set -e' and grep may  properly return non-0, causing the script to exit
#
flavor=""
grubLegacy=0
if [[ -f "/etc/redhat-release" ]]; then
    flavor="redhat"
    if [[ -f "/sbin/grub" ]]; then
        grubLegacy=1
    fi
elif [[ -f "/etc/debian_version" ]]; then
    flavor="debian"
    if [[ -n "$(dpkg -l | sed -n '/grub-legacy/p')" ]]; then
        grubLegacy=1
    fi
else
    echo "Unsupported Linux distribution (currently support: redhat, debian)"
    exit 1
fi


#
# Extract images and grub support files
#
cd $targetDir
tar -zxf $tarFile 
[[ -e menu.lst ]] && chown root:root menu.lst
[[ -e 51_ribbon_iso ]] && chown root:root 51_ribbon_iso

#
# move images to the boot directory
#
mv -f iso_vmlinuz   $bootDir/.
mv -f iso_initrd.gz $bootDir/.

#
# gather information needed to maintain connectivity after the OS is loaded
# note: the grep should never fail here...
#
prefix=$(ip addr show|grep inet|grep $systemIP|awk -F"$systemIP/" '{print $2}' | awk '{print $1}')
gw=$(ip route show|grep 'default via'|awk '{print $3}')
convNetprefixToNetmask $prefix netmask

#
# Setup the menu items in accordance with the grub version
#
if [ $grubLegacy -eq 1 ]; then
    # back up the appropriate menu file. if the iso fails before partitioning, we
    # can more easily boot into the old state by reverting to the saved menu.lst.
    cp -f $grubMenuFile "${grubMenuFile}.pre-iso" || true
    mv -f menu.lst $grubMenuFile

    # update the kernel command line so we can maintain connectivity
    sed -i -e "s/REPAVE_IP=/REPAVE_IP=$systemIP/" \
           -e "s/REPAVE_NM=/REPAVE_NM=$netmask/" \
           -e "s/REPAVE_GW=/REPAVE_GW=$gw/" $grubMenuFile

else
    # remove execute permission from default grub menu handlers so we only get
    # the custom ISO setup. also remove permission from ribbon files in case
    # we are reloading a box.
    chmod a-x $grubCfgDir/[123]0_*

    ribbonConfig=$(ls $grubCfgDir/50_ribbon_* 2> /dev/null | wc -l)
    if [[ $ribbonConfig -ne 0 ]]; then
        chmod a-x $grubCfgDir/50_ribbon_*
    fi

    # install new custom menu script and make sure it is executable
    # note: no need to back it up since it is a different file name than what
    # is used for the SBC
    mv -f 51_ribbon_iso $grubISOCfg
    chmod a+x $grubISOCfg
    grubDefaultEntry=1

    # we used to add vga=788 to the iso args. this has been replaced by gfxpayload,
    # which is best set via the default file
    sed -i -e 's/^#GRUB_GFXMODE/GRUB_GFXMODE/' \
           -e '/GRUB_GFXMODE/s/=.*/=800x600x16,800x600/' $grubDefaultFile

    # update the kernel command line so we can maintain connectivity
    sed -i -e "s/REPAVE_IP=/REPAVE_IP=$systemIP/" \
           -e "s/REPAVE_NM=/REPAVE_NM=$netmask/" \
           -e "s/REPAVE_GW=/REPAVE_GW=$gw/" $grubISOCfg

    # rebuild grub.cfg and set the default menu entry
    # note: there is output regarding leaked fds.  hide the output, but
    # save it for troubleshooting
    if [[ "$flavor" == "debian" ]]; then
        update-grub &> $targetDir/update_grub.log
        grub-set-default $grubDefaultEntry
    elif [[ "$flavor" == "redhat" ]]; then
        grub2-mkconfig -o /boot/grub2/grub.cfg &> $targetDir/update_grub.log
        grub2-set-default $grubDefaultEntry
    fi

    # disable the modified grub default setting in case the user chooses not to ISO
    # i.e. restore the file to its original contents
    sed -i -e 's/^GRUB_GFXMODE/#GRUB_GFXMODE/' \
           -e '/GRUB_GFXMODE/s/=.*/=640x480/' $grubDefaultFile
fi


if [ -e "$isoFile.sha256" ]
then
  CHECKSUM="sha256sum"
  isoFileChecksum="$isoFile.sha256"
fi

#
# Delete all .iso and iso (md5|sha256) files (except our iso/(md5|sha256))
# note: avoid /proc since timing can cause 'no such file or directory' errors
# note: use exec and not delete since prune cannot be used with delete
#
find $targetDir -path '/proc' -prune -o \( -name '*.iso' -a ! -name "$isoFile" \) -exec rm -f {} \;
find $targetDir -path '/proc' -prune -o \( -name '*.iso.md5' -a ! -name "$isoFileChecksum" \) -exec rm -f {} \;
find $targetDir -path '/proc' -prune -o \( -name '*.iso.sha256' -a ! -name "$isoFileChecksum" \) -exec rm -f {} \;

#
# Validate that the ISO copied properly
#
if [ -e $isoFileChecksum ]; then
  # when doing a wget from artifactory, the (md5|sha256) file only has the sum and not
  # the name.  add the name for proper comparison.
  numCols=$(awk '{print NF}' $isoFileChecksum)
  if [ $numCols -eq 1 ]; then
    awk -v name=$isoFile '{print $1 "  " name}' $isoFileChecksum > /tmp/$isoFileChecksum
    mv -f /tmp/$isoFileChecksum $isoFileChecksum
  fi

  # NOTE: the expect script traps on the error, don't change it!
  # NOTE: due to usage of 'set -e', if the comparison failed we exit immediately
  # and the subsequent error code is NOT executed.  Leave it in place in case we
  # ever remove the 'set -e'.
  $CHECKSUM --status -c $isoFileChecksum
  if [ $? -ne 0 ]; then
    echo "ISO file not copied properly ($CHECKSUM check failed)!"
    exit 1
  fi
else
  echo "ISO checksum file does not exist ($CHECKSUM check failed)!"
  exit 1
fi

# success: need to return 0 in order to avoid falsely trapping on success.
exit 0
