#!/bin/bash

set -e

trap 'res=$? && test $res -ne 0 && echo "command ($BASH_COMMAND) return $res"' EXIT

tarFile=$1
isoFile=$2
targetDir=$3
bmcLogging=$4
hw5xSupported=$5

CHECKSUM="md5sum"
isoFileChecksum="$isoFile.md5"

bootDir='/boot'
grubDir="$bootDir/grub"
grubCfgDir="/etc/grub.d"
grubMenuFile="$grubDir/menu.lst"
grubISOCfg="$grubCfgDir/51_ribbon_iso"
grubRibbonCfg="$grubCfgDir/50_ribbon_sda1"
grubDefaultFile="/etc/default/grub"

#
# Validate installing on this system is allowed
#
if [[ $hw5xSupported -eq 0 ]]; then
    hwType=`dmidecode -t 1 | grep -m 1 "Product Name" | awk '{print $3}'`
    if [[ $hwType != "SBC7000" && $hwType != "SBC5400" ]]; then
        # NOTE: the expect script traps on the error, don't change it!
        echo "52xx/51xx hardware NOT supported for version 11.0 and higher"
        # don't trigger an error since we want to catch the returned msg
        exit 0
    fi
fi

#
# Extract images and grub support files
#
cd $targetDir
tar -zxf $tarFile 
[[ -e menu.lst ]] && chown root:root menu.lst
[[ -e 51_ribbon_iso ]] && chown root:root 51_ribbon_iso

#
# move images to the boot directory
#
mv -f iso_vmlinuz   $bootDir/.
mv -f iso_initrd.gz $bootDir/.

#
# Check to see whether the system is running grub-legacy or grub2 and setup
# the menu items accordingly.
#
# If the BMC supports logging, add a flag to the grub menu in order
# to alert postInstall.sh that logging-to-console should be enabled
#
# NOTE: cannot use grep to check the currently installed grub since we are
# using 'set -e' and grep may  properly return non-0, causing the script to exit
#
legacy=$(dpkg -l | sed -n '/grub-legacy/p')
if [ -n "$legacy" ]; then
    # back up the appropriate menu file. if the iso fails before partitioning, we
    # can more easily boot into the old state by reverting to the saved menu.lst.
    cp -f $grubMenuFile "${grubMenuFile}.pre-iso" || true
    mv -f menu.lst $grubMenuFile
    if [ $bmcLogging -eq 1 ]; then
        sed -i -e '/iso_vmlinuz/s/connexip/connexip RIBBON_ISOSBX/' $grubMenuFile
    fi
else
    # remove execute permission from default grub menu handlers so we only get
    # the custom ISO setup.  also remove permission from our SBC menu handlers.
    chmod a-x $grubCfgDir/[123]0_* $grubCfgDir/50_ribbon_*

    # install new custom menu script and make sure it is executable
    # note: no need to back it up since it is a different file name than what
    # is used for the SBC
    mv -f 51_ribbon_iso $grubISOCfg
    chmod a+x $grubISOCfg

    # make sure we have an 'out' to choose the currently existing OS
    # rather than install
    # note: we convert existing entry into an array. in order to do so, change quoted
    # params so their spaces are @, and then change them back when we use them.
    # note: ok to use grep here even though we have set -e, it should always succeed
    if [ -e "$grubRibbonCfg" ]; then
       currEntry=$(grep 'sonus-curr-ver' $grubRibbonCfg)
       currEntryItems=($(echo $currEntry | perl -pe 's{("[^\"]*")}{($x=$1)=~s/ /@/g;$x}ge'))
       sed -i -e "/sonus-curr-ver/s/CURR_KERNEL/${currEntryItems[1]}/" \
              -e "/sonus-curr-ver/s|CURR_DEV|${currEntryItems[2]}|" \
              -e "/sonus-curr-ver/s|CURR_OPTS|${currEntryItems[3]//@/ }|" \
              -e "/sonus-curr-ver/s|CURR_TITLE|${currEntryItems[4]//@/ }|" $grubISOCfg

       singleUserEntry=$(grep 'single-user' $grubRibbonCfg)
       singleUserEntryItems=($(echo $singleUserEntry | perl -pe 's{("[^\"]*")}{($x=$1)=~s/ /@/g;$x}ge'))
       sed -i -e "/single-user/s/CURR_KERNEL/${singleUserEntryItems[1]}/" \
              -e "/single-user/s|CURR_DEV|${singleUserEntryItems[2]}|" \
              -e "/single-user/s|CURR_OPTS|${singleUserEntryItems[3]//@/ }|" \
              -e "/single-user/s|CURR_TITLE|${singleUserEntryItems[4]//@/ }|" $grubISOCfg     

       grubDefaultEntry=2
    else
        # leave only the iso entry since we don't know what the base install is
        sed -i -e '/sonus-curr-ver/d' -e '/single-user/d' $grubISOCfg

       grubDefaultEntry=1
    fi

    # we used to add vga=788 to the iso args. this has been replaced by gfxpayload,
    # which is best set via the default file
    sed -i -e 's/^#GRUB_GFXMODE/GRUB_GFXMODE/' \
           -e '/GRUB_GFXMODE/s/=.*/=800x600x16,800x600/' $grubDefaultFile

    # add extra logging support
    if [ $bmcLogging -eq 1 ]; then
        sed -i -e '/isoArgs=/s/connexip/connexip RIBBON_ISOSBX/' $grubISOCfg
    fi

    # rebuild grub.cfg
    # note: there is output regarding leaked fds.  hide the output, but
    # save it for troubleshooting
    update-grub &> $targetDir/update_grub.log

    # set the default menu entry
    grub-set-default $grubDefaultEntry

    # disbale the modified grub default setting in case the user chooses not to ISO
    sed -i -e 's/^GRUB_GFXMODE/#GRUB_GFXMODE/' \
           -e '/GRUB_GFXMODE/s/=.*/=640x480/' $grubDefaultFile
fi

if [ -e "$isoFile.sha256" ]
then
  CHECKSUM="sha256sum"
  isoFileChecksum="$isoFile.sha256"
fi

#
# Delete all .iso and iso (md5|sha256) files (except our iso/(md5|sha256))
# note: avoid /proc since timing can cause 'no such file or directory' errors
# note: use exec and not delete since prune cannot be used with delete
#
find $targetDir -path '/proc' -prune -o \( -name '*.iso' -a ! -name "$isoFile" \) -exec rm -f {} \;
find $targetDir -path '/proc' -prune -o \( -name '*.iso.md5' -a ! -name "$isoFileChecksum" \) -exec rm -f {} \;
find $targetDir -path '/proc' -prune -o \( -name '*.iso.sha256' -a ! -name "$isoFileChecksum" \) -exec rm -f {} \;

#
# Validate that the ISO copied properly
#
if [ -e $isoFileChecksum ]; then
  # when doing a wget from artifactory, the (md5|sha256) file only has the sum and not
  # the name.  add the name for proper comparison.
  numCols=$(awk '{print NF}' $isoFileChecksum)
  if [ $numCols -eq 1 ]; then
    awk -v name=$isoFile '{print $1 "  " name}' $isoFileChecksum > /tmp/$isoFileChecksum
    mv -f /tmp/$isoFileChecksum $isoFileChecksum
  fi

  # NOTE: the expect script traps on the error, don't change it!
  # NOTE: due to usage of 'set -e', if the comparison failed we exit immediately
  # and the subsequent error code is NOT executed.  Leave it in place in case we
  # ever remove the 'set -e'.
  $CHECKSUM --status -c $isoFileChecksum
  if [ $? -ne 0 ]; then
    echo "ISO file not copied properly ($CHECKSUM check failed)!"
    exit 1
  fi
else
  echo "ISO checksum file does not exist ($CHECKSUM check failed)!"
  exit 1
fi

# success: need to return 0 in order to avoid falsely trapping on success.
exit 0
