#!/bin/bash
#############################################################
#
# Copyright (c) 2010 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# Module Description:
#    Script to check available disk space.
#
# NOTE: the specific error message is looked for by other
# scripts, so do not change it without also updating them!
#############################################################

# Sourcing Command Variables
if [ -e /opt/sonus/staging/sonusCommands.sh ]
then
    source /opt/sonus/staging/sonusCommands.sh
elif [ -e /opt/sonus/bin/sonusCommands.sh ]
then
    source /opt/sonus/bin/sonusCommands.sh
else
    /bin/echo "Could not locate sonusCommands.sh Exiting..."
    exit 1
fi

if [ -e /opt/sonus/staging/sonusCommonFiles.sh ]
then
    source /opt/sonus/staging/sonusCommonFiles.sh
elif [ -e /opt/sonus/bin/sonusCommonFiles.sh ]
then
    source /opt/sonus/bin/sonusCommonFiles.sh
else
    $ECHO "Could not locate sonusCommonFiles.sh Exiting..."
    exit 1
fi

if [ -x $STAGING_SONUS_UTILS_SH ]
then
    source $STAGING_SONUS_UTILS_SH
elif [ -x $SONUS_UTILS_SH ]
then
    source $SONUS_UTILS_SH
else
    echo "sonusUtils.sh is missing or has no execution permission!"
    exit 1;
fi

hostType=$(getHwType)
haMode=$(getHaMode)

# need to act differently depending on our caller
parentPid=$PPID
parentName=`$PS -p $parentPid -ocommand=`;

atStartup="false"
# require 20% disk space, with a finite floor of 7G
lvcount=$($LVSCAN 2>/dev/null | $WC -l)
if [ $lvcount == 3 ]; then
   requiredDiskAvail=3
   requiredDiskPercent=30
else
   if [[ "$hostType" != "ConnexIP5000" && "$prodString" != "5100" ]]; then
      requiredDiskAvail=20
   else
      requiredDiskAvail=7 
   fi

   requiredDiskPercent=20
fi

mountedRoot=$($DF -kh |$GREP debian-root |$HEAD -n1 |$AWK '{print $1}')
if [[ "$hostSubType" == "virtualCloud" && -e $SBX_START_SH ]] ; then
   # On cloud, during upgrade, content of qcow2 image is copied over to disk. This content will be around ~8GB.
   # Here as a conservative measure, we expect at least 9GB of free space.
   requiredDiskAvail=9
fi

verboseMode="n"

function provide_feedback()
{
  local logMsg=$1
  local userMsg=$2

  # when called by sysdump we provide direct feedback, otherwise
  # we are called as part of install/update and use the special
  # pre-install logging.
  if [[ $parentName =~ sysDump ]]; then
      $ECHO -e "$logMsg"
  else
      preInstallLogMsg "$logMsg" "$userMsg"
  fi
}

function check_disk_space()
{
  fileSystem=$1

  # note: for disk avail just use a whole number (i.e. strip of decimal spots)
  diskUsage=`$DF -k -h -P | $GREP $fileSystem`
  if [ "$diskUsage" != "" ]; then
    diskSize=`$ECHO $diskUsage | $AWK '{print $2}' | $SED -e 's/[MG]$//'`
    diskSizeLabel=`$ECHO $diskUsage | $AWK '{print $2}' | $SED -e 's/[0-9\.]*//'`
    diskUsed=`$ECHO $diskUsage | $AWK '{print $3}' | $SED -e 's/[MG]$//'`
    diskUsedLabel=`$ECHO $diskUsage | $AWK '{print $3}' | $SED -e 's/[0-9\.]*//'`
    diskAvailTotal=`$ECHO $diskUsage | $AWK '{print $4}' | $SED -e 's/[MG]$//'`
    diskAvail=`$ECHO $diskAvailTotal | $SED -e 's/\.[0-9]//'`
    diskAvailLabel=`$ECHO $diskUsage | $AWK '{print $4}' | $SED -e 's/[0-9\.]*//'`
    diskUsedPercent=`$ECHO $diskUsage | $AWK '{print $5}' | $SED -e 's/%//'`
    mountPoint=`$ECHO $diskUsage |$AWK '{print $6}'`
    diskAvailPercent=$((100-$diskUsedPercent))

    if [ $diskAvailPercent -lt $requiredDiskPercent -o $diskAvail -lt $requiredDiskAvail ]; then
      provide_feedback "ERROR: Not enough available disk space; Exiting..." "user"
      if [ $verboseMode = "y" ]; then
        provide_feedback "      Disk space required: minimum ${requiredDiskAvail}${diskAvailLabel} available, minimum ${requiredDiskPercent}% available" "user"
      fi
      provide_feedback "      Cleanup required for filesystem '$mountPoint': size=${diskSize}${diskSizeLabel}  used=${diskUsed}${diskUsedLabel}/${diskUsedPercent}%" "user"
      exit 1
    elif [ $verboseMode = "y" ]; then
      provide_feedback "Filesystem '$mountPoint': size=${diskSize}${diskSizeLabel}\tused=${diskUsed}${diskUsedLabel}/${diskUsedPercent}%\tavail=${diskAvailTotal}${diskAvailLabel}/${diskAvailPercent}%"
    fi
    if [[ ! $parentName =~ sysDump ]]; then
      provide_feedback "partition: ${fileSystem}, totalSpace: ${diskSize}${diskSizeLabel}, usage: ${diskUsed}${diskUsedLabel}" 
    fi
  elif [ $verboseMode = "y" ]; then
    provide_feedback "Filesystem $fileSystem not found"
  fi

}

# check all disks for enough space
function check_disks()
{
    if [ $verboseMode = "y" ]; then
      provide_feedback "Disk space required: minimum ${requiredDiskAvail}G, minimum ${requiredDiskPercent}% available"
    fi

    check_disk_space $mountedRoot

    requiredDiskAvail=7
    if [[ "$hostSubType" != "virtualCloud" && "$haMode" != "Nto1" ]] ; then 
       check_disk_space drbd0
    fi

    if [ $lvcount == 3 ]; then
       if [[ "$prodString" == "5110" && -e $VIRSH && $($VIRSH list --all | grep ePSX) ]]; then
          requiredDiskAvail=5
          requiredDiskPercent=5
       else
          # common partition should have atleast 7GB of free space
          requiredDiskAvail=7
          requiredDiskPercent=20
       fi
       check_disk_space debian-common
    fi

    # boot partition should have atleast 40MB of free space
    requiredDiskAvail=40
    requiredDiskPercent=20
    # for those SBCs that have been upgraded since /boot patition was 88MB 
    local bootDiskSize=`$DF -k -h -P | $GREP /boot | $AWK '{print $2}' | $SED -e 's/[MG]$//'`
    if [[ $bootDiskSize -lt 90 && $atStartup == "true" ]]; then
        requiredDiskAvail=2
        requiredDiskPercent=2
    fi
    check_disk_space /boot
}

usage()
{
$CAT << EOF
usage: `$BASENAME $0` [-h] [-m min_avail] [-p percent_avail] [-v]

OPTIONS:
   -h               Print this message.
   -m #             Absolute minimum GB that must be available (default 7GB).
   -p #             Minimum percentage of disk space that must be available (default 20%).
   -v               Verbose mode (always print stats).
EOF
}

while getopts "hm:p:v" OPTION
do
   case $OPTION in
   h)
      usage
      exit 1
      ;;
   m)
      requiredDiskAvail=$OPTARG
      ;;
   p)
      requiredDiskPercent=$OPTARG
      atStartup="true"
      ;;
   v)
      verboseMode="y"
      ;;
   ?)
      usage
      exit 1
      ;;
   esac
done

shift $(($OPTIND - 1))

check_disks

exit 0
