#!/bin/bash
#############################################################
#
# Copyright (c) 2016 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# sbxHwImageBasedUpgrade.sh
#
# 10/5/2016
#
# Module Description:
# Script to facilitate qcow2 image based in place upgrade
#
# Example:
#
#     sh sbxHwImageBasedUpgrade.sh -L /opt/sonus/staging/upgrade.out
#
#############################################################

# Source sonus commands
if [ -e /opt/sonus/staging/sonusCommands.sh ] && [ -e /opt/sonus/staging/sonusCommonFiles.sh ]
then
    source /opt/sonus/staging/sonusCommands.sh
    source /opt/sonus/staging/sonusCommonFiles.sh
else
    /bin/echo "sonusCommands.sh and/or sonusCommonFiles.sh not found. Exiting..."
    exit 1
fi

basePath=/mnt/target
logFile=
lswu="n"
sonusRoot=$SONUS_DIR
sonusStaging=$SONUS_STAGING_DIR
hostTypeFile=$HOST_TYPE
hostSubTypeFile=$HOST_SUB_TYPE
statusUpdater=$STATUS_UPDATER_PL
sftpChrootDir=$HOME_SFTPROOT
upgradeBaseDir=/home/log/sonus/upgrade
installBaseDir=/home/log/sonus/install
perfLogDir=/home/log/sonus/sbxPerf
emaLogDir=/home/log/sonus/ema
bfdLogDir=/home/log/sonus/bfd

if [ -f $STAGING_SONUS_UTILS_SH ]; then
   . $STAGING_SONUS_UTILS_SH
else
   . $SONUS_UTILS_SH
fi

if [ -f $SBXCONF_FILE ]; then
   sonusConfigFile=$SBXCONF_FILE
else
   sonusConfigFile=$SBX_CONF
fi

ceName=`$GREP ceName $sonusConfigFile | $AWK -F "="  '{print $2}'`

# Note: Kernel versions are pulled from the kernel environment during packaging
kernelVersion="5.10.234"
rescueVersion=""

usage()
{
  $CAT << EOF
  usage: $0 options

  OPTIONS:
     -h     Print this message.
     -L     logFile name
     -l     flag to indicate lswu.
EOF
}

while getopts "hL:l" OPTION
do
  case $OPTION in
  h)
     usage
     exit 1
     ;;
  L)
     logFile=$OPTARG
     ;;
  l)
     lswu="y"
     ;;

  ?)
     usage
     exit $E_BADARGS
     ;;
  esac
done

if [ -f $hostTypeFile ]; then
   hostType=`$CAT $hostTypeFile`
else
   # Use prodString/hwType from sonusUtils
   if [[ $prodString == "5110" ]]; then
      hostType="ConnexIP5100"
   else
      hostType=$hwType
   fi
   logMsg $logFile "File $hostTypeFile does not exist..hostType is set to $hostType"
fi

getReqOSVersion
targetVersion=$($GREP "SBC Version:" $STAGING_BUILDINFO | $AWK '{print $3}')
targetQcow=`$ECHO $targetVersion | $SED 's/-//'`
qcowImageName=$($BASENAME `$LS $sonusStaging/sbc-${targetQcow}-connexip-os_${reqOSVersion}_*amd64.qcow2.sha256 | $SED 's/.sha256//'`)
qcowSubDir=`$ECHO $qcowImageName | $SED 's/sbc-V/sbc_V/'`

if [ -f $SONUS_EXTERNAL_DIR/$qcowImageName ]; then
   qcowImagePath=$SONUS_EXTERNAL_DIR/$qcowImageName
elif [ -f $SONUS_EXTERNAL_DIR/$qcowSubDir/$qcowImageName ]; then
   # For upgrades from old versions like 4.0/4.1, qcow2 image can be present in sub-dir
   qcowImagePath=$SONUS_EXTERNAL_DIR/$qcowSubDir/$qcowImageName
else
   logMsg $logFile "ERROR: qcow2 image $qcowImageName not found in $SONUS_EXTERNAL_DIR and sub-dir $SONUS_EXTERNAL_DIR/$qcowSubDir. Exiting..."
   exit 1;
fi

if [ "$lswu" == "y" ]; then
   logMsg $logFile "--------Starting IMAGE UPGRADE--------"
   $PERL $statusUpdater -r ImageUpgrade -s Status -v "inProgress"
   $PERL $statusUpdater -r ImageUpgrade -s StartTime -v "`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`"
fi

currentLv=$($DF -kh |$GREP debian-root |$HEAD -n1 | $AWK '{print $1}')
if [ $currentLv == "/dev/mapper/debian-root1" ]; then
   newLv=/dev/mapper/debian-root2
else
   newLv=/dev/mapper/debian-root1
fi

$RM -fr $basePath/*
$MKDIR -p $basePath
$GREP -q $basePath /etc/mtab
if [ $? -ne 0 ]; then
   logMsg $logFile "Mounting $newLv on $basePath..."
   $MOUNT $newLv $basePath
fi

# Mount the qocw2 image on /tmp/root and /tmp/boot
performQcow2Mount $logFile $qcowImagePath

# Copy qcow2 contents (/tmp/root) to basePath
performQcow2Copy $logFile $basePath

# Copying staging passwdadmin homedir
if [ -d /opt/sonus/shared/passwdadmin ]; then
   $RM -rf $basePath/opt/sonus/shared/passwdadmin/
   basePathPasswdadminID=$($GREP 'passwdadmin' $basePath/etc/passwd | $CUT -f 3 -d ':')
   $CP -af /opt/sonus/shared/passwdadmin $basePath/opt/sonus/shared/passwdadmin/
   $CHOWN -R $basePathPasswdadminID:$basePathPasswdadminID $basePath/opt/sonus/shared/passwdadmin/
   $CHMOD --reference=/tmp/root/opt/sonus/shared/passwdadmin $basePath/opt/sonus/shared/passwdadmin

else
   $RM -rf $basePath/opt/sonus/shared/passwdadmin/
   basePathPasswdadminID=$($GREP 'passwdadmin' $basePath/etc/passwd | $CUT -f 3 -d ':')
   $CP -af /home/passwdadmin $basePath/opt/sonus/shared/passwdadmin/
   $CHOWN -R $basePathPasswdadminID:$basePathPasswdadminID $basePath/opt/sonus/shared/passwdadmin/
   $CHMOD --reference=/tmp/root/home/passwdadmin $basePath/opt/sonus/shared/passwdadmin/
fi

checkForNBDError $logFile

$MKDIR -p $basePath/home

# Update /home directory
$CP -pf /tmp/root/home/common/lvinfo.pl $LVINFO_PL
$CP -pf /tmp/root/home/sonusadmin/.bashrc /home/sonusadmin/.bashrc

$RM -fr /home/confd/

# note: grub configuration support is under /home/etc to be
# persistent across the root systems.  Don't handle it here though.
# It is handled separately later. 
if [ -d "/tmp/root/home" ]; then
   pushd /tmp/root/home &> /dev/null
   dirArray=( $(ls -d * | grep -v 'etc') )
   logMsg $logFile "Updating ownership and permissions in /home directory..."
   for dirElem in "${dirArray[@]}"
   do
    if [ -d "/home/$dirElem" ]; then
       $STAT -c "%n %u %g %a"  /home/$dirElem > /tmp/oldPerm
       $STAT -c "%n %u %g %a"  $dirElem > /tmp/dirPerm
       name=`$CAT /tmp/dirPerm | $AWK -F" " '{print $1}'`
       owner=`$CAT /tmp/dirPerm | $AWK -F" " '{print $2}'`
       group=`$CAT /tmp/dirPerm | $AWK -F" " '{print $3}'`
       perm=`$CAT /tmp/dirPerm | $AWK -F" " '{print $4}'`
       if [[ $dirElem == "sftproot"  ||  $dirElem == "log"  ||  $dirElem == "etc" ]];then
           # Dont do recursive change for sftproot, log and etc
           $CHOWN $owner:$group /home/$dirElem
           $CHMOD $perm /home/$dirElem
       else 
           logMsg $logFile "Updating /home/$dirElem to $owner:$group/$perm..."
           $CHOWN -R $owner:$group /home/$dirElem
           $CHMOD -R $perm /home/$dirElem
       fi
    else
       # postgresql is handled separately. ignore postgresql here. 
       if [[ "$dirElem" != "postgresql" ]] ; then
       logMsg $logFile "Copying $dirElem to /home..."
       $CP -af $dirElem /home/
    fi
    fi
   done
   popd &> /dev/null
fi

# Move dirs and Create links
if [ -d /home/external ]; then
   # this is needed for an upgrade from 51R0 to 62 as $sftpChrootDir/external exists already.
   # if not checked then external is moved inside $sftpChrootDir/external  
   if [ -d $sftpChrootDir/external ]; then
        $CP -prf /home/external/* $sftpChrootDir/external/
        $RM -rf /home/external
    else
        $MV /home/external $sftpChrootDir/external
    fi
    $RM -f $SONUS_EXTERNAL_DIR
    $LN -s $sftpChrootDir/external $SONUS_EXTERNAL_DIR
fi
if [ -d /home/staging ]; then
   $MV /home/staging $sftpChrootDir/staging
   $RM -f $sonusStaging
   $LN -s $sftpChrootDir/staging $sonusStaging
fi

# Move log back to home 
if [ -d $sftpChrootDir/log ]
then
   # covering upgrade path from 60 and 61 
   logMsg $logFile "Moving $sftpChrootDir/log back to /home/log"
   $MV $sftpChrootDir/log /home/log
   $RM -f /var/log
   $LN -s /home/log /var/log
else
   if [[ -d "/var/log" && ! -L "/var/log" ]]
   then
      # covering upgrade path from 51
      $RM -rf /home/log
      $MV /var/log /home/log   		
      $LN -s /home/log /var/log
      logMsg $logFile "Moving /var/log to /home/log"
   fi
fi

if [ -d /home/libvirt ]; then
   $RM -fr /home/libvirt
fi

# Update /home/log/sonus/upgrade
if [ ! -d $upgradeBaseDir ];then
   $MKDIR -p $upgradeBaseDir
fi
$CHOWN -R --reference=/tmp/varlog/sonus/upgrade $upgradeBaseDir
$CHMOD -R --reference=/tmp/varlog/sonus/upgrade $upgradeBaseDir

# Update /home/log/sonus/install
if [ ! -d $installBaseDir ];then
   $MKDIR -p $installBaseDir
fi
$CHOWN -R --reference=/tmp/varlog/sonus/install $installBaseDir
$CHMOD -R --reference=/tmp/varlog/sonus/install $installBaseDir
$CP -pf /tmp/varlog/sonus/install/install-log $installBaseDir/install-log.qcow2

if [ ! -f /home/log/cron.log ] ; then
   $TOUCH /home/log/cron.log
fi
$CHMOD --reference=/tmp/varlog/cron.log /home/log/cron.log

$RM -rf /home/log/audit
$CP -af /tmp/varlog/audit /home/log/

$CP -f /tmp/root/home/cnxipmadmin/.bashrc /home/cnxipmadmin/.bashrc
$CHOWN --reference=/tmp/root/home/cnxipmadmin/.bashrc /home/cnxipmadmin/.bashrc
$CHMOD --reference=/tmp/root/home/cnxipmadmin/.bashrc /home/cnxipmadmin/.bashrc

$MKDIR -m 755 -p $perfLogDir

$MKDIR -p /home/log/sonus/cnxipm
$CHOWN -R --reference=/tmp/varlog/sonus/cnxipm /home/log/sonus/cnxipm
$CHMOD -R --reference=/tmp/varlog/sonus/cnxipm /home/log/sonus/cnxipm
$MKDIR -p /home/log/sonus/cnxipm/admin/tmp
$CHOWN -R --reference=/tmp/varlog/sonus/cnxipm/admin /home/log/sonus/cnxipm/admin
$CHMOD -R 770 /home/log/sonus/cnxipm/admin

# make bfd log directory in upgrade case
if [ ! -d $bfdLogDir ];then
   $MKDIR -p $bfdLogDir
fi
if [ ! -f $bfdLogDir/bfdd.log ] ; then
   $TOUCH $bfdLogDir/bfdd.log
fi
$CHMOD -R 755 $bfdLogDir     
 

if [ ! -d $emaLogDir ];then
    $CP -af /tmp/varlog/sonus/ema /home/log/sonus/ 
else
    $CHOWN -R --reference=/tmp/varlog/sonus/ema $emaLogDir
    $CHMOD --reference=/tmp/varlog/sonus/ema $emaLogDir
fi

$RM -f /home/log/sonus/sbx/.rebootCount /home/log/sonus/sbx/.softRestartCount

$CHOWN -R --reference=/tmp/root/opt/sonus/external/ $sftpChrootDir/external
$CHMOD -R --reference=/tmp/root/opt/sonus/external/ $sftpChrootDir/external
$CHOWN -R --reference=/tmp/root/opt/sonus/staging/ $sftpChrootDir/staging
$CHMOD -R --reference=/tmp/root/opt/sonus/staging/ $sftpChrootDir/staging
$CHOWN --reference=/tmp/varlog/ /home/log
$CHMOD --reference=/tmp/varlog/ /home/log

$RM -rf $basePath/opt/sonus/external
$LN -fs $sftpChrootDir/external $basePath/opt/sonus/external
$RM -rf $basePath/opt/sonus/staging
$LN -fs $sftpChrootDir/staging $basePath/opt/sonus/staging
$RM -rf $basePath/var/log
$LN -fs /home/log $basePath/var/log

if [[ -d /home/sftpadmin ]]; then
   $RM -fr $sftpChrootDir/sftpadmin
   $MV /home/sftpadmin $sftpChrootDir/sftpadmin
   $USERMOD -s /usr/lib/sftp-server -d $sftpChrootDir/sftpadmin sftpadmin
   $LN -fs ../../external $sftpChrootDir/sftpadmin
fi

if [ ! -d /home/log/session ]; then
    $MKDIR -p /home/log/session
fi 
$CHOWN --reference=/tmp/varlog/session /home/log/session
$CHMOD --reference=/tmp/varlog/session /home/log/session

if [ ! -d /home/log/spool/rsyslog ]; then
    $MKDIR -p /home/log/spool/rsyslog
fi 
$CHOWN -R --reference=/tmp/varlog/spool/rsyslog /home/log/spool/rsyslog
$CHMOD -R --reference=/tmp/varlog/spool/rsyslog /home/log/spool/rsyslog

# Retain cnxipmadmin .ssh directory for inter CE communication using public key to work
if [ -e /home/cnxipmadmin/.ssh/authorized_keys ]; then
   logMsg $logFile "Found /home/cnxipmadmin/.ssh/authorized_keys"
   # Check if /$basePath/home/cnxipmadmin/ exists, if not create the parent directory
   if [ ! -d "$basePath/home/cnxipmadmin/" ]; then
       logMsg $logFile "$basePath/home/cnxipmadmin not found. Creating parent path"
       $MKDIR -p $basePath/home/cnxipmadmin/
   else
       logMsg $logFile "Deleting $basePath/home/cnxipmadmin/.ssh"
       $RM -rf $basePath/home/cnxipmadmin/.ssh/
   fi
   $CP -af /home/cnxipmadmin/.ssh $basePath/home/cnxipmadmin/.ssh/
   $CHOWN -R --reference=/tmp/root/home/cnxipmadmin/.ssh $basePath/home/cnxipmadmin/.ssh/
   $CHMOD --reference=/tmp/root/home/cnxipmadmin/.ssh $basePath/home/cnxipmadmin/.ssh/
   logMsg $logFile "Backed up cnxipmadmin authorized keys to $basePath..."
else
   logMsg $logFile "Not found: /home/cnxipmadmin/.ssh/authorized_keys (INFO)"
fi

# Also retain root .ssh directory for inter CE communication using public key to work
if [ -e /root/.ssh/id_rsa ]; then
   logMsg $logFile "Found /root/.ssh/id_rsa"
   $RM -rf $basePath/root/.ssh/
   $CP -af /root/.ssh $basePath/root/.ssh/
   $CHOWN -R --reference=/tmp/root/root/.ssh $basePath/root/.ssh/
   $CHMOD --reference=/tmp/root/root/.ssh $basePath/root/.ssh/
   logMsg $logFile "Backed up root ssh keys to $basePath..."
else
   logMsg $logFile "Not found: /root/.ssh/id_rsa (INFO)"
fi

# Database part
logMsg $logFile "Copying qcow DB data to /home/admin/db_data..."
if [ -d /home/admin/db_data ] ; then
    $RM -rf /home/admin/db_data/*
else
    $MKDIR -p /home/admin/db_data
fi
$CP -Rpf /tmp/root/home/postgresql /home/admin/db_data
# Copy sonusMetadata.xml
$CP -pf $basePath/opt/sonus/sbx/tailf/var/confd/cdb/sonusMetadata.xml.orig $basePath/opt/sonus/sbx/tailf/var/confd/cdb/sonusMetadata.xml

checkForNBDError $logFile

# Save off qcow2 boot dir contents for future updating of /boot
$RM -rf /tmp/qcow2BootDir
$MKDIR -p /tmp/qcow2BootDir
$CP -af /tmp/boot/ /tmp/qcow2BootDir

# grub2 configuration uses files from under /etc.  We move those
# files to /home so all root partitions use the same setup.  Copy
# the new setup from the qcow.  The config will be updated later
# after unmounting the qcow.
# note: virtual installs don't setup /home/etc, so we need to copy
# grub.d and default/grub from qcow's /etc dir.
# note: /home/etc will exist if and only if running grub2. we
# created it/started using it only to support grub2.
# note: make backups of the current setup for revert purposes
grub2Installed=0
if [[ -z "$($DPKG -l | $GREP grub-legacy)" ]]; then
   grub2Installed=1
   logMsg $logFile "Saving existing grub2 config for revert..."
   $RM -fr $HOME_ETC_GRUBD_PREV_DIR
   $MV $HOME_ETC_GRUBD_DIR $HOME_ETC_GRUBD_PREV_DIR
   $MKDIR -p $HOME_ETC_DEFAULT_PREV_DIR
   $RM -fr $HOME_ETC_DEFAULT_PREV_DIR/grub
   $MV $HOME_ETC_DEFAULT_DIR/grub $HOME_ETC_DEFAULT_PREV_DIR

   # can't cleanup grub-legacy till we know it is no longer going
   # to be used. and that is only after upgrading from a grub2
   # release (i.e. grub->grub2->grub2).  can't on first update
   # to grub2 since we may need to revert.
   [[ -d "/boot/grub.prev" ]] && $RM -fr /boot/grub.prev
else
   logMsg $logFile "Saving existing grub config for revert..."
   $CP -pr /boot/grub /boot/grub.prev

   # only need to create default dir since entire grub.d dir is copied
   logMsg $logFile "Creating /home/etc for grub2 support..."
   $MKDIR -p $HOME_ETC_DEFAULT_DIR
   $CHOWN --reference=/tmp/root/etc /home/etc
   $CHMOD --reference=/tmp/root/etc /home/etc
   $CHOWN --reference=/tmp/root/etc/default $HOME_ETC_DEFAULT_DIR
   $CHMOD --reference=/tmp/root/etc/default $HOME_ETC_DEFAULT_DIR
fi

# note: Make sure only the sda1 file is active so we get the appropriate
# menu items when update-grub is run
logMsg $logFile "Copying qcow grub2 configuration files to /home/etc..."
$CP -af /tmp/root${ETC_GRUBD_DIR} /home/etc
$CP -af /tmp/root${ETC_DEFAULT_GRUB} $HOME_ETC_DEFAULT_DIR
$CHMOD a-x /home${SONUS_GRUB_MENU_BASE}*
$CHMOD a+x /home${SONUS_GRUB_MENU_SDA1_FILE}

logMsg $logFile "Creating grub configuration symlinks in $basePath/etc..."
$RM -rf ${basePath}${ETC_GRUBD_DIR}
$LN -s $HOME_ETC_GRUBD_DIR ${basePath}${ETC_GRUBD_DIR}
$RM -f ${basePath}${ETC_DEFAULT_GRUB}
$LN -s $HOME_ETC_DEFAULT_DIR/grub ${basePath}${ETC_DEFAULT_GRUB}

performQcow2Unmount $logFile $qcowImagePath

# NOTE: reference the grub configuration file in basePath so that we don't need
# special processing for upgrades from a version without grub2 (they are a link
# to /home, same as if we already have grub2)
if [ -d /tmp/qcow2BootDir ]; then
   logMsg $logFile "Setting up grub to boot from new kernel/initrd..."

   # get current running SBC version
   currSbcVer=$($SWINFO_SH -l | $GREP "SBC:" | $AWK '{print $2}')

   # remove all but the current image from the boot directory
   cleanupBootFiles /tmp/qcow2BootDir/boot

   # Delete the base-os entry now that both partitions will have
   # SBC versions.  Update the single-user entry to the new partition.
   # Delete the previous and current entries as we will make the
   # previous release current entry the previous one, and will add a
   # new current one.
   # note: recall the file we are dealing with here is the qcow file
   # note: no reason to save a copy since if we are already grub2 the entire
   # grub.d directory was saved above
   $SED -i -e '/base-os/d' \
           -e "/single-user/s|/dev/mapper/debian-root[^ ]* |$newLv |" \
           -e '/sonus-prev-ver/d' \
           -e '/sonus-curr-ver/d' ${basePath}${SONUS_GRUB_MENU_SDA1_FILE}

   # get current kernel parameters from existing config: either grub.cfg
   # file or menu.lst, depending on whether already running grub2
   if [[ $grub2Installed -eq 1 ]]; then
       # can't get options from sd1 file since we will be using configureGrub of the
       # new version.  We know all the info we need for that call, except we have to
       # pull the options used from grub.cfg.
       prevOptions="$($SED -n '/sonus-curr-ver/,/}/p' $GRUB_CONFIG | $GREP vmlinuz | $SED 's|.*/dev/mapper/debian-root[^ ]* ||')"

       # maintain the existing grub2 password (if one existed)
	   # note: get the user/password from grub.cfg so keep all platforms the same
       # note: the password hash will generate differently each time the utility
       # grub-mkpasswd-pbkdf2 is run.  there is therefore no reason to check if
       # it is different.  just perform the update with the original value.
       cfgUserArg=""
       origGrubUser=$($GREP "set superusers" $GRUB_CONFIG | $AWK -F'"' '{print $2}')
       origGrubPwd=$($GREP "password_pbkdf2" $GRUB_CONFIG | $AWK '{print $3}')
       newGrubUser=$($GREP "^SUPERUSER" ${basePath}${SONUS_GRUB_MENU_PWD_FILE} | $AWK -F'"' '{print $2}')

       if [[ -n "$origGrubUser" && "$origGrubUser" != "$newGrubUser" ]]; then
           cfgUserArg="-u $origGrubUser"
   fi
       if [ -n "$origGrubPwd" ]; then
           # -s option species skip calling update-grub.  no need to call it when
           # configuring the password since that will happen below
           ${basePath}${CONFIGURE_GRUB_PWD_SH} -s $cfgUserArg -P $origGrubPwd -f ${basePath}${SONUS_GRUB_MENU_PWD_FILE} 2>&1 | $TEE -a $logFile
   fi
   else
       # menu.lst had the options starting right after the root device
       prevOptions="$($GREP BOOTIF /boot/grub/menu.lst | $TAIL -1 | $SED 's|.*/dev/mapper/debian-root[^ ]* ||')"
   fi

   # udpate the grub default file so kernel parameters are set for hardware
   ${basePath}${CONFIGURE_GRUB_SH} -g -f ${basePath}${ETC_DEFAULT_GRUB}

   # get currently running kernel version (still running orig software right now)
   currKernelVer=$($UNAME -a | $AWK '{print $3}')

   # get current time to be used as 'last used' timestamp 
   timeStamp=$($DATE "+%FT%T%z")

   # Add the previous version menu entry for the currently running version,
   # using the correct kernel arguments for that version
   ${basePath}${CONFIGURE_GRUB_SH} -k $currKernelVer -d $currentLv -v $currSbcVer -t "Sonus SBC $currSbcVer" -i "sonus-prev-ver" -T $timeStamp -P "$prevOptions" -f ${basePath}${SONUS_GRUB_MENU_SDA1_FILE}

   # Add the menu entry for the new version, making sure to use the new
   # version of the utility.
   ${basePath}${CONFIGURE_GRUB_SH} -k $kernelVersion -d $newLv -v $targetVersion -t "Sonus SBC $targetVersion" -i "sonus-curr-ver" -f ${basePath}${SONUS_GRUB_MENU_SDA1_FILE}

   # Need to mount a few items in the target path in order to generate the
   # new grub.cfg in a chroot session
   $MOUNT -t devtmpfs none $basePath/dev
   $MOUNT -t proc none $basePath/proc
   $MOUNT --bind /sys  $basePath/sys
   $MOUNT --bind /boot $basePath/boot
   $MOUNT --bind /home $basePath/home
   # To fix an lvm2 bug
   $MOUNT --bind /run $basePath/run

   # install grub2 to the MBR if not already installed
   if [[ $grub2Installed -eq 0 ]]; then
       logMsg $logFile "grub-legacy currently installed: Installing grub2 to the MBR on ${primary_device}..."
       $CHROOT $basePath $GRUB_INSTALL $primary_device 2>&1 | $TEE -a $logFile
       $RM -f ${GRUB_MENU_LST}* /boot/grub/{{xfs,reiserfs,e2fs,fat,jfs,minix}_stage1_5,stage{1,2}}
   fi

   # Work around an lvm2 bug causing error output during update-grub command
   # Refer to the function for more info
   updateLvmConf "hw"

   # copying apparmor.cfg
   if [ -f $APPARMOR_GRUB ] ; then
       logMsg $logFile "copying Apparmor.cfg..."
       $CP -pf $APPARMOR_GRUB $basePath/etc/default/grub.d
   fi

   # generate a new grub.cfg
   # note: update-grub spits a lot of junk to stderr; lets trash it...
   logMsg $logFile "Generating new grub config..."
   $CHROOT $basePath $UPDATE_GRUB_SH 2> /dev/null

   # update the default entry
   grubEntries=$($GREP -c "^sonus_linux_entry " ${basePath}${SONUS_GRUB_MENU_SDA1_FILE})
   newGrubDefault=$((grubEntries-1))
   logMsg $logFile "Updating grub default to menu entry $newGrubDefault..."
   $CHROOT $basePath $GRUB_SET_DEFAULT $newGrubDefault

   # unmount the temporary mounts we created
   $UMOUNT $basePath/dev $basePath/proc $basePath/sys $basePath/boot $basePath/home $basePath/run

   $RM -rf /tmp/qcow2BootDir
else
   logMsg $logFile "Failed to find qcow2 image boot directory. Exiting..."
   exit 1;
fi

$ECHO "$hostType" > $basePath/etc/hostType
if [ -e $hostSubTypeFile ]; then
   hostSubType=`$CAT $hostSubTypeFile`
   $CP -pf $hostSubTypeFile $basePath/etc/hostSubType
else
   hostSubType=$prodString
   logMsg $logFile "File $hostSubTypeFile does not exist..hostSubType is set to $hostSubType"
   $ECHO "$hostSubType" > $basePath/etc/hostSubType
fi

# Perform all hostType specific updates
performHostSpecificUpdatesForUpgrade $logFile $basePath $lswu $hostType $kernelVersion

if [ -d $SONUS_INSTALLUPGRADE_LOG_DIR ]; then
      $ECHO "Marker created by `basename $0` for Revert on $currentLv " > $LSWU_COMMIT_PENDING
else
      $ECHO "Marker created by `basename $0` for Revert on $currentLv " > $sonusRoot/lswuCommitPending
fi

# unmount the new partition
logMsg $logFile "Unmounting $basePath..."
$UMOUNT $basePath

if [ "$lswu" == "y" ]; then
   $PERL $statusUpdater -r ImageUpgrade -s Status -v "Complete"
   $PERL $statusUpdater -r ImageUpgrade -s EndTime -v "`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`"
   $PERL $statusUpdater -r ImageUpgrade -s Reason -v "Successful_Completion"
   logMsg $logFile "--------Completed IMAGE UPGRADE--------"
fi


logMsg $logFile "Rebooting to perform image based upgrade..."

if [ "$lswu" == "y" ]; then
   logMsg $logFile "--------Starting REBOOT--------"
   $PERL $statusUpdater -r Reboot -s Status -v "inProgress"
   $PERL $statusUpdater -r Reboot -s StartTime -v "`$DATE +'%a %b %e %H:%M:%S %Z %Y'|$AWK '{print $4}'`"
fi

# Enable fstrim in the backed up partition so that in case of rollback we have the fstrim enabled.
logMsg $logFile "Enabling fstrim in the backup partition..."
# Change 'Persistent' field value to 'false' so that timer does not get triggered on running 'systemctl daemon-reload'
$SED -i 's/Persistent=.*$/Persistent=false/' /lib/systemd/system/fstrim.timer
$SYSTEMCTL enable --now fstrim.timer

# 6.2 jessie changes (systemd) caused a change to how we handle our
# custom reboot script.  in upgrading from a release prior to 6.2,
# the new version of the custom script won't exist.  for that
# case we need to revert to using the previous override script.
if [ -e $REBOOT ]; then
   logMsg $logFile "Rebooting via $REBOOT..."
   $REBOOT 2>&1 | tee -a $logFile
else
   logMsg $logFile "Rebooting via /sbin/reboot..."
   /sbin/reboot 2>&1 | tee -a $logFile
fi

exit 0
