#!/bin/bash
#############################################################
# Copyright (c) 2015 Sonus Networks, Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# sbxVerifyDrbdPartition.sh
#
#
# Module Description:
# Script to verify the partition size of the server.
# sbxVerifyDrbdPartition.sh is called during pre-install checks for
# install, upgrade, or LSWU. This script first checks the difference
# between mirrorPartitionSize and drbdMetadata, which should be equal to
# drbd0Size. If this check fails, it performs the same check peer side
# and that also fails, we compare size of drbd0 from local and peer.
# Table below explains when drbd0Size will be compared.
#
#  LocalCheckFailed      PeerCheckFailed    compareCheckDrbd0
#         0                    0                   0
#         0                    1                   1
#         1                    1                   1
#         1                    0                   1
#
#######################################################################
#
# RETURN Values  0 -  No issue detected between the backing device
#                     and drbd partition
#                1 -  Internal Error or Size mismatch between drbd and the
#                     backing partition.
# Example:
#
#    sh sbxVerifyDrbdPartition.sh
#
#############################################################

# Sourcing Command Variables
if [ -e /opt/sonus/staging/sonusCommands.sh ]
then
    source /opt/sonus/staging/sonusCommands.sh
elif [ -e /opt/sonus/bin/sonusCommands.sh ]
then
    source /opt/sonus/bin/sonusCommands.sh
else
    /bin/echo "Could not locate sonusCommands.sh Exiting..."
    exit 1
fi

if [ -e /opt/sonus/staging/sonusCommonFiles.sh ]
then
    source /opt/sonus/staging/sonusCommonFiles.sh
elif [ -e /opt/sonus/bin/sonusCommonFiles.sh ]
then
    source /opt/sonus/bin/sonusCommonFiles.sh
else
    $ECHO "Could not locate sonusCommonFiles.sh Exiting..."
    exit 1
fi


# include sonusUtils - Assuming sbxVerifyDrbdPartition.sh script is called from staging,
# make sure sonusUtils.sh script exists in the same path.
# If the path is changed later, need to make sure if the sonusUtils.sh is sourced
# from the right path.
installRoot=$(dirname $0)
. ${installRoot}/sonusUtils.sh

setPeerConnectInfo()
{  # This script is trying to work across multiple releases.
   # The script sonusPeerCnx has changed from a bash script to an expect sript.
   # The sonusPeerCnx script when it is an expect script, deterimes the peer ip
   # address instead of relying on the caller to supply it. This function
   # hides the release dependent options of sonusPeerCnx.  The variable
   # cnxExt is set to { sh | expect } and the variable peerServer is set to { "" | "peer ip address" }

   haPortName="bond0"

   peerCfg=$(checkIface $haPortName)
   if [ $peerCfg -ne 0 ]; then
       haPortName="ha0"
       peerCfg=$(checkIface $haPortName)
   fi

   if [ -e  "$SBXCONF_FILE" ]; then
      $CAT $SBXCONF_FILE | $GREP ^peerCeName | $GREP -q none$
      if [ $? -eq 0 ]; then
          # this is a stand alone system
          peerCfg=1
      fi
   fi

   if [ $peerCfg -eq 0 ]; then
      if [ -e  $SONUS_PEER_CNX_SH ]; then
         cnxExt="sh"
         tmpPeerServer=$(getIpAddr $haPortName)
         if [[ $tmpPeerServer == "169.254.99.1" ]]; then
            peerIpAddr="169.254.88.1"
         elif [[ $tmpPeerServer == "169.254.88.1" ]]; then
            peerIpAddr="169.254.99.1"
         else
           # could not resolve peer ip address
           peerCfg=1
         fi
      elif [ ! -e  $SONUS_PEER_CNX_EXPECT ]; then
           # could not find the peer connect script
           peerCfg=1
      fi

      # is the bond port configured on the peer side
      if [ $peerCfg -eq 0 ]; then
         $sonusPeerCnxExpectScript $peerIpAddr $LS -ltr /tmp > /dev/null
         peerCfg=$?
      fi
   fi
}

rtnVal=0
origCstate=""

$GREP -q "\"mirror\"" $ETC_DRBD_CONF

if [ $? -eq 0 ]; then
   peerCfg=1
   peerIpAddr=""
   cnxExt="expect"
   sonusScriptDir="$SONUS_SBX_SCRIPT_DIR"
   # Get the backing device name ( physical device )
   partName=`$DRBDADM sh-md-dev  mirror 2> /dev/null`
   mirrorPartition=`$DRBDADM sh-md-dev  mirror | cut -d'/' -f3-`
   $DPKG -l | $GREP -q "libexpect-perl"
   if [ $? -eq 0 ]; then
      if [ -e $SONUS_STAGING_DIR/sonusPeerCnx.$cnxExt ]; then
         sonusPeerCnxExpectScript="$SONUS_STAGING_DIR/sonusPeerCnx.$cnxExt"
      else
         sonusPeerCnxExpectScript="$sonusScriptDir/sonusPeerCnx.$cnxExt"
      fi
   else
      sonusPeerCnxExpectScript="$sonusScriptDir/sonusPeerCnx.$cnxExt"
   fi

   setPeerConnectInfo
   # Mirror partition configured. Verify the partition size.
   # check to see if drbd is running
   $LSMOD | $GREP -q drbd
   if [ $? -ne 0 ]; then
      # Check for mirrorPartition due to a reboot during an os upgrade. This script
      # could be called twice, once for the os upgrade, and once for the app. upgrade.
      $UMOUNT $partName 2> /dev/null

      $MODPROBE drbd  > /dev/null
      if [ $? -eq 0 ]; then
         $DRBDADM attach mirror
      else
        # if we cannot load the device driver
        # Assume that we just updated the kernel with the os
        # and the drbd.ko is not setup.
        # DO NOT return an error.
        exit $rtnVal
      fi
   fi

   role=`$DRBDADM role mirror | $AWK -F / {'printf $1'}`
   localFailed=0
   peerFailed=0
   drbd0Size=`$DRBDMETA -f 0 v08 $partName internal dump-md | $GREP la-size-sect  | $AWK {'print $2'} | $TR -dc '0-9, '`
   # meta data is reported in sectors convert 512 sectors into  Kb
   drbd0Size=$((($drbd0Size * 512)/1024))
   mirrorPartitionSize=$($GREP -e $mirrorPartition /proc/partitions | $AWK '{print $3}')
   drbdMetadataSize=$((((($mirrorPartitionSize/32768)+36+4)/4)*4))

   if (( $mirrorPartitionSize - $drbdMetadataSize != $drbd0Size )); then
      if [ $peerCfg -eq 0 ]; then
         preInstallLogMsg "WARNING:local drbd fs mismatch MetaData from drbdmeta. mirrorPartitionSize =$mirrorPartitionSize drbdMetadataSize =$drbdMetadataSize drbd0Size =$drbd0Size . Trying to match on peer side and drbd0 size of both. " "user"
         $localFailed=1
      else
         preInstallLogMsg "ERROR:local drbd fs mismatch MetaData from drbdmeta $drbd0Size $mirrorPartition $mirrorPartitionSize drbdMetadataSize $drbdMetadataSize drbd0 $drbd0Size" "user"
      fi
      $rtnVal=1
   fi

   #If the above check passed for local machine, we need to check it for peer.
   #If the peer doesn't exist, the rtnVal solely depends on the local.
   if [ $peerCfg -eq 0 ]; then
      $sonusPeerCnxExpectScript $peerIpAddr  $LSMOD | $GREP -q drbd
      if [ $? -ne 0 ]; then
         # Check for mirrorPartition due to a reboot during an os upgrade. This script
         # could be called twice, once for the os upgrade, and once for the app. upgrade.
         $sonusPeerCnxExpectScript $peerIpAddr $SUDO $DRBD_CMD_SH UNMOUNT_ATTACH $partName > /dev/null # sudoers-cnxipmadmin
      fi

      peerDrbd0Size=$($sonusPeerCnxExpectScript $peerIpAddr $SUDO $DRBD_CMD_SH META_DUMP_MD $partName | $GREP la-size-sect  | $AWK {'printf $2'} | $TR -dc '0-9, ') # sudoers-cnxipmadmin
      # issuing this command from a newer version of drbd to an older fails ( give partial data ), but going to same version
      # to same version or old version to new does not have an issue. Lets retry if null.
      peerMetRtry=0
      while [ -z "$peerDrbd0Size" ]
      do
        $SLEEP 1
        peerMetRtry=$($EXPR $peerMetRtry + 1)
        if [ $peerMetRtry -gt 4 ]; then
           preInstallLogMsg "local data validation only continuing"
           exit $rtnVal
        fi
        peerDrbd0Size=$($sonusPeerCnxExpectScript $peerIpAddr $SUDO $DRBD_CMD_SH META_DUMP_MD $partName | $GREP la-size-sect  | $AWK {'printf $2'} | $TR -dc '0-9, ') # sudoers-cnxipmadmin
      done

      peerMirrorPartitionSize=$($sonusPeerCnxExpectScript $peerIpAddr $GREP -e $mirrorPartition /proc/partitions | $AWK '{print $3}') # sudoers-cnxipmadmin
      peerDrbdMetadataSize=$((((($peerMirrorPartitionSize/32768)+36+4)/4)*4))
      # meta data is reported in sectors convert 512 sectors into  Kb
      peerDrbd0Size=$((($peerDrbd0Size * 512)/1024))


      if (( "$peerMirrorPartitionSize" - "$peerDrbdMetadataSize" != $peerDrbd0Size )); then
         preInstallLogMsg "WARNING: peer drbd fs mismatch MetaData from drbdmeta. peerMirrorPartitionSize =$peerMirrorPartitionSize peerDrbdMetadataSize =$peerDrbdMetadataSize peerDrbd0Size= $peerDrbd0Size . Trying to match drbd0 size of both"  "user"
         $peerFailed=1
      fi
      if [ $peerFailed -eq 1 -o $localFailed -eq 1 ]; then

         #If previous check failed on any of the local or peer, we need to check if
         #drbd0 size on both are same or not
         if ((  $drbd0Size != $peerDrbd0Size )); then
            preInstallLogMsg "ERROR: mismatch in drbd0 size of local and peer machine localDrbd0: $drbd0Size peerDrbd0: $peerDrbd0Size" "user"
            rtnVal=1
            exit $rtnVal
         else
            $rtnVal=0
            exit $rtnVal
         fi
      else
         exit $rtnVal
      fi

   fi
fi

exit $rtnVal

