#!/bin/bash
#############################################################
# Usage: sh switch_boot.sh root2|root1|sda6|sda7 [lvRootDir]
#############################################################

# Sourcing Command Variables
if [ -e /opt/sonus/staging/sonusCommonUtils.sh ]; then
   source /opt/sonus/staging/sonusCommonUtils.sh
elif [ -e /opt/sonus/bin/sonusCommonUtils.sh ];then
   source /opt/sonus/bin/sonusCommonUtils.sh
else
   /bin/echo "Could not locate sonusCommonUtils.sh Exiting..."
   exit 1
fi

usage()
{
   errMsg="$1"
   $ECHO -e "$errMsg\nUsage: $0 root1|root2|${primary_disk}7|${primary_disk}6 [lvRootDir]"
   $ECHO -e "\tOptional param to also reset /etc/grub.d files in lvRootDir (used during partitioning)"
}

# determine the option # for the last menu item
# note: this function uses the custom menu file in /etc/grub.d and not grub.cfg
lastOption()
{
   local menuFile=$1
 
   # menu options are 0-based!
   # note: differentiate between the function calls and definition by looking
   # for a space after the function name. this avoids finding the prototype
   # which will have "()" after the name.  If whitespace in the file changes
   # or the file format changes, this cide could/would need updating!
   local totalOpts=$($GREP -c "^sonus_linux_entry " $menuFile)
   local lastOpt=$((totalOpts-1))
   $ECHO $lastOpt
}

boot=$1
lvRootDir=$2

if [[ "$boot" != "root1" && "$boot" != "root2" && "$boot" != "${primary_disk}7" && "$boot" != "${primary_disk}6" ]]; then
   usage "Invalid boot specification"
   exit 1
fi
if [[ -n "$lvRootDir" && ! -d "$lvRootDir" ]]; then
   usage "$lvRootDir does not exist!"
   exit 1
fi

# After an iso installation and we come up after reboot, we always detect /dev/sda 
# as the primary partition. This is because we dont have any kernel driver for usb 
# which makes sense as we dont support usb based installation. So inorder for 
# partition tool to work fine on /dev/sda6 or /dev/sda7 we should not change the grub
# config file to /dev/sdc as it wont be detected after install. So adding the below variable 
# for grub purposes here.
connexipPrimaryDevice=/dev/sda

# get current default setting
currentDefault=$($GRUB_EDITENV list | $GREP saved_entry | $AWK -F= '{ print $2 }')

# set variables depending on boot param: they are used to setup the grub menu
newDefault=
newRoot=
newConfigFile=

case "$boot" in
       'root1' | 'root2')
         newConfigFile=$SONUS_GRUB_MENU_SDA1_FILE
         newRoot="/dev/mapper/debian-$boot"

         # setup per menu item devices
         bDev=$( [ "$boot" == "root1" ] && echo "root2" || echo "root1")
         sDev=$( [ "$boot" == "root1" ] && echo "root1" || echo "root2")
         pDev=$( [ "$boot" == "root1" ] && echo "root2" || echo "root1")
         cDev=$( [ "$boot" == "root1" ] && echo "root1" || echo "root2")

         # determine current boot partition
         bootedRoot=$($SED -n '/partitioning/,/}/p' $GRUB_CONFIG | $GREP "vmlinuz" | $AWK -F"root=" '{print $2}' | $AWK '{print $1}')
         if [ -z "$bootedRoot" ]; then
             bootedRoot=$($SED -n '/sonus-curr-ver/,/}/p' $GRUB_CONFIG | $GREP "vmlinuz" | $AWK -F"root=" '{print $2}' | $AWK '{print $1}' | $AWK -F'-' '{print $2}')
         fi

         if [[ $bootedRoot == "${primary_device}7" || $bootedRoot == "${primary_device}6" ]];then
            $SED -i -e "/base-os/s|/dev/mapper/debian-root|/dev/mapper/debian-$bDev|" \
                    -e "/single-user/s|/dev/mapper/debian-root|/dev/mapper/debian-$sDev|" \
                    -e "/sonus-prev-ver/s|/dev/mapper/debian-root|/dev/mapper/debian-$pDev|" \
                    -e "/sonus-curr-ver/s|/dev/mapper/debian-root|/dev/mapper/debian-$cDev|" $newConfigFile
            newDefault=$(lastOption $newConfigFile)
         elif [[ "$bootedRoot" != "$boot" ]]; then
             lastOpt=$(lastOption $newConfigFile)
             if [[ $lastOpt -eq $currentDefault ]]; then
                newDefault=$((currentDefault-1))
             else
                newDefault=$lastOpt
             fi
             $SED -i -e 's/sonus-prev-ver/sonus-newcurr-ver/' \
                     -e 's/sonus-curr-ver/sonus-prev-ver/' -e  's/sonus-newcurr-ver/sonus-curr-ver/' $newConfigFile
         else
            newDefault=$currentDefault
         fi
         ;;

       "${primary_disk}6")
         newConfigFile=$SONUS_GRUB_MENU_SDA6_FILE
         newRoot="${connexipPrimaryDevice}6"
         newDefault=$(lastOption $SONUS_GRUB_MENU_SDA6_FILE)
         ;;
 
       "${primary_disk}7")
         newConfigFile=$SONUS_GRUB_MENU_SDA7_FILE
         newRoot="${connexipPrimaryDevice}7"
         newDefault=$(lastOption $SONUS_GRUB_MENU_SDA7_FILE)
         ;;
       *)
         usage "Unknown boot specification"
         exit 1
         ;;
esac

# update /etc/grub.d file permissions so only the desired/needed config
# file generates menu items.  During partitioning and upgrades we also need
# to update permissions on a second logical partition or those permissions won't
# be correct. /boot is a seperate physical partition and there is no issue
# with it/it is changed for all root partitions.
newConfigFileName=`$BASENAME $newConfigFile`
$ECHO "Switch_Boot: Update /etc/grub.d permissions to activate $newConfigFileName"
$CHMOD a-x ${SONUS_GRUB_MENU_BASE}*
$CHMOD a+x $newConfigFile

if [[ -n "$lvRootDir" ]]; then
   if [[ "$boot" == "root1" || "$boot" == "root2" ]]; then
       $ECHO "Switch_Boot: Mimicking $newConfigFileName changes in $lvRootDir/etc/grub.d"
       $CP -af $newConfigFile ${lvRootDir}${newConfigFile}
   fi

   $ECHO "Switch_Boot: Update $lvRootDir/etc/grub.d permissions to activate $newConfigFileName"
   $CHMOD a-x ${lvRootDir}${SONUS_GRUB_MENU_BASE}*
   $CHMOD a+x ${lvRootDir}/${newConfigFile}
fi

# update the grub.cfg file
# note: also outputting as an echo to capture in install and partition logs
$LOGGER -t Switch_Boot "Changing boot menu to boot from $newRoot"
$ECHO "Switch_Boot: Update grub.cfg to boot from $newRoot"
$UPDATE_GRUB_SH

# set the default entry
if [[ $newDefault -ne $currentDefault ]]; then
   $LOGGER -t Switch_Boot "Set default boot to menu entry $newDefault" 
   $ECHO "Switch_Boot: Set default boot to menu entry $newDefault"
   $GRUB_SET_DEFAULT $newDefault
fi

exit 0
