#!/usr/bin/env python3

# import re
import sys
import shutil
import os

import datetime
import hashlib
import tarfile
import traceback
import glob
import subprocess
from distutils.dir_util import copy_tree
import json
import uuid
import codecs

# from pprint import pprint
# from StringIO import StringIO

warning_msg = ""
sub_version = ""
prod_version = "11.00.00"
script_dir = os.path.dirname(os.path.realpath(__file__))
vnfd_template = ""
vnfd_proprietary_template = os.path.join(script_dir, "vnfmVnfdTemplate.yaml")
vnfd_sol001_template = os.path.join(script_dir, "vnfmSol001VnfdTemplate.yaml")
vnfd_old_sol001_template = os.path.join(script_dir, "vnfm_pre_21_2_Sol001VnfdTemplate.yaml")
vnfd_ribbon_custom_types_template = os.path.join(script_dir, "vnfmSol001VnfdCustomTypes.yaml")
etsi_vnfd_types_file = os.path.join(script_dir, "etsi_nfv_sol001_vnfd_types.yaml")
etsi_common_types_file = os.path.join(script_dir, "etsi_nfv_sol001_common_types.yaml")
qcow2_file = ""
affinity = "AntiAffinity"
no_dhcp_ha = False
certificate_file = os.path.expanduser("~/certs/certificate.pem")
private_key_file = os.path.expanduser("~/certs/key.pem")
sbc_public_key = "ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABAQC/JjfvXORWn/lAq+9C8cRft"
sbc_public_key += "SksCwqsUHJ6CgCweIQdaKgs9KbDtgWhQ2fNmwlT+L7mLcYOBaL7/LIy59kE1"
sbc_public_key += "VZt9FQgOvZ6UYCRLP8etJeA01VmgRZUEMcvAeIbe+BApxSfJh4Lzm/laOEw3"
sbc_public_key += "7/oDGs2wPTSeG6pMdGnvPGH8hNjYt0rxagwUN5vRP7i7VUi8oalGxr7XhwZK"
sbc_public_key += "JM6aXAjD2BTaogj40uwygDNvK9ZRw7IHSZLExA7vPKKRzQ0/MjdAqCWgaUs9"
sbc_public_key += "XCmFFHli2rlh8X3MInKV+byJhnd3oN0Um3t3znoIq7cJZGy5QYKIOSar8G8J"
sbc_public_key += "ofl5tobn+y7M4EbIXP7 root@S3Ott07"

userids = {}

# This could be configurable for 0 or 1, but for now it's hard-coded to 1
first_rg_index = 1
first_pkt0_trunk_index = 1
first_pkt1_trunk_index = 1


class CustomParm:
    def __init__(self, name, default, desc, editableState):
        self.name = name
        self.default = default
        self.desc = desc
        self.editableState = editableState


class CustomEnv:
    def __init__(self, env, val):
        self.env = env
        self.val = val


class AltPktIps:
    def __init__(self, pktn, alt_index, num_ips):
        self.pkt_index = pktn
        self.alt_index = alt_index
        self.num_ips = num_ips
        global env_var_to_print
        global warning_msg
        # Check for an env var like ALT_PKT0_1_KEY, ALT_PKT0_1_IFNAME, or ALT_PKT0_1_IPS
        env_alt_key = "ALT_PKT" + pktn + "_" + alt_index + "_KEY"
        if env_alt_key in os.environ:
            env_alt_key_val = str(os.environ[env_alt_key])
            env_var_to_print.append(env_alt_key + '="' + env_alt_key_val + '"')
            self.alt_if_key = expand_custom_name(env_alt_key_val)
            if not self.alt_if_key.startswith('ALT_'):
                warning_msg += "Warning: environment variable:" + env_alt_key + " does not start with ALT_\n"
            if self.alt_if_key.find('${alt_if_ip_index}') == -1:
                # Just issue a warning -- this could be valid if there is only one.
                # But if there are more than one, they need to be unique.
                warning_msg += "Warning: environment variable:" + env_alt_key + " does not contain %alt_if_ip_index%\n"
        else:
            # Use a default key
            self.alt_if_key = "ALT_PKT" + pktn + "${opt_alt_index" + pktn + "}_${alt_if_ip_index}"
        env_alt_name = "ALT_PKT" + pktn + "_" + alt_index + "_IFNAME"
        if env_alt_name in os.environ:
            env_alt_name_val = str(os.environ[env_alt_name])
            env_var_to_print.append(env_alt_name + '="' + env_alt_name_val + '"')
            self.alt_if_name = expand_custom_name(env_alt_name_val)
        else:
            # Use a default name
            self.alt_if_name = "IF" + str(int(pktn) + 2)
        env_alt_ip = "ALT_PKT" + pktn + "_" + alt_index + "_IPS"
        self.alt_if_ips = []
        if env_alt_ip in os.environ:
            env_alt_ip_val = str(os.environ[env_alt_ip]).split(',')
            env_var_to_print.append(env_alt_ip + '="' + str(os.environ[env_alt_ip]) + '"')
            self.alt_if_ips = env_alt_ip_val
        # Add default values (if there was no environment variable, or not enough ips)
        if len(self.alt_if_ips) < int(num_ips):
            for i in range(1, int(num_ips)+1):
                if len(self.alt_if_ips) < i:
                    self.alt_if_ips.append("127." + pktn + "." + alt_index + "." + str(i))


class PrefixToInfo:
    def __init__(self, prefix, var0, var1):
        self.prefix = prefix
        self.var0 = var0
        self.var1 = var1


def usage(command_name, verbose=False):
    print("Usage:" + command_name + " [options]")
    print("")
    print("-f <flavor size> [dev|std] (default: all)")
    print("-h <ha type> [simplex|ha|n_1_ha] (default: all)")
    print("-i <interface type> [virtio|sriov|provider] (default: all)")
    print("-p <prod_version>")
    print("-q <qcow_image_file> (default: image file omitted)")
    print("-s <subversion>")
    print("-v <vnfd_template>")
    print("-t <vnfd_custom_types>")
    print("--old_vnfm <true/false> (Use older VNFD for use with Ribbon VNFMs before V21.2. default: false)")
    print("--cinder_boot (Allocates a cinder disk for boot image. default: no cinder boot)")
    print("--cinder_log (Allocates a cinder disk for logging. default: no cinder device for logging)")
    print("--ha_io_type (specify a different interface type for ha network [virtio|sriov|provider] (default: virtio))")
    print('  (ha_io_type is deprecated: use "--ha0 [virtio|sriov|provider]" instead)')
    print("--image_name <image_name> (default: makes image name from product and version strings)")
    print("--ipv6 (use ipv6 as default network value instead of ipv4)")
    print("--mgt_io_type (specify a different interface type for mgmt network [virtio|sriov|provider] (default: off))")
    print('  (mgt_io_type is deprecated: use "--mgt0 [virtio|sriov|provider]" instead)')
    print("--mgt_vip (add a vip to mgt -- io_type must not be virtio) (default: no vip)")
    print("--num_oam_instances <num_oam_instances> (default: 0)")
    print("--num_instances <n> (max n_1_instances, default: 5 for 4_1)")
    print("--num_sbc_rg <num_sbc_rg> (Sets the number of SBC Redundancy Groups. default: 1, max: 8)")
    print("--personality <isbc|msbc|ssbc|tsbc|mrfp|slb> (default: all)")
    print("--pkt_vip (add a vip to pkt -- io_type must not be virtio) (default: no vip)")
    print("--pkt_oam_io_type (specify an interface type for oam VMs pkt network) (default: disabled)")
    print("--prov_mgmt_virtio (default: off, provider io has provider mgmt)")
    # print("--sbc_tar <file> Specifies the input tar file, or 'omit' to skip. default: template_dir + '/sbc-V*.tar.gz'")
    print("--sbc_tar <file> Specifies the input tar file, 'search' to look in template_dir or 'omit' to skip. default: omit")
    print("          Note: Required for automated upgrades, not used otherwise")
    print("--sbc_public_key <file> (file containing a public key, such as id_rsa.pub, authorized to ssh to the SBC)")
    print("--sriov_redundancy (adds two additional packet interfaces if SRIOV is the io_type)")
    print("--userid <userid>[:<public key file>] additional userid to authorize, with their public key")
    print("          Note: this argument can be repeated multiple times for multiple users")
    print("          Note: if the <public key file> is omitted, the sbc_public_key value is used")
    print("--vCPU <num_vCPU> | --RAM <ram_mb> | --DISK <disk_gb> (Optional custom sizing configurations. RAM in MB and DISK in GB)")
    print("--oam_vCPU <num_vCPU> | --oam_RAM <ram_mb> | --oam_DISK <disk_gb> (Optional OAM node custom sizing configurations. RAM in MB and DISK in GB)")
    print("--cinder_oam_DISK <disk_gb> | --cinder_sbc_DISK <disk_gb> | --cinder_log_DISK <disk_gb> (Optional disk sizing if cinder is used)")
    print("--zone <name> | --oam_zone <name> (specifies the availability zone, default: omit)")
    print("--affinity_across_rgs (specify if two redundancy groups can be placed on the same server) (default: disabled)")
    print("--redundancy_ratio <n:m> (specify the redundancy ratio of for a N:1 setup) (default: num_instances-1:1)")

    if verbose:
        print("\n\nAdditional options intended for specific customer configurations")

        print("\nCinder Additional Attributes")
        print('--cinder_disktype <name> (eg: "solidfire", default none)')

        print("\nInterface rules set additional options for a given interface.")
        print("--mgt0:<options> --ha0:<options> --pkt0:<options> --pkt1:<options>")
        print("--oampkt:<options> (which override pkt0/1 for the OAM VM)")
        print("  The options can be a colon separated list of values of the following:")
        print('  "ipv4" or "ipv6" or "ipvn": only the specified ip version is given.')
        print('     (The default is ipvn, which implies both are available, selected during')
        print('     orchestration, except for --oampkt, which has a default of ipv4).')
        print('  "rules": Adds security rules for the interface')
        print('     (default: no rules, unless --security_restrict is set)')
        print('  "rulesoam": Adds separate rules for OAM (only for mgt0 and ha0)')
        print('     (default: same rules apply to mgt0 and ha0 on both OAM and SBC')
        print('  "rip=n": The number of remote IPs to add to rules [n=0-2] default=2')
        print('  "ems_rip=n": The number of remote IPs to add to rules for EMS IPs [n=0-2] default=2')
        print('  "virtio" or "provider" or "sriov": set the interface type for ha0 or mgt0')
        print('  "ALT_M=N" For pkt0 or pkt1 only, create an alternate interface M, with N IPs.')
        print('     M and N must be numbers, such as "pkt0 ALT_1=5:ALT_2=5" to add two sets of 5 IPs')
        print("\nIpv6 uses additional security groups. If ipv6 isn't required, the support")
        print("for it can be removed from the CSAR.")
        print("--omit_ipv6 (omit ipv6 and the choice of ip vers)")

        print("\nPlacement Groups affect which affinity policies are assigned to VMs and")
        print("the name selected affects the server group name.")
        print('--placement_sbc_name <name> (default is "Sbc%VNFTYPE%VmPlacement")')
        print('--placement_oam_name <name> (default is "Oam%VNFTYPE%VmPlacement")')

        print("\nSecurity Groups settings: Security groups typically require manual modification")
        print("after orchestration. By default, all ports are allowed, but with this setting")
        print("additional rules restrict the ports to the recommended settings.")
        print("--security_restrict (security groups rules restrict access to recommended ports)")

        print('\nScaling Options can be enabled with the following options:')
        print('--enable_scaling (enable scaling options)')
        print('    Note: OAM nodes are required to enable scaling')
        print('--max_sbc_rg <maximum> (The maximum number of redundancy groups that a single VDU can scale out to. Default 8)')

#       new command line arguement to disable using get_attribute function in VNFD file.
#       When this option is provided, default values are assigned to corresponding variables.
        print("--no_get_attrib (this shuould be enabled when get_attribute() is not supported)")

        print("--num_pkt0_trunk <num_trunk_pkt0> (Sets the number of trunk ports for pkt0 default: 0, max: 4024)")
        print("--num_pkt1_trunk <num_trunk_pkt1> (Sets the number of trunk ports for pkt1 default: 0, max: 4024)")

        print('\nDNS configuration can be added to the VNFD using a JSON DNS configuration file.')
        print('DNS can be configured on OAM Mgt0, SBC Mgt0, SBC Pkt0 and SBC Pkt1,')
        print('when those interface io types are virtio.')
        print("""The configuration file should be in the format:
{
    "oam": {
        "fqdn": "<fqdn>",
        "server_ip": "<server IP>",
        "dns_type": "<dns record type>",
        "dns_protocol": "<protocol for dns>",
        "dns_service": "<dns service name>"
    },
    "mgt": {
        "fqdn": "<fqdn>",
        "server_ip": "<server IP>",
        "dns_type": "<dns record type>",
        "dns_protocol": "<protocol for dns>",
        "dns_service": "<dns service name>"
    },
    "pkt0": {
        "fqdn": "<fqdn>",
        "server_ip": "<server IP>",
        "dns_type": "<dns record type>",
        "dns_protocol": "<protocol for dns>",
        "dns_service": "<dns service name>"
    },
    "pkt1": {
        "fqdn": "<fqdn>",
        "server_ip": "<server IP>",
        "dns_type": "<dns record type>",
        "dns_protocol": "<protocol for dns>",
        "dns_service": "<dns service name>"
    }
}""")  ##Triple speech marks allow multiline
        print('Removing whichever configuration is not needed.')
        print('The keys should be given the following values:')
        print('    - fqdn: The FQDN for this inteface')
        print('    - server_ip: The DNS server IP')
        print('    - dns_type: The DNS record type. Values should only be A/AAAA/SRV/NAPTR')
        print('    - dns_protocol: The transport protocol of the desired service')
        print('    - dns_service: The symbolic name of the desired service')
        print('To add DNS configuration use the following parameter:')
        print('  --dns_file <dns options file path> (The file path to the DNS configuration)')

        print("\n--timeout <seconds> (The timeout at the CLI / bash shell. Default is 300)")

        print("\nVersion suffix is added to the package version in the CSAR's .mf file.")
        print("This allows very similar CSAR with the same version loaded separately by the VNFM")
        print("--version_suffix <name> (default is "" for no suffix)")

        print("\n\nAdditional options intended for design or debugging:")
        print("\nAffinity policy should generally be set to the default \"AntiAffinity\", but when")
        print("trying to distribute a large number of VMs on different physical CPUs, there may")
        print("be problems if the number of CPUs is small. Using \"None\" will allocate them on")
        print("whichever CPU is least loaded (which may give better performance). Using")
        print("\"Affinity\" will force them onto the same CPU (which may stress a single CPU).")
        print("--affinity <policy> [AntiAffinity|Affinity|None] (default: AntiAffinity)")

        print("\nCertificate files were required by an early version of the SOL001 standard, but")
        print("generally a sha256 of the CSAR is now preferred. If you have a certificate")
        print("file configured and want to use it, add it as follows:")
        print("--certificate_file <file> (default: ~/certs/certificate.pem) if missing, create sha256 instead")
        print("--private_key_file <file> (default: ~/certs/key.pem)")

        print("\nCinder volume id's aren't supplied by VNFM, so SBC attempts to figure out the")
        print("volume id from the device name automatically in newer releases, such as")
        print("recent 8.2 builds. However in older releases, when both cinder_boot and")
        print("cinder_log are specified, you need to specify if the volume name for logging")
        print("will be vdb (normal) or an different device (like vda)")
        print('--cinder_log_device <device_name> (eg: "vdb", default none)')

        print("\nMop directory can be added to the CSAR for automated upgrades, but generally")
        print("the --sbc_tar specified already contains a mop_dir. To explicitly add a mop_dir")
        print("when it's missing from the sbc_tar, use the following:")
        print("--mop_dir <dir> (Specifies input mop dir (or tar file) added to CSAR. default: ${COMMON_ROOT}/VNFR/lib/mop)")
        print("     Note: The mop_dir parameter is ignored if the sbc_tar already contains a mop directory")

        print("\nGenerally the default DHCP settings should be used, but this can be changed:")
        print("--no_dhcp_ha (default: dhcp is enabled on internal ha network)")

        print("\nInsert a name and string value into userData.json:")
        print("--insert_userdata <name1>=<defaultVal1>:<name2>=<defaultVal2>:...")
        print("--dos_support_sec_pkt_ports <value> (True|False|default): equivalent to: --insert_userdata DosSupportSecPktPorts:value")
        print("     The default value is normally True, but is set to False for certain configurations")

        print("\nPassword access via the console may be required to debug startup problems,")
        print("but by default password access is not enabled. The following commands inject")
        print("an encrypted password of your choosing at startup:")
        print("--passwd_admin <encrypted passwd> (initial admin password)")
        print("--passwd_linuxadmin <encrypted passwd> (initial linuxadmin password)")
        print("--passwd_default (enable admin and linuxadmin with default encrypted passwords)")
        print("     Passwords are not recommended, but if they are used, set them as follows:")
        print("        pwd_a=$(mkpasswd --method=SHA-512 --rounds=4096)")
        print("        pwd_l=$(mkpasswd --method=SHA-512 --rounds=4096)")
        print('        ... --passwd_admin "${pwd_a}" --passwd_linuxadmin "${pwd_l}"')

        print("")
    else:
        print("\n To see additional debugging options, use --help\n")

    print("For additional information on customized naming and configuration, use --help_custom")
    global warning_msg
    if warning_msg != "":
        sys.stderr.write(warning_msg)
        warning_msg = ""

    # Undocumented option: "--ha_mode (Allow the user to specify the HA mode, default: default, which makes the system pick 1:1 or N:1)"
    # Undocumented option: "--no_custom_naming (Disable custom interface naming, required for some older VNFM version)"
    # Undocumented option: "--single_system_name (Use a single system_name for all RGs. This may become the default if it works properly)"

    # Additional note: If you want to use certificates, and don't have them,
    # you can create self-signed ones with a command like:
    #    mkdir ~/certs
    #    openssl req -newkey rsa:2048 -nodes -keyout ~/certs/key.pem -x509 -subj "/C=CA/ST=ON/L=Ottawa/O=Ribbon/CN=10.6.40.191" -days 36524 -out ~/certs/certificate.pem
    # Just replace the "subj" with your location and the ipaddress of the machine you're using to run the script.


def usage_custom():
    print("\nCustom Naming:")
    print("A custom name can be specified for CeNames and other entities, using command line")
    print("options and/or environment variables. The names may include the following tokens:")
    print("   %rg_index% : for the redundancy group index")
    print("   %vnftype% or %VNFTYPE% : for the vnftype (eg: SSBC) in lower or uppercase.")
    print("   For PKT Alternate IF names only: %alt_if_ip_index%, %alt_if_index%.")
    print("   VNFM tokens: %orchname%, %basename%, %index%, %vmname%, %vmtype%, %get_param:parmname%")
    print("       The %index% field can support leading 0's, using %index%{02d}")
    print("       VNFM documentation describes which VNFM tokens can apply to which entities.")
    print("\nCommand line options:")
    # print("--custom_first_rg_index <index> (default: 1)")
    print("--custom_oam_name <name> (default: \"OAM\")")
    print("--custom_sbc_name <name> (default: \"RG%rg_index%-%VNFTYPE%\" or \"%VNFTYPE%\" if num_sbc_rg = 1)")
    print("--custom_oam_server_group_name (default: %orchname%-OAM-%VNFTYPE%-SG)")
    print("--custom_sbc_server_group_name (default: %orchname%-SBC-%VNFTYPE%%rg_index%-SG")
    print("--custom_system_name <name> (default: SystemName). The oam_name or sbc_name is appended.")
    print("--custom_vmname_dash: Adds a dash in the ceName before the VM index (default: no dash)")

    print("\nIn addition to the command line options, the following environment variables")
    print("provide the ability to override the default names for other entities:")
    print("CUSTOM_COMMON_SYSTEM_NAME              Default: SystemName")
    print("CUSTOM_OAM_HA_PATTERN                  Default: %vmname%-ha0")
    print("CUSTOM_OAM_MGT_PATTERN                 Default: %vmname%-mgt0")
    print("CUSTOM_OAM_PKT0_PATTERN                Default: %vmname%-pkt0 (if pkt0 is used on OAM)")
    print("CUSTOM_OAM_PKT1_PATTERN                Default: %vmname%-pkt1 (if pkt1 is used on OAM)")
    print("CUSTOM_OAM_SERVER_GROUP_NAME           Default: %orchname%-OAM-${VNF_TYPE}-SG")
    print("CUSTOM_OAM_SYSTEM_NAME                 Default: ${system_name}-${oam_name}")
    print("CUSTOM_OAM_VMGT_PATTERN                Default: %vmname%-vmgt0 (if vmgt0 is used)")
    print("CUSTOM_OAM_VMNAME_PATTERN              Default: %orchname%-%basename%-%index%")
    print("CUSTOM_SBC_HA_PATTERN                  Default: %vmname%-ha0")
    print("CUSTOM_SBC_MGT_PATTERN                 Default: %vmname%-mgt0")
    print("CUSTOM_SBC_PARM_*_DEFAULT              Default: None")
    print("CUSTOM_SBC_PARM_*_DESC                 Default: None")
    print("CUSTOM_SBC_PARM_*_EDITABLESTATE        Default: [UPGRADE]")
    print("CUSTOM_SBC_PARMS                       Default: None")
    print("CUSTOM_SBC_PKT0_PATTERN                Default: %vmname%-pkt0")
    print("CUSTOM_SBC_PKT0S1_PATTERN              Default: %vmname%-pkt0S1")
    print("CUSTOM_SBC_PKT1_PATTERN                Default: %vmname%-pkt1")
    print("CUSTOM_SBC_PKT1S1_PATTERN              Default: %vmname%-pkt1S1")
    print("CUSTOM_SBC_SERVER_GROUP_NAME           Default: %orchname%-SBC-${VNF_TYPE}-SG")
    print("CUSTOM_SBC_SYSTEM_NAME                 Default: ${system_name}-${sbc_name}")
    print("CUSTOM_SBC_VMGT_PATTERN                Default: %vmname%-vmgt0 (if vmgt0 is used)")
    print("CUSTOM_SBC_VMNAME_PATTERN              Default: %orchname%-%basename%-%index%")
    print("CUSTOM_SBC_VPKT0_PATTERN               Default: %vmname%-vpkt0 (if vpkt0 is used)")
    print("CUSTOM_SBC_VPKT1_PATTERN               Default: %vmname%-pkt1")
    print("CUSTOM_SG_NAME_HA                      Default: [Use VNFM default]")
    print("CUSTOM_SG_NAME_MGT                     Default: [Use VNFM default]")
    print("CUSTOM_SG_NAME_OHA                     Default: [Use VNFM default]")
    print("CUSTOM_SG_NAME_OMGT                    Default: [Use VNFM default]")
    print("CUSTOM_SG_NAME_PKT                     Default: [Use VNFM default]")

    print("\nSince there can be many environment variables, it's suggested that a config file")
    print("be created for a given customer (eg: custX_cfg.sh) that exports the desired")
    print("variables when it's sourced.")

    print("\nWhile most of the variables are reasonably simple, the CUSTOM_SBC_PARMS, ")
    print("CUSTOM_SBC_PARM_DESC, and CUSTOM_SBC_PARM_DEFAULT need a special explanation.")
    print("The CUSTOM_SBC_PARMS specifies a comma separated list of parmameters to create. ")
    print("Then for each parameter, a Description and Default can be supplied. For example:")

    print("    export CUSTOM_SBC_PARMS=\"CLLI,Subtype\"")
    print("    export CUSTOM_SBC_PARM_CLLI_DEFAULT=\"WNCKCAAA\"")
    print("    export CUSTOM_SBC_PARM_CLLI_DESC=\"CLLI Location code\"")
    print("    export CUSTOM_SBC_PARM_SUBTYPE_DEFAULT=\"CIGW\"")
    print("    export CUSTOM_SBC_PARM_SUBTYPE_DESC=\"Subtype PS - P-SBC Signalling; PM - P-SBC Media; CIGW\"")

    print("\nThese parameters (with their default value and description) will then appear on")
    print("the VNFM orchestration screen, where the operator can override the default with")
    print("their own selection. The parameter can then be referred to in other variables using")
    print("\"%get_param:parmname%\". In this way, the \"CLLI Location code\" (as an example)")
    print("could be specified by the operator, and prepended to other variables.")

    print("\nWhile there is some validation of these values, the user is ultimately")
    print("responsible for making sure these values lead to unique names where")
    print("required. For example, if the default CUSTOM_OAM_SYSTEM_NAME and CUSTOM_SBC_SYSTEM_NAME")
    print("are changed, the user must ensure that they are different. Simiarly CUSTOM_OAM_VMNAME_PATTERN")
    print("and CUSTOM_SBC_VMNAME_PATTERN must end with an \"%index%\".")

    print("\nNotice that some of these environment variables overlap with previous")
    print("command-line options:")
    print("    --custom_oam_server_group_name")
    print("    --custom_sbc_server_group_name")
    print("    --custom_system_name")
    print("The command line options are left for backwards compatibility, but should be ")
    print("considered deprecated. However, if you use both the command line option")
    print("and the corresponding environment variable, the command line option is chosen.")
    print('\nIf option "--pkt[X] ALT_[M]=[N]" is used to add alternate pkt interfaces, then')
    print('the following environment variables can be used to name those interfaces:')
    print('ALT_PKTX_M_KEY                Default: ALT_PKTX_M_%alt_if_ip_index%')
    print('ALT_PKTX_M_IFNAME             Default: IF[M+2]')
    print('ALT_PKTX_M_IPS                Default: "127.X.M.1,127.X.M.2,...127.X.M.N"')
    print("\nCustom Configuration:")
    print("In addition to custom naming, some additional environment variables are used")
    print("to configure other values to a different default in the VNFD file:")
    print("CUSTOM_SBC_MGT_PREFIXV4                Default: 24")
    print("CUSTOM_SBC_HA_PREFIXV4                 Default: 24")
    print("CUSTOM_SBC_PKT0_PREFIXV4               Default: 24")
    print("CUSTOM_SBC_PKT1_PREFIXV4               Default: 24")
    print("CUSTOM_SBC_MGT_PREFIXV6                Default: 60")
    print("CUSTOM_SBC_HA_PREFIXV6                 Default: 60")
    print("CUSTOM_SBC_PKT0_PREFIXV6               Default: 60")
    print("CUSTOM_SBC_PKT1_PREFIXV6               Default: 60")
    print("CUSTOM_SBC_HA_VLANID                   Default: 3005")
    print("CUSTOM_SBC_PKT0_VLANID                 Default: 3003")
    print("CUSTOM_SBC_PKT1_VLANID                 Default: 3004")

    print("\nAll custom naming and configuration values are only used if the chosen")
    print("configuration supports it. So OAM values on apply if there is an OAM VM,")
    print("VLANID only apply if the io_type is SRIOV, etc.")

    print("\nAll command line options and environment variables used are shown in a comment")
    print("block at the start of the VNFD file. This allows a given CSAR to be easily")
    print("recreated or modified at a later date.")


def expand_custom_name(input_name):
    if not input_name:
        return ""

    output_name = input_name.replace(" ", "")
    output_name = output_name.replace("%rg_index%", "${sbc_rg_index}")
    output_name = output_name.replace("%VNFTYPE%", "${VNF_TYPE}")
    output_name = output_name.replace("%vnftype%", "${vnftype}")
    # These are only replaced in the #sbc_alt_ip# section of the template:
    output_name = output_name.replace('%alt_if_index%', '${alt_if_index}')
    output_name = output_name.replace('%alt_if_ip_index%', '${alt_if_ip_index}')

    return output_name


def get_args(argv=None):
    # To try to get this to run on older python versions, don't use argparse
    global cmdline_args
    cmdline_args = ' '.join(sys.argv[1:])
    print("cmdline:" + cmdline_args)
    global sub_version
    global vnfd_template
    global prod_version
    global qcow2_file
    # For provider networks, should the mgmt if also be provider?
    global cinder_boot
    cinder_boot = False
    global cinder_log
    cinder_log = False
    global designer_debug
    designer_debug = False
    global no_dhcp_ha
    global dash_index
    dash_index = False
    global oam_name
    oam_name = "OAM"
    global sbc_name
    sbc_name = "${SBC_RG_PREFIX_DASH}${VNF_TYPE}"
    global placement_sbc_name
    placement_sbc_name = "Sbc${VNF_TYPE}${sbc_rg_index}VmPlacement"
    global placement_oam_name
    placement_oam_name = "Oam${VNF_TYPE}VmPlacement"
    global system_name
    system_name = ""
    global oam_server_group
    oam_server_group = ""
    global sbc_server_group
    sbc_server_group = ""
    global default_mgt_ip_vers
    default_mgt_ip_vers = "ipv4"
    global default_ha_ip_vers
    default_ha_ip_vers = "ipv4"
    global default_pkt_ip_vers
    default_pkt_ip_vers = "ipv4"
    global dsspp
    dsspp = "default"
    global omit_ipv6
    omit_ipv6 = False
    # Set to True if at least one network in IPV6
    global select_ipv6
    select_ipv6 = False
    # Set to True if at least on network is IPVn
    global select_ipvn
    select_ipvn = False
    global security_open
    security_open = True
    global single_system_name
    single_system_name = False
    global sriov_redundancy
    sriov_redundancy = False
    global custom_naming
    custom_naming = True
    global custom_parm_list
    custom_parm_list = []
    global vdu_cp
    vdu_cp = "ribbon.nodes.nfv.VduCp"
    global vip_cp
    vip_cp = "tosca.nodes.nfv.VipCp"
    global vdu_vbs
    vdu_vbs = "ribbon.nodes.nfv.Vdu.VirtualBlockStorage"
    global ha_mode
    ha_mode = False
    global sbc_ha_mode
    sbc_ha_mode = "default"
    global flavor_size_param
    flavor_size_param = "all"
    global ha_param
    ha_param = "all"
    global interface_param
    interface_param = "all"
    global personality_param
    personality_param = "all"
    global insert_userdata
    insert_userdata = ""
    global insert_userdata_topology
    insert_userdata_topology = ""
    global ip_vers_mgt0_var
    ip_vers_mgt0_var = ""
    global ip_vers_ha0_var
    ip_vers_ha0_var = ""
    global ip_vers_pkt0_var
    ip_vers_pkt0_var = ""
    global ip_vers_pkt1_var
    ip_vers_pkt1_var = ""
    global oam_ip_vers_pkt_var
    oam_ip_vers_pkt_var = ""
    global mgt_num_rip
    mgt_num_rip = 2
    global mgt_num_ems_rip
    mgt_num_ems_rip = 2
    global mgt_rules
    mgt_rules = ""
    global ha_rules
    ha_rules = ""
    global separate_oam_mgt_rules
    separate_oam_mgt_rules = False
    global separate_oam_ha_rules
    separate_oam_ha_rules = False
    global pkt_rules
    pkt_rules = ""
    global pkt_alt_ips
    pkt_alt_ips = []
    global cinder_log_device
    cinder_log_device = ""
    global cinder_boot_attributes
    cinder_boot_attributes = '"bootable":"true"'
    global cinder_log_attributes
    cinder_log_attributes = '"bootable":"false", "retain":"true"'
    global passwd_set
    passwd_set = False
    global passwd_admin
    passwd_admin = "$6$rounds=4096$CAcJbPuoC$bZACUTw7lLUAwu6JjuKd88WvHsW5jI4yp0QTPy81kka5JNRIKtewukrilqmjG9l5zM9yNCq88ikcVDVQRwJpD0"
    global passwd_linuxadmin
    passwd_linuxadmin = "$6$rounds=4096$MJXyCUlSn7eQkaDq$Qtz5TODzQJyEWBcuIiLdDkSXro76Cmt/2wINiho0KjFw7MUS6HXfKAczhsTX8h3/VaIbY3wqLUuMGVhfYAkMO0"
    global mop_input_dir
    mop_input_dir = ""
    # The default number of oam instances is zero (i.e., no OAM VMs)
    # It might be overridden to be 1(simplex) or 2(1+1 HA)
    global num_oam_instances
    num_oam_instances = "0"
    # The default number of instances in an n:1 config is 5 (for 4:1)
    # It might be overridden to be 3 (for a small test config).
    global num_instances
    num_instances = "5"
    global private_key_file
    global certificate_file
    global sbc_public_key
    global userids
    global affinity
    global num_sbc_rg
    num_sbc_rg = 1
    global mgt_io_type
    mgt_io_type = "default"
    global mgt_vip
    mgt_vip = False
    global ha_io_type
    ha_io_type = "default"
    global pkt_oam_io_type
    pkt_oam_io_type = "default"
    global pkt_vip
    pkt_vip = False
    global version_suffix
    version_suffix = ""
    global vcpu
    vcpu = -1
    global ram
    ram = -1
    global disk
    disk = -1
    global oam_vcpu
    oam_vcpu = -1
    global oam_ram
    oam_ram = -1
    global oam_disk
    oam_disk = -1
    global cinder_oam_disk
    cinder_oam_disk = -1
    global cinder_sbc_disk
    cinder_sbc_disk = -1
    global cinder_log_disk
    cinder_log_disk = -1
    global sbc_tar_file
    sbc_tar_file = "omit"
    global timeout_shell
    timeout_shell = -1
    global image_name
    image_name = ""
    global zone
    zone = "omit"
    global oam_zone
    # Default oam_zone of "zone" means match zone variable (omit, by default)
    oam_zone = "zone"
    global scaling
    scaling = False
    global max_sbc_rg
    max_sbc_rg = 8
    global redund_ratio
    redund_ratio = ""
    global affinity_across_rgs
    affinity_across_rgs = 'false'
    global dns_config_dic
    dns_config_dic = []
    global old_vnfm
    old_vnfm = False
    global vnfd_custom_types_file
    vnfd_custom_types_file = ""
    global tosca_def_v
    tosca_def_v = "tosca_simple_yaml_1_3"
    global template_version
    template_version = "3.5.1"
    global qcow2_hash
    qcow2_hash = "" #to be manually populated if qcow2 is not part of csar package
    global get_attribute_supported
    get_attribute_supported = True
    global SGRemoteIpV4Mgt1 
    SGRemoteIpV4Mgt1 = "0.0.0.0/0"
    global SGRemoteIpV4Mgt2
    SGRemoteIpV4Mgt2 = "0.0.0.0/32"
    global SGRemoteIpV6Mgt1
    SGRemoteIpV6Mgt1 = "::/0"
    global SGRemoteIpV6Mgt2 
    SGRemoteIpV6Mgt2 = "::/128"
    global get_number_oam_instances
    get_number_oam_instances = "0"
    global CpNamePrefix
    CpNamePrefix = ""
    global EMSIP0
    EMSIP0 = ""
    global EMSIP1
    EMSIP1 = ""
    global SGRemoteIpV6Vnfm
    SGRemoteIpV6Vnfm = "::/0"
    global SGRemoteIpV4Vnfm 
    SGRemoteIpV4Vnfm = "0.0.0.0/0"
    global VnfrPort
    VnfrPort = "8099"
    global default_SubPort0_segmentation_id
    default_SubPort0_segmentation_id = 0
    global default_SubPort1_segmentation_id
    default_SubPort1_segmentation_id = 0
    global num_trunk_pkt0
    num_trunk_pkt0 = 0
    global num_trunk_pkt1
    num_trunk_pkt1 = 0

    global warning_msg
    global env_var_to_print
    env_var_to_print = []
    next_arg = ""
    if len(argv) > 1:
        for arg in argv[1:]:
            if next_arg != "":
                if arg.startswith('-'):
                    warning_msg += "Warning: Unusual parameter: " + next_arg + " set to " + arg + "\n"
                if next_arg == "sub_version":
                    sub_version = arg
                elif next_arg == "vnfd":
                    vnfd_template = arg
                elif next_arg == "prod_version":
                    prod_version = arg
                elif next_arg == "private_key_file":
                    private_key_file = os.path.expanduser(arg)
                elif next_arg == "certificate_file":
                    certificate_file = os.path.expanduser(arg)
                elif next_arg == "custom_oam_name":
                    oam_name = expand_custom_name(arg)
                elif next_arg == "custom_sbc_name":
                    sbc_name = expand_custom_name(arg)
                elif next_arg == "custom_oam_server_group_name":
                    oam_server_group = expand_custom_name(arg)
                elif next_arg == "custom_sbc_server_group_name":
                    sbc_server_group = expand_custom_name(arg)
                elif next_arg == "custom_system_name":
                    system_name = arg

                elif next_arg == "mgt0_options":
                    options_list = arg.split(':')
                    for option_var in options_list:
                        if (option_var.lower() == "ipv4" or
                                option_var.lower() == "ipv6" or
                                option_var.lower() == "ipvn"):
                            ip_vers_mgt0_var = option_var.lower()
                            if mgt_rules != "":
                                mgt_rules = ip_vers_mgt0_var
                            if ip_vers_mgt0_var == "ipv6":
                                select_ipv6 = True
                                if omit_ipv6:
                                    warning_msg += "Warning: selected --omit_ipv6 and --mgt0 ipv6\n"
                            if ip_vers_mgt0_var == "ipvn":
                                select_ipvn = True
                                if omit_ipv6:
                                    warning_msg += "Warning: selected --omit_ipv6 and --mgt0 ipvn\n"
                        elif option_var.lower() == "rules":
                            if ip_vers_mgt0_var == "":
                                mgt_rules = "ipvn"
                            else:
                                mgt_rules = ip_vers_mgt0_var
                        elif option_var.lower() == "rulesoam":
                            separate_oam_mgt_rules = True
                        elif (option_var.lower() == "virtio" or
                                option_var.lower() == "sriov" or
                                option_var.lower() == "provider"):
                            mgt_io_type = option_var.lower()
                        elif option_var.lower() == "rip=0":
                            mgt_num_rip = 0
                        elif option_var.lower() == "rip=1":
                            mgt_num_rip = 1
                        elif option_var.lower() == "rip=2":
                            mgt_num_rip = 2
                        elif option_var.lower() == "ems_rip=0":
                            mgt_num_ems_rip = 0
                        elif option_var.lower() == "ems_rip=1":
                            mgt_num_ems_rip = 1
                        elif option_var.lower() == "ems_rip=2":
                            mgt_num_ems_rip = 2
                        else:
                            print("Invalid option for mgt0=" + arg + " variable=" + option_var)
                            usage(argv[0])
                elif next_arg == "ha0_options":
                    options_list = arg.split(':')
                    for option_var in options_list:
                        if (option_var.lower() == "ipv4" or
                                option_var.lower() == "ipv6" or
                                option_var.lower() == "ipvn"):
                            ip_vers_ha0_var = option_var.lower()
                            if ip_vers_ha0_var == "ipv6":
                                select_ipv6 = True
                                if omit_ipv6:
                                    warning_msg += "Warning: selected --omit_ipv6 and --ha0 ipv6\n"
                            if ip_vers_ha0_var == "ipvn":
                                select_ipvn = True
                                if omit_ipv6:
                                    warning_msg += "Warning: selected --omit_ipv6 and --ha0 ipvn\n"
                        elif option_var.lower() == "rules":
                            ha_rules = "ipvn"
                        elif option_var.lower() == "rulesoam":
                            separate_oam_ha_rules = True
                        elif (option_var.lower() == "virtio" or
                                option_var.lower() == "sriov" or
                                option_var.lower() == "provider"):
                            ha_io_type = option_var.lower()
                        else:
                            print("Invalid option for ha0=" + arg + " variable=" + option_var)
                            usage(argv[0])
                elif next_arg == "pkt0_options":
                    options_list = arg.split(':')
                    for option_var in options_list:
                        if (option_var.lower() == "ipv4" or
                                option_var.lower() == "ipv6" or
                                option_var.lower() == "ipvn"):
                            ip_vers_pkt0_var = option_var.lower()
                            if ip_vers_pkt0_var == "ipv6":
                                select_ipv6 = True
                                if omit_ipv6:
                                    warning_msg += "Warning: selected --omit_ipv6 and --pkt0 ipv6\n"
                            if ip_vers_pkt0_var == "ipvn":
                                select_ipvn = True
                                if omit_ipv6:
                                    warning_msg += "Warning: selected --omit_ipv6 and --pkt0 ipvn\n"
                        elif option_var.lower() == "rules":
                            pkt_rules = "ipvn"
                        elif option_var.lower().startswith('alt_'):
                            # Expecting format "ALT_1=4" which means set 1 is a list of 4 ip addresses.
                            alt_option = option_var.lower().split('_')[1]
                            alt_parms = alt_option.split('=')
                            if len(alt_parms) == 2 and alt_parms[0].isdigit() and alt_parms[1].isdigit():
                                pkt_alt_ips.append(AltPktIps('0', alt_parms[0], alt_parms[1]))
                            else:
                                print("Invalid option for pkt0 : " + arg + " len:" + str(len(alt_parms)) + " ap:" + str(alt_parms) + " ap0:" + alt_parms[0] + " ap1:" + alt_parms[1])
                                print("expecting --pkt0 parms:ALT_n=m")
                                usage(argv[0])
                                return False
                        else:
                            print("Invalid option for pkt0 = " + arg)
                            usage(argv[0])
                elif next_arg == "pkt1_options":
                    options_list = arg.split(':')
                    for option_var in options_list:
                        if (option_var.lower() == "ipv4" or
                                option_var.lower() == "ipv6" or
                                option_var.lower() == "ipvn"):
                            ip_vers_pkt1_var = option_var.lower()
                            if ip_vers_pkt1_var == "ipv6":
                                select_ipv6 = True
                                if omit_ipv6:
                                    warning_msg += "Warning: selected --omit_ipv6 and --pkt1 ipv6\n"
                            if ip_vers_pkt1_var == "ipvn":
                                select_ipvn = True
                                if omit_ipv6:
                                    warning_msg += "Warning: selected --omit_ipv6 and --pkt1 ipvn\n"
                        elif option_var.lower() == "rules":
                            pkt_rules = "ipvn"
                        elif option_var.lower().startswith('alt_'):
                            # Expecting format "ALT_1=4" which means set 1 is a list of 4 ip addresses.
                            alt_option = option_var.lower().split('_')[1]
                            alt_parms = alt_option.split('=')
                            if len(alt_parms) == 2 and alt_parms[0].isdigit() and alt_parms[1].isdigit():
                                pkt_alt_ips.append(AltPktIps('1', alt_parms[0], alt_parms[1]))
                            else:
                                print("Invalid option for pkt1 : " + arg + " len:" + str(len(alt_parms)) + " ap:" + str(alt_parms) + " ap0:" + alt_parms[0] + " ap1:" + alt_parms[1])
                                print("expecting --pkt1 parms:ALT_n=m")
                                usage(argv[0])
                                return False
                        else:
                            print("Invalid option for pkt1 = " + arg)
                            usage(argv[0])
                elif next_arg == "oampkt_options":
                    options_list = arg.split(':')
                    for option_var in options_list:
                        if (option_var.lower() == "ipv4" or
                                option_var.lower() == "ipv6" or
                                option_var.lower() == "ipvn"):
                            oam_ip_vers_pkt_var = option_var.lower()
                            if oam_ip_vers_pkt_var == "ipv6":
                                select_ipv6 = True
                                if omit_ipv6:
                                    warning_msg += "Warning: selected --omit_ipv6 and --oampkt ipv6\n"
                            if oam_ip_vers_pkt_var == "ipvn":
                                select_ipvn = True
                                if omit_ipv6:
                                    warning_msg += "Warning: selected --omit_ipv6 and --oampkt ipvn\n"
                        elif option_var.lower() == "rules":
                            pkt_rules = "ipvn"
                        else:
                            print("Invalid value for oampkt = " + arg)
                            usage(argv[0])
                elif next_arg == "sbc_public_key":
                    sbc_public_key_file = os.path.expanduser(arg)
                    if not os.path.exists(sbc_public_key_file):
                        print('File ' + sbc_public_key_file + ' not found.')
                        return False
                    with open(sbc_public_key_file, 'r') as in_file:
                        first_line = in_file.readline().strip()
                        if len(first_line) < 64:
                            print('Key in file ' + sbc_public_key_file + ' not a valid openssh key')
                            return False
                        sbc_public_key = first_line
                elif next_arg == "userid":
                    userid, filename = arg.partition(':')[::2]
                    if filename == '':
                        if userid in userids:
                            userids[userid].append('')
                        else:
                            userids[userid] = ['']
                    else:
                        key_file = os.path.expanduser(filename)
                        if not os.path.exists(key_file):
                            print('File ' + key_file + ' not found.')
                            return False
                        with open(key_file, 'r') as in_file:
                            first_line = in_file.readline().strip()
                            if len(first_line) < 64:
                                print('Key in file ' + key_file + ' not a valid openssh key')
                                return False
                            if userid in userids:
                                userids[userid].append(first_line)
                            else:
                                userids[userid] = [first_line]

                elif next_arg == "num_instances":
                    num_instances = arg
                elif next_arg == "num_oam_instances":
                    num_oam_instances = arg
                elif next_arg == "affinity":
                    affinity = arg
                    if affinity != "AntiAffinity" and affinity != "Affinity" and affinity != "None":
                        print("Invalid value for affinity: " + affinity)
                        return False
                elif next_arg == "qcow2_file":
                    qcow2_file = arg
                    if not os.path.exists(qcow2_file):
                        print('File ' + qcow2_file + ' does not exist.')
                        return False
                    if not os.path.splitext(qcow2_file)[1] == '.qcow2':
                        print('File ' + qcow2_file + ' does not have the required extension .qcow2')
                        return False
                elif next_arg == "mop_dir_param":
                    mop_input_dir = arg
                    if not os.path.isdir(mop_input_dir):
                        # If it's not a directory, check if it's a tar file
                        if not (os.path.isfile(mop_input_dir) and tarfile.is_tarfile(mop_input_dir)):
                            print('Directory ' + mop_input_dir + ' does not exist')
                            return False
                elif next_arg == "sbc_tar_param":
                    sbc_tar_file = arg
                    if not os.path.isfile(sbc_tar_file) and not sbc_tar_file == "omit" and not sbc_tar_file == "search":
                        print('Directory ' + sbc_tar_file + ' does not exist')
                        return False
                elif next_arg == "personality_param":
                    if arg == "isbc" or arg == "msbc" or arg == "ssbc" or arg == "tsbc" or arg == "mrfp" or arg == "slb":
                        personality_param = arg
                    else:
                        print("Invalid value for personality = " + arg)
                        usage(argv[0])
                        return False
                elif next_arg == "insert_userdata":
                    insert_list = arg.split(':')
                    for insert_var in insert_list:
                        insert_val = insert_var.split('=')
                        if len(insert_val) == 2:
                            insert_userdata = (insert_userdata +
                                               '      "INSERT:' + insert_val[0] + '":\n' +
                                               '          type: string\n' +
                                               '          description: ' + insert_val[0] + '\n' +
                                               '          default: "' + insert_val[1] + '"\n')
                            insert_userdata_topology = insert_userdata_topology + '#space_delim#"INSERT:' + insert_val[0] + '": ' + insert_val[1] + '\n'
                        else:
                            print('Invalid value. Substring ' + insert_var + ' must contain one "=" character')
                            usage(argv[0])
                            return False
                elif next_arg == "dos_support_sec_pkt_ports":
                    if arg.lower() == "true":
                        dsspp = "True"
                    elif arg.lower() == "false":
                        dsspp = "False"
                    elif arg.lower() == "default":
                        dsspp = "default"
                    else:
                        print('Invalid value for dos_support_sec_pkt_ports:' + arg)
                        usage(argv[0])
                        return False

                elif next_arg == "cinder_disktype":
                    cinder_boot_attributes = cinder_boot_attributes + ', "disktype":' + '"' + arg + '"'
                    cinder_log_attributes = cinder_log_attributes + ', "disktype":' + '"' + arg + '"'
                elif next_arg == "cinder_log_device":
                    cinder_log_device = arg
                elif next_arg == "passwd_admin":
                    # A base64 encoded SHA512 should be much longer than 64 chars.
                    # This is just a very simple test to make sure they didn't use a non-encrypted passwd.
                    if len(arg) > 64:
                        passwd_admin = arg
                        passwd_set = True
                    else:
                        print("Supplied admin passwd is too short. Please use an encrypted password with SHA512.")
                        usage(argv[0])
                        return False
                elif next_arg == "passwd_linuxadmin":
                    # A base64 encoded SHA512 should be much longer than 64 chars.
                    # This is just a very simple test to make sure they didn't use a non-encrypted passwd.
                    if len(arg) > 64:
                        passwd_linuxadmin = arg
                        passwd_set = True
                    else:
                        print("Supplied linuxadmin passwd is too short. Please use an encrypted password with SHA512.")
                        usage(argv[0])
                        return False
                elif next_arg == "placement_sbc_name":
                    placement_sbc_name = expand_custom_name(arg)
                elif next_arg == "placement_oam_name":
                    placement_oam_name = expand_custom_name(arg)
                elif next_arg == "version_suffix":
                    version_suffix = arg
                elif next_arg == "flavor_size_param":
                    if arg == "dev" or arg == "std":
                        flavor_size_param = arg + "_size"
                    else:
                        print("Invalid value for flavor_size_param = " + arg)
                        usage(argv[0])
                        return False
                elif next_arg == "ha_param":
                    if arg == "simplex":
                        ha_param = arg
                        ha_mode = True
                        sbc_ha_mode = "Nto1"
                    elif arg == "n_1_ha":
                        ha_param = arg
                    elif arg == "ha":
                        ha_param = "active_standby"
                    else:
                        print("Invalid value for ha_param = " + arg)
                        usage(argv[0])
                        return False
                elif next_arg == "interface_param":
                    if arg == "virtio" or arg == "sriov" or arg == "provider":
                        interface_param = arg
                    else:
                        print("Invalid value for interface_param = " + arg)
                        usage(argv[0])
                        return False
                elif next_arg == "mgt_io_type":
                    if arg == "virtio" or arg == "sriov" or arg == "provider":
                        mgt_io_type = arg
                    else:
                        print("Invalid value for mgt_io_type = " + arg)
                        usage(argv[0])
                        return False
                elif next_arg == "ha_io_type":
                    if arg == "virtio" or arg == "sriov" or arg == "provider":
                        ha_io_type = arg
                    else:
                        print("Invalid value for ha_io_type = " + arg)
                        usage(argv[0])
                        return False
                elif next_arg == "pkt_oam_io_type":
                    if arg == "virtio" or arg == "sriov" or arg == "provider" or arg == "disabled":
                        pkt_oam_io_type = arg
                    else:
                        print("Invalid value for pkt_oam_io_type = " + arg)
                        usage(argv[0])
                        return False
                elif next_arg == "num_sbc_rg":
                    if arg.isdigit() and (1 <= int(arg)) and (int(arg) <= 8):
                        num_sbc_rg = int(arg)
                    else:
                        print("Invalid value for num_sbc_rg = " + arg)
                        usage(argv[0])
                        return False
                elif next_arg == "num_pkt0_trunk":
                    if arg.isdigit() and (1 <= int(arg)) and (int(arg) <= 8):
                        num_trunk_pkt0 = int(arg)
                    else:
                        print("Invalid value for num_pkt0_trunk = " + arg)
                        usage(argv[0])
                        return False
                elif next_arg == "num_pkt1_trunk":
                    if arg.isdigit() and (1 <= int(arg)) and (int(arg) <= 8):
                        num_trunk_pkt1 = int(arg)
                    else:
                        print("Invalid value for num_pkt1_trunk = " + arg)
                        usage(argv[0])
                        return False

                elif next_arg == "vcpu":
                    vcpu = int(arg)
                elif next_arg == "ram":
                    ram = int(arg)
                elif next_arg == "disk":
                    disk = int(arg)
                elif next_arg == "oam_vcpu":
                    oam_vcpu = int(arg)
                elif next_arg == "oam_ram":
                    oam_ram = int(arg)
                elif next_arg == "oam_disk":
                    oam_disk = int(arg)
                elif next_arg == "cinder_oam_disk":
                    cinder_oam_disk = int(arg)
                elif next_arg == "cinder_sbc_disk":
                    cinder_sbc_disk = int(arg)
                elif next_arg == "cinder_log_disk":
                    cinder_log_disk = int(arg)
                elif next_arg == "timeout_shell":
                    timeout_shell = int(arg)
                elif next_arg == "image_name":
                    image_name = arg
                elif next_arg == "zone":
                    zone = arg
                elif next_arg == "oam_zone":
                    oam_zone = arg
                elif next_arg == "max_sbc_rg":
                    max_sbc_rg = int(arg)
                elif next_arg == "affinity_across_rgs":
                    if arg == 'true' or arg == 'false':
                        affinity_across_rgs = arg
                    else:
                        print("Inalid option (%s) for --affinity_across_rgs. Should be true or false")
                        return False
                elif next_arg == "redund_ratio":
                    ratio = arg.split(':')
                    if len(ratio) != 2:
                        print("redundancy_ratio (%s) in incorrect format" % (arg))
                        return False
                    if int(ratio[0]) < 1 or int(ratio[0]) > 7:
                        print("redundancy_ratio (%s) has an invalid number of actives" % (arg))
                        return False
                    if int(ratio[1]) != 1 and int(ratio[1]) != 0:
                        print("redundancy_ratio (%s) has an invalid number of standby (1 or 0)" % (arg))
                        return False
                    redund_ratio = arg
                elif next_arg == "dns_file":
                    if not os.path.isfile(arg):
                        print("Supplied DNS configuration file (%s) not found!" % (arg))
                        return False
                    try:
                        with open(arg, 'r') as f:
                            dns_config_dic = json.load(f)
                    except Exception as e:
                        print("Failed to load DNS config JSON. Exception: %s" % (str(e)))
                        return False
                elif next_arg == "types_file":
                    if not os.path.isfile(arg):
                        print("Supplied types file (%s) not found!" % (arg))
                        return False
                    vnfd_custom_types_file = arg
                elif next_arg == "old_vnfm":
                    if arg.lower() == "true":
                        old_vnfm = True
                        tosca_def_v = "tosca_simple_yaml_1_2"
                    elif arg.lower() == "false":
                        old_vnfm = False
                        tosca_def_v = "tosca_simple_yaml_1_3"
                    else:
                        print("Invalid argument for --old_vnfm: " + arg)
                        return False
                else:
                    print("Invalid argument: " + arg)
                    if next_arg != "":
                        print("Expected argument for " + next_arg)
                    usage(argv[0])
                    return False
                next_arg = ""
            elif arg.startswith('--'):
                if arg == "--num_instances":
                    next_arg = "num_instances"
                elif arg == "--num_oam_instances":
                    next_arg = "num_oam_instances"
                elif arg == "--affinity":
                    next_arg = "affinity"
                elif arg == "--private_key_file":
                    next_arg = "private_key_file"
                elif arg == "--certificate_file":
                    next_arg = "certificate_file"
                elif arg == "--custom_oam_name":
                    next_arg = "custom_oam_name"
                elif arg == "--custom_sbc_name":
                    next_arg = "custom_sbc_name"
                elif arg == "--custom_oam_server_group_name":
                    next_arg = "custom_oam_server_group_name"
                elif arg == "--custom_sbc_server_group_name":
                    next_arg = "custom_sbc_server_group_name"
                elif arg == "--custom_system_name":
                    next_arg = "custom_system_name"
                # Temporary, no longer used
                elif arg == "--custom_param":
                    print("Use environment variable CUSTOM_SBC_PARMS instead")
                    usage(argv[0])
                    return False
                # Temporary, no longer used
                elif arg == "--custom_param_set":
                    print("Use environment variable CUSTOM_SBC_PARMS instead")
                    usage(argv[0])
                    return False
                elif arg == "--custom_vmname_dash":
                    dash_index = True
                elif arg == "--mgt0":
                    next_arg = "mgt0_options"
                elif arg == "--ha0":
                    next_arg = "ha0_options"
                elif arg == "--pkt0":
                    next_arg = "pkt0_options"
                elif arg == "--pkt1":
                    next_arg = "pkt1_options"
                elif arg == "--oampkt":
                    next_arg = "oampkt_options"
                elif arg == "--sbc_public_key":
                    next_arg = "sbc_public_key"
                elif arg == "--userid":
                    next_arg = "userid"
                elif arg == "--mgt_io_type":
                    next_arg = "mgt_io_type"
                elif arg == "--mgt_vip":
                    mgt_vip = True
                elif arg == "--ha_io_type":
                    next_arg = "ha_io_type"
                elif arg == "--pkt_oam_io_type":
                    next_arg = "pkt_oam_io_type"
                elif arg == "--pkt_vip":
                    pkt_vip = True
                elif arg == "--placement_sbc_name":
                    next_arg = "placement_sbc_name"
                elif arg == "--placement_oam_name":
                    next_arg = "placement_oam_name"
                elif arg == "--version_suffix":
                    next_arg = "version_suffix"
                elif arg == "--cinder_boot":
                    cinder_boot = True
                elif arg == "--cinder_log":
                    cinder_log = True
                elif arg == "--designer_debug":
                    designer_debug = True
                elif arg == "--usage":
                    usage(argv[0], False)
                elif arg == "--help":
                    usage(argv[0], True)
                    return False
                elif arg == "--help_custom":
                    usage_custom()
                    return False
                elif arg == "--ipv6":
                    # The ha network is always ipv4 by default, the others are both set by this one option.
                    default_mgt_ip_vers = "ipv6"
                    default_ha_ip_vers = "ipv4"
                    default_pkt_ip_vers = "ipv6"
                    select_ipv6 = True
                    if omit_ipv6:
                        warning_msg += "Warning: selected --ipv6 and --omit_ipv6"
                elif arg == "--omit_ipv6":
                    omit_ipv6 = True
                    if select_ipv6:
                        warning_msg += "Warning: selected --omit_ipv6 and an ipv6 network.\n"
                    if select_ipvn:
                        warning_msg += "Warning: selected --omit_ipv6 and an ipvn network.\n"
                elif arg == "--security_restrict":
                    security_open = False
                elif arg == "--single_system_name":
                    single_system_name = True
                elif arg == "--ha_mode":
                    ha_mode = True
                elif arg == "--no_custom_naming":
                    custom_naming = False
                    vdu_cp = "tosca.nodes.nfv.VduCp"
                    vdu_vbs = "tosca.nodes.nfv.Vdu.VirtualBlockStorage"
                # Temporary, while transitioning to SOL1
                elif arg == "--mop_dir":
                    next_arg = "mop_dir_param"
                elif arg == "--sbc_tar":
                    next_arg = "sbc_tar_param"
                elif arg == "--personality":
                    next_arg = "personality_param"
                elif arg == "--insert_userdata":
                    next_arg = "insert_userdata"
                elif arg == "--dos_support_sec_pkt_ports":
                    next_arg = "dos_support_sec_pkt_ports"
                elif arg == "--cinder_disktype":
                    next_arg = "cinder_disktype"
                elif arg == "--cinder_log_device":
                    next_arg = "cinder_log_device"
                elif arg == "--passwd_admin":
                    next_arg = "passwd_admin"
                elif arg == "--passwd_linuxadmin":
                    next_arg = "passwd_linuxadmin"
                elif arg == "--passwd_default":
                    passwd_set = True
                elif arg == "--sriov_redundancy":
                    sriov_redundancy = True
                # Temporary, for debugging
                elif arg == "--prov_mgmt_virtio":
                    mgt_io_type = "virtio"
                elif arg == "--no_dhcp_ha":
                    no_dhcp_ha = True
                elif arg == "--num_sbc_rg":
                    next_arg = "num_sbc_rg"
                elif arg == "--num_pkt0_trunk":
                    next_arg = "num_pkt0_trunk"
                elif arg == "--num_pkt1_trunk":
                    next_arg = "num_pkt1_trunk"
                # Ignore the unusual case for flavors, accept any match in lowercase.
                elif arg.lower() == "--vcpu":
                    next_arg = "vcpu"
                elif arg.lower() == "--ram":
                    next_arg = "ram"
                elif arg.lower() == "--disk":
                    next_arg = "disk"
                elif arg.lower() == "--oam_vcpu":
                    next_arg = "oam_vcpu"
                elif arg.lower() == "--oam_ram":
                    next_arg = "oam_ram"
                elif arg.lower() == "--oam_disk":
                    next_arg = "oam_disk"
                elif arg.lower() == "--cinder_oam_disk":
                    next_arg = "cinder_oam_disk"
                elif arg.lower() == "--cinder_sbc_disk":
                    next_arg = "cinder_sbc_disk"
                elif arg.lower() == "--cinder_log_disk":
                    next_arg = "cinder_log_disk"
                elif arg == "--timeout":
                    next_arg = "timeout_shell"

                elif arg == "--image_name":
                    next_arg = "image_name"
                elif arg == "--zone":
                    next_arg = "zone"
                elif arg == "--oam_zone":
                    next_arg = "oam_zone"
                
                elif arg == "--enable_scaling":
                    scaling = True
                elif arg == "--max_sbc_rg":
                    next_arg = "max_sbc_rg"
                elif arg == "--redundancy_ratio":
                    next_arg = "redund_ratio"
                elif arg == "--affinity_across_rgs":
                    next_arg = "affinity_across_rgs"
                elif arg == "--dns_file":
                    next_arg = "dns_file"
                elif arg == "--old_vnfm":
                    next_arg = "old_vnfm"
                elif arg == "--no_get_attrib":
                    get_attribute_supported = False

                else:
                    print("unexpected option:" + arg)
                    usage(argv[0])
                    return False
            elif arg.startswith('-'):
                if arg == "-s":
                    next_arg = "sub_version"
                elif arg == "-v":
                    next_arg = "vnfd"
                elif arg == "-p":
                    next_arg = "prod_version"
                elif arg == "-q":
                    next_arg = "qcow2_file"
                elif arg == "-f":
                    next_arg = "flavor_size_param"
                elif arg == "-h":
                    next_arg = "ha_param"
                elif arg == "-i":
                    next_arg = "interface_param"
                elif arg == "-t":
                    next_arg = "types_file"
                else:
                    print("Unexpected option: " + arg)
                    usage(argv[0])
                    return False
            else:
                print("Invalid argument: " + arg)
                usage(argv[0])
                return False

    if placement_sbc_name == placement_oam_name:
        print("Invalid options: placement_sbc_name and placement_oam_name both set to " + placement_sbc_name)
        return False

    if scaling:
        if int(num_oam_instances) == 0:
            print("Invalid option: scaling requires OAM instances")
            return False
        if num_sbc_rg != 1:
            print("Invalid option: --enable_scaling cannot be used with --num_sbc_rg")
            return False
        if ha_mode and sbc_ha_mode == "1to1":
            print("Invalid option: scaling does not work with ha_mode 1to1")
            return False

    return True


def add_upgrade_mop(scripts_dir):
    # Test if we have an input mop.
    global sbc_tar_file
    global mop_input_dir
    global qcow2_hash
    if sbc_tar_file == "omit":
        print('Omitting SBC tar file, which is required for automated upgrades')
        return

    # Upgrade MOP for SBC contains following components:
    #   1) SBC tar ball which contains various utilities/scripts/LSWU matrix etc
    #   2) Upgrade MOP common directory structure
    #   3) SBC specific list of commands to be populated in various directories
    # all of these needs to copied to Scripts and mop directories before making CSAR

    global vnfd_sol001_template
    template_dir = os.path.dirname(vnfd_sol001_template)

    # 1) copy SBC tar ball content
    try:
        # First look in the same directory as the template:
        if sbc_tar_file == "" or sbc_tar_file == "search":
            sbc_tar_list = glob.glob(os.path.join(template_dir, 'sbc-V*.tar.gz'))
            if sbc_tar_list and sbc_tar_list[0] and os.path.exists(sbc_tar_list[0]):
                sbc_tar_file = sbc_tar_list[0]

            if sbc_tar_file == "" or sbc_tar_file == "search":
                if "ORCA_ROOT" in os.environ:
                    sbc_tar_list = glob.glob(os.path.join(os.environ['ORCA_ROOT'], 'rel', 'sbc-V*.tar.gz'))
                if sbc_tar_list and sbc_tar_list[0] and os.path.exists(sbc_tar_list[0]):
                    sbc_tar_file = sbc_tar_list[0]

        if sbc_tar_file == "" or sbc_tar_file == "search":
            print('Warning: SBC tar file not found. Csar will not include SBC tar ball content (for automated upgrades).')
        else:
            print("Adding tarfiles from: " + sbc_tar_file)
            tar = tarfile.open(sbc_tar_file, 'r:gz', errorlevel=1)
            for f in tar:
                # Don't use tar.extractall, because of possible permission problems.
                # tar.extractall(path=scripts_dir)
                try:
                    tar.extract(f, scripts_dir)
                except IOError as e:
                    os.remove(f.name)
                    tar.extract(f, scripts_dir)

                if not (f.name.find("qcow2.sha256")) == -1:
                    with open(scripts_dir + "/" + f.name, 'r') as in_qcow2_file:
                        qcow2_hash = in_qcow2_file.readline().split()[0]

    except Exception as e:
        print(traceback.format_exc())
        print('Error copying SBC tar ball content, continuing...')

    # If Scripts/mop already exists in the mop output directory, ignore mop input directory.
    # Assume it already contains the correct files, and the correct preUpgradeChecks.list.
    mop_output_dir = os.path.join(scripts_dir, 'mop')
    if os.path.exists(mop_output_dir):
        if not mop_input_dir == "":
            print('Using pre-existing mop directory, ignoring ' + mop_input_dir)
    else:
        os.mkdir(mop_output_dir)

        # 2) copy MOP directory structure
        print('Building upgrade MOP...')
        if mop_input_dir == "":
            # Otherwise use COMMON_ROOT (if we're in a perforce context):
            if "COMMON_ROOT" in os.environ:
                mop_input_dir = os.path.join(os.environ['COMMON_ROOT'], 'VNFR', 'lib', 'mop')
                if not os.path.isdir(mop_input_dir):
                    print('Directory ' + mop_input_dir + ' does not exist')
        if mop_input_dir == "":
            print('Directory mop_dir not specified, upgrade scripts omitted.')
            return

        try:
            if os.path.isdir(mop_input_dir):
                copy_tree(mop_input_dir, mop_output_dir)
            elif os.path.isfile(mop_input_dir) and tarfile.is_tarfile(mop_input_dir):
                mode = "r"
                if mop_input_dir.endswith("gz"):
                    mode = "r:gz"
                tf = tarfile.open(mop_input_dir, mode)
                tf.extractall(path=mop_output_dir)
                tf.close()

        except Exception as e:
            print(traceback.format_exc())
            print('Error copying MOP directory structure, continuing...')

        # 3) copy SBC specific list
        try:
            # Look for a Pre Upgrade Checks dir in template_dir or ORCA_ROOT/install (if it exists) or mop_input_dir.
            puc_input_dir = ""
            if os.path.exists(os.path.join(template_dir, 'preUpgradeChecks.list')):
                puc_input_dir = template_dir
            elif "ORCA_ROOT" in os.environ:
                puc_input_dir = os.path.join(os.environ['ORCA_ROOT'], 'install')
                if not os.path.isdir(puc_input_dir):
                    puc_input_dir = ""
                elif not os.path.exists(os.path.join(puc_input_dir, 'preUpgradeChecks.list')):
                    puc_input_dir = ""
            if puc_input_dir == "":
                if os.path.exists(os.path.join(mop_input_dir, 'preUpgradeChecks', 'preUpgradeChecks.list')):
                    puc_input_dir = os.path.join(mop_input_dir, 'preUpgradeChecks')
            if puc_input_dir == "":
                print('Warning: Could not find preUpgradeChecks in ' + mop_input_dir)
            else:
                print("Adding pre-upgrade-checks from: " + os.path.join(puc_input_dir, "preUpgradeChecks.list"))
                puc_output_dir = os.path.join(mop_output_dir, 'preUpgradeChecks')
                if not os.path.exists(puc_output_dir):
                    os.mkdir(puc_output_dir)
                # Change permissions:
                for dirpath, dirnames, filenames in os.walk(puc_output_dir):
                    os.chmod(dirpath, 0o750)
                    for f in filenames:
                        os.chmod(os.path.join(dirpath, f), 0o750)

                shutil.copy2(os.path.join(puc_input_dir, 'preUpgradeChecks.list'),
                             os.path.join(puc_output_dir, 'preUpgradeChecks.list'))
                # os.system('cp -f ' + puc_input_dir + '/preUpgradeChecks.list ' + puc_output_dir + '/preUpgradeChecks.list')
        except Exception as e:
            print(traceback.format_exc())
            print('Error copying SBC MOP lists, continuing...')

    # 4) set execute permissions for everything under scripts
    try:
        # On Windows (ie "nt"), we can't set the expected permissions.
        if os.name == 'nt':
            print("Warning: Permissions may not be set correctly in Scripts dir")
        # Change permissions:
        for dirpath, dirnames, filenames in os.walk(scripts_dir):
            os.chmod(dirpath, 0o750)
            for f in filenames:
                os.chmod(os.path.join(dirpath, f), 0o750)
    except Exception as e:
        print(traceback.format_exc())
        print('Error changing permissions of scripts dir, continuing...')


def hash_file_sha256(file_to_hash):
    # Since this function could be used for very large files like the qcow2
    # image, it's important to only read small blocks of it into memory at a
    # time. Use 64k blocks as a reasonable size to read into memory.
    BLOCKSIZE = 65536
    hasher = hashlib.sha256()
    with open(file_to_hash, 'rb') as afile:
        buf = afile.read(BLOCKSIZE)
        while len(buf) > 0:
            hasher.update(buf)
            buf = afile.read(BLOCKSIZE)
    return hasher.hexdigest()


def sbc_rg_prefix(sbc_rg_num):
    # If the redundancy group prefix needs to be changed, modify the return of this function
    return "rg" + str(sbc_rg_num)

def get_instantiation_instances(instantiation_level, ha_model, redund_actives):
    """Helper function to work out number of instances for
       an instantiaion level, depending on HA type"""
    instance_num = 0
    if ha_model == "simplex":
        instance_num = instantiation_level + 1
    elif ha_model == "active_standby":
        instance_num = (instantiation_level + 1) * 2
    elif ha_model == "n_1_ha":
        instance_num = (redund_actives + 1) * instantiation_level
    return instance_num
      
def get_aspect_scale_level(instantiation_level, ha_model, redund_actives):
    """Helper function to work out the apect scale level for
       an instantiaion level, depending on HA type"""
    scale_level = 0
    if ha_model == "n_1_ha":
        scale_level = (instantiation_level * redund_actives) - 1
    else:
        scale_level = instantiation_level
    return scale_level

def xsbc_create_csar(prod_vers, sub_vers, vnfd_file, customtypes_file, csar_dir, vt, ha_model, io_type, flavor_size, num_oam):
    # The subversion parameter is an optional parameter to create a CSAR file.
    # If it's not set, we tag the VNFD file to "latest", with no subversion set.
    if sub_vers == "":
        full_vers = prod_vers + "_latest"
    else:
        full_vers = prod_vers + sub_vers

    oam_vm = False
    oam_ha_model = "active_standby"
    if int(num_oam) >= 1:
        oam_vm = True
        if int(num_oam) == 1:
            oam_ha_model = "simplex"

    global version_tag
    # Convert 8.00.00 or 10.00.00 into V08.00.00 or V10.00.00.
    if not full_vers[1] == '.':
        version_tag = "V"+full_vers
    else:
        version_tag = "V0"+full_vers

    global ip_vers_mgt0_var
    global ip_vers_ha0_var
    global ip_vers_pkt0_var
    global ip_vers_pkt1_var
    global oam_ip_vers_pkt_var
    global image_name
    global qcow2_hash
    global SGRemoteIpV4Mgt1
    global SGRemoteIpV4Mgt2
    global SGRemoteIpV6Mgt1
    global SGRemoteIpV6Mgt2
    global get_number_oam_instances
    global CpNamePrefix
    global EMSIP0
    global EMSIP1
    global SGRemoteIpV6Vnfm
    global SGRemoteIpV4Vnfm
    global default_SubPort0_segmentation_id
    global default_SubPort1_segmentation_id
    global VnfrPort
    global zone
    global oam_zone
    global cinder_log_device
    global warning_msg
    if image_name == '':
        image_name = "sbc-" + version_tag
    if oam_zone == "zone":
        oam_zone = zone
    version_digits = prod_vers.split('.')
    maj_version = version_digits[0].lstrip('0') or '0'
    min_version = version_digits[1].lstrip('0') or '0'
    compact_version = maj_version + '.' + min_version

    if not os.path.exists(vnfd_file):
        print("ERROR: " + vnfd_file + " file does not exist.")
        return

    current_dir = os.getcwd()

    n_active_instances = str(int(num_instances) - 1)
    ha_brief = ha_model
    num_pkt0_vlan = []
    num_pkt1_vlan = []

    num_instances_var = []
    if ha_model == "simplex":
        ha_desc = "Simplex"
        num_instances_sol001 = "1"
        num_instances_var.append("1")
        # packet_if = "external_access: 'true'"
    elif ha_model == "active_standby":
        ha_desc = "1:1 HA"
        num_instances_sol001 = "2"
        if num_sbc_rg > 1:
            for i in range(first_rg_index, num_sbc_rg+first_rg_index):
                num_instances_var.append("2")
        else:
            num_instances_var.append("2")
        ha_brief = "ha"
    elif ha_model == "n_1_ha":
        ha_desc = "n:1 HA"
        num_instances_sol001 = num_instances
        if num_sbc_rg > 1:
            for i in range(first_rg_index, num_sbc_rg+first_rg_index):
                if get_attribute_supported:
                    num_instances_var.append("{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, number_" + sbc_rg_prefix(i) + "_" + "sbc_instances ] }")
                else:
                    num_instances_var.append( str(num_instances_sol001) )
        else:
            if get_attribute_supported:
                num_instances_var.append("{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, number_sbc_instances ] }")
            else:
                num_instances_var.append( str(num_instances_sol001) )
        ha_brief = "n_1"
    else:
        print("Unknown value for ha_model: " + ha_model)
        return

    if num_sbc_rg > 1:
        ha_brief = ha_brief + "_rg" + str(num_sbc_rg)

    if flavor_size == "dev_size":
        flavor_brief = ""
    elif flavor_size == "extra_large":
        flavor_brief = "xlarge_"
    else:
        flavor_brief = flavor_size + '_'

    custom_sizing = not (vcpu == -1 and ram == -1 and disk == -1)
    if custom_sizing:
        flavor_brief = "custom_"
    name_suffix = '_'

    if cinder_boot:
        name_suffix = name_suffix + "cinderboot_"

    if cinder_log:
        name_suffix = name_suffix + "cinderlog_"

    if affinity == "None":
        name_suffix = name_suffix + "noAffinity_"
    if affinity == "Affinity":
        name_suffix = name_suffix + "Affinity_"

    if scaling:
        name_suffix = name_suffix + "scaling_"
    if dns_config_dic != []:
        name_suffix = name_suffix + "dns_"

    vt_name = vt + '_' + ha_brief + '_' + flavor_brief + io_type + name_suffix + compact_version + version_suffix

    output_dir = os.path.join(csar_dir, vt_name)
    def_dir = os.path.join(output_dir, "Definitions")
    files_dir = os.path.join(output_dir, "Files")
    tosca_dir = os.path.join(output_dir, "TOSCA-Metadata")
    scripts_dir = os.path.join(output_dir, "Scripts")
    if not os.path.exists(output_dir):
        os.mkdir(output_dir)
    else:
        print("Warning: Directory " + output_dir + " already exists. Adding to existing content, not recreating it.")
    if not os.path.exists(def_dir):
        os.mkdir(def_dir)
    if not os.path.exists(files_dir):
        os.mkdir(files_dir)
    if not os.path.exists(tosca_dir):
        os.mkdir(tosca_dir)
    if os.path.exists(scripts_dir) and sbc_tar_file == "omit" and len(os.listdir(scripts_dir)) != 0:
        # If they requested that we "omit" the sbc_tar_file, and there is already a
        # "scripts_dir", recursively remove it, so we can create it fresh.
        print("Removing previously existing directory " + scripts_dir + " to make sure sbc_tar_file is omitted")
        shutil.rmtree(scripts_dir)
    if not os.path.exists(scripts_dir):
        os.mkdir(scripts_dir)
    # No longer used
    # packet_if = ""

    short_personality = vt
    if vt == "isbc":
        long_personality = "Integrated"
    elif vt == "msbc":
        long_personality = "Media"
    elif vt == "ssbc":
        long_personality = "Signaling"
    elif vt == "tsbc":
        long_personality = "Transcoding"
        # Normally the personality is the same as the vnftype.
        # But the Transcoding SBC is really a special type of Media SBC.
        short_personality = "msbc"
    elif vt == "mrfp":
        long_personality = "MediaResourceFunctionProcessor"
    elif vt == "slb":
        long_personality = "Load Balancing"
    else:
        long_personality = "Unknown"
        # packet_if = "external_access: 'true'"

    # If the name ends with a digit, we need a dash between it and the vm index.
    global dash_index
    if (oam_name[-1].isdigit() or sbc_name[-1].isdigit() or
            oam_name.endswith("${sbc_rg_index}") or
            sbc_name.endswith("${sbc_rg_index}")):
        dash_index = True

    # The custom names for VMs and interfaces may be long, so read them from environment variables.
    # If the environment variable isn't found, the specified default is used.
    if dash_index:
        optional_dash = "-"
    else:
        optional_dash = ""
    custom_vm_name = {
        "${env_custom_sbc_vmname_pattern}": CustomEnv("CUSTOM_SBC_VMNAME_PATTERN", "%orchname%-%basename%" + optional_dash + "%index%"),
        "${env_custom_sbc_vmgt_pattern}": CustomEnv("CUSTOM_SBC_VMGT_PATTERN", "%vmname%-vmgt0"),
        "${env_custom_sbc_mgt_pattern}": CustomEnv("CUSTOM_SBC_MGT_PATTERN", "%vmname%-mgt0"),
        "${env_custom_sbc_ha_pattern}": CustomEnv("CUSTOM_SBC_HA_PATTERN", "%vmname%-ha0"),
        "${env_custom_sbc_vpkt0_pattern}": CustomEnv("CUSTOM_SBC_VPKT0_PATTERN", "%vmname%-vpkt0"),
        "${env_custom_sbc_pkt0_pattern}": CustomEnv("CUSTOM_SBC_PKT0_PATTERN", "%vmname%-pkt0"),
        "${env_custom_sbc_vpkt1_pattern}": CustomEnv("CUSTOM_SBC_VPKT1_PATTERN", "%vmname%-vpkt1"),
        "${env_custom_sbc_pkt1_pattern}": CustomEnv("CUSTOM_SBC_PKT1_PATTERN", "%vmname%-pkt1"),
        "${env_custom_sbc_pkt0S1_pattern}": CustomEnv("CUSTOM_SBC_PKT0S1_PATTERN", "%vmname%-pkt0S1"),
        "${env_custom_sbc_pkt1S1_pattern}": CustomEnv("CUSTOM_SBC_PKT1S1_PATTERN", "%vmname%-pkt1S1"),
        "${env_custom_oam_vmname_pattern}": CustomEnv("CUSTOM_OAM_VMNAME_PATTERN", "%orchname%-%basename%" + optional_dash + "%index%"),
        "${env_custom_oam_vmgt_pattern}": CustomEnv("CUSTOM_OAM_VMGT_PATTERN", "%vmname%-vmgt0"),
        "${env_custom_oam_mgt_pattern}": CustomEnv("CUSTOM_OAM_MGT_PATTERN", "%vmname%-mgt0"),
        "${env_custom_oam_ha_pattern}": CustomEnv("CUSTOM_OAM_HA_PATTERN", "%vmname%-ha0"),
        "${env_custom_oam_pkt0_pattern}": CustomEnv("CUSTOM_OAM_PKT0_PATTERN", "%vmname%-pkt0"),
        "${env_custom_oam_pkt1_pattern}": CustomEnv("CUSTOM_OAM_PKT1_PATTERN", "%vmname%-pkt1"),
        "${env_custom_oam_system_name}": CustomEnv("CUSTOM_OAM_SYSTEM_NAME", "${system_name}-${oam_name}"),
        "${env_custom_sbc_system_name}": CustomEnv("CUSTOM_SBC_SYSTEM_NAME", "${system_name}-${sbc_name}"),
        # The following could be specified in an environment variable or the command line:
        "${oam_server_group}": CustomEnv("CUSTOM_OAM_SERVER_GROUP_NAME", "%orchname%-OAM-${VNF_TYPE}-SG"),
        "${sbc_server_group}": CustomEnv("CUSTOM_SBC_SERVER_GROUP_NAME", "%orchname%-SBC-${VNF_TYPE}-SG"),
        "${system_name}": CustomEnv("CUSTOM_COMMON_SYSTEM_NAME", "SystemName"),
        "${env_custom_oam_block_storage1_pattern}": CustomEnv("ENV_CUSTOM_OAM_BLOCK_STORAGE1_PATTERN", "%vmname%-VOL-01"),
        "${env_custom_oam_block_storage2_pattern}": CustomEnv("ENV_CUSTOM_OAM_BLOCK_STORAGE2_PATTERN", "%vmname%-VOL-02"),

    }
    # If these are specified as an cmdline parameter, that overrides the envrionment variable.
    global oam_server_group
    if oam_server_group != "":
        custom_vm_name["${oam_server_group}"] = CustomEnv("cmdline", oam_server_group)
    global sbc_server_group
    if sbc_server_group != "":
        custom_vm_name["${sbc_server_group}"] = CustomEnv("cmdline", sbc_server_group)
    global system_name
    if system_name != "":
        custom_vm_name["${system_name}"] = CustomEnv("cmdline", system_name)

    # Also check for "CUSTOM_SBC_PARMS" environment variable, and add any that are found.
    custom_parms_split = []
    global env_var_to_print
    env_var_to_print = []
    custom_parm_list = []
    if "CUSTOM_SBC_PARMS" in os.environ:
        custom_sbc_parms = str(os.environ["CUSTOM_SBC_PARMS"])
        custom_parms_split = custom_sbc_parms.split(',')
        env_var_to_print.append('CUSTOM_SBC_PARMS="' + custom_sbc_parms + '"')
    for custom_parm_name in custom_parms_split:
        parm_name = custom_parm_name.strip()
        parm_default = ""
        env_default = "CUSTOM_SBC_PARM_" + parm_name.upper() + "_DEFAULT"
        if env_default in os.environ:
            parm_default = str(os.environ[env_default]).strip()
            env_var_to_print.append(env_default + '="' + parm_default + '"')
        else:
            warning_msg += "Warning: environment variable " + env_default + " not set.\n"
        parm_desc = parm_name
        env_desc = "CUSTOM_SBC_PARM_" + parm_name.upper() + "_DESC"
        if env_desc in os.environ:
            parm_desc = str(os.environ[env_desc]).strip()
            env_var_to_print.append(env_desc + '="' + parm_desc + '"')
        else:
            warning_msg += "Warning: environment variable " + env_desc + " not set.\n"
        parm_editableState = "" 
        env_editableState = "CUSTOM_SBC_PARM_" + parm_name.upper() + "_EDITABLESTATE"
        if env_editableState in os.environ:
            parm_editableState = str(os.environ[env_editableState]).strip()
            env_var_to_print.append(env_editableState + '="' + parm_editableState + '"')
        else:
            warning_msg += "Warning: environment variable " + env_editableState + " not set.\n"
        custom_parm_list.append(CustomParm(parm_name, parm_default, parm_desc, parm_editableState))

    # Similar to the VM names above, the custom variables for the security groups could be very long,
    # so they're set with environment variables below. They're handled separately from VM names,
    # because in this case, if the environment variable isn't specified, the group_name is deleted
    # (so the VNFM default name will be used).
    custom_sg_name = {
        "mgt": CustomEnv("CUSTOM_SG_NAME_MGT", ""),
        "ha": CustomEnv("CUSTOM_SG_NAME_HA", ""),
        "pkt": CustomEnv("CUSTOM_SG_NAME_PKT", ""),
        "omgt": CustomEnv("CUSTOM_SG_NAME_OMGT", ""),
        "oha": CustomEnv("CUSTOM_SG_NAME_OHA", "")
    }
    # Also similar to the VM Names, but it's not related to naming, so
    # no name expansion is done.
    custom_config = {
        "${env_custom_sbc_mgt_prefixv4}": CustomEnv("CUSTOM_SBC_MGT_PREFIXV4", "24"),
        "${env_custom_sbc_ha_prefixv4}": CustomEnv("CUSTOM_SBC_HA_PREFIXV4", "24"),
        "${env_custom_sbc_pkt0_prefixv4}": CustomEnv("CUSTOM_SBC_PKT0_PREFIXV4", "24"),
        "${env_custom_sbc_pkt1_prefixv4}": CustomEnv("CUSTOM_SBC_PKT1_PREFIXV4", "24"),
        "${env_custom_sbc_mgt_prefixv6}": CustomEnv("CUSTOM_SBC_MGT_PREFIXV6", "60"),
        "${env_custom_sbc_ha_prefixv6}": CustomEnv("CUSTOM_SBC_HA_PREFIXV6", "60"),
        "${env_custom_sbc_pkt0_prefixv6}": CustomEnv("CUSTOM_SBC_PKT0_PREFIXV6", "60"),
        "${env_custom_sbc_pkt1_prefixv6}": CustomEnv("CUSTOM_SBC_PKT1_PREFIXV6", "60"),
        "${env_custom_sbc_ha_vlanid}": CustomEnv("CUSTOM_SBC_HA_VLANID", "3005"),
        "${env_custom_sbc_pkt0_vlanid}": CustomEnv("CUSTOM_SBC_PKT0_VLANID", "3003"),
        "${env_custom_sbc_pkt1_vlanid}": CustomEnv("CUSTOM_SBC_PKT1_VLANID", "3004"),
    }

    # Read all the security group keys from os.environ, if they exist. Also expand each name if applicable.
    for key in custom_vm_name:
        env_name = str(custom_vm_name[key].env)
        if env_name != "cmdline" and env_name in os.environ:
            custom_vm_name[key].val = str(os.environ[env_name])
        expanded_name = expand_custom_name(custom_vm_name[key].val)
        custom_vm_name[key].val = expanded_name
    for key in custom_sg_name:
        env_name = str(custom_sg_name[key].env)
        if env_name != "cmdline" and env_name in os.environ:
            custom_sg_name[key].val = str(os.environ[env_name])
        expanded_name = expand_custom_name(custom_sg_name[key].val)
        custom_sg_name[key].val = expanded_name
    for key in custom_config:
        env_name = str(custom_config[key].env)
        if env_name != "cmdline" and env_name in os.environ:
            custom_config[key].val = str(os.environ[env_name])

    # Need to set system_name variable in case it changed above:
    if system_name == "":
        system_name = custom_vm_name["${system_name}"].val

    # Give warnings for some important patterns:
    if (not custom_vm_name['${env_custom_sbc_vmname_pattern}'].val.endswith('%index%') and
            not custom_vm_name['${env_custom_sbc_vmname_pattern}'].val.endswith('%index%{02d}')):
        warning_msg += ("Warning: environment variable " + custom_vm_name['${env_custom_sbc_vmname_pattern}'].env +
                        " does not end with %index% and may not work\n\n")
    if (not custom_vm_name['${env_custom_oam_vmname_pattern}'].val.endswith('%index%') and
            not custom_vm_name['${env_custom_oam_vmname_pattern}'].val.endswith('%index%{02d}')):
        warning_msg += ("Warning: environment variable " + custom_vm_name['${env_custom_oam_vmname_pattern}'].env +
                        " does not end with %index% and may not work\n\n")
    prefix_to_info_list = [
        PrefixToInfo("#ipv4_mgt_gn#",  "mgt", "omgt"),
        PrefixToInfo("#ipv6_mgt_gn#",  "mgt", "omgt"),
        PrefixToInfo("#ipv4_ha_gn#", "ha", "oha"),
        PrefixToInfo("#ipv6_ha_gn#", "ha", "oha"),
        PrefixToInfo("#ipv4_pkt_gn#",  "pkt", "pkt"),
        PrefixToInfo("#ipv6_pkt_gn#",  "pkt", "pkt")
    ]

    # Issue a warning if they specify a pattern that's not in the CUSTOM_PARMS list
    parm_keys = {}
    offset_to_get_parm_var = 11
    for key in custom_vm_name:
        env_var = custom_vm_name[key].val
        parm_start = env_var.find("%get_param:")
        while parm_start != -1:
            parm_end = env_var.find("%", parm_start+offset_to_get_parm_var)
            parm_keys[env_var[parm_start+offset_to_get_parm_var:parm_end]] = ""
            parm_start = env_var.find("%get_param:", parm_end)
    for key in custom_sg_name:
        env_var = custom_sg_name[key].val
        parm_start = env_var.find("%get_param:")
        while parm_start != -1:
            parm_end = env_var.find("%", parm_start+offset_to_get_parm_var)
            parm_keys[env_var[parm_start+offset_to_get_parm_var:parm_end]] = ""
            parm_start = env_var.find("%get_param:", parm_end)
    for key in parm_keys:
        if key not in custom_parms_split:
            warning_msg += ("Warning: environment variable COMMON_SBC_PARMS does not include pattern " + key +
                            ". This csar may not work\n\n")

    if default_pkt_ip_vers == "ipv6":
        if custom_config["${env_custom_sbc_pkt0_prefixv6}"].val == "":
            pkt0_prefix = "PrefixV6"
        else:
            pkt0_prefix = "PrefixV6:" + custom_config["${env_custom_sbc_pkt0_prefixv6}"].val
        if custom_config["${env_custom_sbc_pkt1_prefixv6}"].val == "":
            pkt1_prefix = "PrefixV6"
        else:
            pkt1_prefix = "PrefixV6:" + custom_config["${env_custom_sbc_pkt1_prefixv6}"].val
    else:
        if custom_config["${env_custom_sbc_pkt0_prefixv4}"].val == "":
            pkt0_prefix = "PrefixV4"
        else:
            pkt0_prefix = "PrefixV4:" + custom_config["${env_custom_sbc_pkt0_prefixv4}"].val
        if custom_config["${env_custom_sbc_pkt1_prefixv4}"].val == "":
            pkt1_prefix = "PrefixV4"
        else:
            pkt1_prefix = "PrefixV4:" + custom_config["${env_custom_sbc_pkt1_prefixv4}"].val

    if default_mgt_ip_vers == "ipv6":
        if custom_config["${env_custom_sbc_mgt_prefixv6}"].val == "":
            mgt_prefix = "PrefixV6"
        else:
            mgt_prefix = "PrefixV6:" + custom_config["${env_custom_sbc_mgt_prefixv6}"].val
    else:
        if custom_config["${env_custom_sbc_mgt_prefixv4}"].val == "":
            mgt_prefix = "PrefixV4"
        else:
            mgt_prefix = "PrefixV4:" + custom_config["${env_custom_sbc_mgt_prefixv4}"].val

    # If they set a specific ip_vers variable, that overrides the default ip_vers.
    if ip_vers_pkt0_var != "":
        if ip_vers_pkt0_var == "ipv4":
            if custom_config["${env_custom_sbc_pkt0_prefixv4}"].val == "":
                pkt0_prefix = "PrefixV4"
            else:
                pkt0_prefix = "PrefixV4:" + custom_config["${env_custom_sbc_pkt0_prefixv4}"].val
        elif ip_vers_pkt0_var == "ipv6":
            if custom_config["${env_custom_sbc_pkt0_prefixv6}"].val == "":
                pkt0_prefix = "PrefixV6"
            else:
                pkt0_prefix = "PrefixV6:" + custom_config["${env_custom_sbc_pkt0_prefixv6}"].val
        else:
            pkt0_prefix = "PrefixVn"

    if ip_vers_pkt1_var != "":
        if ip_vers_pkt1_var == "ipv4":
            if custom_config["${env_custom_sbc_pkt1_prefixv4}"].val == "":
                pkt1_prefix = "PrefixV4"
            else:
                pkt1_prefix = "PrefixV4:" + custom_config["${env_custom_sbc_pkt1_prefixv4}"].val
        elif ip_vers_pkt1_var == "ipv6":
            if custom_config["${env_custom_sbc_pkt1_prefixv6}"].val == "":
                pkt1_prefix = "PrefixV6"
            else:
                pkt1_prefix = "PrefixV6:" + custom_config["${env_custom_sbc_pkt1_prefixv6}"].val
        else:
            pkt1_prefix = "PrefixVn"

    if ip_vers_mgt0_var != "":
        if ip_vers_mgt0_var == "ipv4":
            if custom_config["${env_custom_sbc_mgt_prefixv4}"].val == "":
                mgt_prefix = "PrefixV4"
            else:
                mgt_prefix = "PrefixV4:" + custom_config["${env_custom_sbc_mgt_prefixv4}"].val
        elif ip_vers_mgt0_var == "ipv6":
            if custom_config["${env_custom_sbc_mgt_prefixv6}"].val == "":
                mgt_prefix = "PrefixV6"
            else:
                mgt_prefix = "PrefixV6:" + custom_config["${env_custom_sbc_mgt_prefixv6}"].val
        else:
            mgt_prefix = "PrefixVn"

    if default_ha_ip_vers == "ipv6":
        if custom_config["${env_custom_sbc_ha_prefixv6}"].val == "":
            ha_prefix = "PrefixV6"
        else:
            ha_prefix = "PrefixV6:" + custom_config["${env_custom_sbc_ha_prefixv6}"].val
    elif default_ha_ip_vers == "ipv4":
        if custom_config["${env_custom_sbc_ha_prefixv4}"].val == "":
            ha_prefix = "PrefixV4"
        else:
            ha_prefix = "PrefixV4:" + custom_config["${env_custom_sbc_ha_prefixv4}"].val
    else:
        ha_prefix = "PrefixVn"

    if no_dhcp_ha:
        ha_dhcp_custom = "DHCP:False"
    else:
        ha_dhcp_custom = "DHCP"

    if io_type == "virtio":
        vnic_type = "virtio-forwarder # tenant network"
        if2_custom = "Pkt0,DHCP,RNat"
        if3_custom = "Pkt1,DHCP,RNat"
    elif io_type == "sriov":
        vnic_type = "sr_iov_direct # this is sr_iov network"
        if2_custom = "Pkt0,IPVn,GWVn,DHCP:False," + pkt0_prefix + ",VlanId:" + custom_config["${env_custom_sbc_pkt0_vlanid}"].val
        if3_custom = "Pkt1,IPVn,GWVn,DHCP:False," + pkt1_prefix + ",VlanId:" + custom_config["${env_custom_sbc_pkt1_vlanid}"].val
    elif io_type == "provider":
        vnic_type = "normal # this is provider network"
        if2_custom = "Pkt0,IPVn,GWVn,DHCP:False," + pkt0_prefix
        if3_custom = "Pkt1,IPVn,GWVn,DHCP:False," + pkt1_prefix
    else:
        print("Unknown value for io_type: " + io_type)
        return

    global mgt_io_type
    if mgt_io_type == "default":
        mgt_io_type = io_type

    if mgt_io_type == "virtio":
        mgt_vnic_type = "virtio-forwarder # tenant network"
        if0_custom = "Mgt0,FIPV4,DHCP"
    elif mgt_io_type == "sriov":
        mgt_vnic_type = "sr_iov_direct # this is sr_iov network"
        if0_custom = "Mgt0,FIPV4,DHCP"
    elif mgt_io_type == "provider":
        mgt_vnic_type = "normal # this is provider network"
        if0_custom = "Mgt0,IPVn,GWVn,DHCP:False," + mgt_prefix
    else:
        print("Unknown value for mgt_io_type: " + mgt_io_type)
        return

    global ha_io_type
    if ha_io_type == "default":
        ha_io_type = "virtio"

    if ha_io_type == "virtio":
        ha_vnic_type = "virtio # tenant network"
        if1_custom = "Ha0,IPVn,GWVn," + ha_dhcp_custom + ",RNat"
    elif ha_io_type == "sriov":
        ha_vnic_type = "sr_iov_direct # this is sr_iov network"
        if1_custom = "Ha0,IPVn,GWVn," + ha_dhcp_custom + "," + ha_prefix + ",VlanId:" + custom_config["${env_custom_sbc_pkt1_vlanid}"].val
    elif ha_io_type == "provider":
        ha_vnic_type = "normal # this is provider network"
        if1_custom = "Ha0,IPVn,GWVn," + ha_dhcp_custom + "," + ha_prefix
    else:
        print("Unknown value for ha_io_type: " + ha_io_type)
        return

    global pkt_oam_io_type
    if pkt_oam_io_type == 'default':
        if prod_version.startswith('8.'):
            pkt_oam_io_type = 'virtio'
        else:
            # Default for version 9.x or later is disabled OAM pkt interfaces
            pkt_oam_io_type = 'disabled'

    global dsspp
    # In case we're called for multiple personalities in a loop, separate an
    # actual_dsspp (that might change in this loop) from the requested dsspp.
    actual_dsspp = dsspp
    if dsspp == "default":
        if vt == "ssbc" or vt == "msbc":
            actual_dsspp = "False"

        # Only needed for a bug in 8.1R1. Ignored after that, could be removed.
        elif sriov_redundancy and io_type == "sriov" and vt == "slb":
            actual_dsspp = "False"

    # Support for DosSupportSecPktPorts was only added to convertVnfm.py in 9.02.
    # If this version of the script is used in an older image version, it can still
    # be added with an "INSERT" directive. This bool indicates support in the image.
    dsspp_support = False
    if actual_dsspp != "default":
        if int(maj_version) > 9 or (int(maj_version) == 9 and int(min_version) >= 2):
            dsspp_support = True

    if pkt_oam_io_type == "virtio":
        pkt_oam_vnic_type = "virtio-forwarder # tenant network"
        oam_if2_custom = "Pkt0,IPVn,GWVn,DHCP,RNat"
        oam_if3_custom = "Pkt1,IPVn,GWVn,DHCP,RNat"
    elif pkt_oam_io_type == "sriov":
        pkt_oam_vnic_type = "sr_iov_direct # this is sr_iov network"
        oam_if2_custom = "Pkt0,IPVn,GWVn,DHCP:False,PrefixVn,VlanId:3006"
        oam_if3_custom = "Pkt1,IPVn,GWVn,DHCP:False,PrefixVn,VlanId:3007"
    elif pkt_oam_io_type == "provider":
        pkt_oam_vnic_type = "normal # this is provider network"
        oam_if2_custom = "Pkt0,IPVn,GWVn,DHCP,RNat,PrefixVn"
        oam_if3_custom = "Pkt1,IPVn,GWVn,DHCP,RNat,PrefixVn"
    elif pkt_oam_io_type == "disabled":
        # For 9.0 and later, this is the normal case.
        pkt_oam_vnic_type = "disabled # This line should be omitted"
        oam_if2_custom = "Pkt0"
        oam_if3_custom = "Pkt1"
    else:
        print("Unknown value for pkt_oam_io_type: " + pkt_oam_io_type)
        return

    # Set default values, which may be overridding by custom values later:
    if flavor_size == "dev_size":
        flavor_cpu = "4"
        flavor_ram = "10240"   # 10 G
        flavor_disk = "80"
    elif flavor_size == "large":
        flavor_cpu = "20"
        flavor_ram = "32768"   # 32 G
        flavor_disk = "100"
    elif flavor_size == "extra_large":
        flavor_cpu = "20"
        flavor_ram = "131072"  # 128 G
        flavor_disk = "100"
    else:
        print("Unknown value for flavor_size: " + flavor_size)
        return

    # And the OAM flavor
    oam_flavor_cpu = "4"
    oam_flavor_ram = "16384"  # 16 G
    oam_flavor_disk = "80"

    # if dev flavor specified, decrease minimum RAM requirement for use in dev environment
    if flavor_size == "dev_size":
        oam_flavor_ram = "10240"

    cinder_boot_size = flavor_disk
    cinder_oam_boot_size = oam_flavor_disk
    # If they specified a cinder boot disk, the ephemeral should be set to 0
    # just for clarity (because it's not used). Allow custom sizes to be set
    # after this, just in case someone wants to override the size to non-zero.
    if cinder_boot:
        flavor_disk = "0"
        oam_flavor_disk = "0"

    # Now allow for custom sizing on individual attributes
    if vcpu != -1:
        flavor_cpu = str(vcpu)
    if ram != -1:
        flavor_ram = str(ram)
    if disk != -1:
        flavor_disk = str(disk)
    if oam_vcpu != -1:
        oam_flavor_cpu = str(oam_vcpu)
    if oam_ram != -1:
        oam_flavor_ram = str(oam_ram)
    if oam_disk != -1:
        oam_flavor_disk = str(oam_disk)
    if cinder_oam_disk != -1:
        cinder_oam_boot_size = str(cinder_oam_disk)
    if cinder_sbc_disk != -1:
        cinder_boot_size = str(cinder_sbc_disk)
    if cinder_log_disk != -1:
        cinder_log_size = str(cinder_log_disk)
    else:
        cinder_log_size = "30"

    if_names_list = []
    mgt_if_names_list = []
    ha_if_names_list = []
    oam_mgt_if_names_list = []
    oam_ha_if_names_list = []
    pkt_if_names_list = []
    # Add each interface to the list, unless it's an SRIOV type.
    # SRIOV interfaces shouldn't have security groups.
    if num_sbc_rg > 1:
        for i in range(first_rg_index, num_sbc_rg+first_rg_index):
            if mgt_io_type != "sriov":
                if_names_list.append(sbc_rg_prefix(i) + "_" + "mgt0")
                mgt_if_names_list.append(sbc_rg_prefix(i) + "_" + "mgt0")
            if ha_io_type != "sriov":
                if_names_list.append(sbc_rg_prefix(i) + "_" + "ha0")
                ha_if_names_list.append(sbc_rg_prefix(i) + "_" + "ha0")
            if io_type != "sriov":
                if_names_list.append(sbc_rg_prefix(i) + "_" + "pkt0")
                if_names_list.append(sbc_rg_prefix(i) + "_" + "pkt1")
                pkt_if_names_list.append(sbc_rg_prefix(i) + "_" + "pkt0")
                pkt_if_names_list.append(sbc_rg_prefix(i) + "_" + "pkt1")
    else:
        if mgt_io_type != "sriov":
            if_names_list.append("mgt0")
            mgt_if_names_list.append("mgt0")
        if ha_io_type != "sriov":
            if_names_list.append("ha0")
            ha_if_names_list.append("ha0")
        if io_type != "sriov":
            if_names_list.append("pkt0")
            if_names_list.append("pkt1")
            pkt_if_names_list.append("pkt0")
            pkt_if_names_list.append("pkt1")

    if oam_vm:
        if mgt_io_type != "sriov":
            if_names_list.append("omgt0")
            if separate_oam_mgt_rules:
                oam_mgt_if_names_list.append("omgt0")
            else:
                mgt_if_names_list.append("omgt0")
        if ha_io_type != "sriov":
            if_names_list.append("oha0")
            if separate_oam_ha_rules:
                oam_ha_if_names_list.append("oha0")
            else:
                ha_if_names_list.append("oha0")
        if pkt_oam_io_type != "sriov" and pkt_oam_io_type != "disabled":
            if_names_list.append("opkt0")
            if_names_list.append("opkt1")
            pkt_if_names_list.append("opkt0")
            pkt_if_names_list.append("opkt1")

    # Generate the comma separated "if_names" string from the list.
    if_names = ", ".join(if_names_list)
    mgt_if_names = ", ".join(mgt_if_names_list)
    ha_if_names = ", ".join(ha_if_names_list)
    oam_mgt_if_names = ", ".join(oam_mgt_if_names_list)
    oam_ha_if_names = ", ".join(oam_ha_if_names_list)
    pkt_if_names = ", ".join(pkt_if_names_list)
    output_vnfd = os.path.join(def_dir, vt + "-vnfd.yaml")
    output_customtypes = os.path.join(def_dir, "ribbon_custom_types.yaml")
    output_etsi_vnfd_types = os.path.join(def_dir, "etsi_nfv_sol001_vnfd_types.yaml")
    output_etsi_common_types = os.path.join(def_dir, "etsi_nfv_sol001_common_types.yaml")
    date = datetime.datetime.utcnow().replace(microsecond=0).isoformat()

    verbose_description = "Deploys a Ribbon " + ha_desc + " " + long_personality + " SBC VNF"
    if not old_vnfm:
        verbose_description = verbose_description + ", based on SOL001 v"+ template_version + ","
    if not ha_model == "simplex":
        verbose_description = verbose_description + " with " + io_type + " networking"
    if oam_vm:
        verbose_description = verbose_description + ", " + str(num_oam) + " OAM VM"
    if int(num_sbc_rg) > 1:
        verbose_description = verbose_description + ", " + str(num_sbc_rg) + " RGs"
    if cinder_boot:
        verbose_description = verbose_description + ", cinder boot disk"
    if cinder_log:
        verbose_description = verbose_description + ", cinder disk for logging"
    if omit_ipv6:
        verbose_description = verbose_description + ", omit ipv6"
    if affinity == "Affinity":
        verbose_description = verbose_description + ", affinity"
    if affinity == "None":
        verbose_description = verbose_description + ", no affinity"
    if scaling:
        verbose_description = verbose_description + ", horizontal scaling"
    if dns_config_dic != []:
        verbose_description = verbose_description + ", DNS configuration"
    if not flavor_size == "dev_size":
        verbose_description = verbose_description + ", " + flavor_cpu + " vcpu, and " + str(int(flavor_ram) / 1024) + "G ram"
    if no_dhcp_ha:
        ha_dhcp = "false"
    else:
        ha_dhcp = "true"

    verbose_description = verbose_description + ", image " + image_name

    if old_vnfm:
        verbose_description = verbose_description + ". For use with VNFM pre v21.2"

    rg_start_line = 0
    param_start_line = 0
    rules_start_line = 0
    sbc_rules_index = 0
    sbc_param_index = 0
    comment_line = "#" + vt + "#"
    initdata_in_scripts = False

    if get_attribute_supported: 
        if ip_vers_mgt0_var == "" or ip_vers_mgt0_var == "ipvn":
            ip_vers_mgt0_var = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, network_mgt0_protocol ] }"
        if ip_vers_ha0_var == "" or ip_vers_ha0_var == "ipvn":
            ip_vers_ha0_var = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, network_ha0_protocol ] }"
        if ip_vers_pkt0_var == "" or ip_vers_pkt0_var == "ipvn":
            ip_vers_pkt0_var = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, network_pkt0_protocol ] }"
        if ip_vers_pkt1_var == "" or ip_vers_pkt1_var == "ipvn":
            ip_vers_pkt1_var = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, network_pkt1_protocol ] }"
 
        SGRemoteIpV4Mgt1 = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, SBC:SGRemoteIpV4Mgt1 ] }"
        SGRemoteIpV4Mgt2 = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, SBC:SGRemoteIpV4Mgt2 ] }"
        SGRemoteIpV6Mgt1 = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, SBC:SGRemoteIpV6Mgt1 ] }"
        SGRemoteIpV6Mgt2 = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, SBC:SGRemoteIpV6Mgt2 ] }"
        get_number_oam_instances = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, number_oam_instances ] }"
        CpNamePrefix = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, CpNamePrefix ] }"
        EMSIP0 = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, SBC:EMSIP0 ] }"
        EMSIP1 = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, SBC:EMSIP1 ] }"
        SGRemoteIpV6Vnfm = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, SBC:SGRemoteIpV6Vnfm ] }"
        SGRemoteIpV4Vnfm = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, SBC:SGRemoteIpV4Vnfm ] }"
        VnfrPort = "{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, SBC:VnfrPort ] }"
    else:
        if ip_vers_mgt0_var == "" or ip_vers_mgt0_var == "ipvn":
            ip_vers_mgt0_var = "ipv4"
        if ip_vers_ha0_var == "" or ip_vers_ha0_var == "ipvn":
            ip_vers_ha0_var = "ipv4"
        if ip_vers_pkt0_var == "" or ip_vers_pkt0_var == "ipvn":
            ip_vers_pkt0_var = "ipv4"
        if ip_vers_pkt1_var == "" or ip_vers_pkt1_var == "ipvn":
            ip_vers_pkt1_var = "ipv4"
        get_number_oam_instances = str(num_oam)

    if num_sbc_rg > 1:
        for i in range(first_rg_index, num_sbc_rg+first_rg_index):
            if get_attribute_supported:
                for j in range(first_pkt0_trunk_index, num_trunk_pkt0+first_pkt0_trunk_index):
                    num_pkt0_vlan.append("{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, " + sbc_rg_prefix(i) + "_" + "Pkt0_SubPort"+ str(j) + "_segmentation_id ] }")
                for k in range(first_pkt1_trunk_index, num_trunk_pkt1+first_pkt1_trunk_index):
                    num_pkt1_vlan.append("{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, " + sbc_rg_prefix(i) + "_" + "Pkt1_SubPort"+ str(k) + "_segmentation_id ] }")
            else:
                num_pkt0_vlan.append( str(default_SubPort0_segmentation_id) )
                num_pkt1_vlan.append( str(default_SubPort1_segmentation_id) )
    else:
        if get_attribute_supported:
            for j in range(first_pkt0_trunk_index, num_trunk_pkt0+first_pkt0_trunk_index):
                num_pkt0_vlan.append("{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, Pkt0_SubPort" + str(j) + "_segmentation_id ] }")
            for k in range(first_pkt1_trunk_index, num_trunk_pkt1+first_pkt1_trunk_index):
                num_pkt1_vlan.append("{ get_attribute : [ " + vt.upper() + "_VNF, modifiable_attributes, extensions, Pkt1_SubPort" + str(k) + "_segmentation_id ] }")
        else:
            num_pkt0_vlan.append( str(default_SubPort0_segmentation_id) )
            num_pkt1_vlan.append( str(default_SubPort1_segmentation_id) )

    if omit_ipv6:
        ip_vers_mgt0_var = "ipv4"
        ip_vers_ha0_var = "ipv4"
        ip_vers_pkt0_var = "ipv4"
        ip_vers_pkt1_var = "ipv4"

    if oam_ip_vers_pkt_var == "" or oam_ip_vers_pkt_var == "ipvn":
        if pkt_oam_io_type == "virtio":
            oam_ip_vers_pkt_var = "ipv4"
        else:
            oam_ip_vers_pkt_var = ip_vers_pkt0_var

    global pkt_alt_ips
    if len(pkt_alt_ips) > 0:
        if io_type != "sriov" and io_type != "provider":
            warning_msg += "Io-type:" + io_type + " not supported with PKT_ALT_IPS\n"
            return
    # The "${opt_alt_indexN}" should be set only if there are more than one index in a given pkt:
    num_pkt0_alt_index = 0
    num_pkt1_alt_index = 0
    for pai in pkt_alt_ips:
        if pai.pkt_index == '0':
            num_pkt0_alt_index += 1
        elif pai.pkt_index == '1':
            num_pkt1_alt_index += 1
    if num_pkt0_alt_index > 1:
        opt_alt_index0 = "_${alt_if_index}"
    else:
        opt_alt_index0 = ""
    if num_pkt1_alt_index > 1:
        opt_alt_index1 = "_${alt_if_index}"
    else:
        opt_alt_index1 = ""


    # We add security rules if:
    # - The interfaces is not SRIOV
    # - The ip version (ipv4 or ipv6) could be used for the given interface
    # - The user requested it with "rules" for the given interface and version
    #   *or* the user requested "--security_restrict"
    if mgt_if_names == "":  # SRIOV, so not allowed
        ipv4_mgt_rules = False
        ipv6_mgt_rules = False
    else:
        if mgt_rules == "ipv4":  # They requested only ipv4
            ipv4_mgt_rules = True
            ipv6_mgt_rules = False
        elif mgt_rules == "ipv6":  # They requested only ipv6
            ipv4_mgt_rules = False
            ipv6_mgt_rules = True
        elif mgt_rules == "ipvn":  # They requested both
            ipv4_mgt_rules = True
            ipv6_mgt_rules = True
        else:  # Default: Depends on security_open (ie: not --security_restrict)
            if security_open:
                ipv4_mgt_rules = False
                ipv6_mgt_rules = False
            else:
                if ip_vers_mgt0_var == "ipv4":  # The interface is ipv4 only
                    ipv4_mgt_rules = True
                    ipv6_mgt_rules = False
                elif ip_vers_mgt0_var == "ipv6":  # The interface is ipv6 only
                    ipv4_mgt_rules = False
                    ipv6_mgt_rules = True
                else:  # The interface could support be either
                    ipv4_mgt_rules = True
                    ipv6_mgt_rules = True
    if ha_if_names == "":  # SRIOV, so not allowed
        ipv4_ha_rules = False
        ipv6_ha_rules = False
    else:
        if ha_rules == "ipv4":  # They requested only ipv4
            ipv4_ha_rules = True
            ipv6_ha_rules = False
        elif ha_rules == "ipv6":  # They requested only ipv6
            ipv4_ha_rules = False
            ipv6_ha_rules = True
        elif ha_rules == "ipvn":  # They requested both
            ipv4_ha_rules = True
            ipv6_ha_rules = True
        else:  # Default: Depends on security_open (ie: not --security_restrict)
            if security_open:
                ipv4_ha_rules = False
                ipv6_ha_rules = False
            else:
                if ip_vers_ha0_var == "ipv4":  # The interface is ipv4 only
                    ipv4_ha_rules = True
                    ipv6_ha_rules = False
                elif ip_vers_ha0_var == "ipv6":  # The interface is ipv6 only
                    ipv4_ha_rules = False
                    ipv6_ha_rules = True
                else:  # The interface could support be either
                    ipv4_ha_rules = True
                    ipv6_ha_rules = True
    if pkt_if_names == "":  # SRIOV, so not allowed
        ipv4_pkt_rules = False
        ipv6_pkt_rules = False
    else:
        if pkt_rules == "ipv4":  # They requested only ipv4
            ipv4_pkt_rules = True
            ipv6_pkt_rules = False
        elif pkt_rules == "ipv6":  # They requested only ipv6
            ipv4_pkt_rules = False
            ipv6_pkt_rules = True
        elif pkt_rules == "ipvn":  # They requested both
            ipv4_pkt_rules = True
            ipv6_pkt_rules = True
        else:  # Default: Depends on security_open (ie: not --security_restrict)
            if security_open:
                ipv4_pkt_rules = False
                ipv6_pkt_rules = False
            else:
                # Both interfaces are ipv4 only
                if ip_vers_pkt0_var == "ipv4" and ip_vers_pkt1_var == "ipv4":
                    ipv4_pkt_rules = True
                    ipv6_pkt_rules = False
                # Both interfaces are ipv6 only
                elif ip_vers_pkt0_var == "ipv6" and ip_vers_pkt1_var == "ipv6":
                    ipv4_pkt_rules = False
                    ipv6_pkt_rules = True
                else:  # At least one interface could be either.
                    ipv4_pkt_rules = True
                    ipv6_pkt_rules = True

    # For interfaces that don't have "rules", add them to the "open" list
    if_v4_names_list = []
    if_v6_names_list = []
    open_v4_if_names_list = []
    open_v6_if_names_list = []
    if num_sbc_rg > 1:
        for i in range(first_rg_index, num_sbc_rg+first_rg_index):
            if mgt_io_type != "sriov":
                if ip_vers_mgt0_var != "ipv6":
                    if_v4_names_list.append(sbc_rg_prefix(i) + "_" + "mgt0")
                    if not ipv4_mgt_rules:
                        open_v4_if_names_list.append(sbc_rg_prefix(i) + "_" + "mgt0")
                if ip_vers_mgt0_var != "ipv4":
                    if_v6_names_list.append(sbc_rg_prefix(i) + "_" + "mgt0")
                    if not ipv6_mgt_rules:
                        open_v6_if_names_list.append(sbc_rg_prefix(i) + "_" + "mgt0")
            if ha_io_type != "sriov":
                if ip_vers_ha0_var != "ipv6":
                    if_v4_names_list.append(sbc_rg_prefix(i) + "_" + "ha0")
                    if not ipv4_ha_rules:
                        open_v4_if_names_list.append(sbc_rg_prefix(i) + "_" + "ha0")
                if ip_vers_ha0_var != "ipv4":
                    if_v6_names_list.append(sbc_rg_prefix(i) + "_" + "ha0")
                    if not ipv6_ha_rules:
                        open_v6_if_names_list.append(sbc_rg_prefix(i) + "_" + "ha0")
            if io_type != "sriov":
                if ip_vers_pkt0_var != "ipv6":
                    if_v4_names_list.append(sbc_rg_prefix(i) + "_" + "pkt0")
                    if not ipv4_pkt_rules:
                        open_v4_if_names_list.append(sbc_rg_prefix(i) + "_" + "pkt0")
                if ip_vers_pkt0_var != "ipv4":
                    if_v6_names_list.append(sbc_rg_prefix(i) + "_" + "pkt0")
                    if not ipv6_pkt_rules:
                        open_v6_if_names_list.append(sbc_rg_prefix(i) + "_" + "pkt0")
                if ip_vers_pkt1_var != "ipv6":
                    if_v4_names_list.append(sbc_rg_prefix(i) + "_" + "pkt1")
                    if not ipv4_pkt_rules:
                        open_v4_if_names_list.append(sbc_rg_prefix(i) + "_" + "pkt1")
                if ip_vers_pkt1_var != "ipv4":
                    if_v6_names_list.append(sbc_rg_prefix(i) + "_" + "pkt1")
                    if not ipv6_pkt_rules:
                        open_v6_if_names_list.append(sbc_rg_prefix(i) + "_" + "pkt1")
    else:
        if mgt_io_type != "sriov":
            if ip_vers_mgt0_var != "ipv6":
                if_v4_names_list.append("mgt0")
                if not ipv4_mgt_rules:
                    open_v4_if_names_list.append("mgt0")
            if ip_vers_mgt0_var != "ipv4":
                if_v6_names_list.append("mgt0")
                if not ipv6_mgt_rules:
                    open_v6_if_names_list.append("mgt0")
        if ha_io_type != "sriov":
            if ip_vers_ha0_var != "ipv6":
                if_v4_names_list.append("ha0")
                if not ipv4_ha_rules:
                    open_v4_if_names_list.append("ha0")
            if ip_vers_ha0_var != "ipv4":
                if_v6_names_list.append("ha0")
                if not ipv6_ha_rules:
                    open_v6_if_names_list.append("ha0")
        if io_type != "sriov":
            if ip_vers_pkt0_var != "ipv6":
                if_v4_names_list.append("pkt0")
                if not ipv4_pkt_rules:
                    open_v4_if_names_list.append("pkt0")
            if ip_vers_pkt0_var != "ipv4":
                if_v6_names_list.append("pkt0")
                if not ipv6_pkt_rules:
                    open_v6_if_names_list.append("pkt0")
            if ip_vers_pkt1_var != "ipv6":
                if_v4_names_list.append("pkt1")
                if not ipv4_pkt_rules:
                    open_v4_if_names_list.append("pkt1")
            if ip_vers_pkt1_var != "ipv4":
                if_v6_names_list.append("pkt1")
                if not ipv6_pkt_rules:
                    open_v6_if_names_list.append("pkt1")

    if oam_vm:
        if mgt_io_type != "sriov":
            if ip_vers_mgt0_var != "ipv6":
                if_v4_names_list.append("omgt0")
                if not ipv4_mgt_rules:
                    open_v4_if_names_list.append("omgt0")
            if ip_vers_mgt0_var != "ipv4":
                if_v6_names_list.append("omgt0")
                if not ipv6_mgt_rules:
                    open_v6_if_names_list.append("omgt0")
        if ha_io_type != "sriov":
            if ip_vers_ha0_var != "ipv6":
                if_v4_names_list.append("oha0")
                if not ipv4_ha_rules:
                    open_v4_if_names_list.append("oha0")
            if ip_vers_ha0_var != "ipv4":
                if_v6_names_list.append("oha0")
                if not ipv6_ha_rules:
                    open_v6_if_names_list.append("oha0")
        if pkt_oam_io_type != "sriov" and pkt_oam_io_type != "disabled":
            if oam_ip_vers_pkt_var != "ipv6":
                if_v4_names_list.append("opkt0")
                if not ipv4_pkt_rules:
                    open_v4_if_names_list.append("opkt0")
            if oam_ip_vers_pkt_var != "ipv4":
                if_v6_names_list.append("opkt0")
                if not ipv6_pkt_rules:
                    open_v6_if_names_list.append("opkt0")
            if oam_ip_vers_pkt_var != "ipv6":
                if_v4_names_list.append("opkt1")
                if not ipv4_pkt_rules:
                    open_v4_if_names_list.append("opkt1")
            if oam_ip_vers_pkt_var != "ipv4":
                if_v6_names_list.append("opkt1")
                if not ipv6_pkt_rules:
                    open_v6_if_names_list.append("opkt1")

    # Generate the comma separated "open_*_if_names" string from the list.
    open_v4_names = ", ".join(open_v4_if_names_list)
    open_v6_names = ", ".join(open_v6_if_names_list)
    if_v4_names = ", ".join(if_v4_names_list)
    if_v6_names = ", ".join(if_v6_names_list)

    for key in custom_sg_name:
        if custom_sg_name[key].env in os.environ:
            env_var_to_print.append(custom_sg_name[key].env + '="' + custom_sg_name[key].val + '"')
    for key in custom_vm_name:
        if custom_vm_name[key].env in os.environ:
            env_var_to_print.append(custom_vm_name[key].env + '="' + custom_vm_name[key].val + '"')
    for key in custom_config:
        if custom_config[key].env in os.environ:
            env_var_to_print.append(custom_config[key].env + '="' + custom_config[key].val + '"')
                    
    # redundancy_ratio logic
    global redund_ratio
    if ha_model == "simplex":
        redund_ratio = "1:0"
    elif ha_model == "active_standby":
        redund_ratio = "1:1"
    elif ha_model == "n_1_ha":
        if redund_ratio == "":
            redund_ratio = n_active_instances+":1"
    rg_actives = int(redund_ratio.split(':')[0])
    rg_standby = int(redund_ratio.split(':')[1])

    # Scalaing variable replacement options
    if scaling:
        # Construct a list of SBC VDUs
        if num_sbc_rg > 1:
            for i in range(first_rg_index, num_sbc_rg+first_rg_index):
                if i == first_rg_index:
                    list_of_sbc_rgs = "ribbon-" + sbc_rg_prefix(i) + "-sbc"
                else:
                    list_of_sbc_rgs += ", ribbon-" + sbc_rg_prefix(i) + "-sbc"
        else:
            list_of_sbc_rgs = "ribbon-sbc"

        max_sbc_instances = str((rg_actives + rg_standby) * max_sbc_rg)
        deltas = ""
        delta_spacing_prefix = "\n                - "
        if ha_model == "simplex":
            max_scale_level = str(max_sbc_rg - 1)
            deltas = delta_spacing_prefix + "sbc_delta_1"
            min_sbc_instances = str(1)
        elif ha_model == "active_standby":
            max_scale_level = str(max_sbc_rg - 1)
            deltas = delta_spacing_prefix + "sbc_delta_2"
            min_sbc_instances = str(2)
        elif ha_model == "n_1_ha":
            max_scale_level = str((rg_actives * max_sbc_rg) - 1)
            for i in range(1, rg_actives):
                deltas += delta_spacing_prefix + "sbc_delta_1" #Active
            deltas += delta_spacing_prefix + "sbc_delta_2" #Act/Sby. New RG
            min_sbc_instances = str(2)
        num_instantiation_lvls = max_sbc_rg if ha_model == "n_1_ha" else (max_sbc_rg - 1)
        instantiation_level = 0

    add_upgrade_mop(scripts_dir)

    """Start Write File"""
    if old_vnfm:
        in_out_file_list = [ (vnfd_file, output_vnfd) ]
    else:
        in_out_file_list = [ (vnfd_file, output_vnfd), (customtypes_file, output_customtypes), (etsi_vnfd_types_file, output_etsi_vnfd_types), (etsi_common_types_file, output_etsi_common_types) ]
    for in_out_file in in_out_file_list:
        with open(in_out_file[0], 'r') as in_file:
            with open(in_out_file[1], 'w') as out_file:
                line = in_file.readline()
                sbc_rg_count = 0
                trunk_pkt0_count = 0
                trunk_pkt1_count = 0
                while line:
                    sbc_rg_count_str = str(sbc_rg_count)
                    skip_line = False

                    # Custom names need to be replaced first, because they contain
                    # other variables replaced below:

                    # Replace custom vm_keys like "${env_custom_sbc_vmgt_pattern}" with their
                    # default value, or whatever pattern we read from the corresponding env var.
                    for vm_key in custom_vm_name:
                        if not line.find(vm_key) == -1:
                            line = line.replace(vm_key, custom_vm_name[vm_key].val)
                    # On lines with a prefix like "#ipv4_mgt_gn#" replace the "${group_name}"
                    # if they set a value. Otherwise skip the line to use the VNFM default name.
                    for ptoi in prefix_to_info_list:
                        if sbc_rules_index == 0:
                            sg_key = ptoi.var0
                        else:
                            sg_key = ptoi.var1
                        if not line.find(ptoi.prefix) == -1:
                            if custom_sg_name[sg_key].val != "":
                                line = line.replace(ptoi.prefix, "")
                                line = line.replace("${group_name}", '"' + custom_sg_name[sg_key].val + '"')
                            else:
                                skip_line = True

                    # These custom names also need to be replaced next, they also could contain other variables:
                    # other variables replaced below:
                    line = line.replace("${oam_name}", oam_name)
                    line = line.replace("${sbc_name}", sbc_name)
                    line = line.replace("${placement_oam_name}", placement_oam_name)
                    line = line.replace("${placement_sbc_name}", placement_sbc_name)
                    line = line.replace("${sbc_rg_index}", sbc_rg_count_str)

                    # ${system_name} was probably already replaced by the custom_vm_name loop above,
                    # but other variables might also contain system_name, so re-replace:
                    line = line.replace("${system_name}", system_name)

                    line = line.replace("${cmdline_args}", cmdline_args)
                    # In addition to cmdline_args, also dump out significant environment vars.
                    if not line.find('#environ_args#') == -1:
                        skip_line = True
                        for i in sorted(env_var_to_print):
                            out_file.write("#    " + i + "\n")

                    line = line.replace("${vnftype}", vt)
                    line = line.replace("${VNF_TYPE}", vt.upper())
                    line = line.replace("${ha_model}", ha_model)
                    line = line.replace("${ha_desc}", ha_desc)
                    line = line.replace("${io_type}", io_type)
                    line = line.replace("${ha_io_type}", ha_io_type)
                    line = line.replace("${mgt_io_type}", mgt_io_type)
                    line = line.replace("${vnic_type}", vnic_type)
                    line = line.replace("${mgt_vnic_type}", mgt_vnic_type)
                    line = line.replace("${ha_vnic_type}", ha_vnic_type)
                    line = line.replace("${verbose_description}", verbose_description)
                    line = line.replace("${affinity}", affinity)
                    line = line.replace("${date}", date)
                    # line = line.replace("${packet_if}", packet_if)
                    line = line.replace("${if0_custom}", if0_custom)
                    line = line.replace("${if1_custom}", if1_custom)
                    line = line.replace("${if2_custom}", if2_custom)
                    line = line.replace("${if3_custom}", if3_custom)
                    line = line.replace("${oam_if2_custom}", oam_if2_custom)
                    line = line.replace("${oam_if3_custom}", oam_if3_custom)
                    line = line.replace("${prod_version}", prod_vers)
                    line = line.replace("${sub_version}", sub_vers)
                    if prod_version.startswith('8.01'):
                        # Special for 8.01: if a version suffix was used in the .mf file,
                        # also used it in the VNFD file. Why: The suffix should only be used
                        # in the .mf file, but in earlier version of VNFM (19.1), the version
                        # in the .mf file needs to be the same as the version in the VNFD
                        # file. However, this causes problems with automated upgrades if the
                        # VNFD version is not a real version that can be looked up in the
                        # database. But since automated upgrades aren't supported in 8.01,
                        # it's fine to use it in 8.01. This means that 8.02 needs to be used
                        # with VNFM 19.2 or later (if a suffix is used), but 8.01 can be used
                        # with VNFM 19.1 or 19.2 (or later).
                        sw_version_suffix = version_suffix
                    else:
                        sw_version_suffix = ""
                    line = line.replace("${sw_version_suffix}", sw_version_suffix)
                    line = line.replace("${version_tag}", version_tag)
                    line = line.replace("${image_name}", image_name)
                    line = line.replace("${default_mgt_ip_vers}", default_mgt_ip_vers)
                    line = line.replace("${default_ha_ip_vers}", default_ha_ip_vers)
                    line = line.replace("${default_pkt_ip_vers}", default_pkt_ip_vers)
                    line = line.replace("${flavor_cpu}", flavor_cpu)
                    line = line.replace("${flavor_ram}", flavor_ram)
                    line = line.replace("${flavor_disk}", flavor_disk)
                    line = line.replace("${oam_flavor_cpu}", oam_flavor_cpu)
                    line = line.replace("${oam_flavor_ram}", oam_flavor_ram)
                    line = line.replace("${oam_flavor_disk}", oam_flavor_disk)
                    line = line.replace("${personality}", short_personality)
                    line = line.replace("${long_personality}", long_personality)
                    line = line.replace("${num_instances}", num_instances)
                    line = line.replace("${num_instances_sol001}", num_instances_sol001)
                    if not line.find("${num_instances_var}") == -1:
                        if scaling:
                            if not line.find("min_") == -1:
                                line = line.replace("${num_instances_var}", min_sbc_instances)
                            if not line.find("max_") == -1:
                                line = line.replace("${num_instances_var}", max_sbc_instances)
                            else:
                                line = line.replace("${num_instances_var}", num_instances_var[sbc_rg_count-first_rg_index])
                        else:
                            line = line.replace("${num_instances_var}", num_instances_var[sbc_rg_count-first_rg_index])
                    line = line.replace("${num_oam_instances}", num_oam)
                    line = line.replace("${n_active_instances}", n_active_instances)
                    line = line.replace("${ha_dhcp}", ha_dhcp)
                    # If cinder_log is used, this is the size in GB.
                    # If cinder_boot is used, this is the size in GB.
                    line = line.replace("${cinder_boot_size}", cinder_boot_size)
                    line = line.replace("${cinder_oam_boot_size}", cinder_oam_boot_size)
                    line = line.replace("${cinder_log_size}", cinder_log_size)
                    line = line.replace("${cinder_boot_attributes}", cinder_boot_attributes)
                    line = line.replace("${cinder_log_attributes}", cinder_log_attributes)
                    line = line.replace(comment_line, "")
                    line = line.replace("${compact_version}", compact_version)
                    line = line.replace("${vdu_cp}", vdu_cp)
                    line = line.replace("${vdu_vbs}", vdu_vbs)
                    if zone == "omit":
                        line = line.replace("${zone_omit}", "### ")
                        line = line.replace("${zone}", "nova")
                    else:
                        line = line.replace("${zone_omit}", "")
                        line = line.replace("${zone}", zone)
                    if oam_zone == "omit":
                        line = line.replace("${oam_zone_omit}", "### ")
                        line = line.replace("${oam_zone}", "nova")
                    else:
                        line = line.replace("${oam_zone_omit}", "")
                        line = line.replace("${oam_zone}", oam_zone)
                
                    line = line.replace("${if_names}", if_names)
                    line = line.replace("${if_v4_names}", if_v4_names)
                    line = line.replace("${if_v6_names}", if_v6_names)
                    line = line.replace("${open_v4_names}", open_v4_names)
                    line = line.replace("${open_v6_names}", open_v6_names)
                    if separate_oam_mgt_rules and sbc_rules_index == 1:
                        line = line.replace("${mgt_if_names}", oam_mgt_if_names)
                        line = line.replace("${separateOamPrefix}", "Oam")
                    else:
                        line = line.replace("${mgt_if_names}", mgt_if_names)
                        line = line.replace("${separateOamPrefix}", "")
                    if separate_oam_ha_rules and sbc_rules_index == 1:
                        line = line.replace("${ha_if_names}", oam_ha_if_names)
                        line = line.replace("${separateOamPrefix}", "Oam")
                    else:
                        line = line.replace("${ha_if_names}", ha_if_names)
                        line = line.replace("${separateOamPrefix}", "")
                    line = line.replace("${pkt_if_names}", pkt_if_names)
                    line = line.replace("${pkt_oam_vnic_type}", pkt_oam_vnic_type)
                    line = line.replace("${pkt_oam_io_type}", pkt_oam_io_type)

                    # Use either "#new_dsspp#" or "#old_dsspp#", depending on dsspp_support.
                    # If dsspp is "default" omit both, and the default (True) in the image is used.
                    if line.find("#new_dsspp#") != -1:
                        if actual_dsspp != "default" and dsspp_support:
                            line = line.replace("#new_dsspp#", "")
                            line = line.replace("${sbc_dsspp}", actual_dsspp)
                        else:
                            skip_line = True
                    if line.find("#old_dsspp#") != -1:
                        if actual_dsspp != "default" and not dsspp_support:
                            line = line.replace("#old_dsspp#", "")
                            line = line.replace("${sbc_dsspp}", actual_dsspp)
                        else:
                            skip_line = True

                    # Note that #system_name# is no longer used, left for backwards compatibility
                    # (using an older version of the template and this script).
                    if int(num_sbc_rg) == 1:
                        line = line.replace("${sbc_rg_prefix}", "")
                        line = line.replace("${SBC_RG_PREFIX_DASH}", "")
                        line = line.replace("${sbc_rg_prefix_dash}", "")
                    else:
                        if not line.find('#system_name#') == -1:
                            line = line.replace("${sbc_rg_prefix}", sbc_rg_prefix(sbc_rg_count) + "-")
                        line = line.replace("${sbc_rg_prefix}", sbc_rg_prefix(sbc_rg_count) + "_")
                        line = line.replace("${SBC_RG_PREFIX_DASH}", sbc_rg_prefix(sbc_rg_count).upper() + "-")
                        line = line.replace("${sbc_rg_prefix_dash}", sbc_rg_prefix(sbc_rg_count) + "-")

                    line = line.replace("${vip_cp}", vip_cp)

                    line = line.replace("${qcow2_hash}", qcow2_hash)
                    line = line.replace("${SGRemoteIpV4Mgt1}", SGRemoteIpV4Mgt1)
                    line = line.replace("${SGRemoteIpV4Mgt2}", SGRemoteIpV4Mgt2)
                    line = line.replace("${SGRemoteIpV6Mgt1}", SGRemoteIpV6Mgt1)
                    line = line.replace("${SGRemoteIpV6Mgt2}", SGRemoteIpV6Mgt2)
                    line = line.replace("${get_number_oam_instances}", get_number_oam_instances)
                    line = line.replace("${CpNamePrefix}", CpNamePrefix)
                    line = line.replace("${EMSIP0}", EMSIP0)
                    line = line.replace("${EMSIP1}", EMSIP1)
                    line = line.replace("${SGRemoteIpV6Vnfm}", SGRemoteIpV6Vnfm)
                    line = line.replace("${SGRemoteIpV4Vnfm}", SGRemoteIpV4Vnfm)
                    line = line.replace("${VnfrPort}", VnfrPort)


                    line = line.replace("#system_name#", "")

                    if not line.find("#insert_userdata#") == -1:
                        if insert_userdata == "":
                            skip_line = True
                        else:
                            line = line.replace("#insert_userdata#", insert_userdata.rstrip())

                    if not line.find("#insert_userdata_inputs#") == -1:
                        if insert_userdata_topology == "":
                            skip_line = True
                        else:
                            line = line.replace("#insert_userdata_inputs#", insert_userdata_topology.replace("#space_delim#", "          ").rstrip())
                    
                    if not line.find("#insert_userdata_VNF#") == -1:
                        if insert_userdata_topology == "":
                            skip_line = True
                        else:
                            line = line.replace("#insert_userdata_VNF#", insert_userdata_topology.replace("#space_delim#", "            ").rstrip())

                    if not line.find("#ip_vers_mgt0_choice#") == -1:
                        if ip_vers_mgt0_var == "ipv4" or ip_vers_mgt0_var == "ipv6":
                            skip_line = True
                        else:
                            line = line.replace("#ip_vers_mgt0_choice#", "")
                    if not line.find("#ip_vers_ha0_choice#") == -1:
                        if ip_vers_ha0_var == "ipv4" or ip_vers_ha0_var == "ipv6":
                            skip_line = True
                        else:
                            line = line.replace("#ip_vers_ha0_choice#", "")
                    if not line.find("#ip_vers_pkt0_choice#") == -1:
                        if ip_vers_pkt0_var == "ipv4" or ip_vers_pkt0_var == "ipv6":
                            skip_line = True
                        else:
                            line = line.replace("#ip_vers_pkt0_choice#", "")
                    if not line.find("#ip_vers_pkt1_choice#") == -1:
                        if ip_vers_pkt1_var == "ipv4" or ip_vers_pkt1_var == "ipv6":
                            skip_line = True
                        else:
                            line = line.replace("#ip_vers_pkt1_choice#", "")

                    if not line.find("#security_open_v4#") == -1:
                        if open_v4_names != "":
                            line = line.replace("#security_open_v4#", "")
                        else:
                            skip_line = True
                    if not line.find("#security_open_v6#") == -1:
                        if open_v6_names != "":
                            line = line.replace("#security_open_v6#", "")
                        else:
                            skip_line = True
                    if not line.find("#ipv4_icmp_rules#") == -1:
                        if if_v4_names != "":
                            line = line.replace("#ipv4_icmp_rules#", "")
                        else:
                            skip_line = True
                    if not line.find("#ipv6_icmp_rules#") == -1:
                        if if_v6_names != "":
                            line = line.replace("#ipv6_icmp_rules#", "")
                        else:
                            skip_line = True

                    if not line.find("#ipv4_mgt_rules#") == -1:
                        if ipv4_mgt_rules:
                            line = line.replace("#ipv4_mgt_rules#", "")
                        else:
                            skip_line = True
                    if not line.find("#ipv6_mgt_rules#") == -1:
                        if ipv6_mgt_rules:
                            line = line.replace("#ipv6_mgt_rules#", "")
                        else:
                            skip_line = True
                    if not line.find("#mgt_rules") == -1:
                        if ipv4_mgt_rules or ipv6_mgt_rules:
                            line = line.replace("#mgt_rules#", "")
                        else:
                            skip_line = True
                    if not line.find("#mgt_rules_rip#") == -1:
                        if mgt_num_rip > 0:
                            line = line.replace("#mgt_rules_rip#", "")
                        else:
                            skip_line = True
                    if not line.find("#mgt_rules_rip2#") == -1:
                        if mgt_num_rip > 1:
                            line = line.replace("#mgt_rules_rip2#", "")
                        else:
                            skip_line = True
                    if not line.find("#mgt_rules_ems_rip#") == -1:
                        if mgt_num_ems_rip > 0:
                            line = line.replace("#mgt_rules_ems_rip#", "")
                        else:
                            skip_line = True
                    if not line.find("#mgt_rules_ems_rip2#") == -1:
                        if mgt_num_ems_rip > 1:
                            line = line.replace("#mgt_rules_ems_rip2#", "")
                        else:
                            skip_line = True
                    if not line.find("#ipv4_ha_rules#") == -1:
                        if ipv4_ha_rules:
                            line = line.replace("#ipv4_ha_rules#", "")
                        else:
                            skip_line = True
                    if not line.find("#ipv6_ha_rules#") == -1:
                        if ipv6_ha_rules:
                            line = line.replace("#ipv6_ha_rules#", "")
                        else:
                            skip_line = True
                    if not line.find("#ipv4_pkt_rules#") == -1:
                        if ipv4_pkt_rules:
                            line = line.replace("#ipv4_pkt_rules#", "")
                        else:
                            skip_line = True
                    if not line.find("#ipv6_pkt_rules#") == -1:
                        if ipv6_pkt_rules:
                            line = line.replace("#ipv6_pkt_rules#", "")
                        else:
                            skip_line = True

                    if single_system_name:
                        line = line.replace("#single_system_name#", "")
                        if not line.find("#different_system_name#") == -1:
                            skip_line = True
                    else:
                        line = line.replace("#different_system_name#", "")
                        if not line.find("#single_system_name#") == -1:
                            skip_line = True

                    if not line.find("#sriov_pkt_redundancy#") == -1:
                        if sriov_redundancy and io_type == "sriov":
                            line = line.replace("#sriov_pkt_redundancy#", "")
                        else:
                            skip_line = True

                    if cinder_boot:
                        line = line.replace("#cinder_boot#", "")
                    else:
                        line = line.replace("#no_cinder_boot#", "")

                    if cinder_log:
                        line = line.replace("#cinder_log#", "")

                    if cinder_boot or cinder_log:
                        line = line.replace("#cinder_any#", "")
                    else:
                        line = line.replace("#cinder_none#", "")

                    if omit_ipv6:
                        if not line.find('#ipv6#') == -1:
                            skip_line = True
                    else:
                        line = line.replace("#ipv6#", "")

                    if ha_mode:
                        line = line.replace("#ha_mode#", "")
                        line = line.replace("${sbc_ha_mode}", sbc_ha_mode)
                    elif not line.find('#ha_mode#') == -1:
                            skip_line = True

                    if ha_model == "n_1_ha":
                        line = line.replace("#n_1_ha#", "")
                    elif not line.find('#n_1_ha#') == -1:
                        skip_line = True

                    line = line.replace("${ip_vers_mgt0_var}", ip_vers_mgt0_var)
                    line = line.replace("${ip_vers_ha0_var}", ip_vers_ha0_var)
                    line = line.replace("${ip_vers_pkt0_var}", ip_vers_pkt0_var)
                    line = line.replace("${ip_vers_pkt1_var}", ip_vers_pkt1_var)
                    line = line.replace("${oam_ip_vers_pkt_var}", oam_ip_vers_pkt_var)

                    if not line.find('#future#') == -1:
                        skip_line = True
                    if not line.find('#comment#') == -1:
                        skip_line = True

                    # handle oam specific lines
                    if oam_vm:
                        line = line.replace("${oam_ha_model}", oam_ha_model)
                        line = line.replace("#oam#", "")
                        if not line.find('#no_oam#') == -1:
                            skip_line = True
                    else:
                        if not line.find('#oam#') == -1:
                            skip_line = True
                        line = line.replace("#no_oam#", "")

                    if not line.find('#oam_pkt_if#') == -1:
                        if pkt_oam_io_type == "disabled":
                            skip_line = True
                        else:
                            line = line.replace('#oam_pkt_if#', '')

                    if not line.find('#slb#') == -1:
                        if vt == "slb":
                            line = line.replace("#oam#", "")
                        else:
                            skip_line = True

                    line = line.replace("#ipv6#", "")

                    # Notice that the different interfaces each are controlled
                    # separately. The pkt0/pkt1 are controlled by "io_type" (and
                    # "pkt_oam_io_type" for the OAM VM), the other interfaces are
                    # controlled by mgt_io_type and ha_io_type. For each one there
                    # are separate prefix tokens to select that type. Enable the
                    # selected interfaces:
                    if io_type == "virtio":
                        line = line.replace("#virtio_pkt#", "")
                    elif io_type == "sriov":
                        line = line.replace("#sriov_pkt#", "")
                    elif io_type == "provider":
                        line = line.replace("#provider_pkt#", "")

                    if pkt_oam_io_type == "virtio":
                        line = line.replace("#virtio_pkt_oam#", "")
                    elif pkt_oam_io_type == "sriov":
                        line = line.replace("#sriov_pkt_oam#", "")
                    elif pkt_oam_io_type == "provider":
                        line = line.replace("#provider_pkt_oam#", "")

                    if pkt_vip and not io_type == "virtio":
                        line = line.replace("#vip_pkt#", "")

                    if mgt_io_type == "virtio":
                        line = line.replace("#virtio_mgmt#", "")
                    # For SRIOV, provider, or anything else, there's nothing to
                    # uncomment except optionally vip_mgt, below.

                    if mgt_vip and not mgt_io_type == "virtio":
                        line = line.replace("#vip_mgmt#", "")
                    else:
                        line = line.replace("#no_vip_mgmt#", "")

                    if ha_io_type == "virtio":
                        line = line.replace("#virtio_ha#", "")
                    # For SRIOV, provider, or anything else, there's nothing to
                    # add to ha, because we don't have a VIP for HA.

                    # Skip other lines containing non-matching io-types
                    if not line.find('#vip_pkt#') == -1:
                        skip_line = True
                    if not line.find('#provider_pkt#') == -1:
                        skip_line = True
                    if not line.find('#sriov_pkt#') == -1:
                        skip_line = True
                    if not line.find('#virtio_pkt#') == -1:
                        skip_line = True
                    if not line.find('#vip_mgmt#') == -1:
                        skip_line = True
                    if not line.find('#no_vip_mgmt#') == -1:
                        skip_line = True
                    if not line.find('#virtio_mgmt#') == -1:
                        skip_line = True
                    if not line.find('#virtio_ha#') == -1:
                        skip_line = True
                    if not line.find('#virtio_pkt_oam#') == -1:
                        skip_line = True
                    if not line.find('#provider_pkt_oam#') == -1:
                        skip_line = True
                    if not line.find('#sriov_pkt_oam#') == -1:
                        skip_line = True

                    if not line.find('#tsbc#') == -1:
                        skip_line = True
                    if not line.find('#cinder_any#') == -1:
                        skip_line = True
                    if not line.find('#cinder_none#') == -1:
                        skip_line = True
                    if not line.find('#cinder_boot#') == -1:
                        skip_line = True
                    if not line.find('#cinder_log#') == -1:
                        skip_line = True
                    if not line.find('#noaffinity#') == -1:
                        if affinity == "None":
                            skip_line = True
                        else:
                            line = line.replace("#noaffinity#", "")
                    if custom_naming:
                        line = line.replace("#custom_name#", "")
                        if not line.find('#no_custom_name#') == -1:
                            skip_line = True
                    else:
                        line = line.replace("#no_custom_name#", "")
                        if not line.find('#custom_name#') == -1:
                            skip_line = True
                    # #dash_index# and #no_dash_index are no longer used -- this could be removed.
                    if dash_index:
                        line = line.replace("#dash_index#", "")
                        if not line.find('#no_dash_index#') == -1:
                            skip_line = True
                    else:
                        line = line.replace("#no_dash_index#", "")
                        if not line.find('#dash_index#') == -1:
                            skip_line = True

                    if not line.find("#sbc_rg_start#") == -1:
                        skip_line = True
                        rg_start_line = in_file.tell()
                        sbc_rg_count = 1

                    if not line.find("#sbc_rg_end#") == -1:
                        skip_line = True
                        if int(sbc_rg_count) < int(num_sbc_rg):
                            in_file.seek(rg_start_line)
                            sbc_rg_count += 1
                        else:
                            # Reset the RG Count, just in case it's used outside the loop.
                            sbc_rg_count = 0

                    if not int(num_trunk_pkt0) == 0:
                        line = line.replace("${trunk_pkt0_prefix}", str(trunk_pkt0_count))
                    if not int(num_trunk_pkt1) == 0:
                        line = line.replace("${trunk_pkt1_prefix}", str(trunk_pkt1_count))

                    if not line.find("#trunk_pkt0_start#") == -1:
                        skip_line = True
                        trunk_pkt0_start_line = in_file.tell()
                        trunk_pkt0_count = 1

                    if not line.find("#trunk_pkt0_end#") == -1:
                        skip_line = True
                        if int(trunk_pkt0_count) < int(num_trunk_pkt0):
                            in_file.seek(trunk_pkt0_start_line)
                            trunk_pkt0_count += 1
                        else:
                            trunk_pkt0_count = 1

                    if not line.find("#trunk_pkt1_start#") == -1:
                        skip_line = True
                        trunk_pkt1_start_line = in_file.tell()
                        trunk_pkt1_count = 1

                    if not line.find("#trunk_pkt1_end#") == -1:
                        skip_line = True
                        if int(trunk_pkt1_count) < int(num_trunk_pkt1):
                            in_file.seek(trunk_pkt1_start_line)
                            trunk_pkt1_count += 1
                        else:
                            trunk_pkt1_count = 1

                    if not line.find("#separate_oam_mgt_rules_start#") == -1:
                        skip_line = True
                        rules_start_line = in_file.tell()
                        sbc_rules_index = 0

                    if not line.find("#separate_oam_mgt_rules_end#") == -1:
                        skip_line = True
                        if separate_oam_mgt_rules and sbc_rules_index == 0:
                            in_file.seek(rules_start_line)
                            sbc_rules_index += 1

                    if not line.find("#separate_oam_ha_rules_start#") == -1:
                        skip_line = True
                        rules_start_line = in_file.tell()
                        sbc_rules_index = 0

                    if not line.find("#separate_oam_ha_rules_end#") == -1:
                        skip_line = True
                        if separate_oam_ha_rules and sbc_rules_index == 0:
                            in_file.seek(rules_start_line)
                            sbc_rules_index += 1

                    if not line.find("#sbc_alt_ip") == -1:
                        if len(pkt_alt_ips) > 0:
                            # There are alt IPs, and we're in a section of the template concerning alt IPs.
                            if not line.find("#sbc_alt_ip_start#") == -1:
                                skip_line = True
                                alt_ip_start_line = in_file.tell()
                                sbc_alt_index = 0
                                sbc_alt_ip_index = 0

                            if not line.find("#sbc_alt_ip_end#") == -1:
                                skip_line = True
                                pai = pkt_alt_ips[sbc_alt_index]
                                if sbc_alt_ip_index + 1 < int(pai.num_ips):
                                    in_file.seek(alt_ip_start_line)
                                    sbc_alt_ip_index += 1
                                elif sbc_alt_index + 1 < len(pkt_alt_ips):
                                    in_file.seek(alt_ip_start_line)
                                    sbc_alt_index += 1
                                    sbc_alt_ip_index = 0

                            if not line.find("#sbc_alt_ip#") == -1:
                                pai = pkt_alt_ips[sbc_alt_index]
                                line = line.replace('#sbc_alt_ip#', '')
                                line = line.replace('${alt_if_key}', pai.alt_if_key)
                                line = line.replace('${alt_if_name}', pai.alt_if_name)
                                line = line.replace('${alt_ip_value}', pai.alt_if_ips[sbc_alt_ip_index])
                                line = line.replace('${opt_alt_index0}', opt_alt_index0)
                                line = line.replace('${opt_alt_index1}', opt_alt_index1)
                                line = line.replace('${alt_if_index}', str(pai.alt_index))
                                line = line.replace('${alt_if_ip_index}', str(sbc_alt_ip_index))
                        else:
                            skip_line = True

                    if not line.find("#sbc_param_start#") == -1:
                        skip_line = True
                        param_start_line = in_file.tell()
                        sbc_param_index = 0

                    if not line.find("#sbc_param_end#") == -1:
                        skip_line = True
                        if sbc_param_index + 1 < len(custom_parm_list):
                            in_file.seek(param_start_line)
                            sbc_param_index += 1

                    if not line.find("#sbc_param#") == -1:
                        if len(custom_parm_list) > 0:
                            line = line.replace("#sbc_param#", "")
                            line = line.replace("${sbc_param_name}", custom_parm_list[sbc_param_index].name)
                            line = line.replace("${sbc_param_default}", custom_parm_list[sbc_param_index].default)
                            line = line.replace("${sbc_param_desc}", custom_parm_list[sbc_param_index].desc)
                            if  line.find("${sbc_param_editableState}") != -1:
                                if  custom_parm_list[sbc_param_index].editableState != "":
                                    line = line.replace("${sbc_param_editableState}", custom_parm_list[sbc_param_index].editableState)
                                else:
                                    skip_line = True
                        else:
                            skip_line = True

                    if not line.find("#ha_ext#") == -1:
                        if ha_io_type == "virtio":
                            skip_line = True
                        else:
                            line = line.replace("#ha_ext#", "")
                    if not line.find("#ha_int#") == -1:
                        if ha_io_type == "virtio":
                            line = line.replace("#ha_int#", "")
                        else:
                            skip_line = True

                    if not line.find("#pkt_oam_ext#") == -1:
                        if pkt_oam_io_type == "virtio":
                            skip_line = True
                        else:
                            line = line.replace("#pkt_oam_ext#", "")
                    if not line.find("#pkt_oam_int#") == -1:
                        if pkt_oam_io_type == "virtio":
                            line = line.replace("#pkt_oam_int#", "")
                        else:
                            skip_line = True

                    if not line.find("#dns#") == -1:
                        if dns_config_dic == []:
                            skip_line = True
                        else:
                            line = line.replace("#dns#", "")
                            if not line.find('${') == -1:
                                #We have several lines in the form intf_key, 
                                #e.g. ${mgt_server_ip} which relates to the value
                                #from the dns file. So we work out which one this is for
                                #here (rather than having loads of separate options)
                                param = line[line.find('${')+len('${') : line.rfind('}')]
                                intf = param.split('_', 1)[0] #e.g. mgt
                                key = param.split('_', 1)[1]  #e.g fqdn
                                try:
                                    val = dns_config_dic[intf][key]
                                    line = line.replace('${'+param+'}', val)
                                except KeyError:
                                    # Interface does not have
                                    # configuration in the DNS file
                                    skip_line = True
                                except Exception as e:
                                    print("FAILURE: Failed to get %s value for interface %s. Exception: %s" % (key,intf, str(e)))
                                    return False

                    if not line.find("#instantiation_start#") == -1:
                        skip_line = True
                        if scaling:
                            instantiation_level_start = in_file.tell()
                            instantiation_level = 1

                    if not line.find("#instantiation_end#") == -1:
                        skip_line = True
                        if scaling:
                            if instantiation_level < num_instantiation_lvls:
                                in_file.seek(instantiation_level_start)
                                instantiation_level += 1
                            else:
                                instantiation_level = 0

                    if not line.find("#scaling#") == -1:
                        if scaling:
                            line = line.replace("#scaling#", "")
                            line = line.replace("${sbc_scale_level}", max_scale_level)
                            line = line.replace("${sbc_deltas}", deltas)
                            line = line.replace("${list_of_sbc_rgs}", list_of_sbc_rgs)
                            line = line.replace("${initial_instantiaition_vms}", min_sbc_instances)
                            line = line.replace("${instantiation_level}", str(instantiation_level))
                            line = line.replace("${instantiation_level_desc}", str(instantiation_level) if ha_model == "n_1_ha" else str(instantiation_level+1))
                            line = line.replace("${scale_number_instances}", str(get_instantiation_instances(instantiation_level, ha_model, rg_actives)))
                            line = line.replace("${aspect_scale_level}", str(get_aspect_scale_level(instantiation_level, ha_model, rg_actives)))
                        else:
                            skip_line = True

                    global affinity_across_rgs
                    line = line.replace("${affinity_across_rgs}", affinity_across_rgs)

                    if not line.find("#redundancy_ratio#") == -1:
                        line = line.replace("#redundancy_ratio#", "")
                        line = line.replace("{sbc_redundancy_ratio}",redund_ratio)

                    line = line.replace("${tosca_definitions_version}", tosca_def_v)
                    line = line.replace("${template_version}", template_version)
                    line = line.replace("${oam_pkt0_priv_sub}", "private_network_oam_pkt0" if oam_ip_vers_pkt_var != ip_vers_pkt0_var else "private_network_pkt0")
                    line = line.replace("${oam_pkt1_priv_sub}", "private_network_oam_pkt1" if oam_ip_vers_pkt_var != ip_vers_pkt1_var else "private_network_pkt1")
                    if not line.find("${descriptor_id}") == -1:
                        uuid_personality = "{0:0<4}".format(vt)[0:4]
                        uuid_version = prod_version.lstrip('V0')
                        if uuid_version.startswith('9.0'):
                            uuid_version = "0" + uuid_version #requires a length of 8 characters including '.'(xx.xx.xx)
                        uuid_subversion = "{0:0<4}".format(sub_version)[0:4]
                        encode_hex = codecs.getencoder('hex')
                        vnfd_uuid = uuid.UUID(encode_hex((uuid_personality+uuid_version+uuid_subversion).encode())[0].decode())
                        line = line.replace("${descriptor_id}", str(vnfd_uuid))

                    if not line.find("#oam_pkt0_ip_ver#") == -1:
                        if oam_ip_vers_pkt_var != ip_vers_pkt0_var:
                            line = line.replace("#oam_pkt0_ip_ver#", "")
                        else:
                            skip_line = True
                    if not line.find("#oam_pkt1_ip_ver#") == -1:
                        if oam_ip_vers_pkt_var != ip_vers_pkt1_var:
                            line = line.replace("#oam_pkt1_ip_ver#", "")
                        else:
                            skip_line = True

                    if not line.find("${pkt0_trunk_mode}") == -1:
                        if num_trunk_pkt0 == 0:
                            line = line.replace("${pkt0_trunk_mode}", "false")
                        else:
                            line = line.replace("${pkt0_trunk_mode}", "true")

                    if not line.find("${pkt1_trunk_mode}") == -1:
                        if num_trunk_pkt1 == 0:
                            line = line.replace("${pkt1_trunk_mode}", "false")
                        else:
                            line = line.replace("${pkt1_trunk_mode}", "true")

                    if not line.find("#trunk_pkt0#") == -1:
                        if not num_trunk_pkt0 == 0:
                            line = line.replace("#trunk_pkt0#", "")
                            if not line.find("${SubPort0_segmentation_id}") == -1:
                                line = line.replace("${SubPort0_segmentation_id}", num_pkt0_vlan[ ((sbc_rg_count-first_rg_index)* num_trunk_pkt0)+trunk_pkt0_count-first_pkt0_trunk_index])
                            if not line.find("${default_SubPort0_segmentation_id}") == -1:
                                line = line.replace("${default_SubPort0_segmentation_id}", str(default_SubPort0_segmentation_id))
                        else:
                            skip_line = True

                    if not line.find("#trunk_pkt1#") == -1:
                        if not num_trunk_pkt1 == 0:
                            line = line.replace("#trunk_pkt1#", "")
                            if not line.find("${SubPort1_segmentation_id}") == -1:
                                line = line.replace("${SubPort1_segmentation_id}", num_pkt1_vlan[ ((sbc_rg_count-first_rg_index)* num_trunk_pkt1)+trunk_pkt1_count-first_pkt1_trunk_index])
                            if not line.find("${default_SubPort1_segmentation_id}") == -1:
                                line = line.replace("${default_SubPort1_segmentation_id}", str(default_SubPort1_segmentation_id))
                        else:
                            skip_line = True

                    if (not line.find("cloud-initdata.cfg") == -1) and (not line.find("Scripts") == -1):
                        initdata_in_scripts = True
    
                    if not skip_line:
                        out_file.write(line)
    
                    line = in_file.readline()

    vnfd_hash = hash_file_sha256(vnfd_file)

    with open(os.path.join(files_dir, "ChangeLog.txt"), "w") as out_file:
        out_file.write('SBX 7.1 Features:\n')
        out_file.write(' SBX-3281  SWe: ARP probes for Link failure detection\n')
        out_file.write(' SBX-57464 Confd should not run as root\n')
        out_file.write(' SBX-58494 invalid v6 ip allowed when configuring remote audit server\n')
        out_file.write(' SBX-59457 Robust rsyslog implementation for at-scale CDR and TRC file transport\n')
        out_file.write(' SBX-59884 Upgrade to latest DPDK LTS release\n')
        out_file.write(' SBX-60609 Hyperthreading support on SWe\n')
        out_file.write(' SBX-60742 Cloud Provisioning Enhancement by using one of the SBC instances as SBC Configurator\n')
        out_file.write(' SBX-61551 JITC: Need to enhance .AUD logs to report additional information\n')
        out_file.write(' SBX-61552 JITC: Need to use cryptographic mechanisms to protect audit information at rest\n')
        out_file.write(' SBX-62393 Enable second 10GB packet PORT on SBC 5400\n')
        out_file.write(' SBX-62422 Remove default passwords and support two privilege levels with key injection\n')
        out_file.write(' SBX-63108 Replacement upgrade for 1:1 HA model\n')
        out_file.write(' SBX-63309 SBC SWe: Support for Intel x710 NIC\n')
        out_file.write(' SBX-63380 Permit SFTP outbound from SBC to a configurable destination TCP port\n')
        out_file.write(' SBX-64007 Support for syslog export of SSH console activity and Linux logs\n')
        out_file.write(' SBX-65323 VNFM Support: SBC support for a subset of ETSI Ve-Vnfm Interface\n')
        out_file.write(' SBX-65325 VNFM Support: CSAR based software packaging\n')
        out_file.write(' SBX-65788 SBC shall populate additional media stats in CDR - 5K/7K support\n')
        out_file.write(' SBX-66266 SWeNP enhancements\n')
        out_file.write(' SBX-67033 SBCaaS 2nd Management Port for SWe\n')
        out_file.write(' SBX-68781 VNFM DR Integration with SBC\n')
        out_file.write(' SBX-68782 Software Upgrade Support with VNFM\n')
        out_file.write('SBX 7.2 Features:\n')
        out_file.write(' SBX-50672 STIR-SHAKEN: Support for signing via Signature Server, Verification Server, and tagging\n')
        out_file.write(' SBX-62287 EVS passthrough with MRF VzW\n')
        out_file.write(' SBX-62919 SILK Transcode\n')
        out_file.write(' SBX-65508 Domain Locked License feature\n')
        out_file.write(' SBX-68156 VNFM SBC orchestration and Life cycle management\n')
        out_file.write(' SBX-70945 Cloud SBC provisioning Via Headend Node\n')
        out_file.write(' SBX-70226 PEM interworking VzW\n')
        out_file.write('SBX 10.1 Features:\n')
        out_file.write(' SBX-76047 SBC VNF Scaling(Horizontal)\n')
        out_file.write(' SBX-109853 SBC Support for SOL001 3.5.1\n')

    with open(os.path.join(tosca_dir, "TOSCA.meta"), "w") as out_file:
        out_file.write('TOSCA-Meta-File-Version: 1.0\n')
        out_file.write('CSAR-Version: 1.1\n')
        out_file.write('Created-by: Onboarding portal\n')
        out_file.write('ETSI-Entry-Definitions: Definitions/' + vt + '-vnfd.yaml\n')
        out_file.write('ETSI-Entry-Manifest: ' + vt + '.mf\n')
        out_file.write('ETSI-Entry-Change-Log: Files/ChangeLog.txt    #Edit to reflect actual path\n')
        out_file.write('ETSI-Entry-Certificate: Files/' + vt + '.cert\n')

    with open(os.path.join(output_dir, vt + ".mf"), "w") as out_file:
        out_file.write('metadata:\n')
        out_file.write('  vnfd_id: ' + str(vnfd_uuid) + '\n')
        out_file.write('  vnf_product_name: ' + vt + '\n')
        out_file.write('  vnf_provider_id: Ribbon\n')
        out_file.write('  vnf_package_version: ' + '\'3.0.0\'' + '\n')
        out_file.write('  vnf_software_version: ' + version_tag + version_suffix + '\n')
        out_file.write('  vnf_release_date_time: ' + date + '\n')
        out_file.write('  compatible_specification_versions: 2.7.1,3.5.1\n')
        out_file.write('  vnfm_info: Ribbon_VNFM\n')
        out_file.write('\n')
        out_file.write('Source: Definitions/' + vt + '-vnfd.yaml\n')
        out_file.write('Algorithm: SHA-256\n')
        out_file.write('Hash: ' + vnfd_hash + '\n')
        out_file.write('\n')
        if not qcow2_file == "":
            qcow2_basename = os.path.basename(qcow2_file)
            qcow2_hash = hash_file_sha256(qcow2_file)
            out_file.write('Source: Files/' + qcow2_basename + '\n')
            out_file.write('Algorithm: SHA-256\n')
            out_file.write('Hash: ' + qcow2_hash + '\n')
            print("Copying qcow2 image into CSAR directory")
            shutil.copy2(qcow2_file, files_dir)

    if initdata_in_scripts:
        initdata_dir = scripts_dir
    else:
        initdata_dir = files_dir

    with open(os.path.join(initdata_dir, vt + "-" + compact_version + "-cloud-initdata.cfg"), "w") as out_file:
        out_file.write('users:\n')
        if 'linuxadmin' not in userids:
            userids['linuxadmin'] = ['']
        if 'admin' not in userids:
            userids['admin'] = ['']
        for userid in userids:
            out_file.write('  - name: ' + userid + '\n')
            out_file.write('    lock_passwd: false\n')
            out_file.write('    ssh-authorized-keys:\n')
            for public_key in userids[userid]:
                if public_key == '':
                    public_key = sbc_public_key
                out_file.write('      - ' + public_key + '\n')
            out_file.write('\n')

        if passwd_set or timeout_shell != -1 or designer_debug:
            out_file.write('runcmd:\n')

        if passwd_set:
            out_file.write('    - usermod -p \'' + passwd_admin + '\' admin\n')
            out_file.write('    - usermod -p \'' + passwd_linuxadmin + '\' linuxadmin\n')

        if timeout_shell != -1:
            out_file.write("    - sed -i '/^TMOUT/s/TMOUT=[0-9].*/TMOUT=" + str(timeout_shell) + "/' /etc/profile.d/autologout.sh\n")

        if designer_debug:
            out_file.write("    - sed -i 's/^ClientAliveInterval.*[0-9]$/ClientAliveInterval 0/' /etc/ssh/sshd_config\n")
            out_file.write("    - sed -i 's/allowSshAccess=n/allowSshAccess=y/' /opt/sonus/conf/sbx.conf\n")
            out_file.write("    - echo -e '\\n# Avoid Password\\nlinuxadmin ALL=(ALL) NOPASSWD:ALL' >> /etc/sudoers\n")

        if cinder_log and cinder_boot:
            # If we have both cinder boot and log, assume log is vdb
            out_file.write('bootcmd:\n')
            if cinder_log_device != "":
                # For backwards compatibility with older release (where "-a" wasn't supported).
                out_file.write("    - /opt/sonus/bin/mountVolume.sh -v $(ls -l /dev/disk/by-id/virtio-* | grep -m1 " + cinder_log_device + " | sed 's+.*\\s/dev/disk/by-id/virtio-++' | sed 's+\\s.*++')")
            else:
                out_file.write("    - /opt/sonus/bin/mountVolume.sh -a")
        elif cinder_log:
            out_file.write('bootcmd:\n')
            out_file.write("    - /opt/sonus/bin/mountVolume.sh -v $(ls /dev/disk/by-id/virtio-* | grep -m1 virtio | sed 's+/dev/disk/by-id/virtio-++')")
            # Eventually (if "-d" is supported) change to:
            # out_file.write('    - /opt/sonus/bin/mountVolume.sh -d /dev/vdc')

    csar_file = os.path.join(csar_dir, vt_name + ".csar")
    print("Creating " + csar_file)
    try:
        import zipfile
        zipf = zipfile.ZipFile(csar_file, 'w', zipfile.ZIP_DEFLATED)
        for root, dirs, files in os.walk(output_dir):
            for d in dirs:
                # Write directories so even empty directories are copied:
                arcname = os.path.relpath(os.path.join(root, d), output_dir)
                # SBX-SBX-115711 - Do not compress folders.
                zipf.write(os.path.join(root, d), arcname, compress_type=zipfile.ZIP_STORED)
            for f in files:
                # SBX-111952 - SBC needs hidden files too for the successful upgrade.
                arcname = os.path.relpath(os.path.join(root, f), output_dir)
                zipf.write(os.path.join(root, f), arcname, compress_type=zipfile.ZIP_DEFLATED)

    except Exception as zip_e:
        if os.path.exists (csar_file):
            os.remove(csar_file)
        print("Could not import zipfile: " + str(zip_e) + ", using zip function instead")
        # SBX-111952 - SBC needs hidden files too for the successful upgrade.
        os.system("cd " + output_dir + ";zip -qr " + csar_file + " . ;cd " + current_dir)

    print("All files in csar are available in directory: " + os.path.join(csar_dir, vt_name))

    if os.path.exists(certificate_file) and os.path.exists(private_key_file):
        sig_file = os.path.join(csar_dir, vt_name + ".signature")
        print("Key and certificates files found, signing csar file " + sig_file)
        openssl_cmd = "openssl cms -sign -in " + csar_file + " -inkey " + private_key_file + " -signer " + certificate_file + " -out " + sig_file + " -outform PEM -nodetach"
        os.system(openssl_cmd)
    else:
        # print("Key (" + private_key_file + ") and certificate (" + certificate_file + ") files not found. Creating SHA instead.")
        csar_hash = hash_file_sha256(csar_file)
        with open(os.path.join(csar_dir, vt_name + ".sha256"), "w") as out_file:
            out_file.write(csar_hash + '\n')
        print("Saved CSAR SHA in " + os.path.join(csar_dir, vt_name + ".sha256"))


def xsbc_create_csars(prod_vers, sub_vers, vnfd_file, customtypes_file):

    global warning_msg
    vnftypes = [
        'isbc:simplex:virtio:dev_size:0',
        'isbc:active_standby:virtio:dev_size:0',
        'isbc:simplex:provider:dev_size:0',
        'isbc:active_standby:provider:dev_size:0',
        'isbc:active_standby:provider:std_size:0',
        'isbc:n_1_ha:provider:dev_size:2',
        'mrfp:active_standby:virtio:dev_size:0',
        'mrfp:n_1_ha:provider:dev_size:2',
        'msbc:n_1_ha:virtio:dev_size:2',
        'msbc:n_1_ha:provider:dev_size:2',
        'msbc:n_1_ha:sriov:dev_size:2',
        'msbc:n_1_ha:sriov:std_size:2',
        'msbc:active_standby:virtio:std_size:0',
        'msbc:active_standby:provider:dev_size:0',
        'msbc:active_standby:provider:std_size:0',
        'msbc:active_standby:sriov:dev_size:0',
        'msbc:active_standby:sriov:std_size:0',
        'msbc:active_standby:virtio:dev_size:0',
        'ssbc:n_1_ha:virtio:dev_size:2',
        'ssbc:n_1_ha:provider:dev_size:2',
        'ssbc:active_standby:virtio:std_size:0',
        'ssbc:active_standby:provider:dev_size:0',
        'ssbc:active_standby:provider:std_size:0',
        'ssbc:active_standby:sriov:dev_size:0',
        'ssbc:active_standby:virtio:dev_size:0',
        'tsbc:active_standby:sriov:dev_size:0',
        'tsbc:active_standby:virtio:dev_size:0',
        'tsbc:active_standby:provider:dev_size:0',
        'tsbc:active_standby:provider:std_size:0',
        'slb:active_standby:virtio:std_size:0',
        'slb:active_standby:provider:dev_size:0',
        'slb:active_standby:provider:std_size:0',
        'slb:active_standby:sriov:dev_size:0',
        'slb:active_standby:virtio:dev_size:0']

    if flavor_size_param != "all" and interface_param != "all" and ha_param != "all":
        # They only want specific values, so set them even if it's not a normal configuration
        vnftypes = [
            "isbc:" + ha_param + ":" + interface_param + ":" + flavor_size_param + ":0",
            "msbc:" + ha_param + ":" + interface_param + ":" + flavor_size_param + ":0",
            "ssbc:" + ha_param + ":" + interface_param + ":" + flavor_size_param + ":0",
            "tsbc:" + ha_param + ":" + interface_param + ":" + flavor_size_param + ":0",
            "mrfp:" + ha_param + ":" + interface_param + ":" + flavor_size_param + ":0",
            "slb:" + ha_param + ":" + interface_param + ":" + flavor_size_param + ":0"]

    num_csar_created = 0
    no_csar_warning = ""

    # Start clean to avoid any stale files
    vnfd_dir = os.path.abspath(os.path.dirname(vnfd_file))
    if vnfd_dir == "":
        vnfd_dir = "."
    csar_dir = os.path.join(vnfd_dir, "CSAR")
    if not os.path.exists(csar_dir):
        os.mkdir(csar_dir)
    print("Storing CSAR files in " + csar_dir)

    for vnft in vnftypes:
        vnft_list = vnft.split(':')
        vt = vnft_list[0]
        ha_model = vnft_list[1]
        io_type = vnft_list[2]
        flavor_size = vnft_list[3]
        min_num_oam = vnft_list[4]

        # Compare the global num_oam_instances to the local min_num_oam (for this csar).
        # Use whichever is greater.
        if int(num_oam_instances) >= int(min_num_oam):
            num_oam = num_oam_instances
        else:
            num_oam = min_num_oam

        if io_type == "virtio" and default_pkt_ip_vers == "ipv6":
            if interface_param == "virtio":
                # They explicitly asked for virtio and ipv6:
                no_csar_warning = "Virtio currently not supported with ipv6"
            continue

        if flavor_size_param != flavor_size and flavor_size_param != "all":
            # Not the matching flavor size
            continue

        if interface_param != io_type and interface_param != "all":
            # Not the matching interface type
            continue

        if ha_param != ha_model and ha_param != "all":
            # Not the matching ha ha_model
            continue

        if personality_param != vt and personality_param != "all":
            # Not the matching personality type
            continue

        # Map the standard size to "large" or "extra_large".
        if flavor_size == "std_size":
            if vt == "ssbc":
                actual_size = "extra_large"
            else:
                actual_size = "large"
        else:
            actual_size = flavor_size

        if ha_model == "active_standby" and int(num_sbc_rg) > 1 and int(num_oam) == 0:
            warning_msg += "Warning: Multiple redundancy groups for HA mode 'ha' has not been validated. Problems with this CSAR may occur.\n\n"

        xsbc_create_csar(prod_vers, sub_vers, vnfd_file, customtypes_file, csar_dir, vt, ha_model, io_type, actual_size, num_oam)
        num_csar_created += 1

        # Add certain warnings after successful creation:
        if (ha_model == "n_1_ha") and (int(num_oam) == 0):
            sys.stderr.write("Warning: N:1 HA mode requires OAM VMs. This CSAR may not work.\n\n")
        if (ha_model == "active_standby") and (int(num_oam)) > 0 and not scaling:
            sys.stderr.write("Warning: VNFs with OAM require N:1 HA mode. This CSAR may not work.\n\n")
        if warning_msg != "":
            sys.stderr.write(warning_msg)
            warning_msg = ""

    if num_csar_created == 0:
        # There are a few reasons we might fail to create any CSARs.
        # Try to give a meaningful error message.
        if no_csar_warning != "":
            sys.stderr.write(no_csar_warning + "\n")
        elif flavor_size_param == "all":
            sys.stderr.write("Warning: no matching CSARs created. Please specify: -f [dev|std]")
        elif interface_param == "all":
            sys.stderr.write("Warning: no matching CSARs created. Please specify: -i [virtio|sriov|provider]")
        elif ha_param == "all":
            sys.stderr.write("Warning: no matching CSARs created. Please specify: -h [simplex|ha|n_1_ha]")
        else:
            # Shouldn't happen, but just in case issue a vague warning:
            sys.stderr.write("Warning: no matching CSARS created. Please check for incompatible parameters\n")


# MAIN
if __name__ == "__main__":
    print("Script Dir:" + script_dir)
    if not os.path.exists(vnfd_proprietary_template):
        vnfd_proprietary_template = os.path.join(os.path.dirname(script_dir), "templates", "vnfmVnfdTemplate.yaml")
    if not os.path.exists(vnfd_sol001_template):
        vnfd_sol001_template = os.path.join(os.path.dirname(script_dir), "templates", "vnfmSol001VnfdTemplate.yaml")
    if not os.path.exists(vnfd_old_sol001_template):
        vnfd_old_sol001_template = os.path.join(os.path.dirname(script_dir), "templates", "vnfm_pre_21_2_Sol001VnfdTemplate.yaml")
    if not os.path.exists(vnfd_ribbon_custom_types_template):
        vnfd_ribbon_custom_types_template = os.path.join(os.path.dirname(script_dir), "templates", "vnfmSol001VnfdCustomTypes.yaml")

    if "ORCA_ROOT" in os.environ:
        sbc_tar_file = str(glob.glob(os.path.join(os.environ['ORCA_ROOT'], 'rel', 'sbc-V*.tar.gz')))
        tar_split = sbc_tar_file.split('-')
        if len(tar_split) > 2:
            prod_version = tar_split[1].lstrip('V0')
            sub_version = tar_split[2].split('.')[0]

    ret = get_args(sys.argv)

    if vnfd_template == "":
        vnfd_template = vnfd_sol001_template if not old_vnfm else vnfd_old_sol001_template

    if vnfd_custom_types_file == "":
        vnfd_custom_types_file = vnfd_ribbon_custom_types_template

    if ret:
        if sub_version == "":
            print("Product Version:" + prod_version)
        else:
            print("Product Version:" + prod_version + "Sub-Version:" + sub_version)
        print("VNFD Template:" + vnfd_template)

        xsbc_create_csars(prod_version, sub_version, vnfd_template, vnfd_custom_types_file)

