#!/bin/bash

#############################################################
#
# Copyright (c) 2019 Ribbon Communications Inc.
#
# All Rights Reserved.
# Confidential and Proprietary.
#
# HFE_AZ.sh
#
# Module Description:
# Script to enable HFE(High-Availability Front End) instance as frontend for 
# PKT ports of SBC.
# Call "HFE_AZ.sh setup" - to configure HFE with Secondary or Primary IPs
# The script will perform the following steps when called from cloud-init (setup function):
# 1) Save and clear old iptables rules : preConfigure
# 2) Enable IP Forwarding :configureNATRules
# 3) Read variables from natVars.input/natVars.user configured from customdata: readConfig
# 4) Determine interface names and GW IPs on HFE: setInterfaceMap
# 5) Extract the interface IPs required to configure the HFE for pkt0/pkt1
#    ports: extractHfeConfigIps
# 6) Setup DNATs, SNATs and routes to route traffic through HFE to/from SBC PKT ports: configureNATRules
# 7) Configure Management interface to HFE on eth2: configureMgmtNAT
# 8) Configure customer specified routes: addCusomterRoutes
# 9) Log applied iptables configuration and routes: showCurrentConfig 
#10) Check connectivity to currently Active SBC using ping and switchover to Standby SBC
#    if connection is lost to current Active: checkSbcConnectivity
#
#
# Call "HFE_AZ.sh cleanup" to clear all IP tables and Routes on the HFE.
# 1) Save and clear old iptables rules : preConfigure
# 2) Read variables from natVars.input configured from userdata: readConfig
# 3) Determine interface names and GW IPs on HFE: setInterfaceMap
# 4) Extract the interface IPs required to configure the HFE for pkt0/pkt1
#    ports: extractHfeConfigIps
# 5) Route clean up for Pkt0/Pkt1 subnet and remote machine's public IP: routeCleanUp
#
# This option is useful to debug connectivity of end-point with HFE, after
# calling this no packet is forwarded to SBC, user can ping IP on ens4 to
# make sure connectivity between end-point and HFE is working fine.
# Once this is done user MUST reboot HFE node to restore all forwarding rules
# and routes.
# 
#
# NOTE: This script is run by cloud-init or systemd in HFE instance.
#
# This script should be uploaded to a container in an Azure Storage Account
# HFE Node user data then downloads and runs with cloud-init.
# See Ribbon Documentation for more information.
#
#
#############################################################

declare -A interfaceMap
declare -A commandMap

HFERoot="/opt/HFE"
varFile="$HFERoot/natVars.input"
userVarFile="$HFERoot/natVars.user"
confLogFile="$HFERoot/log/HFE_conf.log"
statusLogFile="$HFERoot/log/HFE.log"
oldRules="$HFERoot/iptables.rules.prev"
hfeLogRotateConf="/etc/logrotate.d/hfe"
APP_INSTALL_MARKER="$HFERoot/app_install_marker"
tempMgtRouteFile="$HFERoot/.tempRoute"
TMP_LOG_DIR="/tmp/sysdump"

IPTABLES="$(command -v iptables)"
CURL="$(command -v curl)"
DHCLIENT="$(command -v dhclient)"

ACTIVE_SBC_IP_ARR=""
STANDBY_SBC_IP_ARR=""
HFE_ETH0_IP_ARR=""
ACTIVE_SBC_IP_PKT1_ARR=""
STANDBY_SBC_IP_PKT1_ARR=""
HFE_ETH1_IP_ARR=""
MODE=""
gRetVal=""
readonly INTF_TYPE_MASTER=2
readonly INTF_TYPE_SLAVE=1
readonly INTF_TYPE_NORMAL=0
readonly MAX_QUEUE_LENGTH=8192

PRIMARY_IP=0

RG_NAME=""    # Resource Group Name
SUB_ID=""     # Subscription ID

# PRIMARY IP DNS VARS
ENABLE_PKT_DNS_QUERY=0
declare -A primaryIpMap
PRIMARY_FILE="$(mktemp)"

# REST VARS 
apiversion="api-version=2018-10-01"
# resource and AZURE_URI var here is set to default azure uri, later at readConfig 
# the right uri is read/sourced from the input file and updated this var. 
AZURE_URI="management.azure.com"
resource="resource=https://$AZURE_URI/"

#netplan vars
NETPLAN_ROOT=/etc/netplan
HFE_NETPLAN_FILE=$NETPLAN_ROOT/hfe-netplan.yaml
CLOUD_INIT_NETPLAN=$NETPLAN_ROOT/50-cloud-init.yaml
macsArr=( )
hfeIntfIpsArr=( )

# custom routes
CUSTOM_ROUTES_ARR=( )

# HA related 
initPubIp=""
initSndIp=""
initSndIpCfg=""
HA_MODE=0
HA_SWITCHOVER_RETRY=2


DEBUG_MODE=0
INIT_LOG=1

PROG=${0##*/}

usage()
{
    echo $1
    echo "usage: $PROG <setup | cleanup | sysdump>"
    echo "Example:"
    echo "$PROG setup - Setup HFE with Secondary IPs" 
    echo "$PROG cleanup - Cleanup IP tables and route on HFE" 
    echo "$PROG sysdump - Collect logs used for debugging HFE" 
    exit
}

timestamp()
{
 date +"%Y-%m-%d %T"
}

timestampDebug()
{
 date +"%F %T,%3N"
}

loggerHfe()
{
    echo $(timestamp) "$1" >> $confLogFile 2>&1
}

statusLoggerHfeDebug()
{
    if [ $DEBUG_MODE -eq 1 ]; then
        echo $(timestamp) "$1" >> $statusLogFile 2>&1
    fi
}

statusLoggerHfe()
{
    echo $(timestamp) "$1" >> $statusLogFile 2>&1
}

doneMessage()
{
    loggerHfe " =========================    DONE HFE_AZ.sh     =========================================="
    exit 
}

errorAndExit()
{
    loggerHfe " Error: $1"
    doneMessage
}


clearOldIptableRules()
{
    # reset the default policies in the filter table.
    $IPTABLES -P INPUT ACCEPT
    $IPTABLES -P FORWARD ACCEPT
    $IPTABLES -P OUTPUT ACCEPT

    # reset the default policies in the nat table.
    $IPTABLES -t nat -P PREROUTING ACCEPT
    $IPTABLES -t nat -P POSTROUTING ACCEPT
    $IPTABLES -t nat -P OUTPUT ACCEPT

    # reset the default policies in the mangle table.
    $IPTABLES -t mangle -P PREROUTING ACCEPT
    $IPTABLES -t mangle -P POSTROUTING ACCEPT
    $IPTABLES -t mangle -P INPUT ACCEPT
    $IPTABLES -t mangle -P OUTPUT ACCEPT
    $IPTABLES -t mangle -P FORWARD ACCEPT

    # flush all the rules in the filter and nat tables.
    $IPTABLES -F
    $IPTABLES -t nat -F
    $IPTABLES -t mangle -F

    # erase all chains that's not default in filter and nat table.
    $IPTABLES -X
    $IPTABLES -t nat -X
    $IPTABLES -t mangle -X
}

setAuthorizationHead()
{
    token=$($CURL "http://169.254.169.254/metadata/identity/oauth2/token?$apiversion&$resource" -H Metadata:true | $JQ .access_token | sed 's/"//g')
    authhead="Authorization:Bearer $token"
}

checkIfAuthFailed()
{
    resp="$1"

    errorCode="$(echo $resp | $JQ .error.code | sed 's/"//g')"
    if [[ $errorCode == "AuthorizationFailed" ]]; then
        return 1
    else
       return 0
    fi
}

getSubnetPrefix()
{
    local SBC_NAME=$1
    if [[ $2 == "PKT0" ]]; then
        IF="2"
    else
        IF="3"
    fi

    vmInstance=$($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Compute/virtualMachines/$SBC_NAME?$apiversion" -H "$authhead")
    networkInterfaceId=$(echo $vmInstance | $JQ .properties.networkProfile.networkInterfaces[$IF].id | sed 's/"//g')
    subnetId=$($CURL "https://$AZURE_URI$networkInterfaceId?$apiversion" -H "$authhead" | $JQ .properties.ipConfigurations[0].properties.subnet.id | sed 's/"//g')
    subnetPrefix=$($CURL "https://$AZURE_URI$subnetId?$apiversion" -H "$authhead" | $JQ .properties.addressPrefix | sed 's/"//g')
    if [[ $subnetPrefix == "null" ]]; then
        #IPv6 is configured in subnet
        addressPrefixes=$($CURL "https://$AZURE_URI$subnetId?$apiversion" -H "$authhead" | $JQ .properties.addressPrefixes)
        for i in $(seq 0 $(($(echo $addressPrefixes | $JQ length) - 1))); do
            if [[ $(echo $addressPrefixes | $JQ .[$i]) =~ .*:.* ]]; then
                #IPv6 address - ignore for now
                continue
            else
                subnetPrefix=$(echo $addressPrefixes | $JQ .[$i] | sed 's/"//g')
            fi
        done
    fi
    loggerHfe " Got Subnet Prefix for $2 - $subnetPrefix"
    echo $subnetPrefix
}

#$1 = Active SBC NAME
#$2 = PKT port
#$3 = Gateway
#$4 = Interface
configureSubnetRoute()
{
    subnetPrefix=$(getSubnetPrefix $1 $2)
    declare -g subnetPrefix_$2="$subnetPrefix"
    statusLoggerHfeDebug "set subnetPrefix_$2 as $subnetPrefix"
    $IP route | grep -E "$subnetPrefix.*$4"
    if [ $? -eq 0 ]; then
        loggerHfe " Route already availabe to $2 subnet"
    else
        $IP route add $subnetPrefix via $3 dev $4
        if [ $? -eq 0 ]; then
            loggerHfe " Set route to reach $2 subnet"
        else
            errorAndExit "Failed to set route to $subnetPrefix"
        fi
    fi
}

# $1 - IP / Subnet Prefix
# $2 - Gateway
# $3 - Interface
removeRoute()
{
    $IP route | grep -E "$1.*$2.*$3"
    if [ $? -eq 0 ]; then
        $IP route del $1 via $2 dev $3
        if [ $? -eq 0 ]; then
            loggerHfe " Route deleted to for $1 "
        else
            errorAndExit " Failed to delete route for $1 "
        fi
    else
        loggerHfe " Route not available for $1 "
    fi
}
# $1 - IP / Subnet Prefix
# $2 - Gateway
# $3 - Interface
verifyRoute()
{
    statusLoggerHfeDebug "verifyRoute $1 $2 $3"
    $IP route | grep -E "$1.*$3"
    if [ $? -ne 0 ]; then
        statusLoggerHfeDebug "Adding missing route for $1."
        $IP route add $1 via $2 dev $3
        return 1
    fi
    return 0
}

verifyRoutes()
{
    ret=0
    if [[ $MODE == "single" ]]; then
        verifyRoute $subnetPrefix_PKT0 $ETH3_GW $ETH3
        [ $? -eq 1 ] && ret=1
        verifyRoute $subnetPrefix_PKT1 $ETH4_GW $ETH4
        [ $? -eq 1 ] && ret=1
    else
        var="subnetPrefix_$SBC_PKT_PORT_NAME"
        verifyRoute "${!var}" $ETH2_GW $ETH2
        [ $? -eq 1 ] && ret=1
    fi

    IFS=',' read -ra SSH_IP_LIST <<< "$REMOTE_SSH_MACHINE_IP"
    for REMOTE_MACHINE_IP in "${SSH_IP_LIST[@]}"; do
        REMOTE_MACHINE_IP="$(echo $REMOTE_MACHINE_IP | sed 's~/32$~~')" #IP route will ignore /32. Therefore will fail to find it.
        verifyRoute $REMOTE_MACHINE_IP $MGMT_GW $MGMT
        [ $? -eq 1 ] && ret=1
    done

    for CUSTOM_ROUTE in "${CUSTOM_ROUTES_ARR[@]}"; do
        verifyRoute "$(echo $CUSTOM_ROUTE | awk -F'_' '{print $1}')" "$(echo $CUSTOM_ROUTE | awk -F'_' '{print $2}')" "$(echo $CUSTOM_ROUTE | awk -F'_' '{print $3}')"
        [ $? -eq 1 ] && ret=1
    done

    return $ret
}

routeCleanUp()
{
    loggerHfe " Route clean up for Pkt0 and Pkt1 IP and remote machine's public IP."

    if [[ $MODE == 'single' ]]; then 
        #PKT0
        pkt0SubnetPrefix=$(getSubnetPrefix $ACTIVE_SBC_VM_NAME "PKT0")
        removeRoute $pkt1SubnetPrefix $ETH3_GW $ETH3
        #PKT1
        pkt1SubnetPrefix=$(getSubnetPrefix $ACTIVE_SBC_VM_NAME "PKT1")
        removeRoute $pkt1SubnetPrefix $ETH4_GW $ETH4
    else
        subnetPrefix=$(getSubnetPrefix $ACTIVE_SBC_VM_NAME $SBC_PKT_PORT_NAME)
        removeRoute $subnetPrefix $ETH2_GW $ETH2
    fi

    ## Cleanup Ip route from Remote Machine to HFE
    IFS=',' read -ra SSH_IP_LIST <<< "$REMOTE_SSH_MACHINE_IP"
    for REMOTE_MACHINE_IP in "${SSH_IP_LIST[@]}"; do
        REMOTE_MACHINE_IP="$(echo $REMOTE_MACHINE_IP | sed 's~/32$~~')" #IP route will ignore /32. Therefore will fail to find it.
        removeRoute $REMOTE_MACHINE_IP $MGMT_GW $MGMT
    done

    # Clean up Custom routes
    IFS=',' read -ra CUSTOMER_ROUTES <<< "$CUSTOM_ROUTES"
    for ROUTE_INFO in "${CUSTOMER_ROUTES[@]}"; do
        DEST_IP="$(echo $ROUTE_INFO | awk -F'_' '{print $1}' | sed 's~/32$~~')"  #IP route will ignore /32. Therefore will fail to find it.
        INTF=${interfaceMap[$(echo $ROUTE_INFO | awk -F'_' '{print $2}')]}
        GW=$(getGatewayIp $INTF)
        removeRoute $DEST_IP $GW $INTF
    done
}

configurePrimaryIpMap()
{
    if [ $ENABLE_PKT_DNS_QUERY -eq 1 ]; then
        while read line; do
            key=$(echo $line | awk -F: '{print $1}')
            ip=$(echo $line | awk -F: '{print $2}')
            primaryIpMap[$key]=$ip
        done < $PRIMARY_FILE
    fi

    rm -f $PRIMARY_FILE
}

handleTempMgmtRoute()
{
    local addRoute=$1
    local causeStr="$2"
    [ -z $IP ] && IP="$(command -v ip)"
    if [ -e $tempMgtRouteFile ]; then
        source $tempMgtRouteFile
    else
        if [[ $MODE == "split" ]]; then
            tempMgtIntf=$ETH1
            [ -z $tempMgtIntf ] && tempMgtIntf="eth1"
        else #Posterity
            tempMgtIntf=$ETH2
            [ -z $tempMgtIntf ] && tempMgtIntf="eth2"
        fi
        tempMgtGw=$(getGatewayIp $tempMgtIntf)
    fi

    if [ $addRoute -eq 1 ]; then
        loggerHfe " $2 failed. Adding temporary default mgmt route to allow access."
        $IP route add 0.0.0.0/0 via $tempMgtGw dev $tempMgtIntf metric 10
        if [ $? -ne 0 ]; then 
            errorAndExit "Failed to add temporary default mgmt route"
        fi
    else
        $IP route | grep -E "default.*$tempMgtGw.*$tempMgtIntf"
        if [ $? -eq 0 ]; then
            loggerHfe " Removing temporary default mgmt route"
            source $tempMgtRouteFile
            $IP route delete 0.0.0.0/0 via $tempMgtGw dev $tempMgtIntf metric 10
            if [ $? -ne 0 ]; then
                errorAndExit "Failed to delete temporary mgmt route"
            fi
        fi
        # No longer need this file
        [ -e $tempMgtRouteFile ] && rm -f $tempMgtRouteFile
    fi
}

verifyPackages()
{
    local missingCmd=0

    commandMap["iptables"]=$IPTABLES
    commandMap["curl"]=$CURL
    commandMap["jq"]=$JQ
    commandMap["conntrack"]=$CONNTRACK
    commandMap["fping"]=$FPING
    commandMap["ip"]=$IP
    commandMap["dhclient"]=$DHCLIENT

    for i in "${!commandMap[@]}"
    do
        if [[ ${commandMap[$i]} == "" ]]; then
            loggerHfe "Required packages $i is missing."
            missingCmd=1
        fi
    done

    if [ $missingCmd -eq 1 ]; then
       errorAndExit "Missing required packages. Exiting!!!"
    fi

}

prepareHFEInstance()
{
    ### Configure ip_forward - Enable ip forwarding
    echo 1 >  /proc/sys/net/ipv4/ip_forward

    if [ ! -f $APP_INSTALL_MARKER ]; then
        ### Install required packages

        if [[ $(command -v yum) == "" ]]; then
            sudo apt-get update
            sudo apt-get -y install jq
            if [ $? -ne 0 ]; then
                handleTempMgmtRoute 1 "Repo update"
                sudo apt-get update
                sudo apt-get -y install jq
            fi
            sudo apt-get -y install conntrack
            sudo apt-get -y install fping
            if [[ -z "$DHCLIENT" ]]; then
                 sudo apt-get -y install isc-dhcp-client
                 # Check again for dhclient after installation
                 DHCLIENT="$(command -v dhclient)"
            fi

            # Kill auto package updates
            sudo systemctl stop apt-daily.service
            sudo systemctl stop apt-daily.timer
            sudo systemctl stop apt-daily-upgrade.timer
            sudo systemctl stop apt-daily-upgrade.service
            sudo systemctl disable apt-daily.service
            sudo systemctl disable apt-daily.timer
            sudo systemctl disable apt-daily-upgrade.timer
            sudo systemctl disable apt-daily-upgrade.service
        else
            curl -s --connect-timeout 10 'https://dl.fedoraproject.org/pub/epel/' > /dev/null
            if [ $? -ne 0 ]; then
                handleTempMgmtRoute 1 "Repo update"
            fi
            sudo yum -y update
            sudo yum -y install epel-release
            if [ $? -ne 0 ]; then
                majorOsVer=$(cat /etc/os-release | grep VERSION_ID | awk -F= '{print $2}' | sed 's/"//g' | sed 's/\..*//')
                sudo yum -y install wget
                cd /tmp/
                wget https://dl.fedoraproject.org/pub/epel/epel-release-latest-$majorOsVer.noarch.rpm
                epelRpm=$(ls epel*.rpm)
                sudo yum -y install $epelRpm
            fi
            sudo yum -y install jq
            sudo yum -y install conntrack
            sudo yum -y install fping
        fi
        setLogRotate
    fi

    JQ="$(command -v jq)"
    CONNTRACK="$(command -v conntrack)"
    FPING="$(command -v fping)"
    IP="$(command -v ip)"
    verifyPackages

    # Add timeout to curl
    CURL="$CURL --connect-timeout 10"

    # If we get this far, packages are good
    echo 1 > $APP_INSTALL_MARKER
}

preConfigure()
{
    if [ -f $confLogFile ]; then
        mv -f $confLogFile $confLogFile.prev
    fi

    ### Redirect all echo $(timestamp) to file after writing ip_forward
    loggerHfe " ==========================   Starting HFE_AZ.sh  ============================"
    loggerHfe " Enabled IP forwarding"
    loggerHfe " This script will setup DNAT, SNAT and IP forwarding."
    loggerHfe " Save old rules in $HFERoot/firewall.rules"

    ### Save old iptable rules
    sudo iptables-save > $oldRules
    if [ "$?" = "0" ]; then
        ### Clear the iptables
        clearOldIptableRules
    else
        errorAndExit "Cound not save old iptables rules. Exiting"
    fi
}

# Azure can supply the interfaces in any order
# in the metadata as long as it is not primary.
# This means the interface names will be wrong in 
# netplan - possibly affecting customer changes. Fix it here.
#
# Interfaces can also fail to come up. If they are not up
# use dhclient on the interface to bring it up and configure
fixInterfaces()
{
    if [ $HA_MODE -eq 0 ]; then    # Because of IP movement. We do not want this is HA mode.
        if [ -e $NETPLAN_ROOT ]; then
            if [ ! -e $HFE_NETPLAN_FILE ]; then
                intfNum=0
                # Disable cloudinit from reapplying
                echo "network: {config: disabled}" > /etc/cloud/cloud.cfg.d/99-disable-network-config.cfg

                # Currently cloudinit spits out the interfaces in order (eth0, eth1)
                # So for each MAC, update it with the MAC address for that interface
                # in that position. If this ever changes - we should consider swapping for python
                while IFS= read line; do
                    if [[ $line == *"macaddress:"* ]]; then
                        line=$(echo -E "$line" | sed "s/macaddress:.*/macaddress: ${macsArr[$intfNum]}/")
                        intfNum=$((intfNum+1))
                    fi
                    echo -E "$line" >> $HFE_NETPLAN_FILE
                done <<< $(cat $CLOUD_INIT_NETPLAN)
                if [ ${#macsArr[@]} -ne $intfNum ]; then
                    errorAndExit "Number of MACs updated ($intfNum) does not equal the amount interfaces for HFE (${#macsArr[@]})"
                fi
                sed -i 's/^/#/g' $CLOUD_INIT_NETPLAN    # Keep for prosperity
                netplan apply
                if [ $? -ne 0 ]; then
                    errorAndExit "Failed to apply netplan configuration"
                fi
                loggerHfe " Created and applied netplan config $HFE_NETPLAN_FILE. Rebooting to apply"
                reboot
                exit
            else
                loggerHfe " $HFE_NETPLAN_FILE already exists. Continuing"
            fi
        else
            loggerHfe " $NETPLAN_ROOT does not exist. Not creating netplan config"
        fi
    fi

 
    local dhclinetRun=0
    local routesfound=0
    local lastIntf=""
    while read intf; do
        $DHCLIENT $intf
        if [ $? -ne 0 ]; then
            errorAndExit "dhclient failed to bring up $intf"
        else
            loggerHfe " Brought up $intf using dhclient"
        fi
        dhclientRun=1
        lastIntf=$intf
    done < <($IP a | grep -E eth.: | grep DOWN | awk -F' ' '{print $2}' | sed 's/://')

    # Wait for everything to configured before continuing
    if [ $dhclinetRun -eq 1 ]; then
        for i in $(seq 0 4); do
            if [[ $(getGatewayIp $lastIntf) == "" ]]; then
                sleep 0.5s
            else
                return    # We are ready to continue
            fi
        done
        errorAndExit "Failed to find routes for $intf after 2 seconds. IP routes: \n\n $($IP route)"
    fi
}

getGatewayIp()
{
    local interface=$1
    cidrIp=$($IP route | grep "$interface proto kernel scope link" | awk -F " " '{print $1}' | awk -F "/" '{print $1}')
    finalOct=$(echo $cidrIp | awk -F "." '{print $4}')
    gwOctet=$(( finalOct + 1 ))
    gwIp=$(echo $cidrIp | awk -v var="$gwOctet" -F. '{$NF=var}1' OFS=.)
    echo $gwIp
}

setInterfaceMap()
{
    local vmTry=0
    local intfTry=0
    while [ $vmTry -lt 2 ]; do
        curlOut=$($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Compute/virtualMachines/$HFE_NAME?$apiversion" -H "$authhead")
        checkIfAuthFailed
        if [ $? -eq 1 ]; then
           loggerHfe " Authorization failed for virtualMachine $HFE_NAME. Sleeping 10s and retrying"
           sleep 10s
           setAuthorizationHead
           vmTry=$((vmTry + 1))
           continue
        fi
        interfaceIds=$(echo $curlOut | $JQ .properties.networkProfile.networkInterfaces)
        interfaceNum=$(echo $interfaceIds | $JQ length)

        for i in $(seq 0 $((interfaceNum-1))); do
            interfaceId=$(echo $interfaceIds | $JQ .[$i].id | sed 's/"//g')
            while [ $intfTry -lt 2 ]; do
                curlOut=$($CURL "https://$AZURE_URI$interfaceId?$apiversion" -H "$authhead")
                checkIfAuthFailed
                if [ $? -eq 1 ]; then
                    loggerHfe " Authorization failed for interfaceId. Sleeping 10s and retrying"
                    sleep 10s
                    setAuthorizationHead
                    intfTry=$((vmTry + 1))
                    continue
                fi
                hfeIntfIpsArr[$i]=$(echo $curlOut | $JQ .properties.ipConfigurations[0].properties.privateIPAddress | sed 's/"//g')
                macsArr[$i]="$(echo $curlOut | $JQ .properties.macAddress | sed 's/"//g' | sed 's/-/:/g')"
                break #break out of intfTry
            done
        done
        break # break out of vmTry
    done

    fixInterfaces   # We need interfaces up for matching correctly

    if [ ${#macsArr[@]} -ne $interfaceNum ] || [ ${#hfeIntfIpsArr[@]} -ne $interfaceNum ]; then
        errorAndExit "Number of MAC addresses (${#macsArr[@]} or HFE primary IPs (${#hfeIntfIpsArr[@]}) does not equal number of HFE interfaces ($interfaceNum)"
    fi

    for i in $(seq 0 $((interfaceNum-1))); do
        privIp="${hfeIntfIpsArr[$i]}"
        while read IF; do
            ifIp=$(ip address show $IF | grep -w inet | awk -F ' ' '{print $2}' | awk -F '/' '{print $1}')
                # handle multiple IPs
                if [[ $(echo $ifIp | grep -cw $privIp) -ne 0 ]]; then
                    interfaceMap["eth$i"]=$IF
                    break
                fi
        done < <($IP address show | egrep ^[0-9]: | awk -F ' ' '{print $2}' | sed 's/://' | grep -v lo)
    done


    ETH0=${interfaceMap['eth0']}
    ETH1=${interfaceMap['eth1']}
    ETH2=${interfaceMap['eth2']}
    if [[ $MODE == "single" ]]; then
        ETH3=${interfaceMap['eth3']}
        ETH4=${interfaceMap['eth4']}
    fi

    if [[ $MODE == "single" ]]; then
        # Gateway is second IP in CIDR
        ETH2_GW=$(getGatewayIp $ETH2)
        ETH3_GW=$(getGatewayIp $ETH3)
        ETH4_GW=$(getGatewayIp $ETH4)
        loggerHfe ""
        loggerHfe " Default GWs:"
        loggerHfe " ETH2_GW          $ETH2_GW"
        loggerHfe " ETH3_GW          $ETH3_GW"
        loggerHfe " ETH4_GW          $ETH4_GW"
    else
        # Gateway is second IP in CIDR
        ETH1_GW=$(getGatewayIp $ETH1)
        ETH2_GW=$(getGatewayIp $ETH2)
        loggerHfe ""
        loggerHfe " Default GWs:"
        loggerHfe " ETH1_GW          $ETH1_GW"
        loggerHfe " ETH2_GW          $ETH2_GW"
    fi

    #Set the MGMT interface information
    if [[ $MODE == "single" ]]; then
        MGMT=$ETH2
        MGMT_GW=$ETH2_GW
    else
        MGMT=$ETH1
        MGMT_GW=$ETH1_GW
    fi
}

readConfig()
{
    source $varFile
    
    # In Azure, updating of custom data is not supported! 
    # Allow customers to use a file to update any natVars
    if [ -e $userVarFile ]; then
        source $userVarFile
    fi

    SBC_PKT_PORT_NAME=$(echo $SBC_PKT_PORT_NAME | awk '{print toupper($0)}') #Force UpperCase
    SWITCHOVER_MODE=$(echo $SWITCHOVER_MODE | awk '{print toupper($0)}') #Force UpperCase
    
    loggerHfe " Cloud NAT Vars:"
    if [[ $SBC_PKT_PORT_NAME == "" ]]; then
        MODE="single"
    else
        MODE="split"
        loggerHfe " SBC_PKT_PORT_NAME $SBC_PKT_PORT_NAME"
    fi

    loggerHfe " ACTIVE_SBC_VM_NAME $ACTIVE_SBC_VM_NAME"
    loggerHfe " STANDBY_SBC_VM_NAME $STANDBY_SBC_VM_NAME"
    loggerHfe " REMOTE_SSH_MACHINE_IP $REMOTE_SSH_MACHINE_IP"
    loggerHfe " ENABLE_PKT_DNS_QUERY $ENABLE_PKT_DNS_QUERY"
    loggerHfe " CUSTOM_ROUTES $CUSTOM_ROUTES"
    loggerHfe " PEER_HFE_NAME $PEER_HFE_NAME"
    loggerHfe " SWITCHOVER_MODE $SWITCHOVER_MODE"
    loggerHfe " ROUTE_TABLE_NAME $ROUTE_TABLE_NAME"
    loggerHfe " AZURE_URI $AZURE_URI"
    loggerHfe ""

    resource="resource=https://$AZURE_URI/"

    if [[ $MODE == "single" ]]; then
        loggerHfe " SBC_PKT_PORT_NAME is not configured. Assuming single HFE instance with 5 interfaces"
        loggerHfe ""
    fi
}


## Parameters passed in to configureNATRules
## $1 - Interface for Public/Private connection to HFE
## $2 - Interface to Pkt0/Pkt1 to Active SBC
## $3 - Gateway for Pkt0/Pkt1 to Active SBC
## $4 - Print configuration messages to HFE log
configureNATRules()
{
    ## Set forward ACCEPT rule for packets coming into HFE
    $IPTABLES -I FORWARD -i $1 -o $2 -j ACCEPT
    if [ $? -eq 0 ]; then
        [[ ! -z $4 ]] && loggerHfe " Set Forward ACCEPT rule all packets coming from outside $1 to $2 towards SBC"
    else
        errorAndExit "Failed to set forward ACCEPT rule for all packets coming on IP($1)"
    fi

    ## Set forward ACCEPT rule for packets coming from SBC
    $IPTABLES -I FORWARD -i $2 -o $1 -j ACCEPT
    if [ $? -eq 0 ]; then
        [[ ! -z $4 ]] && loggerHfe " Set Forward ACCEPT rule all packets coming from SBC ($2) to $1"
    else
        errorAndExit "Failed to set ACCEPT rule all packets coming from SBC ($2) to $1"
    fi

    if [[ $1 == $ETH0 ]]; then
        sbcIpCount="${#ACTIVE_SBC_IP_ARR[@]}"
        hfeIpCount="${#HFE_ETH0_IP_ARR[@]}"
        addNumRoute=$(( hfeIpCount < sbcIpCount ? hfeIpCount : sbcIpCount ))
    else
        sbcIpCount="${#ACTIVE_SBC_IP_PKT1_ARR[@]}"
        hfeIpCount="${#HFE_ETH1_IP_ARR[@]}"
        addNumRoute=$(( hfeIpCount < sbcIpCount ? hfeIpCount : sbcIpCount ))
    fi

    
    for (( idx=0; idx<$addNumRoute; idx++ ))
    do
        if [[ $1 == $ETH0 ]]; then
            hfeIp=${HFE_ETH0_IP_ARR[$idx]}
            sbcIp=${ACTIVE_SBC_IP_ARR[$idx]}
        else
            hfeIp=${HFE_ETH1_IP_ARR[$idx]}
            sbcIp=${ACTIVE_SBC_IP_PKT1_ARR[$idx]}
        fi

        ## Set DNAT for detination IP
        $IPTABLES -t nat -I PREROUTING -d $hfeIp -j DNAT --to $sbcIp
        if [ $? -eq 0 ]; then
            [[ ! -z $4 ]] && loggerHfe " Set up proper DNAT for destination IP $hfeIp to offset $sbcIp."
        else
            errorAndExit "Failed to set DNAT rule for destination IP $hfeIp to offset $sbcIp."
        fi

        ## Set SNAT for external interface to HFE
        $IPTABLES -t nat -I POSTROUTING -o $1 -s $sbcIp -j SNAT --to $hfeIp
        if [ $? -eq 0 ]; then
            [[ ! -z $4 ]] && loggerHfe " Set up POSTROUTING rule (source IP $sbcIp, to offset $hfeIp) for packet sent on $1"
        else
            errorAndExit "Failed to set POSTROUTING rule (source IP $sbcIp, to offset $hfeIp) for packet sent on $1"
        fi
    done

    # Handle DNS requests on PKT ports.
    # This is done using Primary IPs, so create SNAT rules for primary IPs
    if [ $ENABLE_PKT_DNS_QUERY -eq 1 ]; then 
        if [[ $1 == $ETH0 ]]; then
            primarySbcIp=${primaryIpMap["$ACTIVE_SBC_VM_NAME"]}
        else
            primarySbcIp=${primaryIpMap["$ACTIVE_SBC_VM_NAME""-PKT1"]}
        fi
        primaryHfeIp=${primaryIpMap["HFE_$1"]}

        $IPTABLES -t nat -I POSTROUTING -o $1 -s $primarySbcIp -j SNAT --to $primaryHfeIp
        if [ $? -eq 0 ]; then
            [[ ! -z $4 ]] && loggerHfe " Set up primary IP POSTROUTING rule (source IP $primarySbcIp, to offset $primaryHfeIp) for packet sent on $1"
        else
            errorAndExit "Failed to set primary IP POSTROUTING rule (source IP $primarySbcIp, to offset $primaryHfeIp) for packet sent on $1"
        fi
    else
        [[ ! -z $4 ]] && loggerHfe " ENABLE_PKT_DNS_QUERY not enabled. Not forwarding primary PKT IP"
    fi
}

###################################################################################################
####                                         WARNING
####
####                Each call of this function will result in pkt drop.
####
#### Each conntrack flush operation has a associated time penalty (pkt drop), call this function
#### only when you are all set with setting up new iptables rules (and old rules are cleaned up).
####
####
####
###################################################################################################
clearOldRules()
{   
    ## Reset connection tracking
    ## Any packet received on input interfaces before NAT rules are set are not forwarded
    ## to sbc, connection tracking will not forward those packets even
    ## if NAT rules are set after receiving first packet from that source
    ## IP/Port as it has cache entry for source IP and Port.
    ## Reset connection tracking will treat them as new stream and those packets
    ## will be forwarded to SBC once SNAT and DNAT rules are setup
    ## properly.

    $CONNTRACK -F conntrack
    if [ $? -eq 0 ]; then
       [[ ! -z $1 ]] && loggerHfe " Flushing connection tracking rules."
    else
       [[ ! -z $1 ]] && loggerHfe " (WARNING):Flushing connection tracking rules failed."
    fi
}

# Add customer specific routes to specific interfaces
# Example values would be:
#    1.1.1.1_eth1,2.2.2.2_eth2,3.3.3.3_eth3
addCustomerRoutes()
{
    if [ -z $CUSTOM_ROUTES ]; then
        loggerHfe " No custom routes supplied. Continuing"
    else
        IFS=',' read -ra CUSTOMER_ROUTES <<< "$CUSTOM_ROUTES"
        for ROUTE_INFO in "${CUSTOMER_ROUTES[@]}"; do
            DEST_IP="$(echo $ROUTE_INFO | awk -F'_' '{print $1}' | sed 's~/32$~~')"  #IP route will ignore /32. Therefore will fail to find it.
            INTF=${interfaceMap[$(echo $ROUTE_INFO | awk -F'_' '{print $2}')]}
            GW=$(getGatewayIp $INTF)
            CUSTOM_ROUTES_ARR+=( "$DEST_IP""_$GW""_$INTF" ) # Create an array so we don't need to find intf gw everytime
            $IP route | grep -E "$DEST_IP.*$GW.*$INTF"
            if [ $? -ne 0 ]; then
                $IP route add $DEST_IP via $GW dev $INTF
                if [ $? -eq 0 ]; then
                    loggerHfe " Added customer route to $DEST_IP through interface $INTF"
                else
                    errorAndExit "Failed to add custom route $DEST_IP via $GW through interface $INTF"
                fi
            else
                loggerHfe " Route already exists to $DEST_IP through interface $INTF"
            fi
        done
    fi
}

configureMgmtNAT()
{
    loggerHfe " Optional configuration to reach $MGMT using Remote IP. "

    if [ -z "${REMOTE_SSH_MACHINE_IP// }" ]; then
        loggerHfe " No IP is given for REMOTE_SSH_MACHINE_IP field, no route is set for managing this instance over $MGMT"
    else
        loggerHfe " $MGMT is used to manage this HFE instance, we can login using private IP to manage HFE machine without setting default route"
        loggerHfe " default route points to eth0 which will be used to interface all traffic for SBC"

        IFS=',' read -ra SSH_IP_LIST <<< "$REMOTE_SSH_MACHINE_IP"
        for REMOTE_MACHINE_IP in "${SSH_IP_LIST[@]}"; do
            REMOTE_MACHINE_IP="$(echo $REMOTE_MACHINE_IP | sed 's~/32$~~')" #IP route will ignore /32. Therefore will fail to find it.
            $IP route | grep -E "$REMOTE_MACHINE_IP.*$MGMT_GW.*$MGMT"
            if [ "$?" = "0" ]; then
                loggerHfe " Route is already available for remote machine's public IP($REMOTE_MACHINE_IP), from this IP you can SSH to HFE over Remote IP($MGMT)"
            else
                $IP route add $REMOTE_MACHINE_IP via $MGMT_GW dev $MGMT
                if [ "$?" = "0" ]; then
                    loggerHfe " Route added for remote machine's public IP($REMOTE_MACHINE_IP), from this IP you can SSH to HFE over Remote IP($MGMT)"
                else
                    errorAndExit "Failed to add route for ($REMOTE_MACHINE_IP)"
                fi
            fi
        done
    fi
}

showCurrentConfig()
{
    loggerHfe " Applied iptable rules and kernel routing table. "
    natTable=`$IPTABLES -t nat -vnL`

    loggerHfe " NAT tables:"
    loggerHfe " $natTable "

    filterTable=`$IPTABLES -t filter -vnL`

    loggerHfe " Filter tables:"
    loggerHfe " $filterTable "

    routeOutput=`$IP route`
    loggerHfe " Route:"
    loggerHfe " $routeOutput "
}

getSbcPktIps()
{
    local SBC_NAME=$1
    primaryMapKey="$SBC_NAME"
    if [[ $2 == "PKT0" ]]; then
        IF="2"
    else
        IF="3"
        [[ $MODE == "single" ]] && primaryMapKey="$SBC_NAME""-PKT1"
    fi
 
    unsortedList=''
    sortedList=''

    vmInstance=$($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Compute/virtualMachines/$SBC_NAME?$apiversion" -H "$authhead")
    networkInterfaceId=$(echo $vmInstance | $JQ .properties.networkProfile.networkInterfaces[$IF].id | sed 's/"//g')

    # Get Primary IPs or Secondary IPs
    ipConfigs=$($CURL "https://$AZURE_URI$networkInterfaceId?$apiversion" -H "$authhead" | $JQ .properties.ipConfigurations)
    configCount=$(echo $ipConfigs | $JQ . | $JQ length)
    for i in $(seq 0 $((configCount-1))); do
        if [[ $(echo $ipConfigs | $JQ .[$i].properties.primary | sed 's/"//g') == "true" ]]; then
            # Build map 
            ip=$(echo $ipConfigs | $JQ .[$i].properties.privateIPAddress | sed 's/"//g')
            echo "$primaryMapKey:$ip" >> $PRIMARY_FILE
        else
            ip=$(echo $ipConfigs | $JQ .[$i].properties.privateIPAddress | sed 's/"//g')
            unsortedList+="$ip "
        fi
    done

    sortedList=$(echo $unsortedList| tr " " '\n' | sort -n -t . -k1,1 -k2,2 -k 3,3 -k4,4)

    if [[ -z "$sortedList" ]];then
        errorAndExit "Failed to get IP(s) assigned to $2 on $1."
    fi
    loggerHfe " $SBC_NAME - IPs for $2 are $(echo $sortedList | tr '\n' " ")"
    echo $sortedList
}

getHfeIps()
{
    interface=$1
    unsortedList=''
    sortedList=''

    if [[ $interface == $ETH0 ]]; then
        nic=0
    else
        nic=1
    fi

    # In HA mode, we might end up being active
    # without the secondary IP. If that is the case
    # wait to be given IP by standby. If we fail, reboot
    # to trigger switchover
    attemptCount=0
    if [ $HA_MODE -eq 1 ] && [[ $SWITCHOVER_TYPE == "private" ]]; then
        attemptMax=90
    else
        attemptMax=1
    fi

    vmInstance=$($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Compute/virtualMachines/$HFE_NAME?$apiversion" -H "$authhead")
    networkInterfaceId=$(echo $vmInstance | $JQ .properties.networkProfile.networkInterfaces[$nic].id | sed 's/"//g')

    while [ $attemptCount -lt $attemptMax ]; do
        ipConfigs=$($CURL "https://$AZURE_URI$networkInterfaceId?$apiversion" -H "$authhead" | $JQ .properties.ipConfigurations)
        configCount=$(echo $ipConfigs | $JQ . | $JQ length)
        for i in $(seq 0 $((configCount-1))); do
            ip=$(echo $ipConfigs | $JQ .[$i].properties.privateIPAddress | sed 's/"//g')
            if [[ $(echo $ipConfigs | $JQ .[$i].properties.primary | sed 's/"//g') == "true" ]]; then
                echo "HFE_$interface:$ip" >> $PRIMARY_FILE
                [ $HA_MODE -eq 1 ] && [[ $SWITCHOVER_TYPE == "private" ]] && continue    ##In HA Mode we keep the primary IPs in place and move secondary between the HFE nodes when private
            fi
            unsortedList+="$ip "
        done
        sortedList=$(echo $unsortedList| tr " " '\n' | sort -n -t . -k1,1 -k2,2 -k 3,3 -k4,4)
        attemptCount=$((attemptCount+1))
        if [[ -z "$sortedList" ]];then
            if [ $HA_MODE -eq 1 ] && [[ $SWITCHOVER_TYPE == "private" ]]; then
                [ $((attemptCount % 10)) -eq 0 ] && loggerHfe " Waiting to be given secondary IP..."
                sleep 2
            fi
        else
            break    # We have our IPs
        fi
    done

    if [[ -z "$sortedList" ]];then
        if [ $HA_MODE -eq 1 ] && [[ $SWITCHOVER_TYPE == "private" ]]; then
            errorAndExit "Switchover: Did not get given secondary IPs after 3 minutes from standby. Check IP configurations and reboot HFE to recover"
        else
            errorAndExit "Failed to get IP(s) assigned to $HFE_NAME on $interface."
        fi
    else
        if [ $attemptCount -gt 1 ]; then
            # HA MODE, we had to wait for the IPs to be assigned
            # make sure they are written to the interface
            prefix=$($IP addr show $ETH0 | grep -w inet | head -1 | awk -F' ' '{print $2}' | awk -F'/' '{print $2}')
            brd=$($IP addr show $ETH0 | grep -w inet | head -1 | awk -F' ' '{print $4}')
            for (( idx=0; idx<"${#sortedList[@]}"; idx++ )); do
                if [ $( $IP addr show $ETH0 | grep -c "${sortedList[$idx]}/" ) -eq 0 ]; then
                    $IP addr add "${sortedList[$idx]}/$prefix" dev $ETH0 scope global brd $brd
                fi
            done
            if [ $($IP addr show dev $ETH0 | grep -ic 'state down') -gt 0 ]; then
                if [ -e $NETPLAN_ROOT ]; then
                    netplan apply
                    if [ $? -ne 0 ]; then
                        loggerHfe "Switchover: Error. Failed to bring up eth0. Rebooting"
                        reboot
                    fi
                fi
            fi
        fi
    fi
    loggerHfe " $HFE_NAME - IPs for $interface are $(echo $sortedList | tr '\n' " ")"
    echo $sortedList
}

setLogRotate()
{
    # Setup HFE logRotate to backup and rotate logs which report status of connection between HFE and Active SBC

    if [ ! -f $hfeLogRotateConf ]; then
        echo -e "$statusLogFile" >> $hfeLogRotateConf
        echo -e "{" >> $hfeLogRotateConf
        echo -e "    rotate 4" >> $hfeLogRotateConf
        echo -e "    size=250M" >> $hfeLogRotateConf
        echo -e "    missingok" >> $hfeLogRotateConf
        echo -e "    notifempty" >> $hfeLogRotateConf
        echo -e "    dateformat -%d%m%Y" >> $hfeLogRotateConf
        echo -e "    compress" >> $hfeLogRotateConf
        echo -e "}" >> $hfeLogRotateConf
    fi
}


# Verify authorization is successful
verifyAuthorization()
{
    local SBC_NAME=$1 
    local count=0

    loggerHfe " Validating authorization token"

    # Test for timeout
    $CURL https://$AZURE_URI
    if [ $? -eq 28 ]; then
        handleTempMgmtRoute 1 "Authorization verification"
    fi

    vmResult=$($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Compute/virtualMachines/$SBC_NAME?$apiversion" -H "$authhead" | $JQ .error.code)
    while [[ $vmResult != "null" ]] && [ $count -lt 30 ]; do
        loggerHfe " Authorization validation for attempt $count failed. Retrying in 30 seconds"
        let count=$count+1
        sleep 30s
        setAuthorizationHead
        vmResult=$($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Compute/virtualMachines/$SBC_NAME?$apiversion" -H "$authhead" | $JQ .error.code)
    done
    
    if [ $count -eq 30 ]; then
        errorAndExit "Authorization still failing after 15 minutes. Verify identity roles and reboot! Exiting"
    else
        loggerHfe " Authorization validation successful for attempt $count. Continuing to get SBC information"
    fi
}

getAzureVars()
{
    metaDataJson=$($CURL "http://169.254.169.254/metadata/instance/compute?$apiversion" -H Metadata:true)
    RG_NAME=$(echo $metaDataJson | $JQ .resourceGroupName | sed 's/"//g')
    SUB_ID=$(echo $metaDataJson | $JQ .subscriptionId | sed 's/"//g')
    HFE_NAME=$(echo $metaDataJson | $JQ .name | sed 's/"//g')
    
    setAuthorizationHead
}

extractHfeConfigIps()
{
    # Get IPs
    if [[ $MODE == "single" ]]; then
        ACTIVE_SBC_IP_ARR=( $(getSbcPktIps $ACTIVE_SBC_VM_NAME "PKT0") )
        STANDBY_SBC_IP_ARR=( $(getSbcPktIps $STANDBY_SBC_VM_NAME "PKT0") )
        HFE_ETH0_IP_ARR=( $(getHfeIps $ETH0) )
        
        ACTIVE_SBC_IP_PKT1_ARR=( $(getSbcPktIps $ACTIVE_SBC_VM_NAME "PKT1") )
        STANDBY_SBC_IP_PKT1_ARR=( $(getSbcPktIps $STANDBY_SBC_VM_NAME "PKT1") )
        HFE_ETH1_IP_ARR=( $(getHfeIps $ETH1) )
    else
        ACTIVE_SBC_IP_ARR=( $(getSbcPktIps $ACTIVE_SBC_VM_NAME $SBC_PKT_PORT_NAME) )
        STANDBY_SBC_IP_ARR=( $(getSbcPktIps $STANDBY_SBC_VM_NAME $SBC_PKT_PORT_NAME) )
        HFE_ETH0_IP_ARR=( $(getHfeIps $ETH0) )
    fi
 
    # Verify we have everything
    if [ -z $ACTIVE_SBC_IP_ARR ] || [ -z $STANDBY_SBC_IP_ARR ] || [ -z $HFE_ETH0_IP_ARR ] ; then
        errorAndExit "SBC information missing. Exiting"
    elif [[ $MODE == "single" ]]; then
        if [ -z $ACTIVE_SBC_IP_PKT1_ARR ] || [ -z $STANDBY_SBC_IP_PKT1_ARR ] || [ -z $HFE_ETH1_IP_ARR ]; then
            errorAndExit "SBC information missing. Exiting"
        fi
    fi
}

# Returns interface type.
# SLAVE interface will have symlink to it's master.
# if the symlink is present, then it's a salve interface.
# Normal interface will not have any of these files.
getIntfType()
{
    local intf=$1

    if ls /sys/class/net/$intf/lower_* 1> /dev/null 2>&1; then
        gRetVal=$INTF_TYPE_MASTER
        return 0
    fi

    if ls /sys/class/net/$intf/master  1> /dev/null 2>&1; then
        gRetVal=$INTF_TYPE_SLAVE
        return 0
    fi

    gRetVal=$INTF_TYPE_NORMAL
    return 0
}

function setSecondaryIntfQueueLen()
{
    local netdev=""
    local intfType=$INTF_TYPE_NORMAL
    local rxPresetMax=4096
    local txPresetMax=4096
    local rxQLen=0
    local txQLen=0


    for netdev in $(ls /sys/class/net); do
        if [ "$netdev" != "lo" ]; then
            getIntfType $netdev
            intfType=$gRetVal
            if [ $intfType -eq $INTF_TYPE_SLAVE ]; then
                rxQLen=$(ethtool -g $netdev| grep -A 4 "Current hardware settings" | grep RX: | cut -d ':' -f2| xargs)
                if [ $? -ne 0 ]; then
                    errorAndExit "$netdev: Failed to get Current RX queue length"
                fi
                txQLen=$(ethtool -g $netdev| grep -A 4 "Current hardware settings" | grep TX: | cut -d ':' -f2| xargs)
                if [ $? -ne 0 ]; then
                    errorAndExit "$netdev: Failed to get Current TX queue length"
                fi

                #find the preset Max RX queue length
                rxPresetMax=$(ethtool -g $netdev| grep -A 4 "Pre-set maximums" | grep RX: | cut -d ':' -f2| xargs)
                if [ $? -ne 0 ]; then
                    errorAndExit "$netdev: Failed to get Pre-set Maximum RX queue length"
                fi
                #restrict the queue length to Max 8192
                rxPresetMax=$((rxPresetMax>MAX_QUEUE_LENGTH ? MAX_QUEUE_LENGTH : rxPresetMax))

                #find the preset Max TX queue length
                txPresetMax=$(ethtool -g $netdev | grep -A 4 "Pre-set maximums" | grep TX: | cut -d ':' -f2| xargs)
                if [ $? -ne 0 ]; then
                    errorAndExit "$netdev: Failed to get Pre-set Maximum TX queue length"
                fi
                #restrict the queue length to Max 8192
                txPresetMax=$((txPresetMax>MAX_QUEUE_LENGTH ? MAX_QUEUE_LENGTH : txPresetMax))

                if [ $rxQLen -lt $rxPresetMax ]; then
                    loggerHfe " $netdev: changing RX queue length from $rxQLen to $rxPresetMax"
                    ethtool -G $netdev rx $rxPresetMax
                fi
                if [ $txQLen -lt $txPresetMax ]; then
                    loggerHfe " $netdev: changing TX queue length from $txQLen to $txPresetMax"
                    ethtool -G $netdev tx $txPresetMax
                fi
            fi
        fi
    done

    sleep 1
}

# Create a sysdump of common files needed by Ribbon
# for debugging HFE failures in Azure
sysdump()
{
    if [ ! -e $HFERoot/sysdump ]; then
        mkdir $HFERoot/sysdump
    fi

    if [ -e $TMP_LOG_DIR ]; then
        rm -rf $TMP_LOG_DIR
    fi
    mkdir $TMP_LOG_DIR

    $IP a > $TMP_LOG_DIR/interfaces.out
    $IP r > $TMP_LOG_DIR/routes.out
    $IPTABLES -vnL > $TMP_LOG_DIR/iptables.out
    echo -E "\n\n\n\n" >> $TMP_LOG_DIR/iptables.out
    $IPTABLES -t nat -vnL >> $TMP_LOG_DIR/iptables.out
    dmesg > $TMP_LOG_DIR/dmesg.out
    $CONNTRACK -L -o extended > $TMP_LOG_DIR/conntrack.out
    $CONNTRACK -C > $TMP_LOG_DIR/conntrack.count.out
    $CURL -H Metadata:true --noproxy "*" "http://169.254.169.254/metadata/instance?api-version=2021-02-01" > $TMP_LOG_DIR/metadata.json
 
    journalctl -xe > $TMP_LOG_DIR/journalctl.out
    journalctl -t dhclient > $TMP_LOG_DIR/dhclient.log
    journalctl -t systemd-networkd > $TMP_LOG_DIR/system-networkd.log

    cp /var/log/syslog $TMP_LOG_DIR/
    cp /var/log/messages $TMP_LOG_DIR/
    cp /var/log/waagent.log $TMP_LOG_DIR/
    cp /var/log/dpkg.log $TMP_LOG_DIR/
    cp /var/log/cloud-init* $TMP_LOG_DIR/
    cp /var/lib/cloud/instance/user-data.txt* $TMP_LOG_DIR/

    mkdir $TMP_LOG_DIR/bash_history/
    ls /home/*/.bash_history /root/.bash_history | while read hist; do
        cp $hist "$TMP_LOG_DIR/bash_history/$(echo $hist | awk -F/ '{print $(NF-1)}')"
    done

    cp -r $NETPLAN_ROOT $TMP_LOG_DIR/
    cp -r $HFERoot $TMP_LOG_DIR/
    rm -fr $TMP_LOG_DIR/HFE/sysdump/     #Remove previous sysdumps

    fileName="HFE_sysdump_$(hostname)_$(date +%y-%m-%dT%H-%M-%S).tar.gz"
    tar -zcvf $fileName -C $TMP_LOG_DIR .
    mv $fileName $HFERoot/sysdump/
    rm -rf $TMP_LOG_DIR
    echo "SYSDUMP available at $HFERoot/sysdump/$fileName"
    echo "Remove any old sysdumps to preserve disk space!"
}

# Util function to wait until
# we can process the next command
# for IP movement
haWaitForIntfReady()
{
    local id="$1"
    lcoal timer=0
    while [[ $($CURL "https://$AZURE_URI$id?$apiversion" -H "$authhead" | $JQ -r .properties.provisioningState) == "Updating" ]]; do
        timer=$((timer+1))
        if [ $timer -eq 20 ]; then
            loggerHfe "Switchover: Networking interface $(basename $id) is still updating."
            timer=0
        fi
            sleep 0.2s
    done

    state=$($CURL "https://$AZURE_URI$id?$apiversion" -H "$authhead" | $JQ -r .properties.provisioningState)
    if [[ $state != "Succeeded" ]]; then
        errorAndExit "Switchover: Networking interface $(basename $id) is in state $state. Exiting!"
    fi
}

# Runs the REST api requests
# to move the secondary IP between
# the HFE nodes.
haPerformPrivateSwitchover()
{
    local sourceVmJson="$1"
    local destVmJson="$2"
    local secondaryIp="$3"
    local errorExit=$4
    local noDetach=$5

    local sourceVmName="$(echo $sourceVmJson | $JQ -r .name)"
    local destVmName="$(echo $destVmJson | $JQ -r .name)"

    local attempt=0

    #### TODO: If we ever support multiple IPs on Azure SBC, come back and update this to work with all IPs ####
    if [ ! -z "$initSndIpCfg" ] && [[ $(echo $sourceVmJson | $JQ -r --arg IP "$secondaryIp" '.eth0.properties.ipConfigurations[] | select(.properties.privateIPAddress==$IP)') == "" ]]; then
        ##Add the initSecondIpConfig in
        sourceVmJson="$(echo $sourceVmJson | $JQ --arg IP "$secondaryIp" --argjson IPCFG "$initSndIpCfg" '.eth0.properties.ipConfigurations += $IPCFG')"
    fi
    secondaryIpCfg="$(echo $sourceVmJson | $JQ --arg IP "$secondaryIp" --arg ALLOCATION "Static" '.eth0.properties.ipConfigurations[] | select(.properties.privateIPAddress==$IP) | .properties.privateIPAllocationMethod=$ALLOCATION | [.]')"

    if [ $noDetach -eq 0 ]; then
        while [ $attempt -le $HA_SWITCHOVER_RETRY ]; do
            loggerHfe "Switchover: Removing secondary IP ($secondaryIp) from $sourceVmName"
            ###Remove Secondary IP Config###
            resp=$($CURL -s -w "\nAzureRetCode: %{http_code}\n" -H "$authhead" -H "Content-Type: application/json" -X PUT "https://$AZURE_URI/$(echo $sourceVmJson | $JQ -r .eth0.id)?$apiversion" -d "$(echo $sourceVmJson | $JQ --arg IP $secondaryIp '.eth0 | del(.properties.ipConfigurations[] | select(.properties.primary==false) | select(.properties.privateIPAddress==$IP))')" )
            retCode=$(echo $resp | awk -F'AzureRetCode: ' '{print $2}')
            if [ $retCode -ne 200 ] && [ $retCode -ne 201 ]; then
                loggerHfe "   Got bad response from Azure: $resp"
                if [ $attempt -eq $HA_SWITCHOVER_RETRY ]; then
                    if [ $errorExit -ne 0 ]; then
                        errorAndExit "Switchover: Failed to remove secondary IP from $sourceVmName"
                    else
                        return 1
                    fi
                fi
                loggerHfe "Switchover: Failed to remove secondary IP from $sourceVmName. Retrying"
                attempt=$((attempt+1))
            else
                break
            fi
        done
        haWaitForIntfReady "$(echo $sourceVmJson | $JQ -r .eth0.id)"
        loggerHfe "Switchover: Successfully removed secondary IP ($secondaryIp) from $sourceVmName"
    fi

    attempt=0
    loggerHfe "Switchover: Adding secondary IP ($secondaryIp) to $destVmName"
    while [ $attempt -le $HA_SWITCHOVER_RETRY ]; do
         resp=$($CURL -s -w "\nAzureRetCode: %{http_code}\n" -H "$authhead" -H "Content-Type: application/json" -X PUT "https://$AZURE_URI/$(echo $destVmJson | $JQ -r .eth0.id)?$apiversion" -d "$(echo $destVmJson | $JQ --argjson PRIVATEIPCFG "$secondaryIpCfg" '.eth0 | .properties.ipConfigurations += $PRIVATEIPCFG')")
         retCode=$(echo $resp | awk -F'AzureRetCode: ' '{print $2}')
        if [ $retCode -ne 200 ] && [ $retCode -ne 201 ]; then
            loggerHfe "   Got bad response from Azure: $resp"
            if [ $attempt -eq $HA_SWITCHOVER_RETRY ]; then
                if [ $errorExit -ne 0 ]; then
                    errorAndExit "Switchover: Failed to attach IP ($secondaryIp) to $destVmName"
                else
                    return 1
                fi
            fi
            loggerHfe "Switchover: Failed to attach IP ($secondaryIp) to $destVmName"
            attempt=$((attempt+1))
        else
            break
        fi
    done

    haWaitForIntfReady "$(echo $destVmJson | $JQ -r .eth0.id)"
    loggerHfe "Switchover: Successfully attached secondary IP ($secondaryIp) to $destVmName"
    return 0
}

# Runs the REST api requests
# to move the public IP between
# the HFE nodes.
haPerformPublicSwitchover()
{
    local sourceVmJson="$1"
    local destVmJson="$2"
    local publicIp="$3"
    local errorExit=$4
    local noDetach=$5

    local sourceVmName="$(echo $sourceVmJson | $JQ -r .name)"
    local destVmName="$(echo $destVmJson | $JQ -r .name)"
    local pubIpName="$(basename $publicIp)"

    local attempt=0

    if [ $noDetach -eq 0 ]; then
        loggerHfe "Switchover: Detaching public IP ($pubIpName) from $sourceVmName"
        while [ $attempt -le $HA_SWITCHOVER_RETRY ]; do
            #Full Public Switchover
            resp=$($CURL -s -w "\nAzureRetCode: %{http_code}\n" -H "$authhead" -H "Content-Type: application/json" -X PUT "https://$AZURE_URI/$(echo $sourceVmJson | $JQ -r .eth0.id)?$apiversion" -d "$(echo $sourceVmJson | $JQ '.eth0 | del(.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.publicIPAddress)')" )
            retCode=$(echo $resp | awk -F'AzureRetCode: ' '{print $2}')
            if [ $retCode -ne 200 ] && [ $retCode -ne 201 ]; then
                loggerHfe "   Got bad response from Azure: $resp"
                if [ $attempt -eq $HA_SWITCHOVER_RETRY ]; then
                    if [ $errorExit -ne 0 ]; then
                        errorAndExit "Switchover: Failed to remove public IP ($pubIpName) from $sourceVmName"
                    else
                        return 1
                    fi
                fi
                loggerHfe "Switchover: Failed to remove public IP ($pubIpName) from $sourceVmName. Retrying"
                attempt=$((attempt+1))
            else
                break
            fi
        done
        haWaitForIntfReady "$(echo $sourceVmJson | $JQ -r .eth0.id)"
        loggerHfe "Switchover: Successfully removed public IP ($pubIpName) from $sourceVmName."
    fi

    attempt=0
    loggerHfe "Switchover: Attaching public IP ($pubIpName) to $destVmName"
    while [ $attempt -le $HA_SWITCHOVER_RETRY ]; do
        resp=$($CURL -s -w "\nAzureRetCode: %{http_code}\n" -H "$authhead" -H "Content-Type: application/json" -X PUT "https://$AZURE_URI/$(echo $destVmJson | $JQ -r .eth0.id)?$apiversion" -d "$(echo $destVmJson | $JQ --argjson PUBLICIPCFG "{ \"publicIPAddress\" : { \"id\": \"$publicIp\"} }"  '.eth0 | .properties.ipConfigurations |= map(select(.properties.primary==true).properties |= . + $PUBLICIPCFG)')" )
        retCode=$(echo $resp | grep "AzureRetCode: " | awk -F'AzureRetCode::' '{print $2}')
        if [ $retCode -ne 200 ] && [ $retCode -ne 201 ]; then
            loggerHfe "   Got bad response from Azure:  $resp"
            if [ $attempt -eq $HA_SWITCHOVER_RETRY ]; then
                if [ $errorExit -ne 0 ]; then
                    errorAndExit "Switchover: Failed to attach public IP ($pubIpName) to $destVmName"
                else
                    return 1
                fi
            fi
            loggerHfe "Switchover: Failed to attach public IP ($pubIpName) to $destVmName. Retrying"
            attempt=$((attempt+1))
        else
            break
        fi
    done
    haWaitForIntfReady "$(echo $destVmJson | $JQ -r .eth0.id)"
    loggerHfe "Switchover: Successfully attached public IP to $destVmName"
    return 0
}

# Util function to return
# an IP address instead of
# performing full switchover.
# Used when standby has the IPs or
# if neither ce has IPs
haReturnIpAddress()
{
    local sourceVmJson="$1"
    local destVmJson="$2"
    local targetIp="$3"
    local noDetach=$4
    local destVmName="$5"

    if [ $noDetach -eq 0 ]; then
        actionType="return IP address to active"
    else
        actionType="recover IP address from Azure"
    fi

    loggerHfe "Switchover: Attempting to $actionType"
    peerId=$(echo $destVmJson | $JQ -r .eth0.id)
    localId=$(echo $sourceVmJson | $JQ -r .eth0.id)
    if [[ $($CURL "https://$AZURE_URI$peerId?$apiversion" -H "$authhead" | $JQ -r .properties.provisioningState) != "Updating" ]] && [[ $($CURL "https://$AZURE_URI$peerId?$apiversion" -H "$authhead" | $JQ -r .properties.provisioningState) != "Updating" ]]; then
        # No updates are occuring to network interfaces, so no IPs are supposed to be moving.
        # Start sending them back
        
        ##Remove null from JSON
        destVmJson="$(echo $destVmJson | $JQ '.eth0 |= del(.properties.ipConfigurations[] | select(.properties.publicIPAddress==null) | .properties.publicIPAddress)')"
        destVmJson="$(echo $destVmJson | $JQ '.eth0 |= del(.properties.ipConfigurations[] | select(.properties.networkSecurityGroup==null) | .properties.networkSecurityGroup)')"
        sourceVmJson="$(echo $sourceVmJson | $JQ '.eth0 |= del(.properties.ipConfigurations[] | select(.properties.publicIPAddress==null) | .properties.publicIPAddress)')"
        sourceVmJson="$(echo $sourceVmJson | $JQ '.eth0 |= del(.properties.ipConfigurations[] | select(.properties.networkSecurityGroup==null) | .properties.networkSecurityGroup)')"
        if [[ $SWITCHOVER_TYPE == "public" ]]; then
            haPerformPublicSwitchover "$sourceVmJson" "$destVmJson" "$targetIp" 0 $noDetach
            [ $? -ne 0 ] && loggerHfe "Switchover: Failed to perform public switchover to give IP to $destVmName. Will try again next time" && return 1
        else
            haPerformPrivateSwitchover "$sourceVmJson" "$destVmJson" "$targetIp" 0 $noDetach
            [ $? -ne 0 ] && loggerHfe "Switchover: Failed to perform private switchover to give IP to $destVmName. Will try again next time" && return 1
        fi
    else
        loggerHfe "Switchover: Either our network interface or our peer's is currently 'updating'. Not performing any action."
        return 2
    fi
    return 0
}


# Util function to get the relevant HFE info.
# Return in a JSON string format for processing.
# For switchover logic to check who is active
haGetHfeInfoJson()
{
    local targetHfe="$1"
    local simpleHfeJson=""
    local vmTry=0
    local intfTry=0

    local idArr=( )
    local ipForwardArr=( )
    local ipAccelArr=( )
 
    while [ $vmTry -lt 2 ]; do
        vmInfo=$($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Compute/virtualMachines/$targetHfe?$apiversion" -H "$authhead")
        checkIfAuthFailed
        if [ $? -eq 1 ]; then
            loggerHfe " Authorization failed for virtualMachine $targetHfe. Sleeping 10s and retrying"
            sleep 10s
            setAuthorizationHead
            vmTry=$((vmTry + 1))
            continue
        fi
        break
    done
 
    interfaceIds=$(echo $vmInfo | $JQ .properties.networkProfile.networkInterfaces)
    interfaceNum=$(echo $interfaceIds | $JQ length)
    location=$(echo $vmInfo | $JQ -r .location)

    for i in $(seq 0 $((interfaceNum-1))); do
        interfaceId=$(echo $interfaceIds | $JQ -r .[$i].id | sed 's/"//g')
        idArr[$i]="$interfaceId"
        while [ $intfTry -lt 2 ]; do
            intfOut=$($CURL "https://$AZURE_URI$interfaceId?$apiversion" -H "$authhead")
            checkIfAuthFailed
            if [ $? -eq 1 ]; then
                loggerHfe " Authorization failed for interfaceId. Sleeping 10s and retrying"
                sleep 10s
                setAuthorizationHead
                intfTry=$((vmTry + 1))
                continue
            fi

            ## Get all of the IP configurations ##
            updateIpConfigJson=""
            configMax=$(($(echo $intfOut | $JQ '.properties.ipConfigurations | length')-1))
            for j in $(seq 0 $configMax); do
                                                                                       #Make sure all IPs stay with the instances
                updatePropertiesJson="$( echo $intfOut | $JQ --argjson index "$j" '.properties.ipConfigurations[$index].properties.privateIPAllocationMethod="Static" | .properties.ipConfigurations[$index].properties | { privateIPAddress, privateIPAllocationMethod, subnet, primary, privateIPAddress, publicIPAddress }')"
                updateConfigJson="$( echo $intfOut | $JQ --argjson index "$j" --argjson props "$updatePropertiesJson" '.properties.ipConfigurations[$index].properties=$props | .properties.ipConfigurations[$index] | { name, id, properties }')"

                if [ $j -eq 0 ]; then
                    updateIpConfigJson="[ $updateConfigJson"
                else
                    updateIpConfigJson="$updateIpConfigJson, $updateConfigJson"
                fi
            done
            updateIpConfigJson="$updateIpConfigJson ]" #Close JSON array
            declare -g eth${i}ipConfig="$updateIpConfigJson"

            ipForwardArr[$i]=$(echo $intfOut | $JQ -r .properties.enableIPForwarding)
            ipAccelArr[$i]=$(echo $intfOut | $JQ -r .properties.enableAcceleratedNetworking)

            declare -g eth${i}nsgCfg="$(echo $intfOut | $JQ -r .properties.networkSecurityGroup)"

            break #break out of intfTry
        done
    done

    simpleHfeJson=$( $JQ -n \
      --arg ETH0ID "${idArr[0]}" \
      --arg ETH1ID "${idArr[1]}" \
      --arg ETH2ID "${idArr[2]}" \
      --argjson ETH0CNFIG "${eth0ipConfig}" \
      --argjson ETH1CNFIG "${eth1ipConfig}" \
      --argjson ETH2CNFIG "${eth2ipConfig}" \
      --argjson ETH0FWD "${ipForwardArr[0]}" \
      --argjson ETH1FWD "${ipForwardArr[1]}" \
      --argjson ETH2FWD "${ipForwardArr[2]}" \
      --argjson ETH0ACCEL "${ipAccelArr[0]}" \
      --argjson ETH1ACCEL "${ipAccelArr[1]}" \
      --argjson ETH2ACCEL "${ipAccelArr[2]}" \
      --argjson ETH0NSG "${eth0nsgCfg}" \
      --argjson ETH1NSG "${eth1nsgCfg}" \
      --argjson ETH2NSG "${eth2nsgCfg}" \
      --arg HFENAME "$targetHfe" \
      --arg LOCATION "$location" \
    '{
    "eth0": {
        "id": $ETH0ID,
        "properties": {
          "ipConfigurations" : $ETH0CNFIG,
          "enableAcceleratedNetworking": $ETH0ACCEL,
          "enableIPForwarding": $ETH0FWD,
          "networkSecurityGroup": $ETH0NSG
        },
        "location": $LOCATION
    },
    "eth1": {
        "id": $ETH1ID,
        "properties": {
          "ipConfigurations" : $ETH1CNFIG,
          "enableAcceleratedNetworking": $ETH1ACCEL,
          "enableIPForwarding": $ETH1FWD,
          "networkSecurityGroup": $ETH1NSG
        },
        "location": $LOCATION
    },
    "eth2": {
        "id": $ETH2ID,
        "properties": {
          "ipConfigurations" : $ETH2CNFIG,
          "enableAcceleratedNetworking": $ETH2ACCEL,
          "enableIPForwarding": $ETH2FWD,
          "networkSecurityGroup": $ETH2NSG
        },
        "location": $LOCATION
    },
    "name": $HFENAME
}')
 
    echo $simpleHfeJson
}

# Main standby function which performs switchover logic. 
# Will do the following:
#   1. Check if we are in HA mode
#   2. Gather information about the Azure routes, local IPs and peer IPs
#   3. If the Azure routes are pointing to us, we are active. Exit function and continue
#   4. Conitually monitor the active HFE:
#     i. Check we do not have the IPs. If so return it
#    ii. Check maintenance status of active. Perform switchover SWITCHOVER_MODE is FULL
#   iii. Use fping to healthcheck the active.
#  5. Perform switchover logic:
#     i. Update the Azure routes to use our eth2 IP address
#    ii. Move IP address to our eth0
#   iii. Update the private IP at linux level if private switchover
haCheckIfStandby()
{
    if [[ $PEER_HFE_NAME == "" ]]; then
        loggerHfe "Switchover: Natvar PEER_HFE_NAME not configured. Not in HA mode"
        return
    elif [[ $ROUTE_TABLE_NAME == "" ]]; then
        loggerHfe "Switchover: Natvar ROUTE_TABLE_NAME not configured. Not in HA mode"
        return
    fi

    [ -z $SWITCHOVER_MODE ] && loggerHfe "Switchover: NatVar SWITCHOVER_MODE not set. Assuming FULL"

    HA_MODE=1
    SWITCHOVER_TYPE="private"
    loggerHfe "Switchover: We are paired with VM $PEER_HFE_NAME. Checking who is active"
 
    localCeJson="$(haGetHfeInfoJson $HFE_NAME)"
    [ -z "$localCeJson" ] && errorAndExit "Switchover: Failed to get info on HFE $HFE_NAME"
    peerCeJson="$(haGetHfeInfoJson $PEER_HFE_NAME)"
    [ -z "$peerCeJson" ] && errorAndExit "Switchover: Failed to get info on HFE $PEER_HFE_NAME"

    # Get some values immediately so that we can apply them if they get removed
    # If active: apply to self or apply to standby if we are about to reboot
    # If standby: apply when switching over
    peerPubIp="$(echo $peerCeJson | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.publicIPAddress.id')"
    localPubIp="$(echo $localCeJson | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.publicIPAddress.id')"
    if [[ $peerPubIp != "null" ]]; then
        SWITCHOVER_TYPE="public"
        initPubIp=$peerPubIp
    elif [[ $localPubIp != "null" ]]; then
        SWITCHOVER_TYPE="public"
        initPubIp=$localPubIp
    fi

    if [ $(echo $peerCeJson | $JQ -r '.eth0.properties.ipConfigurations | length') -gt 1 ]; then
        initSndIp=$(echo $peerCeJson | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==false) | .properties.privateIPAddress' | head -n 1)
        initSndIpCfg="$(echo $peerCeJson | $JQ --arg IP "$initSndIp" '.eth0.properties.ipConfigurations[] | select(.properties.privateIPAddress==$IP) | del(.properties.publicIPAddress) | [.]')"
    elif [ $(echo $localCeJson | $JQ -r '.eth0.properties.ipConfigurations | length') -gt 1 ]; then
        initSndIp=$(echo $localCeJson | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==false) | .properties.privateIPAddress' | head -n 1)
        initSndIpCfg="$(echo $localCeJson | $JQ --arg IP "$initSndIp" '.eth0.properties.ipConfigurations[] | select(.properties.privateIPAddress==$IP) | del(.properties.publicIPAddress) | [.]')"
    fi
    loggerHfe "Switchover: Got the following initial values: SwitchoverType=$SWITCHOVER_TYPE, PublicIP=$initPubIp, SecondaryIP=$initSndIp, SecondaryIPConfigId=$(echo $initSndIpCfg | $JQ -r .[].id)"


    # eth2 IPs for checking the Azure routes
    localEth2=$(echo $localCeJson | $JQ -r '.eth2.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.privateIPAddress')
    peerEth2=$(echo $peerCeJson | $JQ -r '.eth2.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.privateIPAddress')


    while true; do
        routes="$($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Network/routeTables/$ROUTE_TABLE_NAME?$apiversion" -H "$authhead" | $JQ -r .properties.routes[])"

        localCount=$(echo "$routes" | grep -cw $localEth2) 
        peerCount=$(echo "$routes" | grep -cw $peerEth2)
        if [ $localCount -gt 0 ] && [ $peerCount -gt 0 ]; then
            errorAndExit "Switchover: Azure routetable contains routes for both peer HFE nodes! Unable to determine active. Exiting!"
        elif [ $localCount -eq 0 ] && [ $peerCount -eq 0 ]; then
            loggerHfe "Switchover:  Routetable does not contain eth2 IPs for either peer HFE nodes. Waiting 1 second and retrying"
            sleep 1
            continue
        elif [ $localCount -gt 0 ]; then
            loggerHfe "Switchover: We are the active HFE node. Continuing to startup"
            break
        elif [ $peerCount -gt 0 ]; then
            loggerHfe "Switchover: We are the standby HFE node. Starting monitoring of active HFE."
            hfeMgt=$(echo $peerCeJson | $JQ -r '.eth1.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.privateIPAddress')
            triggerSwitchover=0
            checkHostUpgrade=1
            checkIPs=1
            waitCount=0

            # Add mgmt routes now so we can get on the standby HFE
            setInterfaceMap
            configureMgmtNAT

            while [ $triggerSwitchover -eq 0 ]; do
                if [ $checkIPs -eq 1 ]; then
                    checkIPs=0
                    checkState=0    # Do we need to check the state of the interfaces
                    localCeJson="$(haGetHfeInfoJson $HFE_NAME)"
                    if [ -z "$localCeJson" ]; then
                        loggerHfe "Switchover: Failed to get local HFE information. Will try again next time"
                        checkIPs=1
                    else
                        if [[ $SWITCHOVER_TYPE == "public" ]]; then
                            targetIp="$(echo $localCeJson | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.publicIPAddress.id')"
                            if [[ $targetIp != "null" ]]; then
                                checkState=1
                                loggerHfe "Switchover: Detected we have the public IP. Attempting to return it to active"
                            fi
                        else ##Check private
                            if [ $(echo $localCeJson | $JQ '.eth0.properties.ipConfigurations | length') -gt 1 ]; then
                                checkState=1
                                loggerHfe "Switchover: Detected we have the secondary IPs. Attempting to return to them to active"
                                #### TODO: If we ever support multiple IPs on Azure SBC, come back and update this to work with all IPs ####
                                targetIp=$(echo $localCeJson | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==false) | .properties.privateIPAddress' | head -n 1)
                            fi 
                        fi
                        if [ $checkState -eq 1 ]; then    # We have the IPs but are not active. Give them back
                            peerCeJson="$(haGetHfeInfoJson $PEER_HFE_NAME)"
                            if [ -z "$peerCeJson" ]; then
                                loggerHfe "Switchover: Failed to get $PEER_HFE_NAME information. Will try again next time"
                                checkIPs=1
                            else
                                removeIp=0
                                haReturnIpAddress "$localCeJson" "$peerCeJson" "$targetIp" 0 "$PEER_HFE_NAME"
                                case $? in
                                    1) checkIPs=1 ;;
                                    2) waitCount=50 ;;
                                    *) removeIp=1 ;;
                                esac
                                if [ $removeIp -eq 1 ]; then
                                    if [ $($IP addr show dev $ETH0 | grep -c "$targetIp/") -gt 0 ]; then
                                        #Delete the IP locally
                                        sndIpCidr=$($IP addr show dev $ETH0 | grep "$targetIp/" | awk '{print $2}')
                                        $IP addr del $sndIpCidr dev $ETH0
                                        if [ $($IP addr show dev $ETH0 | grep -ic 'state down') -gt 0 ]; then
                                            if [ -e $NETPLAN_ROOT ]; then
                                                netplan apply
                                                if [ $? -ne 0 ]; then
                                                    loggerHfe "Switchover: Error. Failed to bring up eth0. Rebooting"
                                                    reboot
                                                fi
                                            fi
                                        fi
                                    fi
                                fi
                            fi
                        fi
                    fi
                fi

                if [ $checkHostUpgrade -eq 1 ]; then
                    checkHostUpgrade=0     #Reset
                    peerUpdateCheck=$($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Compute/virtualMachines/$PEER_HFE_NAME/instanceView?$apiversion" -H "$authhead" | $JQ -r instanceView.maintenanceRedeployStatus)
                    if [[ $peerUpdateCheck != "" ]]; then
                        peerUpdateStart=$(echo $peerUpdateCheck | $JQ -r .maintenanceWindowStartTime)
                        localUpdateCheck=$($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Compute/virtualMachines/$PEER_HFE_NAME/instanceView?$apiversion" -H "$authhead" | $JQ -r instanceView.maintenanceRedeployStatus) 
                        if [[ $localUpdateCheck == "" ]]; then
                            msg="Switchover: Peer HFE CE ($PEER_HFE_NAME) is going for maintenance at $peerUpdateStart. No maintenance detected for this CE."
                            if [[ $SWITCHOVER_MODE != "FAULT" && $SWITCHOVER_MODE != "NONE" ]]; then
                                triggerSwitchover=1
                                msg="$msg Performing Switchover"
                            fi
                            loggerHfe "$msg"
                        else
                            # Both CEs are going for maintenance at some point.
                            # Check who is first
                            localUpdateStart=$(echo $localUpdateCheck | $JQ -r .maintenanceWindowStartTime)
                            if [ $(date -d $peerUpdateCheck +%s) -lt $(date -d $localUpdateCheck +%s) ]; then
                                # Peer updating before us. Switchover
                                msg="Switchover: Both hosts for HFE peers are scheduled for maintenance, however peer is scheduled first at $peerUpdateStart."
                                if [[ $SWITCHOVER_MODE != "FAULT" && $SWITCHOVER_MODE != "NONE" ]]; then
                                    msg="$msg Performing Switchover"
                                    triggerSwitchover=1
                                fi
                                loggerHfe "$msg"
                            fi
                        fi
                    fi
                fi
                if [ $triggerSwitchover -ne 1 ]; then
                    $FPING -c 3 -t 100 -p 200 $hfeMgt
                    if [ $? -ne 0 ]; then
                        msg="Switchover: $PEER_HFE_NAME failed to respond in time."
                        if [[ $SWITCHOVER_MODE != "NONE" ]]; then
                            msg="$msg Performing Switchover"
                            triggerSwitchover=1
                        fi
                        loggerHfe "$msg"
                    else
                        waitCount=$((waitCount+1))
                        sleep 0.5s
                        if [ $waitCount -eq 60 ]; then
                            checkHostUpgrade=1 
                            checkIPs=1 
                            waitCount=0
                            setAuthorizationHead # Refesh token
                        fi
                    fi
                fi
            done

            setAuthorizationHead #Get latest auth token
 
            # Re-get routes so we have the latest
            routes="$($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Network/routeTables/$ROUTE_TABLE_NAME?$apiversion" -H "$authhead" | $JQ -r .properties.routes[])"

            ##Update the routes
            loggerHfe "Switchover: Updating Azure routes to new HFE"
            routeIdArr=( )
            routeAddrArr=( )
            while read id; do
                routeIdArr+=( $id )
            done< <(echo $routes | $JQ -r --arg ip "$peerEth2" '. | select(.properties.nextHopIpAddress==$ip) | .id')
            while read addr; do
                routeAddrArr+=( $addr )
            done< <(echo $routes | $JQ -r --arg ip "$peerEth2" '. | select(.properties.nextHopIpAddress==$ip) | .properties.addressPrefix')

            for i in ${!routeIdArr[@]}; do
                resp=$($CURL -s -w "\nAzureRetCode: %{http_code}\n" -H "$authhead" -H "Content-Type: application/json" -d "{ \"properties\": { \"addressPrefix\": \"${routeAddrArr[$i]}\", \"nextHopType\": \"VirtualAppliance\", \"nextHopIpAddress\": \"$localEth2\" } }" -X PUT "https://$AZURE_URI/${routeIdArr[$i]}?$apiversion")
                retCode=$(echo $resp | awk -F'AzureRetCode: ' '{print $2}')
                if [ $retCode -ne 200 ] && [ $retCode -ne 201 ]; then
                    loggerHfe "   Got bad response from Azure: $resp"
                    errorAndExit "Switchover: Failed to update Azure route to address ${routeAddrArr[$i]}"
                fi
                loggerHfe "Switchover: Updated route to ${routeAddrArr[$i]}"
            done

            # Make sure we have the latest information 
            localCeJson="$(haGetHfeInfoJson $HFE_NAME)"
            [ -z "$localCeJson" ] && errorAndExit "Switchover: Failed to get info on HFE $HFE_NAME"
            peerCeJson="$(haGetHfeInfoJson $PEER_HFE_NAME)"
            [ -z "$peerCeJson" ] && errorAndExit "Switchover: Failed to get info on HFE $PEER_HFE_NAME"

            peerPubIp="$(echo $peerCeJson | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.publicIPAddress.id')"
            localPubIp="$(echo $localCeJson | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.publicIPAddress.id')"
 
            ##Remove any null public IPs from eth0
            peerCeJson="$(echo $peerCeJson | $JQ '.eth0 |= del(.properties.ipConfigurations[] | select(.properties.publicIPAddress==null) | .properties.publicIPAddress)')"
            localCeJson="$(echo $localCeJson | $JQ '.eth0 |= del(.properties.ipConfigurations[] | select(.properties.publicIPAddress==null) | .properties.publicIPAddress)')"

            ##Remove any null securtiy groups from eth0
            peerCeJson="$(echo $peerCeJson | $JQ '.eth0 |= del(.properties.ipConfigurations[] | select(.properties.networkSecurityGroup==null) | .properties.networkSecurityGroup)')"
            localCeJson="$(echo $localCeJson | $JQ '.eth0 |= del(.properties.ipConfigurations[] | select(.properties.networkSecurityGroup==null) | .properties.networkSecurityGroup)')"

            if [[ $SWITCHOVER_TYPE == "public" ]]; then 
                if [[ $peerPubIp != "null" ]]; then
                    haPerformPublicSwitchover "$peerCeJson" "$localCeJson" "$peerPubIp" 1 0
                elif [[ $localPubIp != "null" ]]; then
                    # Just update the routetables
                    loggerHfe "Switchover: We are standby but already have public IP. Just updating the Azure routes"
                else
                    loggerHfe "Switchover: Neither HFE nodes have a public IP. Attempting to attach last know public IP $(basename $initPubIp)"
                    haPerformPublicSwitchover "$peerCeJson" "$localCeJson" "$initPubIp" 1 1
                fi
            else
                loggerHfe "Switchover: Private IP switchover" 
                #### TODO: If we ever support multiple IPs on Azure SBC, come back and update this ####
                peerSndIp=$(echo $peerCeJson | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==false) | .properties.privateIPAddress' | head -n 1)
                localSndIp=$(echo $localCeJson | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==false) | .properties.privateIPAddress' | head -n 1)
                localPrimaryIp=$(echo $localCeJson | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.privateIPAddress')

                if [[ $peerSndIp != "" ]]; then
                    haPerformPrivateSwitchover "$peerCeJson" "$localCeJson" "$peerSndIp" 1 0
                elif [[ $localSndIp != "" ]]; then
                    loggerHfe "Switchover: We are standby but already have secondary IP. Just updating the Azure routes"
                else
                    loggerHfe "Switchover: Neither HFE nodes have a secondary IP address. Attempting to use last known secondary IP $initSndIp"
                    haPerformPrivateSwitchover "$peerCeJson" "$localCeJson" "$initSndIp" 1 1
                fi
            fi

            ipMoved=0
            routeUpdated=0
            failTimer=0 #Give 1.5 minute for switchover..
            while [ $ipMoved -ne 1 ] && [ $routeUpdated -ne 1 ] && [ $failTimer -lt 90 ]; do
                failTimer=$((failTimer+2))
                if [ $routeUpdated -ne 1 ]; then
                    if [ $($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Network/routeTables/$ROUTE_TABLE_NAME?$apiversion" -H "$authhead" | $JQ -r .properties.routes[].properties.nextHopIpAddress | grep -cw "$peerEth2") -gt 0 ]; then
                        loggerHfe "Switchover: Some routes still going via peer's eth2. Waiting 2 seconds and retrying"
                        sleep 2
                        continue
                    else
                        loggerHfe "Switchover: Routes are successively updated to our eth2"
                        routeUpdated=1
                    fi
                fi

                if [ $ipMoved -ne 1 ]; then
                    updatedlocalCeJson="$(haGetHfeInfoJson $HFE_NAME)"
                    if [[ $UPGRADE_TYPE == "public" ]]; then
                        if [[ $(echo $updatedlocalCeJson | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.publicIPAddress.id') == "null" ]]; then
                            loggerHfe "Switchover: No public IP found attached to eth0 primary config. Waiting 2 seconds and retrying"
                            sleep 2
                            continue
                        else
                            loggerHfe "Switchover: Public IP successively attached to eth0"
                            ipMoved=1
                        fi
                    else
                        ##Private
                        if [ $(echo $updatedlocalCeJson | $JQ -r '.eth0.properties.ipConfigurations[].properties.privateIPAddress' | grep -c $peerSndIp) -eq 0 ]; then
                            loggerHfe "Switchover: The IP $peerSndIp has not been assigned to eth0 yet. Waiting 2 seconds and retrying"
                            sleep 2
                            continue
                        else
                            loggerHfe "Switchover: Private IP successively attached to eth0"
                            ipMoved=1

                            #inet 10.4.6.5/24 brd 10.4.6.255 scope global eth0
                            loggerHfe "Switchover: Adding new IP to eth0 as secondary"
                            prefix=$($IP addr show $ETH0 | grep $localPrimaryIp | awk -F' ' '{print $2}' | awk -F'/' '{print $2}')
                            brd=$($IP addr show $ETH0 | grep $localPrimaryIp | awk -F' ' '{print $4}')

                            $IP addr add $peerSndIp/$prefix dev $ETH0 scope global brd $brd
                            if [ $($IP addr show dev $ETH0 | grep -ic 'state down') -gt 0 ]; then
                                if [ -e $NETPLAN_ROOT ]; then
                                    netplan apply
                                    if [ $? -ne 0 ]; then
                                        loggerHfe "Switchover: Error. Failed to bring up eth0. Rebooting"
                                        reboot
                                    fi
                                fi
                            fi

                            if [ $($IP addr show $ETH0 | grep -c $peerSndIp) -eq 0 ]; then
                                errorAndExit "Switchover: New IP address not added $ETH0. Exiting!"
                            fi
                        fi
                    fi
                fi
            done

            if [ $failTimer -eq 90 ]; then
                loggerHfe "Switchover: Switchover mechanism failed! Allowing HFE to re-check if old active is up"
                continue
            fi

            loggerHfe "Switchover: Successful switchover to active. Starting HFE setup."
            break
        else
            errorAndExit "Switchover: Value returned for number of IPs in routetable is less than zero! localCount=$localCount peerCount=$peerCount. Exiting!"
        fi
    done
}


# Main logic of HFE
checkSbcConnectivity()
{
    local switched=0
    local counter=0
    local verifyCount=0
    local haCheckCount=0
    local needMgmtRoute=0

    # Test for timeout after all iptable rules are setup.
    # We use this to know if we have to add a route to 
    # the resolved ip address of management.azure.com 
    # to make requests out about the Azure routes.
    # We use this in HA to understand if we are no longer active (and reboot)
    $CURL https://$AZURE_URI
    if [ $? -eq 28 ]; then
        needMgmtRoute=1
    fi


    while :; do
############################################### CAUTION #################################
#
#        -t (value) < -P (value)
#
#########################################################################################
#      -c, --count=N
#
#        Number of request packets to send to each target. In this mode, a line is
#        displayed for each received response (this can suppressed with -q or -Q). Also,
#        statistics about responses for each target are displayed when all requests have
#        been sent (or when interrupted).
#
#
#      -p, --period= MSEC
#
#        In looping or counting modes (-l, -c, or -C), this parameter sets the time in
#        milliseconds that fping waits between successive packets to an individual
#        target. Default is 1000 and minimum is 10.
#
#
#
#      -t, --timeout= MSEC
#
#        Initial target timeout in milliseconds. In the default, non-loop mode,
#        the default timeout is 500ms, and it represents the amount of time that
#        fping waits for a response to its first request. Successive timeouts are
#        multiplied by the backoff factor specified with -B.
#
#        In loop/count mode, the default timeout is automatically adjusted to
#        match the "period" value (but not more than 2000ms). You can still
#        adjust the timeout value with this option, if you wish to, but note that
#        setting a value larger than "period" produces inconsistent results,
#        because the timeout value can be respected only for the last ping.
#
#        Also note that any received replies that are larger than the timeout
#        value, will be discarded.
############################################################################################

        # Use first IP
        $FPING -c 3 -t 100 -p 200 ${ACTIVE_SBC_IP_ARR[0]} &> /dev/null
        if [ $? -ne 0 ]
        then
            if [ $switched -eq 0 ]; then
                statusLoggerHfe "Connection error detected to Active SBC: $ACTIVE_SBC_VM_NAME. ${ACTIVE_SBC_IP_ARR[0]} did not respond. Attempting switchover."
                switched=1
            elif [ $switched -eq 1 ] && [ $(($counter % 10)) -eq 0 ]; then
                statusLoggerHfe "Connection error ongoing - No connection to SBC $SBC_PKT_PORT_NAME from HFE"
            fi
            
            # If single HFE (2.0), this is PKT0 config
            TEMP_SBC_IP_ARR=( ${ACTIVE_SBC_IP_ARR[@]} )
            ACTIVE_SBC_IP_ARR=( ${STANDBY_SBC_IP_ARR[@]} )
            STANDBY_SBC_IP_ARR=( ${TEMP_SBC_IP_ARR[@]} )

            TEMP_SBC_NAME=$ACTIVE_SBC_VM_NAME
            ACTIVE_SBC_VM_NAME=$STANDBY_SBC_VM_NAME
            STANDBY_SBC_VM_NAME=$TEMP_SBC_NAME

            if [[ $MODE == "single" ]]; then
                TEMP_SBC_IP_PKT1_ARR=( ${ACTIVE_SBC_IP_PKT1_ARR[@]} )
                ACTIVE_SBC_IP_PKT1_ARR=( ${STANDBY_SBC_IP_PKT1_ARR[@]} )
                STANDBY_SBC_IP_PKT1_ARR=( ${TEMP_SBC_IP_PKT1_ARR[@]} )
            fi

            statusLoggerHfeDebug "Before clearOldIptableRules"
            clearOldIptableRules
            statusLoggerHfeDebug "After clearOldIptableRules, and just before configureNATRules"

            statusLoggerHfeDebug "Verifying Routes"
            verifyRoutes
            verifyCount=0

            if [[ $MODE == "single" ]]; then
                configureNATRules $ETH0 $ETH3 $ETH3_GW
                configureNATRules $ETH1 $ETH4 $ETH4_GW
            else
                configureNATRules $ETH0 $ETH2 $ETH2_GW
            fi

            statusLoggerHfeDebug "After configureNATRules and just before clearOldRules"
            clearOldRules
            statusLoggerHfeDebug "After clearOldRules"
            
            let counter=$counter+1 #increment log count
            statusLoggerHfeDebug "Wait for SBC to complete switchover."
            statusLoggerHfeDebug ""
            statusLoggerHfeDebug ""
            sleep 2s
	    else
            if [ $INIT_LOG -eq 1 ]; then
                statusLoggerHfe "Initial HFE startup configuration complete. Successfully connected to $ACTIVE_SBC_VM_NAME"
                INIT_LOG=0
                switched=0
                counter=0
            elif [ $switched -eq 1 ]; then
                statusLoggerHfe "Switchover from old Active $TEMP_SBC_NAME to new Active $ACTIVE_SBC_VM_NAME complete. Connection established."
                switched=0
				counter=0
            fi
            statusLoggerHfeDebug "Nothing needs to be done, active is up $ACTIVE_SBC_VM_NAME"
	        sleep 0.5s
        fi
 
        # Verify Routes every ~10 secs
        if [ $verifyCount -eq 20 ]; then
            verifyRoutes
            if [ $? -eq 1 ]; then
                statusLoggerHfe "Routes have changed. Flushing conntrack"
                clearOldRules
            fi
            verifyCount=0
        fi
        let verifyCount=$verifyCount+1

        # Verify every ~30seconds we are still supposed to be active
        if [ $haCheckCount -eq 60 ]; then
            if [ $HA_MODE -eq 1 ]; then
                statusLoggerHfeDebug "Checking if we are still active CE and if IPs are attached"
                recoverIp=""

                setAuthorizationHead ##Refresh token
                GW=$(getGatewayIp $ETH1)
                eth2ip=$($IP addr show dev $ETH2 | grep "inet" | head -1 | awk -F' ' '{print $2}' | awk -F'/' '{print $1}')
                for try in 1 2; do
                    if [ $needMgmtRoute -eq 1 ]; then
                        GW=$(getGatewayIp $ETH1)
                        azureIp="$(dig +short $AZURE_URI | tail -1)"
                        statusLoggerHfeDebug "Adding route to the Azure IP $azureIp via $GW dev $ETH1"
                        $IP route add $azureIp via $GW dev $ETH1
                    fi
                    nextHopIps="$($CURL "https://$AZURE_URI/subscriptions/$SUB_ID/resourceGroups/$RG_NAME/providers/Microsoft.Network/routeTables/$ROUTE_TABLE_NAME?$apiversion" -H "$authhead" | $JQ -r .properties.routes[].properties.nextHopIpAddress)"
                    if [ $? -eq 23 ]; then
                        [ $needMgmtRoute -eq 1 ] && $IP route delete $azureIp via $GW dev $ETH1
                        needMgmtRoute=1   ##We timed out. Retry
                        statusLoggerHfe "Request timed out getting routes. Retrying"
                        continue
                    fi
                    localCeJson="$(haGetHfeInfoJson $HFE_NAME)"
                    if [ -z "$localCeJson" ]; then
                        [ $needMgmtRoute -eq 1 ] && $IP route delete $azureIp via $GW dev $ETH1
                        needMgmtRoute=1   ##We timed out. Retry
                        statusLoggerHfe "Request timed out getting local CE information. Retrying"
                        continue
                    fi
                    peerCeJson="$(haGetHfeInfoJson $PEER_HFE_NAME)"
                    if [ -z "$peerCeJson" ]; then
                        [ $needMgmtRoute -eq 1 ] && $IP route delete $azureIp via $GW dev $ETH1
                        needMgmtRoute=1   ##We timed out. Retry
                        statusLoggerHfe "Request timed out getting peer CE information. Retrying"
                        continue
                    fi
                    if [[ "$SWITCHOVER_TYPE" == "public" ]]; then
                        if [[ $(echo "$localCeJson" | $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.publicIPAddress.id') == "null" ]] && [[ $(echo "$peerCeJson" |  $JQ -r '.eth0.properties.ipConfigurations[] | select(.properties.primary==true) | .properties.publicIPAddress.id') == "null" ]]; then
                            # Neither of us have public IPs. Try to get it back from initPubIp
                            recoverIp="$initPubIp"
                        fi
                    else
                        if [ $(echo "$localCeJson" | $JQ '.eth0.properties.ipConfigurations | length') -lt 2 ] && [ $(echo "$peerCeJson" | $JQ '.eth0.properties.ipConfigurations | length') -lt 2 ]; then
                            recoverIp="$initSndIp"
                        fi
                    fi
                    if [ $(echo "$nextHopIps" | grep -cw "$eth2ip") -eq 0 ]; then
                        statusLoggerHfe "Detected no Azure routes are our IP address. Rebooting to become standby"
                        loggerHfe "Switchover: Detected no Azure routes are our IP address. Rebooting to become standby"
                        if [ ! -z "$recoverIp" ]; then
                            loggerHfe "Switchover: No IPs found for either CE. Attempting to recover $(basename $recoverIp) and give to peer before reboot"
                            haReturnIpAddress "$localCeJson" "$peerCeJson" "$recoverIp" 1 "$PEER_HFE_NAME"
                            if [ $? -ne 0 ]; then
                                loggerHfe "Switchover: Failed to recover $(basename $recoverIp) and give to peer. Will try again before rebooting"
                            else
                                reboot
                            fi
                        else 
                            reboot
                        fi
                    else
                        if [ ! -z "$recoverIp" ] ; then
                            # We are active but no-one has the IPs. Recover them to us
                            haReturnIpAddress "$peerCeJson" "$localCeJson" "$recoverIp" 1 "$HFE_NAME"
                            if [ $? -eq 0 ] && [[ "$SWITCHOVER_TYPE" == "private" ]]; then
                                # Check if we need to add the address
                                if [ $($IP addr show dev $ETH0 | grep -c "$targetIp/") -eq 0 ]; then
                                    prefix=$($IP addr show $ETH0 | grep -w inet | head -1 | awk -F' ' '{print $2}' | awk -F'/' '{print $2}')
                                    brd=$($IP addr show $ETH0 | grep $localPrimaryIp | awk -F' ' '{print $4}')
                                    $IP addr add $recoverIp/$prefix dev $ETH0 scope global brd $brd
                                    if [ $($IP addr show dev $ETH0 | grep -ic 'state down') -gt 0 ]; then
                                        if [ -e $NETPLAN_ROOT ]; then
                                            netplan apply
                                            if [ $? -ne 0 ]; then
                                                loggerHfe "Switchover: Error. Failed to bring up eth0, Rebooting"
                                                reboot
                                            fi
                                        fi
                                    fi
                                fi
                            fi
                        fi
                    fi
                    [ $needMgmtRoute -eq 1 ] && $IP route delete $azureIp via $GW dev $ETH1 && statusLoggerHfeDebug "Deleting route to the Azure IP $azureIp"
                    break
                done
            fi
            haCheckCount=0
        fi
        let haCheckCount=$haCheckCount+1

    done
}

main()
{
    case $1 in
        "setup") 
            prepareHFEInstance
            preConfigure
            readConfig
            getAzureVars
            verifyAuthorization $ACTIVE_SBC_VM_NAME
            haCheckIfStandby
            setInterfaceMap
            extractHfeConfigIps
            configurePrimaryIpMap
            if [[ $MODE == "single" ]]; then
                # Configure Routes for Active SBC PKT0
                configureSubnetRoute $ACTIVE_SBC_VM_NAME "PKT0" $ETH3_GW $ETH3
                configureNATRules $ETH0 $ETH3 $ETH3_GW 1
                # Configure Routes for Active SBC PKT1
                configureSubnetRoute $ACTIVE_SBC_VM_NAME "PKT1" $ETH4_GW $ETH4
                configureNATRules $ETH1 $ETH4 $ETH4_GW 1
            else
                # Configure Routes for Active SBC
                configureSubnetRoute $ACTIVE_SBC_VM_NAME $SBC_PKT_PORT_NAME $ETH2_GW $ETH2
                configureNATRules $ETH0 $ETH2 $ETH2_GW 1
            fi
            handleTempMgmtRoute 0
            configureMgmtNAT
            addCustomerRoutes
            setSecondaryIntfQueueLen
            clearOldRules 1
            showCurrentConfig
            checkSbcConnectivity
            ;;
        "cleanup")
            prepareHFEInstance
            preConfigure
            readConfig
            getAzureVars
            setInterfaceMap
            extractHfeConfigIps
            routeCleanUp
            clearOldRules 1
            doneMessage
            ;;
        "sysdump")
            prepareHFEInstance
            sysdump
            ;;
        *) 
            usage "Unrecognized switch"
            ;;
    esac
}

[[ $# -ne 1 ]] && usage

main $1
