#!/usr/bin/env python3

# import re
import sys
import shutil
import os

import datetime
import hashlib
import tarfile
import traceback
import glob
import subprocess
from distutils.dir_util import copy_tree
import json
import uuid
import codecs
import time
import getpass

warning_msg = ""
template_author = "Ribbon"
def_sub_version = ""
def_product_version = ""
helm_version = "v3"
script_dir = os.path.dirname(os.path.realpath(__file__))
vnfd_template = ""
vnfd_proprietary_template = os.path.join(script_dir, "cnfVnfdTemplate.yaml")
cnf_vnfd_sol001_template = os.path.join(script_dir, "cnfSol001VnfdTemplate.yaml")
vnfd_ribbon_custom_types_template = os.path.join(script_dir, "helm_extensions_1_1.yaml")
etsi_vnfd_types_file = os.path.join(script_dir, "etsi_nfv_sol001_vnfd_2_5_1_types.yaml")
etsi_common_types_file = os.path.join(script_dir, "etsi_nfv_sol001_common_types.yaml")
product_name = "rbbn-core-cnf"
customer_desc_name = "Verizon Communications"
customer_name = "verizon"
customer_short_name = "vz"
privKeyFile = "./csarPrivKey.pem"
pubCertFile = "./csarCert.pem"

env_var_val_dict = {}
env_args = ""
cmd_arg_list_usage = {
        "-h": "Below is the list of command line arguments, See '--help_custom' to set arguments via environment\n  varibles. Additionally we can porvide some of the parmeters through json file. Check the sample\n  json file. The precedence order is json, command line, environment if a parameter is provided\n  in multiple options.\n",
        "--product_version": "--product_version <product_version>\t - Set product version. e.g. - '12.01.00'.",
        "--sub_version": "--sub_version <sub_version>\t\t - Set product sub/release name version. e.g. - 'R002'.",
        "--help_custom": "--help_custom\t\t\t\t - To see the all the possible custom variables that can be set.",
        "--product_name": "--product_name <product name>\t\t - Set product name. It needs to be single word, default is 'rbbn-core-cnf'. ",
        "--provider": "--provider <Provider name>\t\t - Set Provider name. e.g., - 'Ribbon Communications'.",
        "--customer_name": "--customer_name <Customer name>\t - Set Customer name to be set in vnfd definitions like node types. e.g., - 'verizon'.",
        "--customer_desc_name": "--customer_desc_name <Customer name>\t - Set Customer name. e.g., - 'Verizon Communications'.",
        "--customer_short_name": "--customer_short_name <short name>\t - Set short Customer name. e.g., - 'vz'.",
        "--product_build": "--product_build <product_build>\t - Set the product_build E.g. '12.1.1-100'.",
        "--passwd_token": "--passwd_token <token>\t\t - Set password token to download helm chart. There is also a provision to specify\n\t\t\t\t\t   username/password.",
        "--username": "--passwd_token <token>\t\t - Set username along with password token to download helm chart.",
        "--user_password": "--passwd_token <token>\t\t - Set password to download helm chart. This has to be provided when username\n\t\t\t\t\t   is specified instead of passwd_token.",
        "--sign_files": "--sign_files <true/false>\t\t - If set to 'true' generate signature for all file. Default is true and self signed.\n\t\t\t\t\t   To use CA signed certificates, set '--cert_file' and '--key_file'",
        "--cert_file": "--cert_file <cert file>\t\t - Set cert file with aboslute path",
        "--key_file": "--key_file <key file>\t\t\t - Set cert file with aboslute path",
        "--sign_csar": "--sign_csar <csar file>\t\t - Produces new csar from the given csar file by updating manifest file with checksum and signing\n\t\t\t\t\t   all the files. The existing signature file, if any,  will be deleted. The checksum in manifest\n\t\t\t\t\t   file will be updated with newer checksum. 'key' and 'cert' files can be provided for CA signing.",
        "--json_file": "--json_file <json file>\t\t - Set Json file path containing configuration various artifacts\n\t\t\t\t\t   like Helm charts, Images, Test scripts etc., Refer sample Json file. ",
        "--values_file_name_prefix": "--values_file_name_prefix <prefix>\t - Provide prefix to name of values.yaml and i/p json files as per customer naming convention.\n\t\t\t\t\t   The prefix will be added as <prefix>_values.yaml. The normal format for values yaml file\n\t\t\t\t\t   is '<namespace>_<cnf>_values.yaml'. So prefix has to include both namespace and product name."

        }

# CustomEnv members
#   env - environment variable name, default values, validation of 
#   default_val - default values
#   validate - if the validation is required or not
#   mandate_arg - mandates to be given via command line or source it as custom variable
class CustomEnv:
    def __init__(self, env_name, default_val='', validate=False, mandate_arg=False):
        global env_args
        self.env = env_name
        self.validate = validate
        self.mandate_arg = mandate_arg
        if env_name in os.environ:
            self.val = str(os.environ[env_name])
            if env_name != "passwd_token":
                env_args += "#export " + env_name + "=\"" + self.val + "\"\n"
        else:
            self.val = default_val


custom_env_name = {
    "${product_version}": CustomEnv("product_version", def_product_version, False, True),
    "${sub_version}": CustomEnv("sub_version", def_sub_version, False, True),
    "${vnfm}": CustomEnv("vnfm", "k8s", False, True),
    "${flavour_id}": CustomEnv("flavour_id", "default", False, False),
    "${flavour_description}": CustomEnv("flavour_description", "default", False, False),
    "${descriptor_version}": CustomEnv("descriptor_version", "1.0.0_PB1", False, False),
    "${product_name}": CustomEnv("product_name", product_name, False, True),
    "${product_info_name}": CustomEnv("product_info_name", "RIBBON CORE SBC", False, True),
    "${provider}": CustomEnv("provider", "Ribbon Communications", False, True),
    "${customer_name}": CustomEnv("customer_name", customer_name, False, True),
    "${customer_desc_name}": CustomEnv("customer_desc_name", customer_desc_name, False, True),
    "${customer_short_name}": CustomEnv("customer_short_name", customer_short_name, False, True),
    "${product_build}": CustomEnv("product_build", "", False, True),
    "${passwd_token}": CustomEnv("passwd_token", ""),
    "${username}": CustomEnv("username", ""),
    "${user_password}": CustomEnv("user_password", ""),
    "${sign_files}": CustomEnv("sign_files", "true", True, False),
    "${cert_file}": CustomEnv("cert_file", ""),
    "${key_file}": CustomEnv("key_file", ""),
    "${sign_csar}": CustomEnv("sign_csar", ""),
    "${json_file}": CustomEnv("json_file", "./artifacts.json", False, True),
    "${template_version}": CustomEnv("template_version", "2.5.1"),
    "${helm_version}": CustomEnv("helm_version", helm_version),
    "${values_file_name_prefix}": CustomEnv("values_file_name_prefix", ""),
    "${include_certifcate}": CustomEnv("include_certifcate", ""),
    "${skip_cmd_arg}": CustomEnv("skip_cmd_arg", "true"),
    "${skip_env_arg}": CustomEnv("skip_env_arg", "true"),
    "${cpuResourceMinRequest}": CustomEnv("cpuResourceMinRequest", "1.0"),
    "${cpuResourceMaxRequest}": CustomEnv("cpuResourceMaxRequest", "5.0"),
    "${memoryResourceMinRequest}": CustomEnv("memoryResourceMinRequest", "10"),
    "${memoryResourceMaxRequest}": CustomEnv("memoryResourceMaxRequest", "20"),
    "${localStorageResourceMinRequest}": CustomEnv("localStorageResourceMinRequest", "20"),
    "${localStorageResourceMaxRequest}": CustomEnv("localStorageResourceMaxRequest", "50"),
    "${persistentStorageResourceMinRequest}": CustomEnv("persistentStorageResourceMinRequest", "100"),
    "${persistentStorageResourceMaxRequest}": CustomEnv("persistentStorageResourceMaxRequest", "200"),
    "${localStorageResourceMinRequest}": CustomEnv("localStorageResourceMinRequest", "20"),
    "${min_number_of_instances}": CustomEnv("min_number_of_instances", "1"),
    "${max_number_of_instances}": CustomEnv("max_number_of_instances", "1"),
    "${jenkins_csar}": CustomEnv("jenkins_csar", "false"),
    "${job_name}": CustomEnv("job_name", ""),
}


def print_example():
    print("\nExamples:\n\n")
    print("./createCnfCsar.py --product_version \"12.01.02\" --sub_version R002 --product_build \"12.1.2-R002-336\" --product_name \"rbbn-core-cnf\" --provider  \"Ribbon Communications\" --customer_short_name \"vz\" --customer_desc_name \"Verizon Communications Inc.\" --json_file \"./artifacts.json\"")
    print("./createCnfCsar.py --product_version \"12.01.02\" --sub_version R002 --product_build \"12.1.2-155\" --product_name \"rbbn-core-cnf\" --provider  \"Ribbon Communications\" --sign_files false") 
    print("./createCnfCsar.py --product_version \"12.01.02\" --sub_version R002 --product_build \"12.1.2-155\" --product_name \"rbbn-core-cnf\" --provider  \"Ribbon Communications\" --json_file \"./artifacts.json\"") 
    print("./createCnfCsar.py --product_version \"12.01.03\" --sub_version R001 --product_build \"12.1.3-15\" --product_name \"rbbn-core-cnf\" --provider  \"Ribbon Communications\" --cert_file \"/home/tmp/cert.pem\" --key_file \"/home/tmp/key.pem\"") 
    print (" python3 createCnfCsar.py --product_version \"12.01.02\" --sub_version R001 --product_build \"12.1.2-155\" --product_name \"rbbn-core-cnf\" --provider \"Ribbon Communications\"  --customer_short_name \"dt\" --customer_desc_name \"Deutsche Telekom\" --sign_files false ")
    print("./createCnfCsar.py --sign_csar /tmp/rbbn-core-cnf-12.1.2-116.csar")
    print("./createCnfCsar.py --sign_csar /tmp/rbbn-core-cnf-12.1.2-116.csar --cert_file \"/home/tmp/cert.pem\" --key_file \"/home/tmp/key.pem\"")
    print("")

def usage_help():
    print("")
    for argument in cmd_arg_list_usage:
        print("  " + cmd_arg_list_usage[argument])
    print("")
    print_example()

def usage(command_name, arg, verbose=False):
    if arg in cmd_arg_list_usage:
        print(" Invalid value for : " + arg)
        print("\n Usage of " + command_name + " for "+ arg + " is -")
        print("\n\t"+ cmd_arg_list_usage[arg])
        print("")

        print("\n See '-h' for additional options\n\n")
    else:
        print(" Unexpected option: " + arg)
        print("\n\n See '-h' for valid options\n\n")


def usage_help_custom(command_name):
    print("\n Below is the list of custom variables that can be set as environment variables via export command.")
    print(" These are used for customizing the contents of files such as vnfd, mf, meta etc. and the file names.")
    print(" Most of the environment variables name matches with the words within '{}' that are to be replaced in")
    print(" cnfSol001VnfdTemplate.yaml and helm_extensions_1_1.yaml. If variables are not specified, they will be")
    print(" written with default values.\n")
    for custom_name_key in custom_env_name:
        if custom_env_name[custom_name_key].env != "jenkins_csar" and custom_env_name[custom_name_key].env != "job_name":
            print("\t - " + custom_env_name[custom_name_key].env)
    print("\n  Some examples:-\n\n\texport vnfm=\"[0:'HPE']\"\n\texport customer_desc_name=\"Verizon communications\"\n\texport customer_short_name=\"vz\"\n\texport provider=\"Ribbon Communications\"\n")
    print("")

def recom_variables(command_name):
    rec_args = ""
    for iKey in custom_env_name:
        if (custom_env_name[iKey].mandate_arg == True):
            if rec_args == "":
               rec_args = custom_env_name[iKey].env
            else:
                rec_args += ", " + custom_env_name[iKey].env
    print("Warning: No command line arguments are provided. See '-h' for options and examples. It is mandate to provide values to product_version, sub_version and product_build attributes. The attributes can be set either through command line or shell environment variables using export command. It is recommended to provide below attributes to appropriately customize CSAR and its contents.\n The attributes are - " + rec_args + ".\n")

def validateArgValues():
    prod_bld = custom_env_name["${product_build}"].val.lower()
    if prod_bld.find(".") == -1 or prod_bld.find("-") == -1:
        print ("Invalid value '" + prod_bld + "' for 'product_build' ")
        return False
    return True

def get_args(argv=None):
    # To try to get this to run on older python versions, don't use argparse
    global cmdline_args
    cmdline_args = ' '.join(sys.argv[1:])
    print("\ncmdline:" + cmdline_args)
    print("")
    global sub_version
    global vnfd_template
    global product_version
    global helm_version

    global vnfd_custom_types_file
    vnfd_custom_types_file = ""
    global tosca_def_v
    tosca_def_v = "tosca_simple_yaml_1_2"

    global env_var_to_print
    env_var_to_print = []
    env_name = ""
    next_arg = ""
    prev_arg = ""
    if len(argv) > 1:
        for arg in argv[1:]:
            if arg.startswith('---'):
                usage(argv[0], arg)
                return False
            elif env_name != "":
                if prev_arg not in cmd_arg_list_usage and prev_arg != "--jenkins_csar" and prev_arg != "--job_name":
                    print ("[" + prev_arg + "] is invalid command line argument: please use -h option to see valid argumnets and usage")

                    exit()
                if not arg.startswith('-') and not arg.startswith('--'):
                    env_var_val_dict[env_name] = arg
                    env_name = ""
                else:
                    usage(argv[0], arg)
                    return False
            elif arg.startswith('--'):
                if arg == "--help_custom":
                    usage_help_custom(argv[0])                    
                    exit()
                env_name = arg[2:]
                prev_arg = arg
            elif arg.startswith('-'):
                prev_arg = arg
                if arg == "-h":
                    usage_help()                    
                    exit()
                else:
                    usage(argv[0], arg)
                    return False
            else:
                usage(argv[0], arg)
                return False
        if env_name != "":
            if prev_arg in env_var_val_dict:
                print ("The argument [" + prev_arg + "] not passed with a value")
                usage(argv[0], arg)
            else:
                #print("Invalid argument: " + arg)
                usage(argv[0], arg)
                return False
            return False

    else:
    #   usage_help()
    #   exit()
        recom_variables(argv[0])
    return True



def hash_file_sha256(file_to_hash):
    # Since this function could be used for very large files like the qcow2
    # image, it's important to only read small blocks of it into memory at a
    # time. Use 64k blocks as a reasonable size to read into memory.
    BLOCKSIZE = 65536
    hasher = hashlib.sha256()
    with open(file_to_hash, 'rb') as afile:
        buf = afile.read(BLOCKSIZE)
        while len(buf) > 0:
            hasher.update(buf)
            buf = afile.read(BLOCKSIZE)
    return hasher.hexdigest()

def download_container_images(image_artifactory_url, image_name, product_build,  destn_dir, username, passwd):
    if not os.path.exists(destn_dir):
        os.makedirs(destn_dir, exist_ok=True)
    #if len(username) > 0  and len (passwd) > 0:
    #    os.system("docker login -u " + username + " -p " + passwd + "" + image_artifactory_url)

    pull_path = image_artifactory_url + "/" + image_name + ":" + product_build
    #print("PULL PATH :" + pull_path)
    if (os.system("docker pull " + pull_path)) != 0:
        return
    destn_tar = destn_dir + "/" + image_name + "Image.tar"
    imageNameToSave = image_name + ":" + product_build
    if (os.system("docker tag " + pull_path + " " + imageNameToSave)) != 0:
        return
    os.system("docker save " + imageNameToSave + " > " + destn_tar)

def download_file(url, file, destn_dir, token, username, password):

    command = ""
    if (url.startswith("https")):
        source_url =  url + "/" + file
    else:
        source_url = "https://" + url + "/" + file
    if len(token) > 0:
        option = " -sSf -H 'Authorization: Bearer " + token + "' "
        destn_file = " -o " + destn_dir + "/" + file
        command = "curl " + option + source_url + destn_file
        print ("Downloading " + source_url + " using token ")
        #print ("curlCommand " + command)
    elif len(username) > 0  and len(password) > 0 :
        option = " --user=" + username + " --password=" + password + " "
        destn_file = " -O " + destn_dir + "/" + file
        command = "wget " + option + source_url + destn_file
        print ("Downloading " + source_url + " using username/passwd ")
        #print ("wget  " + command)
    else:
        print("Warning: Neither password token nor username/passwrd is provided, Hence unable to download " +  source_url)
        return
    os.system(command)

def calculateCheckSum(csarRootDir="./CSAR"):
    print ("Calculating CheckSum for all the files.... under " + csarRootDir)
    cs_op_str=''
    for root, dirs, files in os.walk(csarRootDir):
        for f in files:
            file = "./" + os.path.relpath(root, csarRootDir) + "/" + f
            if not f.endswith(".mf"):
                cs_op_str += "\n"
                cs_op_str += "Source: " + file + "\n"
                cs_op_str += "Algorithm: SHA-256\n"
                cs_op_str += "Hash: : " + hash_file_sha256(root + "/" + f) + "\n"

    return cs_op_str

def createCsarArtifactsFromJson(csarRootDir, token, username, passwd,  artifact_json, jenkins_build):
    artifacts = {}
    try:
        f = open(artifact_json)
    except:
        return artifacts

    # JSON object dictionary
    try:
        artifacts = json.load(f)
    except:
        print(" Failed to load artifacts!")
        return artifacts

    for i in artifacts:
        for j in artifacts[i]:
            if artifacts[i][j]["archiveOrNot"].lower() != "true":
                continue
            if (artifacts[i][j]["csarPath"] == ""):
                print("Warning: csarPath for '[" + i + "][" + j + "]' does not exist, Continueing...")
                continue
            csar_output_dir = os.path.join(csarRootDir, artifacts[i][j]["csarPath"])
            os.makedirs(csar_output_dir, exist_ok=True)  
            if jenkins_build != "true" and artifacts[i][j]["download"].lower() == "true" :
                if artifacts[i][j]["downloadUrl"] == "":
                    print ("Warning: For Artifacts '[" + i + "][" + j + "]' download is set to true in json i/p file, but url is empty, Continue to next Artifacts...")
                    continue
                if artifacts[i][j]["isDockerImage"].lower() != "true" :
                    files = artifacts[i][j]["files"].split(",")
                    for file in files:
                        download_file(artifacts[i][j]["downloadUrl"], file.strip(), csar_output_dir, token, username, passwd)
                        #print("\t : " + file.strip())
                    continue

                # download docker images
                print ("Warning: If downloading docker image is failing, Re-run after executing 'docker login <image artifactory>' ...")
                images=artifacts[i][j]["images"].split(",")
                for image in images:
                    #print("\t : " + image.strip())
                    download_container_images(artifacts[i][j]["downloadUrl"].strip(), image.strip(), artifacts[i][j]["imageTag"], csar_output_dir, username, passwd)
                continue

            sourcePath = artifacts[i][j]["sourcePath"]
            if sourcePath == "" or not os.path.exists(sourcePath):
                print("Warning: sourcePath " + sourcePath + " for '[" + i + "][" + j + "]' does not exist in json, Continue to next Artifacts...")
            else:
                shutil.copytree(sourcePath, csar_output_dir, dirs_exist_ok=True)
                #print ("csar_output_dir  " +csar_output_dir)
                
    # Closing file
    f.close()
    return artifacts

def buildVnfdAndManifestOutputString(csarRootDir="./CSAR"):
    cs_op_str=''
    for root, dirs, files in os.walk(csarRootDir):
        for f in files:
            file = "./" + os.path.relpath(root, csarRootDir) + "/" + f
            if not f.endswith(".mf"):
                cs_op_str += "\n"
                cs_op_str += "Source: " + file + "\n"
                cs_op_str += "Algorithm: SHA-256\n"
                cs_op_str += "Hash: : " + hash_file_sha256(root + "/" + f) + "\n"

    return cs_op_str

########################################
# Function to generate self-signed key and certificate
########################################
def generateSelfSignedKeyCert(keyFileName="/tmp/key.pem", certFileName="/tmp/cert.pem"):
    cmd = "openssl req -nodes -newkey rsa:2048 -subj '/C=US/ST=Texas/O=RibbonCommunications/localityName=Plano/commonName=rbbn.com/organizationalUnitName=C3/emailAddress=support@rbbn.com' -x509 -days 365 -keyout " + keyFileName + " -out " + certFileName + " -sha256 1>/dev/null" 
    #print("Generating self signed key and certificate. \n" + cmd)
    os.system(cmd)
    return keyFileName, certFileName
    
########################################
# Function to sign files using OpenSSL CMS
########################################
def signFiles(csarRootDir, keyFileName="/tmp/key.pem", certFileName="/tmp/cert.pem"):
    print("Signing all the files.")
    cmd_part1 = "openssl cms -sign -signer " + certFileName + " -inkey " + keyFileName + " -binary -in "
    cmd_part2 = " -outform PEM  -out "

    for root, dirs, files in os.walk(csarRootDir):
        for f in files:
            file = root + "/" + f
            sigFile = file + ".sig"
            if os.path.exists(sigFile):
                os.remove(sigFile)
            cmd = cmd_part1 + file + cmd_part2 + sigFile
            os.system(cmd)
#            if file.endswith(".mf"):
#                os.system("cat " + sigFile + " >> " +  file)
#                os.remove(sigFile)

def extractCsar(csarfile, destnFolder):
    print ("Extracting " + csarfile)
    shutil.rmtree(destnFolder, ignore_errors=True)
    os.system("unzip " + csarfile + " -d " + destnFolder + "  1>/dev/null")

def removeSigFiles(csarRootDir):
    print ("Removing Signature files" )
    for root, dirs, files in os.walk(csarRootDir):
        for f in files:
            file = root + "/" + f
            #print(" files : "+ file)
            if file.find(".sig") != -1:
                os.remove(file)



def updateManifest(outputDir):
    manifestFile = ""
    for file in os.listdir(outputDir):
        if file.endswith(".mf"):
            manifestFile = os.path.join(outputDir, file)

    if manifestFile == "":
        print("Manifest file does not exist")
        return
    print("Updating " + manifestFile + " for checksum")
    with open(manifestFile, "r+") as f:
        fileContents = f.read()
        fileContents =  fileContents[0 : fileContents.find("Source: .") ] + calculateCheckSum(outputDir)
        f.write(fileContents)

    f.close()

def createCsar (output_dir, csar_file):
    try:
        import zipfile
        zipf = zipfile.ZipFile(csar_file, 'w', zipfile.ZIP_DEFLATED)
        for root, dirs, files in os.walk(output_dir):
            for d in dirs:
                # Write directories so even empty directories are copied:
                arcname = os.path.relpath(os.path.join(root, d), output_dir)
                # Do not compress folders.
                zipf.write(os.path.join(root, d), arcname, compress_type=zipfile.ZIP_STORED)
            for f in files:
                # hidden files.
                arcname = os.path.relpath(os.path.join(root, f), output_dir)
                zipf.write(os.path.join(root, f), arcname, compress_type=zipfile.ZIP_DEFLATED)

    except Exception as zip_e:
        if os.path.exists (csar_file):
            os.remove(csar_file)
        print("Could not import zipfile: " + str(zip_e) + ", using zip function instead")
        # Zip including hidden files.
        os.system("cd " + output_dir + ";zip -qr " + csar_file + " . ;cd " + current_dir)


def updateTagAndRebuildHelm(csar_dir, helm_charts_dir, helm_values_dir, product_build):
    values_output_string=""
    tagToReplace = product_build[0:product_build.index("-")] + "-latest"
    helm_charts_dir = os.path.join(csar_dir, helm_charts_dir)
    helm_values_dir = os.path.join(csar_dir, helm_values_dir)
    os.makedirs(helm_charts_dir, exist_ok=True)
    os.makedirs(helm_values_dir, exist_ok=True)

    #Extract all global Values yaml from helm tar ball and copy them helm_values_dir
    if os.path.exists("./helm_temp"):
        shutil.rmtree("./helm_temp")
    os.mkdir("./helm_temp")

    for helm_tar_ball in os.listdir(helm_charts_dir):
        values_output_string += "        deployment_chart:\n"
        values_output_string += "          file: ./" + os.path.relpath(helm_charts_dir, csar_dir) + "/" + helm_tar_ball + "\n"
        values_output_string += "          type: tosca.artifacts.HelmChart\n"

        extractCmd = "tar -zxf " + helm_charts_dir + "/" + helm_tar_ball + " -C ./helm_temp 1>/dev/null"
        os.system (extractCmd)
        helm_temp_dir = ""
        items = os.listdir("./helm_temp/")
        #build helm_temp_dir assuming we have only one directory present in ./helm_temp/
        for d in items:
            if os.path.isdir(os.path.join("./helm_temp/", d)):
                helm_temp_dir = "./helm_temp/"  + d 

        if os.path.exists(helm_temp_dir + "/values.yaml") and os.path.exists(helm_temp_dir + "/values.yaml-production"):
            os.remove(helm_temp_dir + "/values.yaml-production")
            print("both values.yaml and values.yaml-production exists")
        elif os.path.exists(helm_temp_dir + "/values.yaml-production"):
            os.rename(helm_temp_dir + "/values.yaml-production", helm_temp_dir + "/values.yaml")
        for value_yaml in os.listdir(helm_temp_dir):
            #if "values" in value_yaml and "yaml" in value_yaml:
            #copying only values.yaml
            #if value_yaml == "values.yaml-production" or value_yaml == "values.yaml":
            if value_yaml == "values.yaml":
                print("Updating "+ value_yaml + ". Replacing image tag '"+ tagToReplace + "', if any,  with '" + product_build + "'" )

                originalYaml = os.path.join(helm_temp_dir, value_yaml)
                inputYaml = os.path.join(helm_temp_dir, value_yaml + ".tmp")
                os.rename (originalYaml, inputYaml)

                #Add prefix to values yaml
                if ( custom_env_name["${values_file_name_prefix}"].val != ""):
                    value_yaml = custom_env_name["${values_file_name_prefix}"].val + "_" + value_yaml
                outputYaml = os.path.join(helm_temp_dir, value_yaml)
                with open(inputYaml, 'r') as in_file:
                    with open(outputYaml, 'w') as out_file:
                        line = in_file.readline()
                        while line:
                            #print(line)
                            if not line.find(tagToReplace) == -1:
                                #print(line)
                                line = line.replace(tagToReplace, product_build)
                                #print(line)
                            out_file.write(line)
                            line = in_file.readline()

                os.remove(inputYaml)
                shutil.copy(outputYaml, helm_values_dir)
                values_output_string += "        " + value_yaml + ":\n"
                values_output_string += "          file: ./" + os.path.relpath(helm_values_dir, csar_dir) + "/" + value_yaml + "\n"
                values_output_string += "          type: tosca.artifacts.File\n"
            elif value_yaml.find("pfe") != -1 or value_yaml.find("advanced") != -1 or value_yaml.find("functional") != -1:
                outputYaml = os.path.join(helm_temp_dir, value_yaml)
                os.remove(outputYaml)
            elif value_yaml.find("values") != -1 and value_yaml.find("json") != -1:
                inputJson = os.path.join(helm_temp_dir, value_yaml)
                if ( custom_env_name["${values_file_name_prefix}"].val != ""):
                    value_yaml = custom_env_name["${values_file_name_prefix}"].val + "_" + value_yaml
                outputJson = os.path.join(helm_temp_dir, value_yaml)
                os.rename (inputJson, outputJson)

        
        os.remove(helm_charts_dir + "/" + helm_tar_ball)
        os.chdir("./helm_temp")
        for tarDir in os.listdir():
            os.rename(tarDir, tarDir + "-" + product_build)
            achiveCmd = "tar -zcf " +  helm_charts_dir + "/" + helm_tar_ball  + " " + tarDir + "-" + product_build 
            os.system(achiveCmd)
        os.chdir("../")
        #print (values_output_string)
    return values_output_string

#Parse the values.yaml file, download image, re-tag as per build_tag and save it in tar in csar_image_dir
def parseValuesYamlDockerPull(build_tag, valueYamlFile, csar_image_dir):
   print("parsing " + valueYamlFile + " build tag " + build_tag )
   cmd1 = "grep -E 'artifactory.*rbbn.com|repository' " + valueYamlFile + "| grep -v registry  | grep -vE '^\s*#' | cut -d ':' -f2"
   #cmd = "grep artifactory " + valueYamlFile + "| grep -v registry  | grep -vE '^\s*#' | sed 's/^[ \t]*//r' | cut -d ' ' -f2"
   result = subprocess.run(cmd1, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True)
   image_paths1 = [line.lstrip() for line in result.stdout.split('\n')]
   image_paths2 = [line.strip('"') for line in image_paths1]
   image_paths = [line if line.startswith('artifactory-tx.rbbn.com') else 'artifactory-tx.rbbn.com/' + line for line in image_paths2]
   print(image_paths)

   cmd2 = "grep 'tag:' " + valueYamlFile + "| grep -v registry  | grep -vE '^\s*#' | cut -d ':' -f2"
   result = subprocess.run(cmd2, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True)
   image_tags1 = [line.lstrip() for line in result.stdout.split('\n')]
   image_tags = [line.strip('"') for line in image_tags1]
   print(image_tags)

   for i in range(len(image_paths) -1):
       os.makedirs(csar_image_dir, exist_ok=True)  

       path_tag = image_paths[i] + ":" + image_tags[i]
       pull_cmd1 = "docker pull " + path_tag
       print(pull_cmd1)
       os.system(pull_cmd1)

       image_name = image_paths[i][image_paths[i].rfind("/") + 1:]
       #print(image_name)
       destn_tar = csar_image_dir + "/" + image_name + "Image.tar"
       #if imageTar is already created, check for same image another exists.
       if os.path.isfile(destn_tar):
           first_index = image_paths.index(image_paths[i])
           if image_tags[first_index] != image_tags[i]:
               destn_tar = csar_image_dir + "/" + image_name + "Image" + image_tags[i] + ".tar"
               if os.path.isfile(destn_tar):
                   continue;
       imageNameToSave = image_name + ":" + image_tags[i]
       tag_cmd = "docker tag " + path_tag + " " + imageNameToSave
       save_cmd = "docker save " + imageNameToSave + " > " + destn_tar
       print(tag_cmd)
       print(save_cmd)
       os.system(tag_cmd)
       os.system(save_cmd)


def xsbc_create_csar(prod_vers, sub_vers, vnfd_file, customtypes_file, csar_dir):
    # The subversion parameter is an optional parameter to create a CSAR file.
    # If it's not set, we tag the VNFD file to "latest", with no subversion set.
    global version_tag

    global warning_msg
    global product_name
    global customer_desc_name
    global customer_name
    global privKeyFile
    global pubCertFile
    global env_args

    if not os.path.exists(vnfd_file):
        print("ERROR: " + vnfd_file + " file does not exist.")
        return

    # Overwrite environment variables with command line supplied values
    for ikey in env_var_val_dict:
        envKey = "${" + ikey + "}"
        if envKey in custom_env_name:
           custom_env_name[envKey].val = env_var_val_dict[ikey]
           
    include_certifcate = custom_env_name["${include_certifcate}"].val.lower()
    #Check if this is just sigining a provided CSAR file
    if (custom_env_name["${sign_csar}"].val != ''):
        csarFile = custom_env_name["${sign_csar}"].val
        if not os.path.exists (csarFile):
            print(csarFile + " does not exist")
            return
        outputDirTemp = csarFile[0:-5] + "-temp"
        extractCsar (csarFile, outputDirTemp)
        removeSigFiles(outputDirTemp)
        updateManifest(outputDirTemp)
        backupFile = csarFile + "." + str(time.time())
        print("Backing up original csar file in csarFile " + backupFile)
        os.rename(csarFile, backupFile)
        if "cert_file" not in env_var_val_dict or "key_file" not in env_var_val_dict:
            print("Generating self signed key and certificate.")
            generateSelfSignedKeyCert(privKeyFile, pubCertFile)
        else:
            print("Signing with pulblic certificate" )
            privKeyFile = custom_env_name["${key_file}"].val
            pubCertFile = custom_env_name["${cert_file}"].val
        signFiles(outputDirTemp, privKeyFile, pubCertFile)
        if include_certifcate == "true":
            shutil.copy(pubCertFile,  outputDirTemp + "/Files/" + product_name + ".cert")
        createCsar(outputDirTemp, csarFile)
        print("Saved CSAR file in " + csarFile)
        shutil.rmtree(outputDirTemp, ignore_errors=True)
        sig_file = csarFile + ".sig"
        openssl_cmd = "openssl cms -sign -in " + csarFile + " -inkey " + privKeyFile + " -signer " + pubCertFile + " -out " + sig_file + " -outform PEM -binary"
        os.system(openssl_cmd)
        print("Saved CSAR Signature file in " + sig_file)
        csar_hash = hash_file_sha256(csarFile)
        with open(os.path.join(csarFile + ".sha256"), "w") as out_file:
            out_file.write(csar_hash + '\n')
        print("Saved CSAR SHA-256 digest in " + csarFile + ".sha256")
        return

    jenkins_csar = custom_env_name["${jenkins_csar}"].val.lower()
    job_name = custom_env_name["${job_name}"].val.lower()
    if(jenkins_csar == "true") and job_name != '': 
        print("Building csar for jenkins build")
    elif(custom_env_name["${json_file}"].val == ""): 
        print("Warning: Json file with appropriate config should be provided to acrhive artfacts")
    # Overwite product and sub version picked locally from workspace, which is derived from tar file exists in rel folder
    # Also Validate values supplied

    mandArgSet = True
    for iKey in custom_env_name:
        if (custom_env_name[iKey].env == "product_version"):
            if (prod_vers != ""):
                custom_env_name[iKey].val = prod_vers
            else:
                prod_vers = custom_env_name[iKey].val

        if (custom_env_name[iKey].env == "sub_version"):
            if (sub_vers != ""):
                custom_env_name[iKey].val = sub_vers
            else:
                sub_vers = custom_env_name[iKey].val

        if (custom_env_name[iKey].validate):
            if (custom_env_name[iKey].val.lower() != "true" and custom_env_name[iKey].val.lower() != "false"):
                 print("Invalid value for '--" + custom_env_name[iKey].env + "', Usage:  '--" + custom_env_name[iKey].env + " <true/false>'\n")
                 exit()
        if (custom_env_name[iKey].mandate_arg == True):
            if (custom_env_name[iKey].val == ""):
                 print("The value for mandate attribute '" + custom_env_name[iKey].env + "' is neither provided in command line nor sourced as env variable\n")
                 mandArgSet = False

    if mandArgSet == False:
        exit()

    if validateArgValues() == False:
        exit()

    if sub_vers == "":
        full_vers = prod_vers + "_latest"
    else:
        full_vers = prod_vers + sub_vers

    if not full_vers[1] == '.':
        version_tag = "V"+full_vers
    else:
        version_tag = "V0"+full_vers

    verbose_description = "Deploys a " + custom_env_name["${customer_desc_name}"].val + " CNF SBC " + full_vers

    current_dir = os.getcwd()

    product_name = custom_env_name["${product_name}"].val.lower()
    vt_name = product_name + "-" + custom_env_name["${product_build}"].val.lower() 

    #Build and create all directories for CSAR tree
    vnfd_helm_charts_path = "./Artifacts/Scripts/Application-Helm-Charts"
    vnfd_helm_values_path = "./Artifacts/Scripts/Helm-Values"
    csar_app_images_path = "./Artifacts/Images/Application-Images"

    output_dir = os.path.join(csar_dir, vt_name)
    def_dir = os.path.join(output_dir, "Definitions")
    files_dir = os.path.join(output_dir, "Files")
    tosca_dir = os.path.join(output_dir, "TOSCA-Metadata")
    artifacts_dir = os.path.join(output_dir, "Artifacts")
    csar_scripts_dir = "None"
    csar_images_dir = os.path.join(artifacts_dir, "Images")
    helm_values_dir = os.path.join(output_dir, vnfd_helm_values_path)

    if os.path.exists(output_dir):
        shutil.rmtree(output_dir, ignore_errors=True)
    os.mkdir(output_dir)
    os.mkdir(def_dir)
    os.mkdir(files_dir)
    os.mkdir(tosca_dir)

    #Define Directries where pre-downloaded images and helm charts exists
    images_dir = "./Images"
    helm_dir = "./Helm"

    comment_line = "#" + vt_name + "#"

    #build output vnfd files
    template_version = custom_env_name["${template_version}"].val
    customer_short_name = custom_env_name["${customer_short_name}"].val.lower()
    output_vnfd_file = customer_short_name + "_cnf_sol001_vnfd.yaml"
    output_vnfd = os.path.join(def_dir, output_vnfd_file)
    helm_custom_types_yaml = customer_short_name +"_helm_extensions_1_1.yaml"
    output_customtypes = os.path.join(def_dir, helm_custom_types_yaml)
    date = datetime.datetime.utcnow().replace(microsecond=0).isoformat()


    """Start Write VNFD File"""
    if template_version == "2.5.1":
        etsi_vnfd_types_file = os.path.join(script_dir, "etsi_nfv_sol001_vnfd_2_5_1_types.yaml")
        output_etsi_vnfd_types = os.path.join(def_dir, "etsi_nfv_sol001_vnfd_2_5_1_types.yaml")
        in_out_file_list = [ (vnfd_file, output_vnfd), (customtypes_file, output_customtypes), (etsi_vnfd_types_file, output_etsi_vnfd_types)]
    else:
        etsi_vnfd_types_file = os.path.join(script_dir, "etsi_nfv_sol001_vnfd_types.yaml")
        output_etsi_vnfd_types = os.path.join(def_dir, "etsi_nfv_sol001_vnfd_types.yaml")
        output_etsi_common_types_file = os.path.join(def_dir, "etsi_nfv_sol001_common_types.yaml")
        in_out_file_list = [ (vnfd_file, output_vnfd), (customtypes_file, output_customtypes), (etsi_vnfd_types_file, output_etsi_vnfd_types), (etsi_common_types_file, output_etsi_common_types_file)]

    for in_out_file in in_out_file_list:
        with open(in_out_file[0], 'r') as in_file:
            with open(in_out_file[1], 'w') as out_file:
                line = in_file.readline()
                while line:
                    skip_line = False

                    # Dump cmdline_args
                    if not line.find('${cmdline_args}') == -1:
                        if custom_env_name["${skip_cmd_arg}"].val.lower() == "false":
                            line = line.replace("${cmdline_args}", cmdline_args)
                        else:
                            skip_line = True
                    # In addition to cmdline_args, also dump out significant environment vars.
                    if not line.find('#environ_args#') == -1:
                        if custom_env_name["${skip_env_arg}"].val.lower() == "false":
                            line = line.replace("#environ_args#", "#environ_args#\n" + env_args)
                        else:
                            skip_line = True
                    if not line.find("##skip##") == -1:
                        skip_line = True
                    if template_version == "2.5.1":
                        if not line.find("##2.5.1##") == -1:
                            line = line.replace("##2.5.1##", "")
                        if not line.find("##3.5.1##") == -1:
                            skip_line = True
                    else:
                        if not line.find("##3.5.1##") == -1:
                            line = line.replace("##3.5.1##", "")
                        if not line.find("##2.5.1##") == -1:
                            skip_line = True

                    line = line.replace("${template_author}", template_author)
                    line = line.replace("${vnftype}", vt_name)
                    line = line.replace("${verbose_description}", verbose_description)
                    line = line.replace("${date}", date)
                    line = line.replace("${helm_version}", helm_version)
                    sw_version_suffix = ""
                    line = line.replace("${sw_version_suffix}", sw_version_suffix)
                    line = line.replace("${version_tag}", version_tag)
                    line = line.replace(comment_line, "")

                    for custom_name_key in custom_env_name:
                        if not line.find(custom_name_key) == -1:
                            line = line.replace(custom_name_key, custom_env_name[custom_name_key].val)

                    line = line.replace("${tosca_definitions_version}", tosca_def_v)
                    if not line.find("${descriptor_id}") == -1:
                        uuid_personality = "{0:0<4}".format(vt_name)[0:4]

                        uuid_version = prod_vers.lstrip('V0')
                        uuid_subversion = "{0:0<4}".format(sub_vers)[0:4]
                        encode_hex = codecs.getencoder('hex')
                        vnfd_uuid = uuid.UUID(encode_hex((uuid_personality+uuid_version+uuid_subversion).encode())[0].decode())
                        line = line.replace("${descriptor_id}", str(vnfd_uuid))

    
                    if not skip_line:
                        out_file.write(line)
    
                    line = in_file.readline()

    vnfd_hash = hash_file_sha256(vnfd_file)


    #if CSAR to be created during jenkins build, download helm zip file based on the inputs from command
    #line arguments to local dir Helm-Charts. Zip file name needs to be constructed based on subversion or build number
    if(jenkins_csar == "true"): 
        print("Jenkins csar Downloading Helm")
        reldir = custom_env_name["${job_name}"].val
        artifactory_path = "oci://" +  reldir + custom_env_name["${product_name}"].val

        print(custom_env_name["${sub_version}"].val)
        #For GA build, then helm-zip will not have build number(rbbn-core-cnf-12.1.4-R000.tgz)
        #For non GA build, then helm-zip will have build number(rbbn-core-cnf-12.1.5-14.tgz)
        if "R" in custom_env_name["${sub_version}"].val and "R" in custom_env_name["${product_build}"].val :
            second_part =  custom_env_name["${product_version}"].val.replace("0", "") + "-" + custom_env_name["${sub_version}"].val
        else:
            second_part =  custom_env_name["${product_build}"].val
        credentials = " --username " + custom_env_name["${username}"].val + " --password " + custom_env_name["${user_password}"].val + " "
        destn_dir = "./Helm-Charts"
        if os.path.exists(destn_dir):  # Check if the directory exists
            shutil.rmtree(destn_dir)  # Delete the directory
        os.makedirs(destn_dir)
        destination_arg = " --destination " + destn_dir
        helm_pull_cmd = "helm pull " + artifactory_path + " --version " + second_part  + credentials + destination_arg
        print(helm_pull_cmd)
        os.system(helm_pull_cmd)
        print("Downloading dockerImages by parsing values.yaml")

        json_artifacts = createCsarArtifactsFromJson(output_dir, custom_env_name["${passwd_token}"].val, custom_env_name["${username}"].val, custom_env_name["${user_password}"].val, custom_env_name["${json_file}"].val, jenkins_csar)

        #print("Updating vnfd file")
    elif(custom_env_name["${json_file}"].val != ""): 
        json_artifacts = createCsarArtifactsFromJson(output_dir, custom_env_name["${passwd_token}"].val, custom_env_name["${username}"].val, custom_env_name["${user_password}"].val, custom_env_name["${json_file}"].val, jenkins_csar)
        if len(json_artifacts) != 0:
            helm_charts_dir = json_artifacts["rbbn-core-cnf"]["Helm-Charts"]["files"]
            if json_artifacts["rbbn-core-cnf"]["Helm-Charts"]["files"].find(custom_env_name["${product_build}"].val) == -1:
                print("Warning: product_build '" + custom_env_name["${product_build}"].val + "' mismatches with version contained in helm chart tar zip file '" + json_artifacts["rbbn-core-cnf"]["Helm-Charts"]["files"] + " supplied via json file")
        else:
            print("Warning: " + custom_env_name["${json_file}"].val + " is empty or improper !!!")

    #outStr = buildVnfdAndManifestOutputString(output_dir)

    #build string to updated the deatils of artifacts that to be updated in vnfd file
    execOnce = True
    output_string = ""
    imagesArchived = False
    if len(json_artifacts) != 0:
        for i in json_artifacts:
            if json_artifacts[i]["Helm-Charts"]["archiveOrNot"].lower() == "true":
                if len(json_artifacts[i]["Helm-Charts"]["csarPath"]) > 0:
                    json_helm_charts_dir = json_artifacts[i]["Helm-Charts"]["csarPath"]
                    if (json_helm_charts_dir.endswith("/")):
                        json_helm_charts_dir = json_helm_charts_dir[:-1]
                    csar_scripts_dir = json_helm_charts_dir[:json_helm_charts_dir.rfind("/")]
                    helm_values_dir = csar_scripts_dir + "/Helm-Values"
                    helm_charts_dir = os.path.join(output_dir, json_helm_charts_dir)
                    helm_values_dir = os.path.join(output_dir, helm_values_dir)
                    if (csar_scripts_dir.startswith("./")):
                        csar_scripts_dir = csar_scripts_dir[2:]

                    output_string += updateTagAndRebuildHelm(output_dir, helm_charts_dir, helm_values_dir, custom_env_name["${product_build}"].val)
                    #print(" helm_values_dir" + helm_values_dir)
                    if (jenkins_csar == "true" and execOnce == True):
                        execOnce = False
                        if os.path.exists(helm_values_dir + "/values.yaml-production"):
                             valuesYaml = helm_values_dir + "/values.yaml-production"
                        else:
                             valuesYaml = helm_values_dir + "/values.yaml"
                        image_dir = json_artifacts[i]["Images"]["csarPath"]
                        #print(" valuesYaml " + valuesYaml)
                        #print(" image_dir " + image_dir)
                        csar_output_dir = os.path.join(output_dir, image_dir)
                        parseValuesYamlDockerPull(custom_env_name["${product_build}"].val, valuesYaml, csar_output_dir)


            for j in json_artifacts[i]:
                if json_artifacts[i][j]["archiveOrNot"].lower() == "true" and json_artifacts[i][j]["isDockerImage"].lower() == "true":
                    csar_app_images_path = json_artifacts[i][j]["csarPath"]
                    if (csar_app_images_path.startswith("./")):
                        csar_app_images_path = csar_app_images_path[2:]
                    if (csar_app_images_path.endswith("/")):
                        csar_app_images_path = csar_app_images_path[:-1]
                    csar_images_dir = csar_app_images_path[:csar_app_images_path.rfind("/")]

                    app_images_dir = os.path.join(output_dir, csar_app_images_path)
                    if not os.path.exists(app_images_dir):
                        print("Warning: '" + app_images_dir + "' does not exist")
                        continue
                    for file_name in os.listdir(app_images_dir):
                        output_string += "        " + file_name[0:-4] + ":\n"
                        output_string += "          file: " + csar_app_images_path + "/" + file_name + "\n"
                        output_string += "          type: tosca.artifacts.nfv.SwImage\n"
                        imagesArchived = True

    file1 = open(output_vnfd, "a")

    if output_string != "":
        file1.write(output_string)
    file1.close()

    manifest_file_name = custom_env_name["${customer_short_name}"].val.lower() + "_cnf_vnfd.mf"
    template_version = custom_env_name["${template_version}"].val
    with open(os.path.join(files_dir, "ChangeLog.txt"), "w") as out_file:
        out_file.write(' SBC CNF Support with SOL001 ' + template_version + '\n')
    out_file.close()

    with open(os.path.join(tosca_dir, "TOSCA.meta"), "w") as out_file:
        out_file.write('TOSCA-Meta-File-Version: 1.0\n')
        out_file.write('CSAR-Version: 1.1\n')
        out_file.write('Created-By: ' + template_version + '\n')
        if template_version == "2.5.1":
            out_file.write('Entry-Definitions: Definitions/' + output_vnfd_file + '\n')
        else:
            out_file.write('ETSI-Entry-Definitions: Definitions/' + output_vnfd_file + '\n')
        out_file.write('ETSI-Entry-Manifest: ' + manifest_file_name + '\n')
        
        if customer_short_name.lower() == 'vz':
            etsiPrefix = 'VZ'
        else:
            etsiPrefix = 'ETSI'
        if imagesArchived == True:
            out_file.write(etsiPrefix + '-Entry-Images: ' + csar_app_images_path + '\n')
        else:
            out_file.write(etsiPrefix + '-Entry-Images: None\n')
        out_file.write(etsiPrefix + '-Entry-Scripts: ' + csar_scripts_dir + '\n')
        out_file.write('ETSI-Entry-Change-Log: Files/ChangeLog.txt\n')
        if include_certifcate == "true":
            out_file.write('ETSI-Entry-Certificate: Files/' + product_name + '.cert\n')
    out_file.close()

    with open(os.path.join(output_dir, manifest_file_name), "w") as out_file:
        out_file.write('metadata:\n')
        out_file.write('  vnfd_id: ' + str(vnfd_uuid) + '\n')
        out_file.write('  vnf_product_name: ' + product_name + '\n')
        out_file.write('  vnf_provider_id: ' + custom_env_name["${provider}"].val + '\n')
        out_file.write('  vnf_package_version: ' + '\'3.0.0\'' + '\n')
        out_file.write('  vnf_software_version: ' + version_tag + version_suffix + '\n')
        out_file.write('  vnf_release_date_time: ' + date + '\n')
        out_file.write('  compatible_specification_versions: ' + custom_env_name["${template_version}"].val + '\n')
        out_file.write('  vnfm_info: '+ custom_env_name["${vnfm}"].val + '\n')
        out_file.write(calculateCheckSum(output_dir))
        out_file.write('\n')
    out_file.close()

    readmeTxt = "For signature verification use:\n"
    readmeTxt += "openssl cms -verify -in <fileName>.sig -inform PEM -binary -content <fileName> -out <fileName>.verified -noverify"
    with open(os.path.join(output_dir, "README"), "w") as out_file:
        out_file.write(readmeTxt)
    out_file.close()

    if custom_env_name["${sign_files}"].val.lower() == "true":
        if len(custom_env_name["${cert_file}"].val) == 0 or len(custom_env_name["${key_file}"].val) == 0:
            print("Generating self signed key and certificate.")
            generateSelfSignedKeyCert(privKeyFile, pubCertFile)
        else:
            print("Signing with pulblic certificate")
            privKeyFile = custom_env_name["${key_file}"].val
            pubCertFile = custom_env_name["${cert_file}"].val
        signFiles(output_dir, privKeyFile, pubCertFile)

        if include_certifcate == "true":
            shutil.copy(pubCertFile,  output_dir + "/Files/" + product_name + ".cert")
    csar_file = os.path.join(csar_dir, vt_name + ".csar")
    print("Creating " + csar_file)
    
    createCsar(output_dir, csar_file)
    print("All files in csar are available in directory: " + os.path.join(csar_dir, vt_name))
    print("Saved CSAR file in " + csar_file)

    #if os.path.exists(certificate_file) and os.path.exists(private_key_file):
    #if "cert_file" in env_var_val_dict and "key_file" in env_var_val_dict:
    if custom_env_name["${sign_files}"].val.lower() == "true":
        sig_file = os.path.join(csar_dir, vt_name + ".csar.sig")
        openssl_cmd = "openssl cms -sign -in " + csar_file + " -inkey " + privKeyFile + " -signer " + pubCertFile + " -out " + sig_file + " -outform PEM -binary"
        os.system(openssl_cmd)
        print("Saved CSAR Signature file in " + sig_file)
    csar_hash = hash_file_sha256(csar_file)
    with open(os.path.join(csar_dir, vt_name + ".csar.sha256"), "w") as out_file:
            out_file.write(csar_hash + '\n')
    print("Saved CSAR SHA-256 digest in " + os.path.join(csar_dir, vt_name + ".sha256"))
    if(jenkins_csar == "true"):
        files_to_move = glob.glob(os.path.join("./CSAR", "*csar*"))
        for file in files_to_move:
           shutil.move(file, "./")
        shutil.rmtree("CSAR", ignore_errors=True)
        shutil.rmtree("Helm-Charts", ignore_errors=True)
    #remove temporary directories and files
    shutil.rmtree("./helm_temp", ignore_errors=True)
    
    #os.remove(pubCertFile)


def xsbc_create_csars(prod_vers, sub_vers, vnfd_file, customtypes_file):

    global warning_msg
    global version_suffix
    version_suffix = ""

    # Start clean to avoid any stale files
    vnfd_dir = os.path.abspath(os.path.dirname(vnfd_file))
    if vnfd_dir == "":
        vnfd_dir = "."
    csar_dir = os.path.join(vnfd_dir, "CSAR")
    if not os.path.exists(csar_dir):
        os.mkdir(csar_dir)

    xsbc_create_csar(prod_vers, sub_vers, vnfd_file, customtypes_file, csar_dir)


# MAIN
if __name__ == "__main__":
    print("Script Dir:" + script_dir)
    if not os.path.exists(cnf_vnfd_sol001_template):
        cnf_vnfd_sol001_template = os.path.join(os.path.dirname(script_dir), "templates", "cnfSol001VnfdTemplate.yaml")
    if not os.path.exists(vnfd_ribbon_custom_types_template):
        vnfd_ribbon_custom_types_template = os.path.join(os.path.dirname(script_dir), "templates", "helm_extensions_1_1.yaml")

    product_version = ""
    sub_version = ""
    if "ORCA_ROOT" in os.environ:
        sbc_tar_file = str(glob.glob(os.path.join(os.environ['ORCA_ROOT'], 'rel', 'sbc-V*.tar.gz')))
        tar_split = sbc_tar_file.split('-')
        if len(tar_split) > 2:
            product_version = tar_split[1].lstrip('V0')
            sub_version = tar_split[2].split('.')[0]

    ret = get_args(sys.argv)

    if vnfd_template == "":
        vnfd_template = cnf_vnfd_sol001_template

    if vnfd_custom_types_file == "":
        vnfd_custom_types_file = vnfd_ribbon_custom_types_template

    if ret:
        #if sub_version == "":
        #    print("Product Version:" + product_version)
        #else:
        #    print("Product Version:" + product_version + "Sub-Version:" + sub_version)
        print("VNFD Template:" + vnfd_template)

        xsbc_create_csars(product_version, sub_version, vnfd_template, vnfd_custom_types_file)

